<?php namespace Visiosoft\PaymentModule\Payment;

use Illuminate\Foundation\Bus\DispatchesJobs;
use Illuminate\Support\Facades\Auth;
use Visiosoft\ChargingModule\Transaction\Contract\TransactionRepositoryInterface;
use Visiosoft\ConnectModule\Command\CheckRequiredParams;
use Visiosoft\ParkingModule\Park\Contract\PricingCostCalculateHelperInterface;
use Visiosoft\ParkingModule\ParkingSession\Contract\ParkingSessionRepositoryInterface;
use Visiosoft\ParkingModule\Ticket\Contract\TicketRepositoryInterface;
use Visiosoft\ParkingModule\TicketUser\Contract\TicketUserRepositoryInterface;
use Visiosoft\PaymentModule\Category\Contract\CategoryRepositoryInterface;
use Visiosoft\PaymentModule\PaymentMethod\Contract\PaymentMethodRepositoryInterface;
use Visiosoft\PaymentModule\Status\Contract\StatusRepositoryInterface;
use Visiosoft\PaymentStripeExtension\PaymentStripeExtension;
use function Clue\StreamFilter\fun;

class PaymentApiCollection extends PaymentRepository
{
    use DispatchesJobs;

    public function createCard(array $params)
    {
        $userId = Auth::id();
        if (!$create_card = PaymentStripeExtension::createCard($params)) {
            throw new \Exception(trans('visiosoft.module.payment::message.cant_add_card'));
        }
        $create_card = $create_card->toArray();
        if (!empty($create_card['id'])) {
            $create_card['service_payment_method_id'] = $create_card['id'];
            $create_card['service'] = 'stripe';
        }
        unset($create_card['id']);
        unset($create_card['object']);
        $create_card['owner_id'] = $userId;
        $create_card['holder_name'] = $params['name'];
        $paymentMethodRepository = app(PaymentMethodRepositoryInterface::class);
        return $paymentMethodRepository->addPaymentMethod($create_card);
    }

    public function cards()
    {
        $cards = PaymentStripeExtension::cards();
        $paymentMethodIds = [];
        if (!empty($cards)) {
            foreach ($cards as $card) {
                $paymentMethodIds[$card['id']] = $card['id'];
            }
        }
        $results = [];
        $paymentMethodRepository = app(PaymentMethodRepositoryInterface::class);
        $paymentMethods = $paymentMethodRepository->newQuery()
            ->where('owner_id', Auth::id())->whereNull('deleted_at')->get();
        foreach ($paymentMethods as $paymentMethod) {
            if (!empty($paymentMethodIds[$paymentMethod->service_payment_method_id])) {
                $results[] = $paymentMethod;
                unset($paymentMethodIds[$paymentMethod->service_payment_method_id]);
            } else {
                $paymentMethod->delete();
            }
        }
        return $paymentMethodRepository->newQuery()
            ->where('owner_id', Auth::id())->whereNull('deleted_at')->get();
    }

    public function createDraftPayment(array $params)
    {
        $paymentData = [
            'user_id' => $params['user_id'],
            'amount' => $params['amount'],
            'usage' => $params['usage'],
            'status' => 1,
            'category_id' => $params['category_id'],
            'currency' => $this->currency,
        ];
        if (!empty($params['vehicle_id'])) {
            $paymentData['vehicle_id'] = $params['vehicle_id'];
        }
        if (!empty($params['plate'])) {
            $paymentData['plate'] = $params['plate'];
        }
        if (!empty($params['company_id'])) {
            $paymentData['company_id'] = $params['company_id'];
        }
        if (!empty($params['stripe_intent_id'])) {
            $paymentData['stripe_intent_id'] = $params['stripe_intent_id'];
        }
        if (!empty($params['transaction_id'])) {
            $paymentData['transaction_id'] = $params['transaction_id'];
        }
        if (!empty($params['park_session_id'])) {
            $paymentData['park_session_id'] = $params['park_session_id'];
        }
        if (!empty($params['park_id'])) {
            $paymentData['park_id'] = $params['park_id'];
        }
        if (!empty($params['station_id'])) {
            $paymentData['station_id'] = $params['station_id'];
        }
        if (!empty($params['ticket_id'])) {
            $paymentData['ticket_id'] = $params['ticket_id'];
        }
        $create = $this->newQuery()->create($paymentData);

        if (!$create) {
            throw new \Exception(trans('visiosoft.module.payment::message.transaction_create_failed'), 400);
        }
        return $create;
    }


    public function createPayment(array $params)
    {

        $payment = $this->newQuery()->where('transaction_id', $params['transaction_id'])->first();

        $payment_params = ['currency' => $this->currency, 'amount' => $payment['amount'], 'method_id' => $params['method_id']];

        $create_payment = PaymentStripeExtension::createPaymentIntent($payment_params);
        $payment->update(['payment_intent_id' => $create_payment['id']]);
        return $this->confirmPayment(['payment_id' => $create_payment['id']]);
    }

    public function list(array $params)
    {
        $userId = Auth::id();
        $payments = $this->newQuery()
            ->where('user_id', $userId)
            ->orderBy('created_at', 'DESC');
        if (!empty($params['category_id'])) {
            $payments = $payments->where('category_id', $params['category_id']);
        }
        if (!empty($params['status_id'])) {
            $payments = $payments->where('status_id', $params['status_id']);
        }
        if (!empty($params['start_date'])) {
            $payments = $payments->where('created_at', '>=', $params['start_date']);
        }
        if (!empty($params['end_date'])) {
            $payments = $payments->where('created_at', '<=', $params['end_date']);
        }
        if (!empty($params['account_type_id'])) {
            if ($params['account_type_id'] == 1) {
                $payments = $payments->whereNull('company_id');
            }
            if ($params['account_type_id'] == 2) {
                $payments = $payments->whereNotNull('company_id');
            }
        }
        return $payments;
    }

    public function references()
    {
        $categoryRepository = app(CategoryRepositoryInterface::class);
        $statusRepository = app(StatusRepositoryInterface::class);
        return collect([
            [
                'statuses' => $statusRepository->newQuery()->get(),
                'category' => $categoryRepository->newQuery()->get(),
                'account_types' => [['name' => 'Personal', 'id' => 1], ['name' => 'Business', 'id' => 2]]
            ]
        ]);
    }

    public function removeCard(array $params)
    {
        $this->dispatch(new CheckRequiredParams(['card_id'], $params));
        $paymentMethodRepository = app(PaymentMethodRepositoryInterface::class);
        $user = Auth::user();
        if (!$paymentMethod = $paymentMethodRepository->newQuery('owner_id', $user->id)->find($params['card_id'])) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => 'Payment Method']));
        }
        return $detached = PaymentStripeExtension::detachPaymentMethod($paymentMethod->service_payment_method_id, $user->stripe_id);
    }

    public function createPaymentIntent(array $params)
    {
        $this->dispatch(new CheckRequiredParams(['payment_id'], $params));
        if (!$payment = $this->newQuery()->where('user_id', Auth::id())->find($params['payment_id'])) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => 'Payment']));
        }
        $user = Auth::user();
        $amount = $payment->amount * 100;
        $currency = $payment->currency;
        $paymentMethodType = 'card';
        if (!empty($params['payment_method_type'])) {
            if ($params['payment_method_type'] == 'sepa_debit') {
                $currency = 'eur';
                $paymentMethodType = 'sepa_debit';
            }
        }

        if (empty($user->stripe_id)) {
            $description = $user->first_name . ' ' . $user->last_name;
            $createCustomer = PaymentStripeExtension::createCustomer($description, $user->email);
            $customer = $createCustomer->id;
            $user->stripe_id = $customer;
            $user->save();
        } else {
            $customer = $user->stripe_id;
        }
        $paymentIntent = PaymentStripeExtension::createPaymentIntent(['amount' => $amount, 'currency' => $currency, 'customer' => $customer, 'payment_method_type' => $paymentMethodType]);
        if (empty($paymentIntent->id)) {
            throw new \Exception($paymentIntent);
        }
        $payment->update(['stripe_intent_id' => $paymentIntent->id, 'stripe_create_intent' => json_encode($paymentIntent)]);
        return $paymentIntent;
    }

    public function confirmPayment(array $params)
    {
        $this->dispatch(new CheckRequiredParams(['payment_id'], $params));
        if (!$payment = $this->newQuery()->where('user_id', Auth::id())->find($params['payment_id'])) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => 'Payment']));
        }
        $confirmation = PaymentStripeExtension::confirmPaymentIntent(['stripe_intent_id' => $payment->stripe_intent_id]);
        if ($confirmation['status'] == "succeeded" && $confirmation['paid']) {
            $payment->update(['status_id' => 2, 'stripe_confirm_intent' => json_encode($confirmation)]);
        }
        return $confirmation;
    }

    public function pay(array $params)
    {
        $paymentMethodRepository = app(PaymentMethodRepositoryInterface::class);
        $payment = $this->newQuery()->where('id', $params['payment_id'])->first();
        $payment_method = $paymentMethodRepository->newQuery()
            ->where('owner_id', Auth::id())
            ->where('default', 1)
            ->whereNull('deleted_at')
            ->first();

        $details = [];

        // TODO:: This section must be global
        $paymentToArray = $payment->toArray();
        if (!empty($paymentToArray['charging_transaction'])) {
            $transaction = $paymentToArray['charging_transaction'];
            $duration = strtotime($transaction['ended_at']) - strtotime($transaction['started_at']);
            if ($duration < 60) {
                $duration = 60;
            }
            $consumedAmount = round($duration / 60, 0);
            $paymentAmount = $consumedAmount;
            $transactionRepository = app(TransactionRepositoryInterface::class);
            $details = $transactionRepository->getDetail($paymentAmount, $duration);
        }

        if (!empty($payment->park_session)) {
            $session = $payment->park_session;
            $duration = strtotime($session->ended_at) - strtotime($session->started_at);
            if ($duration < 60) {
                $duration = 60;
            }
            $calculator = app(PricingCostCalculateHelperInterface::class);
            $parkSessionRepository = app(ParkingSessionRepositoryInterface::class);
            $cost = $calculator->calculate($session, $duration);
            $details = $parkSessionRepository->getDetail($paymentToArray['amount'], $cost, $duration);
        }
        // TODO:: This section must be global

        $payment_params =
            [
                'currency' => $this->currency,
                'amount' => $details['payment_amount'] * 100,
                'payment_method' => $payment_method->service_payment_method_id,
                'payment_method_type' => 'card',
                'customer' => $payment_method->customer
            ];


        if ($details['payment_amount'] < 3) {
            throw new \Exception(trans('visiosoft.module.payment::message.minimum_price') . $this->currency);
        }

        $create_payment = PaymentStripeExtension::createPaymentIntent($payment_params);
        $payment->update(['payment_intent_id' => $create_payment['id']]);
        $confirm_payment = PaymentStripeExtension::confirmPaymentIntent(['stripe_intent_id' => $create_payment['id']]);


        if (isset($confirm_payment['status']) && $confirm_payment['status'] == "succeeded") {
            $payment->update(['status_id' => 2]);
            return $confirm_payment;
        } else {
            throw new \Exception($confirm_payment);
        }
    }

    public function setDefaultCard(array $params)
    {
        $this->dispatch(new CheckRequiredParams(['payment_method_id'], $params));
        $paymentMethodRepository = app(PaymentMethodRepositoryInterface::class);
        if (!$paymentMethod = $paymentMethodRepository->newQuery()->where('owner_id', Auth::id())->find($params['payment_method_id'])) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => 'Card']));
        }
        $paymentMethodRepository->newQuery()->where('owner_id', Auth::id())->update(['default' => 0]);
        $paymentMethod->update(['default' => 1]);
        return $paymentMethod;
    }

    public function getDefaultCard()
    {
        $paymentMethodRepository = app(PaymentMethodRepositoryInterface::class);
        $defaultCard = $paymentMethodRepository->newQuery()->where('owner_id', Auth::id())->where('default', 1)->first();
        return collect(['default_card' => $defaultCard]);
    }

    public function getPayment(array $params)
    {
        $this->dispatch(new CheckRequiredParams(['payment_id'], $params));
        if (!$payment = $this->newQuery()->where('user_id', Auth::id())->find($params['payment_id'])) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => 'Payment']));
        }
        $paymentCollection = $payment;
        $payment = $payment->toArray();
        $details = [];
        if (!empty($payment['charging_transaction'])) {
            $transaction = $payment['charging_transaction'];
            $duration = strtotime($transaction['ended_at']) - strtotime($transaction['started_at']);
            if ($duration < 60) {
                $duration = 60;
            }
            $consumedAmount = round($duration / 60, 0);
            $paymentAmount = $consumedAmount;
            $transactionRepository = app(TransactionRepositoryInterface::class);
            $details = $transactionRepository->getDetail($paymentAmount, $duration);
        }
        if (!empty($paymentCollection->park_session)) {
            $session = $paymentCollection->park_session;
            $duration = strtotime($session->ended_at) - strtotime($session->started_at);
            if ($duration < 60) {
                $duration = 60;
            }
            $calculator = app(PricingCostCalculateHelperInterface::class);
            $parkSessionRepository = app(ParkingSessionRepositoryInterface::class);
            $cost = $calculator->calculate($session, $duration);
            $details = $parkSessionRepository->getDetail($payment['amount'], $cost, $duration);
        }
        return collect(['payment' => $payment, 'details' => $details]);
    }

    public function startPayment(array $params)
    {
        //todo: payment simulation mode
        $this->dispatch(new CheckRequiredParams(['payment_id'], $params));
        if (!$payment = $this->newQuery()->where('user_id', Auth::id())->find($params['payment_id'])) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => 'Payment']));
        }
        $payment->update(['status_id' => 3]);
        return $payment;
    }

    public function cancelPayment(array $params)
    {
        //todo: payment simulation mode
        $this->dispatch(new CheckRequiredParams(['payment_id'], $params));
        if (!$payment = $this->newQuery()->where('user_id', Auth::id())->find($params['payment_id'])) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => 'Payment']));
        }
        $payment->update(['status_id' => 4]);
        return $payment;
    }

    public function buyTicket(array $params)
    {
        $this->dispatch(new CheckRequiredParams(
            [
                'park_id',
                'ticket_id',
                'plate_id'
            ]
            , $params));


        $user_id = Auth::id();
        $park_id = $params['park_id'];
        $ticket_id = $params['ticket_id'];
        $plate_id = $params['plate_id'];

        $ticketRepository = app(TicketRepositoryInterface::class);
        $categoryRepository = app(CategoryRepositoryInterface::class);
        $category = $categoryRepository->findBy('slug', 'long_term_ticket');
        $ticket = $ticketRepository->find($ticket_id);


        $paymentData = [
            'user_id' => $user_id,
            'amount' => $ticket->price,
            'park_id' => $park_id,
            'ticket_id' => $ticket->id,
            'usage' => '',
            'category_id' => $category->getId(),
            'currency' => $this->currency
        ];

        if ($payment = $this->createDraftPayment($paymentData)) {
            $pay = $this->pay(['payment_id' => $payment->id]);
            if (isset($pay['status']) && $pay['status'] == "succeeded") {
                $ticketUserRepository = app(TicketUserRepositoryInterface::class);
                $started_at = date("Y-m-d H:i:s");
                $ended_at = date('Y-m-d H:i:s', strtotime($started_at . ' + ' . $ticket->time . " " . $ticket->time_unit . ''));

                $createParams = [
                    'plate_id' => $plate_id,
                    'user_id' => $user_id,
                    'ticket_id' => $ticket->getId(),
                    'started_at' => $started_at,
                    'ended_at' => $ended_at,
                    'payment_id' => $payment->id
                ];

                $ticketUserRepository->create($createParams);
                return $pay;
            }
        }
    }
}
