<?php namespace Visiosoft\PaymentModule\Payment;

use Exception;
use Visiosoft\ConnectModule\Command\CheckRequiredParams;
use Visiosoft\CurrencyModule\CurrencyModule;
use Visiosoft\ParkingModule\Park\Contract\PricingCostCalculateHelperInterface;
use Visiosoft\PaymentModule\Event\PaymentCreatedEvent;
use Visiosoft\PaymentModule\Helper\CommonHelper;
use Visiosoft\PaymentModule\Payment\Commands\Card;
use Visiosoft\PaymentModule\Payment\Commands\Pay;
use Visiosoft\PaymentModule\Payment\Contract\PaymentRepositoryInterface;
use Anomaly\Streams\Platform\Entry\EntryRepository;
use Visiosoft\PaymentModule\PaymentMethod\Contract\PaymentMethodRepositoryInterface;

class PaymentRepository extends EntryRepository implements PaymentRepositoryInterface
{

    public string $currency;
    protected $model;
    protected PaymentMethodRepositoryInterface $paymentMethodRepository;
    protected Card $card_provider;
    protected Pay $pay_provider;

    public function __construct(PaymentModel $model, PaymentMethodRepositoryInterface $paymentMethodRepository)
    {
        $this->model = $model;
        $currencyModule = app(CurrencyModule::class);
        $this->currency = $currencyModule->getDefault()->symbol;
        $this->card_provider = new Card();
        $this->pay_provider = new Pay();
        $this->paymentMethodRepository = $paymentMethodRepository;
    }


    /**
     * @param array $params (['payment_id' => int])
     * @return object
     * @throws Exception
     */
    public function getDetail(array $params): object
    {
        $output = [
            'payment_id' => null,
            'location_name' => null,
            'created_at' => null,
            'ended_at' => null,
            'plate' => null,
            'usage_type' => null,
            'is_drifterbox' => null,
            'type' => '',
            'category_id' => null,
            'status_id' => null,
            'park_session_cost' => null,
            'parking_price' => null,
            'park_session_note' => null,
            'charging_price_type' => null,
            'charging_session_cost' => null,
            'digital_code' => null,
            'charging_price' => null,
            'vat' => 0,
            'charging_electricity_consumption' => null,
            'parking_duration' => null,
            'total_amount' => 0,
            'administration_fee' => 0,
            'park_session_status' => null
        ];
        $this->dispatch(new CheckRequiredParams(['payment_id'], $params));
        $payment = $this->newQuery()
            ->whereNull('deleted_at')
            ->find($params['payment_id']);
        if (!$payment) {
            throw new Exception(trans('visiosoft.module.connect::message.not_found', ['name' => trans('visiosoft.module.payment::field.payment')]));
        }
        $output['payment_id'] = $payment->id;
        $output['status_id'] = $payment->status_id;
        $output['category_id'] = $payment->category_id;
        $locationName = '';
        $output['usage_type'] = !empty($payment->company_id) ? 'business' : 'personal';
        if (!empty($payment->charging_transaction)) {
            $chargingPricingType = $payment->charging_transaction->station->pricing_type;
            if (!$chargingPrice = $payment->charging_transaction->transaction_station->charge_price) {
                $chargingPrice = 0;
            }
            $output['charging_price_type'] = $chargingPricingType;
            $output['charging_session_cost'] = $payment->charging_transaction->payment_amount;
            $output['charging_price'] = $chargingPrice;
            $output['charging_electricity_consumption'] = $payment->charging_transaction->total_consumption;
            $locationName = $payment->charging_transaction->station->title;
            $output['created_at'] = $payment->charging_transaction->created_at;
            $output['ended_at'] = $payment->charging_transaction->ended_at;
            $output['plate'] = $payment->charging_transaction->plate_number;
            $output['is_drifterbox'] = !empty($payment->charging_transaction->drifterbox_id) ? 1 : 0;
        }
        if (!empty($payment->park_session)) {
            $parkingCostCalculator = app(PricingCostCalculateHelperInterface::class);
            //todo parking price should submit to the session table. realtime calculation is wrong
            $duration = strtotime($payment->park_session->ended_at) - strtotime($payment->park_session->started_at);
            $costs = $parkingCostCalculator->calculate($payment->park_session, $duration);
            $output['park_session_cost'] = $payment->park_session->payment_amount;
            $output['parking_price'] = $costs['current_price'];
            $output['parking_duration'] = $duration;
            $locationName = $payment->park_session->park->name;
            $output['created_at'] = $payment->park_session->created_at;
            $output['ended_at'] = $payment->park_session->ended_at;
            $output['plate'] = $payment->park_session->plate_number;
            $output['is_drifterbox'] = $payment->park_session->auto_parking;
            $output['park_session_status'] = $payment->park_session->session_status_id == 1 ? 'normal' : 'offense';
            if ($payment->park_session->is_digital_code_active) {
                $output['park_session_note'] = trans('visiosoft.module.payment.message.you_have_used_digital_code');
                $output['digital_code'] = $payment->park_session->digital_code_text;
                $output['park_session_cost'] = 0;
                $output['parking_price'] = 0;
            }
        }
        if ($payment->category_id == 4) {
            $output['type'] = 'park_and_charge';
        }
        if ($payment->category_id == 1) {
            $output['type'] = 'charge';
        }
        if ($payment->category_id == 2) {
            $output['type'] = 'park';
        }
        $output['location_name'] = $locationName;
        $fees = $payment->fees;
        $output['vat'] = $fees['total_vat_amount'];
        $output['total_amount'] = $fees['payment_amount'];
        $output['administration_fee'] = $fees['administration_fee'];
        return collect($output);
    }


    /**
     * @param string $plate
     * @param int $park_id
     * @return object
     */
    public function listByPlateAndPark(string $plate, int $park_id): object
    {

        $list = [];
        $payments = $this->newQuery()
            ->where('amount', '>', 0)
            ->where('plate', $plate)
            ->where('park_id', $park_id)
            ->where('status_id', '!=', 2)
            ->whereNull('deleted_at')
            ->orderBy('created_at', 'DESC')
            ->each(function ($payment) use (&$list) {
                $output = [
                    'payment_id' => $payment->id,
                    'usage_type' => !empty($payment->company_id) ? 'business' : 'personal',
                    'category_id' => $payment->category_id,
                ];

                if (!empty($payment->charging_transaction)) {
                    $chargingPricingType = $payment->charging_transaction->station->pricing_type;
                    if (!$chargingPrice = $payment->charging_transaction->transaction_station->charge_price) {
                        $chargingPrice = 0;
                    }
                    $output['charging_price_type'] = $chargingPricingType;
                    $output['charging_session_cost'] = $payment->charging_transaction->payment_amount;
                    $output['charging_price'] = $chargingPrice;
                    $output['charging_electricity_consumption'] = $payment->charging_transaction->total_consumption;
                    $output['created_at'] = $payment->charging_transaction->created_at;
                    $output['ended_at'] = $payment->charging_transaction->ended_at;
                }
                if (!empty($payment->park_session)) {
                    $parkingCostCalculator = app(PricingCostCalculateHelperInterface::class);
                    //todo parking price should submit to the session table. realtime calculation is wrong
                    if ($payment->park_session->ended_at == null) {
                        $payment->park_session->ended_at = $payment->park_session->started_at;
                    }
                    $duration = strtotime($payment->park_session->ended_at) - strtotime($payment->park_session->started_at);
                    $costs = $parkingCostCalculator->calculate($payment->park_session, $duration);
                    $output['park_session_cost'] = $payment->park_session->payment_amount;
                    $output['park_session_status'] = $payment->park_session->session_status_id == 1 ? 'normal' : 'offense';
                    $output['parking_price'] = $costs['current_price'];
                    if ($duration) {
                        $output['parking_duration'] = CommonHelper::format_duration($duration);
                    } else {
                        $output['parking_duration'] = 0;
                    }
                    $output['created_at'] = $payment->park_session->created_at->format("Y-m-d H:i:s");
                    $output['ended_at'] = $payment->park_session->ended_at->format("Y-m-d H:i:s");
                    if ($payment->park_session->is_digital_code_active) {
                        $output['park_session_note'] = trans('visiosoft.module.payment.message.you_have_used_digital_code');
                        $output['digital_code'] = $payment->park_session->digital_code_text;
                        $output['park_session_cost'] = 0;
                        $output['parking_price'] = 0;
                    }
                }
                if ($payment->category_id == 4) {
                    $output['type'] = 'park_and_charge';
                }
                if ($payment->category_id == 1) {
                    $output['type'] = 'charge';
                }
                if ($payment->category_id == 2) {
                    $output['type'] = 'park';
                }
                $fees = $payment->fees;
                $output['vat'] = $fees['total_vat_amount'];
                $output['total_amount'] = $fees['payment_amount'];
                $output['administration_fee'] = $fees['administration_fee'];

                $list[] = $output;

            }, 100);

        return collect($list);
    }

    /**
     * TODO: Now it only returns amount according to usage needs, for usages in the future, return items from items table(if it does not exist, create it).
     * @param $amount
     * @return array[]
     */
    public function getPaymentItems($amount): array
    {
        return [
            [
                "name" => trans('visiosoft.module.payment::field.charging_service'),
                "price" => $amount,
                "quantity" => 1,
                "description" => trans('visiosoft.module.payment::field.charging_service')
            ]
        ];
    }

    public function doPay($paymentID)
    {
        $payment = $this->newQuery()->find($paymentID);
        if (!$payment) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => trans('visiosoft.module.payment::field.payment')]));
        }

        $payment_method = $this->paymentMethodRepository->getDefaultCardByOwner($payment->user_id);
        if (!$payment_method) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => trans('visiosoft.module.payment::field.card')]));
        };

        $providerUserID = $payment_method->customer;
        $defaultCardID = $payment_method->service_payment_method_id;
        $price = $payment->payment_amount;
        $currency = $this->currency;
        $items = $this->getPaymentItems($price);
        $invoice_id = $payment->id;

        try {
            $paymentRequest = $this->pay_provider->payBySavedCard($providerUserID, $defaultCardID, $price, $currency, $items, [], $invoice_id);
            if ($paymentRequest['data']['status'] == "success") {
                $payment->update([
                    'status_id' => 2,
                    'payment_intent_id' => $paymentRequest['data']['payment_intent_id']
                ]);
            } else if ($paymentRequest['data']['status'] == "fail") {
                $payment_method->update([
                    'status_id' => 3
                ]);
            }
        } catch (\Exception $e) {
            throw new \Exception(trans('visiosoft.module.payment::message.error_payment'));
        }

        return $paymentRequest;
    }

    public function automaticWithdrawal($paymentID)
    {
        $automaticWithdrawal = setting_value('visiosoft.module.payment::automatic_withdrawal');
        if ($automaticWithdrawal) {
            event(new PaymentCreatedEvent($paymentID));
        }
    }
}
