<?php namespace Visiosoft\PaymentModule\Http\Controller;

use Anomaly\Streams\Platform\Http\Controller\PublicController;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Visiosoft\ChargingModule\Transaction\Contract\TransactionRepositoryInterface;
use Visiosoft\ConnectModule\Traits\ApiReturnResponseTrait;
use Visiosoft\CurrencyModule\CurrencyModule;
use Visiosoft\ParkingModule\Park\Contract\ParkRepositoryInterface;
use Visiosoft\ParkingModule\Park\Contract\PricingCostCalculateHelperInterface;
use Visiosoft\ParkingModule\ParkingSession\Contract\ParkingSessionRepositoryInterface;
use Visiosoft\PaymentModule\Event\InvoiceDownloadEvent;
use Visiosoft\PaymentModule\Helper\CommonHelper;
use Visiosoft\PaymentModule\Helper\InvoiceHelper;
use Visiosoft\PaymentModule\Payment\Contract\PaymentRepositoryInterface;
use Visiosoft\PaymentModule\PaymentMethod\PaymentMethodRepository;
use Visiosoft\PaymentModule\PaymentModule;
use Visiosoft\PaymentModule\Status\Contract\StatusRepositoryInterface;
use Visiosoft\PaymentStripeExtension\PaymentStripeExtension;
use Visiosoft\StationModule\Station\Contract\StationRepositoryInterface;

class PaymentController extends PublicController
{
    use ApiReturnResponseTrait;

    protected ParkingSessionRepositoryInterface $parkingSessionRepository;
    protected PaymentRepositoryInterface $paymentRepository;
    protected PricingCostCalculateHelperInterface $pricingCostCalculateHelper;
    protected ParkRepositoryInterface $parkRepository;
    protected TransactionRepositoryInterface $transactionRepository;
    protected StationRepositoryInterface $stationRepository;

    public function __construct(
        ParkingSessionRepositoryInterface   $parkingSessionRepository,
        PaymentRepositoryInterface          $paymentRepository,
        PricingCostCalculateHelperInterface $pricingCostCalculateHelper,
        ParkRepositoryInterface             $parkRepository,
        TransactionRepositoryInterface      $transactionRepository,
        StationRepositoryInterface          $stationRepository
    )
    {
        $this->parkingSessionRepository = $parkingSessionRepository;
        $this->paymentRepository = $paymentRepository;
        $this->pricingCostCalculateHelper = $pricingCostCalculateHelper;
        $this->parkRepository = $parkRepository;
        $this->transactionRepository = $transactionRepository;
        $this->stationRepository = $stationRepository;
        parent::__construct();
    }

    public function successPage(Request $request)
    {
        $inputs = $request->all();
        $payment_intent_id = "";

        if (!empty($inputs['session_id'])) {
            $sessionId = $inputs['session_id'];
            $paymentStripeExtension = app(PaymentStripeExtension::class);
            $stripeSuccessPageData = $paymentStripeExtension->successPageSession($sessionId);
            $payment_intent_id = $stripeSuccessPageData['session']['payment_intent'];
            $customerName = $stripeSuccessPageData['session']['customer_details']['name'];
            $customerEmail = $stripeSuccessPageData['session']['customer_details']['email'];
        }

        if (!empty($inputs['payment_id'])) {
            $payment_intent_id = $inputs['payment_id'];
            $customerName = "Unregistered";
            $customerEmail = "User";
        }

        $payment = $this->paymentRepository->newQuery()
            ->where('stripe_intent_id', $payment_intent_id)
            ->orWhere('payment_intent_id', $payment_intent_id)
            ->first();

        $paymentModule = app(PaymentModule::class);

        if (!empty($payment->invoice_file)) {
            $invoiceUrl = $paymentModule->retrieveInvocieUrl($payment);
        } else {
            $invoiceUrl = $paymentModule->createInvoice($payment, ['customer_email' => $customerEmail, 'customer_name' => $customerName]);
        }


        $data = [
            'currency' => CurrencyModule::getDefault()->symbol,
            'payment' => $payment,
            'customer_name' => $customerName,
            'customer_email' => $customerEmail,
            'invoice_download_url' => $invoiceUrl,
            'device_platform' => CommonHelper::getDevicePlatform($request->userAgent())
        ];

        if (!empty($payment->station_id)) {
            $session = $this->transactionRepository->newQuery()->where('payment_id', $payment->id)->first();
            $station = $this->stationRepository->find($session->station_id);
            $data['station'] = $station;
            $data['type'] = "charge";
        }

        if (!empty($payment->park_id)) {
            $session = $this->parkingSessionRepository->newQuery()->where('id', $payment->park_session_id)->first();
            $park = $this->parkRepository->find($session->park_id);
            $data['parking'] = $park;
            $data['type'] = "park";
        }
        $data['session'] = $session;

        if (!$payment) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => trans('visiosoft.module.payment::field.payment')]));
        }
        if (!empty($inputs['session_id']) && $stripeSuccessPageData['session']['payment_status'] == 'paid') {
            $payment->update(['status_id' => 2]);
        }


        return $this->view->make('visiosoft.module.payment::success-page', $data);
    }

    public function cancelPage(Request $request)
    {
        $inputs = $request->all();
        $sessionId = $inputs['session_id'];

        $paymentStripeExtension = app(PaymentStripeExtension::class);

        $cancelData = $paymentStripeExtension->cancelPageSession($sessionId);
        $payment = $this->paymentRepository->newQuery()
            ->where('payment_intent_id', $cancelData['session']['payment_intent'])
            ->orWhere('stripe_intent_id', $cancelData['session']['payment_intent'])
            ->first();
        $view_params = [
            'currency' => CurrencyModule::getDefault()->symbol,
            'device_platform' => CommonHelper::getDevicePlatform($request->userAgent())
        ];
        if (!empty($payment->station_id)) {
            $session = $this->transactionRepository->newQuery()->where('payment_id', $payment->id)->first();
            $station = $this->stationRepository->find($session->station_id);
            $view_params['station'] = $station;
            $view_params['station_id'] = $session->station_id;
            $view_params['type'] = "charge";
        }

        if (!empty($payment->park_id)) {
            $session = $this->parkingSessionRepository->newQuery()->where('id', $payment->park_session_id)->first();
            $park = $this->parkRepository->find($session->park_id);
            $view_params['parking'] = $park;
            $view_params['park_id'] = $session->park_id;
            $view_params['type'] = "park";
        }

        if (!$session) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => trans('visiosoft.module.payment::field.park_session.name')]));
        }
        $view_params['session'] = $session;


        $this->paymentRepository->newQuery()->where('id', $payment->id)->delete();
        $session->update(['ended_at' => null]);


        return $this->view->make('visiosoft.module.payment::cancel-page', $view_params);
    }

    public function invoice($stripePaymentIntent)
    {
        $payment = $this->paymentRepository->newQuery()->where('stripe_intent_id', $stripePaymentIntent)->where('status_id', 2)->first();
        if (!$payment) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => trans('visiosoft.module.payment::field.payment')]));
        }
        $invoiceUrl = PaymentModule::retrieveInvocieUrl($payment);
        return $this->sendResponse($invoiceUrl);
    }

    public function invoiceEmail(Request $request, $stripePaymentIntent)
    {
        $inputs = $request->all();
        $customerName = $inputs['customer_name'];
        $email = $inputs['customer_email'];
        $link = config('app.url') . '/invoice/download/' . $stripePaymentIntent;
        event(new InvoiceDownloadEvent($customerName, $email, $link));
        return $this->sendResponse([], trans('visiosoft.module.payment::message.email_sent'));
    }

    public function checkPendingPayment()
    {
        $control_range = setting_value('visiosoft.module.payment::payment_control_range');
        $query_time = Carbon::now()->subMinute($control_range);
        $statusRepository = app(StatusRepositoryInterface::class);
        $status_id = $statusRepository->findBySlug('unpaid')->getId();

        $payments = $this->paymentRepository->newQuery()
            ->where('created_at', '<', $query_time->toDateTimeString())
            ->whereHas('status', function ($q) {
                $q->whereIn('slug', ['pending']);
            })->each(function ($payment) use ($status_id) {
                $payment->status = $status_id;
                $payment->save();
            }, 1000);
    }


    /**
     * @deprecated
     */
    public function checkBalanceOverlimit()
    {
        $default_user = setting_value('visiosoft.module.vehicle::default_customer');
        $paymentMethodRepository = app(PaymentMethodRepository::class);
        $statusRepository = app(StatusRepositoryInterface::class);
        $status_id = $statusRepository->findBySlug('paid')->getId();
        $balance_limit = setting_value('visiosoft.module.payment::balance_limit');
        $overlimits = $this->paymentRepository->newQuery()
            ->selectRaw("sum(amount) as amount, user_id as user_id,group_concat(id) as ids,currency")
            ->where('user_id', '!=', $default_user)
            ->havingRaw('(SUM(amount)/100) >= ?', [$balance_limit])
            ->groupBy(['user_id'])
            ->whereHas('status', function ($q) {
                $q->whereIn('slug', ['unpaid']);
            })->each(function ($record) use ($status_id, $paymentMethodRepository) {
                $payment_ids = explode(",", $record->ids);

                $payment_method = $paymentMethodRepository->newQuery()
                    ->where('owner_id', $record->user_id)
                    ->where('default', 1)
                    ->whereNull('deleted_at')
                    ->first();

                if ($payment_method) {
                    $payment_params =
                        [
                            'currency' => $record->currency,
                            'amount' => $record->amount * 100,
                            'payment_method' => $payment_method->service_payment_method_id,
                            'payment_method_type' => 'card',
                            'customer' => $payment_method->customer
                        ];

                    $create_payment = PaymentStripeExtension::createPaymentIntent($payment_params);
                    if ($create_payment && is_array($create_payment)) {
                        $confirm_payment = PaymentStripeExtension::confirmPaymentIntent(['stripe_intent_id' => $create_payment['id']]);
                        if (isset($confirm_payment['status']) && $confirm_payment['status'] == "succeeded") {
                            foreach ($payment_ids as $payment_id) {
                                $payment = $this->paymentRepository->find($payment_id);
                                $payment->payment_method_id = $payment_method->id;
                                $payment->stripe_intent_id = $create_payment['id'];
                                $payment->status = $status_id;
                                $payment->save();
                            }
                        }
                    }

                }
            }, 1000);
    }

    public function getVehicle($plate)
    {
        $invoiceHelper = app(InvoiceHelper::class);
        return $invoiceHelper->getVehicleFromBillupgifter($plate);
    }

    public function listByPlateAndPark($plate, $park_id): object
    {
        return $this->paymentRepository->listByPlateAndPark($plate, $park_id);
    }

}