<?php namespace Visiosoft\PaymentModule\Payment;

use Illuminate\Foundation\Bus\DispatchesJobs;
use Illuminate\Support\Facades\Auth;
use Visiosoft\ConnectModule\Command\CheckRequiredParams;
use Visiosoft\PaymentModule\Category\Contract\CategoryRepositoryInterface;
use Visiosoft\PaymentModule\PaymentMethod\Contract\PaymentMethodRepositoryInterface;
use Visiosoft\PaymentModule\Status\Contract\StatusRepositoryInterface;
use Visiosoft\PaymentStripeExtension\PaymentStripeExtension;

class PaymentApiCollection extends PaymentRepository
{
    use DispatchesJobs;

    public function createCard(array $params)
    {
        $userId = Auth::id();
        if (!$create_card = PaymentStripeExtension::createCard($params)) {
            throw new \Exception(trans('visiosoft.module.payment::message.cant_add_card'));
        }
        $create_card = $create_card->toArray();
        if (!empty($create_card['id'])) {
            $create_card['service_payment_method_id'] = $create_card['id'];
            $create_card['service'] = 'stripe';
        }
        unset($create_card['id']);
        unset($create_card['object']);
        $create_card['owner_id'] = $userId;
        $create_card['holder_name'] = $params['name'];
        $paymentMethodRepository = app(PaymentMethodRepositoryInterface::class);
        return $paymentMethodRepository->addPaymentMethod($create_card);
    }

    public function cards()
    {
        $cards = PaymentStripeExtension::cards();
        $paymentMethodIds = [];
        if (!empty($cards)) {
            foreach ($cards as $card) {
                $paymentMethodIds[$card['id']] = $card['id'];
            }
        }
        $results = [];
        $paymentMethodRepository = app(PaymentMethodRepositoryInterface::class);
        $paymentMethods = $paymentMethodRepository->newQuery()
            ->where('owner_id', Auth::id())->whereNull('deleted_at')->get();
        foreach ($paymentMethods as $paymentMethod) {
            if (!empty($paymentMethodIds[$paymentMethod->service_payment_method_id])) {
                $results[] = $paymentMethod;
                unset($paymentMethodIds[$paymentMethod->service_payment_method_id]);
            } else {
                $paymentMethod->delete();
            }
        }
        return $paymentMethodRepository->newQuery()
            ->where('owner_id', Auth::id())->whereNull('deleted_at')->get();
    }

    public function createDraftPayment(array $params)
    {
        $paymentData = [
            'user_id' => $params['user_id'],
            'amount' => $params['amount'],
            'usage' => $params['usage'],
            'status' => 1,
            'category_id' => $params['category_id'],
            'currency' => "SEK"
        ];
        if (!empty($params['transaction_id'])) {
            $paymentData['transaction_id'] = $params['transaction_id'];
        }
        if (!empty($params['park_session_id'])) {
            $paymentData['park_session_id'] = $params['park_session_id'];
        }
        $create = $this->newQuery()->create($paymentData);

        if (!$create) {
            throw new \Exception(trans('visiosoft.module.payment::message.transaction_create_failed'), 400);
        }
        return $create;
    }


    public function createPayment(array $params)
    {

        $payment = $this->newQuery()->where('transaction_id', $params['transaction_id'])->first();

        $payment_params = ['currency' => "SEK", 'amount' => $payment['amount'], 'method_id' => $params['method_id']];

        $create_payment = PaymentStripeExtension::createPaymentIntent($payment_params);
        $payment->update(['payment_intent_id' => $create_payment['id']]);
        return $this->confirmPayment(['payment_id' => $create_payment['id']]);
    }

    public function list(array $params)
    {
        $userId = Auth::id();
        $payments = $this->newQuery()
            ->where('user_id', $userId)
            ->orderBy('created_at', 'DESC');
        if (!empty($params['category_id'])) {
            $payments = $payments->where('category_id', $params['category_id']);
        }
        if (!empty($params['status_id'])) {
            $payments = $payments->where('status_id', $params['status_id']);
        }
        if (!empty($params['start_date'])) {
            $payments = $payments->where('created_at', '>=', $params['start_date']);
        }
        if (!empty($params['end_date'])) {
            $payments = $payments->where('created_at', '<=', $params['end_date']);
        }
        return $payments->get();
    }

    public function references()
    {
        $categoryRepository = app(CategoryRepositoryInterface::class);
        $statusRepository = app(StatusRepositoryInterface::class);
        return collect([
            [
                'statuses' => $statusRepository->newQuery()->get(),
                'category' => $categoryRepository->newQuery()->get()
            ]
        ]);
    }

    public function removeCard(array $params)
    {
        $this->dispatch(new CheckRequiredParams(['card_id'], $params));
        $paymentMethodRepository = app(PaymentMethodRepositoryInterface::class);
        $user = Auth::user();
        if (!$paymentMethod = $paymentMethodRepository->newQuery('owner_id', $user->id)->find($params['card_id'])) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => 'Payment Method']));
        }
        return $detached = PaymentStripeExtension::detachPaymentMethod($paymentMethod->service_payment_method_id, $user->stripe_id);
    }

    public function createPaymentIntent(array $params)
    {
        $this->dispatch(new CheckRequiredParams(['payment_id'], $params));
        if (!$payment = $this->newQuery()->where('user_id', Auth::id())->find($params['payment_id'])) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => 'Payment']));
        }
        $user = Auth::user();
        $amount = $payment->amount * 100;
        $currency = $payment->currency;
        $description = $user->first_name . ' ' . $user->last_name;
        $createCustomer = PaymentStripeExtension::createCustomer($description, $user->email);
        $customer = $createCustomer->id;
        $user->stripe_id = $customer;
        $user->save();
        $paymentIntent = PaymentStripeExtension::createPaymentIntent(['amount' => $amount, 'currency' => $currency, 'customer' => $customer]);
        if (empty($paymentIntent->id)) {
            throw new \Exception($paymentIntent);
        }
        $payment->update(['stripe_intent_id' => $paymentIntent->id, 'stripe_create_intent' => json_encode($paymentIntent)]);
        return $paymentIntent;
    }

    public function confirmPayment(array $params)
    {
        $this->dispatch(new CheckRequiredParams(['payment_id'], $params));
        if (!$payment = $this->newQuery()->where('user_id', Auth::id())->find($params['payment_id'])) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => 'Payment']));
        }
        $confirmation = PaymentStripeExtension::confirmPaymentIntent(['stripe_intent_id' => $payment->stripe_intent_id]);
        if ($confirmation['status'] == "succeeded" && $confirmation['paid']) {
            $payment->update(['status_id' => 2, 'stripe_confirm_intent' => json_encode($confirmation)]);
        }
        return $confirmation;
    }

    public function pay(array $params)
    {
        //todo: payment simulation mode
        $this->dispatch(new CheckRequiredParams(['payment_id'], $params));
        if (!$payment = $this->newQuery()->where('user_id', Auth::id())->find($params['payment_id'])) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => 'Payment']));
        }
        $payment->update(['status_id' => 2]);
        return $payment;
    }

    public function setDefaultCard(array $params)
    {
        $this->dispatch(new CheckRequiredParams(['payment_method_id'], $params));
        $paymentMethodRepository = app(PaymentMethodRepositoryInterface::class);
        if (!$paymentMethod = $paymentMethodRepository->newQuery()->where('owner_id', Auth::id())->find($params['payment_method_id'])) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => 'Card']));
        }
        $paymentMethodRepository->newQuery()->where('owner_id', Auth::id())->update(['default' => 0]);
        $paymentMethod->update(['default' => 1]);
        return $paymentMethod;
    }

    public function getDefaultCard()
    {
        $paymentMethodRepository = app(PaymentMethodRepositoryInterface::class);
        $defaultCard = $paymentMethodRepository->newQuery()->where('owner_id', Auth::id())->where('default', 1)->first();
        return collect(['default_card' => $defaultCard]);
    }

    public function getPayment(array $params)
    {
        $this->dispatch(new CheckRequiredParams(['payment_id'], $params));
        if (!$payment = $this->newQuery()->where('user_id', Auth::id())->find($params['payment_id'])) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => 'Payment']));
        }
        $payment = $payment->toArray();
        if (!empty($payment['charging_transaction'])) {
            $transaction = $payment['charging_transaction'];
            $chargeTransaction = [];
            $duration = strtotime($transaction['ended_at']) - strtotime($transaction['started_at']);
            if ($duration < 60) {
                $duration = 60;
            }
            $consumedAmount = round($duration / 60, 0);
            $paymentAmount = $consumedAmount;
            $chargeTransaction['charging_cost'] = 1;
            $chargeTransaction['charging_cost_string'] = 1 . ' sek/min';
            $chargeTransaction['overstay_fee'] = 1;
            $chargeTransaction['payment_amount'] = $paymentAmount;
            $chargeTransaction['charging_duration'] = $duration;
            $chargeTransaction['overstay_fee_string'] = 1 . ' sek/min';
            $chargeTransaction['overstay_duration'] = 0;
            $chargeTransaction['overstay_cost_string'] = 0 . ' sek';
            $payment['details'] = $chargeTransaction;
        }
        return collect(['payment' => $payment]);
    }

    public function startPayment(array $params)
    {
        //todo: payment simulation mode
        $this->dispatch(new CheckRequiredParams(['payment_id'], $params));
        if (!$payment = $this->newQuery()->where('user_id', Auth::id())->find($params['payment_id'])) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => 'Payment']));
        }
        $payment->update(['status_id' => 3]);
        return $payment;
    }

    public function cancelPayment(array $params)
    {
        //todo: payment simulation mode
        $this->dispatch(new CheckRequiredParams(['payment_id'], $params));
        if (!$payment = $this->newQuery()->where('user_id', Auth::id())->find($params['payment_id'])) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => 'Payment']));
        }
        $payment->update(['status_id' => 4]);
        return $payment;
    }
}
