<?php namespace Visiosoft\PaymentModule\Helper;

use Anomaly\FilesModule\Folder\Contract\FolderRepositoryInterface;
use Anomaly\UsersModule\User\Contract\UserRepositoryInterface;
use League\Flysystem\MountManager;
use Visiosoft\BillupgifterExtension\BillupgifterExtension;
use Visiosoft\CurrencyModule\CurrencyModule;
use Visiosoft\InvoiceModule\InvoiceItem\Contract\InvoiceItemRepositoryInterface;
use Visiosoft\InvoiceModule\InvoiceUser\Contract\InvoiceUserRepositoryInterface;
use Visiosoft\PaymentModule\PaymentModule;
use Visiosoft\VehicleModule\Vehicle\VehicleWithoutAttributeModel;

class InvoiceHelper
{


    public function getUser($userId)
    {
        $userRepository = app(UserRepositoryInterface::class);
        $user = $userRepository->newQuery()->withTrashed()->select([
            'id',
            'phone',
            'ssn',
            'email',
            'first_name',
            'last_name'
        ])->find($userId);
        if (!$user) {
            return [];
        }
        return $user->toArray();
    }

    public function getDigitalCodeDetails($user, $company, $digitalCodeDetails, $fees, $parkingSessions = [], $chargingSessions = [])
    {
        $currency = app(CurrencyModule::class)->getDefault()->symbol;
        $vatPercent = $this->getVat();
        return [
            'user' => $user,
            'company' => $company,
            'code_usage_type' => $digitalCodeDetails['code_usage_type'],
            'invoice_records' => [],
            'code_type' => $digitalCodeDetails['code_type'],
            'product_slug' => $digitalCodeDetails['product_slug'],
            'code_string' => $digitalCodeDetails['code_string'],
            'parking_slot_count' => $digitalCodeDetails['parking_slot_count'],
            'parking_slot_price' => $digitalCodeDetails['parking_slot_price'],
            'charging_slot_count' => $digitalCodeDetails['charging_slot_count'],
            'charging_slot_price' => $digitalCodeDetails['charging_slot_price'],
            'parking_slot_cost' => $digitalCodeDetails['parking_slot_cost'],
            'charging_slot_cost' => $digitalCodeDetails['charging_slot_cost'],
            'park_sessions' => $parkingSessions,
            'charge_sessions' => $chargingSessions,
            'exclude_vat' => $fees['total_amount_included_vat'] - $fees['total_vat_amount'],
            'payment_amount' => $fees['payment_amount'],
            'vat_percent' => $vatPercent,
            'taxable' => $fees['total_amount_included_vat'],
            'vat_amount' => $fees['total_vat_amount'],
            'administration_fee' => $fees['administration_fee'],
            'administration_fee_vat_amount' => $fees['administration_fee_vat_amount'],
            'currency' => strtoupper($currency),
            'creditor_address' => null,
            'creditor_phone' => null,
            'creditor_email' => null,
            'debtor_address' => null,
            'debtor_phone' => null,
            'debtor_email' => null,
            'creditor_id' => null,
            'debtor_id' => null,
            'creditor_external_id' => null,
            'debtor_external_id' => null
        ];
    }

    public function getInvoiceDetails($user, $vehicle, $vehicleSessions, $plateImageUrls)
    {
        $vatPercent = $this->getVat();
        $taxable = $vehicleSessions['park_cost'] + $vehicleSessions['charge_cost'];
        $calculatedAdministrationFee = $this->calculateAdministrationFee($taxable);
        $administrationFee = $calculatedAdministrationFee['administration_fee'];
        $administrationFeeVatAmount = $calculatedAdministrationFee['administration_fee_tax'];
        $totalExcludeVat = round(($taxable / ((100 + $vatPercent) / 100)), 2);
        $vatAmount = $taxable - $totalExcludeVat;
        $vatInclude = $taxable;
        $paymentAmount = $vatInclude + $administrationFee;
        $currency = app(CurrencyModule::class)->getDefault()->symbol;
        return [
            'user' => $user,
            'vehicle' => $vehicle,
            'invoice_records' => [],
            'park_sessions' => $vehicleSessions['park'],
            'charge_sessions' => $vehicleSessions['charge'],
            'charge_overstay_cost' => $vehicleSessions['charge_overstay_cost'],
            'charge_penalty_cost' => $vehicleSessions['charge_overstay_cost'],
            'charge_cost' => $vehicleSessions['charge_cost'],
            'parking_overstay_cost' => $vehicleSessions['park_overstay_cost'],
            'parking_offense_cost' => $vehicleSessions['park_offense_cost'],
            'parking_penalty_cost' => $vehicleSessions['park_offense_cost'] + $vehicleSessions['park_overstay_cost'],
            'park_cost' => $vehicleSessions['park_cost'],
            'park_duration' => $vehicleSessions['park_duration'],
            'charge_consumption' => $vehicleSessions['charge_consumption'],
            'plate' => $vehicle['plate'],
            'exclude_vat' => $totalExcludeVat + $administrationFee,
            'plate_image_urls' => $plateImageUrls,
            'payment_amount' => $paymentAmount,
            'vat_percent' => $vatPercent,
            'taxable' => $taxable,
            'vat_amount' => $vatAmount,
            'administration_fee' => $administrationFee,
            'administration_fee_vat_amount' => $administrationFeeVatAmount,
            'currency' => strtoupper($currency),
            'creditor_address' => null,
            'creditor_phone' => null,
            'creditor_email' => null,
            'debtor_address' => null,
            'debtor_phone' => null,
            'debtor_email' => null,
            'creditor_id' => null,
            'debtor_id' => null,
            'creditor_external_id' => null,
            'debtor_external_id' => null
        ];
    }

    public function getVehicle($vehicleId, $plate)
    {
        $vehicle = VehicleWithoutAttributeModel::select([
            'id',
            'plate',
            'brand_text',
            'model_text'
        ])->find($vehicleId);
        if ($vehicle) {
            return $vehicle->toArray();
        }
        return ['id' => $vehicleId, 'plate' => $plate];
    }

    public function getVehicleSessions($payment)
    {
        $vehicleSessions = [];
        $vehicles = [];
        $plateImages = [];
        if (!empty($payment)) {
            $vehicleSessions[$payment['vehicle_id']] = [
                'park' => [],
                'charge' => [],
                'park_duration' => 0,
                'charge_consumption' => 0,
                'park_cost' => 0,
                'charge_cost' => 0,
                'charge_penalty_cost' => 0,
                'charge_overstay_cost' => 0,
                'park_penalty_cost' => 0,
                'park_offense_cost' => 0,
                'park_overstay_cost' => 0
            ];
            $vehicles[$payment['vehicle_id']] = $payment['plate'];
            if (!empty($payment['charging_transaction'])) {
                $vehicleSessions[$payment['vehicle_id']]['charge_consumption']
                    = $vehicleSessions[$payment['vehicle_id']]['charge_consumption']
                    + $payment['charging_transaction']['total_consumption'];

                $vehicleSessions[$payment['vehicle_id']]['charge_cost']
                    = $vehicleSessions[$payment['vehicle_id']]['charge_cost']
                    + $payment['charging_transaction']['payment_amount'];

                $vehicleSessions[$payment['vehicle_id']]['charge_overstay_cost']
                    = $vehicleSessions[$payment['vehicle_id']]['charge_overstay_cost']
                    + $payment['charging_transaction']['penalty_amount'];

                $vehicleSessions[$payment['vehicle_id']]['charge_penalty_cost']
                    = $vehicleSessions[$payment['vehicle_id']]['charge_penalty_cost']
                    + $payment['charging_transaction']['penalty_amount'];

                $vehicleSessions[$payment['vehicle_id']]['charge'][] = $payment['charging_transaction'];
            }

            if (!empty($payment['park_session'])) {
                if (!empty($payment['park_session']['plate_image'])) {
                    $plateImages[$payment['vehicle_id']][] = $payment['park_session']['plate_image'];
                }

                $vehicleSessions[$payment['vehicle_id']]['park_duration']
                    = $vehicleSessions[$payment['vehicle_id']]['park_duration']
                    + strtotime($payment['park_session']['ended_at'])
                    - strtotime($payment['park_session']['started_at']);

                $vehicleSessions[$payment['vehicle_id']]['park_cost']
                    = $vehicleSessions[$payment['vehicle_id']]['park_cost']
                    + $payment['park_session']['payment_amount'];

                $vehicleSessions[$payment['vehicle_id']]['park_offense_cost']
                    = $vehicleSessions[$payment['vehicle_id']]['park_offense_cost']
                    + $payment['park_session']['offense_amount'];

                $vehicleSessions[$payment['vehicle_id']]['park_overstay_cost']
                    = $vehicleSessions[$payment['vehicle_id']]['park_overstay_cost']
                    + $payment['park_session']['overstay_amount'];

                $vehicleSessions[$payment['vehicle_id']]['park_penalty_cost']
                    = $vehicleSessions[$payment['vehicle_id']]['park_penalty_cost']
                    + $payment['park_session']['offense_amount']
                    + $payment['park_session']['overstay_amount'];

                $vehicleSessions[$payment['vehicle_id']]['park'][] = $payment['park_session'];
            }
        }
        return ['sessions' => $vehicleSessions, 'vehicles' => $vehicles, 'plate_images' => $plateImages];
    }

    public function calculateAdministrationFee($amount)
    {
        $administrationFee = $this->getAdministrationFee();
        $fees = app(PaymentModule::class)->getFees($amount);
        return [
            'administration_fee' => $fees['administration_fee'],
            'administration_fee_tax' => $fees['administration_fee_vat_amount']
        ];
    }

    public function calculateVAT($amount)
    {
        $vatPercent = $this->getVat();
        return $amount + ($vatPercent / 100) * $amount;
    }

    public function getVat()
    {
        return setting_value('visiosoft.module.invoice::vat');
    }

    public function getAdministrationFee(): array
    {
        return [
            'fee_amount' => setting_value('visiosoft.module.payment::administration_fee'),
            'fee_percent' => setting_value('visiosoft.module.payment::administration_fee_percent')
        ];
    }

    public function formattedInvoiceRecord(
        $sequencer,
        $type,
        $message = '',
        $amount = 0,
        $vat = 0,
        $paymentId = null,
        $invoiceItemId = null,
        $externalServiceId = null,
        $vatAmount = 0
    )
    {
        return [
            'sequence' => $sequencer,
            'type' => $type,
            'message' => $message,
            'amount' => $amount,
            'vat' => $vat,
            'payment_id' => $paymentId,
            'invoice_item_id' => $invoiceItemId,
            'external_service_id' => $externalServiceId,
            'vat_amount' => $vatAmount
        ];
    }

    public function generateDigitalCodeInvoiceRecords($details): array
    {
        $invoiceItemRepository = app(InvoiceItemRepositoryInterface::class);
        $records = [];
        $sequencer = -1;
        $creditorProducts = $invoiceItemRepository->newQuery()->where('invoice_user_id', $details['creditor_id'])->get();
        if (!$creditorProducts) {
            throw new \Exception(trans('visiosoft.module.invoice::message.invoice_products_not_defined'));
        }
        $product = $creditorProducts->where('slug', $details['product_slug'])->first();
        $sequencer = $sequencer + 1;
        $records[] = $this->formattedInvoiceRecord(
            $sequencer,
            'standard',
            trans('visiosoft.module.parking::invoice.digital_code', ['code_type' => $details['code_type']]),
            $details['taxable'],
            $details['vat_percent'],
            null,
            $product->id,
            $product->external_service_id
        );
        $records[] = $this->formattedInvoiceRecord(
            $sequencer,
            'message',
            trans('visiosoft.module.parking::invoice.code_usage_type', ['code_usage_type' => $details['code_usage_type']])
        );
        if (!empty($details['parking_slot_count'])) {
            $records[] = $this->formattedInvoiceRecord(
                $sequencer,
                'message',
                trans('visiosoft.module.parking::invoice.parking_slot_count', ['parking_slot_count' => $details['parking_slot_count']])
            );
            $records[] = $this->formattedInvoiceRecord(
                $sequencer,
                'message',
                trans('visiosoft.module.parking::invoice.parking_slot_price', ['parking_slot_price' => $details['parking_slot_price']])
            );
        }
        if (!empty($details['charging_slot_count'])) {
            $records[] = $this->formattedInvoiceRecord(
                $sequencer,
                'message',
                trans('visiosoft.module.parking::invoice.charging_slot_count', ['charging_slot_count' => $details['charging_slot_count']])
            );
            $records[] = $this->formattedInvoiceRecord(
                $sequencer,
                'message',
                trans('visiosoft.module.parking::invoice.charging_slot_price', ['charging_slot_price' => $details['charging_slot_price']])
            );
        }
        $records[] = $this->formattedInvoiceRecord(
            $sequencer,
            'message',
            trans('visiosoft.module.parking::invoice.code_string', ['code_string' => $details['code_string']])
        );
        $records[] = $this->formattedInvoiceRecord(
            $sequencer,
            'message',
            trans('visiosoft.module.parking::invoice.from', ['from_date' => $details['from']])
        );
        $records[] = $this->formattedInvoiceRecord(
            $sequencer,
            'message',
            trans('visiosoft.module.parking::invoice.to', ['to_date' => $details['to']])
        );
        $records[] = $this->formattedInvoiceRecord(
            $sequencer,
            'message',
            trans('visiosoft.module.parking::invoice.company', ['company' => $details['company']['name']])
        );

        $records[] = $this->formattedInvoiceRecord(
            $sequencer,
            'message',
            trans('visiosoft.module.parking::invoice.address',
                [
                    'address' => $details['company']['address'],
                    'zip_code' => $details['company']['zip_code']
                ])
        );
        $product = $creditorProducts->where('slug', 'administration_fee')->first();
        $vatPercent = $details['administration_fee_vat_amount'] > 0 ? (int)$details['vat_percent'] : 0;
        $records[] = $this->formattedInvoiceRecord(
            $sequencer,
            'administration_fee',
            trans('visiosoft.module.payment::invoice.administration_fee'),
            $details['administration_fee'],
            $vatPercent,
            null,
            $product->id,
            $product->external_service_id,
            $details['administration_fee_vat_amount'],
        );
        return $records;
    }

    public function digitalcodeAttachments($details): array
    {
        $manager = app(MountManager::class);
        $folders = app(FolderRepositoryInterface::class);
        $view = app('Illuminate\Contracts\View\Factory');
        $html = $view->make('visiosoft.module.parking::invoices.digital_code_sessions', [
            'sessions' => $details['park_sessions']
        ])->render();
        $attachments = [];
        $attachments[] = [
            'file' => [
                'content_type' => 'application/pdf',
                'data' => $html,
                'file_name' => 'digital-code-parking-sessions.pdf'
            ],
            'is_cover_sheet' => true,
            'sort_number' => 1
        ];
        return $attachments;
    }

    public function generateInvoiceRecords($details): array
    {
        $invoiceItemRepository = app(InvoiceItemRepositoryInterface::class);
        $records = [];
        $sequencer = -1;
        $creditorProducts = $invoiceItemRepository->newQuery()->where('invoice_user_id', $details['creditor_id'])->get();
        if (!$creditorProducts) {
            throw new \Exception(trans('visiosoft.module.invoice::message.invoice_products_not_defined'));
        }
        if (!empty($details['park_sessions'])) {
            foreach ($details['park_sessions'] as $session) {
                $amount = $session['payment_amount'];
                if ($amount > 0) {
                    $sequencer = $sequencer + 1;
                    $product = $creditorProducts->where('slug', 'parking_cost')->first();
                    $vatPercent = $product->is_vat_include ? (int)$details['vat_percent'] : 0;
                    $records[] = $this->formattedInvoiceRecord(
                        $sequencer,
                        'standard',
                        trans('visiosoft.module.payment::invoice.parking_session_transaction_number', ['transaction_number' => $session['payment_transaction_id']]),
                        $amount,
                        $vatPercent,
                        $session['payment_transaction_id'],
                        $product->id,
                        $product->external_service_id
                    );
                    $records[] = $this->formattedInvoiceRecord(
                        $sequencer,
                        'message',
                        trans('visiosoft.module.payment::invoice.started_at', ['started_at' => date('Y-m-d H:i', strtotime($session['started_at']))])
                    );
                    $records[] = $this->formattedInvoiceRecord(
                        $sequencer,
                        'message',
                        trans('visiosoft.module.payment::invoice.ended_at', ['ended_at' => date('Y-m-d H:i', strtotime($session['ended_at']))])
                    );
                    $records[] = $this->formattedInvoiceRecord(
                        $sequencer,
                        'message',
                        trans('visiosoft.module.payment::invoice.park_slot', ['slot' => $session['slot_text']])
                    );
                    $records[] = $this->formattedInvoiceRecord(
                        $sequencer,
                        'message',
                        trans('visiosoft.module.payment::invoice.location', ['location' => $session['park']['name'] . ' ' . $session['park']['address']]),
                    );
                    $records[] = $this->formattedInvoiceRecord(
                        $sequencer,
                        'message',
                        trans('visiosoft.module.payment::invoice.plate_number', ['plate' => $session['plate_number']])
                    );
                }
                if (!empty($session['penalty_amount'])) {
                    $sequencer = $sequencer + 1;
                    $product = $creditorProducts->where('slug', 'parking_offense')->first();
                    $vatPercent = $product->is_vat_include ? (int)$details['vat_percent'] : 0;
                    $records[] = $this->formattedInvoiceRecord(
                        $sequencer,
                        'standard',
                        trans('visiosoft.module.payment::invoice.parking_session_penalty_cost', ['transaction_number' => $session['payment_transaction_id']]),
                        $session['penalty_amount'],
                        $vatPercent,
                        $session['payment_transaction_id'],
                        $product->id,
                        $product->external_service_id
                    );
                    $records[] = $this->formattedInvoiceRecord(
                        $sequencer,
                        'message',
                        trans('visiosoft.module.payment::invoice.location', ['location' => $session['park']['name'] . ' ' . $session['park']['address']]),
                    );
                    $records[] = $this->formattedInvoiceRecord(
                        $sequencer,
                        'message',
                        trans('visiosoft.module.payment::invoice.park_slot', ['slot' => $session['slot_text']])
                    );
                    $records[] = $this->formattedInvoiceRecord(
                        $sequencer,
                        'message',
                        trans('visiosoft.module.payment::invoice.plate_number', ['plate' => $session['plate_number']]));
                }
            }
        }

        if (!empty($details['charge_sessions'])) {
            foreach ($details['charge_sessions'] as $session) {
                if (!empty($session['payment_amount'])) {
                    $sequencer = $sequencer + 1;
                    $product = $creditorProducts->where('slug', 'charging_cost')->first();
                    $vatPercent = $product->is_vat_include ? (int)$details['vat_percent'] : 0;
                    $records[] = $this->formattedInvoiceRecord(
                        $sequencer,
                        'standard',
                        trans('visiosoft.module.payment::invoice.ev_charging_transaction_no', ['transaction_number' => $session['payment_id']]),
                        $session['payment_amount'],
                        $vatPercent,
                        $session['payment_id'],
                        $product->id,
                        $product->external_service_id
                    );
                    $records[] = $this->formattedInvoiceRecord(
                        $sequencer,
                        'message',
                        trans('visiosoft.module.payment::invoice.electricity_consumption', ['consumption' => $session['total_consumption']])
                    );
                    $records[] = $this->formattedInvoiceRecord(
                        $sequencer,
                        'message',
                        trans('visiosoft.module.payment::invoice.started_at', ['started_at' => date('Y-m-d H:i', strtotime($session['started_at']))])
                    );
                    $records[] = $this->formattedInvoiceRecord(
                        $sequencer,
                        'message',
                        trans('visiosoft.module.payment::invoice.ended_at', ['ended_at' => date('Y-m-d H:i', strtotime($session['ended_at']))])
                    );
                    $records[] = $this->formattedInvoiceRecord(
                        $sequencer,
                        'message',
                        trans('visiosoft.module.payment::invoice.charger', ['charger' => $session['connection']['name']])
                    );
                    $records[] = $this->formattedInvoiceRecord(
                        $sequencer,
                        'message',
                        trans('visiosoft.module.payment::invoice.location', ['location' => $session['station']['name'] . ' ' . $session['station']['address']])
                    );
                    $records[] = $this->formattedInvoiceRecord(
                        $sequencer,
                        'message',
                        trans('visiosoft.module.payment::invoice.plate_number', ['plate' => $session['plate_number']])
                    );
                }
                if (!empty($session['penalty_amount'])) {
                    $sequencer = $sequencer + 1;
                    $product = $creditorProducts->where('slug', 'charging_overstay')->first();
                    $vatPercent = $product->is_vat_include ? (int)$details['vat_percent'] : 0;
                    $records[] = $this->formattedInvoiceRecord(
                        $sequencer,
                        'standard',
                        trans('visiosoft.module.payment::invoice.penalty_cost', ['transaction_number' => $session['payment_id']]),
                        $session['penalty_amount'],
                        $vatPercent,
                        $session['payment_id'],
                        $product->id,
                        $product->external_service_id
                    );
                }
            }
        }
        $sequencer = $sequencer + 1;
        $product = $creditorProducts->where('slug', 'administration_fee')->first();
        $vatPercent = $details['administration_fee_vat_amount'] > 0 ? (int)$details['vat_percent'] : 0;
        $records[] = $this->formattedInvoiceRecord(
            $sequencer,
            'administration_fee',
            trans('visiosoft.module.payment::invoice.administration_fee'),
            $details['administration_fee'],
            $vatPercent,
            null,
            $product->id,
            $product->external_service_id,
            $details['administration_fee_vat_amount']
        );
        return $records;
    }

    public function getUserRepository()
    {
        return app(UserRepositoryInterface::class);
    }

    public function getInvoiceUserRepository()
    {
        return app(InvoiceUserRepositoryInterface::class);
    }

    public function getUserDetail($userId)
    {
        $userRepository = $this->getUserRepository();
        $user = $userRepository->newQuery()->withTrashed()->find($userId);
        if (!$user) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => trans('visiosoft.module.payment::field.user.name')]), 404);
        }
        return $user;
    }

    public function getDebtor($userId, $personType, $creditorExternalId, $company = [])
    {
        $invoiceUserRepository = $this->getInvoiceUserRepository();
        $sender = $invoiceUserRepository->newQuery()->where('external_id', $creditorExternalId)->first();
        if (!$sender) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => trans('visiosoft.module.payment::field.invoice_sender.name')]), 404);
        }
        $invoiceReceiver = $invoiceUserRepository->getUser($userId, 'receiver', $personType, $sender->id);
        $user = $this->getUserDetail($userId);
        if (!$invoiceReceiver) {
            $params = [
                'creditor_id' => $creditorExternalId,
                'ssn' => $user->ssn,
                'name' => $user->name(),
                'address' => $user->extended_address,
                'city' => $user->city,
                'country_code' => $user->country_code,
                'vat_number' => $user->vat_number,
                'phone' => $user->phone,
                'email' => $user->email,
                'invoice_address' => [
                    'address' => $user->extended_address,
                    'zipcode' => $user->zip_code,
                    'city' => $user->city,
                    'country_code' => $user->country_code,
                    'email' => $user->email,
                    'delivery_method' => "Mail"
                ]
            ];
            if ($personType == 'business') {
                $params['name'] = $company['name'];
                $params['address'] = $company['address'];
                $params['invoice_address']['address'] = $company['address'];
                $params['invoice_address']['zipcode'] = $company['zip_code'];
            }
            $invoiceReceiver = $invoiceUserRepository->createUser($userId, 'receiver', $personType, $params, $sender->id);
        }
        return $invoiceReceiver;
    }

    public function getCreditor($userId, $personType)
    {
        $invoiceUserRepository = $this->getInvoiceUserRepository();
        $invoiceSender = $invoiceUserRepository->getUser($userId, 'sender', $personType);
        $user = $this->getUserDetail($userId);
        if (!$invoiceSender) {
            $params = [
                'organization_number' => $user->organization_number,
                'name' => $user->name(),
                'address' => $user->extended_address,
                'city' => $user->city,
                'country_code' => $user->country_code,
                'vat_number' => $user->vat_number,
                'invoice_address' => [
                    'address' => $user->extended_address,
                    'zipcode' => $user->zip_code,
                    'city' => $user->city,
                    'country_code' => $user->country_code,
                    'email' => $user->email,
                    'delivery_method' => "Email"
                ]
            ];
            $invoiceSender = $invoiceUserRepository->createUser($userId, 'sender', $personType, $params);
        }
        return $invoiceSender;
    }

    public function getVehicleFromBillupgifter($plate)
    {
        return BillupgifterExtension::getVehicle($plate);
    }
}