<?php namespace Visiosoft\PaymentModule;

use Anomaly\FilesModule\Folder\Contract\FolderRepositoryInterface;
use Anomaly\Streams\Platform\Addon\Module\Module;
use Anomaly\Streams\Platform\Image\Command\MakeImageInstance;
use Anomaly\Streams\Platform\Ui\Table\Component\View\View;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use League\Flysystem\MountManager;
use Visiosoft\ChargingModule\Transaction\Contract\TransactionRepositoryInterface;
use Visiosoft\ParkingModule\Park\Contract\PricingCostCalculateHelperInterface;
use Visiosoft\ParkingModule\ParkingSession\Contract\ParkingSessionRepositoryInterface;
use Visiosoft\PaymentModule\Job\AutoInvoiceJob;
use Visiosoft\PaymentModule\Payment\Contract\PaymentRepositoryInterface;
use Visiosoft\PaymentModule\Payment\PaymentModel;
use Visiosoft\PaymentModule\Payment\PaymentWithoutAttrModel;

class PaymentModule extends Module
{
    protected $root_menu = 'Drifter';

    protected $navigation = true;

    protected $icon = 'fa fa-puzzle-piece';

    protected $sections = [
        'payment',
        'payment_methods',
        'category' => [
            'buttons' => [
                'new_category',
            ],
        ],
        'status'
    ];

    public function createInvoice(PaymentModel $payment, $customer, $type = "park")
    {
        $invoiceUrl = '';
        if ($payment) {
            if (setting_value('visiosoft.module.payment::invoice')) {
                $manager = app(MountManager::class);
                $folders = app(FolderRepositoryInterface::class);
                $view = app('Illuminate\Contracts\View\Factory');
                $paymentRepo = app(PaymentRepositoryInterface::class);
                $pricingCostCalculateHelper = app(PricingCostCalculateHelperInterface::class);
                if ($type == "park") {
                    $startedAt = strtotime($payment->park_session->started_at);
                    $endedAt = strtotime($payment->park_session->ended_at);
                    if (!$payment->park_session->ended_at) {
                        $endedAt = time();
                    }
                } else if ($type == "transaction") {
                    $startedAt = strtotime($payment->transaction->started_at);
                    $endedAt = strtotime($payment->transaction->ended_at);
                    if (!$payment->transaction->ended_at) {
                        $endedAt = time();
                    }
                }

                $paymentAmount = $payment->amount;
                $duration = $endedAt - $startedAt;
                if ($type == "park") {
                    $parkingSessionRepository = app(ParkingSessionRepositoryInterface::class);
                    $cost = $pricingCostCalculateHelper->calculate($payment->park_session, $duration);
                    $details = $parkingSessionRepository->getDetail($paymentAmount, $cost, $duration);
                } else if ($type == "transaction") {
                    $transactionRepository = app(TransactionRepositoryInterface::class);
                    $details = $transactionRepository->getDetail($paymentAmount, $duration);
                }
                $html = $view->make('visiosoft.module.payment::invoice', [
                    'logo' => '',
                    'type' => $type,
                    'payment' => $payment,
                    'details' => $details,
                    'started_at' => date('Y-m-d H:i', $startedAt),
                    'ended_at' => date('Y-m-d H:i', $endedAt),
                    'customer_name' => $customer['customer_name'],
                    'customer_email' => $customer['customer_email'],
                    'payment_status' => $payment->status->name,
                    'source_of_collection' => ''
                ])->render();
                $pdf = Pdf::loadHTML($html)->output();
                $fileName = '/' . $type . '_session_' . $payment->id . '_' . date('Y-m-d_H-i') . '.pdf';
                $folder = $folders->findBySlug('invoices');
                $file = $manager->put($folder->path($fileName), $pdf);

                $paymnet2 = $paymentRepo->find($payment->id);
                $paymnet2->update(['invoice_file_id' => $file->id]);
                $invoiceUrl = config('app.url') . '/files/' . $file->path();
            }
        }
        return $invoiceUrl;
    }

    public function retrieveInvocieUrl(PaymentModel $payment)
    {
        return config('app.url') . '/files/' . $payment->invoiceFile()->first()->path();
    }

    public function invoice()
    {
        $limit = setting_value('visiosoft.module.payment::minimum_payable_amount');
        $payments = PaymentWithoutAttrModel::with([
            'category',
            'company',
            'parkSession',
            'chargingTransaction',
            'parkSession.slot',
            'parkSession.park' => function ($parking) {
                $parking->select(['id', 'address', 'phone', 'name', 'operator_id']);
            },
            'parkSession.park.operator' => function ($operator) {
                $operator->select(['id', 'title']);
            },
            'chargingTransaction.station' => function ($station) {
                $station->select(['id', 'title as name', 'operator_id', 'address_line_1 as address']);
            },
            'chargingTransaction.station.operator' => function ($operator) {
                $operator->select(['id', 'title']);
            },
            'chargingTransaction.connection'

        ])
            ->whereIn('status_id', [1, 4]) //payment statuses: 1 = unpaid 4 = failed
            ->whereNull('invoice_id')
            ->whereNull('company_id')
            ->where(function ($query) {
                $query->orWhereHas('chargingTransaction')
                    ->orWhereHas('parkSession');
            })
            ->limit(1000)
            ->get();
        if (!empty($payments)) {

            $userPayments = $payments->groupBy('user_id')->toArray();
            foreach ($userPayments as $userId => $userPayment) {
                $amount = 0;
                foreach ($userPayment as $item) {
                    $amount = $amount + $item['amount'];
                }
                if ($amount >= $limit) {
                    AutoInvoiceJob::dispatch($userId, $userPayment);
                }
            }
        }
    }

    public function getFees($includedVatAmount)
    {
        if ($includedVatAmount > 0) {
            $stripeAdministrationFeePercent = setting_value('visiosoft.module.payment::stripe_administration_fee_percent');
            $stripeAdministrationFee = setting_value('visiosoft.module.payment::stripe_administration_fee');
            $fixedAdministrationFee = setting_value('visiosoft.module.payment::fixed_administration_fee');
            $calculatedVatAmount = round($includedVatAmount * $stripeAdministrationFeePercent / 100 + $stripeAdministrationFee, 2);
            $administrationFeeRemain = $fixedAdministrationFee - $calculatedVatAmount;
            $vatPercent = setting_value('visiosoft.module.invoice::vat');
            $administrationFee = $fixedAdministrationFee;
            $administrationFeeVatAmount = 0;
            if ($administrationFeeRemain < 0) {
                $administrationFee = $calculatedVatAmount;
            } else {
                $administrationFeeVatAmount = round($administrationFeeRemain - $administrationFeeRemain / (1 + ($vatPercent / 100)), 2);
            }
            $vatAmount = round($includedVatAmount - $includedVatAmount / (1 + ($vatPercent / 100)), 2);
            $totalVatAmount = $vatAmount + $administrationFeeVatAmount;
            $totalPaymentAmount = $includedVatAmount + $administrationFee;
            return [
                'total_amount_included_vat' => round($includedVatAmount, 2),
                'total_vat_amount' => round($totalVatAmount, 2),
                'payment_amount' => round($totalPaymentAmount, 2),
                'administration_fee' => round($administrationFee, 2),
                'administration_fee_vat_amount' => round($administrationFeeVatAmount, 2)
            ];
        }
        return [
            'total_amount_included_vat' => 0,
            'total_vat_amount' => 0,
            'payment_amount' => 0,
            'administration_fee' => 0,
            'administration_fee_vat_amount' => 0
        ];
    }
}
