<?php

namespace Visiosoft\PaymentModule\Job;

use Carbon\Carbon;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Visiosoft\PaymentModule\Event\UnpaidPaymentEvent;
use Visiosoft\PaymentModule\Event\UnpaidPaymentReminderEvent;
use Visiosoft\PaymentModule\Payment\Contract\PaymentRepositoryInterface;

class UnpaidPaymentControlJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function handle(): void
    {
        $paymentRepository = app(PaymentRepositoryInterface::class);
        $unpaidPayments = $paymentRepository->getPayments(true, null, ['unpaid', 'failed']);

        //Notify once or Periodically
        $remindType = setting_value('visiosoft.module.payment::unpaid_payment_reminder_type');

        //Minimum time to be reminded of payment
        $unpaidReminderTime = setting_value('visiosoft.module.payment::unpaid_payment_reminder_time');

        //Remind Interval
        $notificationInterval = setting_value('visiosoft.module.payment::unpaid_payment_reminder_interval');

        $unpaidPayments
            ->where('amount', '>', 0)
            ->where('created_at', '<=', Carbon::now()->subMinutes((int)$unpaidReminderTime))
            ->where(function ($query) use ($notificationInterval, $remindType) {
                $query->whereNull('last_try');

                if ($remindType != "remind_once") {
                    $query->orWhere('last_try', '<=', Carbon::now()->subMinutes((int)$notificationInterval));
                }
            })
            ->orWhereNull('last_notify')
            ->chunk(10000, function ($unpaidPayments) use ($unpaidReminderTime, $notificationInterval) {
                $this->sendNotifications($unpaidPayments);
            });
    }

    private function sendNotifications($unpaidPayments): void
    {
        foreach ($unpaidPayments as $unpaidPayment) {
            event(new UnpaidPaymentReminderEvent($unpaidPayment));
        }
    }
}