<?php namespace Visiosoft\PaymentModule\Http\Controller;

use \Anomaly\Streams\Platform\Http\Controller\PublicController;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;
use Visiosoft\ConnectModule\Traits\ApiReturnResponseTrait;
use Visiosoft\ParkingModule\Park\Contract\PricingCostCalculateHelperInterface;
use Visiosoft\ParkingModule\ParkingSession\Contract\ParkingSessionRepositoryInterface;
use Visiosoft\PaymentModule\Event\InvoiceDownloadEvent;
use Visiosoft\PaymentModule\Payment\Contract\PaymentRepositoryInterface;
use Visiosoft\PaymentStripeExtension\PaymentStripeExtension;

class PaymentController extends PublicController
{
    use ApiReturnResponseTrait;

    protected ParkingSessionRepositoryInterface $parkingSessionRepository;
    protected PaymentRepositoryInterface $paymentRepository;
    protected PricingCostCalculateHelperInterface $pricingCostCalculateHelper;

    public function __construct(
        ParkingSessionRepositoryInterface   $parkingSessionRepository,
        PaymentRepositoryInterface          $paymentRepository,
        PricingCostCalculateHelperInterface $pricingCostCalculateHelper

    )
    {
        $this->parkingSessionRepository = $parkingSessionRepository;
        $this->paymentRepository = $paymentRepository;
        $this->pricingCostCalculateHelper = $pricingCostCalculateHelper;
        parent::__construct();
    }

    public function successPage(Request $request)
    {
        $inputs = $request->all();
        $sessionId = $inputs['session_id'];

        $stripeSuccessPageData = PaymentStripeExtension::successPageSession($sessionId);

        $session = $this->parkingSessionRepository->newQuery()->find($stripeSuccessPageData['session']['client_reference_id']);
        $payment = $this->paymentRepository->newQuery()->find($session->payment_transaction_id);
        if (!$payment) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => 'Payment']));
        }
        if ($stripeSuccessPageData['session']['payment_status'] == 'paid') {
            $payment->update(['status_id' => 2]);
        }
        $customerName = $stripeSuccessPageData['session']['customer_details']['name'];
        $customerEmail = $stripeSuccessPageData['session']['customer_details']['email'];

        return $this->view->make('visiosoft.module.payment::success-page', [
            'session' => $session,
            'payment' => $payment,
            'customer_name' => $customerName,
            'customer_email' => $customerEmail,
            'invoice_download_url' => ''
        ]);
    }

    public function cancelPage(Request $request)
    {
        $inputs = $request->all();
        $sessionId = $inputs['session_id'];
        $cancelData = PaymentStripeExtension::cancelPageSession($sessionId);
        $session = $this->parkingSessionRepository->newQuery()->where('id', $cancelData['session']['client_reference_id'])->first();
        if (!$session) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => 'Park Session']));
        }
        $payment = $this->paymentRepository->newQuery()->where('park_session_id', $session->id)->delete();

        $session->update(['ended_at' => null]);
        return $this->view->make('visiosoft.module.payment::cancel-page');
    }

    public function invoice(Request $request, $stripePaymentIntent)
    {
        $inputs = $request->all();
        $payment = $this->paymentRepository->newQuery()->where('stripe_intent_id', $stripePaymentIntent)->where('status_id', 2)->first();
        if (!$payment) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => 'Payment']));
        }
        $startedAt = strtotime($payment->park_session->started_at);
        $endedAt = strtotime($payment->park_session->ended_at);
        $prices = $payment->park_session->park->prices;
        $paymentAmount = $payment->amount;
        $duration = $endedAt - $startedAt;
        $cost = $this->pricingCostCalculateHelper->calculate($prices, $duration);
        $detais = $this->parkingSessionRepository->getDetail($paymentAmount, $cost, $duration);
        $html = $this->view->make('visiosoft.module.payment::invoice', [
            'payment' => $payment,
            'details' => $detais,
            'started_at' => date('Y-m-d H:i', $startedAt),
            'ended_at' => date('Y-m-d H:i', $endedAt),
            'customer_name' => $inputs['customer_name'],
            'customer_email' => $inputs['customer_email']
        ])->render();
        $pdf = Pdf::loadHTML($html);
        return $pdf->download('parking_session_invoice.pdf');
    }

    public function invoiceEmail(Request $request, $stripePaymentIntent)
    {
        $inputs = $request->all();
        $customerName = $inputs['customer_name'];
        $email = $inputs['customer_email'];
        $link = config('app.url') . '/invoice/download/' . $stripePaymentIntent . '?' . http_build_query(['customer_name' => $customerName, 'customer_email' => $email]);
        event(new InvoiceDownloadEvent($customerName, $email, $link));
        return $this->sendResponse([], trans('visiosoft.module.payment::message.email_sent'));
    }
}