<?php namespace Visiosoft\PaymentModule\Payment;

use Illuminate\Foundation\Bus\DispatchesJobs;
use Illuminate\Support\Facades\Auth;
use Visiosoft\ChargingModule\Transaction\Contract\TransactionRepositoryInterface;
use Visiosoft\ConnectModule\Command\CheckRequiredParams;
use Visiosoft\ParkingModule\Park\Contract\PricingCostCalculateHelperInterface;
use Visiosoft\ParkingModule\ParkingSession\Contract\ParkingSessionRepositoryInterface;
use Visiosoft\ParkingModule\Ticket\Contract\TicketRepositoryInterface;
use Visiosoft\ParkingModule\TicketUser\Contract\TicketUserRepositoryInterface;
use Visiosoft\PaymentModule\Category\Contract\CategoryRepositoryInterface;
use Visiosoft\PaymentModule\PaymentModule;
use Visiosoft\PaymentModule\Status\Contract\StatusRepositoryInterface;

class PaymentApiCollection extends PaymentRepository
{
    use DispatchesJobs;

    public function createCard(array $params)
    {
        $user = Auth::user();
        try {
            $cardUserKey = $this->paymentMethodRepository->getCustomerIdByService($user->id, $this->card_provider->getProviderSlug());
            $providerResponse = $this->card_provider->createCard(
                $params['name'],
                $params['card']['card_number'],
                $params['card']['expire_year'],
                $params['card']['expire_month'],
                $params['card']['cvc'],
                $user->email,
                $cardUserKey
            );

            if ($providerResponse['status']) {
                $createCard = [
                    'owner_id' => Auth::id(),
                    'service' => $providerResponse['data']['service'],
                    'service_payment_method_id' => $providerResponse['data']['service_payment_method_id'],
                    'customer' => $providerResponse['data']['customer'],
                    'card' => $providerResponse['data']['card'],
                    'holder_name' => $params['name']
                ];

                $this->paymentMethodRepository->addPaymentMethod($createCard);

            } else {
                throw new \Exception(trans('visiosoft.module.payment::message.cant_add_card'));
            }

        } catch (\Exception $e) {
            throw new \Exception(trans('visiosoft.module.payment::message.cant_add_card'));
        }
        return true;
    }

    public function cards()
    {
        return $this->paymentMethodRepository->getCardsByOwner(Auth::id(), ['id', 'card', 'default', 'holder_name', 'created_at']);
    }

    public function createDraftPayment(array $params)
    {
        $paymentData = [
            'user_id' => $params['user_id'],
            'amount' => $params['amount'],
            'usage' => $params['usage'],
            'status' => 1,
            'category_id' => $params['category_id'],
            'currency' => $this->currency,
        ];
        if (!empty($params['vehicle_id'])) {
            $paymentData['vehicle_id'] = $params['vehicle_id'];
        }
        if (!empty($params['plate'])) {
            $paymentData['plate'] = $params['plate'];
        }
        if (!empty($params['company_id'])) {
            $paymentData['company_id'] = $params['company_id'];
        }
        if (!empty($params['stripe_intent_id'])) {
            $paymentData['stripe_intent_id'] = $params['stripe_intent_id'];
        }
        if (!empty($params['transaction_id'])) {
            $paymentData['transaction_id'] = $params['transaction_id'];
        }
        if (!empty($params['park_session_id'])) {
            $paymentData['park_session_id'] = $params['park_session_id'];
        }
        if (!empty($params['park_id'])) {
            $paymentData['park_id'] = $params['park_id'];
        }
        if (!empty($params['station_id'])) {
            $paymentData['station_id'] = $params['station_id'];
        }
        if (!empty($params['ticket_id'])) {
            $paymentData['ticket_id'] = $params['ticket_id'];
        }
        $fees = app(PaymentModule::class)->getFees($paymentData['amount']);
        $paymentData['tax_percent'] = $fees['vat_percent'];
        $paymentData['payment_amount'] = $fees['payment_amount'];
        $paymentData['tax_amount'] = $fees['total_vat_amount'];
        $paymentData['additional_fees'] = $fees['administration_fee'];
        $create = $this->newQuery()->create($paymentData);

        if (!$create) {
            throw new \Exception(trans('visiosoft.module.payment::message.transaction_create_failed'), 400);
        }

        if ($create->payment_amount > 0) {
            $this->createPaymentInvoice($create->id);
            $this->automaticWithdrawal($create->id);
        }

        return $create;
    }

    public function listV2()
    {
        $userId = Auth::id();
        $payments = PaymentWithoutAttrModel::with(['category', 'status'])
            ->select(['id', 'category_id', 'status_id', 'currency', 'amount', 'created_at', 'company_id'])
            ->where('user_id', $userId)
            ->orderBy('created_at', 'DESC');
        if (!empty($params['category_id'])) {
            $payments = $payments->where('category_id', $params['category_id']);
        }
        if (!empty($params['status_id'])) {
            $payments = $payments->where('status_id', $params['status_id']);
        }
        if (!empty($params['start_date'])) {
            $payments = $payments->where('created_at', '>=', $params['start_date']);
        }
        if (!empty($params['end_date'])) {
            $payments = $payments->where('created_at', '<=', $params['end_date']);
        }
        if (!empty($params['account_type_id'])) {
            if ($params['account_type_id'] == 1) {
                $payments = $payments->whereNull('company_id');
            }
            if ($params['account_type_id'] == 2) {
                $payments = $payments->whereNotNull('company_id');
            }
        }
        return $payments;
    }

    /**
     * @deprecated use listV2
     */
    public function list(array $params)
    {
        $userId = Auth::id();
        $payments = $this->newQuery()
            ->where('user_id', $userId)
            ->orderBy('created_at', 'DESC');
        if (!empty($params['category_id'])) {
            $payments = $payments->where('category_id', $params['category_id']);
        }
        if (!empty($params['status_id'])) {
            $payments = $payments->where('status_id', $params['status_id']);
        }
        if (!empty($params['start_date'])) {
            $payments = $payments->where('created_at', '>=', $params['start_date']);
        }
        if (!empty($params['end_date'])) {
            $payments = $payments->where('created_at', '<=', $params['end_date']);
        }
        if (!empty($params['account_type_id'])) {
            if ($params['account_type_id'] == 1) {
                $payments = $payments->whereNull('company_id');
            }
            if ($params['account_type_id'] == 2) {
                $payments = $payments->whereNotNull('company_id');
            }
        }
        return $payments;
    }

    public function references()
    {
        $categoryRepository = app(CategoryRepositoryInterface::class);
        $statusRepository = app(StatusRepositoryInterface::class);
        return collect([
            [
                'statuses' => $statusRepository->newQuery()->get(),
                'category' => $categoryRepository->newQuery()->get(),
                'account_types' => [['name' => 'Personal', 'id' => 1], ['name' => 'Business', 'id' => 2]]
            ]
        ]);
    }

    public function removeCard(array $params)
    {
        $this->dispatch(new CheckRequiredParams(['card_id'], $params));
        if (!$paymentMethod = $this->paymentMethodRepository->find($params['card_id'])) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => trans('visiosoft.module.payment::field.payment_method.name')]));
        }

        if ($paymentMethod->delete()) {
            $deleted_card = $this->card_provider->removeCard($paymentMethod->service_payment_method_id);
            if (!$deleted_card) {
                throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => trans('visiosoft.module.payment::field.card')]));
            }
        }
        return true;
    }

    public function pay(array $params)
    {
        return $this->doPay($params['payment_id']);
    }

    public function setDefaultCard(array $params)
    {
        $this->dispatch(new CheckRequiredParams(['payment_method_id'], $params));
        if (!$paymentMethod = $this->paymentMethodRepository->find($params['payment_method_id'])) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => trans('visiosoft.module.payment::field.card')]));
        }
        $this->paymentMethodRepository->newQuery()->where('owner_id', Auth::id())->update(['default' => 0]);

        if ($paymentMethod->update(['default' => 1])) {
            return true;
        };
        return false;
    }

    public function getDefaultCard()
    {
        return $this->paymentMethodRepository->getDefaultCardByOwner(Auth::id(), ['id', 'card', 'default', 'holder_name', 'created_at']);
    }

    public function getPaymentV2(array $params)
    {
        return $this->getDetail($params);
    }

    public function getPayment(array $params)
    {
        $this->dispatch(new CheckRequiredParams(['payment_id'], $params));
        $payment = $this->newQuery()->where('user_id', Auth::id())->find($params['payment_id']);
        if (!$payment) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => trans('visiosoft.module.payment::field.payment')]));
        }
        $paymentCollection = $payment;
        $payment = $payment->toArray();
        $details = [];
        if (!empty($payment['charging_transaction'])) {
            $transaction = $payment['charging_transaction'];
            $duration = strtotime($transaction['ended_at']) - strtotime($transaction['started_at']);
            if ($duration < 60) {
                $duration = 60;
            }
            $consumedAmount = round($duration / 60, 0);
            $paymentAmount = $consumedAmount;
            $transactionRepository = app(TransactionRepositoryInterface::class);
            $details = $transactionRepository->getDetail($paymentAmount, $duration);
        }
        if (!empty($paymentCollection->park_session)) {
            $session = $paymentCollection->park_session;
            $duration = strtotime($session->ended_at) - strtotime($session->started_at);
            if ($duration < 60) {
                $duration = 60;
            }
            $calculator = app(PricingCostCalculateHelperInterface::class);
            $parkSessionRepository = app(ParkingSessionRepositoryInterface::class);
            $cost = $calculator->calculate($session, $duration);
            $details = $parkSessionRepository->getDetail($payment['amount'], $cost, $duration);
        }
        return collect(['payment' => $payment, 'details' => $details]);
    }

    /**
     * @deprecated delete after check usages.
     */
    public function startPayment(array $params)
    {
        //todo: payment simulation mode
        $this->dispatch(new CheckRequiredParams(['payment_id'], $params));
        if (!$payment = $this->newQuery()->where('user_id', Auth::id())->find($params['payment_id'])) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => trans('visiosoft.module.payment::field.payment')]));
        }
        $payment->update(['status_id' => 3]);
        return $payment;
    }

    /**
     * @deprecated delete after check usages.
     */
    public function cancelPayment(array $params)
    {
        //todo: payment simulation mode
        $this->dispatch(new CheckRequiredParams(['payment_id'], $params));
        if (!$payment = $this->newQuery()->where('user_id', Auth::id())->find($params['payment_id'])) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => trans('visiosoft.module.payment::field.payment')]));
        }
        $payment->update(['status_id' => 4]);
        return $payment;
    }

    public function buyTicket(array $params)
    {
        $this->dispatch(new CheckRequiredParams(
            [
                'park_id',
                'ticket_id',
                'plate_id'
            ]
            , $params));


        $user_id = Auth::id();
        $park_id = $params['park_id'];
        $ticket_id = $params['ticket_id'];
        $plate_id = $params['plate_id'];

        $ticketRepository = app(TicketRepositoryInterface::class);
        $categoryRepository = app(CategoryRepositoryInterface::class);
        $category = $categoryRepository->findBy('slug', 'long_term_ticket');
        $ticket = $ticketRepository->find($ticket_id);


        $paymentData = [
            'user_id' => $user_id,
            'amount' => $ticket->price,
            'park_id' => $park_id,
            'ticket_id' => $ticket->id,
            'usage' => '',
            'category_id' => $category->getId(),
            'currency' => $this->currency
        ];

        if ($payment = $this->createDraftPayment($paymentData)) {
            $pay = $this->pay(['payment_id' => $payment->id]);
            if (isset($pay['status']) && $pay['status'] == "succeeded") {
                $ticketUserRepository = app(TicketUserRepositoryInterface::class);
                $started_at = date("Y-m-d H:i:s");
                $ended_at = date('Y-m-d H:i:s', strtotime($started_at . ' + ' . $ticket->time . " " . $ticket->time_unit . ''));

                $createParams = [
                    'plate_id' => $plate_id,
                    'user_id' => $user_id,
                    'ticket_id' => $ticket->getId(),
                    'started_at' => $started_at,
                    'ended_at' => $ended_at,
                    'payment_id' => $payment->id
                ];

                $ticketUserRepository->create($createParams);
                return $pay;
            }
        }
    }
}
