<?php namespace Visiosoft\PaymentModule\Payment;

use Anomaly\Streams\Platform\Entry\EntryRepository;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Visiosoft\ConnectModule\Command\CheckRequiredParams;
use Visiosoft\PaymentModule\Category\Contract\CategoryRepositoryInterface;
use Visiosoft\PaymentModule\Event\CreateInvoiceEvent;
use Visiosoft\PaymentModule\Event\PaymentCreatedEvent;
use Visiosoft\PaymentModule\Event\PaymentReceiptEvent;
use Visiosoft\PaymentModule\Payment\Commands\Card;
use Visiosoft\PaymentModule\Payment\Commands\Pay;
use Visiosoft\PaymentModule\Payment\Contract\PaymentRepositoryInterface;
use Visiosoft\PaymentModule\PaymentMethod\Contract\PaymentMethodRepositoryInterface;
use Visiosoft\PaymentModule\PaymentModule;
use Visiosoft\PaymentModule\Status\Contract\StatusRepositoryInterface;
use Visiosoft\ProductsModule\Support\Command\Currency;

class PaymentRepository extends EntryRepository implements PaymentRepositoryInterface
{

    public string $currency;
    protected $model;
    protected PaymentMethodRepositoryInterface $paymentMethodRepository;
    protected Card $card_provider;
    protected Pay $pay_provider;

    public function __construct(PaymentModel $model, PaymentMethodRepositoryInterface $paymentMethodRepository)
    {
        $this->model = $model;
        $this->currency = (new Currency())->symbol();
        $this->card_provider = new Card();
        $this->pay_provider = new Pay();
        $this->paymentMethodRepository = $paymentMethodRepository;
    }


    /**
     * @param array $params (['payment_id' => int])
     * @return object
     * @throws Exception
     */
    public function getDetail(array $params): object
    {
        $output = [
            'payment_id' => null,
            'location_name' => null,
            'location_id' => null,
            'created_at' => null,
            'ended_at' => null,
            'plate' => null,
            'usage_type' => null,
            'type' => '',
            'category_id' => null,
            'status_id' => null,
            'status_slug' => null,
            'status_name' => null,
            'charging_price_type' => null,
            'charging_session_cost' => null,
            'digital_code' => null,
            'charging_price' => null,
            'vat' => 0,
            'charging_electricity_consumption' => null,
            'total_amount' => 0,
            'administration_fee' => 0,
            'name' => null,
            'vehicle' => null,
            'charging_duration' => null
        ];
        $this->dispatch(new CheckRequiredParams(['payment_id'], $params));
        $payment = $this->newQuery()
            ->whereNull('deleted_at')
            ->find($params['payment_id']);
        if (!$payment) {
            throw new Exception(trans('visiosoft.module.connect::message.not_found', ['name' => trans('visiosoft.module.payment::field.payment')]));
        }
        $output['payment_id'] = $payment->id;
        $output['status_id'] = $payment->status_id;
        $output['status_slug'] = $payment->status->slug;
        $output['status_name'] = $payment->status->name;
        $output['category_id'] = $payment->category_id;
        $locationName = '';
        $name = '';
        if (!empty($payment->user->name())) {
            $name = $payment->user->name();
        }
        $vehicleName = '';
        if ($payment->vehicle) {
            $vehicleName = $payment->vehicle->brand_text . ' ' . $payment->vehicle->model_text;
        }
        $output['name'] = $name;
        $output['vehicle'] = $vehicleName;
        $output['usage_type'] = !empty($payment->company_id) ? 'business' : 'personal';
        if (!empty($payment->charging_transaction)) {
            $output['location_id'] = $payment->charging_transaction->station_id;
            $duration = strtotime($payment->charging_transaction->ended_at) - strtotime($payment->charging_transaction->started_at);
            $output['charging_duration'] = $duration;
            $chargingPricingType = $payment->charging_transaction->station->pricing_type;
            $output['charging_price_type'] = $chargingPricingType;
            $output['charging_session_cost'] = $payment->charging_transaction->payment_amount;
            $output['charging_price'] = $payment->getUnitPrice();
            $output['charging_electricity_consumption'] = $payment->charging_transaction->total_consumption;
            $locationName = $payment->charging_transaction->station->title;
            $output['created_at'] = $payment->charging_transaction->created_at;
            $output['ended_at'] = $payment->charging_transaction->ended_at;
            $output['plate'] = $payment->charging_transaction->plate_number;
        }
        if ($payment->category_id == 1) {
            $output['type'] = 'charge';
        }
        $output['location_name'] = $locationName;
        $fees = $payment->fees;
        $output['vat'] = $fees['total_vat_amount'];
        $output['total_amount'] = $fees['payment_amount'];
        $output['administration_fee'] = $fees['administration_fee'];
        return collect($output);
    }

    /**
     * TODO: Now it only returns amount according to usage needs, for usages in the future, return items from items table(if it does not exist, create it).
     * @param $amount
     * @return array[]
     */
    public function getPaymentItems($amount): array
    {
        return [
            [
                "name" => trans('visiosoft.module.payment::field.charging_service'),
                "price" => $amount,
                "quantity" => 1,
                "description" => trans('visiosoft.module.payment::field.charging_service')
            ]
        ];
    }

    public function doPay($paymentID, $channel = "")
    {
        $payment = $this->newQuery()->find($paymentID);
        if (!$payment) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => trans('visiosoft.module.payment::field.payment')]));
        }

        $requireMethodValues = ['id', 'customer', 'service_payment_method_id', 'card'];

        $payment_method = $this->paymentMethodRepository->getCardToPay($payment->user_id, $payment->company_id, $requireMethodValues);
        if (!$payment_method) {
            throw new \Exception(trans('visiosoft.module.connect::message.not_found', ['name' => trans('visiosoft.module.payment::field.card.name')]));
        };


        $providerUserID = $payment_method->customer;
        $defaultCardID = $payment_method->service_payment_method_id;
        $price = $payment->payment_amount;
        $currency = $this->currency;
        $items = $this->getPaymentItems($price);
        $invoice_id = $payment->id;

        try {
            $payment->update(['status_id' => 3]); //pending
            $paymentRequest = $this->pay_provider->payBySavedCard($providerUserID, $defaultCardID, $price, $currency, $items, [], $invoice_id);
            $updateParams = ['last_try' => date("Y-m-d H:i:s")];
            $updateParams['payment_method'] = $payment->setPaymentMethod("card", $payment_method->getLast4Digits());

            if ($paymentRequest['data']['status'] == "success") {
                $updateParams['status_id'] = 2; //paid
                $updateParams['payment_intent_id'] = $paymentRequest['data']['payment_intent_id'];
                $updateParams['channel'] = $channel;
                // Add receipt to accounting system
                event(new PaymentReceiptEvent($paymentID));
            } else if ($paymentRequest['data']['status'] == "fail") {
                $updateParams['status_id'] = 4; // fail
                $updateParams['service_message'] = $paymentRequest['data']['service_message'];
                $updateParams['error_code'] = $paymentRequest['data']['error_code'];
            }
            $payment->update($updateParams);


        } catch (\Exception $e) {
            $updateParams['status_id'] = 4; //fail
            $updateParams['service_message'] = $e->getMessage();
            $payment->update($updateParams);
            throw new \Exception(trans('visiosoft.module.payment::message.error_payment'));
        }

        return $paymentRequest;
    }

    public function automaticWithdrawal($paymentID, $create_invoice = true)
    {
        $automaticWithdrawal = setting_value('visiosoft.module.payment::automatic_withdrawal');
        if ($automaticWithdrawal) {
            event(new PaymentCreatedEvent($paymentID));
            if ($create_invoice) {
                event(new CreateInvoiceEvent($paymentID));
            }
        }
    }

    public function refund($paymentID)
    {
        $payment = $this->find($paymentID);
        if (!$payment) {
            return false;
        }

        $refundRequest = $this->pay_provider->refund($payment->payment_intent_id, $payment->id);

        if ($refundRequest['data']['status'] === "success") {
            $paymentStatusRepository = app(StatusRepositoryInterface::class);
            $paymentStatus = $paymentStatusRepository->findBy('slug', 'refunded');
            if (!$paymentStatus) {
                //if the return is successful and the status is not found, we set a dummy number like 99
                //There's a slight chance that status won't be found. but added as a safety measure
                $paymentStatus = 99;
            } else {
                $paymentStatus = $paymentStatus->id;
            }

            $payment->update(['status_id' => $paymentStatus]);
        }
    }

    public function getPaymentByTransactionID($transactionID)
    {
        return $this->model->where('transaction_id', $transactionID)->first();
    }

    /**
     * @param $userID
     * @return numeric | false
     */
    public function getUserDept($userID)
    {
        //status[1] = pending
        //status[4] = failed
        $payment = $this->model
            ->selectRaw('SUM(payment_amount) as debt')
            ->where('user_id', $userID)
            ->where('amount', '>=', 1)
            ->whereIn('status_id', ['1', '4'])
            ->whereNull('deleted_at')
            ->first();

        if (!empty($payment->debt) && $payment->debt > 0) {
            return $payment->debt;
        } else {
            return false;
        }
    }

    public function removeOldVerificationPayment($userID = null)
    {
        $payments = $this->getPayments(false, $userID, ['unpaid', 'pending', 'failed']);
        if ($payments) {
            foreach ($payments as $payment) {
                $paymentCategoryRepository = app(CategoryRepositoryInterface::class);
                $paymentCategory = $paymentCategoryRepository->findBy('slug', 'verification');
                if ($paymentCategory) {
                    if ($paymentCategory->id == $payment->category_id) {
                        $payment->delete();
                    }
                }
            }
        }
    }

    public function createDraftPayment(array $params, $invoice_create = true, $automatic_withdrawal = true)
    {
        $paymentRecord = null;
        $paymentData = [
            'user_id' => $params['user_id'],
            'amount' => $params['amount'],
            'usage' => $params['usage'],
            'status' => 1,
            'category_id' => $params['category_id'],
            'currency' => $this->currency,
            'vehicle_id' => !empty($params['vehicle_id']) ? $params['vehicle_id'] : null,
            'plate' => !empty($params['plate']) ? $params['plate'] : null,
            'company_id' => !empty($params['company_id']) ? $params['company_id'] : null,
            'stripe_intent_id' => !empty($params['stripe_intent_id']) ? $params['stripe_intent_id'] : null,
            'station_id' => !empty($params['station_id']) ? $params['station_id'] : null,
            'ticket_id' => !empty($params['ticket_id']) ? $params['ticket_id'] : null,
            'transaction_id' => !empty($params['transaction_id']) ? $params['transaction_id'] : null,

        ];

        $paymentModule = app(PaymentModule::class);
        $fees = $paymentModule->getFees($params['amount']);
        if ($fees) {
            $paymentData['tax_percent'] = $fees['vat_percent'];
            $paymentData['payment_amount'] = $fees['payment_amount'];
            $paymentData['tax_amount'] = $fees['total_vat_amount'];
            $paymentData['additional_fees'] = $fees['administration_fee'];
        }

        if (!empty($params['transaction_id'])) {
            $paymentData['transaction_id'] = $params['transaction_id'];
            $paymentRecord = $this->getPaymentByTransactionID($paymentData['transaction_id']);
        }


        if (!$paymentRecord) {
            if (!$paymentRecord = $this->newQuery()->create($paymentData)) {
                throw new \Exception(trans('visiosoft.module.payment::message.transaction_create_failed'), 400);
            }
        }

        if ($paymentRecord->payment_amount > 0 && $automatic_withdrawal && $paymentRecord->status_id == 1) {
            $this->automaticWithdrawal($paymentRecord->id, $invoice_create);
        }

        return $paymentRecord;
    }

    /**
     * @param bool $isOpenQuery
     * @param int|null $userID
     * @param string[] $statuses
     * @param string[] $orderBy
     * @return Builder[]|Collection
     */
    public function getPayments(
        bool  $isOpenQuery = false,
        int   $userID = null,
        array $statuses = [
            'unpaid',
            'paid',
            'pending',
            'free',
            'failed'
        ],
              $orderBy = ['id', 'DESC'])
    {
        $model = app(PaymentWithoutAttrModel::class);

        $inStatus = $this->formatStatuses($statuses);
        $payments = $model
            ->whereNull('deleted_at')
            ->whereIn('status_id', $inStatus)
            ->with('status')
            ->orderBy($orderBy[0], $orderBy[1]);


        if (!empty($userID)) {
            $payments->where('user_id', $userID);
        }

        if (!$isOpenQuery) {
            $payments = $payments->get();
        }


        return $payments;
    }

    public function formatStatuses($statuses)
    {
        $formattedStatuses = [];

        foreach ($statuses as $status) {
            switch ($status) {
                case "unpaid":
                    $formattedStatuses[] = 1;
                    break;
                case "paid":
                    $formattedStatuses[] = 2;
                    break;
                case "pending":
                    $formattedStatuses[] = 3;
                    break;
                case "failed":
                    $formattedStatuses[] = 4;
                    break;
                case "free":
                    $formattedStatuses[] = 5;
                    break;
            }
        }


        return $formattedStatuses;

    }

}
