<?php namespace Visiosoft\PatronModule\Debt;

use Visiosoft\PatronModule\Debt\Contract\DebtRepositoryInterface;
use Anomaly\Streams\Platform\Entry\EntryRepository;

class DebtRepository extends EntryRepository implements DebtRepositoryInterface
{

    /**
     * The entry model.
     *
     * @var DebtModel
     */
    protected $model;

    /**
     * Create a new DebtRepository instance.
     *
     * @param DebtModel $model
     */
    public function __construct(DebtModel $model)
    {
        $this->model = $model;
    }

    public function getWeekData($week_ago, $company_id, $currency, $type)
    {
        if ($week_ago == "no_date") {
            return $this->getForDate(null, null, $company_id, $currency, $type);
        }
        $week_start = now()->add($week_ago, 'week')->startOfWeek();
        $week_end = now()->add($week_ago, 'week')->endOfWeek();
        return $this->getForDate($week_start, $week_end, $company_id, $currency, $type);
    }


    public function getForDate($date1, $date2, $company_id, $currency, $type)
    {
        $query = $this
            ->newQuery()
            ->selectRaw('sum(price) as total')
            ->where('type', $type);

        if ($date2 === null) {
            $query = $query->whereNull('expiry_date');
        } elseif ($date2->diffInWeeks() >= 8) {
            $query = $query->where('expiry_date', '<=', $date2);
        } else {
            $query = $query->whereBetween('expiry_date', [$date1, $date2]);
        }

        $query = $query->where('company_id', $company_id)
            ->where('currency', $currency)
            ->groupBy(['company_id'])
            ->get()->first();
        return ($query) ? $query->total : 0;
    }

    public function getCompanies($type)
    {
        $currencies_list = $this->newQuery()
            ->where('type', $type)
            ->select('currency')
            ->groupBy('currency')->get();

        $currencies = [];
        foreach ($currencies_list as $currency) {

            $companies = $this->newQuery()
                ->leftJoin('patron_companies', 'patron_debt.company_id', 'patron_companies.id')
                ->where('currency', $currency->currency)
                ->where('type', $type)
                ->groupBy('company_id')
                ->selectRaw('sum(price) as total,default_patron_debt.*,default_patron_companies.name as name')
                ->get();

            foreach ($companies as $company) {
                $receivables = array();
                $company->no_date = $this->getWeekData('no_date', $company->company_id, $currency->currency, $type);
                for ($week = -9; $week < 1; $week++) {
                    $receivables[$week] = $this->getWeekData($week, $company->company_id, $currency->currency, $type);
                }
                $company->receivables = $receivables;
            }
            $currencies[$currency->currency] = $companies;
        }
        return $currencies;
    }

    public function createDebt($company_id, $type, $price, $currency, $expiry_date)
    {
        return $this->create([
            'company_id' => $company_id,
            'type' => $type,
            'price' => $price,
            'currency' => $currency,
            'expiry_date' => $expiry_date
        ]);
    }
}
