<?php namespace Visiosoft\ParasutModule\Http\Controller;

use Anomaly\SettingsModule\Setting\Contract\SettingRepositoryInterface;
use Anomaly\Streams\Platform\Http\Controller\PublicController;
use Visiosoft\AdvsModule\Adv\AdvModel;
use Visiosoft\DopingsModule\Doping\DopingModel;
use Visiosoft\OrdersModule\Orderdetail\Contract\OrderdetailRepositoryInterface;
use Visiosoft\PackagesModule\Package\PackageModel;
use Visiosoft\PackagesModule\User\UserModel;
use Visiosoft\AdvsModule\Adv\Contract\AdvRepositoryInterface;
use Visiosoft\ParasutModule\Invoice\Contract\InvoiceRepositoryInterface;

class InvoicesController extends PublicController
{
    private $orderdetailRepository;
    private $advRepository;
    private $invoiceRepository;
    private $settingRepository;

    public function __construct(
        OrderdetailRepositoryInterface $orderdetailRepository,
        AdvRepositoryInterface $advRepository,
        InvoiceRepositoryInterface $invoiceRepository,
        SettingRepositoryInterface $settingRepository
    )
    {
        parent::__construct();
        $this->orderdetailRepository = $orderdetailRepository;
        $this->advRepository = $advRepository;
        $this->invoiceRepository = $invoiceRepository;
        $this->settingRepository = $settingRepository;
    }

    public function handlePaymentOrder($user, $order)
    {
        try {
            $requestData = $this->getRequestData($user, $order);
            $parasutInvoiceID = $this->createParasutInvoiceAPI($requestData);

            $this->createInvoice($parasutInvoiceID, $order->id);

            $this->messages->success(trans('visiosoft.module.parasut::message.parasute_invoice_was_created'));
        } catch (\Exception $e) {
            $this->createInvoice(0, $order->id, $e->getMessage());

            $error = json_decode($e->getMessage());

            $this->messages->error($error ? $error->detail : $e->getMessage());
        }
    }

    private function getRequestData($user, $order)
    {
        $ordersDetails = $this->orderdetailRepository->newQuery()->where('order_id', $order->id)->get();

        $products = array();
        foreach ($ordersDetails as $orderDetail) {
            $id = $orderDetail->item_id;
            switch ($orderDetail->item_type) {
                case 'adv':
                    $advModel = new AdvModel();
                    $adv = $advModel->getAdv($id);
                    $item = $adv ? $adv->name : '';
                    break;
                case 'package':
                    $packageUserModel = new UserModel();
                    $PackageModel = new PackageModel();
                    $entry = $packageUserModel->find($id);
                    $item = $entry ? $PackageModel->getPackage($entry->package_id)->name : '';
                    break;
                case 'doping':
                    $DopingModel = new DopingModel();
                    $doping = $DopingModel->newQuery()->find($id);
                    $item = $doping->adv_name->name;
                    break;
                case 'balance':
                    $item = trans('visiosoft.module.orders::field.balance.name');
                    break;
                case 'Subscription':
                    $plan = Plan::query()->find($id);
                    $item = $plan->name . " " . trans('visiosoft.module.subscriptions::field.plan.basic');
                    break;
                case 'store':
                    $store = app(StoreRepositoryInterface::class)->newQuery()->find($id);
                    $item = $store ? $store->name : '';
                    break;
                default:
                    $item = ''.$orderDetail->notes;
            }

            $productID = $this->createProduct($orderDetail->item_id, $item, $orderDetail->price);

            $products[] = [
                'type' => 'sales_invoice_details',
                'attributes' => [
                    'quantity' => $orderDetail->piece,
                    'unit_price' => $orderDetail->total,
                    'vat_rate' => $order->tax,
                ],
                'relationships' => [
                    'product' => [
                        'data' => [
                            'id' => $productID,
                            'type' => 'products',
                        ],
                    ],
                ],
            ];
        }

        $contactID = $this->createContact($user);

        return ['contactID' => $contactID, 'products' => json_encode($products)];
    }

    private function apiUrl()
    {
        $apiUrl = setting_value('visiosoft.module.parasut::api_url');
        $companyID = setting_value('visiosoft.module.parasut::company_id');

        if (!$apiUrl || !$companyID) {
            throw new \Exception(trans('visiosoft.module.parasut::message.please_fill_parasut_settings'));
        }

        return setting_value('visiosoft.module.parasut::api_url') . '/' . setting_value('visiosoft.module.parasut::company_id');
    }

    private function createContact($user)
    {
        $curl = curl_init();

        curl_setopt_array($curl, array(
            CURLOPT_URL => $this->apiUrl() . '/contacts',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 0,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => 'POST',
            CURLOPT_POSTFIELDS => '{
              "data": {
                "type": "contacts",
                "attributes": {
                  "email": "' . $user->email . '",
                  "name": "' . $user->name() . '",
                  "account_type": "customer"
                }
              }
            }',
            CURLOPT_HTTPHEADER => array(
                'Content-Type: application/vnd.api+json',
                'Authorization: Bearer ' . $this->getBearer(),
            ),
        ));

        $response = curl_exec($curl);

        curl_close($curl);

        $response = json_decode($response);

        if (isset($response->errors)) {
            throw new \Exception(json_encode($response->errors[0]));
        }

        return $response->data->id;
    }

    private function createProduct($code, $name, $price)
    {
        $curl = curl_init();

        curl_setopt_array($curl, array(
            CURLOPT_URL => $this->apiUrl() . '/products',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 0,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => 'POST',
            CURLOPT_POSTFIELDS => '{
                  "data": {
                    "type": "products",
                    "attributes": {
                      "code": "' . $code . '",
                      "name": "' . $name . '",
                      "list_price": ' . $price . ',
                      "buying_price": ' . $price . '
                    }
                  }
                }',
            CURLOPT_HTTPHEADER => array(
                'Content-Type: application/vnd.api+json',
                'Authorization: Bearer ' . $this->getBearer(),
            ),
        ));

        $response = curl_exec($curl);

        curl_close($curl);

        $response = json_decode($response);

        if (isset($response->errors)) {
            throw new \Exception(json_encode($response->errors[0]));
        }

        return $response->data->id;
    }

    private function getBearer()
    {
        $oauthTokenUrl = setting_value('visiosoft.module.parasut::oauth_token_url');
        $clientID = setting_value('visiosoft.module.parasut::client_id');
        $clientSecret = setting_value('visiosoft.module.parasut::client_secret');
        $refreshToken = setting_value('visiosoft.module.parasut::refresh_token');

        if (!$oauthTokenUrl || !$clientID || !$clientSecret || !$refreshToken) {
            throw new \Exception(trans('visiosoft.module.parasut::message.please_fill_parasut_settings'));
        }

        $curl = curl_init();

        curl_setopt_array($curl, array(
            CURLOPT_URL => $oauthTokenUrl,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 0,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => 'POST',
            CURLOPT_POSTFIELDS => array(
                'grant_type' => 'refresh_token',
                'client_id' => $clientID,
                'client_secret' => $clientSecret,
                'refresh_token' => $refreshToken
            ),
        ));

        $response = curl_exec($curl);

        curl_close($curl);

        $response = json_decode($response);

        if (isset($response->errors)) {
            throw new \Exception(json_encode($response->errors[0]));
        }

        $this->settingRepository->set('visiosoft.module.parasut::refresh_token', $response->refresh_token);

        return $response->access_token;
    }

    private function createParasutInvoiceAPI($requestInfo)
    {
        $curl = curl_init();

        curl_setopt_array($curl, array(
            CURLOPT_URL => $this->apiUrl() . '/sales_invoices',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 0,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => 'POST',
            CURLOPT_POSTFIELDS => '{
              "data": {
                "type": "sales_invoices",
                "attributes": {
                  "item_type": "invoice",
                  "issue_date": "' . date('Y-m-d') . '",
                  "order_no": "string",
                  "order_date": "' . date('Y-m-d') . '"
                },
                "relationships": {
                  "details": {
                    "data": ' . $requestInfo['products'] . '
                  },
                  "contact": {
                    "data": {
                      "id": "' . $requestInfo['contactID'] . '",
                      "type": "contacts"
                    }
                  }
                }
              }
            }',
            CURLOPT_HTTPHEADER => array(
                'Content-Type: application/vnd.api+json',
                'Authorization: Bearer ' . $this->getBearer(),
            ),
        ));

        $response = curl_exec($curl);

        curl_close($curl);

        $response = json_decode($response);

        if (isset($response->errors)) {
            throw new \Exception(json_encode($response->errors[0]));
        }

        return $response->data->id;
    }

    private function createInvoice($invoiceID, $orderID, $response = null)
    {
        $this->invoiceRepository->create([
            'invoice_id' => $invoiceID,
            'order_id' => $orderID,
            'response' => $response
        ]);
    }
}
