<?php namespace Visiosoft\PagesApiExtension\Page;

use Anomaly\PagesModule\Page\Contract\PageRepositoryInterface;
use Anomaly\PagesModule\Type\Contract\TypeRepositoryInterface;
use Anomaly\Streams\Platform\Entry\EntryRepository;
use Anomaly\Streams\Platform\Model\Pages\PagesDefaultPagesEntryModel;
use Illuminate\Foundation\Bus\DispatchesJobs;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Auth;
use Visiosoft\ConnectModule\Command\CheckPermission;
use Visiosoft\ConnectModule\Command\CheckRequiredParams;
use Visiosoft\ConnectModule\Command\CreateTranslatableValues;

class PageApiCollection extends PageRepository
{
    use DispatchesJobs;

    public function newCreate(array $params)
    {
        $this->checkPermission('create');

        $this->dispatch(new CheckRequiredParams(['title', 'content', 'is_home', 'meta_title', 'meta_description', 'publish_at'], $params));

        $params['slug'] = $params['title'];

        $types = app(TypeRepositoryInterface::class);
        $pages = app(PageRepositoryInterface::class);
        $repository = new EntryRepository();
        $repository->setModel(new PagesDefaultPagesEntryModel());

        // Check Type
        if (!$type = $types->findBySlug('default')) {
            throw new \Exception("Type bulunamadı!", 404);
        }

        // Generate Content
        $content = ['content' => $params['content'], 'created_by_id' => Auth::id()];

        unset($params['content']);

        $content = $this->dispatch(new CreateTranslatableValues($content));

        $content_entry = (new PagesDefaultPagesEntryModel())->create($content);

        // Set Translatable Columns
        $params = $this->dispatch(new CreateTranslatableValues($params, $content_entry->id));

        $create_parameters = array_merge(
            $params,
            [
                'enabled' => true,
                'visible' => true,
                'publish_at' => Carbon::now(),
                'entry' => $content_entry,
                'type' => $type,
                'created_by_id' => Auth::id()
            ]
        );

        $entry = $pages->create($create_parameters);

        return $entry;
    }

    public function getPages()
    {
        $query = new \Anomaly\PagesModule\Page\PageModel();

        $query = $query->newQuery()->where('enabled', true);

        return $query;
    }

    public function getPageDetail(array $params)
    {
        $this->dispatch(new CheckRequiredParams(['slug'], $params));

        $query = new \Anomaly\PagesModule\Page\PageModel();

        $entry = $query->newQuery()->where('pages_pages_translations.slug', $params['slug'])
            ->leftJoin('pages_pages_translations', function ($join) {
                $join->on('pages_pages.id', '=', 'pages_pages_translations.entry_id');
                $join->where('pages_pages_translations.locale', config('app.locale'));
            })
            ->leftJoin('pages_default_pages', 'pages_pages.entry_id', 'pages_default_pages.id')
            ->leftJoin('pages_default_pages_translations', function ($join) {
                $join->on('pages_default_pages.id', '=', 'pages_default_pages_translations.entry_id');
                $join->where('pages_default_pages_translations.locale', config('app.locale'));
            })
            ->select(['pages_pages.*', 'pages_pages_translations.*',
                'pages_default_pages_translations.content as content'])
            ->first();

        return $entry;
    }

    public function deletePage(array $params)
    {
        $this->checkPermission('delete');

        $this->dispatch(new CheckRequiredParams(['id'], $params));

        $entry = $this->checkPage($params['id']);

        $this->checkPermission($entry);

        if ($entry->delete()) {

            return true;
        }

        return false;
    }

    public function updatePage(array $params)
    {
        $this->checkPermission('update');

        $this->dispatch(new CheckRequiredParams(['id'], $params));

        $entry = $this->checkPage($params['id']);

        unset($params['id']);

        $repository = new EntryRepository();
        $repository->setModel(new PagesDefaultPagesEntryModel());


        if (isset($params['content'])) {
            // Generate Content
            $content = ['content' => $params['content'], 'created_by_id' => Auth::id()];

            unset($params['content']);

            $content = $this->dispatch(new CreateTranslatableValues($content));

            $content_entry = (new PagesDefaultPagesEntryModel())->create($content);

            $params['entry'] = $content_entry;
        }

        // Set Translatable Columns
        $params = $this->dispatch(new CreateTranslatableValues($params));

        $update_parameters = array_merge(
            $params,
            [
                'updated_by_id' => Auth::id()
            ]
        );


        $entry->update(
            $update_parameters
        );

        return $entry;
    }

    public function checkPage($id)
    {
        $model = new \Anomaly\PagesModule\Page\PageModel();

        if (!$entry = $model->newQuery()->find($id)) {
            throw new \Exception("Page bulunamadı!", 404);
            die;
        }

        return $entry;
    }

    public function checkPermission($permission)
    {
        $this->dispatch(new CheckPermission('visiosoft.extension.pages_api::pages.' . $permission));
    }
}
