<?php namespace Visiosoft\OrdersModule\Order;

use Illuminate\Foundation\Bus\DispatchesJobs;
use Illuminate\Support\Facades\Auth;
use Visiosoft\AdvsModule\Support\Command\Currency;
use Visiosoft\ConnectModule\Command\CheckRequiredParams;
use Visiosoft\OrdersModule\Order\Command\GetPayTypeName;
use Visiosoft\OrdersModule\Order\Contract\OrderRepositoryInterface;
use Anomaly\Streams\Platform\Entry\EntryRepository;
use Visiosoft\OrdersModule\Orderdetail\Contract\OrderdetailRepositoryInterface;

class OrderRepository extends EntryRepository implements OrderRepositoryInterface
{

    use DispatchesJobs;
    protected $model;
    protected $orderdetail;
    protected $currency;

    public function __construct(OrderModel $model, OrderdetailRepositoryInterface $orderdetail, Currency $currency)
    {
        $this->model = $model;
        $this->orderdetail = $orderdetail;
        $this->currency = $currency;
    }

    public function approveOrder($orderId)
    {
        $this->model->setStatus('approved', $orderId);
    }

    public function rejectOrder($orderId)
    {
        if ($order = $this->newQuery()->find($orderId)) {
            $order->update(['status' => 'rejected']);
        }
    }

    public function createOrder($user_id, $total, $subtotal, $currency, $pay_type, $bill_address_id = null, $delivery_address_id = null, $type = null)
    {
        return $this->create([
            'pay_type' => $pay_type,
            'subtotal' => $subtotal,
            'user_id' => $user_id,
            'total' => $total,
            'currency' => $currency,
            'order_at' => now(),
            'status' => 'pending',
            'bill_address_id' => $bill_address_id,
            'delivery_address_id' => $delivery_address_id,
            'type' => $type,
        ]);
    }

    public function getUsersOrders($usersIDs, $paginate = 3, $startDate = null, $endDate = null)
    {
        $orders = $this->newQuery();

        if ($startDate) {
            $orders = $orders->where('created_at', '>=', $startDate);
        }

        if ($endDate) {
            $orders = $orders->where('created_at', '<=', $endDate);
        }

        $orders = $orders
            ->whereIn('user_id', $usersIDs)
            ->paginate($paginate);

        foreach ($orders as $order) {
            $order->setAttribute('pay_type_name', dispatch_now(new GetPayTypeName($order->pay_type)));
            $order->setAttribute('ordered_by_name', $order->user->name());
            $order->setAttribute('total_formatted', $this->currency->format($order->total, $order->currency));
            $order->setAttribute(
                'post_commission_formatted',
                $this->currency->format((1 - auth()->user()->commission / 100) * $order->total, $order->currency)
            );
        }

        return $orders;
    }

    public function getUsersOrdersTotal($usersIDs, $startDate = null, $endDate = null)
    {
        if (count($usersIDs)) {
            $ordersTotal = $this->newQuery();

            if ($startDate) {
                $ordersTotal = $ordersTotal->where('created_at', '>=', $startDate);
            }

            if ($endDate) {
                $ordersTotal = $ordersTotal->where('created_at', '<=', $endDate);
            }

            $ordersTotal = $ordersTotal
                ->whereIn('user_id', $usersIDs)
                ->where('status', 'approved')
                ->sum('total');

            return $ordersTotal;
        }

        return null;
    }

    public function getUsedPaymentTypes()
    {
        return $this->getModel()->groupBy('pay_type')->pluck('pay_type')->all();
    }

    public function getMySales()
    {
        $order_detail_repository = app(OrderdetailRepositoryInterface::class);

        return $order_detail_repository->mySales();
    }

    public function getMyPurchases()
    {
        return $this->newQuery()->where('user_id', Auth::id());
    }

    public function getSaleOrderInformation(array $params)
    {
        $this->dispatch(new CheckRequiredParams(['sale_id'], $params));

        $order_detail_repository = app(OrderdetailRepositoryInterface::class);

        $detail = $order_detail_repository->find($params['sale_id']);

        if($detail && $detail->seller_id == Auth::id())
        {
            $entry =  $this->newQuery()->find($detail->order_id);

            $entry->setAppends([]);
            $entry->setAttribute('total',null);
            $entry->setAttribute('subtotal',null);
            $entry->setAttribute('shipping',null);
            $entry->setAttribute('tax',null);

            return $entry;
        }

        throw new \Exception(trans('visiosoft.module.orders::message.error_sale_detail'));
    }

    public function getOrdersReportByCity()
    {
        $orders = $this->newQuery()
            ->selectRaw('COUNT(*) as count, name as city')
            ->leftJoin('profile_adress as address', 'orders_order.delivery_address_id', '=', 'address.id')
            ->groupBy('city')
            ->leftJoin('location_cities_translations as cities_trans', function ($join) {
                $join->on('address.city', '=', 'cities_trans.entry_id');
                $join->whereIn('locale', [config('app.locale'), setting_value('streams::default_locale'), 'en']);
            });

        if ($search = request('search.value')) {
            $orders = $orders->where('name', 'LIKE', "%$search%");
        }

        if ($orderDir = request('order.0.dir')) {
            $orders = $orders->orderBy(request('order.0.column') == 1 ? 'count' : 'name', $orderDir);
        }

        $start = request('start');
        $limit = request('length') ?: 10;
        $page = $start ? $start / $limit + 1 : 1;

        return $orders->paginate($limit, ['*'], 'page', $page);
    }

    public function getOrdersReportByPaymentType()
    {
        $orders = $this->newQuery()
            ->selectRaw('COUNT(*) as count, pay_type as payment')
            ->groupBy('pay_type');

        if ($search = request('search.value')) {
            $orders = $orders->where('pay_type', 'LIKE', "%$search%");
        }

        if ($orderDir = request('order.0.dir')) {
            $orders = $orders->orderBy(request('order.0.column') == 1 ? 'count' : 'pay_type', $orderDir);
        }

        $start = request('start');
        $limit = request('length') ?: 10;
        $page = $start ? $start / $limit + 1 : 1;

        return $orders->paginate($limit, ['*'], 'page', $page);
    }

    public function getOrdersReportByMember()
    {
        $orders = $this->newQuery()
            ->selectRaw("COUNT(*) as count, CONCAT_WS('', first_name, ' ', last_name) AS member")
            ->groupBy('user_id')
            ->leftJoin('users_users as users', 'orders_order.user_id', '=', 'users.id');

        if ($search = request('search.value')) {
            $orders = $orders->whereRaw("
                (SELECT CONCAT_WS('', first_name, ' ', last_name) AS member FROM default_users_users WHERE id = user_id) LIKE '%$search%'
            ");
        }

        if ($orderDir = request('order.0.dir')) {
            $orders = $orders->orderBy(request('order.0.column') == 1 ? 'count' : 'member', $orderDir);
        }

        $start = request('start');
        $limit = request('length') ?: 10;
        $page = $start ? $start / $limit + 1 : 1;

        return $orders->paginate($limit, ['*'], 'page', $page);
    }
}
