<?php namespace Visiosoft\InterestsModule\Interest;

use Illuminate\Support\Facades\Auth;
use Visiosoft\CatsModule\Category\Contract\CategoryRepositoryInterface;
use Visiosoft\InterestsModule\Interest\Contract\InterestRepositoryInterface;
use Anomaly\Streams\Platform\Entry\EntryRepository;

class InterestRepository extends EntryRepository implements InterestRepositoryInterface
{

    /**
     * The entry model.
     *
     * @var InterestModel
     */
    protected $model;

    /**
     * Create a new InterestRepository instance.
     *
     * @param InterestModel $model
     */
    public function __construct(InterestModel $model)
    {
        $this->model = $model;
    }

    public function getInterestsWithParentByUser($parent_id, $user_id)
    {
        return $this->newQuery()
            ->where('user_id', $user_id)
            ->where('parent_category_id', $parent_id)
            ->get();
    }

    public function findCategoryByUser($id, $user_id)
    {
        return $this->newQuery()
            ->where('user_id', $user_id)
            ->where('category_id', $id)
            ->first();
    }

    public function isFirstInterest($userID = null)
    {
        $userID = auth_id_if_null($userID);
        return $this->model->withTrashed()->where('user_id', $userID)->first();
    }

    public function createByUser($user_id, $category_id, $parent_id)
    {
        return $this->create([
            'user_id' => $user_id,
            'category_id' => $category_id,
            'parent_category_id' => $parent_id,
        ]);
    }

    public function findAllByCatID($catID)
    {
        return $this->findAllBy('category_id', $catID);
    }

    public function findAllByParentCatID($catID, $userID = null)
    {
        $query = $this->newQuery()->where('parent_category_id', $catID);

        if ($userID) {
            $query = $query->where('user_id', $userID);
        }

        return $query->get();
    }

    public function getInsterestsByUser()
    {
        $category = app(CategoryRepositoryInterface::class);
        $main_categories = $category->getMainCategories();

        foreach ($main_categories as $category) {
            $interest_by_parent = $this->findAllByParentCatID($category->id, auth()->id())->pluck('category_id')->all();
            $category->active = $this->getInterestsByIds($category->id, $interest_by_parent);
            $category->passive = $this->getInterestsByIds($category->id, $interest_by_parent, 'without');
        }
        return $main_categories;
    }

    public function getInterestsByParent($parent_id)
    {
        $item = new \stdClass();
        if (count($interest_by_parent = $this->findAllByParentCatID($parent_id,Auth::id()))) {
            $interest_by_parent = $interest_by_parent->pluck('category_id')->all();
            $item->active = $this->getInterestsByIds($parent_id, $interest_by_parent);
            $item->passive = $this->getInterestsByIds($parent_id, $interest_by_parent, 'without');
        }
        return $item;
    }

    public function getInterestsByIds($parent_id, $ids, $where = 'in')
    {
        $category = app(CategoryRepositoryInterface::class);

        $category = $category->newQuery()
            ->where('parent_category_id', $parent_id);

        $category = ($where == 'in') ? $category->whereIn('cats_category.id', $ids) : $category->whereNotIn('cats_category.id', $ids);

        return $category->leftJoin('cats_category_translations', function ($join) {
            $join->on('cats_category.id', '=', 'cats_category_translations.entry_id');
            $join->whereIn('cats_category_translations.locale', [config('app.locale'), setting_value('streams::default_locale'), 'en']);//active lang
        })
            ->select('cats_category.*', 'cats_category_translations.name as name')
            ->orderBy('name')->get();
    }
}
