<?php namespace Visiosoft\ImportLocationModule\Http\Controller\Admin;

use Anomaly\FilesModule\File\Contract\FileRepositoryInterface;
use Anomaly\Streams\Platform\Http\Controller\AdminController;
use Anomaly\Streams\Platform\Model\Location\LocationCitiesEntryTranslationsModel;
use Anomaly\Streams\Platform\Model\Location\LocationDistrictsEntryTranslationsModel;
use Anomaly\Streams\Platform\Model\Location\LocationNeighborhoodsEntryTranslationsModel;
use Anomaly\Streams\Platform\Ui\Form\FormBuilder;
use Illuminate\Support\Str;
use Visiosoft\LocationModule\City\Contract\CityRepositoryInterface;
use Visiosoft\LocationModule\District\Contract\DistrictRepositoryInterface;
use Visiosoft\LocationModule\Neighborhood\Contract\NeighborhoodRepositoryInterface;

class ImportLocationController extends AdminController
{

    private $fileRepository;
    private $cityRepository;
    private $districtRepository;
    private $neighborhoodRepository;
    private $citiesEntryTranslationsModel;
    private $locationDistrictsEntryTranslationsModel;
    private $neighborhoodsEntryTranslationsModel;

    public function __construct(
        FileRepositoryInterface $fileRepository,
        CityRepositoryInterface $cityRepository,
        DistrictRepositoryInterface $districtRepository,
        NeighborhoodRepositoryInterface $neighborhoodRepository,
        LocationCitiesEntryTranslationsModel $citiesEntryTranslationsModel,
        LocationDistrictsEntryTranslationsModel $locationDistrictsEntryTranslationsModel,
        LocationNeighborhoodsEntryTranslationsModel $neighborhoodsEntryTranslationsModel
    )
    {
        parent::__construct();
        $this->fileRepository = $fileRepository;
        $this->cityRepository = $cityRepository;
        $this->districtRepository = $districtRepository;
        $this->neighborhoodRepository = $neighborhoodRepository;
        $this->citiesEntryTranslationsModel = $citiesEntryTranslationsModel;
        $this->locationDistrictsEntryTranslationsModel = $locationDistrictsEntryTranslationsModel;
        $this->neighborhoodsEntryTranslationsModel = $neighborhoodsEntryTranslationsModel;
    }

    public function index(FormBuilder $formBuilder)
    {
        set_time_limit(0);
        if (request()->action == "create") {
            $file = $this->fileRepository->find(request()->file);

            if ($file->extension !== 'csv') {
                $this->messages->error(trans('visiosoft.module.import_location::message.csv_file_error'));
                return redirect('/admin/import-location');
            } elseif (!setting_value('visiosoft.module.import_location::country')) {
                $this->messages->error(trans('visiosoft.module.import_location::message.select_country_error'));
                return redirect('/admin/import-location');
            }

            $locations = $this->csvToArray($file->name);

            foreach ($locations as $location) {
                $this->createLocation($location);
            }

            $this->messages->success(trans('visiosoft.module.import_location::message.locations_added', ['number' => count($locations)]));
            return redirect('/admin/import-location');
        }
        $formBuilder->setActions(['create']);
        $formBuilder->setFields([
            'file' => [
                "type"   => "anomaly.field_type.file",
                "config" => [
                    "folders" => ["csv_files"],
                ]
            ],
        ]);
        return $formBuilder->render();
    }

    public function csvToArray($filename = '', $delimiter = ',')
    {
        $filePath = app_storage_path() . '/files-module/local/csv_files/' . $filename;
        if (!file_exists($filePath) || !is_readable($filePath)) {
            return false;
        }

        $header = null;
        $data = array();
        if (($handle = fopen($filePath, 'r')) !== false) {
            while (($row = fgetcsv($handle, 1000, $delimiter)) !== false) {
                if (!$header) {
                    $header = $row;
                } else {
                    $data[] = array_combine($header, $row);
                }
            }
            fclose($handle);
        }

        return $data;
    }

    public function createLocation($location)
    {
        // Create city
        $city = null;
        $possibleCities = $this->citiesEntryTranslationsModel->newQuery()->where('name', $location['PROV_NAM'])->get();
        if (count($possibleCities)) {
            foreach ($possibleCities as $possibleCity) {
                $city = $this->cityRepository->find($possibleCity->entry_id);
                if ($city->parent_country_id === setting_value('visiosoft.module.import_location::country')) {
                    break;
                }
            }
        }
        if (!count($possibleCities) || !$city) {
            $city = $this->cityRepository->findBy('slug', Str::slug($location['PROV_NAM'], '_'));
            if (!$city) {
                $city = $this->cityRepository->create([
                    'name' => $location['PROV_NAM'],
                    'slug' => Str::slug($location['PROV_NAM'], '_'),
                    'parent_country_id' => setting_value('visiosoft.module.import_location::country')
                ]);
            }
        }

        // Create district
        $district = null;
        $possibleDistricts = $this->locationDistrictsEntryTranslationsModel->newQuery()
            ->where('name', $location['DIST_NAM'])->get();
        if (count($possibleDistricts)) {
            foreach ($possibleDistricts as $possibleDistrict) {
                $district = $this->districtRepository->find($possibleDistrict->entry_id);
                if ($district->parent_city_id === $city->id) {
                    break;
                }
            }
        }
        if (!count($possibleDistricts) || !$district) {
            $district = $this->districtRepository->findBy('slug', Str::slug($location['DIST_NAM'], '_'));
            if (!$district) {
                $district = $this->districtRepository->create([
                    'name' => $location['DIST_NAM'],
                    'slug' => Str::slug($location['DIST_NAM'], '_'),
                    'parent_city_id' => $city->id
                ]);
            }
        }

        // Create district
        $neighborhood = null;
        $possibleNeighborhoods = $this->neighborhoodsEntryTranslationsModel->newQuery()
            ->where('name', $location['NAME'])->get();
        if (count($possibleNeighborhoods)) {
            foreach ($possibleNeighborhoods as $possibleNeighborhood) {
                $neighborhood = $this->neighborhoodRepository->find($possibleNeighborhood->entry_id);
                if ($neighborhood->parent_district_id === $district->id) {
                    break;
                }
            }
        }
        if (!count($possibleNeighborhoods) || !$neighborhood) {
            $neighborhood = $this->neighborhoodRepository->findBy('slug', Str::slug($location['NAME'], '_'));
            if (!$neighborhood) {
                $neighborhood = $this->neighborhoodRepository->create([
                    'name' => $location['NAME'],
                    'slug' => Str::slug($location['NAME'], '_'),
                    'parent_district_id' => $city->id
                ]);
            }
        }
    }
}
