<?php namespace Visiosoft\ImportModule\Http\Controller\Admin;

use Anomaly\UsersModule\User\Contract\UserRepositoryInterface;
use Carbon\Carbon;
use Illuminate\Support\Str;
use Visiosoft\AdvsModule\Adv\Contract\AdvRepositoryInterface;
use Visiosoft\ImportModule\Events\SendPasswordSMS;
use Visiosoft\ImportModule\FacebookProduct\Contract\FacebookProductRepositoryInterface;
use Anomaly\Streams\Platform\Http\Controller\AdminController;

class FacebookProductsController extends AdminController
{
    private $facebookProductRepository;
    private $userRepository;

    public function __construct(
        FacebookProductRepositoryInterface $facebookProductRepository,
        UserRepositoryInterface $userRepository
    )
    {
        parent::__construct();
        $this->facebookProductRepository = $facebookProductRepository;
        $this->userRepository = $userRepository;
    }

    public function index(UserRepositoryInterface $userRepository)
    {
        $serializedItems = $this->facebookProductRepository->getModel()->take(10)->get();

        $items = array();
        foreach ($serializedItems as $item) {
            $data = unserialize($item->data);

            // Check if user exists
            $phone = $data['text'] ?: $data['post_text'];
            preg_match('/\+?[0-9][0-9()\-\s+]{4,20}[0-9]/', $phone, $matches);
            if (count($matches)) {
                $phone = trim($matches[0]);
                $phone = preg_replace('/\s+/', '', $phone);
                $phone = ltrim($phone, '0');

                if ($user = $userRepository->getModel()->where('gsm_phone', 'LIKE', "%$phone")->first()) {
                    $data['user'] = [
                        'name' => $user->name(),
                        'id' => $user->id,
                    ];
                }
            }

            // Check if price exists
            $price = $data['text'] ?: $data['post_text'];
            $price = preg_replace('/[^0-9]/', '', $price);
            $data['price'] = $price;

            $items[] = array_merge($data, [
                'id' => $item->id
            ]);
        }

        $defaultCountry = setting_value('visiosoft.module.location::default_country');
        if (!$defaultCountry) {
            throw new \Exception('Please select a default country from the location module settings.');
        }

        return view('module::admin/import', compact('items', 'defaultCountry'));
    }

    public function truncate()
    {
        $this->facebookProductRepository->truncate();

        $this->messages->success('Table truncated successfully');

        return redirect('admin/import/facebook_products');
    }

    public function create(AdvRepositoryInterface $advRepository)
    {
        $validations = [
            'product_id' => 'required|exists:import_facebook_products,id',
            'id' => 'required|numeric',
            'name' => 'required',
            'price' => 'required|numeric',
            'cat1' => 'required|exists:cats_category,id',
            'cat2' => 'exists:cats_category,id',
        ];
        if (request()->add_customer) {
            $validations = array_merge($validations, [
                'user_name' => 'required',
                'gsm_phone' => 'required|unique:users_users',
            ]);
        } else {
            $validations = array_merge($validations, [
                'created_by' => 'required|exists:users_users,id',
            ]);
        }
        request()->validate($validations);

        try {
            if (request()->add_customer) {
                $username = Str::slug(request()->user_name) . '_' . Carbon::now()->timestamp;
                $password = Str::random(8);
                $user = $this->userRepository->create([
                    'email' => "$username@example.com",
                    'username' => $username,
                    'display_name' => $username,
                    'first_name' => $username,
                    'last_name' => $username,
                    'activated' => true,
                    'enabled' => true,
                    'gsm_phone' => request()->gsm_phone,
                ]);
                $user->setPasswordAttribute($password);
                $user->save();

                event(new SendPasswordSMS($user, $password));

                $userID = $user->id;
            } else {
                $userID = request()->created_by;
            }

            $date = date('Y-m-d H:i:s');
            $data = [
                'created_by_id' => $userID,
                'name' => request()->name,
                'advs_desc' => request()->advs_desc,
                'price' => request()->price,
                'cat1' => request()->cat1,
                'cat2' => request()->cat2,
                'slug' => Str::slug(request()->name, '_'),
                'currency' => setting_value('streams::currency'),
                'status' => 'approved',
                'publish_at' => $date,
                'finish_at' => date('Y-m-d H:i:s', strtotime($date . ' + 60 day')),
                'config' => json_encode(['fb_id' => request()->id]),
            ];

            $adv = $advRepository
                ->newQuery()
                ->whereJsonContains('config->fb_id', request()->id)
                ->first();
            if (is_null($adv)) {
                $adv = $advRepository->create($data);

//            $files = request('files');
//            if ($files) {
//                $matches = array();
//                $t = preg_match_all('/\'(.*?)\'/s', $files, $matches);
//                $files = $matches[1];
//            }

//            $adImages = $this->savePhoto($files, $adv->id);
//            if (count($adImages) > 0) {
//                $adv->update([
//                    'cover_photo' => $adImages[0]->url()
//                ]);
//            }
            } else {
                $adv->update($data);
            }

            $this->facebookProductRepository->find(request()->product_id)->delete();

            return [
                'success' => true,
                'product' => $adv,
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'msg' => $e->getMessage(),
            ];
        }
    }
}
