<?php namespace Visiosoft\GgTheme;

use Anomaly\NavigationModule\Link\Contract\LinkRepositoryInterface;
use Anomaly\NavigationModule\Menu\Contract\MenuRepositoryInterface;
use Anomaly\SettingsModule\Setting\Contract\SettingRepositoryInterface;
use Anomaly\Streams\Platform\Application\Application;
use Anomaly\Streams\Platform\Assignment\Contract\AssignmentRepositoryInterface;
use Anomaly\Streams\Platform\Database\Seeder\Seeder;
use Anomaly\Streams\Platform\Entry\EntryRepository;
use Anomaly\Streams\Platform\Field\Contract\FieldRepositoryInterface;
use Anomaly\Streams\Platform\Model\Repeater\RepeaterHomepageBannerEntryModel;
use Anomaly\Streams\Platform\Model\Repeater\RepeaterPopularCategoriesEntryModel;
use Anomaly\Streams\Platform\Model\Repeater\RepeaterShowcaseEntryModel;
use Anomaly\Streams\Platform\Model\Repeater\RepeaterTinySliderEntryModel;
use Anomaly\Streams\Platform\Stream\Contract\StreamRepositoryInterface;
use Anomaly\UrlLinkTypeExtension\UrlLinkTypeModel;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Visiosoft\CatsModule\Category\CategoryModel;
use ZipArchive;

class GgThemeSeeder extends Seeder
{
    private $settingRepository;
    private $streamRepository;
    private $fieldRepository;
    private $assignmentRepository;
    private $menus;
    private $links;

    public function __construct(
        SettingRepositoryInterface $settingRepository,
        StreamRepositoryInterface $streamRepository,
        FieldRepositoryInterface $fieldRepository,
        AssignmentRepositoryInterface $assignmentRepository,
        MenuRepositoryInterface $menus,
        LinkRepositoryInterface $links
    )
    {
        parent::__construct();
        $this->settingRepository = $settingRepository;
        $this->streamRepository = $streamRepository;
        $this->fieldRepository = $fieldRepository;
        $this->assignmentRepository = $assignmentRepository;
        $this->menus = $menus;
        $this->links = $links;
    }

    /**
     * Run the seeder.
     */
    public function run()
    {

        // Create repeaters
        $app = app(Application::class)->getReference()."_";

        $repeaters = [
            [
                'name' => 'Homepage Banner',
                'slug' => 'homepage_banner',
                'translatable' => false,
                'fields' => ['large_image', 'medium_image', 'thumbnail_image', 'link']
            ],
            [
                'name' => 'Tiny Slider',
                'slug' => 'tiny_slider',
                'translatable' => true,
                'fields' => ['background', 'link', 'description', 'title']
            ],
            [
                'name' => 'Showcase',
                'slug' => 'showcase',
                'translatable' => true,
                'fields' => ['background', 'title', 'link']
            ],
            [
                'name' => 'Popular Categories',
                'slug' => 'popular_categories',
                'translatable' => true,
                'fields' => ['large_image', 'popular_categories']
            ]
        ];

        $repeatersObjects = array();
        foreach ($repeaters as $repeater) {
            $exists = $this->streamRepository->findBySlugAndNamespace($repeater['slug'], 'repeater');
            $repeatersObjects[$repeater['slug']] = $exists ?: $this->streamRepository->create([
                'name' => $repeater['name'],
                'namespace' => 'repeater',
                'slug' => $repeater['slug'],
                'prefix' => 'repeater_',
                'translatable' => $repeater['translatable'],
            ]);
        }

        // Assign repeater fields
        $repeatersFields = [
            [
                'name' => 'Large Image',
                'namespace' => 'repeater',
                'slug' => 'large_image',
                'type' => 'anomaly.field_type.file',
                'instructions' => 'Recommended 1000x350',
                "config" => [
                    "folders" => ['images'],
                    "mode" => 'upload',
                ],
                "assignmentConfig" => [
                    "required" => true,
                    "translatable" => false,
                ],
            ],
            [
                'name' => 'Medium Image',
                'namespace' => 'repeater',
                'slug' => 'medium_image',
                'type' => 'anomaly.field_type.file',
                'instructions' => 'Recommended 510x384',
                "config" => [
                    "folders" => ['images'],
                    "mode" => 'upload',
                ],
                "assignmentConfig" => [
                    "required" => true,
                    "translatable" => false,
                ],
            ],
            [
                'name' => 'Thumbnail Image',
                'namespace' => 'repeater',
                'slug' => 'thumbnail_image',
                'type' => 'anomaly.field_type.file',
                'instructions' => 'Recommended 65x65',
                "config" => [
                    "folders" => ['images'],
                    "mode" => 'upload',
                ],
                "assignmentConfig" => [
                    "required" => true,
                ],
            ],
            [
                'name' => 'Background',
                'namespace' => 'repeater',
                'slug' => 'background',
                'type' => 'anomaly.field_type.file',
                'instructions' => 'Recommended [280x376 for Showcase]  [718x480 for Tiny Slider]',
                "config" => [
                    "folders" => ['images'],
                    "mode" => 'upload',
                ],
                "assignmentConfig" => [
                    "required" => true,
                ],
            ],
            [
                'name' => 'Title',
                'namespace' => 'repeater',
                'slug' => 'title',
                'type' => 'anomaly.field_type.text',
                "assignmentConfig" => [
                    "required" => true,
                    "translatable" => true,
                ],
            ],
            [
                'name' => 'Link',
                'namespace' => 'repeater',
                'slug' => 'link',
                'type' => 'anomaly.field_type.url',
                "config" => [
                    "default_value" => '#',
                ],
                "assignmentConfig" => [
                    "required" => true,
                ],
            ],
            [
                'name' => 'Description',
                'namespace' => 'repeater',
                'slug' => 'description',
                'type' => 'anomaly.field_type.text',
                "assignmentConfig" => [
                    "required" => true,
                    "translatable" => true,
                ],
            ],
            [
                'name' => 'Popular Categories',
                'namespace' => 'repeater',
                'slug' => 'popular_categories',
                'type' => 'anomaly.field_type.relationship',
                'config' => [
                    "related" => CategoryModel::class,
                ],
                "assignmentConfig" => [
                    "required" => true,
                ],
            ],
        ];

        foreach ($repeatersFields as $repeatersField) {
            $field = $this->fieldRepository->findBySlugAndNamespace($repeatersField['slug'], 'repeater');
            if (!$field) {
                $fieldValue = [
                    'name' => $repeatersField['name'],
                    'namespace' => 'repeater',
                    'slug' => $repeatersField['slug'],
                    'type' => $repeatersField['type'],
                    'locked' => 0,
                ];
                if (isset($repeatersField['config'])) {
                    $fieldValue['config'] = $repeatersField['config'];
                }
                if (isset($repeatersField['instructions'])) {
                    $fieldValue['instructions'] = $repeatersField['instructions'];
                }
                $field = $this->fieldRepository->create($fieldValue);
            }
            foreach ($repeaters as $repeater) {
                if (in_array($field->slug, $repeater['fields'])) {
                    $assign = $this->assignmentRepository->findByStreamAndField(
                        $repeatersObjects[$repeater['slug']],
                        $field
                    );
                    if (!$assign) {
                        $this->assignmentRepository->create(array_merge([
                            'stream_id' => $repeatersObjects[$repeater['slug']]->getId(),
                            'field_id' => $field->id,
                            'label' => $field->name,
                        ], $repeatersField['assignmentConfig']));
                    }
                }
            }
        }

        // Assign pages fields
        $defaultPagesStream = $this->streamRepository->findBySlugAndNamespace($app.'pages', 'pages');

        $pagesFields = [
            [
                'name' => 'Homepage Banner',
                'slug' => 'homepage_banner',
                "config" => [
                    "related" => RepeaterHomepageBannerEntryModel::class,
                    "max" => 14,
                ]
            ],
            [
                'name' => 'Showcase',
                'slug' => 'showcase',
                "config" => [
                    "related" => RepeaterShowcaseEntryModel::class,
                    "max" => 4,
                ]
            ],
            [
                'name' => 'Tiny Slider',
                'slug' => 'tiny_slider',
                "config" => [
                    "related" => RepeaterTinySliderEntryModel::class,
                    "max" => 3,
                ]
            ],
            [
                'name' => 'Popular Categories',
                'slug' => 'popular_categories',
                'config' => [
                    'related' => RepeaterPopularCategoriesEntryModel::class,
                ]
            ],
        ];

        foreach ($pagesFields as $pagesField) {
            $field = $this->fieldRepository->findBySlugAndNamespace($pagesField['slug'], 'pages');
            if (!$field) {
                $field = $this->fieldRepository->create([
                    'name' => $pagesField['name'],
                    'namespace' => 'pages',
                    'slug' => $pagesField['slug'],
                    'type' => 'anomaly.field_type.repeater',
                    'locked' => 0,
                    "config" => $pagesField['config']
                ]);
            }
            $assign = $this->assignmentRepository->findByStreamAndField($defaultPagesStream, $field);
            if (!$assign) {
                $this->assignmentRepository->create([
                    'stream_id' => $defaultPagesStream->getId(),
                    'field_id' => $field->id,
                    'label' => $pagesField['name'],
                ]);
            }
        }

        if (empty($this->menus->findBySlug('horizontal_sub_menu'))){
            $menu = $this->menus->create([
                config('app.locale', 'en')   => [
                    'name'        => 'Horizontal sub-menus',
                ],
                'slug' => 'horizontal_sub_menu',
            ]);

            $entryRepo = new EntryRepository();
            $entryRepo->setModel(new UrlLinkTypeModel());
            $link = $entryRepo->create([
                config('app.locale', 'en')   => [
                    'title' => 'About us',
                ],
                'url' => '#',
            ]);

            $this->links->create([
                'menu' => $menu,
                'target' => '_self',
                'entry'  => $link,
                'type'   => 'anomaly.extension.url_link_type',
            ]);
        }


        // Activate
        $this->settingRepository->set('streams::standard_theme', 'visiosoft.theme.gg');



        $demo = str_replace('{application_reference}', $app, file_get_contents(realpath(dirname(__DIR__)) . '/src/Seed/repeaters_seeder.sql'));
        Model::unguard();
        DB::unprepared($demo);
        Model::reguard();

        $zip = new \ZipArchive();
        $zip->open(realpath(dirname(__DIR__)) . '/src/Seed/images.zip', ZipArchive::CREATE);
        $zip->extractTo(storage_path('streams/'.app(Application::class)->getReference().'/files-module/local/'));
        $zip->close();


        Artisan::call('assets:clear');


    }
}
