<?php namespace Visiosoft\GgTheme;

use Anomaly\SettingsModule\Setting\Contract\SettingRepositoryInterface;
use Anomaly\Streams\Platform\Assignment\Contract\AssignmentRepositoryInterface;
use Anomaly\Streams\Platform\Database\Seeder\Seeder;
use Anomaly\Streams\Platform\Field\Contract\FieldRepositoryInterface;
use Anomaly\Streams\Platform\Model\Repeater\RepeaterHomepageBannerEntryModel;
use Anomaly\Streams\Platform\Model\Repeater\RepeaterShowcaseEntryModel;
use Anomaly\Streams\Platform\Model\Repeater\RepeaterTinySliderEntryModel;
use Anomaly\Streams\Platform\Stream\Contract\StreamRepositoryInterface;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use ZipArchive;

class GgThemeSeeder extends Seeder
{
    private $settingRepository;
    private $streamRepository;
    private $fieldRepository;
    private $assignmentRepository;

    public function __construct(
        SettingRepositoryInterface $settingRepository,
        StreamRepositoryInterface $streamRepository,
        FieldRepositoryInterface $fieldRepository,
        AssignmentRepositoryInterface $assignmentRepository
    )
    {
        parent::__construct();
        $this->settingRepository = $settingRepository;
        $this->streamRepository = $streamRepository;
        $this->fieldRepository = $fieldRepository;
        $this->assignmentRepository = $assignmentRepository;
    }

    public function run()
    {
        // Create repeaters
        $repeaters = [
            [
                'name' => 'Homepage Banner',
                'slug' => 'homepage_banner',
                'translatable' => false,
                'fields' => ['large_image', 'medium_image', 'thumbnail_image', 'link']
            ],
            [
                'name' => 'Tiny Slider',
                'slug' => 'tiny_slider',
                'translatable' => true,
                'fields' => ['background', 'link', 'description', 'title']
            ],
            [
                'name' => 'Showcase',
                'slug' => 'showcase',
                'translatable' => true,
                'fields' => ['background', 'title', 'link']
            ],
        ];

        $repeatersObjects = array();
        foreach ($repeaters as $repeater) {
            $exists = $this->streamRepository->findBySlugAndNamespace($repeater['slug'], 'repeater');
            $repeatersObjects[$repeater['slug']] = $exists ?: $this->streamRepository->create([
                'name' => $repeater['name'],
                'namespace' => 'repeater',
                'slug' => $repeater['slug'],
                'prefix' => 'repeater_',
                'translatable' => $repeater['translatable'],
            ]);
        }

        // Assign repeater fields
        $repeatersFields = [
            [
                'name' => 'Large Image',
                'namespace' => 'repeater',
                'slug' => 'large_image',
                'type' => 'anomaly.field_type.file',
                'instructions' => 'Recommended 1000x350',
                "config" => [
                    "folders" => ['images'],
                    "mode" => 'upload',
                ],
                "assignmentConfig" => [
                    "required" => true,
                ],
            ],
            [
                'name' => 'Medium Image',
                'namespace' => 'repeater',
                'slug' => 'medium_image',
                'type' => 'anomaly.field_type.file',
                'instructions' => 'Recommended 510x384',
                "config" => [
                    "folders" => ['images'],
                    "mode" => 'upload',
                ],
                "assignmentConfig" => [
                    "required" => true,
                ],
            ],
            [
                'name' => 'Thumbnail Image',
                'namespace' => 'repeater',
                'slug' => 'thumbnail_image',
                'type' => 'anomaly.field_type.file',
                'instructions' => 'Recommended 65x65',
                "config" => [
                    "folders" => ['images'],
                    "mode" => 'upload',
                ],
                "assignmentConfig" => [
                    "required" => true,
                ],
            ],
            [
                'name' => 'Background',
                'namespace' => 'repeater',
                'slug' => 'background',
                'type' => 'anomaly.field_type.file',
                'instructions' => 'Recommended [280x376 for Showcase]  [718x480 for Tiny Slider]',
                "config" => [
                    "folders" => ['images'],
                    "mode" => 'upload',
                ],
                "assignmentConfig" => [
                    "required" => true,
                ],
            ],
            [
                'name' => 'Title',
                'namespace' => 'repeater',
                'slug' => 'title',
                'type' => 'anomaly.field_type.text',
                "assignmentConfig" => [
                    "required" => true,
                    "translatable" => true,
                ],
            ],
            [
                'name' => 'Link',
                'namespace' => 'repeater',
                'slug' => 'link',
                'type' => 'anomaly.field_type.url',
                "config" => [
                    "default_value" => '#',
                ],
                "assignmentConfig" => [
                    "required" => true,
                ],
            ],
            [
                'name' => 'Description',
                'namespace' => 'repeater',
                'slug' => 'description',
                'type' => 'anomaly.field_type.text',
                "assignmentConfig" => [
                    "required" => true,
                    "translatable" => true,
                ],
            ],
        ];

        foreach ($repeatersFields as $repeatersField) {
            $field = $this->fieldRepository->findBySlugAndNamespace($repeatersField['slug'], 'repeater');
            if (!$field) {
                $fieldValue = [
                    'name' => $repeatersField['name'],
                    'namespace' => 'repeater',
                    'slug' => $repeatersField['slug'],
                    'type' => $repeatersField['type'],
                    'locked' => 0,
                ];
                if (isset($repeatersField['config'])) {
                    $fieldValue['config'] = $repeatersField['config'];
                }
                if (isset($repeatersField['instructions'])) {
                    $fieldValue['instructions'] = $repeatersField['instructions'];
                }
                $field = $this->fieldRepository->create($fieldValue);
            }
            foreach ($repeaters as $repeater) {
                if (in_array($field->slug, $repeater['fields'])) {
                    $assign = $this->assignmentRepository->findByStreamAndField(
                        $repeatersObjects[$repeater['slug']],
                        $field
                    );
                    if (!$assign) {
                        $this->assignmentRepository->create(array_merge([
                            'stream_id' => $repeatersObjects[$repeater['slug']]->getId(),
                            'field_id' => $field->id,
                            'label' => $field->name,
                        ], $repeatersField['assignmentConfig']));
                    }
                }
            }
        }

        // Assign pages fields
        $defaultPagesStream = $this->streamRepository->findBySlugAndNamespace('default_pages', 'pages');

        $pagesFields = [
            [
                'name' => 'Homepage Banner',
                'slug' => 'homepage_banner',
                "config" => [
                    "related" => RepeaterHomepageBannerEntryModel::class,
                    "max" => 14,
                ]
            ],
            [
                'name' => 'Showcase',
                'slug' => 'showcase',
                "config" => [
                    "related" => RepeaterShowcaseEntryModel::class,
                    "max" => 4,
                ]
            ],
            [
                'name' => 'Tiny Slider',
                'slug' => 'tiny_slider',
                "config" => [
                    "related" => RepeaterTinySliderEntryModel::class,
                    "max" => 3,
                ]
            ],
        ];

        foreach ($pagesFields as $pagesField) {
            $field = $this->fieldRepository->findBySlugAndNamespace($pagesField['slug'], 'pages');
            if (!$field) {
                $field = $this->fieldRepository->create([
                    'name' => $pagesField['name'],
                    'namespace' => 'pages',
                    'slug' => $pagesField['slug'],
                    'type' => 'anomaly.field_type.repeater',
                    'locked' => 0,
                    "config" => $pagesField['config']
                ]);
            }
            $assign = $this->assignmentRepository->findByStreamAndField($defaultPagesStream, $field);
            if (!$assign) {
                $this->assignmentRepository->create([
                    'stream_id' => $defaultPagesStream->getId(),
                    'field_id' => $field->id,
                    'label' => $pagesField['name'],
                ]);
            }
        }

        // Activate
        $this->settingRepository->set('streams::standard_theme', 'visiosoft.theme.gg');


        // Seeder
        Model::unguard();
        DB::unprepared(file_get_contents(__DIR__.'/gidiyor_advs.sql'));
//        DB::unprepared(file_get_contents(__DIR__.'/gidiyor_cats.sql'));
        DB::unprepared(file_get_contents(__DIR__.'/gidiyor_files.sql'));
        DB::unprepared(file_get_contents(__DIR__.'/gidiyor_url.sql'));
        DB::unprepared(file_get_contents(__DIR__.'/gidiyor_repeater.sql'));
        Model::reguard();

        $zip = new ZipArchive();
        $zip->open(__DIR__.'/images.zip', ZipArchive::CREATE);
        $zip->extractTo(storage_path('streams/default/files-module/local/'));
        $zip->close();





        Artisan::call('assets:clear');
    }
}