<?php namespace Anomaly\ForumModule\Discussion;

use Anomaly\ForumModule\Channel\Contract\ChannelInterface;
use Anomaly\ForumModule\Comment\CommentCollection;
use Anomaly\ForumModule\Comment\CommentModel;
use Anomaly\ForumModule\Comment\Contract\CommentInterface;
use Anomaly\ForumModule\Discussion\Contract\DiscussionInterface;
use Anomaly\ForumModule\Participation\Contract\ParticipationInterface;
use Anomaly\ForumModule\Participation\ParticipationCollection;
use Anomaly\ForumModule\Participation\ParticipationModel;
use Anomaly\Streams\Platform\Model\Forum\ForumDiscussionsEntryModel;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;

/**
 * Class DiscussionModel
 *
 * @link   http://pyrocms.com/
 * @author PyroCMS, Inc. <support@pyrocms.com>
 * @author Ryan Thompson <ryan@pyrocms.com>
 */
class DiscussionModel extends ForumDiscussionsEntryModel implements DiscussionInterface
{

    /**
     * Eager load these relations.
     *
     * @var array
     */
    protected $with = [
        'participation',
        'participants',
    ];

    /**
     * Return the read flag.
     *
     * @return bool
     */
    public function isRead()
    {
        if (!$participation = $this->getParticipation()) {
            return false;
        }

        return $participation->isRead();
    }

    /**
     * Return the liked flag.
     *
     * @return bool
     */
    public function isLiked()
    {
        return (bool)$this
            ->reactions()
            ->select('id')
            ->whereType('thumbsup')
            ->whereUserId(auth()->id())
            ->first();
    }

    /**
     * Return the following flag.
     *
     * @return bool
     */
    public function isFollowing()
    {
        if (!$participation = $this->getParticipation()) {
            return false;
        }

        return $participation->isFollowing();
    }

    /**
     * Get the slug.
     *
     * @return string
     */
    public function getSlug()
    {
        return $this->slug;
    }

    /**
     * Get the title.
     *
     * @return string
     */
    public function getTitle()
    {
        return $this->title;
    }

    /**
     * Get the view count.
     *
     * @return int
     */
    public function getViewCount()
    {
        return $this->view_count;
    }

    /**
     * Get the comment count.
     *
     * @return int
     */
    public function getCommentCount()
    {
        return $this->comment_count;
    }

    /**
     * Get the content.
     *
     * @return string
     */
    public function getContent()
    {
        return $this->content;
    }

    /**
     * Return the answered flag.
     *
     * @return bool
     */
    public function isAnswered()
    {
        return (bool)$this->answer_id;
    }

    /**
     * Get the related answer.
     *
     * @return CommentInterface
     */
    public function getAnswer()
    {
        return $this->answer;
    }

    /**
     * Get the related comments.
     *
     * @return CommentCollection
     */
    public function getComments()
    {
        return $this->comments;
    }

    /**
     * Return the comments relation.
     *
     * @return HasMany
     */
    public function comments()
    {
        return $this->hasMany(CommentModel::class, 'discussion_id');
    }

    /**
     * Get the last related comment.
     *
     * @return CommentInterface|null
     */
    public function getLastComment()
    {
        return $this
            ->lastComment()
            ->getResults();
    }

    /**
     * Return the last comment relation.
     *
     * @return HasOne
     */
    public function lastComment()
    {
        return $this
            ->hasOne(CommentModel::class, 'discussion_id')
            ->orderBy('created_at', 'DESC');
    }

    /**
     * Get the related participation.
     *
     * @return ParticipationInterface
     */
    public function getParticipation()
    {
        return $this->participation;
    }

    /**
     * Return the participation relation.
     *
     * @return HasOne
     */
    public function participation()
    {
        return $this
            ->hasOne(ParticipationModel::class, 'discussion_id')
            ->where('user_id', app('auth')->id());
    }

    /**
     * Get the related participants.
     *
     * @return ParticipationCollection
     */
    public function getParticipants()
    {
        return $this->participants;
    }

    /**
     * Return the participants relation.
     *
     * @return hasMany
     */
    public function participants()
    {
        return $this->hasMany(ParticipationModel::class, 'discussion_id');
    }

    /**
     * Return the flagged flag.
     *
     * @return bool
     */
    public function isFlagged()
    {
        return $this->flagged;
    }

    /**
     * Return the approved flag.
     *
     * @return bool
     */
    public function isApproved()
    {
        return $this->approved;
    }

    /**
     * Return the pinned flag.
     *
     * @return bool
     */
    public function isPinned()
    {
        return $this->pinned;
    }

    /**
     * Return routable array.
     *
     * @return array
     */
    public function toRoutableArray()
    {
        $routable = parent::toRoutableArray();

        $routable['channel'] = $this->getChannelSlug();

        return $routable;
    }

    /**
     * Get the channel slug.
     *
     * @return string The related channel slug.
     */
    public function getChannelSlug()
    {
        $channel = $this->getChannel();

        return $channel->getSlug();
    }

    /**
     * Get the related channel.
     *
     * @return ChannelInterface
     */
    public function getChannel()
    {
        return $this->channel;
    }

    /**
     * Get the related followers.
     *
     * @return ParticipationCollection
     */
    public function getFollowers()
    {
        return $this->followers;
    }

    /**
     * Return the followers relation.
     *
     * @return HasOne
     */
    public function followers()
    {
        return $this
            ->participants()
            ->where('following', true);
    }

    /**
     * Get the author's username.
     *
     * @return string
     */
    public function getAuthorUsername()
    {
        return $this
            ->getCreatedBy()
            ->getUsername();
    }

}
