<?php namespace Visiosoft\ForumModule\Http\Controller;

use Visiosoft\ForumModule\Channel\Contract\ChannelInterface;
use Visiosoft\ForumModule\Channel\Contract\ChannelRepositoryInterface;
use Visiosoft\ForumModule\Discussion\Command\IncrementViewCount;
use Visiosoft\ForumModule\Discussion\Command\UpdateParticipation;
use Visiosoft\ForumModule\Discussion\Contract\DiscussionInterface;
use Visiosoft\ForumModule\Discussion\Contract\DiscussionRepositoryInterface;
use Visiosoft\ForumModule\Discussion\DiscussionManager;
use Anomaly\Streams\Platform\Http\Controller\PublicController;
use Anomaly\Streams\Platform\Model\EloquentModel;
use Anomaly\Streams\Platform\User\Contract\UserInterface;
use Illuminate\Contracts\Auth\Guard;

/**
 * Class DiscussionsController
 *
 * @link   http://pyrocms.com/
 * @author PyroCMS, Inc. <support@pyrocms.com>
 * @author Ryan Thompson <ryan@pyrocms.com>
 */
class DiscussionsController extends PublicController
{

    /**
     * Return an index of discussions.
     *
     * @return \Illuminate\Contracts\View\View|mixed
     */
    public function index()
    {
        $this->breadcrumbs->add(
            'visiosoft.module.forum::breadcrumb.forum',
            $this->url->route('visiosoft.module.forum::discussions.index')
        );

        $this->template->set('meta_title', 'visiosoft.module.forum::breadcrumb.forum');

        return $this->view->make('visiosoft.module.forum::discussions.index');
    }

    /**
     * View an existing discussion.
     *
     * @param DiscussionRepositoryInterface $discussions
     * @param ChannelRepositoryInterface $channels
     * @param $channel
     * @param $slug
     * @return \Illuminate\Contracts\View\View
     */
    public function view(
        DiscussionRepositoryInterface $discussions,
        ChannelRepositoryInterface $channels,
        $channel,
        $slug
    ) {
        if (!$discussion = $discussions->findBySlug($slug)) {
            abort(404);
        }

        if (!$channel = $channels->findBySlug($channel)) {
            abort(404);
        }

        if ($discussion->getChannelSlug() != $channel->getSlug()) {
            abort(400);
        }

        $this->breadcrumbs->add(
            'visiosoft.module.forum::breadcrumb.forum',
            $this->url->route('visiosoft.module.forum::discussions.index')
        );

        $this->breadcrumbs->add(
            $channel->getName(),
            $channel->route('view')
        );

        $this->breadcrumbs->add(
            $discussion->getTitle(),
            $discussion->route('view')
        );

        $this->template->set('discussion', $discussion);
        $this->template->set('meta_title', $discussion->getTitle());
        $this->template->set('meta_description', $discussion->getContent());

        $this->dispatchNow(new UpdateParticipation($discussion));
        $this->dispatchNow(new IncrementViewCount($discussion));

        return $this->view->make('visiosoft.module.forum::discussions.view', compact('discussion'));
    }

    /**
     * Create a new discussion.
     *
     * @return \Illuminate\Contracts\View\View
     */
    public function create()
    {
        $this->breadcrumbs->add(
            'visiosoft.module.forum::breadcrumb.forum',
            $this->url->route('visiosoft.module.forum::discussions.index')
        );

        $this->breadcrumbs->add(
            'visiosoft.module.forum::breadcrumb.new_discussion',
            $this->request->path()
        );

        $this->template->set('meta_title', 'visiosoft.module.forum::breadcrumb.new_discussion');

        return $this->view->make('visiosoft.module.forum::discussions.create');
    }

    /**
     * Edit an existing discussion.
     *
     * @param DiscussionRepositoryInterface $discussions
     * @param Guard $auth
     * @param $id
     * @return \Illuminate\Contracts\View\View
     */
    public function edit(DiscussionRepositoryInterface $discussions, Guard $auth, $id)
    {
        if (!$discussion = $discussions->find($id)) {
            abort(404);
        }

        /* @var UserInterface $user */
        if (!$user = $auth->user()) {
            abort(403);
        }

        if ($user->getId() != $discussion->created_by_id) {
            abort(403);
        }

        $this->breadcrumbs->add(
            'visiosoft.module.forum::breadcrumb.forum',
            $this->url->route('visiosoft.module.forum::discussions.index')
        );

        $this->breadcrumbs->add(
            'visiosoft.module.forum::breadcrumb.edit_discussion',
            $this->request->path()
        );

        $this->template->set('meta_title', 'visiosoft.module.forum::breadcrumb.edit_discussion');

        return $this->view->make('visiosoft.module.forum::discussions.edit', compact('discussion'));
    }

    /**
     * Delete a discussion.
     *
     * @param DiscussionRepositoryInterface $discussions
     * @param $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function delete(DiscussionRepositoryInterface $discussions, $id)
    {
        /* @var DiscussionInterface|EloquentModel $discussion */
        if (!$discussion = $discussions->find($id)) {
            abort(404);
        }

        /* @var ChannelInterface $channel */
        $channel = $discussion->getChannel();

        $discussions->delete($discussion);

        return $this->redirect->to($channel->route('view'));
    }

    /**
     * Lock a discussion.
     *
     * @param DiscussionRepositoryInterface $discussions
     * @param $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function lock(DiscussionRepositoryInterface $discussions, $id)
    {
        /* @var DiscussionInterface|EloquentModel $discussion */
        if (!$discussion = $discussions->find($id)) {
            abort(404);
        }

        $discussion->setAttribute('locked', true);

        $discussions->save($discussion);

        return $this->redirect->back();
    }

    /**
     * Unlock a discussion.
     *
     * @param DiscussionRepositoryInterface $discussions
     * @param $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function unlock(DiscussionRepositoryInterface $discussions, $id)
    {
        /* @var DiscussionInterface|EloquentModel $discussion */
        if (!$discussion = $discussions->find($id)) {
            abort(404);
        }

        $discussion->setAttribute('locked', false);

        $discussions->save($discussion);

        return $this->redirect->back();
    }

    /**
     * Pin a discussion.
     *
     * @param DiscussionRepositoryInterface $discussions
     * @param $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function pin(DiscussionRepositoryInterface $discussions, $id)
    {
        /* @var DiscussionInterface|EloquentModel $discussion */
        if (!$discussion = $discussions->find($id)) {
            abort(404);
        }

        $discussion->setAttribute('pinned', true);

        $discussions->save($discussion);

        return $this->redirect->back();
    }

    /**
     * Unpin a discussion.
     *
     * @param DiscussionRepositoryInterface $discussions
     * @param $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function unpin(DiscussionRepositoryInterface $discussions, $id)
    {
        /* @var DiscussionInterface|EloquentModel $discussion */
        if (!$discussion = $discussions->find($id)) {
            abort(404);
        }

        $discussion->setAttribute('pinned', false);

        $discussions->save($discussion);

        return $this->redirect->back();
    }

    /**
     * Follow a discussion.
     *
     * @param DiscussionRepositoryInterface $discussions
     * @param DiscussionManager $manager
     * @param $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function follow(
        DiscussionRepositoryInterface $discussions,
        DiscussionManager $manager,
        $id
    ) {
        /* @var DiscussionInterface $discussion */
        if (!$discussion = $discussions->find($id)) {
            abort(404);
        }

        $manager->follow($discussion);

        return $this->redirect->back();
    }

    /**
     * Unfollow a discussion.
     *
     * @param DiscussionRepositoryInterface $discussions
     * @param DiscussionManager $manager
     * @param $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function unfollow(
        DiscussionRepositoryInterface $discussions,
        DiscussionManager $manager,
        $id
    ) {
        /* @var DiscussionInterface $discussion */
        if (!$discussion = $discussions->find($id)) {
            abort(404);
        }

        $manager->unfollow($discussion);

        return $this->redirect->back();
    }

    /**
     * Create a new flag.
     *
     * @param DiscussionRepositoryInterface $discussions
     * @param DiscussionManager $manager
     * @param $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function flag(DiscussionRepositoryInterface $discussions, DiscussionManager $manager, $id)
    {
        /* @var DiscussionInterface $discussion */
        if (!$discussion = $discussions->find($id)) {
            abort(404);
        }

        $manager->flag($discussion);

        return $this->redirect->back();
    }

    /**
     * Clear flags for a discussion.
     *
     * @param DiscussionRepositoryInterface $discussions
     * @param DiscussionManager $manager
     * @param $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function unflag(
        DiscussionRepositoryInterface $discussions,
        DiscussionManager $manager,
        $id
    ) {
        /* @var DiscussionInterface $discussion */
        if (!$discussion = $discussions->find($id)) {
            abort(404);
        }

        $manager->unflag($discussion);

        return $this->redirect->back();
    }

    /**
     * Like a discussion.
     *
     * @param DiscussionRepositoryInterface $discussions
     * @param DiscussionManager $manager
     * @param $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function like(DiscussionRepositoryInterface $discussions, DiscussionManager $manager, $id)
    {
        /* @var DiscussionInterface $discussion */
        if (!$discussion = $discussions->find($id)) {
            abort(404);
        }

        $manager->like($discussion);

        return $this->redirect->back();
    }

    /**
     * Unlike a discussion.
     *
     * @param DiscussionRepositoryInterface $discussions
     * @param DiscussionManager $manager
     * @param $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function unlike(DiscussionRepositoryInterface $discussions, DiscussionManager $manager, $id)
    {
        /* @var DiscussionInterface $discussion */
        if (!$discussion = $discussions->find($id)) {
            abort(404);
        }

        $manager->unlike($discussion);

        return $this->redirect->back();
    }

}
