<?php namespace Visiosoft\ForumModule\Http\Controller;

use Visiosoft\ForumModule\Comment\CommentManager;
use Visiosoft\ForumModule\Comment\Contract\CommentInterface;
use Visiosoft\ForumModule\Comment\Contract\CommentRepositoryInterface;
use Visiosoft\ForumModule\Discussion\Contract\DiscussionInterface;
use Visiosoft\ForumModule\Discussion\Contract\DiscussionRepositoryInterface;
use Anomaly\Streams\Platform\Http\Controller\PublicController;
use Anomaly\Streams\Platform\Model\EloquentModel;
use Anomaly\Streams\Platform\User\Contract\UserInterface;
use Illuminate\Contracts\Auth\Guard;

class CommentsController extends PublicController
{

    /**
     * Edit an existing discussion.
     *
     * @param CommentRepositoryInterface $comments
     * @param Guard $auth
     * @param $id
     * @return \Illuminate\Contracts\View\View
     */
    public function edit(CommentRepositoryInterface $comments, Guard $auth, $id)
    {
        if (!$comment = $comments->find($id)) {
            abort(404);
        }

        /* @var UserInterface $user */
        if (!$user = $auth->user()) {
            abort(403);
        }

        if ($user->getId() != $comment->created_by_id) {
            abort(403);
        }

        $this->breadcrumbs->add(
            'visiosoft.module.forum::breadcrumb.forum',
            $this->url->route('visiosoft.module.forum::discussions.index')
        );

        $this->breadcrumbs->add('visiosoft.module.forum::breadcrumb.edit_comment', $this->request->path());
        $this->template->set('meta_title', 'visiosoft.module.forum::breadcrumb.edit_comment');

        return $this->view->make('visiosoft.module.forum::comments.edit', compact('comment'));
    }

    /**
     * Delete a discussion.
     *
     * @param CommentRepositoryInterface $comments
     * @param $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function delete(CommentRepositoryInterface $comments, $id)
    {
        /* @var CommentInterface|EloquentModel $comment */
        if (!$comment = $comments->find($id)) {
            abort(404);
        }

        $comments->delete($comment);

        return $this->redirect->back();
    }

    /**
     * Mark a comment as an answer.
     *
     * @param DiscussionRepositoryInterface $discussions
     * @param CommentRepositoryInterface $comments
     * @param Guard $auth
     * @param $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function answer(
        DiscussionRepositoryInterface $discussions,
        CommentRepositoryInterface $comments,
        Guard $auth,
        $id
    ) {
        /* @var CommentInterface $comment */
        if (!$comment = $comments->find($id)) {
            abort(404);
        }

        /* @var DiscussionInterface|EloquentModel $discussion */
        $discussion = $comment->getDiscussion();

        /* @var UserInterface $user */
        if (!$user = $auth->user()) {
            abort(403);
        }

        if (
            $user->getId() != $discussion->created_by_id &&
            !$user->hasPermission('visiosoft.module.forum::comments.answer')
        ) {
            abort(403);
        }

        $discussion->setAttribute('answer', $comment);

        $discussions->save($discussion);

        return $this->redirect->back();
    }

    /**
     * Create a new flag.
     *
     * @param CommentRepositoryInterface $comments
     * @param CommentManager $manager
     * @param $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function flag(CommentRepositoryInterface $comments, CommentManager $manager, $id)
    {
        /* @var CommentInterface $comment */
        if (!$comment = $comments->find($id)) {
            abort(404);
        }

        $manager->flag($comment);

        return $this->redirect->back();
    }

    /**
     * Clear flags for a discussion.
     *
     * @param CommentRepositoryInterface $comments
     * @param CommentManager $manager
     * @param $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function unflag(CommentRepositoryInterface $comments, CommentManager $manager, $id)
    {
        /* @var CommentInterface $comment */
        if (!$comment = $comments->find($id)) {
            abort(404);
        }

        $manager->unflag($comment);

        return $this->redirect->back();
    }

    /**
     * Like a discussion.
     *
     * @param CommentRepositoryInterface $comments
     * @param CommentManager $manager
     * @param $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function like(CommentRepositoryInterface $comments, CommentManager $manager, $id)
    {
        /* @var CommentInterface $comment */
        if (!$comment = $comments->find($id)) {
            abort(404);
        }

        $manager->like($comment);

        return $this->redirect->back();
    }

    /**
     * Unlike a discussion.
     *
     * @param CommentRepositoryInterface $comments
     * @param CommentManager $manager
     * @param $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function unlike(CommentRepositoryInterface $comments, CommentManager $manager, $id)
    {
        /* @var CommentInterface $comment */
        if (!$comment = $comments->find($id)) {
            abort(404);
        }

        $manager->unlike($comment);

        return $this->redirect->back();
    }

}
