<?php namespace Visiosoft\EvdepismisTheme\Http\Controller;

use Anomaly\Streams\Platform\Http\Controller\PublicController;
use Illuminate\Contracts\Bus\Dispatcher;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Cookie;
use Illuminate\Support\Facades\Notification;
use Visiosoft\CatsModule\Category\Contract\CategoryRepositoryInterface;
use Visiosoft\CommentsModule\Comment\Command\getStoreRateFiltered;
use Visiosoft\EvdepismisTheme\Notification\BookTableMail;
use Visiosoft\FavsModule\Favorite\Contract\FavoriteRepositoryInterface;
use Visiosoft\StoreModule\Store\Contract\StoreRepositoryInterface;
use Visiosoft\StoreModule\Category\Contract\CategoryRepositoryInterface as StoreCategory;
use Visiosoft\StoreModule\Store\StoreModel;

class EvdepismisController extends PublicController
{

    private $categoryRepository;
    private $storeCategory;
    private $storeModel;
    private $favoriteRepository;


    public function __construct(
        CategoryRepositoryInterface $categoryRepository,
        StoreCategory               $storeCategory,
        StoreModel                  $storeModel,
        FavoriteRepositoryInterface $favoriteRepository
    )
    {
        parent::__construct();
        $this->categoryRepository = $categoryRepository;
        $this->storeCategory = $storeCategory;
        $this->storeModel = $storeModel;
        $this->favoriteRepository = $favoriteRepository;
    }

    public function returnMapIndex(Request $request)
    {
        $location_name = $request->input('location');

        return view('visiosoft.theme.evdepismis::map/map', compact('location_name'));
    }

    public function getStoresByCoordinates(StoreRepositoryInterface $storeRepository)
    {
        \request()->validate([
            'lat' => 'required',
            'lng' => 'required',
        ]);

        try {
            $stores = $storeRepository->getStoresByCoordinates(\request()->lat, \request()->lng);

            $storesHtml = view('theme::addons/visiosoft/store-module/list/list-item', compact('stores'))->render();

            return [
                'success' => true,
                'stores' => $stores,
                'storesHtml' => $storesHtml,
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'msg' => $e->getMessage()
            ];
        }
    }

    public function listStores(Request $request) {
        $params = $request->toArray();


        $cats = $this->categoryRepository->getMainCategories();

        $filters = [];
        $fields = $this->storeModel->getAssignments()->notLocked();

        foreach ($fields as $field) {
            if ($field->field->type === 'anomaly.field_type.checkboxes'
                || $field->field->type === 'anomaly.field_type.select') {
                $filters[$field->getFieldType()->getSlug()][$field->field->slug] = $field->getFieldType()->getOptions();
            } elseif ($field->field->type === 'visiosoft.field_type.range') {
                $filters[$field->getFieldType()->getSlug()][$field->field->slug] = $field->getFieldType();
            }
        }

        $stores = $this->searchStores($params, null, false, $filters);

        $storeCats = $this->storeCategory->all();
        $hideLocationFilter = false;
        if (
            ($locationFilterCats = setting_value('visiosoft.module.store::hide_location_filter_categories'))
            && is_array($locationFilterCats)
            && count($locationFilterCats)
            && \request()->cat
        ) {
            $hideLocationFilter = in_array(\request()->cat, $locationFilterCats);
        }

        //For AddBlock Override View
        $filter_params = [];
        $compact = compact('stores', 'request', 'cats', 'storeCats', 'fields', 'filters', 'hideLocationFilter', 'filter_params');

        Cookie::queue(Cookie::make('last_search', $request->getRequestUri(), 84000));

        $seoKeyword = setting_value('visiosoft.module.store::list_seo_keywords');
        if ($seoKeyword && count($seoKeyword)) {
            $this->template->set('meta_keywords', implode(', ', $seoKeyword));
        }

        return $this->view->make('visiosoft.module.store::list/list', $compact);

    }

    public function searchStores($param = null, $limit = null, $get_query = false, $filters = [])
    {
        $query = $this->storeModel;
        $query = $query->where('store_store.slug', '!=', "")->where('store_store.status', 'approved');
        $query = $query->rightJoin('store_store_translations', function ($join) {
            $join->on('store_store.id', '=', 'store_store_translations.entry_id');
            $join->where('store_store_translations.locale', '=', Request()->session()->get('_locale', setting_value('streams::default_locale')));
        });

        if (!empty($param['keyword'])) {
            $query = $query->where('store_store.summary', 'LIKE', '%' . $param['keyword'] . '%')
                ->orWhere('store_store_translations.name', 'like', '%' . $param['keyword'] . '%');
        }
        if (!empty($param['country'])) {
            $query = $query->where('country_id', $param['country']);
        }
        if (isset($param['city']) and (!empty($param['city']))) {
            $citiesIds = explode(',', $param['city'][0]);
            $query = $query->whereIn('city', $citiesIds);
        }
        if (!empty($param['cat'])) {
            if (is_array($param['cat'])) {
                $query = $query->whereIn('category', $param['cat']);
            } else {
                $query = $query->where('category', $param['cat']);
            }
        }
        if (!empty($param['gold'])) {
            $query = $query->where('gold', 1);
        }
        if (!empty($param['date'])) {
            if ($param['date'] === 1) {
                $query = $query->where('store_store.created_at', '>=', Carbon::now()->subYears(1));
            } elseif ($param['date'] === 2) {
                $query = $query->where('store_store.created_at', '>=', Carbon::now()->subYears(2));
            } elseif ($param['date'] === 3) {
                $query = $query->where('store_store.created_at', '>=', Carbon::now()->subYears(3));
            }
        }

        foreach ($filters as $filter) {
            foreach ($filter as $key => $value) {
                if (!empty($param[$key])) {
                    if (array_key_exists('rating', $param) && $key === 'rating') {
                        $query = app(Dispatcher::class)->dispatch(new getStoreRateFiltered($query, $param[$key]));
                    } else {
                        $query = $query->where(function ($q) use ($param, $key) {
                            if (is_array($param[$key])) {
                                foreach ($param[$key] as $item) {
                                    $q->orWhere($key, 'like', '%' . $item . '%');
                                }
                            } else {
                                $q->orWhere($key, 'like', '%' . $param[$key] . '%');
                            }
                        });
                    }
                }
            }
        }

        if ($get_query) {
            return $query;
        }

        $query = $query->orderBy('store_store.created_at', 'desc');

        return $query->paginate(setting_value('streams::per_page'));
    }

    public function bookTable(Request $request)
    {
        Notification::send(get_admins(), new BookTableMail($request->all()));
    }

    public function checkBookmark($id, $type)
    {
        return $this->favoriteRepository->newQuery()->where('type', $type)->where('item_id', $id)->first();
    }

    public function toggleBookmark(Request $request){
        if ($favStore = $this->checkBookmark($request->id, $request->type)) {
           return $favStore->delete();
        }

        return $this->favoriteRepository->create([
            'type' => $request->type,
            'item_id' => $request->id,
        ]);
    }

    public function changeOnlineStatus(Request $request){
        if ($store = $this->storeModel->newQuery()->find($request->id)) {
            $store->update(['is_online' => !$store->is_online]);
            $this->messages->success(trans('visiosoft.theme.evdepismis::field.success'));
        } else {
            $this->messages->error(trans('visiosoft.theme.evdepismis::field.error'));
        }
    }
}
