<?php namespace Visiosoft\DopingsModule\Http\Controller\Admin;

use Anomaly\Streams\Platform\Entry\Contract\EntryInterface;
use Anomaly\Streams\Platform\Entry\EntryModel;
use Anomaly\UsersModule\User\Contract\UserRepositoryInterface;
use Anomaly\UsersModule\User\UserModel;
use Carbon\Carbon;
use Illuminate\Support\Str;
use Visiosoft\AdvsModule\Adv\AdvModel;
use Visiosoft\AdvsModule\Adv\Contract\AdvRepositoryInterface;
use Visiosoft\DopingsModule\Doping\Contract\DopingRepositoryInterface;
use Visiosoft\DopingsModule\Doping\Form\DopingFormBuilder;
use Visiosoft\DopingsModule\Doping\Table\DopingTableBuilder;
use Anomaly\Streams\Platform\Http\Controller\AdminController;
use Visiosoft\DopingsModule\Events\DopingApproved;
use Visiosoft\DopingsModule\Package\Contract\PackageRepositoryInterface;
use Visiosoft\DopingsModule\Type\Contract\TypeRepositoryInterface;

class DopingsController extends AdminController
{

    private $dopingTypeRepository;
    private $dopingRepository;
    protected $packageRepository;
    protected $advRepository;

    public function __construct(
        TypeRepositoryInterface $dopingTypeRepository,
        PackageRepositoryInterface $packageRepository,
        DopingRepositoryInterface $dopingRepository,
        AdvRepositoryInterface $advRepository
    )
    {
        parent::__construct();
        $this->dopingTypeRepository = $dopingTypeRepository;
        $this->dopingRepository = $dopingRepository;
        $this->packageRepository = $packageRepository;
        $this->advRepository = $advRepository;
    }

    /**
     * Display an index of existing entries.
     *
     * @param DopingTableBuilder $table
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function index(DopingTableBuilder $table)
    {
        $table->setColumns([
            'name',
            'adv_name' => [
                'value' => function (EntryModel $entry, AdvModel $advModel) {
                    $isAd = $entry->adv_name_id ?
                        $advModel->getAdv($entry->adv_name_id, false, true) :
                        null;
                    if (!is_null($isAd)) {
                        $url = $advModel->getAdvDetailLinkByModel($isAd, 'list');
                        if ($isAd->deleted_at) {
                            return '<a class="font-weight-bold" href="#" >' . $isAd->name . "<a>";
                        }
                        return '<a class="font-weight-bold" href="' . $url . '" >' . $isAd->name . "<a>";
                    }
                    return "<font color='red'>" . trans('visiosoft.module.dopings::field.deleted_ad') . " [id=" . $entry->adv_name_id . "]</font>";
                }
            ],
            'user' => [
                'value' => function (EntryModel $entry, UserRepositoryInterface $user) {
                    if ($user = $user->find($entry->user_id)) {
                        return '<a class="font-weight-bold" href="/admin/users/edit/'.$entry->user_id.'" >' . $user->email . "<a>";
                    }
                    return "<span class='text-muted'>".$entry->user_id."</span>";
                }
            ],
            'status' => [
                'value' => function (EntryModel $entry) {
                    $icon = "fa fa-refresh";
                    if ($entry->status === "approved") {
                        $icon = "fa fa-check";
                    }
                    return '<i class="' . $icon . '"></i>';
                }
            ],
        ]);
        return $table->render();
    }

    public function choose($adId)
    {
        if (empty(\request()->all())) {
            return $this->view->make('module::admin/advs/choose', [
                'dopingTypes' => $this->dopingTypeRepository->all(),
                'adId' => $adId
            ]);
        } else {
            try {
                $this->createDopingForAds(\request()->type_id, [$adId]);

                $this->messages->success(trans('module::messages.doping_added_successfully'));
                return redirect()->route('visiosoft.module.advs::admin_advs');
            } catch (\Exception $e) {
                $this->messages->error($e->getMessage());
                return redirect()->route('visiosoft.module.advs::admin_advs');
            }
        }
    }

    public function bulkChoose()
    {
        return $this->view->make(
            'visiosoft.module.dopings::admin/advs/bulk-choose',
            [
                'dopingTypes' => $this->dopingTypeRepository->all(),
            ]
        );
    }

    public function bulkHandle($typeId)
    {
        try {
            $this->createDopingForAds($typeId, explode(',', request()->get('selected')));

            $this->messages->success(trans('module::messages.doping_added_successfully'));
            return redirect()->route('visiosoft.module.advs::admin_advs');
        } catch (\Exception $e) {
            $this->messages->error($e->getMessage());
            return redirect()->route('visiosoft.module.advs::admin_advs');
        }
    }

    public function createDopingForAds($dopingTypeId, $adsIds)
    {
        $dopingType = $this->dopingTypeRepository->find($dopingTypeId);
        if (!$dopingType) {
            throw new \Exception(trans('visiosoft.module.dopings::messages.doping_type_not_exist'));
        }
        foreach ($adsIds as $id) {
            $this->dopingRepository->create([
                'name' => $dopingType->name,
                'adv_name_id' => $id,
                'doping_type_id' => $dopingType->id,
                'price' => $dopingType->price,
                'currency' => $dopingType->currency,
                'user_id' => auth()->id(),
                'expiry_date' => Carbon::now()->addDays($dopingType->duration),
                'status' => 'approved',
            ]);
        }
    }

    /**
     * Create a new entry.
     *
     * @param DopingFormBuilder $form
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function create(DopingFormBuilder $form)
    {
        return $form->render();
    }

    /**
     * Edit an existing entry.
     *
     * @param DopingFormBuilder $form
     * @param        $id
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function edit(DopingFormBuilder $form, $id)
    {
        return $form->render($id);
    }

    public function approve($id)
    {
        $doping = $this->dopingRepository->find($id);

        $slug = $doping->getType()->getSlug();

        if ($slug === 'updated') {
            if ($ad = $doping->adv_name) {
                $ad->created_at = now();
                $ad->save();
            }
            $doping->approved();

            event(new DopingApproved($doping));
        } else {
            $now = Carbon::now();

            if ($doping->status !== 'approved') {

                if ($doping->package_id) {
                    if ($package = $this->packageRepository->find($doping->package_id)) {
                        $doping->update([
                            'status' => 'approved',
                            'expiry_date' => $now->add($package->interval, Str::lower($package->interval_period)),
                        ]);
                    }
                } else {
                    if ($type = $this->dopingTypeRepository->find($doping->doping_type_id)) {
                        $doping->update([
                            'status' => 'approved',
                            'expiry_date' => $now->addDays($type->duration),
                        ]);
                    }
                }

                event(new DopingApproved($doping));
            }
        }

        return back();
    }

    public function revert($id)
    {
        $doping = $this->dopingRepository->find($id);
        $doping->disapprove();
        return back();
    }

    public function queryAds(AdvRepositoryInterface $advRepository)
    {
        $keyword = request()->term;
        if ($keyword) {
            $query = $advRepository->newQuery();

            $query = $query->leftJoin('advs_advs_translations', function ($join) {
                $join->on('advs_advs.id', '=', 'advs_advs_translations.entry_id');
                $join->where('advs_advs_translations.locale', '=', Request()->session()->get('_locale', setting_value('streams::default_locale')));
            });

            return $query->select('advs_advs_translations.name as name', 'advs_advs.id')
                ->where('name', 'LIKE', "%$keyword%")
                ->orWhere('advs_advs.id', $keyword)
                ->pluck('name', 'advs_advs.id');
        }
    }
}
