<?php namespace Visiosoft\DopingsModule\Http\Controller\Admin;

use Anomaly\Streams\Platform\Entry\Contract\EntryInterface;
use Anomaly\Streams\Platform\Entry\EntryModel;
use Carbon\Carbon;
use Visiosoft\AdvsModule\Adv\AdvModel;
use Visiosoft\DopingsModule\Doping\Contract\DopingRepositoryInterface;
use Visiosoft\DopingsModule\Doping\Form\DopingFormBuilder;
use Visiosoft\DopingsModule\Doping\Table\DopingTableBuilder;
use Anomaly\Streams\Platform\Http\Controller\AdminController;
use Visiosoft\DopingsModule\Events\DopingApproved;
use Visiosoft\DopingsModule\Type\Contract\TypeRepositoryInterface;

class DopingsController extends AdminController
{

    private $dopingTypeRepository;
    private $dopingRepository;

    public function __construct(
        TypeRepositoryInterface $dopingTypeRepository,
        DopingRepositoryInterface $dopingRepository
    )
    {
        parent::__construct();
        $this->dopingTypeRepository = $dopingTypeRepository;
        $this->dopingRepository = $dopingRepository;
    }

    /**
     * Display an index of existing entries.
     *
     * @param DopingTableBuilder $table
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function index(DopingTableBuilder $table)
    {
        $table->setColumns([
            'name',
            'adv_name' => [
                'value' => function (EntryModel $entry, AdvModel $advModel) {
                    $isAd = $entry->adv_name_id ?
                        $advModel->getAdv($entry->adv_name_id, false, true) :
                        null;
                    if (!is_null($isAd)) {
                        $url = $advModel->getAdvDetailLinkByModel($isAd, 'list');
                        if ($isAd->deleted_at) {
                            return '<a class="font-weight-bold" href="#" >' . $isAd->name . "<a>";
                        }
                        return '<a class="font-weight-bold" href="' . $url . '" >' . $isAd->name . "<a>";
                    }
                    return "<font color='red'>" . trans('visiosoft.module.dopings::field.deleted_ad') . " [id=" . $entry->adv_name_id . "]</font>";
                }
            ],
        ]);
        $table->addButtons([
            'status' => [
                'text' => function (EntryInterface $entry) {
                    if ($entry->status == 'approved') {
                        return "visiosoft.module.dopings::button.revert";
                    } else {
                        return "visiosoft.module.dopings::button.approve";
                    }
                },
                'href' => function (EntryInterface $entry) {
                    if ($entry->status == 'approved') {
                        return "/admin/doping/revert/{entry.id}";
                    } else {
                        return "/admin/doping/approve/{entry.id}";
                    }
                },
                'type' => function (EntryInterface $entry) {
                    if ($entry->status == 'approved') {
                        return "danger";
                    } else {
                        return "success";
                    }
                },
            ],
            'edit'
        ]);
        return $table->render();
    }

    public function choose($adId)
    {
        if (empty(\request()->all())) {
            return $this->view->make('module::admin/advs/choose', [
                'dopingTypes' => $this->dopingTypeRepository->all(),
                'adId' => $adId
            ]);
        } else {
            try {
                $this->createDopingForAds(\request()->type_id, [$adId]);

                $this->messages->success(trans('module::messages.doping_added_successfully'));
                return redirect()->route('visiosoft.module.advs::admin_advs');
            } catch (\Exception $e) {
                $this->messages->error($e->getMessage());
                return redirect()->route('visiosoft.module.advs::admin_advs');
            }
        }
    }

    public function bulkChoose()
    {
        return $this->view->make(
            'visiosoft.module.dopings::admin/advs/bulk-choose',
            [
                'dopingTypes' => $this->dopingTypeRepository->all(),
            ]
        );
    }

    public function bulkHandle($typeId)
    {
        try {
            $this->createDopingForAds($typeId, explode(',', request()->get('selected')));

            $this->messages->success(trans('module::messages.doping_added_successfully'));
            return redirect()->route('visiosoft.module.advs::admin_advs');
        } catch (\Exception $e) {
            $this->messages->error($e->getMessage());
            return redirect()->route('visiosoft.module.advs::admin_advs');
        }
    }

    public function createDopingForAds($dopingTypeId, $adsIds)
    {
        $dopingType = $this->dopingTypeRepository->find($dopingTypeId);
        if (!$dopingType) {
            throw new \Exception(trans('visiosoft.module.dopings::messages.doping_type_not_exist'));
        }
        foreach ($adsIds as $id) {
            $this->dopingRepository->create([
                'name' => $dopingType->name,
                'adv_name_id' => $id,
                'doping_type_id' => $dopingType->id,
                'price' => $dopingType->price,
                'currency' => $dopingType->currency,
                'user_id' => auth()->id(),
                'expiry_date' => Carbon::now()->addDays($dopingType->duration),
                'status' => 'approved',
            ]);
        }
    }

    /**
     * Create a new entry.
     *
     * @param DopingFormBuilder $form
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function create(DopingFormBuilder $form)
    {
        return $form->render();
    }

    /**
     * Edit an existing entry.
     *
     * @param DopingFormBuilder $form
     * @param        $id
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function edit(DopingFormBuilder $form, $id)
    {
        return $form->render($id);
    }

    public function approve($id)
    {
        $doping = $this->dopingRepository->find($id);
        $doping_duration = $this->dopingTypeRepository->find($doping->doping_type_id);
        $doping->approve($doping_duration);

        event(new DopingApproved($doping));

        return back();
    }

    public function revert($id)
    {
        $doping = $this->dopingRepository->find($id);
        $doping->disapprove();
        return back();
    }
}
