<?php namespace Visiosoft\DemandModule\Http\Controller;

use Anomaly\Streams\Platform\Http\Controller\PublicController;
use Anomaly\Streams\Platform\Image\Image;
use Anomaly\Streams\Platform\Support\Currency;
use Anomaly\UsersModule\Role\Contract\RoleRepositoryInterface;
use Anomaly\UsersModule\User\Contract\UserRepositoryInterface;
use Illuminate\Contracts\Events\Dispatcher;
use Illuminate\Support\Str;
use Visiosoft\DemandModule\Demand\Contract\DemandRepositoryInterface;
use Visiosoft\DemandModule\Demand\Events\CreateDemand;
use Visiosoft\DemandModule\Option\Contract\OptionRepositoryInterface;
use Visiosoft\DemandModule\Service\Contract\ServiceRepositoryInterface;
use Visiosoft\DemandModule\Step\Contract\StepRepositoryInterface;
use Visiosoft\DemandModule\Type\Contract\TypeRepositoryInterface;
use Visiosoft\LocationModule\City\Contract\CityRepositoryInterface;
use Visiosoft\LocationModule\District\Contract\DistrictRepositoryInterface;
use Visiosoft\LocationModule\Neighborhood\Contract\NeighborhoodRepositoryInterface;

class ApiController extends PublicController
{
    private $image;
    private $cityRepository;
    private $districtRepository;
    private $neighborhoodRepository;
    private $typeRepository;
    private $optionRepository;
    private $demandRepository;
    private $userRepository;
    private $serviceRepository;
    private $dispatcher;
    private $roleRepository;
    private $stepRepository;

    public function __construct(
        Image $image,
        CityRepositoryInterface $cityRepository,
        DistrictRepositoryInterface $districtRepository,
        NeighborhoodRepositoryInterface $neighborhoodRepository,
        TypeRepositoryInterface $typeRepository,
        OptionRepositoryInterface $optionRepository,
        DemandRepositoryInterface $demandRepository,
        UserRepositoryInterface $userRepository,
        RoleRepositoryInterface $roleRepository,
        ServiceRepositoryInterface $serviceRepository,
        Dispatcher $dispatcher,
        StepRepositoryInterface $stepRepository
    )
    {
        parent::__construct();

        $this->image = $image;
        $this->cityRepository = $cityRepository;
        $this->districtRepository = $districtRepository;
        $this->neighborhoodRepository = $neighborhoodRepository;
        $this->typeRepository = $typeRepository;
        $this->optionRepository = $optionRepository;
        $this->demandRepository = $demandRepository;
        $this->userRepository = $userRepository;
        $this->serviceRepository = $serviceRepository;
        $this->dispatcher = $dispatcher;
        $this->roleRepository = $roleRepository;
        $this->stepRepository = $stepRepository;
    }

    public function getInquiries()
    {
        // Get all inquiries
        $inquiries = $this->typeRepository->getInquiries();

        // Check other service inquiry
        $other = $this->typeRepository->getInquiryTypeBySlug('services_other');

        // Select first inquiry as default
        $inquiryTypes = $this->getInquiryTypes($inquiries[0]->type_slug);

        // Get background image
        $bgImage = $this->image->make('theme::images/background.jpg', 'path');

        // Get modal steps
        $steps = $this->stepRepository->all()->sortBy('step')->values();

        return [
            'other' => $other,
            'inquiries' => $inquiries,
            'inquiryTypes' => $inquiryTypes,
            'bgImage' => $bgImage->url(),
            'urls' => [
                'discounted_shipping_service_url' => setting_value('visiosoft.theme.emlak24::discounted_shipping_service_url'),
                'free_tcip_policy_url' => setting_value('visiosoft.theme.emlak24::free_tcip_policy_url'),
                'discounted_furniture_url' => setting_value('visiosoft.theme.emlak24::discounted_furniture_url'),
            ],
            'priceStepSlugs' => setting_value(setting_value('streams::standard_theme') . '::price_step_slugs'),
            'cyprus' => setting_value(setting_value('streams::standard_theme') . '::cyprus'),
            'contact_phone' => setting_value('visiosoft.theme.emlak24::navbar_phone_number'),
            'steps' => $steps,
        ];
    }

    public function getInquiryTypes($slug = null)
    {
        if (is_null($slug)) {
            $slug = $this->request->slug;
        }
        $slug = explode('_', $slug)[0];

        $inquiryTypes = $this->typeRepository->getInquiryTypesBySlug($slug);

        return $inquiryTypes;
    }

    public function getLocation()
    {
        if ($this->request->type == 'city') {
            $locations = $this->cityRepository->newQuery()
                ->where('parent_country_id', setting_value('visiosoft.module.location::default_country'))
                ->get();
        } elseif ($this->request->type == 'district') {
            $locations = $this->districtRepository->newQuery()
                ->where('parent_city_id', $this->request->id)->get();
        } elseif ($this->request->type == 'neighborhood') {
            $locations = $this->neighborhoodRepository->newQuery()
                ->where('parent_district_id', $this->request->id)->get();
        }

        return $locations;
    }

    public function getOptions()
    {
        $inquiryType = $this->typeRepository->getInquiryTypeBySlug($this->request->inquirySlug);

        return $this->optionRepository->getOptionsByInquiryTypeId($inquiryType->id);
    }

    public function submitDemand()
    {
        $allFields = $this->request->all();
        $userFields = $allFields[0];
        $fields = $allFields[1];

        // Check user
        if (key_exists('selectedReference', $userFields)) {
            $user = $this->userRepository->find($userFields['selectedReference']);
        } else {
            $user = $this->checkUser($userFields['fullName'], $userFields['email'], $userFields['phone']);
        }

        // Get demand type
        $inquiryType = $this->typeRepository->getInquiryTypeBySlug($fields['demand']['inquirySlug']);

        // Check if service
        if (strpos($fields['demand']['inquirySlug'], 'services') !== false) {
            // Create a service
            $service = $this->serviceRepository->create([
                'user_id' => $user->id,
                'type_id' => $inquiryType->id,
            ]);

            // Create service custom fields
            $entryModel = $inquiryType->getEntryModel();
            $entry = $entryModel->create($fields['service']);
            $service->update([
                'entry_id' => $entry->id,
                'entry_type' => get_class($entry),
            ]);

            $this->dispatcher->dispatch(new CreateDemand(['demand' => $service, 'type' => 'service'], $user));
        } else {
            // Create a demand
            $demand = $this->demandRepository->create([
                'user_id' => $user->id,
                'city' => $fields['location']['city'],
                'district' => $fields['location']['district'],
                'neighborhood' => $fields['location']['neighborhood'],
                'type_id' => $inquiryType->id,
            ]);

            unset($fields['demand']['inquirySlug']);

            // Create demand custom fields
            $entryModel = $inquiryType->getEntryModel();
            $entry = $entryModel->create($fields['demand']);
            $demand->update([
                'entry_id' => $entry->id,
                'entry_type' => get_class($entry),
            ]);

            $this->dispatcher->dispatch(new CreateDemand(['demand' => $demand, 'type' => 'demand'], $user));
        }

        return ['success' => true];
    }

    public function checkUser($name, $email, $phone)
    {
        $user = $this->userRepository->allWithTrashed()->where('email', $email)->first();
        $userRole = $this->roleRepository->findBySlug('user');

        if (!$userRole) {
            throw new \Exception(trans('visiosoft.module.demand::message.user_role_not_exist'));
        }

        if (is_null($user)) {
            $fullName = explode(" ", $name);
            $lastName = isset($fullName[1]) ? $fullName[1] : null;
            $user = $this->userRepository->create([
                'username' => $name . Str::random(8),
                'first_name' => $fullName[0],
                'last_name' => $lastName,
                'display_name' => $name,
                'email' => $email,
                'activated' => 0,
                'gsm_phone' => $phone,
            ]);
            $password = Str::random(8);
            $user->setAttribute('password', $password);
            $user->setAttribute('roles', $userRole);
            $this->userRepository->save($user);
        };

        return $user;
    }

    public function getDefaultCurrencies(Currency $currency)
    {
        $defaultCurrencies = [$currency->symbol(setting_value('streams::currency'))];
        $enabledCurrencies = setting_value('visiosoft.module.advs::enabled_currencies');
        foreach ($enabledCurrencies as $enabledCurrency) {
            $symbol = $currency->symbol($enabledCurrency);
            $defaultCurrencies[] = $symbol;
        }

        $defaultCurrencies = array_unique($defaultCurrencies);

        return $defaultCurrencies;
    }
}
