<?php

namespace Visiosoft\DataVehicledemoExtension;

use Anomaly\SettingsModule\Setting\Contract\SettingRepositoryInterface;
use Anomaly\Streams\Platform\Application\Application;
use Anomaly\Streams\Platform\Assignment\Contract\AssignmentRepositoryInterface;
use Anomaly\Streams\Platform\Database\Seeder\Seeder;
use Anomaly\Streams\Platform\Field\Contract\FieldRepositoryInterface;
use Anomaly\Streams\Platform\Model\Repeater\RepeaterHomepageBannerEntryModel;
use Anomaly\Streams\Platform\Model\Repeater\RepeaterLocationsEntryModel;
use Anomaly\Streams\Platform\Model\Repeater\RepeaterPopularCategoriesEntryModel;
use Anomaly\Streams\Platform\Model\Repeater\RepeaterShowcaseEntryModel;
use Anomaly\Streams\Platform\Model\Repeater\RepeaterTinySliderEntryModel;
use Anomaly\Streams\Platform\Stream\Contract\StreamRepositoryInterface;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Visiosoft\CatsModule\Category\CategoryModel;
use Visiosoft\CustomfieldsModule\CustomField\Contract\CustomFieldRepositoryInterface;
use Visiosoft\CustomfieldsModule\CustomField\CustomFieldModel;
use Visiosoft\LocationModule\City\CityModel;

class DataVehicledemoExtensionSeeder extends Seeder
{
    private $fieldRepository;
    private $streamRepository;
    private $assignmentRepository;
    private $customFieldRepository;
    private $settingRepository;

    public function __construct(
        FieldRepositoryInterface $fieldRepository,
        StreamRepositoryInterface $streamRepository,
        AssignmentRepositoryInterface $assignmentRepository,
        CustomFieldRepositoryInterface $customFieldRepository,
        SettingRepositoryInterface $settingRepository
    )
    {
        parent::__construct();
        $this->fieldRepository = $fieldRepository;
        $this->streamRepository = $streamRepository;
        $this->assignmentRepository = $assignmentRepository;
        $this->customFieldRepository = $customFieldRepository;
        $this->settingRepository = $settingRepository;
    }

    /**
     * Run the seeder.
     */
    public function run()
    {
        $app = app(Application::class)->getReference()."_";
        $defaultPagesStream = $this->streamRepository->findBySlugAndNamespace($app.'pages', 'pages');

        // Create locations repeater
        $locationsRepeater = $this->streamRepository->findBySlugAndNamespace('locations', 'repeater');
        if (!$locationsRepeater) {
            $locationsRepeater = $this->streamRepository->create([
                'name' => 'Locations',
                'namespace' => 'repeater',
                'slug' => 'locations',
                'prefix' => 'repeater_',
            ]);
        }


        // Assign repeater fields
        $locationsFields = [
            'city' => [
                'name' => 'City',
                'namespace' => 'repeater',
                'slug' => 'city',
                'type' => 'anomaly.field_type.relationship',
                "config" => [
                    "related" => CityModel::class,
                    "mode" => "search",
                ]
            ],
            'cover_photo' => [
                'name' => 'Cover Photo',
                'namespace' => 'repeater',
                'slug' => 'cover_photo',
                'type' => 'anomaly.field_type.file',
                "config" => [
                    "folders" => ['images'],
                ]
            ]
        ];

        foreach ($locationsFields as $locationsField) {
            $field = $this->fieldRepository->findBySlugAndNamespace($locationsField['slug'], $locationsField['namespace']);
            if (!$field) {
                $field = $this->fieldRepository->create([
                    'name' => $locationsField['name'],
                    'namespace' => $locationsField['namespace'],
                    'slug' => $locationsField['slug'],
                    'type' => $locationsField['type'],
                    'locked' => 0,
                    "config" => $locationsField['config']
                ]);

                $this->assignmentRepository->create([
                    'stream_id' => $locationsRepeater->getId(),
                    'field_id' => $field->id,
                    'label' => $locationsField['name'],
                    'config' => [
                        'required' => true,
                        'unique' => true
                    ]
                ]);
            }

        }

        // Assign pages fields
        $pagesFields = [
            'featured_makes' => [
                'name' => 'Featured Makes',
                'namespace' => 'pages',
                'slug' => 'featured_makes',
                'type' => 'anomaly.field_type.multiple',
                "config" => [
                    "related" => CategoryModel::class,
                    "mode" => "lookup",
                ]
            ],
            'locations' => [
                'name' => 'Locations',
                'namespace' => 'pages',
                'slug' => 'locations',
                'type' => 'anomaly.field_type.repeater',
                "config" => [
                    "related" => RepeaterLocationsEntryModel::class,
                    "add_row" => "visiosoft.theme.arabam::button.add_location",
                ]
            ]
        ];

        if (is_module_installed('visiosoft.module.customfields')) {
            $customfield_fields = [
                'body_types' => [
                    'name' => 'Body Types',
                    'namespace' => 'pages',
                    'slug' => 'body_types',
                    'type' => 'anomaly.field_type.relationship',
                    "config" => [
                        "related" => CustomFieldModel::class,
                        "mode" => "search",
                        "title_name" => 'name',
                    ],
                ],
            ];

            $pagesFields = array_merge($customfield_fields, $pagesFields);
        }

        foreach ($pagesFields as $pagesField) {
            $field = $this->fieldRepository->findBySlugAndNamespace($pagesField['slug'], $pagesField['namespace']);
            if (!$field) {
                $field = $this->fieldRepository->create([
                    'name' => $pagesField['name'],
                    'namespace' => $pagesField['namespace'],
                    'slug' => $pagesField['slug'],
                    'type' => $pagesField['type'],
                    'locked' => 0,
                    "config" => $pagesField['config']
                ]);
                $this->assignmentRepository->create([
                    'stream_id' => $defaultPagesStream->getId(),
                    'field_id' => $field->id,
                    'label' => $pagesField['name'],
                ]);
            }
        }

        if (is_module_installed('visiosoft.module.customfields')) {
            // Create detail page custom fields
            $customFields = [
                [
                    'name' => 'City MPG',
                    'slug' => 'city_mpg',
                    'type' => 'integer'
                ],
                [
                    'name' => 'Highway MPG',
                    'slug' => 'highway_mpg',
                    'type' => 'integer'
                ],
            ];

            foreach ($customFields as $customField) {
                $custom = $this->customFieldRepository->findBySlug($customField['slug']);
                if (!$custom) {
                    $this->customFieldRepository->create([
                        'name' => $customField['name'],
                        'slug' => $customField['slug'],
                        'type' => $customField['type'],
                    ]);
                }

            }
        }

        // Create repeaters
        $repeaters = [
            [
                'name' => 'Homepage Banner',
                'slug' => 'homepage_banner',
                'translatable' => false,
                'fields' => ['large_image', 'medium_image', 'thumbnail_image', 'link']
            ],
            [
                'name' => 'Tiny Slider',
                'slug' => 'tiny_slider',
                'translatable' => true,
                'fields' => ['background', 'link', 'description', 'title']
            ],
            [
                'name' => 'Showcase',
                'slug' => 'showcase',
                'translatable' => true,
                'fields' => ['background', 'title', 'link']
            ],
            [
                'name' => 'Popular Categories',
                'slug' => 'popular_categories',
                'translatable' => true,
                'fields' => ['large_image', 'popular_categories']
            ]
        ];

        $repeatersObjects = array();
        foreach ($repeaters as $repeater) {
            $exists = $this->streamRepository->findBySlugAndNamespace($repeater['slug'], 'repeater');
            $repeatersObjects[$repeater['slug']] = $exists ?: $this->streamRepository->create([
                'name' => $repeater['name'],
                'namespace' => 'repeater',
                'slug' => $repeater['slug'],
                'prefix' => 'repeater_',
                'translatable' => $repeater['translatable'],
            ]);
        }

        // Assign repeater fields
        $repeatersFields = [
            [
                'name' => 'Large Image',
                'namespace' => 'repeater',
                'slug' => 'large_image',
                'type' => 'anomaly.field_type.file',
                'instructions' => 'Recommended 1000x350',
                "config" => [
                    "folders" => ['images'],
                    "mode" => 'upload',
                ],
                "assignmentConfig" => [
                    "required" => true,
                    "translatable" => false,
                ],
            ],
            [
                'name' => 'Medium Image',
                'namespace' => 'repeater',
                'slug' => 'medium_image',
                'type' => 'anomaly.field_type.file',
                'instructions' => 'Recommended 510x384',
                "config" => [
                    "folders" => ['images'],
                    "mode" => 'upload',
                ],
                "assignmentConfig" => [
                    "required" => true,
                    "translatable" => false,
                ],
            ],
            [
                'name' => 'Thumbnail Image',
                'namespace' => 'repeater',
                'slug' => 'thumbnail_image',
                'type' => 'anomaly.field_type.file',
                'instructions' => 'Recommended 65x65',
                "config" => [
                    "folders" => ['images'],
                    "mode" => 'upload',
                ],
                "assignmentConfig" => [
                    "required" => true,
                ],
            ],
            [
                'name' => 'Background',
                'namespace' => 'repeater',
                'slug' => 'background',
                'type' => 'anomaly.field_type.file',
                'instructions' => 'Recommended [280x376 for Showcase]  [718x480 for Tiny Slider]',
                "config" => [
                    "folders" => ['images'],
                    "mode" => 'upload',
                ],
                "assignmentConfig" => [
                    "required" => true,
                ],
            ],
            [
                'name' => 'Title',
                'namespace' => 'repeater',
                'slug' => 'title',
                'type' => 'anomaly.field_type.text',
                "assignmentConfig" => [
                    "required" => true,
                    "translatable" => true,
                ],
            ],
            [
                'name' => 'Link',
                'namespace' => 'repeater',
                'slug' => 'link',
                'type' => 'anomaly.field_type.url',
                "config" => [
                    "default_value" => '#',
                ],
                "assignmentConfig" => [
                    "required" => true,
                ],
            ],
            [
                'name' => 'Description',
                'namespace' => 'repeater',
                'slug' => 'description',
                'type' => 'anomaly.field_type.text',
                "assignmentConfig" => [
                    "required" => true,
                    "translatable" => true,
                ],
            ],
            [
                'name' => 'Popular Categories',
                'namespace' => 'repeater',
                'slug' => 'popular_categories',
                'type' => 'anomaly.field_type.relationship',
                'config' => [
                    "related" => CategoryModel::class,
                ],
                "assignmentConfig" => [
                    "required" => true,
                ],
            ],
        ];

        foreach ($repeatersFields as $repeatersField) {
            $field = $this->fieldRepository->findBySlugAndNamespace($repeatersField['slug'], 'repeater');
            if (!$field) {
                $fieldValue = [
                    'name' => $repeatersField['name'],
                    'namespace' => 'repeater',
                    'slug' => $repeatersField['slug'],
                    'type' => $repeatersField['type'],
                    'locked' => 0,
                ];
                if (isset($repeatersField['config'])) {
                    $fieldValue['config'] = $repeatersField['config'];
                }
                if (isset($repeatersField['instructions'])) {
                    $fieldValue['instructions'] = $repeatersField['instructions'];
                }
                $field = $this->fieldRepository->create($fieldValue);
            }
            foreach ($repeaters as $repeater) {
                if (in_array($field->slug, $repeater['fields'])) {
                    $assign = $this->assignmentRepository->findByStreamAndField(
                        $repeatersObjects[$repeater['slug']],
                        $field
                    );
                    if (!$assign) {
                        $this->assignmentRepository->create(array_merge([
                            'stream_id' => $repeatersObjects[$repeater['slug']]->getId(),
                            'field_id' => $field->id,
                            'label' => $field->name,
                        ], $repeatersField['assignmentConfig']));
                    }
                }
            }
        }

        // Assign pages fields
        $defaultPagesStream = $this->streamRepository->findBySlugAndNamespace($app.'pages', 'pages');

        $pagesFields = [
            [
                'name' => 'Homepage Banner',
                'slug' => 'homepage_banner',
                "config" => [
                    "related" => RepeaterHomepageBannerEntryModel::class,
                    "max" => 14,
                ]
            ],
            [
                'name' => 'Showcase',
                'slug' => 'showcase',
                "config" => [
                    "related" => RepeaterShowcaseEntryModel::class,
                    "max" => 4,
                ]
            ],
            [
                'name' => 'Tiny Slider',
                'slug' => 'tiny_slider',
                "config" => [
                    "related" => RepeaterTinySliderEntryModel::class,
                    "max" => 3,
                ]
            ],
            [
                'name' => 'Popular Categories',
                'slug' => 'popular_categories',
                'config' => [
                    'related' => RepeaterPopularCategoriesEntryModel::class,
                ]
            ],
        ];

        foreach ($pagesFields as $pagesField) {
            $field = $this->fieldRepository->findBySlugAndNamespace($pagesField['slug'], 'pages');
            if (!$field) {
                $field = $this->fieldRepository->create([
                    'name' => $pagesField['name'],
                    'namespace' => 'pages',
                    'slug' => $pagesField['slug'],
                    'type' => 'anomaly.field_type.repeater',
                    'locked' => 0,
                    "config" => $pagesField['config']
                ]);
            }
            $assign = $this->assignmentRepository->findByStreamAndField($defaultPagesStream, $field);
            if (!$assign) {
                $this->assignmentRepository->create([
                    'stream_id' => $defaultPagesStream->getId(),
                    'field_id' => $field->id,
                    'label' => $pagesField['name'],
                ]);
            }
        }

        // Activate
        $this->settingRepository->set('streams::standard_theme', 'visiosoft.theme.arabam');

        // Seeder
        $demo = str_replace('{application_reference}', $app, file_get_contents(realpath(dirname(__DIR__)) . '/src/seed/data/vehicle_demodata.sql'));
        Model::unguard();
        DB::unprepared($demo);
        Model::reguard();

        $zip = new \ZipArchive();
        $zip->open(realpath(dirname(__DIR__)) . '/src/seed/data/local.zip', \ZipArchive::CREATE);
        $zip->extractTo(storage_path('streams/' . app(Application::class)->getReference() . '/files-module'));
        $zip->close();

        Artisan::call('assets:clear');
    }
}