<?php namespace Visiosoft\CraftgateExtension\Commands;


use Craftgate\Craftgate;
use Illuminate\Support\Facades\Auth;
use Monolog\Handler\StreamHandler;
use Monolog\Logger;
use Visiosoft\CraftgateExtension\Card\CardModel;

class CardProvider
{
    /**
     * @var Craftgate|null
     */
    protected $craftgate = null;

    /**
     * The informations of craftgate instance.
     *
     * @var string
     */
    public function __construct()
    {
        $this->craftgate = new Craftgate(array(
            'apiKey' => setting_value('visiosoft.extension.craftgate::api_key'),
            'secretKey' => setting_value('visiosoft.extension.craftgate::secret_key'),
            'baseUrl' => setting_value('visiosoft.extension.craftgate::base_url'),
        ));
    }

    /*
    * Retrieve card info by user ID
    *
    * return object
    */
    public function getCardInfo($cardID = null,$is_default = null)
    {
        // start query
        $query = CardModel::query();


        // if card id is not null, add where clause
        if ($cardID) {
            $query = $query->where('id', $cardID);
        }
        if ($is_default) {
            $query = $query->where('is_default', $is_default);
        }


        // Return default card or first card
        return
            $query
                ->where('user_id', Auth::id())
                ->first();
    }

    /*
     * Create a card
     *
     * @param string $cardHolderName
     * @param string $cardNumber
     * @param string $expireYear
     * @param string $expireMonth
     * @param string $cardAlias
     *
     * return object
     */
    public function createCard($cardHolderName, $cardNumber, $expireYear, $expireMonth, $cardAlias = null, $cardUserKey = null)
    {
        try {

            // get users card info
            $cardInfo = $this->getCardInfo();

            // if user has a card in the database, match user_id to the cardUserKey
            if ($cardInfo && isset($cardInfo->user_token)) {
                $cardUserKey = $cardInfo->user_token;
            }

            // create a card
            $response = json_decode($this->craftgate->payment()->storeCard([
                'cardHolderName' => $cardHolderName,
                'cardNumber' => $cardNumber,
                'expireYear' => $expireYear,
                'expireMonth' => $expireMonth,
                'cardAlias' => $cardAlias,
                'cardUserKey' => $cardUserKey
            ]));

            // if the user does not have a card in the database, we will save the cardUserKey
            if ( isset($response->data)) {
                CardModel::create([
                    'user_id' => Auth::id(),
                    'user_token' => $response->data->cardUserKey,
                    'card_token' => $response->data->cardToken,
                    'is_default' => !$this->retrieveDefaultCard(),
                ]);
            }


            return $response;
        } catch (\Exception $e) {
            $this->logError($e);
            return $e->getMessage();
        }

    }

    /*
     * Delete a card
     *
     * @param string $cardUserKey
     * @param string $cardToken
     *
     * return boolean
     */
    public function deleteCard($cardID)
    {
        try {

            $cardInfo = $this->getCardInfo($cardID);
            $userCards = $this->retrieveUserCards();

            if (!$cardInfo) {
                $this->logError('Card not found');
                throw new \Exception('Card not found');
            }

            // Sending delete request to Craftgate
            $cardDelete = json_decode($this->craftgate->payment()->deleteStoredCard([
                'cardUserKey' => $cardInfo->user_token,
                'cardToken' => $cardInfo->card_token
            ]));

            // card delete returns null if it is successful
            // and returns an error message if it fails
            if ($cardDelete != "" && $cardDelete->errors){
                return false;
            }

            // If user has more than one card and the card is default, we will set the first card as default
            if ( $userCards && count($userCards) > 1 && $cardInfo->is_default ) {
                if ( $userCards[0]->id != $cardID)
                    $this->updateDefaultCard($userCards[0]->id);
                else
                    $this->updateDefaultCard(end($userCards)->id);
            }

            // Deleting card from database
            CardModel::query()->where('id', $cardID)->delete();

            return true;

        } catch (\Exception $e) {

            $this->logError($e);

            return $e->getMessage();

        }
    }

    /*
     * Retrieve users cards from craftgate
     *
     * @param string $cardUserKey
     * @param string $cardToken
     *
     * return array
     */
    public function retrieveUserCards()
    {

        try {
            // Users all cards query
            $userCards = CardModel::query()->where('user_id', Auth::id())->get();

            $userKey = $userCards->first()->user_token;


            $response = json_decode($this->craftgate->payment()->searchStoredCards([
                'cardUserKey' => $userKey
            ]));

            $craftgate_cards = $response->data->items;
            // Match user cards with craftgate cards
            // Also add is_default and id to craftgate cards
            foreach ($userCards as $card) {
                foreach ($craftgate_cards as $craftgate_card) {
                    if ($craftgate_card->cardToken == $card->card_token) {
                        $craftgate_card->is_default = $card->is_default;
                        $craftgate_card->id = $card->id;
                    }
                }
            }

            return $craftgate_cards;
        } catch (\Exception $e) {
            $this->logError($e);
            return $e->getMessage();
        }

    }

    /*
     * Retrieve defaut card of user
     *
     * return object
     */
    public function retrieveDefaultCard()
    {

        try {

            return $this->getCardInfo(null, true);

        } catch (\Exception $e) {
            $this->logError($e);
            return $e->getMessage();
        }

    }

    /*
     * Update default card and return updated card
     *
     * @param int $cardID
     *
     * return object
     */
    public function updateDefaultCard($cardID)
    {

        try {
            // Users all cards query
            $query = CardModel::query()->where('user_id', Auth::id());

            // Getting all cards from db and
            // Updating all cards is_default field to false
           $query->update(['is_default' => false]);

            // Updating selected card is_default field to true
            $query->where('id', $cardID)->update(['is_default' => true]);

            // Returning updated card
            return $query->where('is_default', true)->first();

        } catch (\Exception $e) {

            $this->logError($e);

            return $e->getMessage();

        }

    }

    /*
     * Logging errors to the log file under storage path
     */
    public function logError($error)
    {
        $log = new Logger('craftgate');
        $log->pushHandler(new StreamHandler(storage_path('logs/craftgate.log')), Logger::ERROR);
        $log->error($error);
    }

}