<?php namespace Visiosoft\CraftgateExtension\Commands;


use Craftgate\Craftgate;
use Craftgate\Util\Guid;
use Illuminate\Support\Facades\Auth;
use Monolog\Handler\StreamHandler;
use Monolog\Logger;

class CardProvider
{
    /**
     * @var Craftgate|null
     */
    protected $craftgate = null;
    protected $user = null;
    /**
     * The informations of craftgate instance.
     *
     * @var string
     */
    public function __construct()
    {
        $this->craftgate = new Craftgate(array(
            'apiKey' => setting_value('visiosoft.extension.craftgate::api_key'),
            'secretKey' => setting_value('visiosoft.extension.craftgate::secret_key'),
            'baseUrl' => setting_value('visiosoft.extension.craftgate::base_url'),
        ));
        $this->user = Auth::user();
    }

    /*
     * Pay by saved card
     *
     */
    public function payBySavedCard($providerUserID, $defaultCardID, $price, $currency, $items, $buyer, $invoice_id) {

        $request = [
            'price' => $price,
            'paidPrice' => $price,
            'walletPrice' => 0,
            'installment' => 1,
            'currency' => strtoupper($currency),
            'paymentGroup' => 'LISTING_OR_SUBSCRIPTION',
            'card' => [
                'cardUserKey' => $providerUserID,
                'cardToken' => $defaultCardID,
            ],
            'items' => [
                [
                    'externalId' => Guid::generate(),
                    'name' => $items[0]['name'],
                    'price' => $price
                ]
            ]
        ];

        $response = json_decode($this->craftgate->payment()->createPayment($request));

        return $response ?? false;

    }

    /*
     * Create a card
     *
     * @param string $cardHolderName
     * @param string $cardNumber
     * @param string $expireYear
     * @param string $expireMonth
     * @param string $cardAlias
     *
     * return object
     */
    public function createCard(
        $cardHolderName,
        $cardNumber,
        $expireYear,
        $expireMonth,
        $cvc,
        $email,
        $cardUserKey,
        $cardAlias)
    {
        try {

            // request params
            $requestParams = [
                'cardHolderName' => $cardHolderName,
                'cardNumber' => $cardNumber,
                'expireYear' => $expireYear,
                'expireMonth' => $expireMonth,
                'cardAlias' => $cardAlias,
                'cardUserKey' => $cardUserKey
            ];

            // create a card
            $response = json_decode($this->craftgate->payment()->storeCard($requestParams));

            // creating card
            return $response ?? false;

        } catch (\Exception $e) {
            $this->logError($e);
            return $e->getMessage();
        }

    }

    /*
     * Delete a card
     *
     * @param string $cardUserKey
     * @param string $cardToken
     *
     * return boolean
     */
    public function removeCard($cardToken)
    {
        try {

            // Sending delete request to Craftgate
            $cardDelete = json_decode($this->craftgate->payment()->deleteStoredCard([
                'cardToken' => $cardToken
            ]));

            // card delete returns null if it is successful
            // and returns an error message if it fails
            if ($cardDelete != "" && $cardDelete->errors) {
                return false;
            }

            return true;

        } catch (\Exception $e) {

            $this->logError($e);
            return $e->getMessage();

        }
    }


    /*
     * Logging errors to the log file under storage path
     */
    public function logError($error)
    {
        $log = new Logger('craftgate');
        $log->pushHandler(new StreamHandler(storage_path('logs/craftgate.log')), Logger::ERROR);
        $log->error($error);
    }


}