<?php namespace Visiosoft\CraftgateExtension\Commands;


use Craftgate\Craftgate;
use Illuminate\Support\Facades\DB;

class CardProvider
{
    /**
     * @var Craftgate|null
     */
    protected $craftgate = null;

    /**
     * The informations of craftgate instance.
     *
     * @var string
     */
    public function __construct()
    {
        $this->craftgate = new Craftgate(array(
            'apiKey' => setting_value('visiosoft.extension.craftgate::api_key'),
            'secretKey' => setting_value('visiosoft.extension.craftgate::secret_key'),
            'baseUrl' => setting_value('visiosoft.extension.craftgate::base_url'),
        ));
    }

    /*
     * Create a card
     *
     * @param string $cardHolderName
     * @param string $cardNumber
     * @param string $expireYear
     * @param string $expireMonth
     * @param string $cardAlias
     *
     * return object
     */
    public function createCard($userId, $cardHolderName, $cardNumber, $expireYear, $expireMonth, $cardAlias = null, $cardUserKey = null)
    {

        try {

            // check if user has a card in the database
            $relation = DB::table('craftgate_user_user_token')->where('user_id', $userId)->first();

            // if user has a card in the database, we will match to the cardUserKey
            if ($relation && isset($relation->user_token)) {
                $cardUserKey = $relation->user_token;
            }

            // create a card
            $response = json_decode($this->craftgate->payment()->storeCard([
                'cardHolderName' => $cardHolderName,
                'cardNumber' => $cardNumber,
                'expireYear' => $expireYear,
                'expireMonth' => $expireMonth,
                'cardAlias' => $cardAlias,
                'cardUserKey' => $cardUserKey
            ]));

            // if the user does not have a card in the database, we will save the cardUserKey
            if ( isset($response->data) && !isset($relation->user_token)) {
                DB::table('craftgate_user_user_token')->insert([
                    'user_id' => $userId,
                    'user_token' => $response->data->cardUserKey
                ]);
            }


            return $response;
        } catch (\Exception $e) {
            return $e->getMessage();
        }

    }

    /*
     * Update card expire date
     *
     * @param string $cardUserKey
     * @param string $cardToken
     * @param string $expireYear
     * @param string $expireMonth
     *
     * return object
     */
    public function updateCard($userId, $cardToken, $expireYear, $expireMonth)
    {
        // check if user has a card in the database
        $card_info = $this->getUserToken($userId);

        try {
            return $this->craftgate->payment()->updateCard([
                'cardUserKey' => $card_info,
                'cardToken' => $cardToken,
                'expireYear' => $expireYear,
                'expireMonth' => $expireMonth
            ]);
        } catch (\Exception $e) {
            return $e->getMessage();
        }
    }

    /*
     * Delete a card
     *
     * @param string $cardUserKey
     * @param string $cardToken
     *
     * return null
     */
    public function deleteCard($cardUserKey, $cardToken)
    {
        try {

            if ($cardDelete = $this->craftgate->payment()->deleteStoredCard([
                'cardUserKey' => $cardUserKey,
                'cardToken' => $cardToken
            ])) {
                throw new \Exception($cardDelete);
            } else {
                return true;
            }

        } catch (\Exception $e) {
            return $e->getMessage();
        }
    }

    /*
     * Retrieve a card by user ID
     *
     * @param string $cardUserKey
     * @param string $cardToken
     *
     * return object
     */
    public function retrieveUserCards($userId)
    {
        $card_info = $this->getUserToken($userId);

        try {
            return $this->craftgate->payment()->searchStoredCards([
                'cardUserKey' => $card_info
            ]);
        } catch (\Exception $e) {
            return $e->getMessage();
        }

    }

    /*
     * Retrieve user token by user ID
     *
     * @param string $userId
     *
     * return object
     */
    public function getUserToken($userId) {
        $card_info = DB::table('craftgate_user_user_token')->where('user_id', $userId)->first();

        if (!$card_info) {
            throw new \Exception(trans('visiosoft.extension.craftgate::messages.user_has_no_card'));
        }

        return $card_info->user_token;
    }

}