<?php

namespace Visiosoft\ClientTicketsModule\Service;

use Anomaly\SettingsModule\Setting\Contract\SettingRepositoryInterface;
use GuzzleHttp\Client;
use Illuminate\Http\Request;
use Visiosoft\ClientTicketsModule\Service\Contract\TicketApiServiceInterface;

class TicketApiService implements TicketApiServiceInterface
{
    protected $request, $options;

    public function __construct(Request $request)
    {
        $this->request = $request;
        $options = [
            'headers' => [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'locale' => $this->request->session()->get('_locale') ? $this->request->session()->get('_locale') : 'en',
            ],
            'http_errors' => false
        ];
        if (!empty(setting_value('visiosoft.module.client_tickets::token'))) {
            $options['headers']['Authorization'] = 'Bearer ' . setting_value('visiosoft.module.client_tickets::token');
        }
        $this->options = $options;
    }

    public function request($method, $url, $queryString = [], $data = [])
    {
        $client = new Client();
        $options = $this->options;
        if (!empty($data)) {
            $options['body'] = json_encode($data);
        }
        if (!empty($queryString)) {
            $url = $url . '?' . http_build_query($queryString);
        }
        $response = $client->request($method, $url, $options);
        if ($response->getStatusCode() == 401) {
            $this->resetAuth();
            $this->auth(setting_value('visiosoft.module.client_tickets::username'), setting_value('visiosoft.module.client_tickets::password'));
        }
        if ($response->getStatusCode() == 400) {
            json_decode($response->getBody(), true);
            throw new \Exception(trans('visiosoft.module.client_tickets::field.bad_request'));
        }
        return json_decode($response->getBody(), true);
    }

    public function auth($username, $password)
    {
        if (empty(env('AUTO_TOKEN'))) {
            if (empty($username)) {
                $username = setting_value('visiosoft.module.client_tickets::username');
            }
            if (empty($password)) {
                $password = setting_value('visiosoft.module.client_tickets::password');
            }
            $url = $this->retrieveAuthUrl();
            $data = [
                'email' => $username,
                'password' => $password
            ];
            $result = $this->request('post', $url, [], $data);
            $settingRepository = app(SettingRepositoryInterface::class);
            if ($settingRepository->findByKeyOrNew('visiosoft.module.client_tickets::token')) {
                $settingRepository->set('visiosoft.module.client_tickets::token', $result['token']);
            }
            if ($settingRepository->findByKeyOrNew('visiosoft.module.client_tickets::username')) {
                $settingRepository->set('visiosoft.module.client_tickets::username', $username);
            }
            if ($settingRepository->findByKeyOrNew('visiosoft.module.client_tickets::password')) {
                $settingRepository->set('visiosoft.module.client_tickets::password', $password);
            }
            return ['success' => true, 'data' => $result];
        } else {
            return $this->getTokenForSite();
        }
    }


    public function getTokenForSite()
    {
        $url = $this->retrieveSiteTokenUrl();
        $siteToken = env('AUTO_TOKEN');
        $queryString = ['token' => $siteToken];
        $result = $this->request('get', $url, $queryString, []);
        $accessToken = '';
        if ($result['success']) {
            $accessToken = $result['data']['access_token'];
        }
        $settingRepository = app(SettingRepositoryInterface::class);
        if (!empty($settingRepository)) {
            $settingRepository->set('visiosoft.module.client_tickets::token', $accessToken);
        }
        return $result;
    }

    public function table($inputs)
    {
        $url = $this->retrieveTableUrl();
        return $this->request('get', $url, $inputs, []);
    }

    public function create(string $subject, string $message, int $departmentId, int $priorityId, $file = null)
    {
        $params = [
            'subject' => $subject,
            'message' => $message,
            'department_id' => $departmentId,
            'priority_id' => $priorityId,
            'name' => '',
            'email' => '',
            'file' => $file
        ];
        $url = $this->retrieveCreateUrl();
        return $this->request('post', $url, [], $params);
    }

    public function answer(int $ticketId, string $message, $file = null)
    {
        $params = [
            'ticket_id' => $ticketId,
            'content' => $message
        ];
        $url = $this->retrieveAnswerUrl($ticketId);
        return $this->request('put', $url, [], $params);
    }

    public function close(int $ticketId)
    {
        $url = $this->retrieveCloseUrl($ticketId);
        return $this->request('put', $url, [], []);
    }

    public function references()
    {
        $url = $this->retrieveReferencesUrl();
        return $this->request('get', $url, [], []);
    }

    public function thread($ticketId)
    {
        $url = $this->retrieveThreadUrl($ticketId);
        return $this->request('get', $url, [], []);
    }

    public function retrieveBaseUrl()
    {
        return setting_value('visiosoft.module.client_tickets::base_url');
    }


    public function retrieveAuthUrl()
    {
        return $this->retrieveBaseUrl() . setting_value('visiosoft.module.client_tickets::login_url');
    }

    public function retrieveTableUrl()
    {
        return $this->retrieveBaseUrl() . setting_value('visiosoft.module.client_tickets::tickets_url');
    }

    public function retrieveCreateUrl()
    {
        return $this->retrieveBaseUrl() . setting_value('visiosoft.module.client_tickets::create_url');

    }

    public function retrieveAnswerUrl($ticketId)
    {
        return $this->retrieveBaseUrl() . str_replace('{ticketId}', $ticketId, setting_value('visiosoft.module.client_tickets::answer_url'));
    }

    public function retrieveCloseUrl($ticketId)
    {
        return $this->retrieveBaseUrl() . str_replace('{ticketId}', $ticketId, setting_value('visiosoft.module.client_tickets::close_url'));
    }

    public function retrieveThreadUrl($ticketId)
    {
        return $this->retrieveBaseUrl() . str_replace('{ticketId}', $ticketId, setting_value('visiosoft.module.client_tickets::thread_url'));
    }

    public function retrieveReferencesUrl()
    {
        return $this->retrieveBaseUrl() . setting_value('visiosoft.module.client_tickets::references_url');
    }

    public function retrieveSiteTokenUrl()
    {
        return $this->retrieveBaseUrl() . setting_value('visiosoft.module.client_tickets::site_token_url');
    }

    public function resetAuth()
    {
        $settingRepository = app(SettingRepositoryInterface::class);
        $settingRepository->set('visiosoft.module.client_tickets::token', null);
        $settingRepository->set('visiosoft.module.client_tickets::username', null);
        $settingRepository->set('visiosoft.module.client_tickets::password', null);
    }
}