<?php

namespace Visiosoft\ClientTicketsModule\Service;

use Anomaly\SettingsModule\Setting\Contract\SettingRepositoryInterface;
use GuzzleHttp\Client;
use Visiosoft\ClientTicketsModule\Service\Contract\TicketApiServiceInterface;

class TicketApiService implements TicketApiServiceInterface
{
    public function auth(string $username, string $password)
    {
        if (empty($username)) {
            $username = setting_value('visiosoft.module.client_tickets::username');
        }
        if (empty($password)) {
            $password = setting_value('visiosoft.module.client_tickets::password');
        }
        $url = $this->retrieveAuthUrl();
        $client = new Client();
        $options = [
            'headers' => [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json'
            ],
            'body' => json_encode([
                'email' => $username,
                'password' => $password
            ]),
            'http_errors' => false
        ];
        $response = $client->request('post', $url, $options);
        $result = json_decode($response->getBody(), true);
        if ($response->getStatusCode() == 400) {
            return $result;
        }
        $settingRepository = app(SettingRepositoryInterface::class);
        if ($settingRepository->findByKeyOrNew('visiosoft.module.client_tickets::token')) {
            $settingRepository->set('visiosoft.module.client_tickets::token', $result['token']);
        }
        if ($settingRepository->findByKeyOrNew('visiosoft.module.client_tickets::username')) {
            $settingRepository->set('visiosoft.module.client_tickets::username', $username);
        }
        if ($settingRepository->findByKeyOrNew('visiosoft.module.client_tickets::password')) {
            $settingRepository->set('visiosoft.module.client_tickets::password', $password);
        }
        return ['success' => true, 'data' => $result];
    }

    public function table($inputs)
    {
        $url = $this->retrieveTableUrl();
        $client = new Client();
        $options = [
            'headers' => [
                'Authorization' => 'Bearer ' . setting_value('visiosoft.module.client_tickets::token'),
                'Content-Type' => 'application/json',
                'Accept' => 'application/json'
            ],
            'http_errors' => false
        ];
        $response = $client->get($url . '?' . http_build_query($inputs), $options);
        if ($response->getStatusCode() == 401) {
            $this->resetAuth();
        }
        return json_decode($response->getBody(), true);
    }

    public function create(string $subject, string $message, int $departmentId, int $priorityId, string $name, string $email, $file = null)
    {
        $params = [
            'subject' => $subject,
            'message' => $message,
            'department_id' => $departmentId,
            'priority_id' => $priorityId,
            'name' => $name,
            'email' => $email,
            'file' => $file
        ];
        $url = $this->retrieveCreateUrl();
        $client = new Client();
        $options = [
            'headers' => [
                'Authorization' => 'Bearer ' . setting_value('visiosoft.module.client_tickets::token'),
                'Content-Type' => 'application/json',
                'Accept' => 'application/json'
            ],
            'http_errors' => false,
            'body' => json_encode($params)

        ];
        $response = $client->request('post', $url, $options);
        if ($response->getStatusCode() == 401) {
            $this->resetAuth();
        }
        return json_decode($response->getBody(), true);
    }

    public function answer(int $ticketId, string $message, $file = null)
    {
        $params = [
            'ticket_id' => $ticketId,
            'content' => $message
        ];
        $url = $this->retrieveAnswerUrl($ticketId);
        $client = new Client();
        $options = [
            'headers' => [
                'Authorization' => 'Bearer ' . setting_value('visiosoft.module.client_tickets::token'),
                'Content-Type' => 'application/json',
                'Accept' => 'application/json'
            ],
            'http_errors' => false,
            'body' => json_encode($params)

        ];
        $response = $client->request('put', $url, $options);
        if ($response->getStatusCode() == 401) {
            $this->resetAuth();
        }
        return json_decode($response->getBody(), true);
    }

    public function close($ticketId)
    {
        $url = $this->retrieveCloseUrl($ticketId);
        $client = new Client();
        $options = [
            'headers' => [
                'Authorization' => 'Bearer ' . setting_value('visiosoft.module.client_tickets::token'),
                'Content-Type' => 'application/json',
                'Accept' => 'application/json'
            ],
            'http_errors' => false,
        ];
        $response = $client->request('put', $url, $options);
        if ($response->getStatusCode() == 401) {
            $this->resetAuth();
        }
        return json_decode($response->getBody(), true);
    }

    public function references()
    {
        $url = $this->retrieveReferencesUrl();
        $client = new Client();
        $options = [
            'headers' => [
                'Authorization' => 'Bearer ' . setting_value('visiosoft.module.client_tickets::token'),
                'Content-Type' => 'application/json',
                'Accept' => 'application/json'
            ],
        ];
        $response = $client->request('get', $url, $options);
        if ($response->getStatusCode() == 401) {
            $this->resetAuth();
        }
        return json_decode($response->getBody(), true);
    }

    public function thread($ticketId)
    {
        $url = $this->retrieveThreadUrl($ticketId);
        $client = new Client();
        $options = [
            'headers' => [
                'Authorization' => 'Bearer ' . setting_value('visiosoft.module.client_tickets::token'),
                'Content-Type' => 'application/json',
                'Accept' => 'application/json'
            ]
        ];
        $response = $client->request('get', $url, $options);
        if ($response->getStatusCode() == 401) {
            $this->resetAuth();
        }
        return json_decode($response->getBody(), true);
    }

    public function retrieveBaseUrl()
    {
        return setting_value('visiosoft.module.client_tickets::base_url');
    }


    public function retrieveAuthUrl()
    {
        return $this->retrieveBaseUrl() . setting_value('visiosoft.module.client_tickets::login_url');
    }

    public function retrieveTableUrl()
    {
        return $this->retrieveBaseUrl() . setting_value('visiosoft.module.client_tickets::tickets_url');
    }

    public function retrieveCreateUrl()
    {
        return $this->retrieveBaseUrl() . setting_value('visiosoft.module.client_tickets::create_url');

    }

    public function retrieveAnswerUrl($ticketId)
    {
        return $this->retrieveBaseUrl() . str_replace('{ticketId}', $ticketId, setting_value('visiosoft.module.client_tickets::answer_url'));
    }

    public function retrieveCloseUrl($ticketId)
    {
        return $this->retrieveBaseUrl() . str_replace('{ticketId}', $ticketId, setting_value('visiosoft.module.client_tickets::close_url'));
    }

    public function retrieveThreadUrl($ticketId)
    {
        return $this->retrieveBaseUrl() . str_replace('{ticketId}', $ticketId, setting_value('visiosoft.module.client_tickets::thread_url'));
    }

    public function retrieveReferencesUrl()
    {
        return $this->retrieveBaseUrl() . setting_value('visiosoft.module.client_tickets::references_url');
    }

    public function resetAuth()
    {
        $settingRepository = app(SettingRepositoryInterface::class);
        $settingRepository->set('visiosoft.module.client_tickets::token', null);
        $settingRepository->set('visiosoft.module.client_tickets::username', null);
        $settingRepository->set('visiosoft.module.client_tickets::password', null);
    }
}