<?php namespace Visiosoft\BookingModule\Http\Controller;

use Anomaly\Streams\Platform\Http\Controller\ResourceController;
use Anomaly\UsersModule\User\UserModel;
use DateTime;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Response;
use Visiosoft\BookingModule\Booking\Event\paymentPaypal;
use Visiosoft\BookingModule\Booking\BookingModel;
use Anomaly\UsersModule\User\Contract\UserRepositoryInterface;
use Visiosoft\BookingModule\Booking\Events\ActivationBooking;
use Visiosoft\BookingModule\Booking\Events\CreateBooking;
use Visiosoft\BookingModule\Booking\Events\SendUserPassword;
use Visiosoft\BookingModule\Booking\Listener\PaymentItem;
use Visiosoft\BookingModule\Location\Contract\LocationRepositoryInterface;
use Visiosoft\BookingModule\Location\LocationModel;
use Visiosoft\BookingModule\Service\ServiceModel;
use Visiosoft\BookingModule\Staff\StaffModel;
use Illuminate\Contracts\Events\Dispatcher;
use Visiosoft\PaypalMdoule\Event\itemPaypal;
use Visiosoft\PaypalModule\Events\PaypalItems;
use Visiosoft\PaypalModule\Listener\paypal;


class BookingController extends ResourceController
{
    private $booking_model;
    private $services_model;
    private $staff_model;
    private $location_model;
    private $event;
    private $user_model;
    private $users;

    /**
     * BookingController constructor.
     * @param Dispatcher $events
     * @param ServiceModel $serviceModel
     * @param StaffModel $staffModel
     * @param LocationModel $locationModel
     * @param BookingModel $bookingModel
     */
    public function __construct(
        Dispatcher $events,
        ServiceModel $serviceModel,
        StaffModel $staffModel,
        LocationModel $locationModel,
        BookingModel $bookingModel,
        UserModel $user_model,
        UserRepositoryInterface $users
    )
    {
        parent::__construct();
        $this->event = $events;
        $this->booking_model = $bookingModel;
        $this->services_model = $serviceModel;
        $this->staff_model = $staffModel;
        $this->location_model = $locationModel;
        $this->user_model = $user_model;
        $this->users = $users;
    }


    /**
     * @return \Illuminate\Contracts\View\View|mixed
     */
    public function index()
    {
        if (!is_null(Auth::user())) {
            $user = Auth::user();
            $email = $user->email;
            $name = $user->display_name;

        }
        $this->template->set('meta_title', trans('visiosoft.module.booking::field.booking.name'));
        $locations = $this->location_model->all();
        return $this->view->make('visiosoft.module.booking::booking', compact('locations', 'email', 'name'));

    }

    /**
     * @param Request $request
     * @return mixed
     */
    public function srvLocation(Request $request)
    {
        //Return Service for location ID
        $response = $this->services_model->where('location_id', $request->val)->get();
        $response['items'] = $this->ConvertArray($response);
        $response['type'] = 'service';

        return $this->returnResponse($response);
    }

    /**
     * @param Request $request
     * @return mixed
     */
    public function srvService(Request $request)
    {
        //Return Staff for service ID
        $staff_list = $this->staff_model->whereRaw('services REGEXP \'.*;s:[0-9]+:"' . $request->val . '".*\'')->get();
        $response['items'] = $this->ConvertArray($staff_list);
        $response['items'] = $this->staff_model->addFotoStaff($response['items']);
        $response['type'] = 'staff';
        return $this->returnResponse($response);
    }

    /**
     * @param Request $request
     * @return mixed
     */
    public function srvStaff(Request $request)
    {
        //Return Booking Date and Time for service ID, location ID, staff ID
        $parameters = json_decode($request->val, true);

        $response['booking'] = $this->booking_model->getBookingDateListServiceAndStaff($parameters['service'], $parameters['staff']);
        $response['service'] = $this->services_model->addServiceTime($parameters['service']);
        $response['location'] = $this->location_model->addOpenedDay($response['service']['location_id']);
        $response['booking'] = $this->staff_model->addJobTime($response['booking'], $response['service']['service_time'], $parameters['staff'], $response['location']);


        return $this->returnResponse($response);
    }

    /**
     * @param $response
     * @return mixed
     */
    public function returnResponse($response)
    {
        return Response::json($response, 200, array('Access-Control-Allow-Origin' => '*'));
    }

    /**
     * @param $response
     * @return array
     */
    public function ConvertArray($response)
    {
        if (!empty($response)) {
            $response = $response->toArray();
        } else {
            $response = [];
        }
        return $response;
    }

    /**
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     * @throws \Exception
     */
    public function booking(Request $request)
    {
        //Control Required Field
        $fields = ['username', 'phone', 'email', 'service_id', 'staff_id', 'select_time', 'payment_type'];

        if (count($errors = $this->booking_model->requiredFieldControl($request->toArray(), $fields)) != 0) {
            return redirect('/booking')->with('error', $errors);
        }

        $all = $request->all();

        //remove phone first Caracter EX: old = +905xxxxx  new 905xxxx
        $all['phone'] = substr($request->full_phone, 1);
        unset($all['full_phone']);

        // Create user
        $user = $this->user_model->newQuery()->where('email', $all['email'])->first();
        if ($request->create_user == "on") {
            //Control Phone amd Email Field
            $unique = ['username', 'email', 'phone'];
            if (count($errors = $this->booking_model->uniqueFieldControl($all, $unique)) != 0) {
                return redirect('/booking')->with('error', $errors);
            }
            $password = str_random(8);
            if (is_null($user)) {
                $user = $this->user_model->newQuery()->create([
                    'email' => $all['email'],
                    'username' => $all['username'],
                    'phone_number' => $all['phone'],
                    'display_name' => $all['username'],
                    'activated' => 1,
                ]);
                $user->setAttribute('password', $password);
                $user->save();
                $this->event->dispatch(new SendUserPassword($user->id, $password));
                $this->messages->info(trans('visiosoft.module.booking::messages.password_sent', [
                    'email' => $all['email']
                ]));
            }
        }

        //Control old Booking with Date
        $search = $this->booking_model->isBooking($all['select_time'], $all['staff_id'], $all['service_id']);

        if ($search != null) {
            return redirect('/booking')->with('error', [trans('visiosoft.module.booking::messages.already_datetime')]);
        }

        $price = $this->services_model->getServicePrice($all['service_id']);

        $service_time = $this->services_model->getServiceTime($all['service_id']);

        $booking_datetime_end = new DateTime($all['select_time']);
        $booking_datetime_end = $booking_datetime_end
            ->modify('+' . $service_time . ' minutes')
            ->format('Y-m-d H:i:s');

        if (is_null($user)) {
            $user_id = null;
        } else {
            $user_id = $user->id;
        }

        $activation_code = rand(111111, 999999);
        $id = $this->booking_model->createBooking(array_merge($all, [
            'total_price' => $price,
            'user_id' => $user_id,
            'activation_code' => $activation_code,
            'booking_datetime_end' => $booking_datetime_end,
        ]));


        if ($all['payment_type'] == "paypal") {
            $items = $this->event->dispatch(new PaypalItems($id, 'Booking #' . $id, $price, 1, '', '/booking/detail/' . $id));
            return redirect('/payment/paypal/r/' . array_first($items));
        } else {
            $this->event->dispatch(new ActivationBooking($id));
        }
        return redirect('/booking/detail/' . $id)->with('success', [trans('visiosoft.module.booking::messages.success_booking')]);
    }

    /**
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function activateBooking(Request $request)
    {
        //Control Required Field
        $fields = ['activation_code', 'booking_id',];

        if (count($errors = $this->booking_model->requiredFieldControl($request->toArray(), $fields)) != 0) {
            return redirect('/booking/detail/' . $request->booking_id)->with('error', $errors);
        }

        $status = $this->booking_model->activate($request->activation_code, $request->booking_id);
        if (!$status) {
            return redirect('/booking/detail/' . $request->booking_id)->with('error', [trans('visiosoft.module.booking::messages.error_activation_code')]);
        }

        $this->event->dispatch(new CreateBooking($request->booking_id));

        return redirect('/booking/detail/' . $request->booking_id)->with('success', [trans('visiosoft.module.booking::messages.activated_booking')]);
    }


    /**
     * @param $id
     * @return \Illuminate\Contracts\View\View|mixed
     */
    public function detail($id)
    {
        $detail = $this->booking_model->find($id);
        $this->template->set('page_title', trans('visiosoft.module.booking::field.booking_details.name') . ' #' . $id);
        $this->template->set('meta_title', trans('visiosoft.module.booking::field.booking_details.name'));
        return $this->view->make('visiosoft.module.booking::detail', compact('detail', 'id'));
    }

    /**
     * @param Request $request
     */
    public function paypalreq(Request $request)
    {
        $url = $this->event->dispatch(new paymentPaypal(null, 'r', $request->token));
    }

}
