<?php namespace Visiosoft\BookingModule\Http\Controller;

use Anomaly\Streams\Platform\Http\Controller\ResourceController;
use Anomaly\UsersModule\User\UserModel;
use Carbon\Carbon;
use DateTime;
use Hashids\Hashids;
use Visiosoft\BookingModule\Booking\Book\BookFormBuilder;
use Visiosoft\BookingModule\Booking\Events\remindingTomorrow;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Response;
use Visiosoft\BookingModule\Booking\Contract\BookingRepositoryInterface;
use Visiosoft\BookingModule\Booking\Event\paymentPaypal;
use Visiosoft\BookingModule\Booking\BookingModel;
use Visiosoft\BookingModule\Booking\Events\ActivationBooking;
use Visiosoft\BookingModule\Booking\Events\CreateBooking;
use Visiosoft\BookingModule\Booking\Events\SendUserPassword;
use Visiosoft\BookingModule\Holiday\Contract\HolidayRepositoryInterface;
use Visiosoft\BookingModule\Location\LocationModel;
use Visiosoft\BookingModule\Service\ServiceModel;
use Visiosoft\BookingModule\Staff\StaffModel;
use Illuminate\Contracts\Events\Dispatcher;
use Visiosoft\PaypalModule\Events\PaypalItems;
use Visiosoft\PaypalModule\Listener\paypal;

class BookingController extends ResourceController
{
    private $booking_model;
    private $services_model;
    private $staff_model;
    private $location_model;
    private $event;
    private $user_model;
    private $holidays;
    private $bookingRepository;

    public function __construct(
        Dispatcher $events,
        ServiceModel $serviceModel,
        StaffModel $staffModel,
        LocationModel $locationModel,
        BookingModel $bookingModel,
        HolidayRepositoryInterface $holidays,
        UserModel $user_model,
        BookingRepositoryInterface $bookingRepository
    )
    {
        parent::__construct();
        $this->event = $events;
        $this->booking_model = $bookingModel;
        $this->services_model = $serviceModel;
        $this->staff_model = $staffModel;
        $this->location_model = $locationModel;
        $this->user_model = $user_model;
        $this->holidays = $holidays;
        $this->bookingRepository = $bookingRepository;
    }


    public function index(Request $request)
    {
        $email = $name = $phone = null;
        if (!is_null(Auth::user())) {
            $user = Auth::user();
            $email = $user->email;
            $name = $user->display_name;
            $phone = $user->phone_number;

        }
        $this->template->set('meta_title', trans('visiosoft.module.booking::field.booking.name'));
        $locations = $this->location_model->where('enabled', true)->get();

        $location = 'false';
        if (isset($request->location)) {
            $location = $request->location;
        }

        return $this->view->make('visiosoft.module.booking::booking', compact('locations', 'email', 'name', 'phone', 'location'));

    }

    public function srvLocation(Request $request)
    {
        //Return Service for location ID
        $response = $this->services_model
            ->where('enabled', true)
            ->where('location_id', $request->val)
            ->get();
        $response['items'] = $this->ConvertArray($response);
        $response['items'] = $this->services_model->addFotoService($response['items']);
        $response['type'] = 'service';

        return $this->returnResponse($response);
    }

    public function srvService(Request $request)
    {
        //Return Staff for service ID
        $staff_list = $this->staff_model
            ->where('enabled', true)
            ->whereRaw('services REGEXP \'.*;s:[0-9]+:"' . $request->val . '".*\'')
            ->get();
        $response['items'] = $this->ConvertArray($staff_list);
        $response['items'] = $this->staff_model->addFotoStaff($response['items']);
        $response['type'] = 'staff';
        return $this->returnResponse($response);
    }

    public function srvStaff(Request $request)
    {
        //Return Booking Date and Time for service ID, location ID, staff ID
        $parameters = json_decode($request->val, true);
        if ($parameters['staff'] == 0) {
            $bookings_for_staff = array();
            foreach ($this->staff_model->where('enabled', 1)->get() as $staff) {
                $bookings_for_staff[$staff->id] = $this->booking_model->getBookingDateListServiceAndStaff($parameters['service'], $staff->id);
                $noonList = $this->staff_model->addStaffNoonTypeBooking($staff->id, $parameters['service']);
                $bookings_for_staff[$staff->id] = array_merge($bookings_for_staff[$staff->id], $noonList);
            }
            $response['booking'] = $bookings_for_staff;

        } else {
            $response['booking'] = $this->booking_model->getBookingDateListServiceAndStaff($parameters['service'], $parameters['staff']);
        }
        $response['service'] = $this->services_model->addServiceTime($parameters['service']);
        $response['location'] = $this->location_model->addOpenedDay($response['service']['location_id']);
        $response['staff'] = $this->staff_model->addJobTime($parameters['staff']);
        $response['holidays'] = $this->holidays->all();


        return $this->returnResponse($response);
    }

    public function returnResponse($response)
    {
        return Response::json($response, 200, array('Access-Control-Allow-Origin' => '*'));
    }

    public function ConvertArray($response)
    {
        if (!empty($response)) {
            $response = $response->toArray();
        } else {
            $response = [];
        }
        return $response;
    }

    public function booking(Request $request, BookFormBuilder $builder)
    {
        $error = $builder->build()->validate()->getFormErrors()->getMessages();
        if (!empty($error)) {
            return $this->redirect->back();
        }

        $all = $request->all();

        if (Auth::check()) {
            $all['email'] = Auth::user()->email;
        }

        $all['phone'] = preg_replace("/[^0-9]/", "", $all['phone']);;

        //Control Required Field
        $fields = ['username', 'phone', 'email', 'service_id', 'select_time', 'payment_type'];

        if (count($errors = $this->booking_model->requiredFieldControl($all, $fields)) != 0) {
            return redirect('/booking')->with('error', $errors);
        }

        if ($all['staff_id'] == 0) {
            $ineligible_staff = $this->booking_model->getBookingForDate($all['select_time'])->pluck('staff_id');
            $eligible_staff = $this->staff_model->getStaffForNonListWithDate($ineligible_staff, $all['select_time'],$all['service_id']);
            if (!is_null($eligible_staff)) {
                $all['staff_id'] = $eligible_staff->id;
            } else {
                $this->messages->error(trans('visiosoft.module.booking::messages.no_appropriate_staff_found'));
                return redirect('/booking');
            }
        }

        //Check Booking for Selected date and Email (Only 1 appointment per day can be made)
        if ($this->bookingRepository->isBookingForDate($all['email'], $all['select_time'])) {
            $this->messages->error(trans('visiosoft.module.booking::messages.booking_already_on_selected_date'));
            return redirect('/booking');
        }

        // Create user
        $user = $this->user_model->newQuery()->where('email', $all['email'])->first();
        if ($request->create_user == "on") {
            //Control Phone amd Email Field
            $unique = ['username'];
            if (count($errors = $this->booking_model->uniqueFieldControl($all, $unique)) != 0) {
                return redirect('/booking')->with('error', $errors);
            }
            if (is_null($user)) {
                $password = str_random(8);
                $user = $this->user_model->newQuery()->create([
                    'email' => $all['email'],
                    'username' => $all['username'],
                    'phone_number' => $all['phone'],
                    'display_name' => $all['username'],
                    'activated' => 1,
                ]);
                $user->setAttribute('password', $password);
                $user->save();
                $this->event->dispatch(new SendUserPassword($user->id, $password));
                $this->messages->info(trans('visiosoft.module.booking::messages.password_sent', [
                    'email' => $all['email']
                ]));
            }
        }

        //Update New Phone Number for login user
        if (Auth::check()) {
            $user = $this->user_model->find(Auth::id());
            $user->update(['phone_number' => $all['phone']]);
        }

        //Control old Booking with Date
        $service = $this->services_model->find($all['service_id']);

        if (is_null($service)) {
            return redirect('/booking')->with('error', [trans('visiosoft.module.booking::messages.found_service')]);
        }

        $end = $this->getEndTime($service->time, $all['select_time']);
        $search = $this->booking_model->isBooking($all['select_time'], $all['staff_id'], $end);

        if ($search > 0) {
            return redirect('/booking')->with('error', [trans('visiosoft.module.booking::messages.already_datetime')]);
        }


        $price = $this->services_model->getServicePrice($all['service_id']);

        $service_time = $this->services_model->getServiceTime($all['service_id']);

        $booking_datetime_end = new DateTime($all['select_time']);
        $booking_datetime_end = $booking_datetime_end
            ->modify('+' . $service_time . ' minutes')
            ->format('Y-m-d H:i:s');

        //Set user
        $user_id = null;

        if (!is_null($user)) {
            $user_id = $user->id;
        }

        $activation_code = rand(111111, 999999);

        $create_params = array_merge($all, [
            'total_price' => $price,
            'user_id' => $user_id,
            'phone' => $all['phone'],
            'activation_code' => $activation_code,
            'booking_datetime_end' => $booking_datetime_end
        ]);

        if (!setting_value('visiosoft.module.booking::activation_system')) {
            $create_params['activated'] = 1;
            $create_params['status'] = 'active';
        }

        $id = $this->booking_model->createBooking($create_params);

        // Hash the ID
        $hashIds = new Hashids('V1si@b00k1ng', 5);
        $hashedId = $hashIds->encode($id, $activation_code);

        if ($all['payment_type'] == "paypal") {
            $items = $this->event->dispatch(new PaypalItems($id, 'Booking #' . $id, $price, 1, '', '/booking/detail/' . $hashedId));
            return redirect('/payment/paypal/r/' . array_first($items));
        } else if (setting_value('visiosoft.module.booking::activation_system')) {
            $this->event->dispatch(new ActivationBooking($id));
            $this->messages->success(trans('visiosoft.module.booking::messages.success_booking'));
        } else {
            $this->event->dispatch(new CreateBooking($id));
        }
        return redirect('/booking/detail/' . $hashedId);
    }

    public function activateBooking(Request $request)
    {
        //Control Required Field
        $fields = ['activation_code', 'booking_id',];

        if (count($errors = $this->booking_model->requiredFieldControl($request->toArray(), $fields)) != 0) {
            return redirect('/booking/detail/' . $request->booking_id)->with('error', $errors);
        }

        $hashIds = new Hashids('V1si@b00k1ng', 5);
        $bookingId = empty($hashIds->decode($request->booking_id)) ? null : $hashIds->decode($request->booking_id)[0];
        if (!$bookingId) {
            $this->messages->error(['visiosoft.module.booking::messages.booking_not_found']);
            return redirect('/');
        }

        // Check if booking is cancelled
        $booking = $this->bookingRepository->find($bookingId);
        if (!$booking) {
            $this->messages->error(['visiosoft.module.booking::messages.booking_cancelled']);
            return redirect('/');
        }

        $status = $this->booking_model->activate($request->activation_code, $bookingId);
        if (!$status) {
            return redirect('/booking/detail/' . $request->booking_id)->with('error', [trans('visiosoft.module.booking::messages.error_activation_code')]);
        }

        $this->event->dispatch(new CreateBooking($bookingId));

        return redirect('/booking/detail/' . $request->booking_id)->with('success', [trans('visiosoft.module.booking::messages.activated_booking')]);
    }

    public function cancelBooking(Request $request)
    {
        if (isset($this->request->booking_id)) {
            $hashedId = $this->request->booking_id;

            $hashIds = new Hashids('V1si@b00k1ng', 5);
            $id = empty($hashIds->decode($hashedId)) ? null : $hashIds->decode($hashedId)[0];

            if ($id and $detail = $this->booking_model->newQuery()->find($id)) {
                if (isset($this->request->email)) {
                    if ($this->request->email == $detail->email) {
                        $userDetail = $this->request->ip() . "-#" . Auth::id();

                        $detail->update(['updated_by_id' => Auth::id(), 'deleted_by_user' => $userDetail]);

                        $detail->delete();

//                        event(new CancellationBooking($detail));

                        $this->messages->success(['visiosoft.module.booking::messages.cancelled_booking']);

                        return redirect()->route('booking::create');
                    } else {
                        $this->messages->error(['visiosoft.module.booking::messages.confirm_email']);
                    }
                }
                return $this->view->make('visiosoft.module.booking::cancel', compact('detail', 'hashedId'));

            } else {
                $this->messages->error(['visiosoft.module.booking::messages.booking_not_found']);
            }
        }
        return redirect()->route('booking::create');
    }

    public function detail($hashedId)
    {
        $hashIds = new Hashids('V1si@b00k1ng', 5);
        $id = empty($hashIds->decode($hashedId)) ? null : $hashIds->decode($hashedId)[0];
        $detail = $this->bookingRepository->findTrashed($id);
        if ($detail && $detail->trashed()) {
            $this->messages->error(['visiosoft.module.booking::messages.booking_cancelled']);
            return redirect('/');
        } elseif ($detail) {
            $this->template->set('page_title', trans('visiosoft.module.booking::field.booking_details.name') . ' #' . $id);
            $this->template->set('meta_title', trans('visiosoft.module.booking::field.booking_details.name'));
            return $this->view->make('visiosoft.module.booking::detail', compact('detail', 'hashedId'));
        } else {
            $this->messages->error(['visiosoft.module.booking::messages.booking_not_found']);
            return redirect('/');
        }
    }

    public function paypalreq(Request $request)
    {
        $url = $this->event->dispatch(new paymentPaypal(null, 'r', $request->token));
    }

    public function ajaxGetService($id)
    {
        return $this->response->json(['status' => 'success', 'data' => $this->services_model->newQuery()->find($id)]);
    }

    public function forgotPasswordMailSent()
    {
        $this->messages->flush();
        return $this->view->make('visiosoft.theme.barberlin::password/email-sent-confirmation');
    }

    public function passwordResetConfirmation()
    {
        $this->messages->flush();
        return $this->response
            ->view('visiosoft.theme.barberlin::password/password-changed-confirmation')
            ->header("Refresh", "5;url=/login");
    }

    public function remindingCron()
    {
        $tomorrow = date('Y-m-d', strtotime("+1 days"));
        $bookings = $this->booking_model->newQuery()
            ->whereDate('booking_datetime_start', '=', $tomorrow)
            ->where('activated', 1)
            ->where('status', 'active')
            ->where('remindingTomorrow', '!=', true)
            ->get();

        foreach ($bookings as $booking) {
            event(new remindingTomorrow($booking));
            $booking->update(['remindingTomorrow' => true]);
        }
    }

    public function validateBooking()
    {
        $errors = array();

        $all = \request()->all();

        $all['phone'] = preg_replace("/[^0-9]/", "", $all['phone']);

        //Control Required Field
        $fields = ['username', 'phone', 'service_id', 'select_time', 'payment_type'];

        if (!Auth::check()) {
            $fields[] = 'email';
        }

        if (count($requiredErrors = $this->booking_model->requiredFieldControl($all, $fields)) != 0) {
            $errors = array_merge($errors, $requiredErrors);
        }

        if ($all['staff_id'] == 0) {
            $ineligible_staff = $this->booking_model->getBookingForDate($all['select_time'])->pluck('staff_id');
            $eligible_staff = $this->staff_model->getStaffForNonListWithDate($ineligible_staff, $all['select_time'],$all['service_id']);
            if (is_null($eligible_staff)) {
                $errors[] = trans('visiosoft.module.booking::messages.no_appropriate_staff_found');
            } else {
                $all['staff_id'] = $eligible_staff->id;
            }
        }

        if (!isset($all['email'])) {
            $all['email'] = Auth::user()->email;
        }

        //Check Booking for Selected date and Email (Only 1 appointment per day can be made)
        if ($this->bookingRepository->isBookingForDate($all['email'], $all['select_time'])) {
            $errors[] = trans('visiosoft.module.booking::messages.booking_already_on_selected_date');
        }

        //Control old Booking with Date
        $service = $this->services_model->find($all['service_id']);

        if (!is_null($service)) {
            $end = $this->getEndTime($service->time, $all['select_time']);
            $search = $this->booking_model->isBooking($all['select_time'], $all['staff_id'], $end);

            if ($search > 0) {
                $errors[] = trans('visiosoft.module.booking::messages.already_datetime');
            }
        } else {
            $errors[] = trans('visiosoft.module.booking::messages.found_service');
        }

        return \response()->json(['errors' => $errors]);
    }

    public function getEndTime($service_time, $start)
    {
        $end = new DateTime($start);
        $end = $end->modify('+' . $service_time . ' minutes')->format('Y-m-d H:i');
        return $end;
    }
}
