<?php namespace Visiosoft\BookingModule\Http\Controller\Admin;

use Anomaly\Streams\Platform\Entry\EntryModel;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Visiosoft\BookingModule\Booking\BookingExport;
use Visiosoft\BookingModule\Booking\BookingModel;
use Maatwebsite\Excel\Facades\Excel;
use Visiosoft\BookingModule\Booking\Contract\BookingRepositoryInterface;
use Visiosoft\BookingModule\Booking\Events\CancellationBooking;
use Visiosoft\BookingModule\Booking\Form\BookingFormBuilder;
use Visiosoft\BookingModule\Booking\Table\BookingTableBuilder;
use Anomaly\Streams\Platform\Http\Controller\AdminController;
use Visiosoft\BookingModule\Service\ServiceModel;
use Illuminate\Http\Request;
use Visiosoft\BookingModule\Staff\Contract\StaffRepositoryInterface;
use Visiosoft\BookingModule\Staff\StaffModel;

class BookingController extends AdminController
{
    private $model;
    private $request_params;
    private $repository;
    private $staffRepository;

    public function __construct(
        BookingModel $model,
        Request $request,
        BookingRepositoryInterface $repository,
        StaffRepositoryInterface $staffRepository
    )
    {
        parent::__construct();
        $this->model = $model;
        $this->repository = $repository;
        $this->staffRepository = $staffRepository;
        $this->request_params = $request->all();
    }

    public function index(BookingTableBuilder $table)
    {
        if (!isset($this->request->view) || $this->request->view !== "booking") {
            $table->setColumns(['staff', 'booking_datetime_start',
                'services_select' => [
                    'value' => function (EntryModel $entry) {
                        return $this->getServiceName($entry->services_select);
                    }
                ]
                , 'name', 'phone', 'email']);
        }

        return $table->render();
    }

    public function bookingJSON()
    {
        $all_bookings = $this->model
            ->newQuery()
            ->join('booking_services', function ($join) {
                $join->on('booking_booking.services_select', '=', 'booking_services.id');
                if ($locationId = \request()->filter_location) {
                    $join->where('location_id', $locationId);
                }
            })->select([
                'booking_booking.name as title',
                'booking_booking.phone as phone',
                'booking_booking.email as email',
                'booking_booking.booking_datetime_start as start',
                'booking_booking.booking_datetime_end as end',
                'booking_booking.id as id',
                'booking_booking.staff_id as resourceId',
                'booking_services.color as color',
            ])->whereBetween('booking_datetime_start', [\request()->start, \request()->end]);

        if ($this->request->has('group')) {
            $all_bookings = $all_bookings->groupBy($this->request->get('group'));
        }

        $all_bookings = $all_bookings->get();

        return $this->response->json($all_bookings);
    }

    public function staffJSON()
    {
        $all_bookings = StaffModel::query()
            ->newQuery()
            ->select([
                'booking_staff.id as id',
                'booking_staff.name as title',
            ])
            ->get();
        return $this->response->json($all_bookings);
    }

    public function create(BookingFormBuilder $form)
    {
        return $form->render();
    }

    public function edit(BookingFormBuilder $form, $id)
    {
        return $form->render($id);
    }

    public function getServiceName($id)
    {
        $serviceModel = new ServiceModel();
        return $serviceModel->getServiceName($id) . " - " . $serviceModel->getServicePrice($id) . " " . substr(setting_value('streams::currency'), 0, 3);
    }

    public function getDetails(BookingModel $bookingModel, $id)
    {
        return $this->response->json($bookingModel->getDetail($id));
    }

    public function createAjax()
    {
        $params = $this->request_params;
        $status = "error_create";
        $modal = 'show';
        unset($params['locations']);

        $search = $this->model->isBooking($params['booking_datetime_start'], $params['staff'], $params['booking_datetime_end']);

        $check_job_time = $this->checkJobTime($params['staff'], $params['booking_datetime_start'], $params['booking_datetime_end']);

        if ($search == 0 && $check_job_time) {
            $this->repository->create(array_merge($params,[
                'created_at' => Carbon::now(),
            ]));
            $status = "create_success";
            $modal = 'hide';
        }

        return response()->json(['status' => $status, 'form' => $modal]);
    }

    public function editAjax()
    {
        $params = $this->request_params;
        $status = "found_booking";
        $modal = 'hide';
        unset($params['locations']);
        $booking = $this->model->find($params['id']);
        if ($booking) {
            unset($params['id']);
            $params['payment_type'] = $booking->payment_type;
            $params['email'] = $booking->email;
            $params['total_price'] = $booking->total_price;

            $userDetail = $this->request->ip() . "-#" . Auth::id();

            $booking->update(['updated_by_id' => Auth::id(), 'deleted_by_user' => $userDetail]);
            $booking->delete();

            $booking->create($params);
            $status = "edit_success";
        }
        return response()->json(['status' => $status, 'form' => $modal]);

    }

    public function deleteAjax($id)
    {
        $booking = $this->model->newQuery()->find($id);
        $userDetail = $this->request->ip() . "-#" . Auth::id();

        $booking->update(['updated_by_id' => Auth::id(), 'deleted_by_user' => $userDetail]);
        $booking->delete();

        event(new CancellationBooking($booking));


        $this->messages->success(trans('streams::message.delete_success', ['count' => 1]));
        return $this->redirect->to('/admin/booking/booking');
    }

    public function approveAjax($id)
    {
        $this->model->activateById($id);
        return response()->json(['status' => true]);
    }

    public function ApprovePayment($id)
    {
        $this->repository->changePaymentStatus($id, 'success');
        $this->messages->success([trans('visiosoft.module.booking::messages.approvedPaymentMsg')]);
        return $this->redirect->to('/admin/booking/booking');
    }

    public function ApprovePaymentAjax($id)
    {
        $this->repository->changePaymentStatus($id, 'success');
        return response()->json(['status' => true]);
    }

    public function exportExcel()
    {
        return Excel::download(new BookingExport(), 'reservation-' . time() . '.xlsx');
    }

    public function checkJobTime($staff_id, $start, $finish)
    {
        $staff = $this->staffRepository->newQuery()
            ->find($staff_id);

        $start = Carbon::make($start);
        $finish = Carbon::make($finish);

        $staff_today_start = $staff[strtolower($start->format('l')) . "Start"];
        $staff_today_start = Carbon::make($start->format('Y-m-d')." ".$staff_today_start->format('H:i'));

        $staff_today_finish = $staff[strtolower($finish->format('l')) . "End"];
        $staff_today_finish = Carbon::make($finish->format('Y-m-d')." ".$staff_today_finish->format('H:i'));


        // Check Start Job Time
        if ($start->format('U') < $staff_today_start->format('U')) {
            return false;
        }

        // Check Finish Job Time
        if ($finish->format('U') > $staff_today_finish->format('U')) {
            return false;
        }

        $staff_today_start_noon = $staff[strtolower($start->format('l')) . "StartNoon"];
        $staff_today_start_noon = Carbon::make($start->format('Y-m-d')." ".$staff_today_start_noon->format('H:i'));

        $staff_today_finish_noon = $staff[strtolower($finish->format('l')) . "EndNoon"];
        $staff_today_finish_noon = Carbon::make($finish->format('Y-m-d')." ".$staff_today_finish_noon->format('H:i'));


        if ($staff_today_start_noon->format('U') <= $start->format('U') && $start->format('U') < $staff_today_finish_noon->format('U')) {
            return false;
        }


        if ($staff_today_start_noon->format('U') < $finish->format('U') && $finish->format('U') <= $staff_today_finish_noon->format('U')) {
            return false;
        }

        return true;
    }
}
