<?php namespace Visiosoft\BookingModule\Http\Controller\Admin;

use Anomaly\SettingsModule\Setting\Contract\SettingRepositoryInterface;
use Anomaly\Streams\Platform\Entry\EntryModel;
use Anomaly\UsersModule\User\UserModel;
use Illuminate\Support\Collection;
use PHPUnit\Util\Json;
use Visiosoft\BookingModule\Booking\BookingModel;
use Illuminate\Contracts\Config\Repository;
use Visiosoft\BookingModule\Booking\Contract\BookingRepositoryInterface;
use Visiosoft\BookingModule\Booking\Form\BookingFormBuilder;
use Visiosoft\BookingModule\Booking\Table\BookingTableBuilder;
use Anomaly\Streams\Platform\Http\Controller\AdminController;
use Visiosoft\BookingModule\Service\ServiceModel;
use Illuminate\Http\Request;
use Visiosoft\BookingModule\Staff\StaffModel;

class BookingController extends AdminController
{
    private $model;
    private $request_params;
    private $repository;

    public function __construct(BookingModel $model, Request $request, BookingRepositoryInterface $repository)
    {
        parent::__construct();
        $this->model = $model;
        $this->repository = $repository;
        $this->request_params = $request->all();
    }

    /**
     * Display an index of existing entries.
     *
     * @param BookingTableBuilder $table
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function index(BookingTableBuilder $table)
    {
        if (isset($this->request->view) and $this->request->view == "booking") {
            $table->setModel(ServiceModel::class);
        }
        $table->setColumns(['staff', 'booking_datetime_start',
            'services_select' => [
                'value' => function (EntryModel $entry, SettingRepositoryInterface $setting) {
                    return $this->getServiceName($entry->services_select, $setting);
                }
            ]
            , 'name', 'phone', 'email']);
        return $table->render();
    }

    public function bookingJSON()
    {
        $all_bookings = $this->model
            ->newQuery()
            ->leftJoin('booking_services', function ($join) {
                $join->on('booking_services.id', '=', 'booking_booking.services_select');
            })
            ->select([
                'booking_booking.name as title',
                'booking_booking.booking_datetime_start as start',
                'booking_booking.booking_datetime_end as end',
                'booking_booking.id as id',
                'booking_booking.staff_id as resourceId',
                'booking_services.color as color',
            ])
            ->get();


        return $this->response->json($all_bookings);
    }

    public function staffJSON()
    {
        $all_bookings = StaffModel::query()
            ->newQuery()
            ->select([
                'booking_staff.id as id',
                'booking_staff.name as title',
            ])
            ->get();
        return $this->response->json($all_bookings);
    }

    /**
     * Create a new entry.
     *
     * @param BookingFormBuilder $form
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function create(BookingFormBuilder $form)
    {
        return $form->render();
    }

    /**
     * Edit an existing entry.
     *
     * @param BookingFormBuilder $form
     * @param        $id
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function edit(BookingFormBuilder $form, $id)
    {
        return $form->render($id);
    }

    /**
     * @param $id
     * @param $setting
     * @return string
     */
    public function getServiceName($id, $setting)
    {
        $serviceModel = new ServiceModel();
        return $serviceModel->getServiceName($id) . " - " . $serviceModel->getServicePrice($id) . " " . substr($setting->value('streams::currency'), 0, 3);
    }

    /**
     * @param BookingModel $bookingModel
     * @param $id
     */
    public function getDetails(BookingModel $bookingModel, $id)
    {
        return $this->response->json($bookingModel->getDetail($id));
    }

    public function createAjax()
    {
        $params = $this->request_params;
        unset($params['locations']);
        $this->model->create($params);
        return response()->json(['status' => 'success']);
    }

    public function editAjax()
    {
        $params = $this->request_params;
        $status = "error";
        unset($params['locations']);
        $booking = $this->model->find($params['id']);
        if ($booking) {
            if ($booking->payment_type == "paypal") {
                unset($params['payment_type'], $params['email'], $params['total_price']);
            }
            $booking->update($params);
            $status = "success";
        }
        return response()->json(['status' => $status]);

    }

    public function deleteAjax($id)
    {
        $this->model->newQuery()->find($id)->delete();
        $this->messages->success(trans('streams::message.delete_success', ['count' => 1]));
        return $this->redirect->to('/admin/booking/booking');
    }

    public function approveAjax($id)
    {
        $this->model->activateById($id);
        return response()->json(['status' => true]);
    }

    public function ApprovePayment($id)
    {
        $this->repository->changePaymentStatus($id, 'success');
        $this->messages->success([trans('visiosoft.module.booking::messages.approvedPaymentMsg')]);
        return $this->redirect->to('/admin/booking/booking');
    }

    public function ApprovePaymentAjax($id)
    {
        $this->repository->changePaymentStatus($id, 'success');
        return response()->json(['status' => true]);
    }
}
