/*!
 * FullCalendar v3.1.0
 * Docs & License: http://fullcalendar.io/
 * (c) 2016 Adam Shaw
 */

(function(factory) {
	if (typeof define === 'function' && define.amd) {
		define([ 'jquery', 'moment' ], factory);
	}
	else if (typeof exports === 'object') { // Node/CommonJS
		module.exports = factory(require('jquery'), require('moment'));
	}
	else {
		factory(jQuery, moment);
	}
})(function($, moment) {

	;;

	var FC = $.fullCalendar = {
		version: "3.1.0",
		internalApiVersion: 7
	};
	var fcViews = FC.views = {};


	$.fn.fullCalendar = function(options) {
		var args = Array.prototype.slice.call(arguments, 1); // for a possible method call
		var res = this; // what this function will return (this jQuery object by default)

		this.each(function(i, _element) { // loop each DOM element involved
			var element = $(_element);
			var calendar = element.data('fullCalendar'); // get the existing calendar object (if any)
			var singleRes; // the returned value of this single method call

			// a method call
			if (typeof options === 'string') {
				if (calendar && $.isFunction(calendar[options])) {
					singleRes = calendar[options].apply(calendar, args);
					if (!i) {
						res = singleRes; // record the first method call result
					}
					if (options === 'destroy') { // for the destroy method, must remove Calendar object data
						element.removeData('fullCalendar');
					}
				}
			}
			// a new calendar initialization
			else if (!calendar) { // don't initialize twice
				calendar = new Calendar(element, options);
				element.data('fullCalendar', calendar);
				calendar.render();
			}
		});

		return res;
	};


	var complexOptions = [ // names of options that are objects whose properties should be combined
		'header',
		'footer',
		'buttonText',
		'buttonIcons',
		'themeButtonIcons'
	];


// Merges an array of option objects into a single object
	function mergeOptions(optionObjs) {
		return mergeProps(optionObjs, complexOptions);
	}

	;;

// exports
	FC.intersectRanges = intersectRanges;
	FC.applyAll = applyAll;
	FC.debounce = debounce;
	FC.isInt = isInt;
	FC.htmlEscape = htmlEscape;
	FC.cssToStr = cssToStr;
	FC.proxy = proxy;
	FC.capitaliseFirstLetter = capitaliseFirstLetter;


	/* FullCalendar-specific DOM Utilities
----------------------------------------------------------------------------------------------------------------------*/


// Given the scrollbar widths of some other container, create borders/margins on rowEls in order to match the left
// and right space that was offset by the scrollbars. A 1-pixel border first, then margin beyond that.
	function compensateScroll(rowEls, scrollbarWidths) {
		if (scrollbarWidths.left) {
			rowEls.css({
				'border-left-width': 1,
				'margin-left': scrollbarWidths.left - 1
			});
		}
		if (scrollbarWidths.right) {
			rowEls.css({
				'border-right-width': 1,
				'margin-right': scrollbarWidths.right - 1
			});
		}
	}


// Undoes compensateScroll and restores all borders/margins
	function uncompensateScroll(rowEls) {
		rowEls.css({
			'margin-left': '',
			'margin-right': '',
			'border-left-width': '',
			'border-right-width': ''
		});
	}


// Make the mouse cursor express that an event is not allowed in the current area
	function disableCursor() {
		$('body').addClass('fc-not-allowed');
	}


// Returns the mouse cursor to its original look
	function enableCursor() {
		$('body').removeClass('fc-not-allowed');
	}


// Given a total available height to fill, have `els` (essentially child rows) expand to accomodate.
// By default, all elements that are shorter than the recommended height are expanded uniformly, not considering
// any other els that are already too tall. if `shouldRedistribute` is on, it considers these tall rows and
// reduces the available height.
	function distributeHeight(els, availableHeight, shouldRedistribute) {

		// *FLOORING NOTE*: we floor in certain places because zoom can give inaccurate floating-point dimensions,
		// and it is better to be shorter than taller, to avoid creating unnecessary scrollbars.

		var minOffset1 = Math.floor(availableHeight / els.length); // for non-last element
		var minOffset2 = Math.floor(availableHeight - minOffset1 * (els.length - 1)); // for last element *FLOORING NOTE*
		var flexEls = []; // elements that are allowed to expand. array of DOM nodes
		var flexOffsets = []; // amount of vertical space it takes up
		var flexHeights = []; // actual css height
		var usedHeight = 0;

		undistributeHeight(els); // give all elements their natural height

		// find elements that are below the recommended height (expandable).
		// important to query for heights in a single first pass (to avoid reflow oscillation).
		els.each(function(i, el) {
			var minOffset = i === els.length - 1 ? minOffset2 : minOffset1;
			var naturalOffset = $(el).outerHeight(true);

			if (naturalOffset < minOffset) {
				flexEls.push(el);
				flexOffsets.push(naturalOffset);
				flexHeights.push($(el).height());
			}
			else {
				// this element stretches past recommended height (non-expandable). mark the space as occupied.
				usedHeight += naturalOffset;
			}
		});

		// readjust the recommended height to only consider the height available to non-maxed-out rows.
		if (shouldRedistribute) {
			availableHeight -= usedHeight;
			minOffset1 = Math.floor(availableHeight / flexEls.length);
			minOffset2 = Math.floor(availableHeight - minOffset1 * (flexEls.length - 1)); // *FLOORING NOTE*
		}

		// assign heights to all expandable elements
		$(flexEls).each(function(i, el) {
			var minOffset = i === flexEls.length - 1 ? minOffset2 : minOffset1;
			var naturalOffset = flexOffsets[i];
			var naturalHeight = flexHeights[i];
			var newHeight = minOffset - (naturalOffset - naturalHeight); // subtract the margin/padding

			if (naturalOffset < minOffset) { // we check this again because redistribution might have changed things
				$(el).height(newHeight);
			}
		});
	}


// Undoes distrubuteHeight, restoring all els to their natural height
	function undistributeHeight(els) {
		els.height('');
	}


// Given `els`, a jQuery set of <td> cells, find the cell with the largest natural width and set the widths of all the
// cells to be that width.
// PREREQUISITE: if you want a cell to take up width, it needs to have a single inner element w/ display:inline
	function matchCellWidths(els) {
		var maxInnerWidth = 0;

		els.find('> *').each(function(i, innerEl) {
			var innerWidth = $(innerEl).outerWidth();
			if (innerWidth > maxInnerWidth) {
				maxInnerWidth = innerWidth;
			}
		});

		maxInnerWidth++; // sometimes not accurate of width the text needs to stay on one line. insurance

		els.width(maxInnerWidth);

		return maxInnerWidth;
	}


// Given one element that resides inside another,
// Subtracts the height of the inner element from the outer element.
	function subtractInnerElHeight(outerEl, innerEl) {
		var both = outerEl.add(innerEl);
		var diff;

		// effin' IE8/9/10/11 sometimes returns 0 for dimensions. this weird hack was the only thing that worked
		both.css({
			position: 'relative', // cause a reflow, which will force fresh dimension recalculation
			left: -1 // ensure reflow in case the el was already relative. negative is less likely to cause new scroll
		});
		diff = outerEl.outerHeight() - innerEl.outerHeight(); // grab the dimensions
		both.css({ position: '', left: '' }); // undo hack

		return diff;
	}


	/* Element Geom Utilities
----------------------------------------------------------------------------------------------------------------------*/

	FC.getOuterRect = getOuterRect;
	FC.getClientRect = getClientRect;
	FC.getContentRect = getContentRect;
	FC.getScrollbarWidths = getScrollbarWidths;


// borrowed from https://github.com/jquery/jquery-ui/blob/1.11.0/ui/core.js#L51
	function getScrollParent(el) {
		var position = el.css('position'),
			scrollParent = el.parents().filter(function() {
				var parent = $(this);
				return (/(auto|scroll)/).test(
					parent.css('overflow') + parent.css('overflow-y') + parent.css('overflow-x')
				);
			}).eq(0);

		return position === 'fixed' || !scrollParent.length ? $(el[0].ownerDocument || document) : scrollParent;
	}


// Queries the outer bounding area of a jQuery element.
// Returns a rectangle with absolute coordinates: left, right (exclusive), top, bottom (exclusive).
// Origin is optional.
	function getOuterRect(el, origin) {
		var offset = el.offset();
		var left = offset.left - (origin ? origin.left : 0);
		var top = offset.top - (origin ? origin.top : 0);

		return {
			left: left,
			right: left + el.outerWidth(),
			top: top,
			bottom: top + el.outerHeight()
		};
	}


// Queries the area within the margin/border/scrollbars of a jQuery element. Does not go within the padding.
// Returns a rectangle with absolute coordinates: left, right (exclusive), top, bottom (exclusive).
// Origin is optional.
// NOTE: should use clientLeft/clientTop, but very unreliable cross-browser.
	function getClientRect(el, origin) {
		var offset = el.offset();
		var scrollbarWidths = getScrollbarWidths(el);
		var left = offset.left + getCssFloat(el, 'border-left-width') + scrollbarWidths.left - (origin ? origin.left : 0);
		var top = offset.top + getCssFloat(el, 'border-top-width') + scrollbarWidths.top - (origin ? origin.top : 0);

		return {
			left: left,
			right: left + el[0].clientWidth, // clientWidth includes padding but NOT scrollbars
			top: top,
			bottom: top + el[0].clientHeight // clientHeight includes padding but NOT scrollbars
		};
	}


// Queries the area within the margin/border/padding of a jQuery element. Assumed not to have scrollbars.
// Returns a rectangle with absolute coordinates: left, right (exclusive), top, bottom (exclusive).
// Origin is optional.
	function getContentRect(el, origin) {
		var offset = el.offset(); // just outside of border, margin not included
		var left = offset.left + getCssFloat(el, 'border-left-width') + getCssFloat(el, 'padding-left') -
			(origin ? origin.left : 0);
		var top = offset.top + getCssFloat(el, 'border-top-width') + getCssFloat(el, 'padding-top') -
			(origin ? origin.top : 0);

		return {
			left: left,
			right: left + el.width(),
			top: top,
			bottom: top + el.height()
		};
	}


// Returns the computed left/right/top/bottom scrollbar widths for the given jQuery element.
// NOTE: should use clientLeft/clientTop, but very unreliable cross-browser.
	function getScrollbarWidths(el) {
		var leftRightWidth = el.innerWidth() - el[0].clientWidth; // the paddings cancel out, leaving the scrollbars
		var widths = {
			left: 0,
			right: 0,
			top: 0,
			bottom: el.innerHeight() - el[0].clientHeight // the paddings cancel out, leaving the bottom scrollbar
		};

		if (getIsLeftRtlScrollbars() && el.css('direction') == 'rtl') { // is the scrollbar on the left side?
			widths.left = leftRightWidth;
		}
		else {
			widths.right = leftRightWidth;
		}

		return widths;
	}


// Logic for determining if, when the element is right-to-left, the scrollbar appears on the left side

	var _isLeftRtlScrollbars = null;

	function getIsLeftRtlScrollbars() { // responsible for caching the computation
		if (_isLeftRtlScrollbars === null) {
			_isLeftRtlScrollbars = computeIsLeftRtlScrollbars();
		}
		return _isLeftRtlScrollbars;
	}

	function computeIsLeftRtlScrollbars() { // creates an offscreen test element, then removes it
		var el = $('<div><div/></div>')
			.css({
				position: 'absolute',
				top: -1000,
				left: 0,
				border: 0,
				padding: 0,
				overflow: 'scroll',
				direction: 'rtl'
			})
			.appendTo('body');
		var innerEl = el.children();
		var res = innerEl.offset().left > el.offset().left; // is the inner div shifted to accommodate a left scrollbar?
		el.remove();
		return res;
	}


// Retrieves a jQuery element's computed CSS value as a floating-point number.
// If the queried value is non-numeric (ex: IE can return "medium" for border width), will just return zero.
	function getCssFloat(el, prop) {
		return parseFloat(el.css(prop)) || 0;
	}


	/* Mouse / Touch Utilities
----------------------------------------------------------------------------------------------------------------------*/

	FC.preventDefault = preventDefault;


// Returns a boolean whether this was a left mouse click and no ctrl key (which means right click on Mac)
	function isPrimaryMouseButton(ev) {
		return ev.which == 1 && !ev.ctrlKey;
	}


	function getEvX(ev) {
		if (ev.pageX !== undefined) {
			return ev.pageX;
		}
		var touches = ev.originalEvent.touches;
		if (touches) {
			return touches[0].pageX;
		}
	}


	function getEvY(ev) {
		if (ev.pageY !== undefined) {
			return ev.pageY;
		}
		var touches = ev.originalEvent.touches;
		if (touches) {
			return touches[0].pageY;
		}
	}


	function getEvIsTouch(ev) {
		return /^touch/.test(ev.type);
	}


	function preventSelection(el) {
		el.addClass('fc-unselectable')
			.on('selectstart', preventDefault);
	}


// Stops a mouse/touch event from doing it's native browser action
	function preventDefault(ev) {
		ev.preventDefault();
	}


// attach a handler to get called when ANY scroll action happens on the page.
// this was impossible to do with normal on/off because 'scroll' doesn't bubble.
// http://stackoverflow.com/a/32954565/96342
// returns `true` on success.
	function bindAnyScroll(handler) {
		if (window.addEventListener) {
			window.addEventListener('scroll', handler, true); // useCapture=true
			return true;
		}
		return false;
	}


// undoes bindAnyScroll. must pass in the original function.
// returns `true` on success.
	function unbindAnyScroll(handler) {
		if (window.removeEventListener) {
			window.removeEventListener('scroll', handler, true); // useCapture=true
			return true;
		}
		return false;
	}


	/* General Geometry Utils
----------------------------------------------------------------------------------------------------------------------*/

	FC.intersectRects = intersectRects;

// Returns a new rectangle that is the intersection of the two rectangles. If they don't intersect, returns false
	function intersectRects(rect1, rect2) {
		var res = {
			left: Math.max(rect1.left, rect2.left),
			right: Math.min(rect1.right, rect2.right),
			top: Math.max(rect1.top, rect2.top),
			bottom: Math.min(rect1.bottom, rect2.bottom)
		};

		if (res.left < res.right && res.top < res.bottom) {
			return res;
		}
		return false;
	}


// Returns a new point that will have been moved to reside within the given rectangle
	function constrainPoint(point, rect) {
		return {
			left: Math.min(Math.max(point.left, rect.left), rect.right),
			top: Math.min(Math.max(point.top, rect.top), rect.bottom)
		};
	}


// Returns a point that is the center of the given rectangle
	function getRectCenter(rect) {
		return {
			left: (rect.left + rect.right) / 2,
			top: (rect.top + rect.bottom) / 2
		};
	}


// Subtracts point2's coordinates from point1's coordinates, returning a delta
	function diffPoints(point1, point2) {
		return {
			left: point1.left - point2.left,
			top: point1.top - point2.top
		};
	}


	/* Object Ordering by Field
----------------------------------------------------------------------------------------------------------------------*/

	FC.parseFieldSpecs = parseFieldSpecs;
	FC.compareByFieldSpecs = compareByFieldSpecs;
	FC.compareByFieldSpec = compareByFieldSpec;
	FC.flexibleCompare = flexibleCompare;


	function parseFieldSpecs(input) {
		var specs = [];
		var tokens = [];
		var i, token;

		if (typeof input === 'string') {
			tokens = input.split(/\s*,\s*/);
		}
		else if (typeof input === 'function') {
			tokens = [ input ];
		}
		else if ($.isArray(input)) {
			tokens = input;
		}

		for (i = 0; i < tokens.length; i++) {
			token = tokens[i];

			if (typeof token === 'string') {
				specs.push(
					token.charAt(0) == '-' ?
						{ field: token.substring(1), order: -1 } :
						{ field: token, order: 1 }
				);
			}
			else if (typeof token === 'function') {
				specs.push({ func: token });
			}
		}

		return specs;
	}


	function compareByFieldSpecs(obj1, obj2, fieldSpecs) {
		var i;
		var cmp;

		for (i = 0; i < fieldSpecs.length; i++) {
			cmp = compareByFieldSpec(obj1, obj2, fieldSpecs[i]);
			if (cmp) {
				return cmp;
			}
		}

		return 0;
	}


	function compareByFieldSpec(obj1, obj2, fieldSpec) {
		if (fieldSpec.func) {
			return fieldSpec.func(obj1, obj2);
		}
		return flexibleCompare(obj1[fieldSpec.field], obj2[fieldSpec.field]) *
			(fieldSpec.order || 1);
	}


	function flexibleCompare(a, b) {
		if (!a && !b) {
			return 0;
		}
		if (b == null) {
			return -1;
		}
		if (a == null) {
			return 1;
		}
		if ($.type(a) === 'string' || $.type(b) === 'string') {
			return String(a).localeCompare(String(b));
		}
		return a - b;
	}


	/* FullCalendar-specific Misc Utilities
----------------------------------------------------------------------------------------------------------------------*/


// Computes the intersection of the two ranges. Will return fresh date clones in a range.
// Returns undefined if no intersection.
// Expects all dates to be normalized to the same timezone beforehand.
// TODO: move to date section?
	function intersectRanges(subjectRange, constraintRange) {
		var subjectStart = subjectRange.start;
		var subjectEnd = subjectRange.end;
		var constraintStart = constraintRange.start;
		var constraintEnd = constraintRange.end;
		var segStart, segEnd;
		var isStart, isEnd;

		if (subjectEnd > constraintStart && subjectStart < constraintEnd) { // in bounds at all?

			if (subjectStart >= constraintStart) {
				segStart = subjectStart.clone();
				isStart = true;
			}
			else {
				segStart = constraintStart.clone();
				isStart =  false;
			}

			if (subjectEnd <= constraintEnd) {
				segEnd = subjectEnd.clone();
				isEnd = true;
			}
			else {
				segEnd = constraintEnd.clone();
				isEnd = false;
			}

			return {
				start: segStart,
				end: segEnd,
				isStart: isStart,
				isEnd: isEnd
			};
		}
	}


	/* Date Utilities
----------------------------------------------------------------------------------------------------------------------*/

	FC.computeIntervalUnit = computeIntervalUnit;
	FC.divideRangeByDuration = divideRangeByDuration;
	FC.divideDurationByDuration = divideDurationByDuration;
	FC.multiplyDuration = multiplyDuration;
	FC.durationHasTime = durationHasTime;

	var dayIDs = [ 'sun', 'mon', 'tue', 'wed', 'thu', 'fri', 'sat' ];
	var intervalUnits = [ 'year', 'month', 'week', 'day', 'hour', 'minute', 'second', 'millisecond' ];


// Diffs the two moments into a Duration where full-days are recorded first, then the remaining time.
// Moments will have their timezones normalized.
	function diffDayTime(a, b) {
		return moment.duration({
			days: a.clone().stripTime().diff(b.clone().stripTime(), 'days'),
			ms: a.time() - b.time() // time-of-day from day start. disregards timezone
		});
	}


// Diffs the two moments via their start-of-day (regardless of timezone). Produces whole-day durations.
	function diffDay(a, b) {
		return moment.duration({
			days: a.clone().stripTime().diff(b.clone().stripTime(), 'days')
		});
	}


// Diffs two moments, producing a duration, made of a whole-unit-increment of the given unit. Uses rounding.
	function diffByUnit(a, b, unit) {
		return moment.duration(
			Math.round(a.diff(b, unit, true)), // returnFloat=true
			unit
		);
	}


// Computes the unit name of the largest whole-unit period of time.
// For example, 48 hours will be "days" whereas 49 hours will be "hours".
// Accepts start/end, a range object, or an original duration object.
	function computeIntervalUnit(start, end) {
		var i, unit;
		var val;

		for (i = 0; i < intervalUnits.length; i++) {
			unit = intervalUnits[i];
			val = computeRangeAs(unit, start, end);

			if (val >= 1 && isInt(val)) {
				break;
			}
		}

		return unit; // will be "milliseconds" if nothing else matches
	}


// Computes the number of units (like "hours") in the given range.
// Range can be a {start,end} object, separate start/end args, or a Duration.
// Results are based on Moment's .as() and .diff() methods, so results can depend on internal handling
// of month-diffing logic (which tends to vary from version to version).
	function computeRangeAs(unit, start, end) {

		if (end != null) { // given start, end
			return end.diff(start, unit, true);
		}
		else if (moment.isDuration(start)) { // given duration
			return start.as(unit);
		}
		else { // given { start, end } range object
			return start.end.diff(start.start, unit, true);
		}
	}


// Intelligently divides a range (specified by a start/end params) by a duration
	function divideRangeByDuration(start, end, dur) {
		var months;

		if (durationHasTime(dur)) {
			return (end - start) / dur;
		}
		months = dur.asMonths();
		if (Math.abs(months) >= 1 && isInt(months)) {
			return end.diff(start, 'months', true) / months;
		}
		return end.diff(start, 'days', true) / dur.asDays();
	}


// Intelligently divides one duration by another
	function divideDurationByDuration(dur1, dur2) {
		var months1, months2;

		if (durationHasTime(dur1) || durationHasTime(dur2)) {
			return dur1 / dur2;
		}
		months1 = dur1.asMonths();
		months2 = dur2.asMonths();
		if (
			Math.abs(months1) >= 1 && isInt(months1) &&
			Math.abs(months2) >= 1 && isInt(months2)
		) {
			return months1 / months2;
		}
		return dur1.asDays() / dur2.asDays();
	}


// Intelligently multiplies a duration by a number
	function multiplyDuration(dur, n) {
		var months;

		if (durationHasTime(dur)) {
			return moment.duration(dur * n);
		}
		months = dur.asMonths();
		if (Math.abs(months) >= 1 && isInt(months)) {
			return moment.duration({ months: months * n });
		}
		return moment.duration({ days: dur.asDays() * n });
	}


// Returns a boolean about whether the given duration has any time parts (hours/minutes/seconds/ms)
	function durationHasTime(dur) {
		return Boolean(dur.hours() || dur.minutes() || dur.seconds() || dur.milliseconds());
	}


	function isNativeDate(input) {
		return  Object.prototype.toString.call(input) === '[object Date]' || input instanceof Date;
	}


// Returns a boolean about whether the given input is a time string, like "06:40:00" or "06:00"
	function isTimeString(str) {
		return /^\d+\:\d+(?:\:\d+\.?(?:\d{3})?)?$/.test(str);
	}


	/* Logging and Debug
----------------------------------------------------------------------------------------------------------------------*/

	FC.log = function() {
		var console = window.console;

		if (console && console.log) {
			return console.log.apply(console, arguments);
		}
	};

	FC.warn = function() {
		var console = window.console;

		if (console && console.warn) {
			return console.warn.apply(console, arguments);
		}
		else {
			return FC.log.apply(FC, arguments);
		}
	};


	/* General Utilities
----------------------------------------------------------------------------------------------------------------------*/

	var hasOwnPropMethod = {}.hasOwnProperty;


// Merges an array of objects into a single object.
// The second argument allows for an array of property names who's object values will be merged together.
	function mergeProps(propObjs, complexProps) {
		var dest = {};
		var i, name;
		var complexObjs;
		var j, val;
		var props;

		if (complexProps) {
			for (i = 0; i < complexProps.length; i++) {
				name = complexProps[i];
				complexObjs = [];

				// collect the trailing object values, stopping when a non-object is discovered
				for (j = propObjs.length - 1; j >= 0; j--) {
					val = propObjs[j][name];

					if (typeof val === 'object') {
						complexObjs.unshift(val);
					}
					else if (val !== undefined) {
						dest[name] = val; // if there were no objects, this value will be used
						break;
					}
				}

				// if the trailing values were objects, use the merged value
				if (complexObjs.length) {
					dest[name] = mergeProps(complexObjs);
				}
			}
		}

		// copy values into the destination, going from last to first
		for (i = propObjs.length - 1; i >= 0; i--) {
			props = propObjs[i];

			for (name in props) {
				if (!(name in dest)) { // if already assigned by previous props or complex props, don't reassign
					dest[name] = props[name];
				}
			}
		}

		return dest;
	}


// Create an object that has the given prototype. Just like Object.create
	function createObject(proto) {
		var f = function() {};
		f.prototype = proto;
		return new f();
	}
	FC.createObject = createObject;


	function copyOwnProps(src, dest) {
		for (var name in src) {
			if (hasOwnProp(src, name)) {
				dest[name] = src[name];
			}
		}
	}


	function hasOwnProp(obj, name) {
		return hasOwnPropMethod.call(obj, name);
	}


// Is the given value a non-object non-function value?
	function isAtomic(val) {
		return /undefined|null|boolean|number|string/.test($.type(val));
	}


	function applyAll(functions, thisObj, args) {
		if ($.isFunction(functions)) {
			functions = [ functions ];
		}
		if (functions) {
			var i;
			var ret;
			for (i=0; i<functions.length; i++) {
				ret = functions[i].apply(thisObj, args) || ret;
			}
			return ret;
		}
	}


	function firstDefined() {
		for (var i=0; i<arguments.length; i++) {
			if (arguments[i] !== undefined) {
				return arguments[i];
			}
		}
	}


	function htmlEscape(s) {
		return (s + '').replace(/&/g, '&amp;')
			.replace(/</g, '&lt;')
			.replace(/>/g, '&gt;')
			.replace(/'/g, '&#039;')
			.replace(/"/g, '&quot;')
			.replace(/\n/g, '<br />');
	}


	function stripHtmlEntities(text) {
		return text.replace(/&.*?;/g, '');
	}


// Given a hash of CSS properties, returns a string of CSS.
// Uses property names as-is (no camel-case conversion). Will not make statements for null/undefined values.
	function cssToStr(cssProps) {
		var statements = [];

		$.each(cssProps, function(name, val) {
			if (val != null) {
				statements.push(name + ':' + val);
			}
		});

		return statements.join(';');
	}


// Given an object hash of HTML attribute names to values,
// generates a string that can be injected between < > in HTML
	function attrsToStr(attrs) {
		var parts = [];

		$.each(attrs, function(name, val) {
			if (val != null) {
				parts.push(name + '="' + htmlEscape(val) + '"');
			}
		});

		return parts.join(' ');
	}


	function capitaliseFirstLetter(str) {
		return str.charAt(0).toUpperCase() + str.slice(1);
	}


	function compareNumbers(a, b) { // for .sort()
		return a - b;
	}


	function isInt(n) {
		return n % 1 === 0;
	}


// Returns a method bound to the given object context.
// Just like one of the jQuery.proxy signatures, but without the undesired behavior of treating the same method with
// different contexts as identical when binding/unbinding events.
	function proxy(obj, methodName) {
		var method = obj[methodName];

		return function() {
			return method.apply(obj, arguments);
		};
	}


// Returns a function, that, as long as it continues to be invoked, will not
// be triggered. The function will be called after it stops being called for
// N milliseconds. If `immediate` is passed, trigger the function on the
// leading edge, instead of the trailing.
// https://github.com/jashkenas/underscore/blob/1.6.0/underscore.js#L714
	function debounce(func, wait, immediate) {
		var timeout, args, context, timestamp, result;

		var later = function() {
			var last = +new Date() - timestamp;
			if (last < wait) {
				timeout = setTimeout(later, wait - last);
			}
			else {
				timeout = null;
				if (!immediate) {
					result = func.apply(context, args);
					context = args = null;
				}
			}
		};

		return function() {
			context = this;
			args = arguments;
			timestamp = +new Date();
			var callNow = immediate && !timeout;
			if (!timeout) {
				timeout = setTimeout(later, wait);
			}
			if (callNow) {
				result = func.apply(context, args);
				context = args = null;
			}
			return result;
		};
	}

	;;

	/*
GENERAL NOTE on moments throughout the *entire rest* of the codebase:
All moments are assumed to be ambiguously-zoned unless otherwise noted,
with the NOTABLE EXCEOPTION of start/end dates that live on *Event Objects*.
Ambiguously-TIMED moments are assumed to be ambiguously-zoned by nature.
*/

	var ambigDateOfMonthRegex = /^\s*\d{4}-\d\d$/;
	var ambigTimeOrZoneRegex =
		/^\s*\d{4}-(?:(\d\d-\d\d)|(W\d\d$)|(W\d\d-\d)|(\d\d\d))((T| )(\d\d(:\d\d(:\d\d(\.\d+)?)?)?)?)?$/;
	var newMomentProto = moment.fn; // where we will attach our new methods
	var oldMomentProto = $.extend({}, newMomentProto); // copy of original moment methods

// tell momentjs to transfer these properties upon clone
	var momentProperties = moment.momentProperties;
	momentProperties.push('_fullCalendar');
	momentProperties.push('_ambigTime');
	momentProperties.push('_ambigZone');


// Creating
// -------------------------------------------------------------------------------------------------

// Creates a new moment, similar to the vanilla moment(...) constructor, but with
// extra features (ambiguous time, enhanced formatting). When given an existing moment,
// it will function as a clone (and retain the zone of the moment). Anything else will
// result in a moment in the local zone.
	FC.moment = function() {
		return makeMoment(arguments);
	};

// Sames as FC.moment, but forces the resulting moment to be in the UTC timezone.
	FC.moment.utc = function() {
		var mom = makeMoment(arguments, true);

		// Force it into UTC because makeMoment doesn't guarantee it
		// (if given a pre-existing moment for example)
		if (mom.hasTime()) { // don't give ambiguously-timed moments a UTC zone
			mom.utc();
		}

		return mom;
	};

// Same as FC.moment, but when given an ISO8601 string, the timezone offset is preserved.
// ISO8601 strings with no timezone offset will become ambiguously zoned.
	FC.moment.parseZone = function() {
		return makeMoment(arguments, true, true);
	};

// Builds an enhanced moment from args. When given an existing moment, it clones. When given a
// native Date, or called with no arguments (the current time), the resulting moment will be local.
// Anything else needs to be "parsed" (a string or an array), and will be affected by:
//    parseAsUTC - if there is no zone information, should we parse the input in UTC?
//    parseZone - if there is zone information, should we force the zone of the moment?
	function makeMoment(args, parseAsUTC, parseZone) {
		var input = args[0];
		var isSingleString = args.length == 1 && typeof input === 'string';
		var isAmbigTime;
		var isAmbigZone;
		var ambigMatch;
		var mom;

		if (moment.isMoment(input) || isNativeDate(input) || input === undefined) {
			mom = moment.apply(null, args);
		}
		else { // "parsing" is required
			isAmbigTime = false;
			isAmbigZone = false;

			if (isSingleString) {
				if (ambigDateOfMonthRegex.test(input)) {
					// accept strings like '2014-05', but convert to the first of the month
					input += '-01';
					args = [ input ]; // for when we pass it on to moment's constructor
					isAmbigTime = true;
					isAmbigZone = true;
				}
				else if ((ambigMatch = ambigTimeOrZoneRegex.exec(input))) {
					isAmbigTime = !ambigMatch[5]; // no time part?
					isAmbigZone = true;
				}
			}
			else if ($.isArray(input)) {
				// arrays have no timezone information, so assume ambiguous zone
				isAmbigZone = true;
			}
			// otherwise, probably a string with a format

			if (parseAsUTC || isAmbigTime) {
				mom = moment.utc.apply(moment, args);
			}
			else {
				mom = moment.apply(null, args);
			}

			if (isAmbigTime) {
				mom._ambigTime = true;
				mom._ambigZone = true; // ambiguous time always means ambiguous zone
			}
			else if (parseZone) { // let's record the inputted zone somehow
				if (isAmbigZone) {
					mom._ambigZone = true;
				}
				else if (isSingleString) {
					mom.utcOffset(input); // if not a valid zone, will assign UTC
				}
			}
		}

		mom._fullCalendar = true; // flag for extended functionality

		return mom;
	}


// Week Number
// -------------------------------------------------------------------------------------------------


// Returns the week number, considering the locale's custom week number calcuation
// `weeks` is an alias for `week`
	newMomentProto.week = newMomentProto.weeks = function(input) {
		var weekCalc = this._locale._fullCalendar_weekCalc;

		if (input == null && typeof weekCalc === 'function') { // custom function only works for getter
			return weekCalc(this);
		}
		else if (weekCalc === 'ISO') {
			return oldMomentProto.isoWeek.apply(this, arguments); // ISO getter/setter
		}

		return oldMomentProto.week.apply(this, arguments); // local getter/setter
	};


// Time-of-day
// -------------------------------------------------------------------------------------------------

// GETTER
// Returns a Duration with the hours/minutes/seconds/ms values of the moment.
// If the moment has an ambiguous time, a duration of 00:00 will be returned.
//
// SETTER
// You can supply a Duration, a Moment, or a Duration-like argument.
// When setting the time, and the moment has an ambiguous time, it then becomes unambiguous.
	newMomentProto.time = function(time) {

		// Fallback to the original method (if there is one) if this moment wasn't created via FullCalendar.
		// `time` is a generic enough method name where this precaution is necessary to avoid collisions w/ other plugins.
		if (!this._fullCalendar) {
			return oldMomentProto.time.apply(this, arguments);
		}

		if (time == null) { // getter
			return moment.duration({
				hours: this.hours(),
				minutes: this.minutes(),
				seconds: this.seconds(),
				milliseconds: this.milliseconds()
			});
		}
		else { // setter

			this._ambigTime = false; // mark that the moment now has a time

			if (!moment.isDuration(time) && !moment.isMoment(time)) {
				time = moment.duration(time);
			}

			// The day value should cause overflow (so 24 hours becomes 00:00:00 of next day).
			// Only for Duration times, not Moment times.
			var dayHours = 0;
			if (moment.isDuration(time)) {
				dayHours = Math.floor(time.asDays()) * 24;
			}

			// We need to set the individual fields.
			// Can't use startOf('day') then add duration. In case of DST at start of day.
			return this.hours(dayHours + time.hours())
				.minutes(time.minutes())
				.seconds(time.seconds())
				.milliseconds(time.milliseconds());
		}
	};

// Converts the moment to UTC, stripping out its time-of-day and timezone offset,
// but preserving its YMD. A moment with a stripped time will display no time
// nor timezone offset when .format() is called.
	newMomentProto.stripTime = function() {

		if (!this._ambigTime) {

			this.utc(true); // keepLocalTime=true (for keeping *date* value)

			// set time to zero
			this.set({
				hours: 0,
				minutes: 0,
				seconds: 0,
				ms: 0
			});

			// Mark the time as ambiguous. This needs to happen after the .utc() call, which might call .utcOffset(),
			// which clears all ambig flags.
			this._ambigTime = true;
			this._ambigZone = true; // if ambiguous time, also ambiguous timezone offset
		}

		return this; // for chaining
	};

// Returns if the moment has a non-ambiguous time (boolean)
	newMomentProto.hasTime = function() {
		return !this._ambigTime;
	};


// Timezone
// -------------------------------------------------------------------------------------------------

// Converts the moment to UTC, stripping out its timezone offset, but preserving its
// YMD and time-of-day. A moment with a stripped timezone offset will display no
// timezone offset when .format() is called.
	newMomentProto.stripZone = function() {
		var wasAmbigTime;

		if (!this._ambigZone) {

			wasAmbigTime = this._ambigTime;

			this.utc(true); // keepLocalTime=true (for keeping date and time values)

			// the above call to .utc()/.utcOffset() unfortunately might clear the ambig flags, so restore
			this._ambigTime = wasAmbigTime || false;

			// Mark the zone as ambiguous. This needs to happen after the .utc() call, which might call .utcOffset(),
			// which clears the ambig flags.
			this._ambigZone = true;
		}

		return this; // for chaining
	};

// Returns of the moment has a non-ambiguous timezone offset (boolean)
	newMomentProto.hasZone = function() {
		return !this._ambigZone;
	};


// implicitly marks a zone
	newMomentProto.local = function(keepLocalTime) {

		// for when converting from ambiguously-zoned to local,
		// keep the time values when converting from UTC -> local
		oldMomentProto.local.call(this, this._ambigZone || keepLocalTime);

		// ensure non-ambiguous
		// this probably already happened via local() -> utcOffset(), but don't rely on Moment's internals
		this._ambigTime = false;
		this._ambigZone = false;

		return this; // for chaining
	};


// implicitly marks a zone
	newMomentProto.utc = function(keepLocalTime) {

		oldMomentProto.utc.call(this, keepLocalTime);

		// ensure non-ambiguous
		// this probably already happened via utc() -> utcOffset(), but don't rely on Moment's internals
		this._ambigTime = false;
		this._ambigZone = false;

		return this;
	};


// implicitly marks a zone (will probably get called upon .utc() and .local())
	newMomentProto.utcOffset = function(tzo) {

		if (tzo != null) { // setter
			// these assignments needs to happen before the original zone method is called.
			// I forget why, something to do with a browser crash.
			this._ambigTime = false;
			this._ambigZone = false;
		}

		return oldMomentProto.utcOffset.apply(this, arguments);
	};


// Formatting
// -------------------------------------------------------------------------------------------------

	newMomentProto.format = function() {
		if (this._fullCalendar && arguments[0]) { // an enhanced moment? and a format string provided?
			return formatDate(this, arguments[0]); // our extended formatting
		}
		if (this._ambigTime) {
			return oldMomentFormat(this, 'YYYY-MM-DD');
		}
		if (this._ambigZone) {
			return oldMomentFormat(this, 'YYYY-MM-DD[T]HH:mm:ss');
		}
		return oldMomentProto.format.apply(this, arguments);
	};

	newMomentProto.toISOString = function() {
		if (this._ambigTime) {
			return oldMomentFormat(this, 'YYYY-MM-DD');
		}
		if (this._ambigZone) {
			return oldMomentFormat(this, 'YYYY-MM-DD[T]HH:mm:ss');
		}
		return oldMomentProto.toISOString.apply(this, arguments);
	};

	;;

// Single Date Formatting
// -------------------------------------------------------------------------------------------------


// call this if you want Moment's original format method to be used
	function oldMomentFormat(mom, formatStr) {
		return oldMomentProto.format.call(mom, formatStr); // oldMomentProto defined in moment-ext.js
	}


// Formats `date` with a Moment formatting string, but allow our non-zero areas and
// additional token.
	function formatDate(date, formatStr) {
		return formatDateWithChunks(date, getFormatStringChunks(formatStr));
	}


	function formatDateWithChunks(date, chunks) {
		var s = '';
		var i;

		for (i=0; i<chunks.length; i++) {
			s += formatDateWithChunk(date, chunks[i]);
		}

		return s;
	}


// addition formatting tokens we want recognized
	var tokenOverrides = {
		t: function(date) { // "a" or "p"
			return oldMomentFormat(date, 'a').charAt(0);
		},
		T: function(date) { // "A" or "P"
			return oldMomentFormat(date, 'A').charAt(0);
		}
	};


	function formatDateWithChunk(date, chunk) {
		var token;
		var maybeStr;

		if (typeof chunk === 'string') { // a literal string
			return chunk;
		}
		else if ((token = chunk.token)) { // a token, like "YYYY"
			if (tokenOverrides[token]) {
				return tokenOverrides[token](date); // use our custom token
			}
			return oldMomentFormat(date, token);
		}
		else if (chunk.maybe) { // a grouping of other chunks that must be non-zero
			maybeStr = formatDateWithChunks(date, chunk.maybe);
			if (maybeStr.match(/[1-9]/)) {
				return maybeStr;
			}
		}

		return '';
	}


// Date Range Formatting
// -------------------------------------------------------------------------------------------------
// TODO: make it work with timezone offset

// Using a formatting string meant for a single date, generate a range string, like
// "Sep 2 - 9 2013", that intelligently inserts a separator where the dates differ.
// If the dates are the same as far as the format string is concerned, just return a single
// rendering of one date, without any separator.
	function formatRange(date1, date2, formatStr, separator, isRTL) {
		var localeData;

		date1 = FC.moment.parseZone(date1);
		date2 = FC.moment.parseZone(date2);

		localeData = date1.localeData();

		// Expand localized format strings, like "LL" -> "MMMM D YYYY"
		formatStr = localeData.longDateFormat(formatStr) || formatStr;
		// BTW, this is not important for `formatDate` because it is impossible to put custom tokens
		// or non-zero areas in Moment's localized format strings.

		separator = separator || ' - ';

		return formatRangeWithChunks(
			date1,
			date2,
			getFormatStringChunks(formatStr),
			separator,
			isRTL
		);
	}
	FC.formatRange = formatRange; // expose


	function formatRangeWithChunks(date1, date2, chunks, separator, isRTL) {
		var unzonedDate1 = date1.clone().stripZone(); // for formatSimilarChunk
		var unzonedDate2 = date2.clone().stripZone(); // "
		var chunkStr; // the rendering of the chunk
		var leftI;
		var leftStr = '';
		var rightI;
		var rightStr = '';
		var middleI;
		var middleStr1 = '';
		var middleStr2 = '';
		var middleStr = '';

		// Start at the leftmost side of the formatting string and continue until you hit a token
		// that is not the same between dates.
		for (leftI=0; leftI<chunks.length; leftI++) {
			chunkStr = formatSimilarChunk(date1, date2, unzonedDate1, unzonedDate2, chunks[leftI]);
			if (chunkStr === false) {
				break;
			}
			leftStr += chunkStr;
		}

		// Similarly, start at the rightmost side of the formatting string and move left
		for (rightI=chunks.length-1; rightI>leftI; rightI--) {
			chunkStr = formatSimilarChunk(date1, date2, unzonedDate1, unzonedDate2,  chunks[rightI]);
			if (chunkStr === false) {
				break;
			}
			rightStr = chunkStr + rightStr;
		}

		// The area in the middle is different for both of the dates.
		// Collect them distinctly so we can jam them together later.
		for (middleI=leftI; middleI<=rightI; middleI++) {
			middleStr1 += formatDateWithChunk(date1, chunks[middleI]);
			middleStr2 += formatDateWithChunk(date2, chunks[middleI]);
		}

		if (middleStr1 || middleStr2) {
			if (isRTL) {
				middleStr = middleStr2 + separator + middleStr1;
			}
			else {
				middleStr = middleStr1 + separator + middleStr2;
			}
		}

		return leftStr + middleStr + rightStr;
	}


	var similarUnitMap = {
		Y: 'year',
		M: 'month',
		D: 'day', // day of month
		d: 'day', // day of week
		// prevents a separator between anything time-related...
		A: 'second', // AM/PM
		a: 'second', // am/pm
		T: 'second', // A/P
		t: 'second', // a/p
		H: 'second', // hour (24)
		h: 'second', // hour (12)
		m: 'second', // minute
		s: 'second' // second
	};
// TODO: week maybe?


// Given a formatting chunk, and given that both dates are similar in the regard the
// formatting chunk is concerned, format date1 against `chunk`. Otherwise, return `false`.
	function formatSimilarChunk(date1, date2, unzonedDate1, unzonedDate2, chunk) {
		var token;
		var unit;

		if (typeof chunk === 'string') { // a literal string
			return chunk;
		}
		else if ((token = chunk.token)) {
			unit = similarUnitMap[token.charAt(0)];

			// are the dates the same for this unit of measurement?
			// use the unzoned dates for this calculation because unreliable when near DST (bug #2396)
			if (unit && unzonedDate1.isSame(unzonedDate2, unit)) {
				return oldMomentFormat(date1, token); // would be the same if we used `date2`
				// BTW, don't support custom tokens
			}
		}

		return false; // the chunk is NOT the same for the two dates
		// BTW, don't support splitting on non-zero areas
	}


// Chunking Utils
// -------------------------------------------------------------------------------------------------


	var formatStringChunkCache = {};


	function getFormatStringChunks(formatStr) {
		if (formatStr in formatStringChunkCache) {
			return formatStringChunkCache[formatStr];
		}
		return (formatStringChunkCache[formatStr] = chunkFormatString(formatStr));
	}


// Break the formatting string into an array of chunks
	function chunkFormatString(formatStr) {
		var chunks = [];
		var chunker = /\[([^\]]*)\]|\(([^\)]*)\)|(LTS|LT|(\w)\4*o?)|([^\w\[\(]+)/g; // TODO: more descrimination
		var match;

		while ((match = chunker.exec(formatStr))) {
			if (match[1]) { // a literal string inside [ ... ]
				chunks.push(match[1]);
			}
			else if (match[2]) { // non-zero formatting inside ( ... )
				chunks.push({ maybe: chunkFormatString(match[2]) });
			}
			else if (match[3]) { // a formatting token
				chunks.push({ token: match[3] });
			}
			else if (match[5]) { // an unenclosed literal string
				chunks.push(match[5]);
			}
		}

		return chunks;
	}


// Misc Utils
// -------------------------------------------------------------------------------------------------


// granularity only goes up until day
// TODO: unify with similarUnitMap
	var tokenGranularities = {
		Y: { value: 1, unit: 'year' },
		M: { value: 2, unit: 'month' },
		W: { value: 3, unit: 'week' },
		w: { value: 3, unit: 'week' },
		D: { value: 4, unit: 'day' }, // day of month
		d: { value: 4, unit: 'day' } // day of week
	};

// returns a unit string, either 'year', 'month', 'day', or null
// for the most granular formatting token in the string.
	FC.queryMostGranularFormatUnit = function(formatStr) {
		var chunks = getFormatStringChunks(formatStr);
		var i, chunk;
		var candidate;
		var best;

		for (i = 0; i < chunks.length; i++) {
			chunk = chunks[i];
			if (chunk.token) {
				candidate = tokenGranularities[chunk.token.charAt(0)];
				if (candidate) {
					if (!best || candidate.value > best.value) {
						best = candidate;
					}
				}
			}
		}

		if (best) {
			return best.unit;
		}

		return null;
	};

	;;

	FC.Class = Class; // export

// Class that all other classes will inherit from
	function Class() { }


// Called on a class to create a subclass.
// Last argument contains instance methods. Any argument before the last are considered mixins.
	Class.extend = function() {
		var len = arguments.length;
		var i;
		var members;

		for (i = 0; i < len; i++) {
			members = arguments[i];
			if (i < len - 1) { // not the last argument?
				mixIntoClass(this, members);
			}
		}

		return extendClass(this, members || {}); // members will be undefined if no arguments
	};


// Adds new member variables/methods to the class's prototype.
// Can be called with another class, or a plain object hash containing new members.
	Class.mixin = function(members) {
		mixIntoClass(this, members);
	};


	function extendClass(superClass, members) {
		var subClass;

		// ensure a constructor for the subclass, forwarding all arguments to the super-constructor if it doesn't exist
		if (hasOwnProp(members, 'constructor')) {
			subClass = members.constructor;
		}
		if (typeof subClass !== 'function') {
			subClass = members.constructor = function() {
				superClass.apply(this, arguments);
			};
		}

		// build the base prototype for the subclass, which is an new object chained to the superclass's prototype
		subClass.prototype = createObject(superClass.prototype);

		// copy each member variable/method onto the the subclass's prototype
		copyOwnProps(members, subClass.prototype);

		// copy over all class variables/methods to the subclass, such as `extend` and `mixin`
		copyOwnProps(superClass, subClass);

		return subClass;
	}


	function mixIntoClass(theClass, members) {
		copyOwnProps(members, theClass.prototype);
	}
	;;

	/*
Wrap jQuery's Deferred Promise object to be slightly more Promise/A+ compliant.
With the added non-standard feature of synchronously executing handlers on resolved promises,
which doesn't always happen otherwise (esp with nested .then handlers!?),
so, this makes things a lot easier, esp because jQuery 3 changed the synchronicity for Deferred objects.

TODO: write tests and more comments
*/

	function Promise(executor) {
		var deferred = $.Deferred();
		var promise = deferred.promise();

		if (typeof executor === 'function') {
			executor(
				function(value) { // resolve
					if (Promise.immediate) {
						promise._value = value;
					}
					deferred.resolve(value);
				},
				function() { // reject
					deferred.reject();
				}
			);
		}

		if (Promise.immediate) {
			var origThen = promise.then;

			promise.then = function(onFulfilled, onRejected) {
				var state = promise.state();

				if (state === 'resolved') {
					if (typeof onFulfilled === 'function') {
						return Promise.resolve(onFulfilled(promise._value));
					}
				}
				else if (state === 'rejected') {
					if (typeof onRejected === 'function') {
						onRejected();
						return promise; // already rejected
					}
				}

				return origThen.call(promise, onFulfilled, onRejected);
			};
		}

		return promise; // instanceof Promise will break :( TODO: make Promise a real class
	}

	FC.Promise = Promise;

	Promise.immediate = true;


	Promise.resolve = function(value) {
		if (value && typeof value.resolve === 'function') {
			return value.promise();
		}
		if (value && typeof value.then === 'function') {
			return value;
		}
		else {
			var deferred = $.Deferred().resolve(value);
			var promise = deferred.promise();

			if (Promise.immediate) {
				var origThen = promise.then;

				promise._value = value;

				promise.then = function(onFulfilled, onRejected) {
					if (typeof onFulfilled === 'function') {
						return Promise.resolve(onFulfilled(value));
					}
					return origThen.call(promise, onFulfilled, onRejected);
				};
			}

			return promise;
		}
	};


	Promise.reject = function() {
		return $.Deferred().reject().promise();
	};


	Promise.all = function(inputs) {
		var hasAllValues = false;
		var values;
		var i, input;

		if (Promise.immediate) {
			hasAllValues = true;
			values = [];

			for (i = 0; i < inputs.length; i++) {
				input = inputs[i];

				if (input && typeof input.state === 'function' && input.state() === 'resolved' && ('_value' in input)) {
					values.push(input._value);
				}
				else if (input && typeof input.then === 'function') {
					hasAllValues = false;
					break;
				}
				else {
					values.push(input);
				}
			}
		}

		if (hasAllValues) {
			return Promise.resolve(values);
		}
		else {
			return $.when.apply($.when, inputs).then(function() {
				return $.when($.makeArray(arguments));
			});
		}
	};

	;;

// TODO: write tests and clean up code

	function TaskQueue(debounceWait) {
		var q = []; // array of runFuncs

		function addTask(taskFunc) {
			return new Promise(function(resolve) {

				// should run this function when it's taskFunc's turn to run.
				// responsible for popping itself off the queue.
				var runFunc = function() {
					Promise.resolve(taskFunc()) // result might be async, coerce to promise
						.then(resolve) // resolve TaskQueue::push's promise, for the caller. will receive result of taskFunc.
						.then(function() {
							q.shift(); // pop itself off

							// run the next task, if any
							if (q.length) {
								q[0]();
							}
						});
				};

				// always put the task at the end of the queue, BEFORE running the task
				q.push(runFunc);

				// if it's the only task in the queue, run immediately
				if (q.length === 1) {
					runFunc();
				}
			});
		}

		this.add = // potentially debounce, for the public method
			typeof debounceWait === 'number' ?
				debounce(addTask, debounceWait) :
				addTask; // if not a number (null/undefined/false), no debounce at all

		this.addQuickly = addTask; // guaranteed no debounce
	}

	FC.TaskQueue = TaskQueue;

	/*
q = new TaskQueue();

function work(i) {
	return q.push(function() {
		trigger();
		console.log('work' + i);
	});
}

var cnt = 0;

function trigger() {
	if (cnt < 5) {
		cnt++;
		work(cnt);
	}
}

work(9);
*/

	;;

	var EmitterMixin = FC.EmitterMixin = {

		// jQuery-ification via $(this) allows a non-DOM object to have
		// the same event handling capabilities (including namespaces).


		on: function(types, handler) {
			$(this).on(types, this._prepareIntercept(handler));
			return this; // for chaining
		},


		one: function(types, handler) {
			$(this).one(types, this._prepareIntercept(handler));
			return this; // for chaining
		},


		_prepareIntercept: function(handler) {
			// handlers are always called with an "event" object as their first param.
			// sneak the `this` context and arguments into the extra parameter object
			// and forward them on to the original handler.
			var intercept = function(ev, extra) {
				return handler.apply(
					extra.context || this,
					extra.args || []
				);
			};

			// mimick jQuery's internal "proxy" system (risky, I know)
			// causing all functions with the same .guid to appear to be the same.
			// https://github.com/jquery/jquery/blob/2.2.4/src/core.js#L448
			// this is needed for calling .off with the original non-intercept handler.
			if (!handler.guid) {
				handler.guid = $.guid++;
			}
			intercept.guid = handler.guid;

			return intercept;
		},


		off: function(types, handler) {
			$(this).off(types, handler);

			return this; // for chaining
		},


		trigger: function(types) {
			var args = Array.prototype.slice.call(arguments, 1); // arguments after the first

			// pass in "extra" info to the intercept
			$(this).triggerHandler(types, { args: args });

			return this; // for chaining
		},


		triggerWith: function(types, context, args) {

			// `triggerHandler` is less reliant on the DOM compared to `trigger`.
			// pass in "extra" info to the intercept.
			$(this).triggerHandler(types, { context: context, args: args });

			return this; // for chaining
		}

	};

	;;

	/*
Utility methods for easily listening to events on another object,
and more importantly, easily unlistening from them.
*/
	var ListenerMixin = FC.ListenerMixin = (function() {
		var guid = 0;
		var ListenerMixin = {

			listenerId: null,

			/*
		Given an `other` object that has on/off methods, bind the given `callback` to an event by the given name.
		The `callback` will be called with the `this` context of the object that .listenTo is being called on.
		Can be called:
			.listenTo(other, eventName, callback)
		OR
			.listenTo(other, {
				eventName1: callback1,
				eventName2: callback2
			})
		*/
			listenTo: function(other, arg, callback) {
				if (typeof arg === 'object') { // given dictionary of callbacks
					for (var eventName in arg) {
						if (arg.hasOwnProperty(eventName)) {
							this.listenTo(other, eventName, arg[eventName]);
						}
					}
				}
				else if (typeof arg === 'string') {
					other.on(
						arg + '.' + this.getListenerNamespace(), // use event namespacing to identify this object
						$.proxy(callback, this) // always use `this` context
						// the usually-undesired jQuery guid behavior doesn't matter,
						// because we always unbind via namespace
					);
				}
			},

			/*
		Causes the current object to stop listening to events on the `other` object.
		`eventName` is optional. If omitted, will stop listening to ALL events on `other`.
		*/
			stopListeningTo: function(other, eventName) {
				other.off((eventName || '') + '.' + this.getListenerNamespace());
			},

			/*
		Returns a string, unique to this object, to be used for event namespacing
		*/
			getListenerNamespace: function() {
				if (this.listenerId == null) {
					this.listenerId = guid++;
				}
				return '_listener' + this.listenerId;
			}

		};
		return ListenerMixin;
	})();
	;;

// simple class for toggle a `isIgnoringMouse` flag on delay
// initMouseIgnoring must first be called, with a millisecond delay setting.
	var MouseIgnorerMixin = {

		isIgnoringMouse: false, // bool
		delayUnignoreMouse: null, // method


		initMouseIgnoring: function(delay) {
			this.delayUnignoreMouse = debounce(proxy(this, 'unignoreMouse'), delay || 1000);
		},


		// temporarily ignore mouse actions on segments
		tempIgnoreMouse: function() {
			this.isIgnoringMouse = true;
			this.delayUnignoreMouse();
		},


		// delayUnignoreMouse eventually calls this
		unignoreMouse: function() {
			this.isIgnoringMouse = false;
		}

	};

	;;

	/* A rectangular panel that is absolutely positioned over other content
------------------------------------------------------------------------------------------------------------------------
Options:
	- className (string)
	- content (HTML string or jQuery element set)
	- parentEl
	- top
	- left
	- right (the x coord of where the right edge should be. not a "CSS" right)
	- autoHide (boolean)
	- show (callback)
	- hide (callback)
*/

	var Popover = Class.extend(ListenerMixin, {

		isHidden: true,
		options: null,
		el: null, // the container element for the popover. generated by this object
		margin: 10, // the space required between the popover and the edges of the scroll container


		constructor: function(options) {
			this.options = options || {};
		},


		// Shows the popover on the specified position. Renders it if not already
		show: function() {
			if (this.isHidden) {
				if (!this.el) {
					this.render();
				}
				this.el.show();
				this.position();
				this.isHidden = false;
				this.trigger('show');
			}
		},


		// Hides the popover, through CSS, but does not remove it from the DOM
		hide: function() {
			if (!this.isHidden) {
				this.el.hide();
				this.isHidden = true;
				this.trigger('hide');
			}
		},


		// Creates `this.el` and renders content inside of it
		render: function() {
			var _this = this;
			var options = this.options;

			this.el = $('<div class="fc-popover"/>')
				.addClass(options.className || '')
				.css({
					// position initially to the top left to avoid creating scrollbars
					top: 0,
					left: 0
				})
				.append(options.content)
				.appendTo(options.parentEl);

			// when a click happens on anything inside with a 'fc-close' className, hide the popover
			this.el.on('click', '.fc-close', function() {
				_this.hide();
			});

			if (options.autoHide) {
				this.listenTo($(document), 'mousedown', this.documentMousedown);
			}
		},


		// Triggered when the user clicks *anywhere* in the document, for the autoHide feature
		documentMousedown: function(ev) {
			// only hide the popover if the click happened outside the popover
			if (this.el && !$(ev.target).closest(this.el).length) {
				this.hide();
			}
		},


		// Hides and unregisters any handlers
		removeElement: function() {
			this.hide();

			if (this.el) {
				this.el.remove();
				this.el = null;
			}

			this.stopListeningTo($(document), 'mousedown');
		},


		// Positions the popover optimally, using the top/left/right options
		position: function() {
			var options = this.options;
			var origin = this.el.offsetParent().offset();
			var width = this.el.outerWidth();
			var height = this.el.outerHeight();
			var windowEl = $(window);
			var viewportEl = getScrollParent(this.el);
			var viewportTop;
			var viewportLeft;
			var viewportOffset;
			var top; // the "position" (not "offset") values for the popover
			var left; //

			// compute top and left
			top = options.top || 0;
			if (options.left !== undefined) {
				left = options.left;
			}
			else if (options.right !== undefined) {
				left = options.right - width; // derive the left value from the right value
			}
			else {
				left = 0;
			}

			if (viewportEl.is(window) || viewportEl.is(document)) { // normalize getScrollParent's result
				viewportEl = windowEl;
				viewportTop = 0; // the window is always at the top left
				viewportLeft = 0; // (and .offset() won't work if called here)
			}
			else {
				viewportOffset = viewportEl.offset();
				viewportTop = viewportOffset.top;
				viewportLeft = viewportOffset.left;
			}

			// if the window is scrolled, it causes the visible area to be further down
			viewportTop += windowEl.scrollTop();
			viewportLeft += windowEl.scrollLeft();

			// constrain to the view port. if constrained by two edges, give precedence to top/left
			if (options.viewportConstrain !== false) {
				top = Math.min(top, viewportTop + viewportEl.outerHeight() - height - this.margin);
				top = Math.max(top, viewportTop + this.margin);
				left = Math.min(left, viewportLeft + viewportEl.outerWidth() - width - this.margin);
				left = Math.max(left, viewportLeft + this.margin);
			}

			this.el.css({
				top: top - origin.top,
				left: left - origin.left
			});
		},


		// Triggers a callback. Calls a function in the option hash of the same name.
		// Arguments beyond the first `name` are forwarded on.
		// TODO: better code reuse for this. Repeat code
		trigger: function(name) {
			if (this.options[name]) {
				this.options[name].apply(this, Array.prototype.slice.call(arguments, 1));
			}
		}

	});

	;;

	/*
A cache for the left/right/top/bottom/width/height values for one or more elements.
Works with both offset (from topleft document) and position (from offsetParent).

options:
- els
- isHorizontal
- isVertical
*/
	var CoordCache = FC.CoordCache = Class.extend({

		els: null, // jQuery set (assumed to be siblings)
		forcedOffsetParentEl: null, // options can override the natural offsetParent
		origin: null, // {left,top} position of offsetParent of els
		boundingRect: null, // constrain cordinates to this rectangle. {left,right,top,bottom} or null
		isHorizontal: false, // whether to query for left/right/width
		isVertical: false, // whether to query for top/bottom/height

		// arrays of coordinates (offsets from topleft of document)
		lefts: null,
		rights: null,
		tops: null,
		bottoms: null,


		constructor: function(options) {
			this.els = $(options.els);
			this.isHorizontal = options.isHorizontal;
			this.isVertical = options.isVertical;
			this.forcedOffsetParentEl = options.offsetParent ? $(options.offsetParent) : null;
		},


		// Queries the els for coordinates and stores them.
		// Call this method before using and of the get* methods below.
		build: function() {
			var offsetParentEl = this.forcedOffsetParentEl;
			if (!offsetParentEl && this.els.length > 0) {
				offsetParentEl = this.els.eq(0).offsetParent();
			}

			this.origin = offsetParentEl ?
				offsetParentEl.offset() :
				null;

			this.boundingRect = this.queryBoundingRect();

			if (this.isHorizontal) {
				this.buildElHorizontals();
			}
			if (this.isVertical) {
				this.buildElVerticals();
			}
		},


		// Destroys all internal data about coordinates, freeing memory
		clear: function() {
			this.origin = null;
			this.boundingRect = null;
			this.lefts = null;
			this.rights = null;
			this.tops = null;
			this.bottoms = null;
		},


		// When called, if coord caches aren't built, builds them
		ensureBuilt: function() {
			if (!this.origin) {
				this.build();
			}
		},


		// Populates the left/right internal coordinate arrays
		buildElHorizontals: function() {
			var lefts = [];
			var rights = [];

			this.els.each(function(i, node) {
				var el = $(node);
				var left = el.offset().left;
				var width = el.outerWidth();

				lefts.push(left);
				rights.push(left + width);
			});

			this.lefts = lefts;
			this.rights = rights;
		},


		// Populates the top/bottom internal coordinate arrays
		buildElVerticals: function() {
			var tops = [];
			var bottoms = [];

			this.els.each(function(i, node) {
				var el = $(node);
				var top = el.offset().top;
				var height = el.outerHeight();

				tops.push(top);
				bottoms.push(top + height);
			});

			this.tops = tops;
			this.bottoms = bottoms;
		},


		// Given a left offset (from document left), returns the index of the el that it horizontally intersects.
		// If no intersection is made, returns undefined.
		getHorizontalIndex: function(leftOffset) {
			this.ensureBuilt();

			var lefts = this.lefts;
			var rights = this.rights;
			var len = lefts.length;
			var i;

			for (i = 0; i < len; i++) {
				if (leftOffset >= lefts[i] && leftOffset < rights[i]) {
					return i;
				}
			}
		},


		// Given a top offset (from document top), returns the index of the el that it vertically intersects.
		// If no intersection is made, returns undefined.
		getVerticalIndex: function(topOffset) {
			this.ensureBuilt();

			var tops = this.tops;
			var bottoms = this.bottoms;
			var len = tops.length;
			var i;

			for (i = 0; i < len; i++) {
				if (topOffset >= tops[i] && topOffset < bottoms[i]) {
					return i;
				}
			}
		},


		// Gets the left offset (from document left) of the element at the given index
		getLeftOffset: function(leftIndex) {
			this.ensureBuilt();
			return this.lefts[leftIndex];
		},


		// Gets the left position (from offsetParent left) of the element at the given index
		getLeftPosition: function(leftIndex) {
			this.ensureBuilt();
			return this.lefts[leftIndex] - this.origin.left;
		},


		// Gets the right offset (from document left) of the element at the given index.
		// This value is NOT relative to the document's right edge, like the CSS concept of "right" would be.
		getRightOffset: function(leftIndex) {
			this.ensureBuilt();
			return this.rights[leftIndex];
		},


		// Gets the right position (from offsetParent left) of the element at the given index.
		// This value is NOT relative to the offsetParent's right edge, like the CSS concept of "right" would be.
		getRightPosition: function(leftIndex) {
			this.ensureBuilt();
			return this.rights[leftIndex] - this.origin.left;
		},


		// Gets the width of the element at the given index
		getWidth: function(leftIndex) {
			this.ensureBuilt();
			return this.rights[leftIndex] - this.lefts[leftIndex];
		},


		// Gets the top offset (from document top) of the element at the given index
		getTopOffset: function(topIndex) {
			this.ensureBuilt();
			return this.tops[topIndex];
		},


		// Gets the top position (from offsetParent top) of the element at the given position
		getTopPosition: function(topIndex) {
			this.ensureBuilt();
			return this.tops[topIndex] - this.origin.top;
		},

		// Gets the bottom offset (from the document top) of the element at the given index.
		// This value is NOT relative to the offsetParent's bottom edge, like the CSS concept of "bottom" would be.
		getBottomOffset: function(topIndex) {
			this.ensureBuilt();
			return this.bottoms[topIndex];
		},


		// Gets the bottom position (from the offsetParent top) of the element at the given index.
		// This value is NOT relative to the offsetParent's bottom edge, like the CSS concept of "bottom" would be.
		getBottomPosition: function(topIndex) {
			this.ensureBuilt();
			return this.bottoms[topIndex] - this.origin.top;
		},


		// Gets the height of the element at the given index
		getHeight: function(topIndex) {
			this.ensureBuilt();
			return this.bottoms[topIndex] - this.tops[topIndex];
		},


		// Bounding Rect
		// TODO: decouple this from CoordCache

		// Compute and return what the elements' bounding rectangle is, from the user's perspective.
		// Right now, only returns a rectangle if constrained by an overflow:scroll element.
		// Returns null if there are no elements
		queryBoundingRect: function() {
			var scrollParentEl;

			if (this.els.length > 0) {
				scrollParentEl = getScrollParent(this.els.eq(0));

				if (!scrollParentEl.is(document)) {
					return getClientRect(scrollParentEl);
				}
			}

			return null;
		},

		isPointInBounds: function(leftOffset, topOffset) {
			return this.isLeftInBounds(leftOffset) && this.isTopInBounds(topOffset);
		},

		isLeftInBounds: function(leftOffset) {
			return !this.boundingRect || (leftOffset >= this.boundingRect.left && leftOffset < this.boundingRect.right);
		},

		isTopInBounds: function(topOffset) {
			return !this.boundingRect || (topOffset >= this.boundingRect.top && topOffset < this.boundingRect.bottom);
		}

	});

	;;

	/* Tracks a drag's mouse movement, firing various handlers
----------------------------------------------------------------------------------------------------------------------*/
// TODO: use Emitter

	var DragListener = FC.DragListener = Class.extend(ListenerMixin, MouseIgnorerMixin, {

		options: null,
		subjectEl: null,

		// coordinates of the initial mousedown
		originX: null,
		originY: null,

		// the wrapping element that scrolls, or MIGHT scroll if there's overflow.
		// TODO: do this for wrappers that have overflow:hidden as well.
		scrollEl: null,

		isInteracting: false,
		isDistanceSurpassed: false,
		isDelayEnded: false,
		isDragging: false,
		isTouch: false,

		delay: null,
		delayTimeoutId: null,
		minDistance: null,

		handleTouchScrollProxy: null, // calls handleTouchScroll, always bound to `this`


		constructor: function(options) {
			this.options = options || {};
			this.handleTouchScrollProxy = proxy(this, 'handleTouchScroll');
			this.initMouseIgnoring(500);
		},


		// Interaction (high-level)
		// -----------------------------------------------------------------------------------------------------------------


		startInteraction: function(ev, extraOptions) {
			var isTouch = getEvIsTouch(ev);

			if (ev.type === 'mousedown') {
				if (this.isIgnoringMouse) {
					return;
				}
				else if (!isPrimaryMouseButton(ev)) {
					return;
				}
				else {
					ev.preventDefault(); // prevents native selection in most browsers
				}
			}

			if (!this.isInteracting) {

				// process options
				extraOptions = extraOptions || {};
				this.delay = firstDefined(extraOptions.delay, this.options.delay, 0);
				this.minDistance = firstDefined(extraOptions.distance, this.options.distance, 0);
				this.subjectEl = this.options.subjectEl;

				this.isInteracting = true;
				this.isTouch = isTouch;
				this.isDelayEnded = false;
				this.isDistanceSurpassed = false;

				this.originX = getEvX(ev);
				this.originY = getEvY(ev);
				this.scrollEl = getScrollParent($(ev.target));

				this.bindHandlers();
				this.initAutoScroll();
				this.handleInteractionStart(ev);
				this.startDelay(ev);

				if (!this.minDistance) {
					this.handleDistanceSurpassed(ev);
				}
			}
		},


		handleInteractionStart: function(ev) {
			this.trigger('interactionStart', ev);
		},


		endInteraction: function(ev, isCancelled) {
			if (this.isInteracting) {
				this.endDrag(ev);

				if (this.delayTimeoutId) {
					clearTimeout(this.delayTimeoutId);
					this.delayTimeoutId = null;
				}

				this.destroyAutoScroll();
				this.unbindHandlers();

				this.isInteracting = false;
				this.handleInteractionEnd(ev, isCancelled);

				// a touchstart+touchend on the same element will result in the following addition simulated events:
				// mouseover + mouseout + click
				// let's ignore these bogus events
				if (this.isTouch) {
					this.tempIgnoreMouse();
				}
			}
		},


		handleInteractionEnd: function(ev, isCancelled) {
			this.trigger('interactionEnd', ev, isCancelled || false);
		},


		// Binding To DOM
		// -----------------------------------------------------------------------------------------------------------------


		bindHandlers: function() {
			var _this = this;
			var touchStartIgnores = 1;

			if (this.isTouch) {
				this.listenTo($(document), {
					touchmove: this.handleTouchMove,
					touchend: this.endInteraction,
					touchcancel: this.endInteraction,

					// Sometimes touchend doesn't fire
					// (can't figure out why. touchcancel doesn't fire either. has to do with scrolling?)
					// If another touchstart happens, we know it's bogus, so cancel the drag.
					// touchend will continue to be broken until user does a shorttap/scroll, but this is best we can do.
					touchstart: function(ev) {
						if (touchStartIgnores) { // bindHandlers is called from within a touchstart,
							touchStartIgnores--; // and we don't want this to fire immediately, so ignore.
						}
						else {
							_this.endInteraction(ev, true); // isCancelled=true
						}
					}
				});

				// listen to ALL scroll actions on the page
				if (
					!bindAnyScroll(this.handleTouchScrollProxy) && // hopefully this works and short-circuits the rest
					this.scrollEl // otherwise, attach a single handler to this
				) {
					this.listenTo(this.scrollEl, 'scroll', this.handleTouchScroll);
				}
			}
			else {
				this.listenTo($(document), {
					mousemove: this.handleMouseMove,
					mouseup: this.endInteraction
				});
			}

			this.listenTo($(document), {
				selectstart: preventDefault, // don't allow selection while dragging
				contextmenu: preventDefault // long taps would open menu on Chrome dev tools
			});
		},


		unbindHandlers: function() {
			this.stopListeningTo($(document));

			// unbind scroll listening
			unbindAnyScroll(this.handleTouchScrollProxy);
			if (this.scrollEl) {
				this.stopListeningTo(this.scrollEl, 'scroll');
			}
		},


		// Drag (high-level)
		// -----------------------------------------------------------------------------------------------------------------


		// extraOptions ignored if drag already started
		startDrag: function(ev, extraOptions) {
			this.startInteraction(ev, extraOptions); // ensure interaction began

			if (!this.isDragging) {
				this.isDragging = true;
				this.handleDragStart(ev);
			}
		},


		handleDragStart: function(ev) {
			this.trigger('dragStart', ev);
		},


		handleMove: function(ev) {
			var dx = getEvX(ev) - this.originX;
			var dy = getEvY(ev) - this.originY;
			var minDistance = this.minDistance;
			var distanceSq; // current distance from the origin, squared

			if (!this.isDistanceSurpassed) {
				distanceSq = dx * dx + dy * dy;
				if (distanceSq >= minDistance * minDistance) { // use pythagorean theorem
					this.handleDistanceSurpassed(ev);
				}
			}

			if (this.isDragging) {
				this.handleDrag(dx, dy, ev);
			}
		},


		// Called while the mouse is being moved and when we know a legitimate drag is taking place
		handleDrag: function(dx, dy, ev) {
			this.trigger('drag', dx, dy, ev);
			this.updateAutoScroll(ev); // will possibly cause scrolling
		},


		endDrag: function(ev) {
			if (this.isDragging) {
				this.isDragging = false;
				this.handleDragEnd(ev);
			}
		},


		handleDragEnd: function(ev) {
			this.trigger('dragEnd', ev);
		},


		// Delay
		// -----------------------------------------------------------------------------------------------------------------


		startDelay: function(initialEv) {
			var _this = this;

			if (this.delay) {
				this.delayTimeoutId = setTimeout(function() {
					_this.handleDelayEnd(initialEv);
				}, this.delay);
			}
			else {
				this.handleDelayEnd(initialEv);
			}
		},


		handleDelayEnd: function(initialEv) {
			this.isDelayEnded = true;

			if (this.isDistanceSurpassed) {
				this.startDrag(initialEv);
			}
		},


		// Distance
		// -----------------------------------------------------------------------------------------------------------------


		handleDistanceSurpassed: function(ev) {
			this.isDistanceSurpassed = true;

			if (this.isDelayEnded) {
				this.startDrag(ev);
			}
		},


		// Mouse / Touch
		// -----------------------------------------------------------------------------------------------------------------


		handleTouchMove: function(ev) {
			// prevent inertia and touchmove-scrolling while dragging
			if (this.isDragging) {
				ev.preventDefault();
			}

			this.handleMove(ev);
		},


		handleMouseMove: function(ev) {
			this.handleMove(ev);
		},


		// Scrolling (unrelated to auto-scroll)
		// -----------------------------------------------------------------------------------------------------------------


		handleTouchScroll: function(ev) {
			// if the drag is being initiated by touch, but a scroll happens before
			// the drag-initiating delay is over, cancel the drag
			if (!this.isDragging) {
				this.endInteraction(ev, true); // isCancelled=true
			}
		},


		// Utils
		// -----------------------------------------------------------------------------------------------------------------


		// Triggers a callback. Calls a function in the option hash of the same name.
		// Arguments beyond the first `name` are forwarded on.
		trigger: function(name) {
			if (this.options[name]) {
				this.options[name].apply(this, Array.prototype.slice.call(arguments, 1));
			}
			// makes _methods callable by event name. TODO: kill this
			if (this['_' + name]) {
				this['_' + name].apply(this, Array.prototype.slice.call(arguments, 1));
			}
		}


	});

	;;
	/*
this.scrollEl is set in DragListener
*/
	DragListener.mixin({

		isAutoScroll: false,

		scrollBounds: null, // { top, bottom, left, right }
		scrollTopVel: null, // pixels per second
		scrollLeftVel: null, // pixels per second
		scrollIntervalId: null, // ID of setTimeout for scrolling animation loop

		// defaults
		scrollSensitivity: 30, // pixels from edge for scrolling to start
		scrollSpeed: 200, // pixels per second, at maximum speed
		scrollIntervalMs: 50, // millisecond wait between scroll increment


		initAutoScroll: function() {
			var scrollEl = this.scrollEl;

			this.isAutoScroll =
				this.options.scroll &&
				scrollEl &&
				!scrollEl.is(window) &&
				!scrollEl.is(document);

			if (this.isAutoScroll) {
				// debounce makes sure rapid calls don't happen
				this.listenTo(scrollEl, 'scroll', debounce(this.handleDebouncedScroll, 100));
			}
		},


		destroyAutoScroll: function() {
			this.endAutoScroll(); // kill any animation loop

			// remove the scroll handler if there is a scrollEl
			if (this.isAutoScroll) {
				this.stopListeningTo(this.scrollEl, 'scroll'); // will probably get removed by unbindHandlers too :(
			}
		},


		// Computes and stores the bounding rectangle of scrollEl
		computeScrollBounds: function() {
			if (this.isAutoScroll) {
				this.scrollBounds = getOuterRect(this.scrollEl);
				// TODO: use getClientRect in future. but prevents auto scrolling when on top of scrollbars
			}
		},


		// Called when the dragging is in progress and scrolling should be updated
		updateAutoScroll: function(ev) {
			var sensitivity = this.scrollSensitivity;
			var bounds = this.scrollBounds;
			var topCloseness, bottomCloseness;
			var leftCloseness, rightCloseness;
			var topVel = 0;
			var leftVel = 0;

			if (bounds) { // only scroll if scrollEl exists

				// compute closeness to edges. valid range is from 0.0 - 1.0
				topCloseness = (sensitivity - (getEvY(ev) - bounds.top)) / sensitivity;
				bottomCloseness = (sensitivity - (bounds.bottom - getEvY(ev))) / sensitivity;
				leftCloseness = (sensitivity - (getEvX(ev) - bounds.left)) / sensitivity;
				rightCloseness = (sensitivity - (bounds.right - getEvX(ev))) / sensitivity;

				// translate vertical closeness into velocity.
				// mouse must be completely in bounds for velocity to happen.
				if (topCloseness >= 0 && topCloseness <= 1) {
					topVel = topCloseness * this.scrollSpeed * -1; // negative. for scrolling up
				}
				else if (bottomCloseness >= 0 && bottomCloseness <= 1) {
					topVel = bottomCloseness * this.scrollSpeed;
				}

				// translate horizontal closeness into velocity
				if (leftCloseness >= 0 && leftCloseness <= 1) {
					leftVel = leftCloseness * this.scrollSpeed * -1; // negative. for scrolling left
				}
				else if (rightCloseness >= 0 && rightCloseness <= 1) {
					leftVel = rightCloseness * this.scrollSpeed;
				}
			}

			this.setScrollVel(topVel, leftVel);
		},


		// Sets the speed-of-scrolling for the scrollEl
		setScrollVel: function(topVel, leftVel) {

			this.scrollTopVel = topVel;
			this.scrollLeftVel = leftVel;

			this.constrainScrollVel(); // massages into realistic values

			// if there is non-zero velocity, and an animation loop hasn't already started, then START
			if ((this.scrollTopVel || this.scrollLeftVel) && !this.scrollIntervalId) {
				this.scrollIntervalId = setInterval(
					proxy(this, 'scrollIntervalFunc'), // scope to `this`
					this.scrollIntervalMs
				);
			}
		},


		// Forces scrollTopVel and scrollLeftVel to be zero if scrolling has already gone all the way
		constrainScrollVel: function() {
			var el = this.scrollEl;

			if (this.scrollTopVel < 0) { // scrolling up?
				if (el.scrollTop() <= 0) { // already scrolled all the way up?
					this.scrollTopVel = 0;
				}
			}
			else if (this.scrollTopVel > 0) { // scrolling down?
				if (el.scrollTop() + el[0].clientHeight >= el[0].scrollHeight) { // already scrolled all the way down?
					this.scrollTopVel = 0;
				}
			}

			if (this.scrollLeftVel < 0) { // scrolling left?
				if (el.scrollLeft() <= 0) { // already scrolled all the left?
					this.scrollLeftVel = 0;
				}
			}
			else if (this.scrollLeftVel > 0) { // scrolling right?
				if (el.scrollLeft() + el[0].clientWidth >= el[0].scrollWidth) { // already scrolled all the way right?
					this.scrollLeftVel = 0;
				}
			}
		},


		// This function gets called during every iteration of the scrolling animation loop
		scrollIntervalFunc: function() {
			var el = this.scrollEl;
			var frac = this.scrollIntervalMs / 1000; // considering animation frequency, what the vel should be mult'd by

			// change the value of scrollEl's scroll
			if (this.scrollTopVel) {
				el.scrollTop(el.scrollTop() + this.scrollTopVel * frac);
			}
			if (this.scrollLeftVel) {
				el.scrollLeft(el.scrollLeft() + this.scrollLeftVel * frac);
			}

			this.constrainScrollVel(); // since the scroll values changed, recompute the velocities

			// if scrolled all the way, which causes the vels to be zero, stop the animation loop
			if (!this.scrollTopVel && !this.scrollLeftVel) {
				this.endAutoScroll();
			}
		},


		// Kills any existing scrolling animation loop
		endAutoScroll: function() {
			if (this.scrollIntervalId) {
				clearInterval(this.scrollIntervalId);
				this.scrollIntervalId = null;

				this.handleScrollEnd();
			}
		},


		// Get called when the scrollEl is scrolled (NOTE: this is delayed via debounce)
		handleDebouncedScroll: function() {
			// recompute all coordinates, but *only* if this is *not* part of our scrolling animation
			if (!this.scrollIntervalId) {
				this.handleScrollEnd();
			}
		},


		// Called when scrolling has stopped, whether through auto scroll, or the user scrolling
		handleScrollEnd: function() {
		}

	});
	;;

	/* Tracks mouse movements over a component and raises events about which hit the mouse is over.
------------------------------------------------------------------------------------------------------------------------
options:
- subjectEl
- subjectCenter
*/

	var HitDragListener = DragListener.extend({

		component: null, // converts coordinates to hits
		// methods: prepareHits, releaseHits, queryHit

		origHit: null, // the hit the mouse was over when listening started
		hit: null, // the hit the mouse is over
		coordAdjust: null, // delta that will be added to the mouse coordinates when computing collisions


		constructor: function(component, options) {
			DragListener.call(this, options); // call the super-constructor

			this.component = component;
		},


		// Called when drag listening starts (but a real drag has not necessarily began).
		// ev might be undefined if dragging was started manually.
		handleInteractionStart: function(ev) {
			var subjectEl = this.subjectEl;
			var subjectRect;
			var origPoint;
			var point;

			this.computeCoords();

			if (ev) {
				origPoint = { left: getEvX(ev), top: getEvY(ev) };
				point = origPoint;

				// constrain the point to bounds of the element being dragged
				if (subjectEl) {
					subjectRect = getOuterRect(subjectEl); // used for centering as well
					point = constrainPoint(point, subjectRect);
				}

				this.origHit = this.queryHit(point.left, point.top);

				// treat the center of the subject as the collision point?
				if (subjectEl && this.options.subjectCenter) {

					// only consider the area the subject overlaps the hit. best for large subjects.
					// TODO: skip this if hit didn't supply left/right/top/bottom
					if (this.origHit) {
						subjectRect = intersectRects(this.origHit, subjectRect) ||
							subjectRect; // in case there is no intersection
					}

					point = getRectCenter(subjectRect);
				}

				this.coordAdjust = diffPoints(point, origPoint); // point - origPoint
			}
			else {
				this.origHit = null;
				this.coordAdjust = null;
			}

			// call the super-method. do it after origHit has been computed
			DragListener.prototype.handleInteractionStart.apply(this, arguments);
		},


		// Recomputes the drag-critical positions of elements
		computeCoords: function() {
			this.component.prepareHits();
			this.computeScrollBounds(); // why is this here??????
		},


		// Called when the actual drag has started
		handleDragStart: function(ev) {
			var hit;

			DragListener.prototype.handleDragStart.apply(this, arguments); // call the super-method

			// might be different from this.origHit if the min-distance is large
			hit = this.queryHit(getEvX(ev), getEvY(ev));

			// report the initial hit the mouse is over
			// especially important if no min-distance and drag starts immediately
			if (hit) {
				this.handleHitOver(hit);
			}
		},


		// Called when the drag moves
		handleDrag: function(dx, dy, ev) {
			var hit;

			DragListener.prototype.handleDrag.apply(this, arguments); // call the super-method

			hit = this.queryHit(getEvX(ev), getEvY(ev));

			if (!isHitsEqual(hit, this.hit)) { // a different hit than before?
				if (this.hit) {
					this.handleHitOut();
				}
				if (hit) {
					this.handleHitOver(hit);
				}
			}
		},


		// Called when dragging has been stopped
		handleDragEnd: function() {
			this.handleHitDone();
			DragListener.prototype.handleDragEnd.apply(this, arguments); // call the super-method
		},


		// Called when a the mouse has just moved over a new hit
		handleHitOver: function(hit) {
			var isOrig = isHitsEqual(hit, this.origHit);

			this.hit = hit;

			this.trigger('hitOver', this.hit, isOrig, this.origHit);
		},


		// Called when the mouse has just moved out of a hit
		handleHitOut: function() {
			if (this.hit) {
				this.trigger('hitOut', this.hit);
				this.handleHitDone();
				this.hit = null;
			}
		},


		// Called after a hitOut. Also called before a dragStop
		handleHitDone: function() {
			if (this.hit) {
				this.trigger('hitDone', this.hit);
			}
		},


		// Called when the interaction ends, whether there was a real drag or not
		handleInteractionEnd: function() {
			DragListener.prototype.handleInteractionEnd.apply(this, arguments); // call the super-method

			this.origHit = null;
			this.hit = null;

			this.component.releaseHits();
		},


		// Called when scrolling has stopped, whether through auto scroll, or the user scrolling
		handleScrollEnd: function() {
			DragListener.prototype.handleScrollEnd.apply(this, arguments); // call the super-method

			this.computeCoords(); // hits' absolute positions will be in new places. recompute
		},


		// Gets the hit underneath the coordinates for the given mouse event
		queryHit: function(left, top) {

			if (this.coordAdjust) {
				left += this.coordAdjust.left;
				top += this.coordAdjust.top;
			}

			return this.component.queryHit(left, top);
		}

	});


// Returns `true` if the hits are identically equal. `false` otherwise. Must be from the same component.
// Two null values will be considered equal, as two "out of the component" states are the same.
	function isHitsEqual(hit0, hit1) {

		if (!hit0 && !hit1) {
			return true;
		}

		if (hit0 && hit1) {
			return hit0.component === hit1.component &&
				isHitPropsWithin(hit0, hit1) &&
				isHitPropsWithin(hit1, hit0); // ensures all props are identical
		}

		return false;
	}


// Returns true if all of subHit's non-standard properties are within superHit
	function isHitPropsWithin(subHit, superHit) {
		for (var propName in subHit) {
			if (!/^(component|left|right|top|bottom)$/.test(propName)) {
				if (subHit[propName] !== superHit[propName]) {
					return false;
				}
			}
		}
		return true;
	}

	;;

	/* Creates a clone of an element and lets it track the mouse as it moves
----------------------------------------------------------------------------------------------------------------------*/

	var MouseFollower = Class.extend(ListenerMixin, {

		options: null,

		sourceEl: null, // the element that will be cloned and made to look like it is dragging
		el: null, // the clone of `sourceEl` that will track the mouse
		parentEl: null, // the element that `el` (the clone) will be attached to

		// the initial position of el, relative to the offset parent. made to match the initial offset of sourceEl
		top0: null,
		left0: null,

		// the absolute coordinates of the initiating touch/mouse action
		y0: null,
		x0: null,

		// the number of pixels the mouse has moved from its initial position
		topDelta: null,
		leftDelta: null,

		isFollowing: false,
		isHidden: false,
		isAnimating: false, // doing the revert animation?

		constructor: function(sourceEl, options) {
			this.options = options = options || {};
			this.sourceEl = sourceEl;
			this.parentEl = options.parentEl ? $(options.parentEl) : sourceEl.parent(); // default to sourceEl's parent
		},


		// Causes the element to start following the mouse
		start: function(ev) {
			if (!this.isFollowing) {
				this.isFollowing = true;

				this.y0 = getEvY(ev);
				this.x0 = getEvX(ev);
				this.topDelta = 0;
				this.leftDelta = 0;

				if (!this.isHidden) {
					this.updatePosition();
				}

				if (getEvIsTouch(ev)) {
					this.listenTo($(document), 'touchmove', this.handleMove);
				}
				else {
					this.listenTo($(document), 'mousemove', this.handleMove);
				}
			}
		},


		// Causes the element to stop following the mouse. If shouldRevert is true, will animate back to original position.
		// `callback` gets invoked when the animation is complete. If no animation, it is invoked immediately.
		stop: function(shouldRevert, callback) {
			var _this = this;
			var revertDuration = this.options.revertDuration;

			function complete() { // might be called by .animate(), which might change `this` context
				_this.isAnimating = false;
				_this.removeElement();

				_this.top0 = _this.left0 = null; // reset state for future updatePosition calls

				if (callback) {
					callback();
				}
			}

			if (this.isFollowing && !this.isAnimating) { // disallow more than one stop animation at a time
				this.isFollowing = false;

				this.stopListeningTo($(document));

				if (shouldRevert && revertDuration && !this.isHidden) { // do a revert animation?
					this.isAnimating = true;
					this.el.animate({
						top: this.top0,
						left: this.left0
					}, {
						duration: revertDuration,
						complete: complete
					});
				}
				else {
					complete();
				}
			}
		},


		// Gets the tracking element. Create it if necessary
		getEl: function() {
			var el = this.el;

			if (!el) {
				el = this.el = this.sourceEl.clone()
					.addClass(this.options.additionalClass || '')
					.css({
						position: 'absolute',
						visibility: '', // in case original element was hidden (commonly through hideEvents())
						display: this.isHidden ? 'none' : '', // for when initially hidden
						margin: 0,
						right: 'auto', // erase and set width instead
						bottom: 'auto', // erase and set height instead
						width: this.sourceEl.width(), // explicit height in case there was a 'right' value
						height: this.sourceEl.height(), // explicit width in case there was a 'bottom' value
						opacity: this.options.opacity || '',
						zIndex: this.options.zIndex
					});

				// we don't want long taps or any mouse interaction causing selection/menus.
				// would use preventSelection(), but that prevents selectstart, causing problems.
				el.addClass('fc-unselectable');

				el.appendTo(this.parentEl);
			}

			return el;
		},


		// Removes the tracking element if it has already been created
		removeElement: function() {
			if (this.el) {
				this.el.remove();
				this.el = null;
			}
		},


		// Update the CSS position of the tracking element
		updatePosition: function() {
			var sourceOffset;
			var origin;

			this.getEl(); // ensure this.el

			// make sure origin info was computed
			if (this.top0 === null) {
				sourceOffset = this.sourceEl.offset();
				origin = this.el.offsetParent().offset();
				this.top0 = sourceOffset.top - origin.top;
				this.left0 = sourceOffset.left - origin.left;
			}

			this.el.css({
				top: this.top0 + this.topDelta,
				left: this.left0 + this.leftDelta
			});
		},


		// Gets called when the user moves the mouse
		handleMove: function(ev) {
			this.topDelta = getEvY(ev) - this.y0;
			this.leftDelta = getEvX(ev) - this.x0;

			if (!this.isHidden) {
				this.updatePosition();
			}
		},


		// Temporarily makes the tracking element invisible. Can be called before following starts
		hide: function() {
			if (!this.isHidden) {
				this.isHidden = true;
				if (this.el) {
					this.el.hide();
				}
			}
		},


		// Show the tracking element after it has been temporarily hidden
		show: function() {
			if (this.isHidden) {
				this.isHidden = false;
				this.updatePosition();
				this.getEl().show();
			}
		}

	});

	;;

	/* An abstract class comprised of a "grid" of areas that each represent a specific datetime
----------------------------------------------------------------------------------------------------------------------*/

	var Grid = FC.Grid = Class.extend(ListenerMixin, MouseIgnorerMixin, {

		// self-config, overridable by subclasses
		hasDayInteractions: true, // can user click/select ranges of time?

		view: null, // a View object
		isRTL: null, // shortcut to the view's isRTL option

		start: null,
		end: null,

		el: null, // the containing element
		elsByFill: null, // a hash of jQuery element sets used for rendering each fill. Keyed by fill name.

		// derived from options
		eventTimeFormat: null,
		displayEventTime: null,
		displayEventEnd: null,

		minResizeDuration: null, // TODO: hack. set by subclasses. minumum event resize duration

		// if defined, holds the unit identified (ex: "year" or "month") that determines the level of granularity
		// of the date areas. if not defined, assumes to be day and time granularity.
		// TODO: port isTimeScale into same system?
		largeUnit: null,

		dayDragListener: null,
		segDragListener: null,
		segResizeListener: null,
		externalDragListener: null,


		constructor: function(view) {
			this.view = view;
			this.isRTL = view.opt('isRTL');
			this.elsByFill = {};

			this.dayDragListener = this.buildDayDragListener();
			this.initMouseIgnoring();
		},


		/* Options
	------------------------------------------------------------------------------------------------------------------*/


		// Generates the format string used for event time text, if not explicitly defined by 'timeFormat'
		computeEventTimeFormat: function() {
			return this.view.opt('smallTimeFormat');
		},


		// Determines whether events should have their end times displayed, if not explicitly defined by 'displayEventTime'.
		// Only applies to non-all-day events.
		computeDisplayEventTime: function() {
			return true;
		},


		// Determines whether events should have their end times displayed, if not explicitly defined by 'displayEventEnd'
		computeDisplayEventEnd: function() {
			return true;
		},


		/* Dates
	------------------------------------------------------------------------------------------------------------------*/


		// Tells the grid about what period of time to display.
		// Any date-related internal data should be generated.
		setRange: function(range) {
			this.start = range.start.clone();
			this.end = range.end.clone();

			this.rangeUpdated();
			this.processRangeOptions();
		},


		// Called when internal variables that rely on the range should be updated
		rangeUpdated: function() {
		},


		// Updates values that rely on options and also relate to range
		processRangeOptions: function() {
			var view = this.view;
			var displayEventTime;
			var displayEventEnd;

			this.eventTimeFormat =
				view.opt('eventTimeFormat') ||
				view.opt('timeFormat') || // deprecated
				this.computeEventTimeFormat();

			displayEventTime = view.opt('displayEventTime');
			if (displayEventTime == null) {
				displayEventTime = this.computeDisplayEventTime(); // might be based off of range
			}

			displayEventEnd = view.opt('displayEventEnd');
			if (displayEventEnd == null) {
				displayEventEnd = this.computeDisplayEventEnd(); // might be based off of range
			}

			this.displayEventTime = displayEventTime;
			this.displayEventEnd = displayEventEnd;
		},


		// Converts a span (has unzoned start/end and any other grid-specific location information)
		// into an array of segments (pieces of events whose format is decided by the grid).
		spanToSegs: function(span) {
			// subclasses must implement
		},


		// Diffs the two dates, returning a duration, based on granularity of the grid
		// TODO: port isTimeScale into this system?
		diffDates: function(a, b) {
			if (this.largeUnit) {
				return diffByUnit(a, b, this.largeUnit);
			}
			else {
				return diffDayTime(a, b);
			}
		},


		/* Hit Area
	------------------------------------------------------------------------------------------------------------------*/


		// Called before one or more queryHit calls might happen. Should prepare any cached coordinates for queryHit
		prepareHits: function() {
		},


		// Called when queryHit calls have subsided. Good place to clear any coordinate caches.
		releaseHits: function() {
		},


		// Given coordinates from the topleft of the document, return data about the date-related area underneath.
		// Can return an object with arbitrary properties (although top/right/left/bottom are encouraged).
		// Must have a `grid` property, a reference to this current grid. TODO: avoid this
		// The returned object will be processed by getHitSpan and getHitEl.
		queryHit: function(leftOffset, topOffset) {
		},


		// Given position-level information about a date-related area within the grid,
		// should return an object with at least a start/end date. Can provide other information as well.
		getHitSpan: function(hit) {
		},


		// Given position-level information about a date-related area within the grid,
		// should return a jQuery element that best represents it. passed to dayClick callback.
		getHitEl: function(hit) {
		},


		/* Rendering
	------------------------------------------------------------------------------------------------------------------*/


		// Sets the container element that the grid should render inside of.
		// Does other DOM-related initializations.
		setElement: function(el) {
			this.el = el;

			if (this.hasDayInteractions) {
				preventSelection(el);

				this.bindDayHandler('touchstart', this.dayTouchStart);
				this.bindDayHandler('mousedown', this.dayMousedown);
			}

			// attach event-element-related handlers. in Grid.events
			// same garbage collection note as above.
			this.bindSegHandlers();

			this.bindGlobalHandlers();
		},


		bindDayHandler: function(name, handler) {
			var _this = this;

			// attach a handler to the grid's root element.
			// jQuery will take care of unregistering them when removeElement gets called.
			this.el.on(name, function(ev) {
				if (
					!$(ev.target).is(
						_this.segSelector + ',' + // directly on an event element
						_this.segSelector + ' *,' + // within an event element
						'.fc-more,' + // a "more.." link
						'a[data-goto]' // a clickable nav link
					)
				) {
					return handler.call(_this, ev);
				}
			});
		},


		// Removes the grid's container element from the DOM. Undoes any other DOM-related attachments.
		// DOES NOT remove any content beforehand (doesn't clear events or call unrenderDates), unlike View
		removeElement: function() {
			this.unbindGlobalHandlers();
			this.clearDragListeners();

			this.el.remove();

			// NOTE: we don't null-out this.el for the same reasons we don't do it within View::removeElement
		},


		// Renders the basic structure of grid view before any content is rendered
		renderSkeleton: function() {
			// subclasses should implement
		},


		// Renders the grid's date-related content (like areas that represent days/times).
		// Assumes setRange has already been called and the skeleton has already been rendered.
		renderDates: function() {
			// subclasses should implement
		},


		// Unrenders the grid's date-related content
		unrenderDates: function() {
			// subclasses should implement
		},


		/* Handlers
	------------------------------------------------------------------------------------------------------------------*/


		// Binds DOM handlers to elements that reside outside the grid, such as the document
		bindGlobalHandlers: function() {
			this.listenTo($(document), {
				dragstart: this.externalDragStart, // jqui
				sortstart: this.externalDragStart // jqui
			});
		},


		// Unbinds DOM handlers from elements that reside outside the grid
		unbindGlobalHandlers: function() {
			this.stopListeningTo($(document));
		},


		// Process a mousedown on an element that represents a day. For day clicking and selecting.
		dayMousedown: function(ev) {
			if (!this.isIgnoringMouse) {
				this.dayDragListener.startInteraction(ev, {
					//distance: 5, // needs more work if we want dayClick to fire correctly
				});
			}
		},


		dayTouchStart: function(ev) {
			var view = this.view;
			var selectLongPressDelay = view.opt('selectLongPressDelay');

			// HACK to prevent a user's clickaway for unselecting a range or an event
			// from causing a dayClick.
			if (view.isSelected || view.selectedEvent) {
				this.tempIgnoreMouse();
			}

			if (selectLongPressDelay == null) {
				selectLongPressDelay = view.opt('longPressDelay'); // fallback
			}

			this.dayDragListener.startInteraction(ev, {
				delay: selectLongPressDelay
			});
		},


		// Creates a listener that tracks the user's drag across day elements.
		// For day clicking and selecting.
		buildDayDragListener: function() {
			var _this = this;
			var view = this.view;
			var isSelectable = view.opt('selectable');
			var dayClickHit; // null if invalid dayClick
			var selectionSpan; // null if invalid selection

			// this listener tracks a mousedown on a day element, and a subsequent drag.
			// if the drag ends on the same day, it is a 'dayClick'.
			// if 'selectable' is enabled, this listener also detects selections.
			var dragListener = new HitDragListener(this, {
				scroll: view.opt('dragScroll'),
				interactionStart: function() {
					dayClickHit = dragListener.origHit; // for dayClick, where no dragging happens
					selectionSpan = null;
				},
				dragStart: function() {
					view.unselect(); // since we could be rendering a new selection, we want to clear any old one
				},
				hitOver: function(hit, isOrig, origHit) {
					if (origHit) { // click needs to have started on a hit

						// if user dragged to another cell at any point, it can no longer be a dayClick
						if (!isOrig) {
							dayClickHit = null;
						}

						if (isSelectable) {
							selectionSpan = _this.computeSelection(
								_this.getHitSpan(origHit),
								_this.getHitSpan(hit)
							);
							if (selectionSpan) {
								_this.renderSelection(selectionSpan);
							}
							else if (selectionSpan === false) {
								disableCursor();
							}
						}
					}
				},
				hitOut: function() { // called before mouse moves to a different hit OR moved out of all hits
					dayClickHit = null;
					selectionSpan = null;
					_this.unrenderSelection();
				},
				hitDone: function() { // called after a hitOut OR before a dragEnd
					enableCursor();
				},
				interactionEnd: function(ev, isCancelled) {
					if (!isCancelled) {
						if (
							dayClickHit &&
							!_this.isIgnoringMouse // see hack in dayTouchStart
						) {
							view.triggerDayClick(
								_this.getHitSpan(dayClickHit),
								_this.getHitEl(dayClickHit),
								ev
							);
						}
						if (selectionSpan) {
							// the selection will already have been rendered. just report it
							view.reportSelection(selectionSpan, ev);
						}
					}
				}
			});

			return dragListener;
		},


		// Kills all in-progress dragging.
		// Useful for when public API methods that result in re-rendering are invoked during a drag.
		// Also useful for when touch devices misbehave and don't fire their touchend.
		clearDragListeners: function() {
			this.dayDragListener.endInteraction();

			if (this.segDragListener) {
				this.segDragListener.endInteraction(); // will clear this.segDragListener
			}
			if (this.segResizeListener) {
				this.segResizeListener.endInteraction(); // will clear this.segResizeListener
			}
			if (this.externalDragListener) {
				this.externalDragListener.endInteraction(); // will clear this.externalDragListener
			}
		},


		/* Event Helper
	------------------------------------------------------------------------------------------------------------------*/
		// TODO: should probably move this to Grid.events, like we did event dragging / resizing


		// Renders a mock event at the given event location, which contains zoned start/end properties.
		// Returns all mock event elements.
		renderEventLocationHelper: function(eventLocation, sourceSeg) {
			var fakeEvent = this.fabricateHelperEvent(eventLocation, sourceSeg);

			return this.renderHelper(fakeEvent, sourceSeg); // do the actual rendering
		},


		// Builds a fake event given zoned event date properties and a segment is should be inspired from.
		// The range's end can be null, in which case the mock event that is rendered will have a null end time.
		// `sourceSeg` is the internal segment object involved in the drag. If null, something external is dragging.
		fabricateHelperEvent: function(eventLocation, sourceSeg) {
			var fakeEvent = sourceSeg ? createObject(sourceSeg.event) : {}; // mask the original event object if possible

			fakeEvent.start = eventLocation.start.clone();
			fakeEvent.end = eventLocation.end ? eventLocation.end.clone() : null;
			fakeEvent.allDay = null; // force it to be freshly computed by normalizeEventDates
			this.view.calendar.normalizeEventDates(fakeEvent);

			// this extra className will be useful for differentiating real events from mock events in CSS
			fakeEvent.className = (fakeEvent.className || []).concat('fc-helper');

			// if something external is being dragged in, don't render a resizer
			if (!sourceSeg) {
				fakeEvent.editable = false;
			}

			return fakeEvent;
		},


		// Renders a mock event. Given zoned event date properties.
		// Must return all mock event elements.
		renderHelper: function(eventLocation, sourceSeg) {
			// subclasses must implement
		},


		// Unrenders a mock event
		unrenderHelper: function() {
			// subclasses must implement
		},


		/* Selection
	------------------------------------------------------------------------------------------------------------------*/


		// Renders a visual indication of a selection. Will highlight by default but can be overridden by subclasses.
		// Given a span (unzoned start/end and other misc data)
		renderSelection: function(span) {
			this.renderHighlight(span);
		},


		// Unrenders any visual indications of a selection. Will unrender a highlight by default.
		unrenderSelection: function() {
			this.unrenderHighlight();
		},


		// Given the first and last date-spans of a selection, returns another date-span object.
		// Subclasses can override and provide additional data in the span object. Will be passed to renderSelection().
		// Will return false if the selection is invalid and this should be indicated to the user.
		// Will return null/undefined if a selection invalid but no error should be reported.
		computeSelection: function(span0, span1) {
			var span = this.computeSelectionSpan(span0, span1);

			if (span && !this.view.calendar.isSelectionSpanAllowed(span)) {
				return false;
			}

			return span;
		},


		// Given two spans, must return the combination of the two.
		// TODO: do this separation of concerns (combining VS validation) for event dnd/resize too.
		computeSelectionSpan: function(span0, span1) {
			var dates = [ span0.start, span0.end, span1.start, span1.end ];

			dates.sort(compareNumbers); // sorts chronologically. works with Moments

			return { start: dates[0].clone(), end: dates[3].clone() };
		},


		/* Highlight
	------------------------------------------------------------------------------------------------------------------*/


		// Renders an emphasis on the given date range. Given a span (unzoned start/end and other misc data)
		renderHighlight: function(span) {
			this.renderFill('highlight', this.spanToSegs(span));
		},


		// Unrenders the emphasis on a date range
		unrenderHighlight: function() {
			this.unrenderFill('highlight');
		},


		// Generates an array of classNames for rendering the highlight. Used by the fill system.
		highlightSegClasses: function() {
			return [ 'fc-highlight' ];
		},


		/* Business Hours
	------------------------------------------------------------------------------------------------------------------*/


		renderBusinessHours: function() {
		},


		unrenderBusinessHours: function() {
		},


		/* Now Indicator
	------------------------------------------------------------------------------------------------------------------*/


		getNowIndicatorUnit: function() {
		},


		renderNowIndicator: function(date) {
		},


		unrenderNowIndicator: function() {
		},


		/* Fill System (highlight, background events, business hours)
	--------------------------------------------------------------------------------------------------------------------
	TODO: remove this system. like we did in TimeGrid
	*/


		// Renders a set of rectangles over the given segments of time.
		// MUST RETURN a subset of segs, the segs that were actually rendered.
		// Responsible for populating this.elsByFill. TODO: better API for expressing this requirement
		renderFill: function(type, segs) {
			// subclasses must implement
		},


		// Unrenders a specific type of fill that is currently rendered on the grid
		unrenderFill: function(type) {
			var el = this.elsByFill[type];

			if (el) {
				el.remove();
				delete this.elsByFill[type];
			}
		},


		// Renders and assigns an `el` property for each fill segment. Generic enough to work with different types.
		// Only returns segments that successfully rendered.
		// To be harnessed by renderFill (implemented by subclasses).
		// Analagous to renderFgSegEls.
		renderFillSegEls: function(type, segs) {
			var _this = this;
			var segElMethod = this[type + 'SegEl'];
			var html = '';
			var renderedSegs = [];
			var i;

			if (segs.length) {

				// build a large concatenation of segment HTML
				for (i = 0; i < segs.length; i++) {
					html += this.fillSegHtml(type, segs[i]);
				}

				// Grab individual elements from the combined HTML string. Use each as the default rendering.
				// Then, compute the 'el' for each segment.
				$(html).each(function(i, node) {
					var seg = segs[i];
					var el = $(node);

					// allow custom filter methods per-type
					if (segElMethod) {
						el = segElMethod.call(_this, seg, el);
					}

					if (el) { // custom filters did not cancel the render
						el = $(el); // allow custom filter to return raw DOM node

						// correct element type? (would be bad if a non-TD were inserted into a table for example)
						if (el.is(_this.fillSegTag)) {
							seg.el = el;
							renderedSegs.push(seg);
						}
					}
				});
			}

			return renderedSegs;
		},


		fillSegTag: 'div', // subclasses can override


		// Builds the HTML needed for one fill segment. Generic enough to work with different types.
		fillSegHtml: function(type, seg) {

			// custom hooks per-type
			var classesMethod = this[type + 'SegClasses'];
			var cssMethod = this[type + 'SegCss'];

			var classes = classesMethod ? classesMethod.call(this, seg) : [];
			var css = cssToStr(cssMethod ? cssMethod.call(this, seg) : {});

			return '<' + this.fillSegTag +
				(classes.length ? ' class="' + classes.join(' ') + '"' : '') +
				(css ? ' style="' + css + '"' : '') +
				' />';
		},



		/* Generic rendering utilities for subclasses
	------------------------------------------------------------------------------------------------------------------*/


		// Computes HTML classNames for a single-day element
		getDayClasses: function(date, noThemeHighlight) {
			var view = this.view;
			var today = view.calendar.getNow();
			var classes = [ 'fc-' + dayIDs[date.day()] ];

			if (
				view.intervalDuration.as('months') == 1 &&
				date.month() != view.intervalStart.month()
			) {
				classes.push('fc-other-month');
			}

			if (date.isSame(today, 'day')) {
				classes.push('fc-today');

				if (noThemeHighlight !== true) {
					classes.push(view.highlightStateClass);
				}
			}
			else if (date < today) {
				classes.push('fc-past');
			}
			else {
				classes.push('fc-future');
			}

			return classes;
		}

	});

	;;

	/* Event-rendering and event-interaction methods for the abstract Grid class
----------------------------------------------------------------------------------------------------------------------*/

	Grid.mixin({

		// self-config, overridable by subclasses
		segSelector: '.fc-event-container > *', // what constitutes an event element?

		mousedOverSeg: null, // the segment object the user's mouse is over. null if over nothing
		isDraggingSeg: false, // is a segment being dragged? boolean
		isResizingSeg: false, // is a segment being resized? boolean
		isDraggingExternal: false, // jqui-dragging an external element? boolean
		segs: null, // the *event* segments currently rendered in the grid. TODO: rename to `eventSegs`


		// Renders the given events onto the grid
		renderEvents: function(events) {
			var bgEvents = [];
			var fgEvents = [];
			var i;

			for (i = 0; i < events.length; i++) {
				(isBgEvent(events[i]) ? bgEvents : fgEvents).push(events[i]);
			}

			this.segs = [].concat( // record all segs
				this.renderBgEvents(bgEvents),
				this.renderFgEvents(fgEvents)
			);
		},


		renderBgEvents: function(events) {
			var segs = this.eventsToSegs(events);

			// renderBgSegs might return a subset of segs, segs that were actually rendered
			return this.renderBgSegs(segs) || segs;
		},


		renderFgEvents: function(events) {
			var segs = this.eventsToSegs(events);

			// renderFgSegs might return a subset of segs, segs that were actually rendered
			return this.renderFgSegs(segs) || segs;
		},


		// Unrenders all events currently rendered on the grid
		unrenderEvents: function() {
			this.handleSegMouseout(); // trigger an eventMouseout if user's mouse is over an event
			this.clearDragListeners();

			this.unrenderFgSegs();
			this.unrenderBgSegs();

			this.segs = null;
		},


		// Retrieves all rendered segment objects currently rendered on the grid
		getEventSegs: function() {
			return this.segs || [];
		},


		/* Foreground Segment Rendering
	------------------------------------------------------------------------------------------------------------------*/


		// Renders foreground event segments onto the grid. May return a subset of segs that were rendered.
		renderFgSegs: function(segs) {
			// subclasses must implement
		},


		// Unrenders all currently rendered foreground segments
		unrenderFgSegs: function() {
			// subclasses must implement
		},


		// Renders and assigns an `el` property for each foreground event segment.
		// Only returns segments that successfully rendered.
		// A utility that subclasses may use.
		renderFgSegEls: function(segs, disableResizing) {
			var view = this.view;
			var html = '';
			var renderedSegs = [];
			var i;

			if (segs.length) { // don't build an empty html string

				// build a large concatenation of event segment HTML
				for (i = 0; i < segs.length; i++) {
					html += this.fgSegHtml(segs[i], disableResizing);
				}

				// Grab individual elements from the combined HTML string. Use each as the default rendering.
				// Then, compute the 'el' for each segment. An el might be null if the eventRender callback returned false.
				$(html).each(function(i, node) {
					var seg = segs[i];
					var el = view.resolveEventEl(seg.event, $(node));

					if (el) {
						el.data('fc-seg', seg); // used by handlers
						seg.el = el;
						renderedSegs.push(seg);
					}
				});
			}

			return renderedSegs;
		},


		// Generates the HTML for the default rendering of a foreground event segment. Used by renderFgSegEls()
		fgSegHtml: function(seg, disableResizing) {
			// subclasses should implement
		},


		/* Background Segment Rendering
	------------------------------------------------------------------------------------------------------------------*/


		// Renders the given background event segments onto the grid.
		// Returns a subset of the segs that were actually rendered.
		renderBgSegs: function(segs) {
			return this.renderFill('bgEvent', segs);
		},


		// Unrenders all the currently rendered background event segments
		unrenderBgSegs: function() {
			this.unrenderFill('bgEvent');
		},


		// Renders a background event element, given the default rendering. Called by the fill system.
		bgEventSegEl: function(seg, el) {
			return this.view.resolveEventEl(seg.event, el); // will filter through eventRender
		},


		// Generates an array of classNames to be used for the default rendering of a background event.
		// Called by fillSegHtml.
		bgEventSegClasses: function(seg) {
			var event = seg.event;
			var source = event.source || {};

			return [ 'fc-bgevent' ].concat(
				event.className,
				source.className || []
			);
		},


		// Generates a semicolon-separated CSS string to be used for the default rendering of a background event.
		// Called by fillSegHtml.
		bgEventSegCss: function(seg) {
			return {
				'background-color': this.getSegSkinCss(seg)['background-color']
			};
		},


		// Generates an array of classNames to be used for the rendering business hours overlay. Called by the fill system.
		// Called by fillSegHtml.
		businessHoursSegClasses: function(seg) {
			return [ 'fc-nonbusiness', 'fc-bgevent' ];
		},


		/* Business Hours
	------------------------------------------------------------------------------------------------------------------*/


		// Compute business hour segs for the grid's current date range.
		// Caller must ask if whole-day business hours are needed.
		// If no `businessHours` configuration value is specified, assumes the calendar default.
		buildBusinessHourSegs: function(wholeDay, businessHours) {
			return this.eventsToSegs(
				this.buildBusinessHourEvents(wholeDay, businessHours)
			);
		},


		// Compute business hour *events* for the grid's current date range.
		// Caller must ask if whole-day business hours are needed.
		// If no `businessHours` configuration value is specified, assumes the calendar default.
		buildBusinessHourEvents: function(wholeDay, businessHours) {
			var calendar = this.view.calendar;
			var events;

			if (businessHours == null) {
				// fallback
				// access from calendawr. don't access from view. doesn't update with dynamic options.
				businessHours = calendar.options.businessHours;
			}

			events = calendar.computeBusinessHourEvents(wholeDay, businessHours);

			// HACK. Eventually refactor business hours "events" system.
			// If no events are given, but businessHours is activated, this means the entire visible range should be
			// marked as *not* business-hours, via inverse-background rendering.
			if (!events.length && businessHours) {
				events = [
					$.extend({}, BUSINESS_HOUR_EVENT_DEFAULTS, {
						start: this.view.end, // guaranteed out-of-range
						end: this.view.end,   // "
						dow: null
					})
				];
			}

			return events;
		},


		/* Handlers
	------------------------------------------------------------------------------------------------------------------*/


		// Attaches event-element-related handlers for *all* rendered event segments of the view.
		bindSegHandlers: function() {
			this.bindSegHandlersToEl(this.el);
		},


		// Attaches event-element-related handlers to an arbitrary container element. leverages bubbling.
		bindSegHandlersToEl: function(el) {
			this.bindSegHandlerToEl(el, 'touchstart', this.handleSegTouchStart);
			this.bindSegHandlerToEl(el, 'touchend', this.handleSegTouchEnd);
			this.bindSegHandlerToEl(el, 'mouseenter', this.handleSegMouseover);
			this.bindSegHandlerToEl(el, 'mouseleave', this.handleSegMouseout);
			this.bindSegHandlerToEl(el, 'mousedown', this.handleSegMousedown);
			this.bindSegHandlerToEl(el, 'click', this.handleSegClick);
		},


		// Executes a handler for any a user-interaction on a segment.
		// Handler gets called with (seg, ev), and with the `this` context of the Grid
		bindSegHandlerToEl: function(el, name, handler) {
			var _this = this;

			el.on(name, this.segSelector, function(ev) {
				var seg = $(this).data('fc-seg'); // grab segment data. put there by View::renderEvents

				// only call the handlers if there is not a drag/resize in progress
				if (seg && !_this.isDraggingSeg && !_this.isResizingSeg) {
					return handler.call(_this, seg, ev); // context will be the Grid
				}
			});
		},


		handleSegClick: function(seg, ev) {
			var res = this.view.publiclyTrigger('eventClick', seg.el[0], seg.event, ev); // can return `false` to cancel
			if (res === false) {
				ev.preventDefault();
			}
		},


		// Updates internal state and triggers handlers for when an event element is moused over
		handleSegMouseover: function(seg, ev) {
			if (
				!this.isIgnoringMouse &&
				!this.mousedOverSeg
			) {
				this.mousedOverSeg = seg;
				if (this.view.isEventResizable(seg.event)) {
					seg.el.addClass('fc-allow-mouse-resize');
				}
				this.view.publiclyTrigger('eventMouseover', seg.el[0], seg.event, ev);
			}
		},


		// Updates internal state and triggers handlers for when an event element is moused out.
		// Can be given no arguments, in which case it will mouseout the segment that was previously moused over.
		handleSegMouseout: function(seg, ev) {
			ev = ev || {}; // if given no args, make a mock mouse event

			if (this.mousedOverSeg) {
				seg = seg || this.mousedOverSeg; // if given no args, use the currently moused-over segment
				this.mousedOverSeg = null;
				if (this.view.isEventResizable(seg.event)) {
					seg.el.removeClass('fc-allow-mouse-resize');
				}
				this.view.publiclyTrigger('eventMouseout', seg.el[0], seg.event, ev);
			}
		},


		handleSegMousedown: function(seg, ev) {
			var isResizing = this.startSegResize(seg, ev, { distance: 5 });

			if (!isResizing && this.view.isEventDraggable(seg.event)) {
				this.buildSegDragListener(seg)
					.startInteraction(ev, {
						distance: 5
					});
			}
		},


		handleSegTouchStart: function(seg, ev) {
			var view = this.view;
			var event = seg.event;
			var isSelected = view.isEventSelected(event);
			var isDraggable = view.isEventDraggable(event);
			var isResizable = view.isEventResizable(event);
			var isResizing = false;
			var dragListener;
			var eventLongPressDelay;

			if (isSelected && isResizable) {
				// only allow resizing of the event is selected
				isResizing = this.startSegResize(seg, ev);
			}

			if (!isResizing && (isDraggable || isResizable)) { // allowed to be selected?

				eventLongPressDelay = view.opt('eventLongPressDelay');
				if (eventLongPressDelay == null) {
					eventLongPressDelay = view.opt('longPressDelay'); // fallback
				}

				dragListener = isDraggable ?
					this.buildSegDragListener(seg) :
					this.buildSegSelectListener(seg); // seg isn't draggable, but still needs to be selected

				dragListener.startInteraction(ev, { // won't start if already started
					delay: isSelected ? 0 : eventLongPressDelay // do delay if not already selected
				});
			}

			// a long tap simulates a mouseover. ignore this bogus mouseover.
			this.tempIgnoreMouse();
		},


		handleSegTouchEnd: function(seg, ev) {
			// touchstart+touchend = click, which simulates a mouseover.
			// ignore this bogus mouseover.
			this.tempIgnoreMouse();
		},


		// returns boolean whether resizing actually started or not.
		// assumes the seg allows resizing.
		// `dragOptions` are optional.
		startSegResize: function(seg, ev, dragOptions) {
			if ($(ev.target).is('.fc-resizer')) {
				this.buildSegResizeListener(seg, $(ev.target).is('.fc-start-resizer'))
					.startInteraction(ev, dragOptions);
				return true;
			}
			return false;
		},



		/* Event Dragging
	------------------------------------------------------------------------------------------------------------------*/


		// Builds a listener that will track user-dragging on an event segment.
		// Generic enough to work with any type of Grid.
		// Has side effect of setting/unsetting `segDragListener`
		buildSegDragListener: function(seg) {
			var _this = this;
			var view = this.view;
			var calendar = view.calendar;
			var el = seg.el;
			var event = seg.event;
			var isDragging;
			var mouseFollower; // A clone of the original element that will move with the mouse
			var dropLocation; // zoned event date properties

			if (this.segDragListener) {
				return this.segDragListener;
			}

			// Tracks mouse movement over the *view's* coordinate map. Allows dragging and dropping between subcomponents
			// of the view.
			var dragListener = this.segDragListener = new HitDragListener(view, {
				scroll: view.opt('dragScroll'),
				subjectEl: el,
				subjectCenter: true,
				interactionStart: function(ev) {
					seg.component = _this; // for renderDrag
					isDragging = false;
					mouseFollower = new MouseFollower(seg.el, {
						additionalClass: 'fc-dragging',
						parentEl: view.el,
						opacity: dragListener.isTouch ? null : view.opt('dragOpacity'),
						revertDuration: view.opt('dragRevertDuration'),
						zIndex: 2 // one above the .fc-view
					});
					mouseFollower.hide(); // don't show until we know this is a real drag
					mouseFollower.start(ev);
				},
				dragStart: function(ev) {
					if (dragListener.isTouch && !view.isEventSelected(event)) {
						// if not previously selected, will fire after a delay. then, select the event
						view.selectEvent(event);
					}
					isDragging = true;
					_this.handleSegMouseout(seg, ev); // ensure a mouseout on the manipulated event has been reported
					_this.segDragStart(seg, ev);
					view.hideEvent(event); // hide all event segments. our mouseFollower will take over
				},
				hitOver: function(hit, isOrig, origHit) {
					var dragHelperEls;

					// starting hit could be forced (DayGrid.limit)
					if (seg.hit) {
						origHit = seg.hit;
					}

					// since we are querying the parent view, might not belong to this grid
					dropLocation = _this.computeEventDrop(
						origHit.component.getHitSpan(origHit),
						hit.component.getHitSpan(hit),
						event
					);

					if (dropLocation && !calendar.isEventSpanAllowed(_this.eventToSpan(dropLocation), event)) {
						disableCursor();
						dropLocation = null;
					}

					// if a valid drop location, have the subclass render a visual indication
					if (dropLocation && (dragHelperEls = view.renderDrag(dropLocation, seg))) {

						dragHelperEls.addClass('fc-dragging');
						if (!dragListener.isTouch) {
							_this.applyDragOpacity(dragHelperEls);
						}

						mouseFollower.hide(); // if the subclass is already using a mock event "helper", hide our own
					}
					else {
						mouseFollower.show(); // otherwise, have the helper follow the mouse (no snapping)
					}

					if (isOrig) {
						dropLocation = null; // needs to have moved hits to be a valid drop
					}
				},
				hitOut: function() { // called before mouse moves to a different hit OR moved out of all hits
					view.unrenderDrag(); // unrender whatever was done in renderDrag
					mouseFollower.show(); // show in case we are moving out of all hits
					dropLocation = null;
				},
				hitDone: function() { // Called after a hitOut OR before a dragEnd
					enableCursor();
				},
				interactionEnd: function(ev) {
					delete seg.component; // prevent side effects

					// do revert animation if hasn't changed. calls a callback when finished (whether animation or not)
					mouseFollower.stop(!dropLocation, function() {
						if (isDragging) {
							view.unrenderDrag();
							_this.segDragStop(seg, ev);
						}

						if (dropLocation) {
							// no need to re-show original, will rerender all anyways. esp important if eventRenderWait
							view.reportEventDrop(event, dropLocation, _this.largeUnit, el, ev);
						}
						else {
							view.showEvent(event);
						}
					});
					_this.segDragListener = null;
				}
			});

			return dragListener;
		},


		// seg isn't draggable, but let's use a generic DragListener
		// simply for the delay, so it can be selected.
		// Has side effect of setting/unsetting `segDragListener`
		buildSegSelectListener: function(seg) {
			var _this = this;
			var view = this.view;
			var event = seg.event;

			if (this.segDragListener) {
				return this.segDragListener;
			}

			var dragListener = this.segDragListener = new DragListener({
				dragStart: function(ev) {
					if (dragListener.isTouch && !view.isEventSelected(event)) {
						// if not previously selected, will fire after a delay. then, select the event
						view.selectEvent(event);
					}
				},
				interactionEnd: function(ev) {
					_this.segDragListener = null;
				}
			});

			return dragListener;
		},


		// Called before event segment dragging starts
		segDragStart: function(seg, ev) {
			this.isDraggingSeg = true;
			this.view.publiclyTrigger('eventDragStart', seg.el[0], seg.event, ev, {}); // last argument is jqui dummy
		},


		// Called after event segment dragging stops
		segDragStop: function(seg, ev) {
			this.isDraggingSeg = false;
			this.view.publiclyTrigger('eventDragStop', seg.el[0], seg.event, ev, {}); // last argument is jqui dummy
		},


		// Given the spans an event drag began, and the span event was dropped, calculates the new zoned start/end/allDay
		// values for the event. Subclasses may override and set additional properties to be used by renderDrag.
		// A falsy returned value indicates an invalid drop.
		// DOES NOT consider overlap/constraint.
		computeEventDrop: function(startSpan, endSpan, event) {
			var calendar = this.view.calendar;
			var dragStart = startSpan.start;
			var dragEnd = endSpan.start;
			var delta;
			var dropLocation; // zoned event date properties

			if (dragStart.hasTime() === dragEnd.hasTime()) {
				delta = this.diffDates(dragEnd, dragStart);

				// if an all-day event was in a timed area and it was dragged to a different time,
				// guarantee an end and adjust start/end to have times
				if (event.allDay && durationHasTime(delta)) {
					dropLocation = {
						start: event.start.clone(),
						end: calendar.getEventEnd(event), // will be an ambig day
						allDay: false // for normalizeEventTimes
					};
					calendar.normalizeEventTimes(dropLocation);
				}
				// othewise, work off existing values
				else {
					dropLocation = pluckEventDateProps(event);
				}

				dropLocation.start.add(delta);
				if (dropLocation.end) {
					dropLocation.end.add(delta);
				}
			}
			else {
				// if switching from day <-> timed, start should be reset to the dropped date, and the end cleared
				dropLocation = {
					start: dragEnd.clone(),
					end: null, // end should be cleared
					allDay: !dragEnd.hasTime()
				};
			}

			return dropLocation;
		},


		// Utility for apply dragOpacity to a jQuery set
		applyDragOpacity: function(els) {
			var opacity = this.view.opt('dragOpacity');

			if (opacity != null) {
				els.css('opacity', opacity);
			}
		},


		/* External Element Dragging
	------------------------------------------------------------------------------------------------------------------*/


		// Called when a jQuery UI drag is initiated anywhere in the DOM
		externalDragStart: function(ev, ui) {
			var view = this.view;
			var el;
			var accept;

			if (view.opt('droppable')) { // only listen if this setting is on
				el = $((ui ? ui.item : null) || ev.target);

				// Test that the dragged element passes the dropAccept selector or filter function.
				// FYI, the default is "*" (matches all)
				accept = view.opt('dropAccept');
				if ($.isFunction(accept) ? accept.call(el[0], el) : el.is(accept)) {
					if (!this.isDraggingExternal) { // prevent double-listening if fired twice
						this.listenToExternalDrag(el, ev, ui);
					}
				}
			}
		},


		// Called when a jQuery UI drag starts and it needs to be monitored for dropping
		listenToExternalDrag: function(el, ev, ui) {
			var _this = this;
			var calendar = this.view.calendar;
			var meta = getDraggedElMeta(el); // extra data about event drop, including possible event to create
			var dropLocation; // a null value signals an unsuccessful drag

			// listener that tracks mouse movement over date-associated pixel regions
			var dragListener = _this.externalDragListener = new HitDragListener(this, {
				interactionStart: function() {
					_this.isDraggingExternal = true;
				},
				hitOver: function(hit) {
					dropLocation = _this.computeExternalDrop(
						hit.component.getHitSpan(hit), // since we are querying the parent view, might not belong to this grid
						meta
					);

					if ( // invalid hit?
						dropLocation &&
						!calendar.isExternalSpanAllowed(_this.eventToSpan(dropLocation), dropLocation, meta.eventProps)
					) {
						disableCursor();
						dropLocation = null;
					}

					if (dropLocation) {
						_this.renderDrag(dropLocation); // called without a seg parameter
					}
				},
				hitOut: function() {
					dropLocation = null; // signal unsuccessful
				},
				hitDone: function() { // Called after a hitOut OR before a dragEnd
					enableCursor();
					_this.unrenderDrag();
				},
				interactionEnd: function(ev) {
					if (dropLocation) { // element was dropped on a valid hit
						_this.view.reportExternalDrop(meta, dropLocation, el, ev, ui);
					}
					_this.isDraggingExternal = false;
					_this.externalDragListener = null;
				}
			});

			dragListener.startDrag(ev); // start listening immediately
		},


		// Given a hit to be dropped upon, and misc data associated with the jqui drag (guaranteed to be a plain object),
		// returns the zoned start/end dates for the event that would result from the hypothetical drop. end might be null.
		// Returning a null value signals an invalid drop hit.
		// DOES NOT consider overlap/constraint.
		computeExternalDrop: function(span, meta) {
			var calendar = this.view.calendar;
			var dropLocation = {
				start: calendar.applyTimezone(span.start), // simulate a zoned event start date
				end: null
			};

			// if dropped on an all-day span, and element's metadata specified a time, set it
			if (meta.startTime && !dropLocation.start.hasTime()) {
				dropLocation.start.time(meta.startTime);
			}

			if (meta.duration) {
				dropLocation.end = dropLocation.start.clone().add(meta.duration);
			}

			return dropLocation;
		},



		/* Drag Rendering (for both events and an external elements)
	------------------------------------------------------------------------------------------------------------------*/


		// Renders a visual indication of an event or external element being dragged.
		// `dropLocation` contains hypothetical start/end/allDay values the event would have if dropped. end can be null.
		// `seg` is the internal segment object that is being dragged. If dragging an external element, `seg` is null.
		// A truthy returned value indicates this method has rendered a helper element.
		// Must return elements used for any mock events.
		renderDrag: function(dropLocation, seg) {
			// subclasses must implement
		},


		// Unrenders a visual indication of an event or external element being dragged
		unrenderDrag: function() {
			// subclasses must implement
		},


		/* Resizing
	------------------------------------------------------------------------------------------------------------------*/


		// Creates a listener that tracks the user as they resize an event segment.
		// Generic enough to work with any type of Grid.
		buildSegResizeListener: function(seg, isStart) {
			var _this = this;
			var view = this.view;
			var calendar = view.calendar;
			var el = seg.el;
			var event = seg.event;
			var eventEnd = calendar.getEventEnd(event);
			var isDragging;
			var resizeLocation; // zoned event date properties. falsy if invalid resize

			// Tracks mouse movement over the *grid's* coordinate map
			var dragListener = this.segResizeListener = new HitDragListener(this, {
				scroll: view.opt('dragScroll'),
				subjectEl: el,
				interactionStart: function() {
					isDragging = false;
				},
				dragStart: function(ev) {
					isDragging = true;
					_this.handleSegMouseout(seg, ev); // ensure a mouseout on the manipulated event has been reported
					_this.segResizeStart(seg, ev);
				},
				hitOver: function(hit, isOrig, origHit) {
					var origHitSpan = _this.getHitSpan(origHit);
					var hitSpan = _this.getHitSpan(hit);

					resizeLocation = isStart ?
						_this.computeEventStartResize(origHitSpan, hitSpan, event) :
						_this.computeEventEndResize(origHitSpan, hitSpan, event);

					if (resizeLocation) {
						if (!calendar.isEventSpanAllowed(_this.eventToSpan(resizeLocation), event)) {
							disableCursor();
							resizeLocation = null;
						}
						// no change? (FYI, event dates might have zones)
						else if (
							resizeLocation.start.isSame(event.start.clone().stripZone()) &&
							resizeLocation.end.isSame(eventEnd.clone().stripZone())
						) {
							resizeLocation = null;
						}
					}

					if (resizeLocation) {
						view.hideEvent(event);
						_this.renderEventResize(resizeLocation, seg);
					}
				},
				hitOut: function() { // called before mouse moves to a different hit OR moved out of all hits
					resizeLocation = null;
					view.showEvent(event); // for when out-of-bounds. show original
				},
				hitDone: function() { // resets the rendering to show the original event
					_this.unrenderEventResize();
					enableCursor();
				},
				interactionEnd: function(ev) {
					if (isDragging) {
						_this.segResizeStop(seg, ev);
					}

					if (resizeLocation) { // valid date to resize to?
						// no need to re-show original, will rerender all anyways. esp important if eventRenderWait
						view.reportEventResize(event, resizeLocation, _this.largeUnit, el, ev);
					}
					else {
						view.showEvent(event);
					}
					_this.segResizeListener = null;
				}
			});

			return dragListener;
		},


		// Called before event segment resizing starts
		segResizeStart: function(seg, ev) {
			this.isResizingSeg = true;
			this.view.publiclyTrigger('eventResizeStart', seg.el[0], seg.event, ev, {}); // last argument is jqui dummy
		},


		// Called after event segment resizing stops
		segResizeStop: function(seg, ev) {
			this.isResizingSeg = false;
			this.view.publiclyTrigger('eventResizeStop', seg.el[0], seg.event, ev, {}); // last argument is jqui dummy
		},


		// Returns new date-information for an event segment being resized from its start
		computeEventStartResize: function(startSpan, endSpan, event) {
			return this.computeEventResize('start', startSpan, endSpan, event);
		},


		// Returns new date-information for an event segment being resized from its end
		computeEventEndResize: function(startSpan, endSpan, event) {
			return this.computeEventResize('end', startSpan, endSpan, event);
		},


		// Returns new zoned date information for an event segment being resized from its start OR end
		// `type` is either 'start' or 'end'.
		// DOES NOT consider overlap/constraint.
		computeEventResize: function(type, startSpan, endSpan, event) {
			var calendar = this.view.calendar;
			var delta = this.diffDates(endSpan[type], startSpan[type]);
			var resizeLocation; // zoned event date properties
			var defaultDuration;

			// build original values to work from, guaranteeing a start and end
			resizeLocation = {
				start: event.start.clone(),
				end: calendar.getEventEnd(event),
				allDay: event.allDay
			};

			// if an all-day event was in a timed area and was resized to a time, adjust start/end to have times
			if (resizeLocation.allDay && durationHasTime(delta)) {
				resizeLocation.allDay = false;
				calendar.normalizeEventTimes(resizeLocation);
			}

			resizeLocation[type].add(delta); // apply delta to start or end

			// if the event was compressed too small, find a new reasonable duration for it
			if (!resizeLocation.start.isBefore(resizeLocation.end)) {

				defaultDuration =
					this.minResizeDuration || // TODO: hack
					(event.allDay ?
						calendar.defaultAllDayEventDuration :
						calendar.defaultTimedEventDuration);

				if (type == 'start') { // resizing the start?
					resizeLocation.start = resizeLocation.end.clone().subtract(defaultDuration);
				}
				else { // resizing the end?
					resizeLocation.end = resizeLocation.start.clone().add(defaultDuration);
				}
			}

			return resizeLocation;
		},


		// Renders a visual indication of an event being resized.
		// `range` has the updated dates of the event. `seg` is the original segment object involved in the drag.
		// Must return elements used for any mock events.
		renderEventResize: function(range, seg) {
			// subclasses must implement
		},


		// Unrenders a visual indication of an event being resized.
		unrenderEventResize: function() {
			// subclasses must implement
		},


		/* Rendering Utils
	------------------------------------------------------------------------------------------------------------------*/


		// Compute the text that should be displayed on an event's element.
		// `range` can be the Event object itself, or something range-like, with at least a `start`.
		// If event times are disabled, or the event has no time, will return a blank string.
		// If not specified, formatStr will default to the eventTimeFormat setting,
		// and displayEnd will default to the displayEventEnd setting.
		getEventTimeText: function(range, formatStr, displayEnd) {

			if (formatStr == null) {
				formatStr = this.eventTimeFormat;
			}

			if (displayEnd == null) {
				displayEnd = this.displayEventEnd;
			}

			if (this.displayEventTime && range.start.hasTime()) {
				if (displayEnd && range.end) {
					return this.view.formatRange(range, formatStr);
				}
				else {
					return range.start.format(formatStr);
				}
			}

			return '';
		},


		// Generic utility for generating the HTML classNames for an event segment's element
		getSegClasses: function(seg, isDraggable, isResizable) {
			var view = this.view;
			var classes = [
				'fc-event',
				seg.isStart ? 'fc-start' : 'fc-not-start',
				seg.isEnd ? 'fc-end' : 'fc-not-end'
			].concat(this.getSegCustomClasses(seg));

			if (isDraggable) {
				classes.push('fc-draggable');
			}
			if (isResizable) {
				classes.push('fc-resizable');
			}

			// event is currently selected? attach a className.
			if (view.isEventSelected(seg.event)) {
				classes.push('fc-selected');
			}

			return classes;
		},


		// List of classes that were defined by the caller of the API in some way
		getSegCustomClasses: function(seg) {
			var event = seg.event;

			return [].concat(
				event.className, // guaranteed to be an array
				event.source ? event.source.className : []
			);
		},


		// Utility for generating event skin-related CSS properties
		getSegSkinCss: function(seg) {
			return {
				'background-color': this.getSegBackgroundColor(seg),
				'border-color': this.getSegBorderColor(seg),
				color: this.getSegTextColor(seg)
			};
		},


		// Queries for caller-specified color, then falls back to default
		getSegBackgroundColor: function(seg) {
			return seg.event.backgroundColor ||
				seg.event.color ||
				this.getSegDefaultBackgroundColor(seg);
		},


		getSegDefaultBackgroundColor: function(seg) {
			var source = seg.event.source || {};

			return source.backgroundColor ||
				source.color ||
				this.view.opt('eventBackgroundColor') ||
				this.view.opt('eventColor');
		},


		// Queries for caller-specified color, then falls back to default
		getSegBorderColor: function(seg) {
			return seg.event.borderColor ||
				seg.event.color ||
				this.getSegDefaultBorderColor(seg);
		},


		getSegDefaultBorderColor: function(seg) {
			var source = seg.event.source || {};

			return source.borderColor ||
				source.color ||
				this.view.opt('eventBorderColor') ||
				this.view.opt('eventColor');
		},


		// Queries for caller-specified color, then falls back to default
		getSegTextColor: function(seg) {
			return seg.event.textColor ||
				this.getSegDefaultTextColor(seg);
		},


		getSegDefaultTextColor: function(seg) {
			var source = seg.event.source || {};

			return source.textColor ||
				this.view.opt('eventTextColor');
		},


		/* Converting events -> eventRange -> eventSpan -> eventSegs
	------------------------------------------------------------------------------------------------------------------*/


		// Generates an array of segments for the given single event
		// Can accept an event "location" as well (which only has start/end and no allDay)
		eventToSegs: function(event) {
			return this.eventsToSegs([ event ]);
		},


		eventToSpan: function(event) {
			return this.eventToSpans(event)[0];
		},


		// Generates spans (always unzoned) for the given event.
		// Does not do any inverting for inverse-background events.
		// Can accept an event "location" as well (which only has start/end and no allDay)
		eventToSpans: function(event) {
			var range = this.eventToRange(event);
			return this.eventRangeToSpans(range, event);
		},



		// Converts an array of event objects into an array of event segment objects.
		// A custom `segSliceFunc` may be given for arbitrarily slicing up events.
		// Doesn't guarantee an order for the resulting array.
		eventsToSegs: function(allEvents, segSliceFunc) {
			var _this = this;
			var eventsById = groupEventsById(allEvents);
			var segs = [];

			$.each(eventsById, function(id, events) {
				var ranges = [];
				var i;

				for (i = 0; i < events.length; i++) {
					ranges.push(_this.eventToRange(events[i]));
				}

				// inverse-background events (utilize only the first event in calculations)
				if (isInverseBgEvent(events[0])) {
					ranges = _this.invertRanges(ranges);

					for (i = 0; i < ranges.length; i++) {
						segs.push.apply(segs, // append to
							_this.eventRangeToSegs(ranges[i], events[0], segSliceFunc));
					}
				}
				// normal event ranges
				else {
					for (i = 0; i < ranges.length; i++) {
						segs.push.apply(segs, // append to
							_this.eventRangeToSegs(ranges[i], events[i], segSliceFunc));
					}
				}
			});

			return segs;
		},


		// Generates the unzoned start/end dates an event appears to occupy
		// Can accept an event "location" as well (which only has start/end and no allDay)
		eventToRange: function(event) {
			var calendar = this.view.calendar;
			var start = event.start.clone().stripZone();
			var end = (
				event.end ?
					event.end.clone() :
					// derive the end from the start and allDay. compute allDay if necessary
					calendar.getDefaultEventEnd(
						event.allDay != null ?
							event.allDay :
							!event.start.hasTime(),
						event.start
					)
			).stripZone();

			// hack: dynamic locale change forgets to upate stored event localed
			calendar.localizeMoment(start);
			calendar.localizeMoment(end);

			return { start: start, end: end };
		},


		// Given an event's range (unzoned start/end), and the event itself,
		// slice into segments (using the segSliceFunc function if specified)
		eventRangeToSegs: function(range, event, segSliceFunc) {
			var spans = this.eventRangeToSpans(range, event);
			var segs = [];
			var i;

			for (i = 0; i < spans.length; i++) {
				segs.push.apply(segs, // append to
					this.eventSpanToSegs(spans[i], event, segSliceFunc));
			}

			return segs;
		},


		// Given an event's unzoned date range, return an array of "span" objects.
		// Subclasses can override.
		eventRangeToSpans: function(range, event) {
			return [ $.extend({}, range) ]; // copy into a single-item array
		},


		// Given an event's span (unzoned start/end and other misc data), and the event itself,
		// slices into segments and attaches event-derived properties to them.
		eventSpanToSegs: function(span, event, segSliceFunc) {
			var segs = segSliceFunc ? segSliceFunc(span) : this.spanToSegs(span);
			var i, seg;

			for (i = 0; i < segs.length; i++) {
				seg = segs[i];
				seg.event = event;
				seg.eventStartMS = +span.start; // TODO: not the best name after making spans unzoned
				seg.eventDurationMS = span.end - span.start;
			}

			return segs;
		},


		// Produces a new array of range objects that will cover all the time NOT covered by the given ranges.
		// SIDE EFFECT: will mutate the given array and will use its date references.
		invertRanges: function(ranges) {
			var view = this.view;
			var viewStart = view.start.clone(); // need a copy
			var viewEnd = view.end.clone(); // need a copy
			var inverseRanges = [];
			var start = viewStart; // the end of the previous range. the start of the new range
			var i, range;

			// ranges need to be in order. required for our date-walking algorithm
			ranges.sort(compareRanges);

			for (i = 0; i < ranges.length; i++) {
				range = ranges[i];

				// add the span of time before the event (if there is any)
				if (range.start > start) { // compare millisecond time (skip any ambig logic)
					inverseRanges.push({
						start: start,
						end: range.start
					});
				}

				start = range.end;
			}

			// add the span of time after the last event (if there is any)
			if (start < viewEnd) { // compare millisecond time (skip any ambig logic)
				inverseRanges.push({
					start: start,
					end: viewEnd
				});
			}

			return inverseRanges;
		},


		sortEventSegs: function(segs) {
			segs.sort(proxy(this, 'compareEventSegs'));
		},


		// A cmp function for determining which segments should take visual priority
		compareEventSegs: function(seg1, seg2) {
			return seg1.eventStartMS - seg2.eventStartMS || // earlier events go first
				seg2.eventDurationMS - seg1.eventDurationMS || // tie? longer events go first
				seg2.event.allDay - seg1.event.allDay || // tie? put all-day events first (booleans cast to 0/1)
				compareByFieldSpecs(seg1.event, seg2.event, this.view.eventOrderSpecs);
		}

	});


	/* Utilities
----------------------------------------------------------------------------------------------------------------------*/


	function pluckEventDateProps(event) {
		return {
			start: event.start.clone(),
			end: event.end ? event.end.clone() : null,
			allDay: event.allDay // keep it the same
		};
	}
	FC.pluckEventDateProps = pluckEventDateProps;


	function isBgEvent(event) { // returns true if background OR inverse-background
		var rendering = getEventRendering(event);
		return rendering === 'background' || rendering === 'inverse-background';
	}
	FC.isBgEvent = isBgEvent; // export


	function isInverseBgEvent(event) {
		return getEventRendering(event) === 'inverse-background';
	}


	function getEventRendering(event) {
		return firstDefined((event.source || {}).rendering, event.rendering);
	}


	function groupEventsById(events) {
		var eventsById = {};
		var i, event;

		for (i = 0; i < events.length; i++) {
			event = events[i];
			(eventsById[event._id] || (eventsById[event._id] = [])).push(event);
		}

		return eventsById;
	}


// A cmp function for determining which non-inverted "ranges" (see above) happen earlier
	function compareRanges(range1, range2) {
		return range1.start - range2.start; // earlier ranges go first
	}


	/* External-Dragging-Element Data
----------------------------------------------------------------------------------------------------------------------*/

// Require all HTML5 data-* attributes used by FullCalendar to have this prefix.
// A value of '' will query attributes like data-event. A value of 'fc' will query attributes like data-fc-event.
	FC.dataAttrPrefix = '';

// Given a jQuery element that might represent a dragged FullCalendar event, returns an intermediate data structure
// to be used for Event Object creation.
// A defined `.eventProps`, even when empty, indicates that an event should be created.
	function getDraggedElMeta(el) {
		var prefix = FC.dataAttrPrefix;
		var eventProps; // properties for creating the event, not related to date/time
		var startTime; // a Duration
		var duration;
		var stick;

		if (prefix) { prefix += '-'; }
		eventProps = el.data(prefix + 'event') || null;

		if (eventProps) {
			if (typeof eventProps === 'object') {
				eventProps = $.extend({}, eventProps); // make a copy
			}
			else { // something like 1 or true. still signal event creation
				eventProps = {};
			}

			// pluck special-cased date/time properties
			startTime = eventProps.start;
			if (startTime == null) { startTime = eventProps.time; } // accept 'time' as well
			duration = eventProps.duration;
			stick = eventProps.stick;
			delete eventProps.start;
			delete eventProps.time;
			delete eventProps.duration;
			delete eventProps.stick;
		}

		// fallback to standalone attribute values for each of the date/time properties
		if (startTime == null) { startTime = el.data(prefix + 'start'); }
		if (startTime == null) { startTime = el.data(prefix + 'time'); } // accept 'time' as well
		if (duration == null) { duration = el.data(prefix + 'duration'); }
		if (stick == null) { stick = el.data(prefix + 'stick'); }

		// massage into correct data types
		startTime = startTime != null ? moment.duration(startTime) : null;
		duration = duration != null ? moment.duration(duration) : null;
		stick = Boolean(stick);

		return { eventProps: eventProps, startTime: startTime, duration: duration, stick: stick };
	}


	;;

	/*
A set of rendering and date-related methods for a visual component comprised of one or more rows of day columns.
Prerequisite: the object being mixed into needs to be a *Grid*
*/
	var DayTableMixin = FC.DayTableMixin = {

		breakOnWeeks: false, // should create a new row for each week?
		dayDates: null, // whole-day dates for each column. left to right
		dayIndices: null, // for each day from start, the offset
		daysPerRow: null,
		rowCnt: null,
		colCnt: null,
		colHeadFormat: null,


		// Populates internal variables used for date calculation and rendering
		updateDayTable: function() {
			var view = this.view;
			var date = this.start.clone();
			var dayIndex = -1;
			var dayIndices = [];
			var dayDates = [];
			var daysPerRow;
			var firstDay;
			var rowCnt;

			while (date.isBefore(this.end)) { // loop each day from start to end
				if (view.isHiddenDay(date)) {
					dayIndices.push(dayIndex + 0.5); // mark that it's between indices
				}
				else {
					dayIndex++;
					dayIndices.push(dayIndex);
					dayDates.push(date.clone());
				}
				date.add(1, 'days');
			}

			if (this.breakOnWeeks) {
				// count columns until the day-of-week repeats
				firstDay = dayDates[0].day();
				for (daysPerRow = 1; daysPerRow < dayDates.length; daysPerRow++) {
					if (dayDates[daysPerRow].day() == firstDay) {
						break;
					}
				}
				rowCnt = Math.ceil(dayDates.length / daysPerRow);
			}
			else {
				rowCnt = 1;
				daysPerRow = dayDates.length;
			}

			this.dayDates = dayDates;
			this.dayIndices = dayIndices;
			this.daysPerRow = daysPerRow;
			this.rowCnt = rowCnt;

			this.updateDayTableCols();
		},


		// Computes and assigned the colCnt property and updates any options that may be computed from it
		updateDayTableCols: function() {
			this.colCnt = this.computeColCnt();
			this.colHeadFormat = this.view.opt('columnFormat') || this.computeColHeadFormat();
		},


		// Determines how many columns there should be in the table
		computeColCnt: function() {
			return this.daysPerRow;
		},


		// Computes the ambiguously-timed moment for the given cell
		getCellDate: function(row, col) {
			return this.dayDates[
				this.getCellDayIndex(row, col)
				].clone();
		},


		// Computes the ambiguously-timed date range for the given cell
		getCellRange: function(row, col) {
			var start = this.getCellDate(row, col);
			var end = start.clone().add(1, 'days');

			return { start: start, end: end };
		},


		// Returns the number of day cells, chronologically, from the first of the grid (0-based)
		getCellDayIndex: function(row, col) {
			return row * this.daysPerRow + this.getColDayIndex(col);
		},


		// Returns the numner of day cells, chronologically, from the first cell in *any given row*
		getColDayIndex: function(col) {
			if (this.isRTL) {
				return this.colCnt - 1 - col;
			}
			else {
				return col;
			}
		},


		// Given a date, returns its chronolocial cell-index from the first cell of the grid.
		// If the date lies between cells (because of hiddenDays), returns a floating-point value between offsets.
		// If before the first offset, returns a negative number.
		// If after the last offset, returns an offset past the last cell offset.
		// Only works for *start* dates of cells. Will not work for exclusive end dates for cells.
		getDateDayIndex: function(date) {
			var dayIndices = this.dayIndices;
			var dayOffset = date.diff(this.start, 'days');

			if (dayOffset < 0) {
				return dayIndices[0] - 1;
			}
			else if (dayOffset >= dayIndices.length) {
				return dayIndices[dayIndices.length - 1] + 1;
			}
			else {
				return dayIndices[dayOffset];
			}
		},


		/* Options
	------------------------------------------------------------------------------------------------------------------*/


		// Computes a default column header formatting string if `colFormat` is not explicitly defined
		computeColHeadFormat: function() {
			// if more than one week row, or if there are a lot of columns with not much space,
			// put just the day numbers will be in each cell
			if (this.rowCnt > 1 || this.colCnt > 10) {
				return 'ddd'; // "Sat"
			}
			// multiple days, so full single date string WON'T be in title text
			else if (this.colCnt > 1) {
				return this.view.opt('dayOfMonthFormat'); // "Sat 12/10"
			}
			// single day, so full single date string will probably be in title text
			else {
				return 'dddd'; // "Saturday"
			}
		},


		/* Slicing
	------------------------------------------------------------------------------------------------------------------*/


		// Slices up a date range into a segment for every week-row it intersects with
		sliceRangeByRow: function(range) {
			var daysPerRow = this.daysPerRow;
			var normalRange = this.view.computeDayRange(range); // make whole-day range, considering nextDayThreshold
			var rangeFirst = this.getDateDayIndex(normalRange.start); // inclusive first index
			var rangeLast = this.getDateDayIndex(normalRange.end.clone().subtract(1, 'days')); // inclusive last index
			var segs = [];
			var row;
			var rowFirst, rowLast; // inclusive day-index range for current row
			var segFirst, segLast; // inclusive day-index range for segment

			for (row = 0; row < this.rowCnt; row++) {
				rowFirst = row * daysPerRow;
				rowLast = rowFirst + daysPerRow - 1;

				// intersect segment's offset range with the row's
				segFirst = Math.max(rangeFirst, rowFirst);
				segLast = Math.min(rangeLast, rowLast);

				// deal with in-between indices
				segFirst = Math.ceil(segFirst); // in-between starts round to next cell
				segLast = Math.floor(segLast); // in-between ends round to prev cell

				if (segFirst <= segLast) { // was there any intersection with the current row?
					segs.push({
						row: row,

						// normalize to start of row
						firstRowDayIndex: segFirst - rowFirst,
						lastRowDayIndex: segLast - rowFirst,

						// must be matching integers to be the segment's start/end
						isStart: segFirst === rangeFirst,
						isEnd: segLast === rangeLast
					});
				}
			}

			return segs;
		},


		// Slices up a date range into a segment for every day-cell it intersects with.
		// TODO: make more DRY with sliceRangeByRow somehow.
		sliceRangeByDay: function(range) {
			var daysPerRow = this.daysPerRow;
			var normalRange = this.view.computeDayRange(range); // make whole-day range, considering nextDayThreshold
			var rangeFirst = this.getDateDayIndex(normalRange.start); // inclusive first index
			var rangeLast = this.getDateDayIndex(normalRange.end.clone().subtract(1, 'days')); // inclusive last index
			var segs = [];
			var row;
			var rowFirst, rowLast; // inclusive day-index range for current row
			var i;
			var segFirst, segLast; // inclusive day-index range for segment

			for (row = 0; row < this.rowCnt; row++) {
				rowFirst = row * daysPerRow;
				rowLast = rowFirst + daysPerRow - 1;

				for (i = rowFirst; i <= rowLast; i++) {

					// intersect segment's offset range with the row's
					segFirst = Math.max(rangeFirst, i);
					segLast = Math.min(rangeLast, i);

					// deal with in-between indices
					segFirst = Math.ceil(segFirst); // in-between starts round to next cell
					segLast = Math.floor(segLast); // in-between ends round to prev cell

					if (segFirst <= segLast) { // was there any intersection with the current row?
						segs.push({
							row: row,

							// normalize to start of row
							firstRowDayIndex: segFirst - rowFirst,
							lastRowDayIndex: segLast - rowFirst,

							// must be matching integers to be the segment's start/end
							isStart: segFirst === rangeFirst,
							isEnd: segLast === rangeLast
						});
					}
				}
			}

			return segs;
		},


		/* Header Rendering
	------------------------------------------------------------------------------------------------------------------*/


		renderHeadHtml: function() {
			var view = this.view;

			return '' +
				'<div class="fc-row ' + view.widgetHeaderClass + '">' +
				'<table>' +
				'<thead>' +
				this.renderHeadTrHtml() +
				'</thead>' +
				'</table>' +
				'</div>';
		},


		renderHeadIntroHtml: function() {
			return this.renderIntroHtml(); // fall back to generic
		},


		renderHeadTrHtml: function() {
			return '' +
				'<tr>' +
				(this.isRTL ? '' : this.renderHeadIntroHtml()) +
				this.renderHeadDateCellsHtml() +
				(this.isRTL ? this.renderHeadIntroHtml() : '') +
				'</tr>';
		},


		renderHeadDateCellsHtml: function() {
			var htmls = [];
			var col, date;

			for (col = 0; col < this.colCnt; col++) {
				date = this.getCellDate(0, col);
				htmls.push(this.renderHeadDateCellHtml(date));
			}

			return htmls.join('');
		},


		// TODO: when internalApiVersion, accept an object for HTML attributes
		// (colspan should be no different)
		renderHeadDateCellHtml: function(date, colspan, otherAttrs) {
			var view = this.view;
			var classNames = [
				'fc-day-header',
				view.widgetHeaderClass
			];

			// if only one row of days, the classNames on the header can represent the specific days beneath
			if (this.rowCnt === 1) {
				classNames = classNames.concat(
					// includes the day-of-week class
					// noThemeHighlight=true (don't highlight the header)
					this.getDayClasses(date, true)
				);
			}
			else {
				classNames.push('fc-' + dayIDs[date.day()]); // only add the day-of-week class
			}

			return '' +
				'<th class="' + classNames.join(' ') + '"' +
				(this.rowCnt === 1 ?
					' data-date="' + date.format('YYYY-MM-DD') + '"' :
					'') +
				(colspan > 1 ?
					' colspan="' + colspan + '"' :
					'') +
				(otherAttrs ?
					' ' + otherAttrs :
					'') +
				'>' +
				// don't make a link if the heading could represent multiple days, or if there's only one day (forceOff)
				view.buildGotoAnchorHtml(
					{ date: date, forceOff: this.rowCnt > 1 || this.colCnt === 1 },
					htmlEscape(date.format(this.colHeadFormat)) // inner HTML
				) +
				'</th>';
		},


		/* Background Rendering
	------------------------------------------------------------------------------------------------------------------*/


		renderBgTrHtml: function(row) {
			return '' +
				'<tr>' +
				(this.isRTL ? '' : this.renderBgIntroHtml(row)) +
				this.renderBgCellsHtml(row) +
				(this.isRTL ? this.renderBgIntroHtml(row) : '') +
				'</tr>';
		},


		renderBgIntroHtml: function(row) {
			return this.renderIntroHtml(); // fall back to generic
		},


		renderBgCellsHtml: function(row) {
			var htmls = [];
			var col, date;

			for (col = 0; col < this.colCnt; col++) {
				date = this.getCellDate(row, col);
				htmls.push(this.renderBgCellHtml(date));
			}

			return htmls.join('');
		},


		renderBgCellHtml: function(date, otherAttrs) {
			var view = this.view;
			var classes = this.getDayClasses(date);

			classes.unshift('fc-day', view.widgetContentClass);

			return '<td class="' + classes.join(' ') + '"' +
				' data-date="' + date.format('YYYY-MM-DD') + '"' + // if date has a time, won't format it
				(otherAttrs ?
					' ' + otherAttrs :
					'') +
				'></td>';
		},


		/* Generic
	------------------------------------------------------------------------------------------------------------------*/


		// Generates the default HTML intro for any row. User classes should override
		renderIntroHtml: function() {
		},


		// TODO: a generic method for dealing with <tr>, RTL, intro
		// when increment internalApiVersion
		// wrapTr (scheduler)


		/* Utils
	------------------------------------------------------------------------------------------------------------------*/


		// Applies the generic "intro" and "outro" HTML to the given cells.
		// Intro means the leftmost cell when the calendar is LTR and the rightmost cell when RTL. Vice-versa for outro.
		bookendCells: function(trEl) {
			var introHtml = this.renderIntroHtml();

			if (introHtml) {
				if (this.isRTL) {
					trEl.append(introHtml);
				}
				else {
					trEl.prepend(introHtml);
				}
			}
		}

	};

	;;

	/* A component that renders a grid of whole-days that runs horizontally. There can be multiple rows, one per week.
----------------------------------------------------------------------------------------------------------------------*/

	var DayGrid = FC.DayGrid = Grid.extend(DayTableMixin, {

		numbersVisible: false, // should render a row for day/week numbers? set by outside view. TODO: make internal
		bottomCoordPadding: 0, // hack for extending the hit area for the last row of the coordinate grid

		rowEls: null, // set of fake row elements
		cellEls: null, // set of whole-day elements comprising the row's background
		helperEls: null, // set of cell skeleton elements for rendering the mock event "helper"

		rowCoordCache: null,
		colCoordCache: null,


		// Renders the rows and columns into the component's `this.el`, which should already be assigned.
		// isRigid determins whether the individual rows should ignore the contents and be a constant height.
		// Relies on the view's colCnt and rowCnt. In the future, this component should probably be self-sufficient.
		renderDates: function(isRigid) {
			var view = this.view;
			var rowCnt = this.rowCnt;
			var colCnt = this.colCnt;
			var html = '';
			var row;
			var col;

			for (row = 0; row < rowCnt; row++) {
				html += this.renderDayRowHtml(row, isRigid);
			}
			this.el.html(html);

			this.rowEls = this.el.find('.fc-row');
			this.cellEls = this.el.find('.fc-day');

			this.rowCoordCache = new CoordCache({
				els: this.rowEls,
				isVertical: true
			});
			this.colCoordCache = new CoordCache({
				els: this.cellEls.slice(0, this.colCnt), // only the first row
				isHorizontal: true
			});

			// trigger dayRender with each cell's element
			for (row = 0; row < rowCnt; row++) {
				for (col = 0; col < colCnt; col++) {
					view.publiclyTrigger(
						'dayRender',
						null,
						this.getCellDate(row, col),
						this.getCellEl(row, col)
					);
				}
			}
		},


		unrenderDates: function() {
			this.removeSegPopover();
		},


		renderBusinessHours: function() {
			var segs = this.buildBusinessHourSegs(true); // wholeDay=true
			this.renderFill('businessHours', segs, 'bgevent');
		},


		unrenderBusinessHours: function() {
			this.unrenderFill('businessHours');
		},


		// Generates the HTML for a single row, which is a div that wraps a table.
		// `row` is the row number.
		renderDayRowHtml: function(row, isRigid) {
			var view = this.view;
			var classes = [ 'fc-row', 'fc-week', view.widgetContentClass ];

			if (isRigid) {
				classes.push('fc-rigid');
			}

			return '' +
				'<div class="' + classes.join(' ') + '">' +
				'<div class="fc-bg">' +
				'<table>' +
				this.renderBgTrHtml(row) +
				'</table>' +
				'</div>' +
				'<div class="fc-content-skeleton">' +
				'<table>' +
				(this.numbersVisible ?
						'<thead>' +
						this.renderNumberTrHtml(row) +
						'</thead>' :
						''
				) +
				'</table>' +
				'</div>' +
				'</div>';
		},


		/* Grid Number Rendering
	------------------------------------------------------------------------------------------------------------------*/


		renderNumberTrHtml: function(row) {
			return '' +
				'<tr>' +
				(this.isRTL ? '' : this.renderNumberIntroHtml(row)) +
				this.renderNumberCellsHtml(row) +
				(this.isRTL ? this.renderNumberIntroHtml(row) : '') +
				'</tr>';
		},


		renderNumberIntroHtml: function(row) {
			return this.renderIntroHtml();
		},


		renderNumberCellsHtml: function(row) {
			var htmls = [];
			var col, date;

			for (col = 0; col < this.colCnt; col++) {
				date = this.getCellDate(row, col);
				htmls.push(this.renderNumberCellHtml(date));
			}

			return htmls.join('');
		},


		// Generates the HTML for the <td>s of the "number" row in the DayGrid's content skeleton.
		// The number row will only exist if either day numbers or week numbers are turned on.
		renderNumberCellHtml: function(date) {
			var html = '';
			var classes;
			var weekCalcFirstDoW;

			if (!this.view.dayNumbersVisible && !this.view.cellWeekNumbersVisible) {
				// no numbers in day cell (week number must be along the side)
				return '<td/>'; //  will create an empty space above events :(
			}

			classes = this.getDayClasses(date);
			classes.unshift('fc-day-top');

			if (this.view.cellWeekNumbersVisible) {
				// To determine the day of week number change under ISO, we cannot
				// rely on moment.js methods such as firstDayOfWeek() or weekday(),
				// because they rely on the locale's dow (possibly overridden by
				// our firstDay option), which may not be Monday. We cannot change
				// dow, because that would affect the calendar start day as well.
				if (date._locale._fullCalendar_weekCalc === 'ISO') {
					weekCalcFirstDoW = 1;  // Monday by ISO 8601 definition
				}
				else {
					weekCalcFirstDoW = date._locale.firstDayOfWeek();
				}
			}

			html += '<td class="' + classes.join(' ') + '" data-date="' + date.format() + '">';

			if (this.view.cellWeekNumbersVisible && (date.day() == weekCalcFirstDoW)) {
				html += this.view.buildGotoAnchorHtml(
					{ date: date, type: 'week' },
					{ 'class': 'fc-week-number' },
					date.format('w') // inner HTML
				);
			}

			if (this.view.dayNumbersVisible) {
				html += this.view.buildGotoAnchorHtml(
					date,
					{ 'class': 'fc-day-number' },
					date.date() // inner HTML
				);
			}

			html += '</td>';

			return html;
		},


		/* Options
	------------------------------------------------------------------------------------------------------------------*/


		// Computes a default event time formatting string if `timeFormat` is not explicitly defined
		computeEventTimeFormat: function() {
			return this.view.opt('extraSmallTimeFormat'); // like "6p" or "6:30p"
		},


		// Computes a default `displayEventEnd` value if one is not expliclty defined
		computeDisplayEventEnd: function() {
			return this.colCnt == 1; // we'll likely have space if there's only one day
		},


		/* Dates
	------------------------------------------------------------------------------------------------------------------*/


		rangeUpdated: function() {
			this.updateDayTable();
		},


		// Slices up the given span (unzoned start/end with other misc data) into an array of segments
		spanToSegs: function(span) {
			var segs = this.sliceRangeByRow(span);
			var i, seg;

			for (i = 0; i < segs.length; i++) {
				seg = segs[i];
				if (this.isRTL) {
					seg.leftCol = this.daysPerRow - 1 - seg.lastRowDayIndex;
					seg.rightCol = this.daysPerRow - 1 - seg.firstRowDayIndex;
				}
				else {
					seg.leftCol = seg.firstRowDayIndex;
					seg.rightCol = seg.lastRowDayIndex;
				}
			}

			return segs;
		},


		/* Hit System
	------------------------------------------------------------------------------------------------------------------*/


		prepareHits: function() {
			this.colCoordCache.build();
			this.rowCoordCache.build();
			this.rowCoordCache.bottoms[this.rowCnt - 1] += this.bottomCoordPadding; // hack
		},


		releaseHits: function() {
			this.colCoordCache.clear();
			this.rowCoordCache.clear();
		},


		queryHit: function(leftOffset, topOffset) {
			if (this.colCoordCache.isLeftInBounds(leftOffset) && this.rowCoordCache.isTopInBounds(topOffset)) {
				var col = this.colCoordCache.getHorizontalIndex(leftOffset);
				var row = this.rowCoordCache.getVerticalIndex(topOffset);

				if (row != null && col != null) {
					return this.getCellHit(row, col);
				}
			}
		},


		getHitSpan: function(hit) {
			return this.getCellRange(hit.row, hit.col);
		},


		getHitEl: function(hit) {
			return this.getCellEl(hit.row, hit.col);
		},


		/* Cell System
	------------------------------------------------------------------------------------------------------------------*/
		// FYI: the first column is the leftmost column, regardless of date


		getCellHit: function(row, col) {
			return {
				row: row,
				col: col,
				component: this, // needed unfortunately :(
				left: this.colCoordCache.getLeftOffset(col),
				right: this.colCoordCache.getRightOffset(col),
				top: this.rowCoordCache.getTopOffset(row),
				bottom: this.rowCoordCache.getBottomOffset(row)
			};
		},


		getCellEl: function(row, col) {
			return this.cellEls.eq(row * this.colCnt + col);
		},


		/* Event Drag Visualization
	------------------------------------------------------------------------------------------------------------------*/
		// TODO: move to DayGrid.event, similar to what we did with Grid's drag methods


		// Renders a visual indication of an event or external element being dragged.
		// `eventLocation` has zoned start and end (optional)
		renderDrag: function(eventLocation, seg) {

			// always render a highlight underneath
			this.renderHighlight(this.eventToSpan(eventLocation));

			// if a segment from the same calendar but another component is being dragged, render a helper event
			if (seg && seg.component !== this) {
				return this.renderEventLocationHelper(eventLocation, seg); // returns mock event elements
			}
		},


		// Unrenders any visual indication of a hovering event
		unrenderDrag: function() {
			this.unrenderHighlight();
			this.unrenderHelper();
		},


		/* Event Resize Visualization
	------------------------------------------------------------------------------------------------------------------*/


		// Renders a visual indication of an event being resized
		renderEventResize: function(eventLocation, seg) {
			this.renderHighlight(this.eventToSpan(eventLocation));
			return this.renderEventLocationHelper(eventLocation, seg); // returns mock event elements
		},


		// Unrenders a visual indication of an event being resized
		unrenderEventResize: function() {
			this.unrenderHighlight();
			this.unrenderHelper();
		},


		/* Event Helper
	------------------------------------------------------------------------------------------------------------------*/


		// Renders a mock "helper" event. `sourceSeg` is the associated internal segment object. It can be null.
		renderHelper: function(event, sourceSeg) {
			var helperNodes = [];
			var segs = this.eventToSegs(event);
			var rowStructs;

			segs = this.renderFgSegEls(segs); // assigns each seg's el and returns a subset of segs that were rendered
			rowStructs = this.renderSegRows(segs);

			// inject each new event skeleton into each associated row
			this.rowEls.each(function(row, rowNode) {
				var rowEl = $(rowNode); // the .fc-row
				var skeletonEl = $('<div class="fc-helper-skeleton"><table/></div>'); // will be absolutely positioned
				var skeletonTop;

				// If there is an original segment, match the top position. Otherwise, put it at the row's top level
				if (sourceSeg && sourceSeg.row === row) {
					skeletonTop = sourceSeg.el.position().top;
				}
				else {
					skeletonTop = rowEl.find('.fc-content-skeleton tbody').position().top;
				}

				skeletonEl.css('top', skeletonTop)
					.find('table')
					.append(rowStructs[row].tbodyEl);

				rowEl.append(skeletonEl);
				helperNodes.push(skeletonEl[0]);
			});

			return ( // must return the elements rendered
				this.helperEls = $(helperNodes) // array -> jQuery set
			);
		},


		// Unrenders any visual indication of a mock helper event
		unrenderHelper: function() {
			if (this.helperEls) {
				this.helperEls.remove();
				this.helperEls = null;
			}
		},


		/* Fill System (highlight, background events, business hours)
	------------------------------------------------------------------------------------------------------------------*/


		fillSegTag: 'td', // override the default tag name


		// Renders a set of rectangles over the given segments of days.
		// Only returns segments that successfully rendered.
		renderFill: function(type, segs, className) {
			var nodes = [];
			var i, seg;
			var skeletonEl;

			segs = this.renderFillSegEls(type, segs); // assignes `.el` to each seg. returns successfully rendered segs

			for (i = 0; i < segs.length; i++) {
				seg = segs[i];
				skeletonEl = this.renderFillRow(type, seg, className);
				this.rowEls.eq(seg.row).append(skeletonEl);
				nodes.push(skeletonEl[0]);
			}

			this.elsByFill[type] = $(nodes);

			return segs;
		},


		// Generates the HTML needed for one row of a fill. Requires the seg's el to be rendered.
		renderFillRow: function(type, seg, className) {
			var colCnt = this.colCnt;
			var startCol = seg.leftCol;
			var endCol = seg.rightCol + 1;
			var skeletonEl;
			var trEl;

			className = className || type.toLowerCase();

			skeletonEl = $(
				'<div class="fc-' + className + '-skeleton">' +
				'<table><tr/></table>' +
				'</div>'
			);
			trEl = skeletonEl.find('tr');

			if (startCol > 0) {
				trEl.append('<td colspan="' + startCol + '"/>');
			}

			trEl.append(
				seg.el.attr('colspan', endCol - startCol)
			);

			if (endCol < colCnt) {
				trEl.append('<td colspan="' + (colCnt - endCol) + '"/>');
			}

			this.bookendCells(trEl);

			return skeletonEl;
		}

	});

	;;

	/* Event-rendering methods for the DayGrid class
----------------------------------------------------------------------------------------------------------------------*/

	DayGrid.mixin({

		rowStructs: null, // an array of objects, each holding information about a row's foreground event-rendering


		// Unrenders all events currently rendered on the grid
		unrenderEvents: function() {
			this.removeSegPopover(); // removes the "more.." events popover
			Grid.prototype.unrenderEvents.apply(this, arguments); // calls the super-method
		},


		// Retrieves all rendered segment objects currently rendered on the grid
		getEventSegs: function() {
			return Grid.prototype.getEventSegs.call(this) // get the segments from the super-method
				.concat(this.popoverSegs || []); // append the segments from the "more..." popover
		},


		// Renders the given background event segments onto the grid
		renderBgSegs: function(segs) {

			// don't render timed background events
			var allDaySegs = $.grep(segs, function(seg) {
				return seg.event.allDay;
			});

			return Grid.prototype.renderBgSegs.call(this, allDaySegs); // call the super-method
		},


		// Renders the given foreground event segments onto the grid
		renderFgSegs: function(segs) {
			var rowStructs;

			// render an `.el` on each seg
			// returns a subset of the segs. segs that were actually rendered
			segs = this.renderFgSegEls(segs);

			rowStructs = this.rowStructs = this.renderSegRows(segs);

			// append to each row's content skeleton
			this.rowEls.each(function(i, rowNode) {
				$(rowNode).find('.fc-content-skeleton > table').append(
					rowStructs[i].tbodyEl
				);
			});

			return segs; // return only the segs that were actually rendered
		},


		// Unrenders all currently rendered foreground event segments
		unrenderFgSegs: function() {
			var rowStructs = this.rowStructs || [];
			var rowStruct;

			while ((rowStruct = rowStructs.pop())) {
				rowStruct.tbodyEl.remove();
			}

			this.rowStructs = null;
		},


		// Uses the given events array to generate <tbody> elements that should be appended to each row's content skeleton.
		// Returns an array of rowStruct objects (see the bottom of `renderSegRow`).
		// PRECONDITION: each segment shoud already have a rendered and assigned `.el`
		renderSegRows: function(segs) {
			var rowStructs = [];
			var segRows;
			var row;

			segRows = this.groupSegRows(segs); // group into nested arrays

			// iterate each row of segment groupings
			for (row = 0; row < segRows.length; row++) {
				rowStructs.push(
					this.renderSegRow(row, segRows[row])
				);
			}

			return rowStructs;
		},


		// Builds the HTML to be used for the default element for an individual segment
		fgSegHtml: function(seg, disableResizing) {
			var view = this.view;
			var event = seg.event;
			var isDraggable = view.isEventDraggable(event);
			var isResizableFromStart = !disableResizing && event.allDay &&
				seg.isStart && view.isEventResizableFromStart(event);
			var isResizableFromEnd = !disableResizing && event.allDay &&
				seg.isEnd && view.isEventResizableFromEnd(event);
			var classes = this.getSegClasses(seg, isDraggable, isResizableFromStart || isResizableFromEnd);
			var skinCss = cssToStr(this.getSegSkinCss(seg));
			var timeHtml = '';
			var timeText;
			var titleHtml;

			classes.unshift('fc-day-grid-event', 'fc-h-event');

			// Only display a timed events time if it is the starting segment
			if (seg.isStart) {
				timeText = this.getEventTimeText(event);
				if (timeText) {
					timeHtml = '<span class="fc-time">' + htmlEscape(timeText) + '</span>';
				}
			}

			titleHtml =
				'<span class="fc-title">' +
				(htmlEscape(event.title || '') || '&nbsp;') + // we always want one line of height
				'</span>';

			return '<a class="' + classes.join(' ') + '"' +
				(event.url ?
						' href="' + htmlEscape(event.url) + '"' :
						''
				) +
				(skinCss ?
						' style="' + skinCss + '"' :
						''
				) +
				'>' +
				'<div class="fc-content">' +
				(this.isRTL ?
						titleHtml + ' ' + timeHtml : // put a natural space in between
						timeHtml + ' ' + titleHtml   //
				) +
				'</div>' +
				(isResizableFromStart ?
						'<div class="fc-resizer fc-start-resizer" />' :
						''
				) +
				(isResizableFromEnd ?
						'<div class="fc-resizer fc-end-resizer" />' :
						''
				) +
				'</a>';
		},


		// Given a row # and an array of segments all in the same row, render a <tbody> element, a skeleton that contains
		// the segments. Returns object with a bunch of internal data about how the render was calculated.
		// NOTE: modifies rowSegs
		renderSegRow: function(row, rowSegs) {
			var colCnt = this.colCnt;
			var segLevels = this.buildSegLevels(rowSegs); // group into sub-arrays of levels
			var levelCnt = Math.max(1, segLevels.length); // ensure at least one level
			var tbody = $('<tbody/>');
			var segMatrix = []; // lookup for which segments are rendered into which level+col cells
			var cellMatrix = []; // lookup for all <td> elements of the level+col matrix
			var loneCellMatrix = []; // lookup for <td> elements that only take up a single column
			var i, levelSegs;
			var col;
			var tr;
			var j, seg;
			var td;

			// populates empty cells from the current column (`col`) to `endCol`
			function emptyCellsUntil(endCol) {
				while (col < endCol) {
					// try to grab a cell from the level above and extend its rowspan. otherwise, create a fresh cell
					td = (loneCellMatrix[i - 1] || [])[col];
					if (td) {
						td.attr(
							'rowspan',
							parseInt(td.attr('rowspan') || 1, 10) + 1
						);
					}
					else {
						td = $('<td/>');
						tr.append(td);
					}
					cellMatrix[i][col] = td;
					loneCellMatrix[i][col] = td;
					col++;
				}
			}

			for (i = 0; i < levelCnt; i++) { // iterate through all levels
				levelSegs = segLevels[i];
				col = 0;
				tr = $('<tr/>');

				segMatrix.push([]);
				cellMatrix.push([]);
				loneCellMatrix.push([]);

				// levelCnt might be 1 even though there are no actual levels. protect against this.
				// this single empty row is useful for styling.
				if (levelSegs) {
					for (j = 0; j < levelSegs.length; j++) { // iterate through segments in level
						seg = levelSegs[j];

						emptyCellsUntil(seg.leftCol);

						// create a container that occupies or more columns. append the event element.
						td = $('<td class="fc-event-container"/>').append(seg.el);
						if (seg.leftCol != seg.rightCol) {
							td.attr('colspan', seg.rightCol - seg.leftCol + 1);
						}
						else { // a single-column segment
							loneCellMatrix[i][col] = td;
						}

						while (col <= seg.rightCol) {
							cellMatrix[i][col] = td;
							segMatrix[i][col] = seg;
							col++;
						}

						tr.append(td);
					}
				}

				emptyCellsUntil(colCnt); // finish off the row
				this.bookendCells(tr);
				tbody.append(tr);
			}

			return { // a "rowStruct"
				row: row, // the row number
				tbodyEl: tbody,
				cellMatrix: cellMatrix,
				segMatrix: segMatrix,
				segLevels: segLevels,
				segs: rowSegs
			};
		},


		// Stacks a flat array of segments, which are all assumed to be in the same row, into subarrays of vertical levels.
		// NOTE: modifies segs
		buildSegLevels: function(segs) {
			var levels = [];
			var i, seg;
			var j;

			// Give preference to elements with certain criteria, so they have
			// a chance to be closer to the top.
			this.sortEventSegs(segs);

			for (i = 0; i < segs.length; i++) {
				seg = segs[i];

				// loop through levels, starting with the topmost, until the segment doesn't collide with other segments
				for (j = 0; j < levels.length; j++) {
					if (!isDaySegCollision(seg, levels[j])) {
						break;
					}
				}
				// `j` now holds the desired subrow index
				seg.level = j;

				// create new level array if needed and append segment
				(levels[j] || (levels[j] = [])).push(seg);
			}

			// order segments left-to-right. very important if calendar is RTL
			for (j = 0; j < levels.length; j++) {
				levels[j].sort(compareDaySegCols);
			}

			return levels;
		},


		// Given a flat array of segments, return an array of sub-arrays, grouped by each segment's row
		groupSegRows: function(segs) {
			var segRows = [];
			var i;

			for (i = 0; i < this.rowCnt; i++) {
				segRows.push([]);
			}

			for (i = 0; i < segs.length; i++) {
				segRows[segs[i].row].push(segs[i]);
			}

			return segRows;
		}

	});


// Computes whether two segments' columns collide. They are assumed to be in the same row.
	function isDaySegCollision(seg, otherSegs) {
		var i, otherSeg;

		for (i = 0; i < otherSegs.length; i++) {
			otherSeg = otherSegs[i];

			if (
				otherSeg.leftCol <= seg.rightCol &&
				otherSeg.rightCol >= seg.leftCol
			) {
				return true;
			}
		}

		return false;
	}


// A cmp function for determining the leftmost event
	function compareDaySegCols(a, b) {
		return a.leftCol - b.leftCol;
	}

	;;

	/* Methods relate to limiting the number events for a given day on a DayGrid
----------------------------------------------------------------------------------------------------------------------*/
// NOTE: all the segs being passed around in here are foreground segs

	DayGrid.mixin({

		segPopover: null, // the Popover that holds events that can't fit in a cell. null when not visible
		popoverSegs: null, // an array of segment objects that the segPopover holds. null when not visible


		removeSegPopover: function() {
			if (this.segPopover) {
				this.segPopover.hide(); // in handler, will call segPopover's removeElement
			}
		},


		// Limits the number of "levels" (vertically stacking layers of events) for each row of the grid.
		// `levelLimit` can be false (don't limit), a number, or true (should be computed).
		limitRows: function(levelLimit) {
			var rowStructs = this.rowStructs || [];
			var row; // row #
			var rowLevelLimit;

			for (row = 0; row < rowStructs.length; row++) {
				this.unlimitRow(row);

				if (!levelLimit) {
					rowLevelLimit = false;
				}
				else if (typeof levelLimit === 'number') {
					rowLevelLimit = levelLimit;
				}
				else {
					rowLevelLimit = this.computeRowLevelLimit(row);
				}

				if (rowLevelLimit !== false) {
					this.limitRow(row, rowLevelLimit);
				}
			}
		},


		// Computes the number of levels a row will accomodate without going outside its bounds.
		// Assumes the row is "rigid" (maintains a constant height regardless of what is inside).
		// `row` is the row number.
		computeRowLevelLimit: function(row) {
			var rowEl = this.rowEls.eq(row); // the containing "fake" row div
			var rowHeight = rowEl.height(); // TODO: cache somehow?
			var trEls = this.rowStructs[row].tbodyEl.children();
			var i, trEl;
			var trHeight;

			function iterInnerHeights(i, childNode) {
				trHeight = Math.max(trHeight, $(childNode).outerHeight());
			}

			// Reveal one level <tr> at a time and stop when we find one out of bounds
			for (i = 0; i < trEls.length; i++) {
				trEl = trEls.eq(i).removeClass('fc-limited'); // reset to original state (reveal)

				// with rowspans>1 and IE8, trEl.outerHeight() would return the height of the largest cell,
				// so instead, find the tallest inner content element.
				trHeight = 0;
				trEl.find('> td > :first-child').each(iterInnerHeights);

				if (trEl.position().top + trHeight > rowHeight) {
					return i;
				}
			}

			return false; // should not limit at all
		},


		// Limits the given grid row to the maximum number of levels and injects "more" links if necessary.
		// `row` is the row number.
		// `levelLimit` is a number for the maximum (inclusive) number of levels allowed.
		limitRow: function(row, levelLimit) {
			var _this = this;
			var rowStruct = this.rowStructs[row];
			var moreNodes = []; // array of "more" <a> links and <td> DOM nodes
			var col = 0; // col #, left-to-right (not chronologically)
			var levelSegs; // array of segment objects in the last allowable level, ordered left-to-right
			var cellMatrix; // a matrix (by level, then column) of all <td> jQuery elements in the row
			var limitedNodes; // array of temporarily hidden level <tr> and segment <td> DOM nodes
			var i, seg;
			var segsBelow; // array of segment objects below `seg` in the current `col`
			var totalSegsBelow; // total number of segments below `seg` in any of the columns `seg` occupies
			var colSegsBelow; // array of segment arrays, below seg, one for each column (offset from segs's first column)
			var td, rowspan;
			var segMoreNodes; // array of "more" <td> cells that will stand-in for the current seg's cell
			var j;
			var moreTd, moreWrap, moreLink;

			// Iterates through empty level cells and places "more" links inside if need be
			function emptyCellsUntil(endCol) { // goes from current `col` to `endCol`
				while (col < endCol) {
					segsBelow = _this.getCellSegs(row, col, levelLimit);
					if (segsBelow.length) {
						td = cellMatrix[levelLimit - 1][col];
						moreLink = _this.renderMoreLink(row, col, segsBelow);
						moreWrap = $('<div/>').append(moreLink);
						td.append(moreWrap);
						moreNodes.push(moreWrap[0]);
					}
					col++;
				}
			}

			if (levelLimit && levelLimit < rowStruct.segLevels.length) { // is it actually over the limit?
				levelSegs = rowStruct.segLevels[levelLimit - 1];
				cellMatrix = rowStruct.cellMatrix;

				limitedNodes = rowStruct.tbodyEl.children().slice(levelLimit) // get level <tr> elements past the limit
					.addClass('fc-limited').get(); // hide elements and get a simple DOM-nodes array

				// iterate though segments in the last allowable level
				for (i = 0; i < levelSegs.length; i++) {
					seg = levelSegs[i];
					emptyCellsUntil(seg.leftCol); // process empty cells before the segment

					// determine *all* segments below `seg` that occupy the same columns
					colSegsBelow = [];
					totalSegsBelow = 0;
					while (col <= seg.rightCol) {
						segsBelow = this.getCellSegs(row, col, levelLimit);
						colSegsBelow.push(segsBelow);
						totalSegsBelow += segsBelow.length;
						col++;
					}

					if (totalSegsBelow) { // do we need to replace this segment with one or many "more" links?
						td = cellMatrix[levelLimit - 1][seg.leftCol]; // the segment's parent cell
						rowspan = td.attr('rowspan') || 1;
						segMoreNodes = [];

						// make a replacement <td> for each column the segment occupies. will be one for each colspan
						for (j = 0; j < colSegsBelow.length; j++) {
							moreTd = $('<td class="fc-more-cell"/>').attr('rowspan', rowspan);
							segsBelow = colSegsBelow[j];
							moreLink = this.renderMoreLink(
								row,
								seg.leftCol + j,
								[ seg ].concat(segsBelow) // count seg as hidden too
							);
							moreWrap = $('<div/>').append(moreLink);
							moreTd.append(moreWrap);
							segMoreNodes.push(moreTd[0]);
							moreNodes.push(moreTd[0]);
						}

						td.addClass('fc-limited').after($(segMoreNodes)); // hide original <td> and inject replacements
						limitedNodes.push(td[0]);
					}
				}

				emptyCellsUntil(this.colCnt); // finish off the level
				rowStruct.moreEls = $(moreNodes); // for easy undoing later
				rowStruct.limitedEls = $(limitedNodes); // for easy undoing later
			}
		},


		// Reveals all levels and removes all "more"-related elements for a grid's row.
		// `row` is a row number.
		unlimitRow: function(row) {
			var rowStruct = this.rowStructs[row];

			if (rowStruct.moreEls) {
				rowStruct.moreEls.remove();
				rowStruct.moreEls = null;
			}

			if (rowStruct.limitedEls) {
				rowStruct.limitedEls.removeClass('fc-limited');
				rowStruct.limitedEls = null;
			}
		},


		// Renders an <a> element that represents hidden event element for a cell.
		// Responsible for attaching click handler as well.
		renderMoreLink: function(row, col, hiddenSegs) {
			var _this = this;
			var view = this.view;

			return $('<a class="fc-more"/>')
				.text(
					this.getMoreLinkText(hiddenSegs.length)
				)
				.on('click', function(ev) {
					var clickOption = view.opt('eventLimitClick');
					var date = _this.getCellDate(row, col);
					var moreEl = $(this);
					var dayEl = _this.getCellEl(row, col);
					var allSegs = _this.getCellSegs(row, col);

					// rescope the segments to be within the cell's date
					var reslicedAllSegs = _this.resliceDaySegs(allSegs, date);
					var reslicedHiddenSegs = _this.resliceDaySegs(hiddenSegs, date);

					if (typeof clickOption === 'function') {
						// the returned value can be an atomic option
						clickOption = view.publiclyTrigger('eventLimitClick', null, {
							date: date,
							dayEl: dayEl,
							moreEl: moreEl,
							segs: reslicedAllSegs,
							hiddenSegs: reslicedHiddenSegs
						}, ev);
					}

					if (clickOption === 'popover') {
						_this.showSegPopover(row, col, moreEl, reslicedAllSegs);
					}
					else if (typeof clickOption === 'string') { // a view name
						view.calendar.zoomTo(date, clickOption);
					}
				});
		},


		// Reveals the popover that displays all events within a cell
		showSegPopover: function(row, col, moreLink, segs) {
			var _this = this;
			var view = this.view;
			var moreWrap = moreLink.parent(); // the <div> wrapper around the <a>
			var topEl; // the element we want to match the top coordinate of
			var options;

			if (this.rowCnt == 1) {
				topEl = view.el; // will cause the popover to cover any sort of header
			}
			else {
				topEl = this.rowEls.eq(row); // will align with top of row
			}

			options = {
				className: 'fc-more-popover',
				content: this.renderSegPopoverContent(row, col, segs),
				parentEl: this.view.el, // attach to root of view. guarantees outside of scrollbars.
				top: topEl.offset().top,
				autoHide: true, // when the user clicks elsewhere, hide the popover
				viewportConstrain: view.opt('popoverViewportConstrain'),
				hide: function() {
					// kill everything when the popover is hidden
					// notify events to be removed
					if (_this.popoverSegs) {
						var seg;
						for (var i = 0; i < _this.popoverSegs.length; ++i) {
							seg = _this.popoverSegs[i];
							view.publiclyTrigger('eventDestroy', seg.event, seg.event, seg.el);
						}
					}
					_this.segPopover.removeElement();
					_this.segPopover = null;
					_this.popoverSegs = null;
				}
			};

			// Determine horizontal coordinate.
			// We use the moreWrap instead of the <td> to avoid border confusion.
			if (this.isRTL) {
				options.right = moreWrap.offset().left + moreWrap.outerWidth() + 1; // +1 to be over cell border
			}
			else {
				options.left = moreWrap.offset().left - 1; // -1 to be over cell border
			}

			this.segPopover = new Popover(options);
			this.segPopover.show();

			// the popover doesn't live within the grid's container element, and thus won't get the event
			// delegated-handlers for free. attach event-related handlers to the popover.
			this.bindSegHandlersToEl(this.segPopover.el);
		},


		// Builds the inner DOM contents of the segment popover
		renderSegPopoverContent: function(row, col, segs) {
			var view = this.view;
			var isTheme = view.opt('theme');
			var title = this.getCellDate(row, col).format(view.opt('dayPopoverFormat'));
			var content = $(
				'<div class="fc-header ' + view.widgetHeaderClass + '">' +
				'<span class="fc-close ' +
				(isTheme ? 'ui-icon ui-icon-closethick' : 'fc-icon fc-icon-x') +
				'"></span>' +
				'<span class="fc-title">' +
				htmlEscape(title) +
				'</span>' +
				'<div class="fc-clear"/>' +
				'</div>' +
				'<div class="fc-body ' + view.widgetContentClass + '">' +
				'<div class="fc-event-container"></div>' +
				'</div>'
			);
			var segContainer = content.find('.fc-event-container');
			var i;

			// render each seg's `el` and only return the visible segs
			segs = this.renderFgSegEls(segs, true); // disableResizing=true
			this.popoverSegs = segs;

			for (i = 0; i < segs.length; i++) {

				// because segments in the popover are not part of a grid coordinate system, provide a hint to any
				// grids that want to do drag-n-drop about which cell it came from
				this.prepareHits();
				segs[i].hit = this.getCellHit(row, col);
				this.releaseHits();

				segContainer.append(segs[i].el);
			}

			return content;
		},


		// Given the events within an array of segment objects, reslice them to be in a single day
		resliceDaySegs: function(segs, dayDate) {

			// build an array of the original events
			var events = $.map(segs, function(seg) {
				return seg.event;
			});

			var dayStart = dayDate.clone();
			var dayEnd = dayStart.clone().add(1, 'days');
			var dayRange = { start: dayStart, end: dayEnd };

			// slice the events with a custom slicing function
			segs = this.eventsToSegs(
				events,
				function(range) {
					var seg = intersectRanges(range, dayRange); // undefind if no intersection
					return seg ? [ seg ] : []; // must return an array of segments
				}
			);

			// force an order because eventsToSegs doesn't guarantee one
			this.sortEventSegs(segs);

			return segs;
		},


		// Generates the text that should be inside a "more" link, given the number of events it represents
		getMoreLinkText: function(num) {
			var opt = this.view.opt('eventLimitText');

			if (typeof opt === 'function') {
				return opt(num);
			}
			else {
				return '+' + num + ' ' + opt;
			}
		},


		// Returns segments within a given cell.
		// If `startLevel` is specified, returns only events including and below that level. Otherwise returns all segs.
		getCellSegs: function(row, col, startLevel) {
			var segMatrix = this.rowStructs[row].segMatrix;
			var level = startLevel || 0;
			var segs = [];
			var seg;

			while (level < segMatrix.length) {
				seg = segMatrix[level][col];
				if (seg) {
					segs.push(seg);
				}
				level++;
			}

			return segs;
		}

	});

	;;

	/* A component that renders one or more columns of vertical time slots
----------------------------------------------------------------------------------------------------------------------*/
// We mixin DayTable, even though there is only a single row of days

	var TimeGrid = FC.TimeGrid = Grid.extend(DayTableMixin, {

		slotDuration: null, // duration of a "slot", a distinct time segment on given day, visualized by lines
		snapDuration: null, // granularity of time for dragging and selecting
		snapsPerSlot: null,
		minTime: null, // Duration object that denotes the first visible time of any given day
		maxTime: null, // Duration object that denotes the exclusive visible end time of any given day
		labelFormat: null, // formatting string for times running along vertical axis
		labelInterval: null, // duration of how often a label should be displayed for a slot

		colEls: null, // cells elements in the day-row background
		slatContainerEl: null, // div that wraps all the slat rows
		slatEls: null, // elements running horizontally across all columns
		nowIndicatorEls: null,

		colCoordCache: null,
		slatCoordCache: null,


		constructor: function() {
			Grid.apply(this, arguments); // call the super-constructor

			this.processOptions();
		},


		// Renders the time grid into `this.el`, which should already be assigned.
		// Relies on the view's colCnt. In the future, this component should probably be self-sufficient.
		renderDates: function() {
			this.el.html(this.renderHtml());
			this.colEls = this.el.find('.fc-day');
			this.slatContainerEl = this.el.find('.fc-slats');
			this.slatEls = this.slatContainerEl.find('tr');

			this.colCoordCache = new CoordCache({
				els: this.colEls,
				isHorizontal: true
			});
			this.slatCoordCache = new CoordCache({
				els: this.slatEls,
				isVertical: true
			});

			this.renderContentSkeleton();
		},


		// Renders the basic HTML skeleton for the grid
		renderHtml: function() {
			return '' +
				'<div class="fc-bg">' +
				'<table>' +
				this.renderBgTrHtml(0) + // row=0
				'</table>' +
				'</div>' +
				'<div class="fc-slats">' +
				'<table>' +
				this.renderSlatRowHtml() +
				'</table>' +
				'</div>';
		},


		// Generates the HTML for the horizontal "slats" that run width-wise. Has a time axis on a side. Depends on RTL.
		renderSlatRowHtml: function() {
			var view = this.view;
			var isRTL = this.isRTL;
			var html = '';
			var slotTime = moment.duration(+this.minTime); // wish there was .clone() for durations
			var slotDate; // will be on the view's first day, but we only care about its time
			var isLabeled;
			var axisHtml;

			// Calculate the time for each slot
			while (slotTime < this.maxTime) {
				slotDate = this.start.clone().time(slotTime);
				isLabeled = isInt(divideDurationByDuration(slotTime, this.labelInterval));

				axisHtml =
					'<td class="fc-axis fc-time ' + view.widgetContentClass + '" ' + view.axisStyleAttr() + '>' +
					(isLabeled ?
							'<span>' + // for matchCellWidths
							htmlEscape(slotDate.format(this.labelFormat)) +
							'</span>' :
							''
					) +
					'</td>';

				html +=
					'<tr data-time="' + slotDate.format('HH:mm:ss') + '"' +
					(isLabeled ? '' : ' class="fc-minor"') +
					'>' +
					(!isRTL ? axisHtml : '') +
					'<td class="' + view.widgetContentClass + '"/>' +
					(isRTL ? axisHtml : '') +
					"</tr>";

				slotTime.add(this.slotDuration);
			}

			return html;
		},


		/* Options
	------------------------------------------------------------------------------------------------------------------*/


		// Parses various options into properties of this object
		processOptions: function() {
			var view = this.view;
			var slotDuration = view.opt('slotDuration');
			var snapDuration = view.opt('snapDuration');
			var input;

			slotDuration = moment.duration(slotDuration);
			snapDuration = snapDuration ? moment.duration(snapDuration) : slotDuration;

			this.slotDuration = slotDuration;
			this.snapDuration = snapDuration;
			this.snapsPerSlot = slotDuration / snapDuration; // TODO: ensure an integer multiple?

			this.minResizeDuration = snapDuration; // hack

			this.minTime = moment.duration(view.opt('minTime'));
			this.maxTime = moment.duration(view.opt('maxTime'));

			// might be an array value (for TimelineView).
			// if so, getting the most granular entry (the last one probably).
			input = view.opt('slotLabelFormat');
			if ($.isArray(input)) {
				input = input[input.length - 1];
			}

			this.labelFormat =
				input ||
				view.opt('smallTimeFormat'); // the computed default

			input = view.opt('slotLabelInterval');
			this.labelInterval = input ?
				moment.duration(input) :
				this.computeLabelInterval(slotDuration);
		},


		// Computes an automatic value for slotLabelInterval
		computeLabelInterval: function(slotDuration) {
			var i;
			var labelInterval;
			var slotsPerLabel;

			// find the smallest stock label interval that results in more than one slots-per-label
			for (i = AGENDA_STOCK_SUB_DURATIONS.length - 1; i >= 0; i--) {
				labelInterval = moment.duration(AGENDA_STOCK_SUB_DURATIONS[i]);
				slotsPerLabel = divideDurationByDuration(labelInterval, slotDuration);
				if (isInt(slotsPerLabel) && slotsPerLabel > 1) {
					return labelInterval;
				}
			}

			return moment.duration(slotDuration); // fall back. clone
		},


		// Computes a default event time formatting string if `timeFormat` is not explicitly defined
		computeEventTimeFormat: function() {
			return this.view.opt('noMeridiemTimeFormat'); // like "6:30" (no AM/PM)
		},


		// Computes a default `displayEventEnd` value if one is not expliclty defined
		computeDisplayEventEnd: function() {
			return true;
		},


		/* Hit System
	------------------------------------------------------------------------------------------------------------------*/


		prepareHits: function() {
			this.colCoordCache.build();
			this.slatCoordCache.build();
		},


		releaseHits: function() {
			this.colCoordCache.clear();
			// NOTE: don't clear slatCoordCache because we rely on it for computeTimeTop
		},


		queryHit: function(leftOffset, topOffset) {
			var snapsPerSlot = this.snapsPerSlot;
			var colCoordCache = this.colCoordCache;
			var slatCoordCache = this.slatCoordCache;

			if (colCoordCache.isLeftInBounds(leftOffset) && slatCoordCache.isTopInBounds(topOffset)) {
				var colIndex = colCoordCache.getHorizontalIndex(leftOffset);
				var slatIndex = slatCoordCache.getVerticalIndex(topOffset);

				if (colIndex != null && slatIndex != null) {
					var slatTop = slatCoordCache.getTopOffset(slatIndex);
					var slatHeight = slatCoordCache.getHeight(slatIndex);
					var partial = (topOffset - slatTop) / slatHeight; // floating point number between 0 and 1
					var localSnapIndex = Math.floor(partial * snapsPerSlot); // the snap # relative to start of slat
					var snapIndex = slatIndex * snapsPerSlot + localSnapIndex;
					var snapTop = slatTop + (localSnapIndex / snapsPerSlot) * slatHeight;
					var snapBottom = slatTop + ((localSnapIndex + 1) / snapsPerSlot) * slatHeight;

					return {
						col: colIndex,
						snap: snapIndex,
						component: this, // needed unfortunately :(
						left: colCoordCache.getLeftOffset(colIndex),
						right: colCoordCache.getRightOffset(colIndex),
						top: snapTop,
						bottom: snapBottom
					};
				}
			}
		},


		getHitSpan: function(hit) {
			var start = this.getCellDate(0, hit.col); // row=0
			var time = this.computeSnapTime(hit.snap); // pass in the snap-index
			var end;

			start.time(time);
			end = start.clone().add(this.snapDuration);

			return { start: start, end: end };
		},


		getHitEl: function(hit) {
			return this.colEls.eq(hit.col);
		},


		/* Dates
	------------------------------------------------------------------------------------------------------------------*/


		rangeUpdated: function() {
			this.updateDayTable();
		},


		// Given a row number of the grid, representing a "snap", returns a time (Duration) from its start-of-day
		computeSnapTime: function(snapIndex) {
			return moment.duration(this.minTime + this.snapDuration * snapIndex);
		},


		// Slices up the given span (unzoned start/end with other misc data) into an array of segments
		spanToSegs: function(span) {
			var segs = this.sliceRangeByTimes(span);
			var i;

			for (i = 0; i < segs.length; i++) {
				if (this.isRTL) {
					segs[i].col = this.daysPerRow - 1 - segs[i].dayIndex;
				}
				else {
					segs[i].col = segs[i].dayIndex;
				}
			}

			return segs;
		},


		sliceRangeByTimes: function(range) {
			var segs = [];
			var seg;
			var dayIndex;
			var dayDate;
			var dayRange;

			for (dayIndex = 0; dayIndex < this.daysPerRow; dayIndex++) {
				dayDate = this.dayDates[dayIndex].clone(); // TODO: better API for this?
				dayRange = {
					start: dayDate.clone().time(this.minTime),
					end: dayDate.clone().time(this.maxTime)
				};
				seg = intersectRanges(range, dayRange); // both will be ambig timezone
				if (seg) {
					seg.dayIndex = dayIndex;
					segs.push(seg);
				}
			}

			return segs;
		},


		/* Coordinates
	------------------------------------------------------------------------------------------------------------------*/


		updateSize: function(isResize) { // NOT a standard Grid method
			this.slatCoordCache.build();

			if (isResize) {
				this.updateSegVerticals(
					[].concat(this.fgSegs || [], this.bgSegs || [], this.businessSegs || [])
				);
			}
		},


		getTotalSlatHeight: function() {
			return this.slatContainerEl.outerHeight();
		},


		// Computes the top coordinate, relative to the bounds of the grid, of the given date.
		// A `startOfDayDate` must be given for avoiding ambiguity over how to treat midnight.
		computeDateTop: function(date, startOfDayDate) {
			return this.computeTimeTop(
				moment.duration(
					date - startOfDayDate.clone().stripTime()
				)
			);
		},


		// Computes the top coordinate, relative to the bounds of the grid, of the given time (a Duration).
		computeTimeTop: function(time) {
			var len = this.slatEls.length;
			var slatCoverage = (time - this.minTime) / this.slotDuration; // floating-point value of # of slots covered
			var slatIndex;
			var slatRemainder;

			// compute a floating-point number for how many slats should be progressed through.
			// from 0 to number of slats (inclusive)
			// constrained because minTime/maxTime might be customized.
			slatCoverage = Math.max(0, slatCoverage);
			slatCoverage = Math.min(len, slatCoverage);

			// an integer index of the furthest whole slat
			// from 0 to number slats (*exclusive*, so len-1)
			slatIndex = Math.floor(slatCoverage);
			slatIndex = Math.min(slatIndex, len - 1);

			// how much further through the slatIndex slat (from 0.0-1.0) must be covered in addition.
			// could be 1.0 if slatCoverage is covering *all* the slots
			slatRemainder = slatCoverage - slatIndex;

			return this.slatCoordCache.getTopPosition(slatIndex) +
				this.slatCoordCache.getHeight(slatIndex) * slatRemainder;
		},



		/* Event Drag Visualization
	------------------------------------------------------------------------------------------------------------------*/


		// Renders a visual indication of an event being dragged over the specified date(s).
		// A returned value of `true` signals that a mock "helper" event has been rendered.
		renderDrag: function(eventLocation, seg) {

			if (seg) { // if there is event information for this drag, render a helper event

				// returns mock event elements
				// signal that a helper has been rendered
				return this.renderEventLocationHelper(eventLocation, seg);
			}
			else {
				// otherwise, just render a highlight
				this.renderHighlight(this.eventToSpan(eventLocation));
			}
		},


		// Unrenders any visual indication of an event being dragged
		unrenderDrag: function() {
			this.unrenderHelper();
			this.unrenderHighlight();
		},


		/* Event Resize Visualization
	------------------------------------------------------------------------------------------------------------------*/


		// Renders a visual indication of an event being resized
		renderEventResize: function(eventLocation, seg) {
			return this.renderEventLocationHelper(eventLocation, seg); // returns mock event elements
		},


		// Unrenders any visual indication of an event being resized
		unrenderEventResize: function() {
			this.unrenderHelper();
		},


		/* Event Helper
	------------------------------------------------------------------------------------------------------------------*/


		// Renders a mock "helper" event. `sourceSeg` is the original segment object and might be null (an external drag)
		renderHelper: function(event, sourceSeg) {
			return this.renderHelperSegs(this.eventToSegs(event), sourceSeg); // returns mock event elements
		},


		// Unrenders any mock helper event
		unrenderHelper: function() {
			this.unrenderHelperSegs();
		},


		/* Business Hours
	------------------------------------------------------------------------------------------------------------------*/


		renderBusinessHours: function() {
			this.renderBusinessSegs(
				this.buildBusinessHourSegs()
			);
		},


		unrenderBusinessHours: function() {
			this.unrenderBusinessSegs();
		},


		/* Now Indicator
	------------------------------------------------------------------------------------------------------------------*/


		getNowIndicatorUnit: function() {
			return 'minute'; // will refresh on the minute
		},


		renderNowIndicator: function(date) {
			// seg system might be overkill, but it handles scenario where line needs to be rendered
			//  more than once because of columns with the same date (resources columns for example)
			var segs = this.spanToSegs({ start: date, end: date });
			var top = this.computeDateTop(date, date);
			var nodes = [];
			var i;

			// render lines within the columns
			for (i = 0; i < segs.length; i++) {
				nodes.push($('<div class="fc-now-indicator fc-now-indicator-line"></div>')
					.css('top', top)
					.appendTo(this.colContainerEls.eq(segs[i].col))[0]);
			}

			// render an arrow over the axis
			if (segs.length > 0) { // is the current time in view?
				nodes.push($('<div class="fc-now-indicator fc-now-indicator-arrow"></div>')
					.css('top', top)
					.appendTo(this.el.find('.fc-content-skeleton'))[0]);
			}

			this.nowIndicatorEls = $(nodes);
		},


		unrenderNowIndicator: function() {
			if (this.nowIndicatorEls) {
				this.nowIndicatorEls.remove();
				this.nowIndicatorEls = null;
			}
		},


		/* Selection
	------------------------------------------------------------------------------------------------------------------*/


		// Renders a visual indication of a selection. Overrides the default, which was to simply render a highlight.
		renderSelection: function(span) {
			if (this.view.opt('selectHelper')) { // this setting signals that a mock helper event should be rendered

				// normally acceps an eventLocation, span has a start/end, which is good enough
				this.renderEventLocationHelper(span);
			}
			else {
				this.renderHighlight(span);
			}
		},


		// Unrenders any visual indication of a selection
		unrenderSelection: function() {
			this.unrenderHelper();
			this.unrenderHighlight();
		},


		/* Highlight
	------------------------------------------------------------------------------------------------------------------*/


		renderHighlight: function(span) {
			this.renderHighlightSegs(this.spanToSegs(span));
		},


		unrenderHighlight: function() {
			this.unrenderHighlightSegs();
		}

	});

	;;

	/* Methods for rendering SEGMENTS, pieces of content that live on the view
 ( this file is no longer just for events )
----------------------------------------------------------------------------------------------------------------------*/

	TimeGrid.mixin({

		colContainerEls: null, // containers for each column

		// inner-containers for each column where different types of segs live
		fgContainerEls: null,
		bgContainerEls: null,
		helperContainerEls: null,
		highlightContainerEls: null,
		businessContainerEls: null,

		// arrays of different types of displayed segments
		fgSegs: null,
		bgSegs: null,
		helperSegs: null,
		highlightSegs: null,
		businessSegs: null,


		// Renders the DOM that the view's content will live in
		renderContentSkeleton: function() {
			var cellHtml = '';
			var i;
			var skeletonEl;

			for (i = 0; i < this.colCnt; i++) {
				cellHtml +=
					'<td>' +
					'<div class="fc-content-col">' +
					'<div class="fc-event-container fc-helper-container"></div>' +
					'<div class="fc-event-container"></div>' +
					'<div class="fc-highlight-container"></div>' +
					'<div class="fc-bgevent-container"></div>' +
					'<div class="fc-business-container"></div>' +
					'</div>' +
					'</td>';
			}

			skeletonEl = $(
				'<div class="fc-content-skeleton">' +
				'<table>' +
				'<tr>' + cellHtml + '</tr>' +
				'</table>' +
				'</div>'
			);

			this.colContainerEls = skeletonEl.find('.fc-content-col');
			this.helperContainerEls = skeletonEl.find('.fc-helper-container');
			this.fgContainerEls = skeletonEl.find('.fc-event-container:not(.fc-helper-container)');
			this.bgContainerEls = skeletonEl.find('.fc-bgevent-container');
			this.highlightContainerEls = skeletonEl.find('.fc-highlight-container');
			this.businessContainerEls = skeletonEl.find('.fc-business-container');

			this.bookendCells(skeletonEl.find('tr')); // TODO: do this on string level
			this.el.append(skeletonEl);
		},


		/* Foreground Events
	------------------------------------------------------------------------------------------------------------------*/


		renderFgSegs: function(segs) {
			segs = this.renderFgSegsIntoContainers(segs, this.fgContainerEls);
			this.fgSegs = segs;
			return segs; // needed for Grid::renderEvents
		},


		unrenderFgSegs: function() {
			this.unrenderNamedSegs('fgSegs');
		},


		/* Foreground Helper Events
	------------------------------------------------------------------------------------------------------------------*/


		renderHelperSegs: function(segs, sourceSeg) {
			var helperEls = [];
			var i, seg;
			var sourceEl;

			segs = this.renderFgSegsIntoContainers(segs, this.helperContainerEls);

			// Try to make the segment that is in the same row as sourceSeg look the same
			for (i = 0; i < segs.length; i++) {
				seg = segs[i];
				if (sourceSeg && sourceSeg.col === seg.col) {
					sourceEl = sourceSeg.el;
					seg.el.css({
						left: sourceEl.css('left'),
						right: sourceEl.css('right'),
						'margin-left': sourceEl.css('margin-left'),
						'margin-right': sourceEl.css('margin-right')
					});
				}
				helperEls.push(seg.el[0]);
			}

			this.helperSegs = segs;

			return $(helperEls); // must return rendered helpers
		},


		unrenderHelperSegs: function() {
			this.unrenderNamedSegs('helperSegs');
		},


		/* Background Events
	------------------------------------------------------------------------------------------------------------------*/


		renderBgSegs: function(segs) {
			segs = this.renderFillSegEls('bgEvent', segs); // TODO: old fill system
			this.updateSegVerticals(segs);
			this.attachSegsByCol(this.groupSegsByCol(segs), this.bgContainerEls);
			this.bgSegs = segs;
			return segs; // needed for Grid::renderEvents
		},


		unrenderBgSegs: function() {
			this.unrenderNamedSegs('bgSegs');
		},


		/* Highlight
	------------------------------------------------------------------------------------------------------------------*/


		renderHighlightSegs: function(segs) {
			segs = this.renderFillSegEls('highlight', segs); // TODO: old fill system
			this.updateSegVerticals(segs);
			this.attachSegsByCol(this.groupSegsByCol(segs), this.highlightContainerEls);
			this.highlightSegs = segs;
		},


		unrenderHighlightSegs: function() {
			this.unrenderNamedSegs('highlightSegs');
		},


		/* Business Hours
	------------------------------------------------------------------------------------------------------------------*/


		renderBusinessSegs: function(segs) {
			segs = this.renderFillSegEls('businessHours', segs); // TODO: old fill system
			this.updateSegVerticals(segs);
			this.attachSegsByCol(this.groupSegsByCol(segs), this.businessContainerEls);
			this.businessSegs = segs;
		},


		unrenderBusinessSegs: function() {
			this.unrenderNamedSegs('businessSegs');
		},


		/* Seg Rendering Utils
	------------------------------------------------------------------------------------------------------------------*/


		// Given a flat array of segments, return an array of sub-arrays, grouped by each segment's col
		groupSegsByCol: function(segs) {
			var segsByCol = [];
			var i;

			for (i = 0; i < this.colCnt; i++) {
				segsByCol.push([]);
			}

			for (i = 0; i < segs.length; i++) {
				segsByCol[segs[i].col].push(segs[i]);
			}

			return segsByCol;
		},


		// Given segments grouped by column, insert the segments' elements into a parallel array of container
		// elements, each living within a column.
		attachSegsByCol: function(segsByCol, containerEls) {
			var col;
			var segs;
			var i;

			for (col = 0; col < this.colCnt; col++) { // iterate each column grouping
				segs = segsByCol[col];

				for (i = 0; i < segs.length; i++) {
					containerEls.eq(col).append(segs[i].el);
				}
			}
		},


		// Given the name of a property of `this` object, assumed to be an array of segments,
		// loops through each segment and removes from DOM. Will null-out the property afterwards.
		unrenderNamedSegs: function(propName) {
			var segs = this[propName];
			var i;

			if (segs) {
				for (i = 0; i < segs.length; i++) {
					segs[i].el.remove();
				}
				this[propName] = null;
			}
		},



		/* Foreground Event Rendering Utils
	------------------------------------------------------------------------------------------------------------------*/


		// Given an array of foreground segments, render a DOM element for each, computes position,
		// and attaches to the column inner-container elements.
		renderFgSegsIntoContainers: function(segs, containerEls) {
			var segsByCol;
			var col;

			segs = this.renderFgSegEls(segs); // will call fgSegHtml
			segsByCol = this.groupSegsByCol(segs);

			for (col = 0; col < this.colCnt; col++) {
				this.updateFgSegCoords(segsByCol[col]);
			}

			this.attachSegsByCol(segsByCol, containerEls);

			return segs;
		},


		// Renders the HTML for a single event segment's default rendering
		fgSegHtml: function(seg, disableResizing) {
			var view = this.view;
			var event = seg.event;
			var isDraggable = view.isEventDraggable(event);
			var isResizableFromStart = !disableResizing && seg.isStart && view.isEventResizableFromStart(event);
			var isResizableFromEnd = !disableResizing && seg.isEnd && view.isEventResizableFromEnd(event);
			var classes = this.getSegClasses(seg, isDraggable, isResizableFromStart || isResizableFromEnd);
			var skinCss = cssToStr(this.getSegSkinCss(seg));
			var timeText;
			var fullTimeText; // more verbose time text. for the print stylesheet
			var startTimeText; // just the start time text

			classes.unshift('fc-time-grid-event', 'fc-v-event');

			if (view.isMultiDayEvent(event)) { // if the event appears to span more than one day...
				// Don't display time text on segments that run entirely through a day.
				// That would appear as midnight-midnight and would look dumb.
				// Otherwise, display the time text for the *segment's* times (like 6pm-midnight or midnight-10am)
				if (seg.isStart || seg.isEnd) {
					timeText = this.getEventTimeText(seg);
					fullTimeText = this.getEventTimeText(seg, 'LT');
					startTimeText = this.getEventTimeText(seg, null, false); // displayEnd=false
				}
			} else {
				// Display the normal time text for the *event's* times
				timeText = this.getEventTimeText(event);
				fullTimeText = this.getEventTimeText(event, 'LT');
				startTimeText = this.getEventTimeText(event, null, false); // displayEnd=false
			}

			return '<a class="' + classes.join(' ') + '"' +
				(event.url ?
						' href="' + htmlEscape(event.url) + '"' :
						''
				) +
				(skinCss ?
						' style="' + skinCss + '"' :
						''
				) +
				'>' +
				'<div class="fc-content">' +
				(timeText ?
						'<div class="fc-time"' +
						' data-start="' + htmlEscape(startTimeText) + '"' +
						' data-full="' + htmlEscape(fullTimeText) + '"' +
						'>' +
						'<span>' + htmlEscape(timeText) + '</span>' +
						'</div>' :
						''
				) +
				(event.title ?
						'<div class="fc-title">' +
						htmlEscape(event.title) +
						'</div>' :
						''
				) +
				'</div>' +
				'<div class="fc-bg"/>' +
				/* TODO: write CSS for this
				(isResizableFromStart ?
					'<div class="fc-resizer fc-start-resizer" />' :
					''
					) +
				*/
				(isResizableFromEnd ?
						'<div class="fc-resizer fc-end-resizer" />' :
						''
				) +
				'</a>';
		},


		/* Seg Position Utils
	------------------------------------------------------------------------------------------------------------------*/


		// Refreshes the CSS top/bottom coordinates for each segment element.
		// Works when called after initial render, after a window resize/zoom for example.
		updateSegVerticals: function(segs) {
			this.computeSegVerticals(segs);
			this.assignSegVerticals(segs);
		},


		// For each segment in an array, computes and assigns its top and bottom properties
		computeSegVerticals: function(segs) {
			var i, seg;

			for (i = 0; i < segs.length; i++) {
				seg = segs[i];
				seg.top = this.computeDateTop(seg.start, seg.start);
				seg.bottom = this.computeDateTop(seg.end, seg.start);
			}
		},


		// Given segments that already have their top/bottom properties computed, applies those values to
		// the segments' elements.
		assignSegVerticals: function(segs) {
			var i, seg;

			for (i = 0; i < segs.length; i++) {
				seg = segs[i];
				seg.el.css(this.generateSegVerticalCss(seg));
			}
		},


		// Generates an object with CSS properties for the top/bottom coordinates of a segment element
		generateSegVerticalCss: function(seg) {
			return {
				top: seg.top,
				bottom: -seg.bottom // flipped because needs to be space beyond bottom edge of event container
			};
		},


		/* Foreground Event Positioning Utils
	------------------------------------------------------------------------------------------------------------------*/


		// Given segments that are assumed to all live in the *same column*,
		// compute their verical/horizontal coordinates and assign to their elements.
		updateFgSegCoords: function(segs) {
			this.computeSegVerticals(segs); // horizontals relies on this
			this.computeFgSegHorizontals(segs); // compute horizontal coordinates, z-index's, and reorder the array
			this.assignSegVerticals(segs);
			this.assignFgSegHorizontals(segs);
		},


		// Given an array of segments that are all in the same column, sets the backwardCoord and forwardCoord on each.
		// NOTE: Also reorders the given array by date!
		computeFgSegHorizontals: function(segs) {
			var levels;
			var level0;
			var i;

			this.sortEventSegs(segs); // order by certain criteria
			levels = buildSlotSegLevels(segs);
			computeForwardSlotSegs(levels);

			if ((level0 = levels[0])) {

				for (i = 0; i < level0.length; i++) {
					computeSlotSegPressures(level0[i]);
				}

				for (i = 0; i < level0.length; i++) {
					this.computeFgSegForwardBack(level0[i], 0, 0);
				}
			}
		},


		// Calculate seg.forwardCoord and seg.backwardCoord for the segment, where both values range
		// from 0 to 1. If the calendar is left-to-right, the seg.backwardCoord maps to "left" and
		// seg.forwardCoord maps to "right" (via percentage). Vice-versa if the calendar is right-to-left.
		//
		// The segment might be part of a "series", which means consecutive segments with the same pressure
		// who's width is unknown until an edge has been hit. `seriesBackwardPressure` is the number of
		// segments behind this one in the current series, and `seriesBackwardCoord` is the starting
		// coordinate of the first segment in the series.
		computeFgSegForwardBack: function(seg, seriesBackwardPressure, seriesBackwardCoord) {
			var forwardSegs = seg.forwardSegs;
			var i;

			if (seg.forwardCoord === undefined) { // not already computed

				if (!forwardSegs.length) {

					// if there are no forward segments, this segment should butt up against the edge
					seg.forwardCoord = 1;
				}
				else {

					// sort highest pressure first
					this.sortForwardSegs(forwardSegs);

					// this segment's forwardCoord will be calculated from the backwardCoord of the
					// highest-pressure forward segment.
					this.computeFgSegForwardBack(forwardSegs[0], seriesBackwardPressure + 1, seriesBackwardCoord);
					seg.forwardCoord = forwardSegs[0].backwardCoord;
				}

				// calculate the backwardCoord from the forwardCoord. consider the series
				seg.backwardCoord = seg.forwardCoord -
					(seg.forwardCoord - seriesBackwardCoord) / // available width for series
					(seriesBackwardPressure + 1); // # of segments in the series

				// use this segment's coordinates to computed the coordinates of the less-pressurized
				// forward segments
				for (i=0; i<forwardSegs.length; i++) {
					this.computeFgSegForwardBack(forwardSegs[i], 0, seg.forwardCoord);
				}
			}
		},


		sortForwardSegs: function(forwardSegs) {
			forwardSegs.sort(proxy(this, 'compareForwardSegs'));
		},


		// A cmp function for determining which forward segment to rely on more when computing coordinates.
		compareForwardSegs: function(seg1, seg2) {
			// put higher-pressure first
			return seg2.forwardPressure - seg1.forwardPressure ||
				// put segments that are closer to initial edge first (and favor ones with no coords yet)
				(seg1.backwardCoord || 0) - (seg2.backwardCoord || 0) ||
				// do normal sorting...
				this.compareEventSegs(seg1, seg2);
		},


		// Given foreground event segments that have already had their position coordinates computed,
		// assigns position-related CSS values to their elements.
		assignFgSegHorizontals: function(segs) {
			var i, seg;

			for (i = 0; i < segs.length; i++) {
				seg = segs[i];
				seg.el.css(this.generateFgSegHorizontalCss(seg));

				// if the height is short, add a className for alternate styling
				if (seg.bottom - seg.top < 30) {
					seg.el.addClass('fc-short');
				}
			}
		},


		// Generates an object with CSS properties/values that should be applied to an event segment element.
		// Contains important positioning-related properties that should be applied to any event element, customized or not.
		generateFgSegHorizontalCss: function(seg) {
			var shouldOverlap = this.view.opt('slotEventOverlap');
			var backwardCoord = seg.backwardCoord; // the left side if LTR. the right side if RTL. floating-point
			var forwardCoord = seg.forwardCoord; // the right side if LTR. the left side if RTL. floating-point
			var props = this.generateSegVerticalCss(seg); // get top/bottom first
			var left; // amount of space from left edge, a fraction of the total width
			var right; // amount of space from right edge, a fraction of the total width

			if (shouldOverlap) {
				// double the width, but don't go beyond the maximum forward coordinate (1.0)
				forwardCoord = Math.min(1, backwardCoord + (forwardCoord - backwardCoord) * 2);
			}

			if (this.isRTL) {
				left = 1 - forwardCoord;
				right = backwardCoord;
			}
			else {
				left = backwardCoord;
				right = 1 - forwardCoord;
			}

			props.zIndex = seg.level + 1; // convert from 0-base to 1-based
			props.left = left * 100 + '%';
			props.right = right * 100 + '%';

			if (shouldOverlap && seg.forwardPressure) {
				// add padding to the edge so that forward stacked events don't cover the resizer's icon
				props[this.isRTL ? 'marginLeft' : 'marginRight'] = 10 * 2; // 10 is a guesstimate of the icon's width
			}

			return props;
		}

	});


// Builds an array of segments "levels". The first level will be the leftmost tier of segments if the calendar is
// left-to-right, or the rightmost if the calendar is right-to-left. Assumes the segments are already ordered by date.
	function buildSlotSegLevels(segs) {
		var levels = [];
		var i, seg;
		var j;

		for (i=0; i<segs.length; i++) {
			seg = segs[i];

			// go through all the levels and stop on the first level where there are no collisions
			for (j=0; j<levels.length; j++) {
				if (!computeSlotSegCollisions(seg, levels[j]).length) {
					break;
				}
			}

			seg.level = j;

			(levels[j] || (levels[j] = [])).push(seg);
		}

		return levels;
	}


// For every segment, figure out the other segments that are in subsequent
// levels that also occupy the same vertical space. Accumulate in seg.forwardSegs
	function computeForwardSlotSegs(levels) {
		var i, level;
		var j, seg;
		var k;

		for (i=0; i<levels.length; i++) {
			level = levels[i];

			for (j=0; j<level.length; j++) {
				seg = level[j];

				seg.forwardSegs = [];
				for (k=i+1; k<levels.length; k++) {
					computeSlotSegCollisions(seg, levels[k], seg.forwardSegs);
				}
			}
		}
	}


// Figure out which path forward (via seg.forwardSegs) results in the longest path until
// the furthest edge is reached. The number of segments in this path will be seg.forwardPressure
	function computeSlotSegPressures(seg) {
		var forwardSegs = seg.forwardSegs;
		var forwardPressure = 0;
		var i, forwardSeg;

		if (seg.forwardPressure === undefined) { // not already computed

			for (i=0; i<forwardSegs.length; i++) {
				forwardSeg = forwardSegs[i];

				// figure out the child's maximum forward path
				computeSlotSegPressures(forwardSeg);

				// either use the existing maximum, or use the child's forward pressure
				// plus one (for the forwardSeg itself)
				forwardPressure = Math.max(
					forwardPressure,
					1 + forwardSeg.forwardPressure
				);
			}

			seg.forwardPressure = forwardPressure;
		}
	}


// Find all the segments in `otherSegs` that vertically collide with `seg`.
// Append into an optionally-supplied `results` array and return.
	function computeSlotSegCollisions(seg, otherSegs, results) {
		results = results || [];

		for (var i=0; i<otherSegs.length; i++) {
			if (isSlotSegCollision(seg, otherSegs[i])) {
				results.push(otherSegs[i]);
			}
		}

		return results;
	}


// Do these segments occupy the same vertical space?
	function isSlotSegCollision(seg1, seg2) {
		return seg1.bottom > seg2.top && seg1.top < seg2.bottom;
	}

	;;

	/* An abstract class from which other views inherit from
----------------------------------------------------------------------------------------------------------------------*/

	var View = FC.View = Class.extend(EmitterMixin, ListenerMixin, {

		type: null, // subclass' view name (string)
		name: null, // deprecated. use `type` instead
		title: null, // the text that will be displayed in the header's title

		calendar: null, // owner Calendar object
		options: null, // hash containing all options. already merged with view-specific-options
		el: null, // the view's containing element. set by Calendar

		isDateSet: false,
		isDateRendered: false,
		dateRenderQueue: null,

		isEventsBound: false,
		isEventsSet: false,
		isEventsRendered: false,
		eventRenderQueue: null,

		// range the view is actually displaying (moments)
		start: null,
		end: null, // exclusive

		// range the view is formally responsible for (moments)
		// may be different from start/end. for example, a month view might have 1st-31st, excluding padded dates
		intervalStart: null,
		intervalEnd: null, // exclusive
		intervalDuration: null,
		intervalUnit: null, // name of largest unit being displayed, like "month" or "week"

		isRTL: false,
		isSelected: false, // boolean whether a range of time is user-selected or not
		selectedEvent: null,

		eventOrderSpecs: null, // criteria for ordering events when they have same date/time

		// classNames styled by jqui themes
		widgetHeaderClass: null,
		widgetContentClass: null,
		highlightStateClass: null,

		// for date utils, computed from options
		nextDayThreshold: null,
		isHiddenDayHash: null,

		// now indicator
		isNowIndicatorRendered: null,
		initialNowDate: null, // result first getNow call
		initialNowQueriedMs: null, // ms time the getNow was called
		nowIndicatorTimeoutID: null, // for refresh timing of now indicator
		nowIndicatorIntervalID: null, // "


		constructor: function(calendar, type, options, intervalDuration) {

			this.calendar = calendar;
			this.type = this.name = type; // .name is deprecated
			this.options = options;
			this.intervalDuration = intervalDuration || moment.duration(1, 'day');

			this.nextDayThreshold = moment.duration(this.opt('nextDayThreshold'));
			this.initThemingProps();
			this.initHiddenDays();
			this.isRTL = this.opt('isRTL');

			this.eventOrderSpecs = parseFieldSpecs(this.opt('eventOrder'));

			this.dateRenderQueue = new TaskQueue();
			this.eventRenderQueue = new TaskQueue(this.opt('eventRenderWait'));

			this.initialize();
		},


		// A good place for subclasses to initialize member variables
		initialize: function() {
			// subclasses can implement
		},


		// Retrieves an option with the given name
		opt: function(name) {
			return this.options[name];
		},


		// Triggers handlers that are view-related. Modifies args before passing to calendar.
		publiclyTrigger: function(name, thisObj) { // arguments beyond thisObj are passed along
			var calendar = this.calendar;

			return calendar.publiclyTrigger.apply(
				calendar,
				[name, thisObj || this].concat(
					Array.prototype.slice.call(arguments, 2), // arguments beyond thisObj
					[ this ] // always make the last argument a reference to the view. TODO: deprecate
				)
			);
		},


		// Returns a proxy of the given promise that will be rejected if the given event fires
		// before the promise resolves.
		rejectOn: function(eventName, promise) {
			var _this = this;

			return new Promise(function(resolve, reject) {
				_this.one(eventName, reject);

				function cleanup() {
					_this.off(eventName, reject);
				}

				promise.then(function(res) { // success
					cleanup();
					resolve(res);
				}, function() { // failure
					cleanup();
					reject();
				});
			});
		},


		/* Date Computation
	------------------------------------------------------------------------------------------------------------------*/


		// Updates all internal dates for displaying the given unzoned range.
		setRange: function(range) {
			$.extend(this, range); // assigns every property to this object's member variables
			this.updateTitle();
		},


		// Given a single current unzoned date, produce information about what range to display.
		// Subclasses can override. Must return all properties.
		computeRange: function(date) {
			var intervalUnit = computeIntervalUnit(this.intervalDuration);
			var intervalStart = date.clone().startOf(intervalUnit);
			var intervalEnd = intervalStart.clone().add(this.intervalDuration);
			var start, end;

			// normalize the range's time-ambiguity
			if (/year|month|week|day/.test(intervalUnit)) { // whole-days?
				intervalStart.stripTime();
				intervalEnd.stripTime();
			}
			else { // needs to have a time?
				if (!intervalStart.hasTime()) {
					intervalStart = this.calendar.time(0); // give 00:00 time
				}
				if (!intervalEnd.hasTime()) {
					intervalEnd = this.calendar.time(0); // give 00:00 time
				}
			}

			start = intervalStart.clone();
			start = this.skipHiddenDays(start);
			end = intervalEnd.clone();
			end = this.skipHiddenDays(end, -1, true); // exclusively move backwards

			return {
				intervalUnit: intervalUnit,
				intervalStart: intervalStart,
				intervalEnd: intervalEnd,
				start: start,
				end: end
			};
		},


		// Computes the new date when the user hits the prev button, given the current date
		computePrevDate: function(date) {
			return this.massageCurrentDate(
				date.clone().startOf(this.intervalUnit).subtract(this.intervalDuration), -1
			);
		},


		// Computes the new date when the user hits the next button, given the current date
		computeNextDate: function(date) {
			return this.massageCurrentDate(
				date.clone().startOf(this.intervalUnit).add(this.intervalDuration)
			);
		},


		// Given an arbitrarily calculated current date of the calendar, returns a date that is ensured to be completely
		// visible. `direction` is optional and indicates which direction the current date was being
		// incremented or decremented (1 or -1).
		massageCurrentDate: function(date, direction) {
			if (this.intervalDuration.as('days') <= 1) { // if the view displays a single day or smaller
				if (this.isHiddenDay(date)) {
					date = this.skipHiddenDays(date, direction);
					date.startOf('day');
				}
			}

			return date;
		},


		/* Title and Date Formatting
	------------------------------------------------------------------------------------------------------------------*/


		// Sets the view's title property to the most updated computed value
		updateTitle: function() {
			this.title = this.computeTitle();
			this.calendar.setToolbarsTitle(this.title);
		},


		// Computes what the title at the top of the calendar should be for this view
		computeTitle: function() {
			return this.formatRange(
				{
					// in case intervalStart/End has a time, make sure timezone is correct
					start: this.calendar.applyTimezone(this.intervalStart),
					end: this.calendar.applyTimezone(this.intervalEnd)
				},
				this.opt('titleFormat') || this.computeTitleFormat(),
				this.opt('titleRangeSeparator')
			);
		},


		// Generates the format string that should be used to generate the title for the current date range.
		// Attempts to compute the most appropriate format if not explicitly specified with `titleFormat`.
		computeTitleFormat: function() {
			if (this.intervalUnit == 'year') {
				return 'YYYY';
			}
			else if (this.intervalUnit == 'month') {
				return this.opt('monthYearFormat'); // like "September 2014"
			}
			else if (this.intervalDuration.as('days') > 1) {
				return 'll'; // multi-day range. shorter, like "Sep 9 - 10 2014"
			}
			else {
				return 'LL'; // one day. longer, like "September 9 2014"
			}
		},


		// Utility for formatting a range. Accepts a range object, formatting string, and optional separator.
		// Displays all-day ranges naturally, with an inclusive end. Takes the current isRTL into account.
		// The timezones of the dates within `range` will be respected.
		formatRange: function(range, formatStr, separator) {
			var end = range.end;

			if (!end.hasTime()) { // all-day?
				end = end.clone().subtract(1); // convert to inclusive. last ms of previous day
			}

			return formatRange(range.start, end, formatStr, separator, this.opt('isRTL'));
		},


		getAllDayHtml: function() {
			return this.opt('allDayHtml') || htmlEscape(this.opt('allDayText'));
		},


		/* Navigation
	------------------------------------------------------------------------------------------------------------------*/


		// Generates HTML for an anchor to another view into the calendar.
		// Will either generate an <a> tag or a non-clickable <span> tag, depending on enabled settings.
		// `gotoOptions` can either be a moment input, or an object with the form:
		// { date, type, forceOff }
		// `type` is a view-type like "day" or "week". default value is "day".
		// `attrs` and `innerHtml` are use to generate the rest of the HTML tag.
		buildGotoAnchorHtml: function(gotoOptions, attrs, innerHtml) {
			var date, type, forceOff;
			var finalOptions;

			if ($.isPlainObject(gotoOptions)) {
				date = gotoOptions.date;
				type = gotoOptions.type;
				forceOff = gotoOptions.forceOff;
			}
			else {
				date = gotoOptions; // a single moment input
			}
			date = FC.moment(date); // if a string, parse it

			finalOptions = { // for serialization into the link
				date: date.format('YYYY-MM-DD'),
				type: type || 'day'
			};

			if (typeof attrs === 'string') {
				innerHtml = attrs;
				attrs = null;
			}

			attrs = attrs ? ' ' + attrsToStr(attrs) : ''; // will have a leading space
			innerHtml = innerHtml || '';

			if (!forceOff && this.opt('navLinks')) {
				return '<a' + attrs +
					' data-goto="' + htmlEscape(JSON.stringify(finalOptions)) + '">' +
					innerHtml +
					'</a>';
			}
			else {
				return '<span' + attrs + '>' +
					innerHtml +
					'</span>';
			}
		},


		// Rendering Non-date-related Content
		// -----------------------------------------------------------------------------------------------------------------


		// Sets the container element that the view should render inside of, does global DOM-related initializations,
		// and renders all the non-date-related content inside.
		setElement: function(el) {
			this.el = el;
			this.bindGlobalHandlers();
			this.renderSkeleton();
		},


		// Removes the view's container element from the DOM, clearing any content beforehand.
		// Undoes any other DOM-related attachments.
		removeElement: function() {
			this.unsetDate();
			this.unrenderSkeleton();

			this.unbindGlobalHandlers();

			this.el.remove();
			// NOTE: don't null-out this.el in case the View was destroyed within an API callback.
			// We don't null-out the View's other jQuery element references upon destroy,
			//  so we shouldn't kill this.el either.
		},


		// Renders the basic structure of the view before any content is rendered
		renderSkeleton: function() {
			// subclasses should implement
		},


		// Unrenders the basic structure of the view
		unrenderSkeleton: function() {
			// subclasses should implement
		},


		// Date Setting/Unsetting
		// -----------------------------------------------------------------------------------------------------------------


		setDate: function(date) {
			var isReset = this.isDateSet;

			this.isDateSet = true;
			this.handleDate(date, isReset);
			this.trigger(isReset ? 'dateReset' : 'dateSet', date);
		},


		unsetDate: function() {
			if (this.isDateSet) {
				this.isDateSet = false;
				this.handleDateUnset();
				this.trigger('dateUnset');
			}
		},


		// Date Handling
		// -----------------------------------------------------------------------------------------------------------------


		handleDate: function(date, isReset) {
			var _this = this;

			this.unbindEvents(); // will do nothing if not already bound
			this.requestDateRender(date).then(function() {
				// wish we could start earlier, but setRange/computeRange needs to execute first
				_this.bindEvents(); // will request events
			});
		},


		handleDateUnset: function() {
			this.unbindEvents();
			this.requestDateUnrender();
		},


		// Date Render Queuing
		// -----------------------------------------------------------------------------------------------------------------


		// if date not specified, uses current
		requestDateRender: function(date) {
			var _this = this;

			return this.dateRenderQueue.add(function() {
				return _this.executeDateRender(date);
			});
		},


		requestDateUnrender: function() {
			var _this = this;

			return this.dateRenderQueue.add(function() {
				return _this.executeDateUnrender();
			});
		},


		// Date High-level Rendering
		// -----------------------------------------------------------------------------------------------------------------


		// if date not specified, uses current
		executeDateRender: function(date) {
			var _this = this;

			// if rendering a new date, reset scroll to initial state (scrollTime)
			if (date) {
				this.captureInitialScroll();
			}
			else {
				this.captureScroll(); // a rerender of the current date
			}

			this.freezeHeight();

			return this.executeDateUnrender().then(function() {

				if (date) {
					_this.setRange(_this.computeRange(date));
				}

				if (_this.render) {
					_this.render(); // TODO: deprecate
				}

				_this.renderDates();
				_this.updateSize();
				_this.renderBusinessHours(); // might need coordinates, so should go after updateSize()
				_this.startNowIndicator();

				_this.thawHeight();
				_this.releaseScroll();

				_this.isDateRendered = true;
				_this.onDateRender();
				_this.trigger('dateRender');
			});
		},


		executeDateUnrender: function() {
			var _this = this;

			if (_this.isDateRendered) {
				return this.requestEventsUnrender().then(function() {

					_this.unselect();
					_this.stopNowIndicator();
					_this.triggerUnrender();
					_this.unrenderBusinessHours();
					_this.unrenderDates();

					if (_this.destroy) {
						_this.destroy(); // TODO: deprecate
					}

					_this.isDateRendered = false;
					_this.trigger('dateUnrender');
				});
			}
			else {
				return Promise.resolve();
			}
		},


		// Date Rendering Triggers
		// -----------------------------------------------------------------------------------------------------------------


		onDateRender: function() {
			this.triggerRender();
		},


		// Date Low-level Rendering
		// -----------------------------------------------------------------------------------------------------------------


		// date-cell content only
		renderDates: function() {
			// subclasses should implement
		},


		// date-cell content only
		unrenderDates: function() {
			// subclasses should override
		},


		// Misc view rendering utils
		// -------------------------


		// Signals that the view's content has been rendered
		triggerRender: function() {
			this.publiclyTrigger('viewRender', this, this, this.el);
		},


		// Signals that the view's content is about to be unrendered
		triggerUnrender: function() {
			this.publiclyTrigger('viewDestroy', this, this, this.el);
		},


		// Binds DOM handlers to elements that reside outside the view container, such as the document
		bindGlobalHandlers: function() {
			this.listenTo($(document), 'mousedown', this.handleDocumentMousedown);
			this.listenTo($(document), 'touchstart', this.processUnselect);
		},


		// Unbinds DOM handlers from elements that reside outside the view container
		unbindGlobalHandlers: function() {
			this.stopListeningTo($(document));
		},


		// Initializes internal variables related to theming
		initThemingProps: function() {
			var tm = this.opt('theme') ? 'ui' : 'fc';

			this.widgetHeaderClass = tm + '-widget-header';
			this.widgetContentClass = tm + '-widget-content';
			this.highlightStateClass = tm + '-state-highlight';
		},


		/* Business Hours
	------------------------------------------------------------------------------------------------------------------*/


		// Renders business-hours onto the view. Assumes updateSize has already been called.
		renderBusinessHours: function() {
			// subclasses should implement
		},


		// Unrenders previously-rendered business-hours
		unrenderBusinessHours: function() {
			// subclasses should implement
		},


		/* Now Indicator
	------------------------------------------------------------------------------------------------------------------*/


		// Immediately render the current time indicator and begins re-rendering it at an interval,
		// which is defined by this.getNowIndicatorUnit().
		// TODO: somehow do this for the current whole day's background too
		startNowIndicator: function() {
			var _this = this;
			var unit;
			var update;
			var delay; // ms wait value

			if (this.opt('nowIndicator')) {
				unit = this.getNowIndicatorUnit();
				if (unit) {
					update = proxy(this, 'updateNowIndicator'); // bind to `this`

					this.initialNowDate = this.calendar.getNow();
					this.initialNowQueriedMs = +new Date();
					this.renderNowIndicator(this.initialNowDate);
					this.isNowIndicatorRendered = true;

					// wait until the beginning of the next interval
					delay = this.initialNowDate.clone().startOf(unit).add(1, unit) - this.initialNowDate;
					this.nowIndicatorTimeoutID = setTimeout(function() {
						_this.nowIndicatorTimeoutID = null;
						update();
						delay = +moment.duration(1, unit);
						delay = Math.max(100, delay); // prevent too frequent
						_this.nowIndicatorIntervalID = setInterval(update, delay); // update every interval
					}, delay);
				}
			}
		},


		// rerenders the now indicator, computing the new current time from the amount of time that has passed
		// since the initial getNow call.
		updateNowIndicator: function() {
			if (this.isNowIndicatorRendered) {
				this.unrenderNowIndicator();
				this.renderNowIndicator(
					this.initialNowDate.clone().add(new Date() - this.initialNowQueriedMs) // add ms
				);
			}
		},


		// Immediately unrenders the view's current time indicator and stops any re-rendering timers.
		// Won't cause side effects if indicator isn't rendered.
		stopNowIndicator: function() {
			if (this.isNowIndicatorRendered) {

				if (this.nowIndicatorTimeoutID) {
					clearTimeout(this.nowIndicatorTimeoutID);
					this.nowIndicatorTimeoutID = null;
				}
				if (this.nowIndicatorIntervalID) {
					clearTimeout(this.nowIndicatorIntervalID);
					this.nowIndicatorIntervalID = null;
				}

				this.unrenderNowIndicator();
				this.isNowIndicatorRendered = false;
			}
		},


		// Returns a string unit, like 'second' or 'minute' that defined how often the current time indicator
		// should be refreshed. If something falsy is returned, no time indicator is rendered at all.
		getNowIndicatorUnit: function() {
			// subclasses should implement
		},


		// Renders a current time indicator at the given datetime
		renderNowIndicator: function(date) {
			// subclasses should implement
		},


		// Undoes the rendering actions from renderNowIndicator
		unrenderNowIndicator: function() {
			// subclasses should implement
		},


		/* Dimensions
	------------------------------------------------------------------------------------------------------------------*/


		// Refreshes anything dependant upon sizing of the container element of the grid
		updateSize: function(isResize) {

			if (isResize) {
				this.captureScroll();
			}

			this.updateHeight(isResize);
			this.updateWidth(isResize);
			this.updateNowIndicator();

			if (isResize) {
				this.releaseScroll();
			}
		},


		// Refreshes the horizontal dimensions of the calendar
		updateWidth: function(isResize) {
			// subclasses should implement
		},


		// Refreshes the vertical dimensions of the calendar
		updateHeight: function(isResize) {
			var calendar = this.calendar; // we poll the calendar for height information

			this.setHeight(
				calendar.getSuggestedViewHeight(),
				calendar.isHeightAuto()
			);
		},


		// Updates the vertical dimensions of the calendar to the specified height.
		// if `isAuto` is set to true, height becomes merely a suggestion and the view should use its "natural" height.
		setHeight: function(height, isAuto) {
			// subclasses should implement
		},


		/* Scroller
	------------------------------------------------------------------------------------------------------------------*/


		capturedScroll: null,
		capturedScrollDepth: 0,


		captureScroll: function() {
			if (!(this.capturedScrollDepth++)) {
				this.capturedScroll = this.isDateRendered ? this.queryScroll() : {}; // require a render first
				return true; // root?
			}
			return false;
		},


		captureInitialScroll: function(forcedScroll) {
			if (this.captureScroll()) { // root?
				this.capturedScroll.isInitial = true;

				if (forcedScroll) {
					$.extend(this.capturedScroll, forcedScroll);
				}
				else {
					this.capturedScroll.isComputed = true;
				}
			}
		},


		releaseScroll: function() {
			var scroll = this.capturedScroll;
			var isRoot = this.discardScroll();

			if (scroll.isComputed) {
				if (isRoot) {
					// only compute initial scroll if it will actually be used (is the root capture)
					$.extend(scroll, this.computeInitialScroll());
				}
				else {
					scroll = null; // scroll couldn't be computed. don't apply it to the DOM
				}
			}

			if (scroll) {
				// we act immediately on a releaseScroll operation, as opposed to captureScroll.
				// if capture/release wraps a render operation that screws up the scroll,
				// we still want to restore it a good state after, regardless of depth.

				if (scroll.isInitial) {
					this.hardSetScroll(scroll); // outsmart how browsers set scroll on initial DOM
				}
				else {
					this.setScroll(scroll);
				}
			}
		},


		discardScroll: function() {
			if (!(--this.capturedScrollDepth)) {
				this.capturedScroll = null;
				return true; // root?
			}
			return false;
		},


		computeInitialScroll: function() {
			return {};
		},


		queryScroll: function() {
			return {};
		},


		hardSetScroll: function(scroll) {
			var _this = this;
			var exec = function() { _this.setScroll(scroll); };
			exec();
			setTimeout(exec, 0); // to surely clear the browser's initial scroll for the DOM
		},


		setScroll: function(scroll) {
		},


		/* Height Freezing
	------------------------------------------------------------------------------------------------------------------*/


		freezeHeight: function() {
			this.calendar.freezeContentHeight();
		},


		thawHeight: function() {
			this.calendar.thawContentHeight();
		},


		// Event Binding/Unbinding
		// -----------------------------------------------------------------------------------------------------------------


		bindEvents: function() {
			var _this = this;

			if (!this.isEventsBound) {
				this.isEventsBound = true;
				this.rejectOn('eventsUnbind', this.requestEvents()).then(function(events) { // TODO: test rejection
					_this.listenTo(_this.calendar, 'eventsReset', _this.setEvents);
					_this.setEvents(events);
				});
			}
		},


		unbindEvents: function() {
			if (this.isEventsBound) {
				this.isEventsBound = false;
				this.stopListeningTo(this.calendar, 'eventsReset');
				this.unsetEvents();
				this.trigger('eventsUnbind');
			}
		},


		// Event Setting/Unsetting
		// -----------------------------------------------------------------------------------------------------------------


		setEvents: function(events) {
			var isReset = this.isEventSet;

			this.isEventsSet = true;
			this.handleEvents(events, isReset);
			this.trigger(isReset ? 'eventsReset' : 'eventsSet', events);
		},


		unsetEvents: function() {
			if (this.isEventsSet) {
				this.isEventsSet = false;
				this.handleEventsUnset();
				this.trigger('eventsUnset');
			}
		},


		whenEventsSet: function() {
			var _this = this;

			if (this.isEventsSet) {
				return Promise.resolve(this.getCurrentEvents());
			}
			else {
				return new Promise(function(resolve) {
					_this.one('eventsSet', resolve);
				});
			}
		},


		// Event Handling
		// -----------------------------------------------------------------------------------------------------------------


		handleEvents: function(events, isReset) {
			this.requestEventsRender(events);
		},


		handleEventsUnset: function() {
			this.requestEventsUnrender();
		},


		// Event Render Queuing
		// -----------------------------------------------------------------------------------------------------------------


		// assumes any previous event renders have been cleared already
		requestEventsRender: function(events) {
			var _this = this;

			return this.eventRenderQueue.add(function() { // might not return a promise if debounced!? bad
				return _this.executeEventsRender(events);
			});
		},


		requestEventsUnrender: function() {
			var _this = this;

			if (this.isEventsRendered) {
				return this.eventRenderQueue.addQuickly(function() {
					return _this.executeEventsUnrender();
				});
			}
			else {
				return Promise.resolve();
			}
		},


		requestCurrentEventsRender: function() {
			if (this.isEventsSet) {
				this.requestEventsRender(this.getCurrentEvents());
			}
			else {
				return Promise.reject();
			}
		},


		// Event High-level Rendering
		// -----------------------------------------------------------------------------------------------------------------


		executeEventsRender: function(events) {
			var _this = this;

			this.captureScroll();
			this.freezeHeight();

			return this.executeEventsUnrender().then(function() {
				_this.renderEvents(events);

				_this.thawHeight();
				_this.releaseScroll();

				_this.isEventsRendered = true;
				_this.onEventsRender();
				_this.trigger('eventsRender');
			});
		},


		executeEventsUnrender: function() {
			if (this.isEventsRendered) {
				this.onBeforeEventsUnrender();

				this.captureScroll();
				this.freezeHeight();

				if (this.destroyEvents) {
					this.destroyEvents(); // TODO: deprecate
				}

				this.unrenderEvents();

				this.thawHeight();
				this.releaseScroll();

				this.isEventsRendered = false;
				this.trigger('eventsUnrender');
			}

			return Promise.resolve(); // always synchronous
		},


		// Event Rendering Triggers
		// -----------------------------------------------------------------------------------------------------------------


		// Signals that all events have been rendered
		onEventsRender: function() {
			this.renderedEventSegEach(function(seg) {
				this.publiclyTrigger('eventAfterRender', seg.event, seg.event, seg.el);
			});
			this.publiclyTrigger('eventAfterAllRender');
		},


		// Signals that all event elements are about to be removed
		onBeforeEventsUnrender: function() {
			this.renderedEventSegEach(function(seg) {
				this.publiclyTrigger('eventDestroy', seg.event, seg.event, seg.el);
			});
		},


		// Event Low-level Rendering
		// -----------------------------------------------------------------------------------------------------------------


		// Renders the events onto the view.
		renderEvents: function(events) {
			// subclasses should implement
		},


		// Removes event elements from the view.
		unrenderEvents: function() {
			// subclasses should implement
		},


		// Event Data Access
		// -----------------------------------------------------------------------------------------------------------------


		requestEvents: function() {
			return this.calendar.requestEvents(this.start, this.end);
		},


		getCurrentEvents: function() {
			return this.calendar.getPrunedEventCache();
		},


		// Event Rendering Utils
		// -----------------------------------------------------------------------------------------------------------------


		// Given an event and the default element used for rendering, returns the element that should actually be used.
		// Basically runs events and elements through the eventRender hook.
		resolveEventEl: function(event, el) {
			var custom = this.publiclyTrigger('eventRender', event, event, el);

			if (custom === false) { // means don't render at all
				el = null;
			}
			else if (custom && custom !== true) {
				el = $(custom);
			}

			return el;
		},


		// Hides all rendered event segments linked to the given event
		showEvent: function(event) {
			this.renderedEventSegEach(function(seg) {
				seg.el.css('visibility', '');
			}, event);
		},


		// Shows all rendered event segments linked to the given event
		hideEvent: function(event) {
			this.renderedEventSegEach(function(seg) {
				seg.el.css('visibility', 'hidden');
			}, event);
		},


		// Iterates through event segments that have been rendered (have an el). Goes through all by default.
		// If the optional `event` argument is specified, only iterates through segments linked to that event.
		// The `this` value of the callback function will be the view.
		renderedEventSegEach: function(func, event) {
			var segs = this.getEventSegs();
			var i;

			for (i = 0; i < segs.length; i++) {
				if (!event || segs[i].event._id === event._id) {
					if (segs[i].el) {
						func.call(this, segs[i]);
					}
				}
			}
		},


		// Retrieves all the rendered segment objects for the view
		getEventSegs: function() {
			// subclasses must implement
			return [];
		},


		/* Event Drag-n-Drop
	------------------------------------------------------------------------------------------------------------------*/


		// Computes if the given event is allowed to be dragged by the user
		isEventDraggable: function(event) {
			return this.isEventStartEditable(event);
		},


		isEventStartEditable: function(event) {
			return firstDefined(
				event.startEditable,
				(event.source || {}).startEditable,
				this.opt('eventStartEditable'),
				this.isEventGenerallyEditable(event)
			);
		},


		isEventGenerallyEditable: function(event) {
			return firstDefined(
				event.editable,
				(event.source || {}).editable,
				this.opt('editable')
			);
		},


		// Must be called when an event in the view is dropped onto new location.
		// `dropLocation` is an object that contains the new zoned start/end/allDay values for the event.
		reportEventDrop: function(event, dropLocation, largeUnit, el, ev) {
			var calendar = this.calendar;
			var mutateResult = calendar.mutateEvent(event, dropLocation, largeUnit);
			var undoFunc = function() {
				mutateResult.undo();
				calendar.reportEventChange();
			};

			this.triggerEventDrop(event, mutateResult.dateDelta, undoFunc, el, ev);
			calendar.reportEventChange(); // will rerender events
		},


		// Triggers event-drop handlers that have subscribed via the API
		triggerEventDrop: function(event, dateDelta, undoFunc, el, ev) {
			this.publiclyTrigger('eventDrop', el[0], event, dateDelta, undoFunc, ev, {}); // {} = jqui dummy
		},


		/* External Element Drag-n-Drop
	------------------------------------------------------------------------------------------------------------------*/


		// Must be called when an external element, via jQuery UI, has been dropped onto the calendar.
		// `meta` is the parsed data that has been embedded into the dragging event.
		// `dropLocation` is an object that contains the new zoned start/end/allDay values for the event.
		reportExternalDrop: function(meta, dropLocation, el, ev, ui) {
			var eventProps = meta.eventProps;
			var eventInput;
			var event;

			// Try to build an event object and render it. TODO: decouple the two
			if (eventProps) {
				eventInput = $.extend({}, eventProps, dropLocation);
				event = this.calendar.renderEvent(eventInput, meta.stick)[0]; // renderEvent returns an array
			}

			this.triggerExternalDrop(event, dropLocation, el, ev, ui);
		},


		// Triggers external-drop handlers that have subscribed via the API
		triggerExternalDrop: function(event, dropLocation, el, ev, ui) {

			// trigger 'drop' regardless of whether element represents an event
			this.publiclyTrigger('drop', el[0], dropLocation.start, ev, ui);

			if (event) {
				this.publiclyTrigger('eventReceive', null, event); // signal an external event landed
			}
		},


		/* Drag-n-Drop Rendering (for both events and external elements)
	------------------------------------------------------------------------------------------------------------------*/


		// Renders a visual indication of a event or external-element drag over the given drop zone.
		// If an external-element, seg will be `null`.
		// Must return elements used for any mock events.
		renderDrag: function(dropLocation, seg) {
			// subclasses must implement
		},


		// Unrenders a visual indication of an event or external-element being dragged.
		unrenderDrag: function() {
			// subclasses must implement
		},


		/* Event Resizing
	------------------------------------------------------------------------------------------------------------------*/


		// Computes if the given event is allowed to be resized from its starting edge
		isEventResizableFromStart: function(event) {
			return this.opt('eventResizableFromStart') && this.isEventResizable(event);
		},


		// Computes if the given event is allowed to be resized from its ending edge
		isEventResizableFromEnd: function(event) {
			return this.isEventResizable(event);
		},


		// Computes if the given event is allowed to be resized by the user at all
		isEventResizable: function(event) {
			var source = event.source || {};

			return firstDefined(
				event.durationEditable,
				source.durationEditable,
				this.opt('eventDurationEditable'),
				event.editable,
				source.editable,
				this.opt('editable')
			);
		},


		// Must be called when an event in the view has been resized to a new length
		reportEventResize: function(event, resizeLocation, largeUnit, el, ev) {
			var calendar = this.calendar;
			var mutateResult = calendar.mutateEvent(event, resizeLocation, largeUnit);
			var undoFunc = function() {
				mutateResult.undo();
				calendar.reportEventChange();
			};

			this.triggerEventResize(event, mutateResult.durationDelta, undoFunc, el, ev);
			calendar.reportEventChange(); // will rerender events
		},


		// Triggers event-resize handlers that have subscribed via the API
		triggerEventResize: function(event, durationDelta, undoFunc, el, ev) {
			this.publiclyTrigger('eventResize', el[0], event, durationDelta, undoFunc, ev, {}); // {} = jqui dummy
		},


		/* Selection (time range)
	------------------------------------------------------------------------------------------------------------------*/


		// Selects a date span on the view. `start` and `end` are both Moments.
		// `ev` is the native mouse event that begin the interaction.
		select: function(span, ev) {
			this.unselect(ev);
			this.renderSelection(span);
			this.reportSelection(span, ev);
		},


		// Renders a visual indication of the selection
		renderSelection: function(span) {
			// subclasses should implement
		},


		// Called when a new selection is made. Updates internal state and triggers handlers.
		reportSelection: function(span, ev) {
			this.isSelected = true;
			this.triggerSelect(span, ev);
		},


		// Triggers handlers to 'select'
		triggerSelect: function(span, ev) {
			this.publiclyTrigger(
				'select',
				null,
				this.calendar.applyTimezone(span.start), // convert to calendar's tz for external API
				this.calendar.applyTimezone(span.end), // "
				ev
			);
		},


		// Undoes a selection. updates in the internal state and triggers handlers.
		// `ev` is the native mouse event that began the interaction.
		unselect: function(ev) {
			if (this.isSelected) {
				this.isSelected = false;
				if (this.destroySelection) {
					this.destroySelection(); // TODO: deprecate
				}
				this.unrenderSelection();
				this.publiclyTrigger('unselect', null, ev);
			}
		},


		// Unrenders a visual indication of selection
		unrenderSelection: function() {
			// subclasses should implement
		},


		/* Event Selection
	------------------------------------------------------------------------------------------------------------------*/


		selectEvent: function(event) {
			if (!this.selectedEvent || this.selectedEvent !== event) {
				this.unselectEvent();
				this.renderedEventSegEach(function(seg) {
					seg.el.addClass('fc-selected');
				}, event);
				this.selectedEvent = event;
			}
		},


		unselectEvent: function() {
			if (this.selectedEvent) {
				this.renderedEventSegEach(function(seg) {
					seg.el.removeClass('fc-selected');
				}, this.selectedEvent);
				this.selectedEvent = null;
			}
		},


		isEventSelected: function(event) {
			// event references might change on refetchEvents(), while selectedEvent doesn't,
			// so compare IDs
			return this.selectedEvent && this.selectedEvent._id === event._id;
		},


		/* Mouse / Touch Unselecting (time range & event unselection)
	------------------------------------------------------------------------------------------------------------------*/
		// TODO: move consistently to down/start or up/end?
		// TODO: don't kill previous selection if touch scrolling


		handleDocumentMousedown: function(ev) {
			if (isPrimaryMouseButton(ev)) {
				this.processUnselect(ev);
			}
		},


		processUnselect: function(ev) {
			this.processRangeUnselect(ev);
			this.processEventUnselect(ev);
		},


		processRangeUnselect: function(ev) {
			var ignore;

			// is there a time-range selection?
			if (this.isSelected && this.opt('unselectAuto')) {
				// only unselect if the clicked element is not identical to or inside of an 'unselectCancel' element
				ignore = this.opt('unselectCancel');
				if (!ignore || !$(ev.target).closest(ignore).length) {
					this.unselect(ev);
				}
			}
		},


		processEventUnselect: function(ev) {
			if (this.selectedEvent) {
				if (!$(ev.target).closest('.fc-selected').length) {
					this.unselectEvent();
				}
			}
		},


		/* Day Click
	------------------------------------------------------------------------------------------------------------------*/


		// Triggers handlers to 'dayClick'
		// Span has start/end of the clicked area. Only the start is useful.
		triggerDayClick: function(span, dayEl, ev) {
			this.publiclyTrigger(
				'dayClick',
				dayEl,
				this.calendar.applyTimezone(span.start), // convert to calendar's timezone for external API
				ev
			);
		},


		/* Date Utils
	------------------------------------------------------------------------------------------------------------------*/


		// Initializes internal variables related to calculating hidden days-of-week
		initHiddenDays: function() {
			var hiddenDays = this.opt('hiddenDays') || []; // array of day-of-week indices that are hidden
			var isHiddenDayHash = []; // is the day-of-week hidden? (hash with day-of-week-index -> bool)
			var dayCnt = 0;
			var i;

			if (this.opt('weekends') === false) {
				hiddenDays.push(0, 6); // 0=sunday, 6=saturday
			}

			for (i = 0; i < 7; i++) {
				if (
					!(isHiddenDayHash[i] = $.inArray(i, hiddenDays) !== -1)
				) {
					dayCnt++;
				}
			}

			if (!dayCnt) {
				throw 'invalid hiddenDays'; // all days were hidden? bad.
			}

			this.isHiddenDayHash = isHiddenDayHash;
		},


		// Is the current day hidden?
		// `day` is a day-of-week index (0-6), or a Moment
		isHiddenDay: function(day) {
			if (moment.isMoment(day)) {
				day = day.day();
			}
			return this.isHiddenDayHash[day];
		},


		// Incrementing the current day until it is no longer a hidden day, returning a copy.
		// If the initial value of `date` is not a hidden day, don't do anything.
		// Pass `isExclusive` as `true` if you are dealing with an end date.
		// `inc` defaults to `1` (increment one day forward each time)
		skipHiddenDays: function(date, inc, isExclusive) {
			var out = date.clone();
			inc = inc || 1;
			while (
				this.isHiddenDayHash[(out.day() + (isExclusive ? inc : 0) + 7) % 7]
				) {
				out.add(inc, 'days');
			}
			return out;
		},


		// Returns the date range of the full days the given range visually appears to occupy.
		// Returns a new range object.
		computeDayRange: function(range) {
			var startDay = range.start.clone().stripTime(); // the beginning of the day the range starts
			var end = range.end;
			var endDay = null;
			var endTimeMS;

			if (end) {
				endDay = end.clone().stripTime(); // the beginning of the day the range exclusively ends
				endTimeMS = +end.time(); // # of milliseconds into `endDay`

				// If the end time is actually inclusively part of the next day and is equal to or
				// beyond the next day threshold, adjust the end to be the exclusive end of `endDay`.
				// Otherwise, leaving it as inclusive will cause it to exclude `endDay`.
				if (endTimeMS && endTimeMS >= this.nextDayThreshold) {
					endDay.add(1, 'days');
				}
			}

			// If no end was specified, or if it is within `startDay` but not past nextDayThreshold,
			// assign the default duration of one day.
			if (!end || endDay <= startDay) {
				endDay = startDay.clone().add(1, 'days');
			}

			return { start: startDay, end: endDay };
		},


		// Does the given event visually appear to occupy more than one day?
		isMultiDayEvent: function(event) {
			var range = this.computeDayRange(event); // event is range-ish

			return range.end.diff(range.start, 'days') > 1;
		}

	});

	;;

	/*
Embodies a div that has potential scrollbars
*/
	var Scroller = FC.Scroller = Class.extend({

		el: null, // the guaranteed outer element
		scrollEl: null, // the element with the scrollbars
		overflowX: null,
		overflowY: null,


		constructor: function(options) {
			options = options || {};
			this.overflowX = options.overflowX || options.overflow || 'auto';
			this.overflowY = options.overflowY || options.overflow || 'auto';
		},


		render: function() {
			this.el = this.renderEl();
			this.applyOverflow();
		},


		renderEl: function() {
			return (this.scrollEl = $('<div class="fc-scroller"></div>'));
		},


		// sets to natural height, unlocks overflow
		clear: function() {
			this.setHeight('auto');
			this.applyOverflow();
		},


		destroy: function() {
			this.el.remove();
		},


		// Overflow
		// -----------------------------------------------------------------------------------------------------------------


		applyOverflow: function() {
			this.scrollEl.css({
				'overflow-x': this.overflowX,
				'overflow-y': this.overflowY
			});
		},


		// Causes any 'auto' overflow values to resolves to 'scroll' or 'hidden'.
		// Useful for preserving scrollbar widths regardless of future resizes.
		// Can pass in scrollbarWidths for optimization.
		lockOverflow: function(scrollbarWidths) {
			var overflowX = this.overflowX;
			var overflowY = this.overflowY;

			scrollbarWidths = scrollbarWidths || this.getScrollbarWidths();

			if (overflowX === 'auto') {
				overflowX = (
					scrollbarWidths.top || scrollbarWidths.bottom || // horizontal scrollbars?
					// OR scrolling pane with massless scrollbars?
					this.scrollEl[0].scrollWidth - 1 > this.scrollEl[0].clientWidth
					// subtract 1 because of IE off-by-one issue
				) ? 'scroll' : 'hidden';
			}

			if (overflowY === 'auto') {
				overflowY = (
					scrollbarWidths.left || scrollbarWidths.right || // vertical scrollbars?
					// OR scrolling pane with massless scrollbars?
					this.scrollEl[0].scrollHeight - 1 > this.scrollEl[0].clientHeight
					// subtract 1 because of IE off-by-one issue
				) ? 'scroll' : 'hidden';
			}

			this.scrollEl.css({ 'overflow-x': overflowX, 'overflow-y': overflowY });
		},


		// Getters / Setters
		// -----------------------------------------------------------------------------------------------------------------


		setHeight: function(height) {
			this.scrollEl.height(height);
		},


		getScrollTop: function() {
			return this.scrollEl.scrollTop();
		},


		setScrollTop: function(top) {
			this.scrollEl.scrollTop(top);
		},


		getClientWidth: function() {
			return this.scrollEl[0].clientWidth;
		},


		getClientHeight: function() {
			return this.scrollEl[0].clientHeight;
		},


		getScrollbarWidths: function() {
			return getScrollbarWidths(this.scrollEl);
		}

	});

	;;
	function Iterator(items) {
		this.items = items || [];
	}


	/* Calls a method on every item passing the arguments through */
	Iterator.prototype.proxyCall = function(methodName) {
		var args = Array.prototype.slice.call(arguments, 1);
		var results = [];

		this.items.forEach(function(item) {
			results.push(item[methodName].apply(item, args));
		});

		return results;
	};

	;;

	/* Toolbar with buttons and title
----------------------------------------------------------------------------------------------------------------------*/

	function Toolbar(calendar, toolbarOptions) {
		var t = this;

		// exports
		t.setToolbarOptions = setToolbarOptions;
		t.render = render;
		t.removeElement = removeElement;
		t.updateTitle = updateTitle;
		t.activateButton = activateButton;
		t.deactivateButton = deactivateButton;
		t.disableButton = disableButton;
		t.enableButton = enableButton;
		t.getViewsWithButtons = getViewsWithButtons;
		t.el = null; // mirrors local `el`

		// locals
		var el;
		var viewsWithButtons = [];
		var tm;

		// method to update toolbar-specific options, not calendar-wide options
		function setToolbarOptions(newToolbarOptions) {
			toolbarOptions = newToolbarOptions;
		}

		// can be called repeatedly and will rerender
		function render() {
			var sections = toolbarOptions.layout;

			tm = calendar.options.theme ? 'ui' : 'fc';

			if (sections) {
				if (!el) {
					el = this.el = $("<div class='fc-toolbar "+ toolbarOptions.extraClasses + "'/>");
				}
				else {
					el.empty();
				}
				el.append(renderSection('left'))
					.append(renderSection('right'))
					.append(renderSection('center'))
					.append('<div class="fc-clear"/>');
			}
			else {
				removeElement();
			}
		}


		function removeElement() {
			if (el) {
				el.remove();
				el = t.el = null;
			}
		}


		function renderSection(position) {
			var sectionEl = $('<div class="fc-' + position + '"/>');
			var buttonStr = toolbarOptions.layout[position];

			if (buttonStr) {
				$.each(buttonStr.split(' '), function(i) {
					var groupChildren = $();
					var isOnlyButtons = true;
					var groupEl;

					$.each(this.split(','), function(j, buttonName) {
						var customButtonProps;
						var viewSpec;
						var buttonClick;
						var overrideText; // text explicitly set by calendar's constructor options. overcomes icons
						var defaultText;
						var themeIcon;
						var normalIcon;
						var innerHtml;
						var classes;
						var button; // the element

						if (buttonName == 'title') {
							groupChildren = groupChildren.add($('<h2>&nbsp;</h2>')); // we always want it to take up height
							isOnlyButtons = false;
						}
						else {
							if ((customButtonProps = (calendar.options.customButtons || {})[buttonName])) {
								buttonClick = function(ev) {
									if (customButtonProps.click) {
										customButtonProps.click.call(button[0], ev);
									}
								};
								overrideText = ''; // icons will override text
								defaultText = customButtonProps.text;
							}
							else if ((viewSpec = calendar.getViewSpec(buttonName))) {
								buttonClick = function() {
									calendar.changeView(buttonName);
								};
								viewsWithButtons.push(buttonName);
								overrideText = viewSpec.buttonTextOverride;
								defaultText = viewSpec.buttonTextDefault;
							}
							else if (calendar[buttonName]) { // a calendar method
								buttonClick = function() {
									calendar[buttonName]();
								};
								overrideText = (calendar.overrides.buttonText || {})[buttonName];
								defaultText = calendar.options.buttonText[buttonName]; // everything else is considered default
							}

							if (buttonClick) {

								themeIcon =
									customButtonProps ?
										customButtonProps.themeIcon :
										calendar.options.themeButtonIcons[buttonName];

								normalIcon =
									customButtonProps ?
										customButtonProps.icon :
										calendar.options.buttonIcons[buttonName];

								if (overrideText) {
									innerHtml = htmlEscape(overrideText);
								}
								else if (themeIcon && calendar.options.theme) {
									innerHtml = "<span class='ui-icon ui-icon-" + themeIcon + "'></span>";
								}
								else if (normalIcon && !calendar.options.theme) {
									innerHtml = "<span class='fc-icon fc-icon-" + normalIcon + "'></span>";
								}
								else {
									innerHtml = htmlEscape(defaultText);
								}

								classes = [
									'fc-' + buttonName + '-button',
									tm + '-button',
									tm + '-state-default'
								];

								button = $( // type="button" so that it doesn't submit a form
									'<button type="button" class="' + classes.join(' ') + '">' +
									innerHtml +
									'</button>'
								)
									.click(function(ev) {
										// don't process clicks for disabled buttons
										if (!button.hasClass(tm + '-state-disabled')) {

											buttonClick(ev);

											// after the click action, if the button becomes the "active" tab, or disabled,
											// it should never have a hover class, so remove it now.
											if (
												button.hasClass(tm + '-state-active') ||
												button.hasClass(tm + '-state-disabled')
											) {
												button.removeClass(tm + '-state-hover');
											}
										}
									})
									.mousedown(function() {
										// the *down* effect (mouse pressed in).
										// only on buttons that are not the "active" tab, or disabled
										button
											.not('.' + tm + '-state-active')
											.not('.' + tm + '-state-disabled')
											.addClass(tm + '-state-down');
									})
									.mouseup(function() {
										// undo the *down* effect
										button.removeClass(tm + '-state-down');
									})
									.hover(
										function() {
											// the *hover* effect.
											// only on buttons that are not the "active" tab, or disabled
											button
												.not('.' + tm + '-state-active')
												.not('.' + tm + '-state-disabled')
												.addClass(tm + '-state-hover');
										},
										function() {
											// undo the *hover* effect
											button
												.removeClass(tm + '-state-hover')
												.removeClass(tm + '-state-down'); // if mouseleave happens before mouseup
										}
									);

								groupChildren = groupChildren.add(button);
							}
						}
					});

					if (isOnlyButtons) {
						groupChildren
							.first().addClass(tm + '-corner-left').end()
							.last().addClass(tm + '-corner-right').end();
					}

					if (groupChildren.length > 1) {
						groupEl = $('<div/>');
						if (isOnlyButtons) {
							groupEl.addClass('fc-button-group');
						}
						groupEl.append(groupChildren);
						sectionEl.append(groupEl);
					}
					else {
						sectionEl.append(groupChildren); // 1 or 0 children
					}
				});
			}

			return sectionEl;
		}


		function updateTitle(text) {
			if (el) {
				el.find('h2').text(text);
			}
		}


		function activateButton(buttonName) {
			if (el) {
				el.find('.fc-' + buttonName + '-button')
					.addClass(tm + '-state-active');
			}
		}


		function deactivateButton(buttonName) {
			if (el) {
				el.find('.fc-' + buttonName + '-button')
					.removeClass(tm + '-state-active');
			}
		}


		function disableButton(buttonName) {
			if (el) {
				el.find('.fc-' + buttonName + '-button')
					.prop('disabled', true)
					.addClass(tm + '-state-disabled');
			}
		}


		function enableButton(buttonName) {
			if (el) {
				el.find('.fc-' + buttonName + '-button')
					.prop('disabled', false)
					.removeClass(tm + '-state-disabled');
			}
		}


		function getViewsWithButtons() {
			return viewsWithButtons;
		}

	}

	;;

	var Calendar = FC.Calendar = Class.extend({

		dirDefaults: null, // option defaults related to LTR or RTL
		localeDefaults: null, // option defaults related to current locale
		overrides: null, // option overrides given to the fullCalendar constructor
		dynamicOverrides: null, // options set with dynamic setter method. higher precedence than view overrides.
		options: null, // all defaults combined with overrides
		viewSpecCache: null, // cache of view definitions
		view: null, // current View object
		header: null,
		footer: null,
		loadingLevel: 0, // number of simultaneous loading tasks


		// a lot of this class' OOP logic is scoped within this constructor function,
		// but in the future, write individual methods on the prototype.
		constructor: Calendar_constructor,


		// Subclasses can override this for initialization logic after the constructor has been called
		initialize: function() {
		},


		// Computes the flattened options hash for the calendar and assigns to `this.options`.
		// Assumes this.overrides and this.dynamicOverrides have already been initialized.
		populateOptionsHash: function() {
			var locale, localeDefaults;
			var isRTL, dirDefaults;

			locale = firstDefined( // explicit locale option given?
				this.dynamicOverrides.locale,
				this.overrides.locale
			);
			localeDefaults = localeOptionHash[locale];
			if (!localeDefaults) { // explicit locale option not given or invalid?
				locale = Calendar.defaults.locale;
				localeDefaults = localeOptionHash[locale] || {};
			}

			isRTL = firstDefined( // based on options computed so far, is direction RTL?
				this.dynamicOverrides.isRTL,
				this.overrides.isRTL,
				localeDefaults.isRTL,
				Calendar.defaults.isRTL
			);
			dirDefaults = isRTL ? Calendar.rtlDefaults : {};

			this.dirDefaults = dirDefaults;
			this.localeDefaults = localeDefaults;
			this.options = mergeOptions([ // merge defaults and overrides. lowest to highest precedence
				Calendar.defaults, // global defaults
				dirDefaults,
				localeDefaults,
				this.overrides,
				this.dynamicOverrides
			]);
			populateInstanceComputableOptions(this.options); // fill in gaps with computed options
		},


		// Gets information about how to create a view. Will use a cache.
		getViewSpec: function(viewType) {
			var cache = this.viewSpecCache;

			return cache[viewType] || (cache[viewType] = this.buildViewSpec(viewType));
		},


		// Given a duration singular unit, like "week" or "day", finds a matching view spec.
		// Preference is given to views that have corresponding buttons.
		getUnitViewSpec: function(unit) {
			var viewTypes;
			var i;
			var spec;

			if ($.inArray(unit, intervalUnits) != -1) {

				// put views that have buttons first. there will be duplicates, but oh well
				viewTypes = this.header.getViewsWithButtons(); // TODO: include footer as well?
				$.each(FC.views, function(viewType) { // all views
					viewTypes.push(viewType);
				});

				for (i = 0; i < viewTypes.length; i++) {
					spec = this.getViewSpec(viewTypes[i]);
					if (spec) {
						if (spec.singleUnit == unit) {
							return spec;
						}
					}
				}
			}
		},


		// Builds an object with information on how to create a given view
		buildViewSpec: function(requestedViewType) {
			var viewOverrides = this.overrides.views || {};
			var specChain = []; // for the view. lowest to highest priority
			var defaultsChain = []; // for the view. lowest to highest priority
			var overridesChain = []; // for the view. lowest to highest priority
			var viewType = requestedViewType;
			var spec; // for the view
			var overrides; // for the view
			var duration;
			var unit;

			// iterate from the specific view definition to a more general one until we hit an actual View class
			while (viewType) {
				spec = fcViews[viewType];
				overrides = viewOverrides[viewType];
				viewType = null; // clear. might repopulate for another iteration

				if (typeof spec === 'function') { // TODO: deprecate
					spec = { 'class': spec };
				}

				if (spec) {
					specChain.unshift(spec);
					defaultsChain.unshift(spec.defaults || {});
					duration = duration || spec.duration;
					viewType = viewType || spec.type;
				}

				if (overrides) {
					overridesChain.unshift(overrides); // view-specific option hashes have options at zero-level
					duration = duration || overrides.duration;
					viewType = viewType || overrides.type;
				}
			}

			spec = mergeProps(specChain);
			spec.type = requestedViewType;
			if (!spec['class']) {
				return false;
			}

			if (duration) {
				duration = moment.duration(duration);
				if (duration.valueOf()) { // valid?
					spec.duration = duration;
					unit = computeIntervalUnit(duration);

					// view is a single-unit duration, like "week" or "day"
					// incorporate options for this. lowest priority
					if (duration.as(unit) === 1) {
						spec.singleUnit = unit;
						overridesChain.unshift(viewOverrides[unit] || {});
					}
				}
			}

			spec.defaults = mergeOptions(defaultsChain);
			spec.overrides = mergeOptions(overridesChain);

			this.buildViewSpecOptions(spec);
			this.buildViewSpecButtonText(spec, requestedViewType);

			return spec;
		},


		// Builds and assigns a view spec's options object from its already-assigned defaults and overrides
		buildViewSpecOptions: function(spec) {
			spec.options = mergeOptions([ // lowest to highest priority
				Calendar.defaults, // global defaults
				spec.defaults, // view's defaults (from ViewSubclass.defaults)
				this.dirDefaults,
				this.localeDefaults, // locale and dir take precedence over view's defaults!
				this.overrides, // calendar's overrides (options given to constructor)
				spec.overrides, // view's overrides (view-specific options)
				this.dynamicOverrides // dynamically set via setter. highest precedence
			]);
			populateInstanceComputableOptions(spec.options);
		},


		// Computes and assigns a view spec's buttonText-related options
		buildViewSpecButtonText: function(spec, requestedViewType) {

			// given an options object with a possible `buttonText` hash, lookup the buttonText for the
			// requested view, falling back to a generic unit entry like "week" or "day"
			function queryButtonText(options) {
				var buttonText = options.buttonText || {};
				return buttonText[requestedViewType] ||
					// view can decide to look up a certain key
					(spec.buttonTextKey ? buttonText[spec.buttonTextKey] : null) ||
					// a key like "month"
					(spec.singleUnit ? buttonText[spec.singleUnit] : null);
			}

			// highest to lowest priority
			spec.buttonTextOverride =
				queryButtonText(this.dynamicOverrides) ||
				queryButtonText(this.overrides) || // constructor-specified buttonText lookup hash takes precedence
				spec.overrides.buttonText; // `buttonText` for view-specific options is a string

			// highest to lowest priority. mirrors buildViewSpecOptions
			spec.buttonTextDefault =
				queryButtonText(this.localeDefaults) ||
				queryButtonText(this.dirDefaults) ||
				spec.defaults.buttonText || // a single string. from ViewSubclass.defaults
				queryButtonText(Calendar.defaults) ||
				(spec.duration ? this.humanizeDuration(spec.duration) : null) || // like "3 days"
				requestedViewType; // fall back to given view name
		},


		// Given a view name for a custom view or a standard view, creates a ready-to-go View object
		instantiateView: function(viewType) {
			var spec = this.getViewSpec(viewType);

			return new spec['class'](this, viewType, spec.options, spec.duration);
		},


		// Returns a boolean about whether the view is okay to instantiate at some point
		isValidViewType: function(viewType) {
			return Boolean(this.getViewSpec(viewType));
		},


		// Should be called when any type of async data fetching begins
		pushLoading: function() {
			if (!(this.loadingLevel++)) {
				this.publiclyTrigger('loading', null, true, this.view);
			}
		},


		// Should be called when any type of async data fetching completes
		popLoading: function() {
			if (!(--this.loadingLevel)) {
				this.publiclyTrigger('loading', null, false, this.view);
			}
		},


		// Given arguments to the select method in the API, returns a span (unzoned start/end and other info)
		buildSelectSpan: function(zonedStartInput, zonedEndInput) {
			var start = this.moment(zonedStartInput).stripZone();
			var end;

			if (zonedEndInput) {
				end = this.moment(zonedEndInput).stripZone();
			}
			else if (start.hasTime()) {
				end = start.clone().add(this.defaultTimedEventDuration);
			}
			else {
				end = start.clone().add(this.defaultAllDayEventDuration);
			}

			return { start: start, end: end };
		}

	});


	Calendar.mixin(EmitterMixin);


	function Calendar_constructor(element, overrides) {
		var t = this;


		// Exports
		// -----------------------------------------------------------------------------------

		t.render = render;
		t.destroy = destroy;
		t.rerenderEvents = rerenderEvents;
		t.changeView = renderView; // `renderView` will switch to another view
		t.select = select;
		t.unselect = unselect;
		t.prev = prev;
		t.next = next;
		t.prevYear = prevYear;
		t.nextYear = nextYear;
		t.today = today;
		t.gotoDate = gotoDate;
		t.incrementDate = incrementDate;
		t.zoomTo = zoomTo;
		t.getDate = getDate;
		t.getCalendar = getCalendar;
		t.getView = getView;
		t.option = option; // getter/setter method
		t.publiclyTrigger = publiclyTrigger;


		// Options
		// -----------------------------------------------------------------------------------

		t.dynamicOverrides = {};
		t.viewSpecCache = {};
		t.optionHandlers = {}; // for Calendar.options.js
		t.overrides = $.extend({}, overrides); // make a copy

		t.populateOptionsHash(); // sets this.options



		// Locale-data Internals
		// -----------------------------------------------------------------------------------
		// Apply overrides to the current locale's data

		var localeData;

		// Called immediately, and when any of the options change.
		// Happens before any internal objects rebuild or rerender, because this is very core.
		t.bindOptions([
			'locale', 'monthNames', 'monthNamesShort', 'dayNames', 'dayNamesShort', 'firstDay', 'weekNumberCalculation'
		], function(locale, monthNames, monthNamesShort, dayNames, dayNamesShort, firstDay, weekNumberCalculation) {

			// normalize
			if (weekNumberCalculation === 'iso') {
				weekNumberCalculation = 'ISO'; // normalize
			}

			localeData = createObject( // make a cheap copy
				getMomentLocaleData(locale) // will fall back to en
			);

			if (monthNames) {
				localeData._months = monthNames;
			}
			if (monthNamesShort) {
				localeData._monthsShort = monthNamesShort;
			}
			if (dayNames) {
				localeData._weekdays = dayNames;
			}
			if (dayNamesShort) {
				localeData._weekdaysShort = dayNamesShort;
			}

			if (firstDay == null && weekNumberCalculation === 'ISO') {
				firstDay = 1;
			}
			if (firstDay != null) {
				var _week = createObject(localeData._week); // _week: { dow: # }
				_week.dow = firstDay;
				localeData._week = _week;
			}

			if ( // whitelist certain kinds of input
				weekNumberCalculation === 'ISO' ||
				weekNumberCalculation === 'local' ||
				typeof weekNumberCalculation === 'function'
			) {
				localeData._fullCalendar_weekCalc = weekNumberCalculation; // moment-ext will know what to do with it
			}

			// If the internal current date object already exists, move to new locale.
			// We do NOT need to do this technique for event dates, because this happens when converting to "segments".
			if (date) {
				localizeMoment(date); // sets to localeData
			}
		});


		// Calendar-specific Date Utilities
		// -----------------------------------------------------------------------------------


		t.defaultAllDayEventDuration = moment.duration(t.options.defaultAllDayEventDuration);
		t.defaultTimedEventDuration = moment.duration(t.options.defaultTimedEventDuration);


		// Builds a moment using the settings of the current calendar: timezone and locale.
		// Accepts anything the vanilla moment() constructor accepts.
		t.moment = function() {
			var mom;

			if (t.options.timezone === 'local') {
				mom = FC.moment.apply(null, arguments);

				// Force the moment to be local, because FC.moment doesn't guarantee it.
				if (mom.hasTime()) { // don't give ambiguously-timed moments a local zone
					mom.local();
				}
			}
			else if (t.options.timezone === 'UTC') {
				mom = FC.moment.utc.apply(null, arguments); // process as UTC
			}
			else {
				mom = FC.moment.parseZone.apply(null, arguments); // let the input decide the zone
			}

			localizeMoment(mom);

			return mom;
		};


		// Updates the given moment's locale settings to the current calendar locale settings.
		function localizeMoment(mom) {
			mom._locale = localeData;
		}
		t.localizeMoment = localizeMoment;


		// Returns a boolean about whether or not the calendar knows how to calculate
		// the timezone offset of arbitrary dates in the current timezone.
		t.getIsAmbigTimezone = function() {
			return t.options.timezone !== 'local' && t.options.timezone !== 'UTC';
		};


		// Returns a copy of the given date in the current timezone. Has no effect on dates without times.
		t.applyTimezone = function(date) {
			if (!date.hasTime()) {
				return date.clone();
			}

			var zonedDate = t.moment(date.toArray());
			var timeAdjust = date.time() - zonedDate.time();
			var adjustedZonedDate;

			// Safari sometimes has problems with this coersion when near DST. Adjust if necessary. (bug #2396)
			if (timeAdjust) { // is the time result different than expected?
				adjustedZonedDate = zonedDate.clone().add(timeAdjust); // add milliseconds
				if (date.time() - adjustedZonedDate.time() === 0) { // does it match perfectly now?
					zonedDate = adjustedZonedDate;
				}
			}

			return zonedDate;
		};


		// Returns a moment for the current date, as defined by the client's computer or from the `now` option.
		// Will return an moment with an ambiguous timezone.
		t.getNow = function() {
			var now = t.options.now;
			if (typeof now === 'function') {
				now = now();
			}
			return t.moment(now).stripZone();
		};


		// Get an event's normalized end date. If not present, calculate it from the defaults.
		t.getEventEnd = function(event) {
			if (event.end) {
				return event.end.clone();
			}
			else {
				return t.getDefaultEventEnd(event.allDay, event.start);
			}
		};


		// Given an event's allDay status and start date, return what its fallback end date should be.
		// TODO: rename to computeDefaultEventEnd
		t.getDefaultEventEnd = function(allDay, zonedStart) {
			var end = zonedStart.clone();

			if (allDay) {
				end.stripTime().add(t.defaultAllDayEventDuration);
			}
			else {
				end.add(t.defaultTimedEventDuration);
			}

			if (t.getIsAmbigTimezone()) {
				end.stripZone(); // we don't know what the tzo should be
			}

			return end;
		};


		// Produces a human-readable string for the given duration.
		// Side-effect: changes the locale of the given duration.
		t.humanizeDuration = function(duration) {
			return duration.locale(t.options.locale).humanize();
		};



		// Imports
		// -----------------------------------------------------------------------------------


		EventManager.call(t);



		// Locals
		// -----------------------------------------------------------------------------------


		var _element = element[0];
		var toolbarsManager;
		var header;
		var footer;
		var content;
		var tm; // for making theme classes
		var currentView; // NOTE: keep this in sync with this.view
		var viewsByType = {}; // holds all instantiated view instances, current or not
		var suggestedViewHeight;
		var windowResizeProxy; // wraps the windowResize function
		var ignoreWindowResize = 0;
		var date; // unzoned



		// Main Rendering
		// -----------------------------------------------------------------------------------


		// compute the initial ambig-timezone date
		if (t.options.defaultDate != null) {
			date = t.moment(t.options.defaultDate).stripZone();
		}
		else {
			date = t.getNow(); // getNow already returns unzoned
		}


		function render() {
			if (!content) {
				initialRender();
			}
			else if (elementVisible()) {
				// mainly for the public API
				calcSize();
				renderView();
			}
		}


		function initialRender() {
			element.addClass('fc');

			// event delegation for nav links
			element.on('click.fc', 'a[data-goto]', function(ev) {
				var anchorEl = $(this);
				var gotoOptions = anchorEl.data('goto'); // will automatically parse JSON
				var date = t.moment(gotoOptions.date);
				var viewType = gotoOptions.type;

				// property like "navLinkDayClick". might be a string or a function
				var customAction = currentView.opt('navLink' + capitaliseFirstLetter(viewType) + 'Click');

				if (typeof customAction === 'function') {
					customAction(date, ev);
				}
				else {
					if (typeof customAction === 'string') {
						viewType = customAction;
					}
					zoomTo(date, viewType);
				}
			});

			// called immediately, and upon option change
			t.bindOption('theme', function(theme) {
				tm = theme ? 'ui' : 'fc'; // affects a larger scope
				element.toggleClass('ui-widget', theme);
				element.toggleClass('fc-unthemed', !theme);
			});

			// called immediately, and upon option change.
			// HACK: locale often affects isRTL, so we explicitly listen to that too.
			t.bindOptions([ 'isRTL', 'locale' ], function(isRTL) {
				element.toggleClass('fc-ltr', !isRTL);
				element.toggleClass('fc-rtl', isRTL);
			});

			content = $("<div class='fc-view-container'/>").prependTo(element);

			var toolbars = buildToolbars();
			toolbarsManager = new Iterator(toolbars);

			header = t.header = toolbars[0];
			footer = t.footer = toolbars[1];

			renderHeader();
			renderFooter();
			renderView(t.options.defaultView);

			if (t.options.handleWindowResize) {
				windowResizeProxy = debounce(windowResize, t.options.windowResizeDelay); // prevents rapid calls
				$(window).resize(windowResizeProxy);
			}
		}


		function destroy() {

			if (currentView) {
				currentView.removeElement();

				// NOTE: don't null-out currentView/t.view in case API methods are called after destroy.
				// It is still the "current" view, just not rendered.
			}

			toolbarsManager.proxyCall('removeElement');
			content.remove();
			element.removeClass('fc fc-ltr fc-rtl fc-unthemed ui-widget');

			element.off('.fc'); // unbind nav link handlers

			if (windowResizeProxy) {
				$(window).unbind('resize', windowResizeProxy);
			}
		}


		function elementVisible() {
			return element.is(':visible');
		}



		// View Rendering
		// -----------------------------------------------------------------------------------


		// Renders a view because of a date change, view-type change, or for the first time.
		// If not given a viewType, keep the current view but render different dates.
		// Accepts an optional scroll state to restore to.
		function renderView(viewType, forcedScroll) {
			ignoreWindowResize++;

			var needsClearView = currentView && viewType && currentView.type !== viewType;

			// if viewType is changing, remove the old view's rendering
			if (needsClearView) {
				freezeContentHeight(); // prevent a scroll jump when view element is removed
				clearView();
			}

			// if viewType changed, or the view was never created, create a fresh view
			if (!currentView && viewType) {
				currentView = t.view =
					viewsByType[viewType] ||
					(viewsByType[viewType] = t.instantiateView(viewType));

				currentView.setElement(
					$("<div class='fc-view fc-" + viewType + "-view' />").appendTo(content)
				);
				toolbarsManager.proxyCall('activateButton', viewType);
			}

			if (currentView) {

				// in case the view should render a period of time that is completely hidden
				date = currentView.massageCurrentDate(date);

				// render or rerender the view
				if (
					!currentView.isDateSet ||
					!( // NOT within interval range signals an implicit date window change
						date >= currentView.intervalStart &&
						date < currentView.intervalEnd
					)
				) {
					if (elementVisible()) {

						if (forcedScroll) {
							currentView.captureInitialScroll(forcedScroll);
						}

						currentView.setDate(date, forcedScroll);

						if (forcedScroll) {
							currentView.releaseScroll();
						}

						// need to do this after View::render, so dates are calculated
						// NOTE: view updates title text proactively
						updateToolbarsTodayButton();
					}
				}
			}

			if (needsClearView) {
				thawContentHeight();
			}

			ignoreWindowResize--;
		}


		// Unrenders the current view and reflects this change in the Header.
		// Unregsiters the `currentView`, but does not remove from viewByType hash.
		function clearView() {
			toolbarsManager.proxyCall('deactivateButton', currentView.type);
			currentView.removeElement();
			currentView = t.view = null;
		}


		// Destroys the view, including the view object. Then, re-instantiates it and renders it.
		// Maintains the same scroll state.
		// TODO: maintain any other user-manipulated state.
		function reinitView() {
			ignoreWindowResize++;
			freezeContentHeight();

			var viewType = currentView.type;
			var scrollState = currentView.queryScroll();
			clearView();
			calcSize();
			renderView(viewType, scrollState);

			thawContentHeight();
			ignoreWindowResize--;
		}



		// Resizing
		// -----------------------------------------------------------------------------------


		t.getSuggestedViewHeight = function() {
			if (suggestedViewHeight === undefined) {
				calcSize();
			}
			return suggestedViewHeight;
		};


		t.isHeightAuto = function() {
			return t.options.contentHeight === 'auto' || t.options.height === 'auto';
		};


		function updateSize(shouldRecalc) {
			if (elementVisible()) {

				if (shouldRecalc) {
					_calcSize();
				}

				ignoreWindowResize++;
				currentView.updateSize(true); // isResize=true. will poll getSuggestedViewHeight() and isHeightAuto()
				ignoreWindowResize--;

				return true; // signal success
			}
		}


		function calcSize() {
			if (elementVisible()) {
				_calcSize();
			}
		}


		function _calcSize() { // assumes elementVisible
			var contentHeightInput = t.options.contentHeight;
			var heightInput = t.options.height;

			if (typeof contentHeightInput === 'number') { // exists and not 'auto'
				suggestedViewHeight = contentHeightInput;
			}
			else if (typeof contentHeightInput === 'function') { // exists and is a function
				suggestedViewHeight = contentHeightInput();
			}
			else if (typeof heightInput === 'number') { // exists and not 'auto'
				suggestedViewHeight = heightInput - queryToolbarsHeight();
			}
			else if (typeof heightInput === 'function') { // exists and is a function
				suggestedViewHeight = heightInput() - queryToolbarsHeight();
			}
			else if (heightInput === 'parent') { // set to height of parent element
				suggestedViewHeight = element.parent().height() - queryToolbarsHeight();
			}
			else {
				suggestedViewHeight = Math.round(content.width() / Math.max(t.options.aspectRatio, .5));
			}
		}


		function queryToolbarsHeight() {
			return toolbarsManager.items.reduce(function(accumulator, toolbar) {
				var toolbarHeight = toolbar.el ? toolbar.el.outerHeight(true) : 0; // includes margin
				return accumulator + toolbarHeight;
			}, 0);
		}


		function windowResize(ev) {
			if (
				!ignoreWindowResize &&
				ev.target === window && // so we don't process jqui "resize" events that have bubbled up
				currentView.start // view has already been rendered
			) {
				if (updateSize(true)) {
					currentView.publiclyTrigger('windowResize', _element);
				}
			}
		}



		/* Event Rendering
	-----------------------------------------------------------------------------*/


		function rerenderEvents() { // API method. destroys old events if previously rendered.
			if (elementVisible()) {
				t.reportEventChange(); // will re-trasmit events to the view, causing a rerender
			}
		}



		/* Toolbars
	-----------------------------------------------------------------------------*/


		function buildToolbars() {
			return [
				new Toolbar(t, computeHeaderOptions()),
				new Toolbar(t, computeFooterOptions())
			];
		}


		function computeHeaderOptions() {
			return {
				extraClasses: 'fc-header-toolbar',
				layout: t.options.header
			};
		}


		function computeFooterOptions() {
			return {
				extraClasses: 'fc-footer-toolbar',
				layout: t.options.footer
			};
		}


		// can be called repeatedly and Header will rerender
		function renderHeader() {
			header.setToolbarOptions(computeHeaderOptions());
			header.render();
			if (header.el) {
				element.prepend(header.el);
			}
		}


		// can be called repeatedly and Footer will rerender
		function renderFooter() {
			footer.setToolbarOptions(computeFooterOptions());
			footer.render();
			if (footer.el) {
				element.append(footer.el);
			}
		}


		t.setToolbarsTitle = function(title) {
			toolbarsManager.proxyCall('updateTitle', title);
		};


		function updateToolbarsTodayButton() {
			var now = t.getNow();
			if (now >= currentView.intervalStart && now < currentView.intervalEnd) {
				toolbarsManager.proxyCall('disableButton', 'today');
			}
			else {
				toolbarsManager.proxyCall('enableButton', 'today');
			}
		}



		/* Selection
	-----------------------------------------------------------------------------*/


		// this public method receives start/end dates in any format, with any timezone
		function select(zonedStartInput, zonedEndInput) {
			currentView.select(
				t.buildSelectSpan.apply(t, arguments)
			);
		}


		function unselect() { // safe to be called before renderView
			if (currentView) {
				currentView.unselect();
			}
		}



		/* Date
	-----------------------------------------------------------------------------*/


		function prev() {
			date = currentView.computePrevDate(date);
			renderView();
		}


		function next() {
			date = currentView.computeNextDate(date);
			renderView();
		}


		function prevYear() {
			date.add(-1, 'years');
			renderView();
		}


		function nextYear() {
			date.add(1, 'years');
			renderView();
		}


		function today() {
			date = t.getNow();
			renderView();
		}


		function gotoDate(zonedDateInput) {
			date = t.moment(zonedDateInput).stripZone();
			renderView();
		}


		function incrementDate(delta) {
			date.add(moment.duration(delta));
			renderView();
		}


		// Forces navigation to a view for the given date.
		// `viewType` can be a specific view name or a generic one like "week" or "day".
		function zoomTo(newDate, viewType) {
			var spec;

			viewType = viewType || 'day'; // day is default zoom
			spec = t.getViewSpec(viewType) || t.getUnitViewSpec(viewType);

			date = newDate.clone();
			renderView(spec ? spec.type : null);
		}


		// for external API
		function getDate() {
			return t.applyTimezone(date); // infuse the calendar's timezone
		}



		/* Height "Freezing"
	-----------------------------------------------------------------------------*/


		t.freezeContentHeight = freezeContentHeight;
		t.thawContentHeight = thawContentHeight;

		var freezeContentHeightDepth = 0;


		function freezeContentHeight() {
			if (!(freezeContentHeightDepth++)) {
				content.css({
					width: '100%',
					height: content.height(),
					overflow: 'hidden'
				});
			}
		}


		function thawContentHeight() {
			if (!(--freezeContentHeightDepth)) {
				content.css({
					width: '',
					height: '',
					overflow: ''
				});
			}
		}



		/* Misc
	-----------------------------------------------------------------------------*/


		function getCalendar() {
			return t;
		}


		function getView() {
			return currentView;
		}


		function option(name, value) {
			var newOptionHash;

			if (typeof name === 'string') {
				if (value === undefined) { // getter
					return t.options[name];
				}
				else { // setter for individual option
					newOptionHash = {};
					newOptionHash[name] = value;
					setOptions(newOptionHash);
				}
			}
			else if (typeof name === 'object') { // compound setter with object input
				setOptions(name);
			}
		}


		function setOptions(newOptionHash) {
			var optionCnt = 0;
			var optionName;

			for (optionName in newOptionHash) {
				t.dynamicOverrides[optionName] = newOptionHash[optionName];
			}

			t.viewSpecCache = {}; // the dynamic override invalidates the options in this cache, so just clear it
			t.populateOptionsHash(); // this.options needs to be recomputed after the dynamic override

			// trigger handlers after this.options has been updated
			for (optionName in newOptionHash) {
				t.triggerOptionHandlers(optionName); // recall bindOption/bindOptions
				optionCnt++;
			}

			// special-case handling of single option change.
			// if only one option change, `optionName` will be its name.
			if (optionCnt === 1) {
				if (optionName === 'height' || optionName === 'contentHeight' || optionName === 'aspectRatio') {
					updateSize(true); // true = allow recalculation of height
					return;
				}
				else if (optionName === 'defaultDate') {
					return; // can't change date this way. use gotoDate instead
				}
				else if (optionName === 'businessHours') {
					if (currentView) {
						currentView.unrenderBusinessHours();
						currentView.renderBusinessHours();
					}
					return;
				}
				else if (optionName === 'timezone') {
					t.rezoneArrayEventSources();
					t.refetchEvents();
					return;
				}
			}

			// catch-all. rerender the header and footer and rebuild/rerender the current view
			renderHeader();
			renderFooter();
			viewsByType = {}; // even non-current views will be affected by this option change. do before rerender
			reinitView();
		}


		function publiclyTrigger(name, thisObj) {
			var args = Array.prototype.slice.call(arguments, 2);

			thisObj = thisObj || _element;
			this.triggerWith(name, thisObj, args); // Emitter's method

			if (t.options[name]) {
				return t.options[name].apply(thisObj, args);
			}
		}

		t.initialize();
	}

	;;
	/*
Options binding/triggering system.
*/
	Calendar.mixin({

		// A map of option names to arrays of handler objects. Initialized to {} in Calendar.
		// Format for a handler object:
		// {
		//   func // callback function to be called upon change
		//   names // option names whose values should be given to func
		// }
		optionHandlers: null,

		// Calls handlerFunc immediately, and when the given option has changed.
		// handlerFunc will be given the option value.
		bindOption: function(optionName, handlerFunc) {
			this.bindOptions([ optionName ], handlerFunc);
		},

		// Calls handlerFunc immediately, and when any of the given options change.
		// handlerFunc will be given each option value as ordered function arguments.
		bindOptions: function(optionNames, handlerFunc) {
			var handlerObj = { func: handlerFunc, names: optionNames };
			var i;

			for (i = 0; i < optionNames.length; i++) {
				this.registerOptionHandlerObj(optionNames[i], handlerObj);
			}

			this.triggerOptionHandlerObj(handlerObj);
		},

		// Puts the given handler object into the internal hash
		registerOptionHandlerObj: function(optionName, handlerObj) {
			(this.optionHandlers[optionName] || (this.optionHandlers[optionName] = []))
				.push(handlerObj);
		},

		// Reports that the given option has changed, and calls all appropriate handlers.
		triggerOptionHandlers: function(optionName) {
			var handlerObjs = this.optionHandlers[optionName] || [];
			var i;

			for (i = 0; i < handlerObjs.length; i++) {
				this.triggerOptionHandlerObj(handlerObjs[i]);
			}
		},

		// Calls the callback for a specific handler object, passing in the appropriate arguments.
		triggerOptionHandlerObj: function(handlerObj) {
			var optionNames = handlerObj.names;
			var optionValues = [];
			var i;

			for (i = 0; i < optionNames.length; i++) {
				optionValues.push(this.options[optionNames[i]]);
			}

			handlerObj.func.apply(this, optionValues); // maintain the Calendar's `this` context
		}

	});

	;;

	Calendar.defaults = {

		titleRangeSeparator: ' \u2013 ', // en dash
		monthYearFormat: 'MMMM YYYY', // required for en. other locales rely on datepicker computable option

		defaultTimedEventDuration: '02:00:00',
		defaultAllDayEventDuration: { days: 1 },
		forceEventDuration: false,
		nextDayThreshold: '09:00:00', // 9am

		// display
		defaultView: 'month',
		aspectRatio: 1.35,
		header: {
			left: 'title',
			center: '',
			right: 'today prev,next'
		},
		weekends: true,
		weekNumbers: false,

		weekNumberTitle: 'W',
		weekNumberCalculation: 'local',

		//editable: false,

		//nowIndicator: false,

		scrollTime: '06:00:00',

		// event ajax
		lazyFetching: true,
		startParam: 'start',
		endParam: 'end',
		timezoneParam: 'timezone',

		timezone: false,

		//allDayDefault: undefined,

		// locale
		isRTL: false,
		buttonText: {
			prev: "prev",
			next: "next",
			prevYear: "prev year",
			nextYear: "next year",
			year: 'year', // TODO: locale files need to specify this
			today: 'today',
			month: 'month',
			week: 'week',
			day: 'day'
		},

		buttonIcons: {
			prev: 'left-single-arrow',
			next: 'right-single-arrow',
			prevYear: 'left-double-arrow',
			nextYear: 'right-double-arrow'
		},

		allDayText: 'all-day',

		// jquery-ui theming
		theme: false,
		themeButtonIcons: {
			prev: 'circle-triangle-w',
			next: 'circle-triangle-e',
			prevYear: 'seek-prev',
			nextYear: 'seek-next'
		},

		//eventResizableFromStart: false,
		dragOpacity: .75,
		dragRevertDuration: 500,
		dragScroll: true,

		//selectable: false,
		unselectAuto: true,

		dropAccept: '*',

		eventOrder: 'title',
		//eventRenderWait: null,

		eventLimit: false,
		eventLimitText: 'more',
		eventLimitClick: 'popover',
		dayPopoverFormat: 'LL',

		handleWindowResize: true,
		windowResizeDelay: 100, // milliseconds before an updateSize happens

		longPressDelay: 1000

	};


	Calendar.englishDefaults = { // used by locale.js
		dayPopoverFormat: 'dddd, MMMM D'
	};


	Calendar.rtlDefaults = { // right-to-left defaults
		header: { // TODO: smarter solution (first/center/last ?)
			left: 'next,prev today',
			center: '',
			right: 'title'
		},
		buttonIcons: {
			prev: 'right-single-arrow',
			next: 'left-single-arrow',
			prevYear: 'right-double-arrow',
			nextYear: 'left-double-arrow'
		},
		themeButtonIcons: {
			prev: 'circle-triangle-e',
			next: 'circle-triangle-w',
			nextYear: 'seek-prev',
			prevYear: 'seek-next'
		}
	};

	;;

	var localeOptionHash = FC.locales = {}; // initialize and expose


// TODO: document the structure and ordering of a FullCalendar locale file


// Initialize jQuery UI datepicker translations while using some of the translations
// Will set this as the default locales for datepicker.
	FC.datepickerLocale = function(localeCode, dpLocaleCode, dpOptions) {

		// get the FullCalendar internal option hash for this locale. create if necessary
		var fcOptions = localeOptionHash[localeCode] || (localeOptionHash[localeCode] = {});

		// transfer some simple options from datepicker to fc
		fcOptions.isRTL = dpOptions.isRTL;
		fcOptions.weekNumberTitle = dpOptions.weekHeader;

		// compute some more complex options from datepicker
		$.each(dpComputableOptions, function(name, func) {
			fcOptions[name] = func(dpOptions);
		});

		// is jQuery UI Datepicker is on the page?
		if ($.datepicker) {

			// Register the locale data.
			// FullCalendar and MomentJS use locale codes like "pt-br" but Datepicker
			// does it like "pt-BR" or if it doesn't have the locale, maybe just "pt".
			// Make an alias so the locale can be referenced either way.
			$.datepicker.regional[dpLocaleCode] =
				$.datepicker.regional[localeCode] = // alias
					dpOptions;

			// Alias 'en' to the default locale data. Do this every time.
			$.datepicker.regional.en = $.datepicker.regional[''];

			// Set as Datepicker's global defaults.
			$.datepicker.setDefaults(dpOptions);
		}
	};


// Sets FullCalendar-specific translations. Will set the locales as the global default.
	FC.locale = function(localeCode, newFcOptions) {
		var fcOptions;
		var momOptions;

		// get the FullCalendar internal option hash for this locale. create if necessary
		fcOptions = localeOptionHash[localeCode] || (localeOptionHash[localeCode] = {});

		// provided new options for this locales? merge them in
		if (newFcOptions) {
			fcOptions = localeOptionHash[localeCode] = mergeOptions([ fcOptions, newFcOptions ]);
		}

		// compute locale options that weren't defined.
		// always do this. newFcOptions can be undefined when initializing from i18n file,
		// so no way to tell if this is an initialization or a default-setting.
		momOptions = getMomentLocaleData(localeCode); // will fall back to en
		$.each(momComputableOptions, function(name, func) {
			if (fcOptions[name] == null) {
				fcOptions[name] = func(momOptions, fcOptions);
			}
		});

		// set it as the default locale for FullCalendar
		Calendar.defaults.locale = localeCode;
	};


// NOTE: can't guarantee any of these computations will run because not every locale has datepicker
// configs, so make sure there are English fallbacks for these in the defaults file.
	var dpComputableOptions = {

		buttonText: function(dpOptions) {
			return {
				// the translations sometimes wrongly contain HTML entities
				prev: stripHtmlEntities(dpOptions.prevText),
				next: stripHtmlEntities(dpOptions.nextText),
				today: stripHtmlEntities(dpOptions.currentText)
			};
		},

		// Produces format strings like "MMMM YYYY" -> "September 2014"
		monthYearFormat: function(dpOptions) {
			return dpOptions.showMonthAfterYear ?
				'YYYY[' + dpOptions.yearSuffix + '] MMMM' :
				'MMMM YYYY[' + dpOptions.yearSuffix + ']';
		}

	};

	var momComputableOptions = {

		// Produces format strings like "ddd M/D" -> "Fri 9/15"
		dayOfMonthFormat: function(momOptions, fcOptions) {
			var format = momOptions.longDateFormat('l'); // for the format like "M/D/YYYY"

			// strip the year off the edge, as well as other misc non-whitespace chars
			format = format.replace(/^Y+[^\w\s]*|[^\w\s]*Y+$/g, '');

			if (fcOptions.isRTL) {
				format += ' ddd'; // for RTL, add day-of-week to end
			}
			else {
				format = 'ddd ' + format; // for LTR, add day-of-week to beginning
			}
			return format;
		},

		// Produces format strings like "h:mma" -> "6:00pm"
		mediumTimeFormat: function(momOptions) { // can't be called `timeFormat` because collides with option
			return momOptions.longDateFormat('LT')
				.replace(/\s*a$/i, 'a'); // convert AM/PM/am/pm to lowercase. remove any spaces beforehand
		},

		// Produces format strings like "h(:mm)a" -> "6pm" / "6:30pm"
		smallTimeFormat: function(momOptions) {
			return momOptions.longDateFormat('LT')
				.replace(':mm', '(:mm)')
				.replace(/(\Wmm)$/, '($1)') // like above, but for foreign locales
				.replace(/\s*a$/i, 'a'); // convert AM/PM/am/pm to lowercase. remove any spaces beforehand
		},

		// Produces format strings like "h(:mm)t" -> "6p" / "6:30p"
		extraSmallTimeFormat: function(momOptions) {
			return momOptions.longDateFormat('LT')
				.replace(':mm', '(:mm)')
				.replace(/(\Wmm)$/, '($1)') // like above, but for foreign locales
				.replace(/\s*a$/i, 't'); // convert to AM/PM/am/pm to lowercase one-letter. remove any spaces beforehand
		},

		// Produces format strings like "ha" / "H" -> "6pm" / "18"
		hourFormat: function(momOptions) {
			return momOptions.longDateFormat('LT')
				.replace(':mm', '')
				.replace(/(\Wmm)$/, '') // like above, but for foreign locales
				.replace(/\s*a$/i, 'a'); // convert AM/PM/am/pm to lowercase. remove any spaces beforehand
		},

		// Produces format strings like "h:mm" -> "6:30" (with no AM/PM)
		noMeridiemTimeFormat: function(momOptions) {
			return momOptions.longDateFormat('LT')
				.replace(/\s*a$/i, ''); // remove trailing AM/PM
		}

	};


// options that should be computed off live calendar options (considers override options)
// TODO: best place for this? related to locale?
// TODO: flipping text based on isRTL is a bad idea because the CSS `direction` might want to handle it
	var instanceComputableOptions = {

		// Produces format strings for results like "Mo 16"
		smallDayDateFormat: function(options) {
			return options.isRTL ?
				'D dd' :
				'dd D';
		},

		// Produces format strings for results like "Wk 5"
		weekFormat: function(options) {
			return options.isRTL ?
				'w[ ' + options.weekNumberTitle + ']' :
				'[' + options.weekNumberTitle + ' ]w';
		},

		// Produces format strings for results like "Wk5"
		smallWeekFormat: function(options) {
			return options.isRTL ?
				'w[' + options.weekNumberTitle + ']' :
				'[' + options.weekNumberTitle + ']w';
		}

	};

	function populateInstanceComputableOptions(options) {
		$.each(instanceComputableOptions, function(name, func) {
			if (options[name] == null) {
				options[name] = func(options);
			}
		});
	}


// Returns moment's internal locale data. If doesn't exist, returns English.
	function getMomentLocaleData(localeCode) {
		return moment.localeData(localeCode) || moment.localeData('en');
	}


// Initialize English by forcing computation of moment-derived options.
// Also, sets it as the default.
	FC.locale('en', Calendar.englishDefaults);

	;;

	FC.sourceNormalizers = [];
	FC.sourceFetchers = [];

	var ajaxDefaults = {
		dataType: 'json',
		cache: false
	};

	var eventGUID = 1;


	function EventManager() { // assumed to be a calendar
		var t = this;


		// exports
		t.requestEvents = requestEvents;
		t.reportEventChange = reportEventChange;
		t.isFetchNeeded = isFetchNeeded;
		t.fetchEvents = fetchEvents;
		t.fetchEventSources = fetchEventSources;
		t.refetchEvents = refetchEvents;
		t.refetchEventSources = refetchEventSources;
		t.getEventSources = getEventSources;
		t.getEventSourceById = getEventSourceById;
		t.addEventSource = addEventSource;
		t.removeEventSource = removeEventSource;
		t.removeEventSources = removeEventSources;
		t.updateEvent = updateEvent;
		t.updateEvents = updateEvents;
		t.renderEvent = renderEvent;
		t.renderEvents = renderEvents;
		t.removeEvents = removeEvents;
		t.clientEvents = clientEvents;
		t.mutateEvent = mutateEvent;
		t.normalizeEventDates = normalizeEventDates;
		t.normalizeEventTimes = normalizeEventTimes;


		// locals
		var stickySource = { events: [] };
		var sources = [ stickySource ];
		var rangeStart, rangeEnd;
		var pendingSourceCnt = 0; // outstanding fetch requests, max one per source
		var cache = []; // holds events that have already been expanded
		var prunedCache; // like cache, but only events that intersect with rangeStart/rangeEnd


		$.each(
			(t.options.events ? [ t.options.events ] : []).concat(t.options.eventSources || []),
			function(i, sourceInput) {
				var source = buildEventSource(sourceInput);
				if (source) {
					sources.push(source);
				}
			}
		);



		function requestEvents(start, end) {
			if (!t.options.lazyFetching || isFetchNeeded(start, end)) {
				return fetchEvents(start, end);
			}
			else {
				return Promise.resolve(prunedCache);
			}
		}


		function reportEventChange() {
			prunedCache = filterEventsWithinRange(cache);
			t.trigger('eventsReset', prunedCache);
		}


		function filterEventsWithinRange(events) {
			var filteredEvents = [];
			var i, event;

			for (i = 0; i < events.length; i++) {
				event = events[i];

				if (
					event.start.clone().stripZone() < rangeEnd &&
					t.getEventEnd(event).stripZone() > rangeStart
				) {
					filteredEvents.push(event);
				}
			}

			return filteredEvents;
		}


		t.getEventCache = function() {
			return cache;
		};


		t.getPrunedEventCache = function() {
			return prunedCache;
		};



		/* Fetching
	-----------------------------------------------------------------------------*/


		// start and end are assumed to be unzoned
		function isFetchNeeded(start, end) {
			return !rangeStart || // nothing has been fetched yet?
				start < rangeStart || end > rangeEnd; // is part of the new range outside of the old range?
		}


		function fetchEvents(start, end) {
			rangeStart = start;
			rangeEnd = end;
			return refetchEvents();
		}


		// poorly named. fetches all sources with current `rangeStart` and `rangeEnd`.
		function refetchEvents() {
			return fetchEventSources(sources, 'reset');
		}


		// poorly named. fetches a subset of event sources.
		function refetchEventSources(matchInputs) {
			return fetchEventSources(getEventSourcesByMatchArray(matchInputs));
		}


		// expects an array of event source objects (the originals, not copies)
		// `specialFetchType` is an optimization parameter that affects purging of the event cache.
		function fetchEventSources(specificSources, specialFetchType) {
			var i, source;

			if (specialFetchType === 'reset') {
				cache = [];
			}
			else if (specialFetchType !== 'add') {
				cache = excludeEventsBySources(cache, specificSources);
			}

			for (i = 0; i < specificSources.length; i++) {
				source = specificSources[i];

				// already-pending sources have already been accounted for in pendingSourceCnt
				if (source._status !== 'pending') {
					pendingSourceCnt++;
				}

				source._fetchId = (source._fetchId || 0) + 1;
				source._status = 'pending';
			}

			for (i = 0; i < specificSources.length; i++) {
				source = specificSources[i];
				tryFetchEventSource(source, source._fetchId);
			}

			if (pendingSourceCnt) {
				return new Promise(function(resolve) {
					t.one('eventsReceived', resolve); // will send prunedCache
				});
			}
			else { // executed all synchronously, or no sources at all
				return Promise.resolve(prunedCache);
			}
		}


		// fetches an event source and processes its result ONLY if it is still the current fetch.
		// caller is responsible for incrementing pendingSourceCnt first.
		function tryFetchEventSource(source, fetchId) {
			_fetchEventSource(source, function(eventInputs) {
				var isArraySource = $.isArray(source.events);
				var i, eventInput;
				var abstractEvent;

				if (
					// is this the source's most recent fetch?
					// if not, rely on an upcoming fetch of this source to decrement pendingSourceCnt
					fetchId === source._fetchId &&
					// event source no longer valid?
					source._status !== 'rejected'
				) {
					source._status = 'resolved';

					if (eventInputs) {
						for (i = 0; i < eventInputs.length; i++) {
							eventInput = eventInputs[i];

							if (isArraySource) { // array sources have already been convert to Event Objects
								abstractEvent = eventInput;
							}
							else {
								abstractEvent = buildEventFromInput(eventInput, source);
							}

							if (abstractEvent) { // not false (an invalid event)
								cache.push.apply( // append
									cache,
									expandEvent(abstractEvent) // add individual expanded events to the cache
								);
							}
						}
					}

					decrementPendingSourceCnt();
				}
			});
		}


		function rejectEventSource(source) {
			var wasPending = source._status === 'pending';

			source._status = 'rejected';

			if (wasPending) {
				decrementPendingSourceCnt();
			}
		}


		function decrementPendingSourceCnt() {
			pendingSourceCnt--;
			if (!pendingSourceCnt) {
				reportEventChange(cache); // updates prunedCache
				t.trigger('eventsReceived', prunedCache);
			}
		}


		function _fetchEventSource(source, callback) {
			var i;
			var fetchers = FC.sourceFetchers;
			var res;

			for (i=0; i<fetchers.length; i++) {
				res = fetchers[i].call(
					t, // this, the Calendar object
					source,
					rangeStart.clone(),
					rangeEnd.clone(),
					t.options.timezone,
					callback
				);

				if (res === true) {
					// the fetcher is in charge. made its own async request
					return;
				}
				else if (typeof res == 'object') {
					// the fetcher returned a new source. process it
					_fetchEventSource(res, callback);
					return;
				}
			}

			var events = source.events;
			if (events) {
				if ($.isFunction(events)) {
					t.pushLoading();
					events.call(
						t, // this, the Calendar object
						rangeStart.clone(),
						rangeEnd.clone(),
						t.options.timezone,
						function(events) {
							callback(events);
							t.popLoading();
						}
					);
				}
				else if ($.isArray(events)) {
					callback(events);
				}
				else {
					callback();
				}
			}else{
				var url = source.url;
				if (url) {
					var success = source.success;
					var error = source.error;
					var complete = source.complete;

					// retrieve any outbound GET/POST $.ajax data from the options
					var customData;
					if ($.isFunction(source.data)) {
						// supplied as a function that returns a key/value object
						customData = source.data();
					}
					else {
						// supplied as a straight key/value object
						customData = source.data;
					}

					// use a copy of the custom data so we can modify the parameters
					// and not affect the passed-in object.
					var data = $.extend({}, customData || {});

					var startParam = firstDefined(source.startParam, t.options.startParam);
					var endParam = firstDefined(source.endParam, t.options.endParam);
					var timezoneParam = firstDefined(source.timezoneParam, t.options.timezoneParam);

					if (startParam) {
						data[startParam] = rangeStart.format();
					}
					if (endParam) {
						data[endParam] = rangeEnd.format();
					}
					if (t.options.timezone && t.options.timezone != 'local') {
						data[timezoneParam] = t.options.timezone;
					}

					t.pushLoading();
					$.ajax($.extend({}, ajaxDefaults, source, {
						data: data,
						success: function(events) {
							events = events || [];
							var res = applyAll(success, this, arguments);
							if ($.isArray(res)) {
								events = res;
							}
							callback(events);
						},
						error: function() {
							applyAll(error, this, arguments);
							callback();
						},
						complete: function() {
							applyAll(complete, this, arguments);
							t.popLoading();
						}
					}));
				}else{
					callback();
				}
			}
		}



		/* Sources
	-----------------------------------------------------------------------------*/


		function addEventSource(sourceInput) {
			var source = buildEventSource(sourceInput);
			if (source) {
				sources.push(source);
				fetchEventSources([ source ], 'add'); // will eventually call reportEventChange
			}
		}


		function buildEventSource(sourceInput) { // will return undefined if invalid source
			var normalizers = FC.sourceNormalizers;
			var source;
			var i;

			if ($.isFunction(sourceInput) || $.isArray(sourceInput)) {
				source = { events: sourceInput };
			}
			else if (typeof sourceInput === 'string') {
				source = { url: sourceInput };
			}
			else if (typeof sourceInput === 'object') {
				source = $.extend({}, sourceInput); // shallow copy
			}

			if (source) {

				// TODO: repeat code, same code for event classNames
				if (source.className) {
					if (typeof source.className === 'string') {
						source.className = source.className.split(/\s+/);
					}
					// otherwise, assumed to be an array
				}
				else {
					source.className = [];
				}

				// for array sources, we convert to standard Event Objects up front
				if ($.isArray(source.events)) {
					source.origArray = source.events; // for removeEventSource
					source.events = $.map(source.events, function(eventInput) {
						return buildEventFromInput(eventInput, source);
					});
				}

				for (i=0; i<normalizers.length; i++) {
					normalizers[i].call(t, source);
				}

				return source;
			}
		}


		function removeEventSource(matchInput) {
			removeSpecificEventSources(
				getEventSourcesByMatch(matchInput)
			);
		}


		// if called with no arguments, removes all.
		function removeEventSources(matchInputs) {
			if (matchInputs == null) {
				removeSpecificEventSources(sources, true); // isAll=true
			}
			else {
				removeSpecificEventSources(
					getEventSourcesByMatchArray(matchInputs)
				);
			}
		}


		function removeSpecificEventSources(targetSources, isAll) {
			var i;

			// cancel pending requests
			for (i = 0; i < targetSources.length; i++) {
				rejectEventSource(targetSources[i]);
			}

			if (isAll) { // an optimization
				sources = [];
				cache = [];
			}
			else {
				// remove from persisted source list
				sources = $.grep(sources, function(source) {
					for (i = 0; i < targetSources.length; i++) {
						if (source === targetSources[i]) {
							return false; // exclude
						}
					}
					return true; // include
				});

				cache = excludeEventsBySources(cache, targetSources);
			}

			reportEventChange();
		}


		function getEventSources() {
			return sources.slice(1); // returns a shallow copy of sources with stickySource removed
		}


		function getEventSourceById(id) {
			return $.grep(sources, function(source) {
				return source.id && source.id === id;
			})[0];
		}


		// like getEventSourcesByMatch, but accepts multple match criteria (like multiple IDs)
		function getEventSourcesByMatchArray(matchInputs) {

			// coerce into an array
			if (!matchInputs) {
				matchInputs = [];
			}
			else if (!$.isArray(matchInputs)) {
				matchInputs = [ matchInputs ];
			}

			var matchingSources = [];
			var i;

			// resolve raw inputs to real event source objects
			for (i = 0; i < matchInputs.length; i++) {
				matchingSources.push.apply( // append
					matchingSources,
					getEventSourcesByMatch(matchInputs[i])
				);
			}

			return matchingSources;
		}


		// matchInput can either by a real event source object, an ID, or the function/URL for the source.
		// returns an array of matching source objects.
		function getEventSourcesByMatch(matchInput) {
			var i, source;

			// given an proper event source object
			for (i = 0; i < sources.length; i++) {
				source = sources[i];
				if (source === matchInput) {
					return [ source ];
				}
			}

			// an ID match
			source = getEventSourceById(matchInput);
			if (source) {
				return [ source ];
			}

			return $.grep(sources, function(source) {
				return isSourcesEquivalent(matchInput, source);
			});
		}


		function isSourcesEquivalent(source1, source2) {
			return source1 && source2 && getSourcePrimitive(source1) == getSourcePrimitive(source2);
		}


		function getSourcePrimitive(source) {
			return (
					(typeof source === 'object') ? // a normalized event source?
						(source.origArray || source.googleCalendarId || source.url || source.events) : // get the primitive
						null
				) ||
				source; // the given argument *is* the primitive
		}


		// util
		// returns a filtered array without events that are part of any of the given sources
		function excludeEventsBySources(specificEvents, specificSources) {
			return $.grep(specificEvents, function(event) {
				for (var i = 0; i < specificSources.length; i++) {
					if (event.source === specificSources[i]) {
						return false; // exclude
					}
				}
				return true; // keep
			});
		}



		/* Manipulation
	-----------------------------------------------------------------------------*/


		// Only ever called from the externally-facing API
		function updateEvent(event) {
			updateEvents([ event ]);
		}


		// Only ever called from the externally-facing API
		function updateEvents(events) {
			var i, event;

			for (i = 0; i < events.length; i++) {
				event = events[i];

				// massage start/end values, even if date string values
				event.start = t.moment(event.start);
				if (event.end) {
					event.end = t.moment(event.end);
				}
				else {
					event.end = null;
				}

				mutateEvent(event, getMiscEventProps(event)); // will handle start/end/allDay normalization
			}

			reportEventChange(); // reports event modifications (so we can redraw)
		}


		// Returns a hash of misc event properties that should be copied over to related events.
		function getMiscEventProps(event) {
			var props = {};

			$.each(event, function(name, val) {
				if (isMiscEventPropName(name)) {
					if (val !== undefined && isAtomic(val)) { // a defined non-object
						props[name] = val;
					}
				}
			});

			return props;
		}

		// non-date-related, non-id-related, non-secret
		function isMiscEventPropName(name) {
			return !/^_|^(id|allDay|start|end)$/.test(name);
		}


		// returns the expanded events that were created
		function renderEvent(eventInput, stick) {
			return renderEvents([ eventInput ], stick);
		}


		// returns the expanded events that were created
		function renderEvents(eventInputs, stick) {
			var renderedEvents = [];
			var renderableEvents;
			var abstractEvent;
			var i, j, event;

			for (i = 0; i < eventInputs.length; i++) {
				abstractEvent = buildEventFromInput(eventInputs[i]);

				if (abstractEvent) { // not false (a valid input)
					renderableEvents = expandEvent(abstractEvent);

					for (j = 0; j < renderableEvents.length; j++) {
						event = renderableEvents[j];

						if (!event.source) {
							if (stick) {
								stickySource.events.push(event);
								event.source = stickySource;
							}
							cache.push(event);
						}
					}

					renderedEvents = renderedEvents.concat(renderableEvents);
				}
			}

			if (renderedEvents.length) { // any new events rendered?
				reportEventChange();
			}

			return renderedEvents;
		}


		function removeEvents(filter) {
			var eventID;
			var i;

			if (filter == null) { // null or undefined. remove all events
				filter = function() { return true; }; // will always match
			}
			else if (!$.isFunction(filter)) { // an event ID
				eventID = filter + '';
				filter = function(event) {
					return event._id == eventID;
				};
			}

			// Purge event(s) from our local cache
			cache = $.grep(cache, filter, true); // inverse=true

			// Remove events from array sources.
			// This works because they have been converted to official Event Objects up front.
			// (and as a result, event._id has been calculated).
			for (i=0; i<sources.length; i++) {
				if ($.isArray(sources[i].events)) {
					sources[i].events = $.grep(sources[i].events, filter, true);
				}
			}

			reportEventChange();
		}


		function clientEvents(filter) {
			if ($.isFunction(filter)) {
				return $.grep(cache, filter);
			}
			else if (filter != null) { // not null, not undefined. an event ID
				filter += '';
				return $.grep(cache, function(e) {
					return e._id == filter;
				});
			}
			return cache; // else, return all
		}


		// Makes sure all array event sources have their internal event objects
		// converted over to the Calendar's current timezone.
		t.rezoneArrayEventSources = function() {
			var i;
			var events;
			var j;

			for (i = 0; i < sources.length; i++) {
				events = sources[i].events;
				if ($.isArray(events)) {

					for (j = 0; j < events.length; j++) {
						rezoneEventDates(events[j]);
					}
				}
			}
		};

		function rezoneEventDates(event) {
			event.start = t.moment(event.start);
			if (event.end) {
				event.end = t.moment(event.end);
			}
			backupEventDates(event);
		}


		/* Event Normalization
	-----------------------------------------------------------------------------*/


		// Given a raw object with key/value properties, returns an "abstract" Event object.
		// An "abstract" event is an event that, if recurring, will not have been expanded yet.
		// Will return `false` when input is invalid.
		// `source` is optional
		function buildEventFromInput(input, source) {
			var out = {};
			var start, end;
			var allDay;

			if (t.options.eventDataTransform) {
				input = t.options.eventDataTransform(input);
			}
			if (source && source.eventDataTransform) {
				input = source.eventDataTransform(input);
			}

			// Copy all properties over to the resulting object.
			// The special-case properties will be copied over afterwards.
			$.extend(out, input);

			if (source) {
				out.source = source;
			}

			out._id = input._id || (input.id === undefined ? '_fc' + eventGUID++ : input.id + '');

			if (input.className) {
				if (typeof input.className == 'string') {
					out.className = input.className.split(/\s+/);
				}
				else { // assumed to be an array
					out.className = input.className;
				}
			}
			else {
				out.className = [];
			}

			start = input.start || input.date; // "date" is an alias for "start"
			end = input.end;

			// parse as a time (Duration) if applicable
			if (isTimeString(start)) {
				start = moment.duration(start);
			}
			if (isTimeString(end)) {
				end = moment.duration(end);
			}

			if (input.dow || moment.isDuration(start) || moment.isDuration(end)) {

				// the event is "abstract" (recurring) so don't calculate exact start/end dates just yet
				out.start = start ? moment.duration(start) : null; // will be a Duration or null
				out.end = end ? moment.duration(end) : null; // will be a Duration or null
				out._recurring = true; // our internal marker
			}
			else {

				if (start) {
					start = t.moment(start);
					if (!start.isValid()) {
						return false;
					}
				}

				if (end) {
					end = t.moment(end);
					if (!end.isValid()) {
						end = null; // let defaults take over
					}
				}

				allDay = input.allDay;
				if (allDay === undefined) { // still undefined? fallback to default
					allDay = firstDefined(
						source ? source.allDayDefault : undefined,
						t.options.allDayDefault
					);
					// still undefined? normalizeEventDates will calculate it
				}

				assignDatesToEvent(start, end, allDay, out);
			}

			t.normalizeEvent(out); // hook for external use. a prototype method

			return out;
		}
		t.buildEventFromInput = buildEventFromInput;


		// Normalizes and assigns the given dates to the given partially-formed event object.
		// NOTE: mutates the given start/end moments. does not make a copy.
		function assignDatesToEvent(start, end, allDay, event) {
			event.start = start;
			event.end = end;
			event.allDay = allDay;
			normalizeEventDates(event);
			backupEventDates(event);
		}


		// Ensures proper values for allDay/start/end. Accepts an Event object, or a plain object with event-ish properties.
		// NOTE: Will modify the given object.
		function normalizeEventDates(eventProps) {

			normalizeEventTimes(eventProps);

			if (eventProps.end && !eventProps.end.isAfter(eventProps.start)) {
				eventProps.end = null;
			}

			if (!eventProps.end) {
				if (t.options.forceEventDuration) {
					eventProps.end = t.getDefaultEventEnd(eventProps.allDay, eventProps.start);
				}
				else {
					eventProps.end = null;
				}
			}
		}


		// Ensures the allDay property exists and the timeliness of the start/end dates are consistent
		function normalizeEventTimes(eventProps) {
			if (eventProps.allDay == null) {
				eventProps.allDay = !(eventProps.start.hasTime() || (eventProps.end && eventProps.end.hasTime()));
			}

			if (eventProps.allDay) {
				eventProps.start.stripTime();
				if (eventProps.end) {
					// TODO: consider nextDayThreshold here? If so, will require a lot of testing and adjustment
					eventProps.end.stripTime();
				}
			}
			else {
				if (!eventProps.start.hasTime()) {
					eventProps.start = t.applyTimezone(eventProps.start.time(0)); // will assign a 00:00 time
				}
				if (eventProps.end && !eventProps.end.hasTime()) {
					eventProps.end = t.applyTimezone(eventProps.end.time(0)); // will assign a 00:00 time
				}
			}
		}


		// If the given event is a recurring event, break it down into an array of individual instances.
		// If not a recurring event, return an array with the single original event.
		// If given a falsy input (probably because of a failed buildEventFromInput call), returns an empty array.
		// HACK: can override the recurring window by providing custom rangeStart/rangeEnd (for businessHours).
		function expandEvent(abstractEvent, _rangeStart, _rangeEnd) {
			var events = [];
			var dowHash;
			var dow;
			var i;
			var date;
			var startTime, endTime;
			var start, end;
			var event;

			_rangeStart = _rangeStart || rangeStart;
			_rangeEnd = _rangeEnd || rangeEnd;

			if (abstractEvent) {
				if (abstractEvent._recurring) {

					// make a boolean hash as to whether the event occurs on each day-of-week
					if ((dow = abstractEvent.dow)) {
						dowHash = {};
						for (i = 0; i < dow.length; i++) {
							dowHash[dow[i]] = true;
						}
					}

					// iterate through every day in the current range
					date = _rangeStart.clone().stripTime(); // holds the date of the current day
					while (date.isBefore(_rangeEnd)) {

						if (!dowHash || dowHash[date.day()]) { // if everyday, or this particular day-of-week

							startTime = abstractEvent.start; // the stored start and end properties are times (Durations)
							endTime = abstractEvent.end; // "
							start = date.clone();
							end = null;

							if (startTime) {
								start = start.time(startTime);
							}
							if (endTime) {
								end = date.clone().time(endTime);
							}

							event = $.extend({}, abstractEvent); // make a copy of the original
							assignDatesToEvent(
								start, end,
								!startTime && !endTime, // allDay?
								event
							);
							events.push(event);
						}

						date.add(1, 'days');
					}
				}
				else {
					events.push(abstractEvent); // return the original event. will be a one-item array
				}
			}

			return events;
		}
		t.expandEvent = expandEvent;



		/* Event Modification Math
	-----------------------------------------------------------------------------------------*/


		// Modifies an event and all related events by applying the given properties.
		// Special date-diffing logic is used for manipulation of dates.
		// If `props` does not contain start/end dates, the updated values are assumed to be the event's current start/end.
		// All date comparisons are done against the event's pristine _start and _end dates.
		// Returns an object with delta information and a function to undo all operations.
		// For making computations in a granularity greater than day/time, specify largeUnit.
		// NOTE: The given `newProps` might be mutated for normalization purposes.
		function mutateEvent(event, newProps, largeUnit) {
			var miscProps = {};
			var oldProps;
			var clearEnd;
			var startDelta;
			var endDelta;
			var durationDelta;
			var undoFunc;

			// diffs the dates in the appropriate way, returning a duration
			function diffDates(date1, date0) { // date1 - date0
				if (largeUnit) {
					return diffByUnit(date1, date0, largeUnit);
				}
				else if (newProps.allDay) {
					return diffDay(date1, date0);
				}
				else {
					return diffDayTime(date1, date0);
				}
			}

			newProps = newProps || {};

			// normalize new date-related properties
			if (!newProps.start) {
				newProps.start = event.start.clone();
			}
			if (newProps.end === undefined) {
				newProps.end = event.end ? event.end.clone() : null;
			}
			if (newProps.allDay == null) { // is null or undefined?
				newProps.allDay = event.allDay;
			}
			normalizeEventDates(newProps);

			// create normalized versions of the original props to compare against
			// need a real end value, for diffing
			oldProps = {
				start: event._start.clone(),
				end: event._end ? event._end.clone() : t.getDefaultEventEnd(event._allDay, event._start),
				allDay: newProps.allDay // normalize the dates in the same regard as the new properties
			};
			normalizeEventDates(oldProps);

			// need to clear the end date if explicitly changed to null
			clearEnd = event._end !== null && newProps.end === null;

			// compute the delta for moving the start date
			startDelta = diffDates(newProps.start, oldProps.start);

			// compute the delta for moving the end date
			if (newProps.end) {
				endDelta = diffDates(newProps.end, oldProps.end);
				durationDelta = endDelta.subtract(startDelta);
			}
			else {
				durationDelta = null;
			}

			// gather all non-date-related properties
			$.each(newProps, function(name, val) {
				if (isMiscEventPropName(name)) {
					if (val !== undefined) {
						miscProps[name] = val;
					}
				}
			});

			// apply the operations to the event and all related events
			undoFunc = mutateEvents(
				clientEvents(event._id), // get events with this ID
				clearEnd,
				newProps.allDay,
				startDelta,
				durationDelta,
				miscProps
			);

			return {
				dateDelta: startDelta,
				durationDelta: durationDelta,
				undo: undoFunc
			};
		}


		// Modifies an array of events in the following ways (operations are in order):
		// - clear the event's `end`
		// - convert the event to allDay
		// - add `dateDelta` to the start and end
		// - add `durationDelta` to the event's duration
		// - assign `miscProps` to the event
		//
		// Returns a function that can be called to undo all the operations.
		//
		// TODO: don't use so many closures. possible memory issues when lots of events with same ID.
		//
		function mutateEvents(events, clearEnd, allDay, dateDelta, durationDelta, miscProps) {
			var isAmbigTimezone = t.getIsAmbigTimezone();
			var undoFunctions = [];

			// normalize zero-length deltas to be null
			if (dateDelta && !dateDelta.valueOf()) { dateDelta = null; }
			if (durationDelta && !durationDelta.valueOf()) { durationDelta = null; }

			$.each(events, function(i, event) {
				var oldProps;
				var newProps;

				// build an object holding all the old values, both date-related and misc.
				// for the undo function.
				oldProps = {
					start: event.start.clone(),
					end: event.end ? event.end.clone() : null,
					allDay: event.allDay
				};
				$.each(miscProps, function(name) {
					oldProps[name] = event[name];
				});

				// new date-related properties. work off the original date snapshot.
				// ok to use references because they will be thrown away when backupEventDates is called.
				newProps = {
					start: event._start,
					end: event._end,
					allDay: allDay // normalize the dates in the same regard as the new properties
				};
				normalizeEventDates(newProps); // massages start/end/allDay

				// strip or ensure the end date
				if (clearEnd) {
					newProps.end = null;
				}
				else if (durationDelta && !newProps.end) { // the duration translation requires an end date
					newProps.end = t.getDefaultEventEnd(newProps.allDay, newProps.start);
				}

				if (dateDelta) {
					newProps.start.add(dateDelta);
					if (newProps.end) {
						newProps.end.add(dateDelta);
					}
				}

				if (durationDelta) {
					newProps.end.add(durationDelta); // end already ensured above
				}

				// if the dates have changed, and we know it is impossible to recompute the
				// timezone offsets, strip the zone.
				if (
					isAmbigTimezone &&
					!newProps.allDay &&
					(dateDelta || durationDelta)
				) {
					newProps.start.stripZone();
					if (newProps.end) {
						newProps.end.stripZone();
					}
				}

				$.extend(event, miscProps, newProps); // copy over misc props, then date-related props
				backupEventDates(event); // regenerate internal _start/_end/_allDay

				undoFunctions.push(function() {
					$.extend(event, oldProps);
					backupEventDates(event); // regenerate internal _start/_end/_allDay
				});
			});

			return function() {
				for (var i = 0; i < undoFunctions.length; i++) {
					undoFunctions[i]();
				}
			};
		}

	}


// hook for external libs to manipulate event properties upon creation.
// should manipulate the event in-place.
	Calendar.prototype.normalizeEvent = function(event) {
	};


// Does the given span (start, end, and other location information)
// fully contain the other?
	Calendar.prototype.spanContainsSpan = function(outerSpan, innerSpan) {
		var eventStart = outerSpan.start.clone().stripZone();
		var eventEnd = this.getEventEnd(outerSpan).stripZone();

		return innerSpan.start >= eventStart && innerSpan.end <= eventEnd;
	};


// Returns a list of events that the given event should be compared against when being considered for a move to
// the specified span. Attached to the Calendar's prototype because EventManager is a mixin for a Calendar.
	Calendar.prototype.getPeerEvents = function(span, event) {
		var cache = this.getEventCache();
		var peerEvents = [];
		var i, otherEvent;

		for (i = 0; i < cache.length; i++) {
			otherEvent = cache[i];
			if (
				!event ||
				event._id !== otherEvent._id // don't compare the event to itself or other related [repeating] events
			) {
				peerEvents.push(otherEvent);
			}
		}

		return peerEvents;
	};


// updates the "backup" properties, which are preserved in order to compute diffs later on.
	function backupEventDates(event) {
		event._allDay = event.allDay;
		event._start = event.start.clone();
		event._end = event.end ? event.end.clone() : null;
	}


	/* Overlapping / Constraining
-----------------------------------------------------------------------------------------*/


// Determines if the given event can be relocated to the given span (unzoned start/end with other misc data)
	Calendar.prototype.isEventSpanAllowed = function(span, event) {
		var source = event.source || {};

		var constraint = firstDefined(
			event.constraint,
			source.constraint,
			this.options.eventConstraint
		);

		var overlap = firstDefined(
			event.overlap,
			source.overlap,
			this.options.eventOverlap
		);

		return this.isSpanAllowed(span, constraint, overlap, event) &&
			(!this.options.eventAllow || this.options.eventAllow(span, event) !== false);
	};


// Determines if an external event can be relocated to the given span (unzoned start/end with other misc data)
	Calendar.prototype.isExternalSpanAllowed = function(eventSpan, eventLocation, eventProps) {
		var eventInput;
		var event;

		// note: very similar logic is in View's reportExternalDrop
		if (eventProps) {
			eventInput = $.extend({}, eventProps, eventLocation);
			event = this.expandEvent(
				this.buildEventFromInput(eventInput)
			)[0];
		}

		if (event) {
			return this.isEventSpanAllowed(eventSpan, event);
		}
		else { // treat it as a selection

			return this.isSelectionSpanAllowed(eventSpan);
		}
	};


// Determines the given span (unzoned start/end with other misc data) can be selected.
	Calendar.prototype.isSelectionSpanAllowed = function(span) {
		return this.isSpanAllowed(span, this.options.selectConstraint, this.options.selectOverlap) &&
			(!this.options.selectAllow || this.options.selectAllow(span) !== false);
	};


// Returns true if the given span (caused by an event drop/resize or a selection) is allowed to exist
// according to the constraint/overlap settings.
// `event` is not required if checking a selection.
	Calendar.prototype.isSpanAllowed = function(span, constraint, overlap, event) {
		var constraintEvents;
		var anyContainment;
		var peerEvents;
		var i, peerEvent;
		var peerOverlap;

		// the range must be fully contained by at least one of produced constraint events
		if (constraint != null) {

			// not treated as an event! intermediate data structure
			// TODO: use ranges in the future
			constraintEvents = this.constraintToEvents(constraint);
			if (constraintEvents) { // not invalid

				anyContainment = false;
				for (i = 0; i < constraintEvents.length; i++) {
					if (this.spanContainsSpan(constraintEvents[i], span)) {
						anyContainment = true;
						break;
					}
				}

				if (!anyContainment) {
					return false;
				}
			}
		}

		peerEvents = this.getPeerEvents(span, event);

		for (i = 0; i < peerEvents.length; i++)  {
			peerEvent = peerEvents[i];

			// there needs to be an actual intersection before disallowing anything
			if (this.eventIntersectsRange(peerEvent, span)) {

				// evaluate overlap for the given range and short-circuit if necessary
				if (overlap === false) {
					return false;
				}
				// if the event's overlap is a test function, pass the peer event in question as the first param
				else if (typeof overlap === 'function' && !overlap(peerEvent, event)) {
					return false;
				}

				// if we are computing if the given range is allowable for an event, consider the other event's
				// EventObject-specific or Source-specific `overlap` property
				if (event) {
					peerOverlap = firstDefined(
						peerEvent.overlap,
						(peerEvent.source || {}).overlap
						// we already considered the global `eventOverlap`
					);
					if (peerOverlap === false) {
						return false;
					}
					// if the peer event's overlap is a test function, pass the subject event as the first param
					if (typeof peerOverlap === 'function' && !peerOverlap(event, peerEvent)) {
						return false;
					}
				}
			}
		}

		return true;
	};


// Given an event input from the API, produces an array of event objects. Possible event inputs:
// 'businessHours'
// An event ID (number or string)
// An object with specific start/end dates or a recurring event (like what businessHours accepts)
	Calendar.prototype.constraintToEvents = function(constraintInput) {

		if (constraintInput === 'businessHours') {
			return this.getCurrentBusinessHourEvents();
		}

		if (typeof constraintInput === 'object') {
			if (constraintInput.start != null) { // needs to be event-like input
				return this.expandEvent(this.buildEventFromInput(constraintInput));
			}
			else {
				return null; // invalid
			}
		}

		return this.clientEvents(constraintInput); // probably an ID
	};


// Does the event's date range intersect with the given range?
// start/end already assumed to have stripped zones :(
	Calendar.prototype.eventIntersectsRange = function(event, range) {
		var eventStart = event.start.clone().stripZone();
		var eventEnd = this.getEventEnd(event).stripZone();

		return range.start < eventEnd && range.end > eventStart;
	};


	/* Business Hours
-----------------------------------------------------------------------------------------*/

	var BUSINESS_HOUR_EVENT_DEFAULTS = {
		id: '_fcBusinessHours', // will relate events from different calls to expandEvent
		start: '09:00',
		end: '17:00',
		dow: [ 1, 2, 3, 4, 5 ], // monday - friday
		rendering: 'inverse-background'
		// classNames are defined in businessHoursSegClasses
	};

// Return events objects for business hours within the current view.
// Abuse of our event system :(
	Calendar.prototype.getCurrentBusinessHourEvents = function(wholeDay) {
		return this.computeBusinessHourEvents(wholeDay, this.options.businessHours);
	};

// Given a raw input value from options, return events objects for business hours within the current view.
	Calendar.prototype.computeBusinessHourEvents = function(wholeDay, input) {
		if (input === true) {
			return this.expandBusinessHourEvents(wholeDay, [ {} ]);
		}
		else if ($.isPlainObject(input)) {
			return this.expandBusinessHourEvents(wholeDay, [ input ]);
		}
		else if ($.isArray(input)) {
			return this.expandBusinessHourEvents(wholeDay, input, true);
		}
		else {
			return [];
		}
	};

// inputs expected to be an array of objects.
// if ignoreNoDow is true, will ignore entries that don't specify a day-of-week (dow) key.
	Calendar.prototype.expandBusinessHourEvents = function(wholeDay, inputs, ignoreNoDow) {
		var view = this.getView();
		var events = [];
		var i, input;

		for (i = 0; i < inputs.length; i++) {
			input = inputs[i];

			if (ignoreNoDow && !input.dow) {
				continue;
			}

			// give defaults. will make a copy
			input = $.extend({}, BUSINESS_HOUR_EVENT_DEFAULTS, input);

			// if a whole-day series is requested, clear the start/end times
			if (wholeDay) {
				input.start = null;
				input.end = null;
			}

			events.push.apply(events, // append
				this.expandEvent(
					this.buildEventFromInput(input),
					view.start,
					view.end
				)
			);
		}

		return events;
	};

	;;

	/* An abstract class for the "basic" views, as well as month view. Renders one or more rows of day cells.
----------------------------------------------------------------------------------------------------------------------*/
// It is a manager for a DayGrid subcomponent, which does most of the heavy lifting.
// It is responsible for managing width/height.

	var BasicView = FC.BasicView = View.extend({

		scroller: null,

		dayGridClass: DayGrid, // class the dayGrid will be instantiated from (overridable by subclasses)
		dayGrid: null, // the main subcomponent that does most of the heavy lifting

		dayNumbersVisible: false, // display day numbers on each day cell?
		colWeekNumbersVisible: false, // display week numbers along the side?
		cellWeekNumbersVisible: false, // display week numbers in day cell?

		weekNumberWidth: null, // width of all the week-number cells running down the side

		headContainerEl: null, // div that hold's the dayGrid's rendered date header
		headRowEl: null, // the fake row element of the day-of-week header


		initialize: function() {
			this.dayGrid = this.instantiateDayGrid();

			this.scroller = new Scroller({
				overflowX: 'hidden',
				overflowY: 'auto'
			});
		},


		// Generates the DayGrid object this view needs. Draws from this.dayGridClass
		instantiateDayGrid: function() {
			// generate a subclass on the fly with BasicView-specific behavior
			// TODO: cache this subclass
			var subclass = this.dayGridClass.extend(basicDayGridMethods);

			return new subclass(this);
		},


		// Sets the display range and computes all necessary dates
		setRange: function(range) {
			View.prototype.setRange.call(this, range); // call the super-method

			this.dayGrid.breakOnWeeks = /year|month|week/.test(this.intervalUnit); // do before setRange
			this.dayGrid.setRange(range);
		},


		// Compute the value to feed into setRange. Overrides superclass.
		computeRange: function(date) {
			var range = View.prototype.computeRange.call(this, date); // get value from the super-method

			// year and month views should be aligned with weeks. this is already done for week
			if (/year|month/.test(range.intervalUnit)) {
				range.start.startOf('week');
				range.start = this.skipHiddenDays(range.start);

				// make end-of-week if not already
				if (range.end.weekday()) {
					range.end.add(1, 'week').startOf('week');
					range.end = this.skipHiddenDays(range.end, -1, true); // exclusively move backwards
				}
			}

			return range;
		},


		// Renders the view into `this.el`, which should already be assigned
		renderDates: function() {

			this.dayNumbersVisible = this.dayGrid.rowCnt > 1; // TODO: make grid responsible
			if (this.opt('weekNumbers')) {
				if (this.opt('weekNumbersWithinDays')) {
					this.cellWeekNumbersVisible = true;
					this.colWeekNumbersVisible = false;
				}
				else {
					this.cellWeekNumbersVisible = false;
					this.colWeekNumbersVisible = true;
				};
			}
			this.dayGrid.numbersVisible = this.dayNumbersVisible ||
				this.cellWeekNumbersVisible || this.colWeekNumbersVisible;

			this.el.addClass('fc-basic-view').html(this.renderSkeletonHtml());
			this.renderHead();

			this.scroller.render();
			var dayGridContainerEl = this.scroller.el.addClass('fc-day-grid-container');
			var dayGridEl = $('<div class="fc-day-grid" />').appendTo(dayGridContainerEl);
			this.el.find('.fc-body > tr > td').append(dayGridContainerEl);

			this.dayGrid.setElement(dayGridEl);
			this.dayGrid.renderDates(this.hasRigidRows());
		},


		// render the day-of-week headers
		renderHead: function() {
			this.headContainerEl =
				this.el.find('.fc-head-container')
					.html(this.dayGrid.renderHeadHtml());
			this.headRowEl = this.headContainerEl.find('.fc-row');
		},


		// Unrenders the content of the view. Since we haven't separated skeleton rendering from date rendering,
		// always completely kill the dayGrid's rendering.
		unrenderDates: function() {
			this.dayGrid.unrenderDates();
			this.dayGrid.removeElement();
			this.scroller.destroy();
		},


		renderBusinessHours: function() {
			this.dayGrid.renderBusinessHours();
		},


		unrenderBusinessHours: function() {
			this.dayGrid.unrenderBusinessHours();
		},


		// Builds the HTML skeleton for the view.
		// The day-grid component will render inside of a container defined by this HTML.
		renderSkeletonHtml: function() {
			return '' +
				'<table>' +
				'<thead class="fc-head">' +
				'<tr>' +
				'<td class="fc-head-container ' + this.widgetHeaderClass + '"></td>' +
				'</tr>' +
				'</thead>' +
				'<tbody class="fc-body">' +
				'<tr>' +
				'<td class="' + this.widgetContentClass + '"></td>' +
				'</tr>' +
				'</tbody>' +
				'</table>';
		},


		// Generates an HTML attribute string for setting the width of the week number column, if it is known
		weekNumberStyleAttr: function() {
			if (this.weekNumberWidth !== null) {
				return 'style="width:' + this.weekNumberWidth + 'px"';
			}
			return '';
		},


		// Determines whether each row should have a constant height
		hasRigidRows: function() {
			var eventLimit = this.opt('eventLimit');
			return eventLimit && typeof eventLimit !== 'number';
		},


		/* Dimensions
	------------------------------------------------------------------------------------------------------------------*/


		// Refreshes the horizontal dimensions of the view
		updateWidth: function() {
			if (this.colWeekNumbersVisible) {
				// Make sure all week number cells running down the side have the same width.
				// Record the width for cells created later.
				this.weekNumberWidth = matchCellWidths(
					this.el.find('.fc-week-number')
				);
			}
		},


		// Adjusts the vertical dimensions of the view to the specified values
		setHeight: function(totalHeight, isAuto) {
			var eventLimit = this.opt('eventLimit');
			var scrollerHeight;
			var scrollbarWidths;

			// reset all heights to be natural
			this.scroller.clear();
			uncompensateScroll(this.headRowEl);

			this.dayGrid.removeSegPopover(); // kill the "more" popover if displayed

			// is the event limit a constant level number?
			if (eventLimit && typeof eventLimit === 'number') {
				this.dayGrid.limitRows(eventLimit); // limit the levels first so the height can redistribute after
			}

			// distribute the height to the rows
			// (totalHeight is a "recommended" value if isAuto)
			scrollerHeight = this.computeScrollerHeight(totalHeight);
			this.setGridHeight(scrollerHeight, isAuto);

			// is the event limit dynamically calculated?
			if (eventLimit && typeof eventLimit !== 'number') {
				this.dayGrid.limitRows(eventLimit); // limit the levels after the grid's row heights have been set
			}

			if (!isAuto) { // should we force dimensions of the scroll container?

				this.scroller.setHeight(scrollerHeight);
				scrollbarWidths = this.scroller.getScrollbarWidths();

				if (scrollbarWidths.left || scrollbarWidths.right) { // using scrollbars?

					compensateScroll(this.headRowEl, scrollbarWidths);

					// doing the scrollbar compensation might have created text overflow which created more height. redo
					scrollerHeight = this.computeScrollerHeight(totalHeight);
					this.scroller.setHeight(scrollerHeight);
				}

				// guarantees the same scrollbar widths
				this.scroller.lockOverflow(scrollbarWidths);
			}
		},


		// given a desired total height of the view, returns what the height of the scroller should be
		computeScrollerHeight: function(totalHeight) {
			return totalHeight -
				subtractInnerElHeight(this.el, this.scroller.el); // everything that's NOT the scroller
		},


		// Sets the height of just the DayGrid component in this view
		setGridHeight: function(height, isAuto) {
			if (isAuto) {
				undistributeHeight(this.dayGrid.rowEls); // let the rows be their natural height with no expanding
			}
			else {
				distributeHeight(this.dayGrid.rowEls, height, true); // true = compensate for height-hogging rows
			}
		},


		/* Scroll
	------------------------------------------------------------------------------------------------------------------*/


		computeInitialScroll: function() {
			return { top: 0 };
		},


		queryScroll: function() {
			return { top: this.scroller.getScrollTop() };
		},


		setScroll: function(scroll) {
			this.scroller.setScrollTop(scroll.top);
		},


		/* Hit Areas
	------------------------------------------------------------------------------------------------------------------*/
		// forward all hit-related method calls to dayGrid


		prepareHits: function() {
			this.dayGrid.prepareHits();
		},


		releaseHits: function() {
			this.dayGrid.releaseHits();
		},


		queryHit: function(left, top) {
			return this.dayGrid.queryHit(left, top);
		},


		getHitSpan: function(hit) {
			return this.dayGrid.getHitSpan(hit);
		},


		getHitEl: function(hit) {
			return this.dayGrid.getHitEl(hit);
		},


		/* Events
	------------------------------------------------------------------------------------------------------------------*/


		// Renders the given events onto the view and populates the segments array
		renderEvents: function(events) {
			this.dayGrid.renderEvents(events);

			this.updateHeight(); // must compensate for events that overflow the row
		},


		// Retrieves all segment objects that are rendered in the view
		getEventSegs: function() {
			return this.dayGrid.getEventSegs();
		},


		// Unrenders all event elements and clears internal segment data
		unrenderEvents: function() {
			this.dayGrid.unrenderEvents();

			// we DON'T need to call updateHeight() because
			// a renderEvents() call always happens after this, which will eventually call updateHeight()
		},


		/* Dragging (for both events and external elements)
	------------------------------------------------------------------------------------------------------------------*/


		// A returned value of `true` signals that a mock "helper" event has been rendered.
		renderDrag: function(dropLocation, seg) {
			return this.dayGrid.renderDrag(dropLocation, seg);
		},


		unrenderDrag: function() {
			this.dayGrid.unrenderDrag();
		},


		/* Selection
	------------------------------------------------------------------------------------------------------------------*/


		// Renders a visual indication of a selection
		renderSelection: function(span) {
			this.dayGrid.renderSelection(span);
		},


		// Unrenders a visual indications of a selection
		unrenderSelection: function() {
			this.dayGrid.unrenderSelection();
		}

	});


// Methods that will customize the rendering behavior of the BasicView's dayGrid
	var basicDayGridMethods = {


		// Generates the HTML that will go before the day-of week header cells
		renderHeadIntroHtml: function() {
			var view = this.view;

			if (view.colWeekNumbersVisible) {
				return '' +
					'<th class="fc-week-number ' + view.widgetHeaderClass + '" ' + view.weekNumberStyleAttr() + '>' +
					'<span>' + // needed for matchCellWidths
					htmlEscape(view.opt('weekNumberTitle')) +
					'</span>' +
					'</th>';
			}

			return '';
		},


		// Generates the HTML that will go before content-skeleton cells that display the day/week numbers
		renderNumberIntroHtml: function(row) {
			var view = this.view;
			var weekStart = this.getCellDate(row, 0);

			if (view.colWeekNumbersVisible) {
				return '' +
					'<td class="fc-week-number" ' + view.weekNumberStyleAttr() + '>' +
					view.buildGotoAnchorHtml( // aside from link, important for matchCellWidths
						{ date: weekStart, type: 'week', forceOff: this.colCnt === 1 },
						weekStart.format('w') // inner HTML
					) +
					'</td>';
			}

			return '';
		},


		// Generates the HTML that goes before the day bg cells for each day-row
		renderBgIntroHtml: function() {
			var view = this.view;

			if (view.colWeekNumbersVisible) {
				return '<td class="fc-week-number ' + view.widgetContentClass + '" ' +
					view.weekNumberStyleAttr() + '></td>';
			}

			return '';
		},


		// Generates the HTML that goes before every other type of row generated by DayGrid.
		// Affects helper-skeleton and highlight-skeleton rows.
		renderIntroHtml: function() {
			var view = this.view;

			if (view.colWeekNumbersVisible) {
				return '<td class="fc-week-number" ' + view.weekNumberStyleAttr() + '></td>';
			}

			return '';
		}

	};

	;;

	/* A month view with day cells running in rows (one-per-week) and columns
----------------------------------------------------------------------------------------------------------------------*/

	var MonthView = FC.MonthView = BasicView.extend({

		// Produces information about what range to display
		computeRange: function(date) {
			var range = BasicView.prototype.computeRange.call(this, date); // get value from super-method
			var rowCnt;

			// ensure 6 weeks
			if (this.isFixedWeeks()) {
				rowCnt = Math.ceil(range.end.diff(range.start, 'weeks', true)); // could be partial weeks due to hiddenDays
				range.end.add(6 - rowCnt, 'weeks');
			}

			return range;
		},


		// Overrides the default BasicView behavior to have special multi-week auto-height logic
		setGridHeight: function(height, isAuto) {

			// if auto, make the height of each row the height that it would be if there were 6 weeks
			if (isAuto) {
				height *= this.rowCnt / 6;
			}

			distributeHeight(this.dayGrid.rowEls, height, !isAuto); // if auto, don't compensate for height-hogging rows
		},


		isFixedWeeks: function() {
			return this.opt('fixedWeekCount');
		}

	});

	;;

	fcViews.basic = {
		'class': BasicView
	};

	fcViews.basicDay = {
		type: 'basic',
		duration: { days: 1 }
	};

	fcViews.basicWeek = {
		type: 'basic',
		duration: { weeks: 1 }
	};

	fcViews.month = {
		'class': MonthView,
		duration: { months: 1 }, // important for prev/next
		defaults: {
			fixedWeekCount: true
		}
	};
	;;

	/* An abstract class for all agenda-related views. Displays one more columns with time slots running vertically.
----------------------------------------------------------------------------------------------------------------------*/
// Is a manager for the TimeGrid subcomponent and possibly the DayGrid subcomponent (if allDaySlot is on).
// Responsible for managing width/height.

	var AgendaView = FC.AgendaView = View.extend({

		scroller: null,

		timeGridClass: TimeGrid, // class used to instantiate the timeGrid. subclasses can override
		timeGrid: null, // the main time-grid subcomponent of this view

		dayGridClass: DayGrid, // class used to instantiate the dayGrid. subclasses can override
		dayGrid: null, // the "all-day" subcomponent. if all-day is turned off, this will be null

		axisWidth: null, // the width of the time axis running down the side

		headContainerEl: null, // div that hold's the timeGrid's rendered date header
		noScrollRowEls: null, // set of fake row elements that must compensate when scroller has scrollbars

		// when the time-grid isn't tall enough to occupy the given height, we render an <hr> underneath
		bottomRuleEl: null,


		initialize: function() {
			this.timeGrid = this.instantiateTimeGrid();

			if (this.opt('allDaySlot')) { // should we display the "all-day" area?
				this.dayGrid = this.instantiateDayGrid(); // the all-day subcomponent of this view
			}

			this.scroller = new Scroller({
				overflowX: 'hidden',
				overflowY: 'auto'
			});
		},


		// Instantiates the TimeGrid object this view needs. Draws from this.timeGridClass
		instantiateTimeGrid: function() {
			var subclass = this.timeGridClass.extend(agendaTimeGridMethods);

			return new subclass(this);
		},


		// Instantiates the DayGrid object this view might need. Draws from this.dayGridClass
		instantiateDayGrid: function() {
			var subclass = this.dayGridClass.extend(agendaDayGridMethods);

			return new subclass(this);
		},


		/* Rendering
	------------------------------------------------------------------------------------------------------------------*/


		// Sets the display range and computes all necessary dates
		setRange: function(range) {
			View.prototype.setRange.call(this, range); // call the super-method

			this.timeGrid.setRange(range);
			if (this.dayGrid) {
				this.dayGrid.setRange(range);
			}
		},


		// Renders the view into `this.el`, which has already been assigned
		renderDates: function() {

			this.el.addClass('fc-agenda-view').html(this.renderSkeletonHtml());
			this.renderHead();

			this.scroller.render();
			var timeGridWrapEl = this.scroller.el.addClass('fc-time-grid-container');
			var timeGridEl = $('<div class="fc-time-grid" />').appendTo(timeGridWrapEl);
			this.el.find('.fc-body > tr > td').append(timeGridWrapEl);

			this.timeGrid.setElement(timeGridEl);
			this.timeGrid.renderDates();

			// the <hr> that sometimes displays under the time-grid
			this.bottomRuleEl = $('<hr class="fc-divider ' + this.widgetHeaderClass + '"/>')
				.appendTo(this.timeGrid.el); // inject it into the time-grid

			if (this.dayGrid) {
				this.dayGrid.setElement(this.el.find('.fc-day-grid'));
				this.dayGrid.renderDates();

				// have the day-grid extend it's coordinate area over the <hr> dividing the two grids
				this.dayGrid.bottomCoordPadding = this.dayGrid.el.next('hr').outerHeight();
			}

			this.noScrollRowEls = this.el.find('.fc-row:not(.fc-scroller *)'); // fake rows not within the scroller
		},


		// render the day-of-week headers
		renderHead: function() {
			this.headContainerEl =
				this.el.find('.fc-head-container')
					.html(this.timeGrid.renderHeadHtml());
		},


		// Unrenders the content of the view. Since we haven't separated skeleton rendering from date rendering,
		// always completely kill each grid's rendering.
		unrenderDates: function() {
			this.timeGrid.unrenderDates();
			this.timeGrid.removeElement();

			if (this.dayGrid) {
				this.dayGrid.unrenderDates();
				this.dayGrid.removeElement();
			}

			this.scroller.destroy();
		},


		// Builds the HTML skeleton for the view.
		// The day-grid and time-grid components will render inside containers defined by this HTML.
		renderSkeletonHtml: function() {
			return '' +
				'<table>' +
				'<thead class="fc-head">' +
				'<tr>' +
				'<td class="fc-head-container ' + this.widgetHeaderClass + '"></td>' +
				'</tr>' +
				'</thead>' +
				'<tbody class="fc-body">' +
				'<tr>' +
				'<td class="' + this.widgetContentClass + '">' +
				(this.dayGrid ?
						'<div class="fc-day-grid"/>' +
						'<hr class="fc-divider ' + this.widgetHeaderClass + '"/>' :
						''
				) +
				'</td>' +
				'</tr>' +
				'</tbody>' +
				'</table>';
		},


		// Generates an HTML attribute string for setting the width of the axis, if it is known
		axisStyleAttr: function() {
			if (this.axisWidth !== null) {
				return 'style="width:' + this.axisWidth + 'px"';
			}
			return '';
		},


		/* Business Hours
	------------------------------------------------------------------------------------------------------------------*/


		renderBusinessHours: function() {
			this.timeGrid.renderBusinessHours();

			if (this.dayGrid) {
				this.dayGrid.renderBusinessHours();
			}
		},


		unrenderBusinessHours: function() {
			this.timeGrid.unrenderBusinessHours();

			if (this.dayGrid) {
				this.dayGrid.unrenderBusinessHours();
			}
		},


		/* Now Indicator
	------------------------------------------------------------------------------------------------------------------*/


		getNowIndicatorUnit: function() {
			return this.timeGrid.getNowIndicatorUnit();
		},


		renderNowIndicator: function(date) {
			this.timeGrid.renderNowIndicator(date);
		},


		unrenderNowIndicator: function() {
			this.timeGrid.unrenderNowIndicator();
		},


		/* Dimensions
	------------------------------------------------------------------------------------------------------------------*/


		updateSize: function(isResize) {
			this.timeGrid.updateSize(isResize);

			View.prototype.updateSize.call(this, isResize); // call the super-method
		},


		// Refreshes the horizontal dimensions of the view
		updateWidth: function() {
			// make all axis cells line up, and record the width so newly created axis cells will have it
			this.axisWidth = matchCellWidths(this.el.find('.fc-axis'));
		},


		// Adjusts the vertical dimensions of the view to the specified values
		setHeight: function(totalHeight, isAuto) {
			var eventLimit;
			var scrollerHeight;
			var scrollbarWidths;

			// reset all dimensions back to the original state
			this.bottomRuleEl.hide(); // .show() will be called later if this <hr> is necessary
			this.scroller.clear(); // sets height to 'auto' and clears overflow
			uncompensateScroll(this.noScrollRowEls);

			// limit number of events in the all-day area
			if (this.dayGrid) {
				this.dayGrid.removeSegPopover(); // kill the "more" popover if displayed

				eventLimit = this.opt('eventLimit');
				if (eventLimit && typeof eventLimit !== 'number') {
					eventLimit = AGENDA_ALL_DAY_EVENT_LIMIT; // make sure "auto" goes to a real number
				}
				if (eventLimit) {
					this.dayGrid.limitRows(eventLimit);
				}
			}

			if (!isAuto) { // should we force dimensions of the scroll container?

				scrollerHeight = this.computeScrollerHeight(totalHeight);
				this.scroller.setHeight(scrollerHeight);
				scrollbarWidths = this.scroller.getScrollbarWidths();

				if (scrollbarWidths.left || scrollbarWidths.right) { // using scrollbars?

					// make the all-day and header rows lines up
					compensateScroll(this.noScrollRowEls, scrollbarWidths);

					// the scrollbar compensation might have changed text flow, which might affect height, so recalculate
					// and reapply the desired height to the scroller.
					scrollerHeight = this.computeScrollerHeight(totalHeight);
					this.scroller.setHeight(scrollerHeight);
				}

				// guarantees the same scrollbar widths
				this.scroller.lockOverflow(scrollbarWidths);

				// if there's any space below the slats, show the horizontal rule.
				// this won't cause any new overflow, because lockOverflow already called.
				if (this.timeGrid.getTotalSlatHeight() < scrollerHeight) {
					this.bottomRuleEl.show();
				}
			}
		},


		// given a desired total height of the view, returns what the height of the scroller should be
		computeScrollerHeight: function(totalHeight) {
			return totalHeight -
				subtractInnerElHeight(this.el, this.scroller.el); // everything that's NOT the scroller
		},


		/* Scroll
	------------------------------------------------------------------------------------------------------------------*/


		// Computes the initial pre-configured scroll state prior to allowing the user to change it
		computeInitialScroll: function() {
			var scrollTime = moment.duration(this.opt('scrollTime'));
			var top = this.timeGrid.computeTimeTop(scrollTime);

			// zoom can give weird floating-point values. rather scroll a little bit further
			top = Math.ceil(top);

			if (top) {
				top++; // to overcome top border that slots beyond the first have. looks better
			}

			return { top: top };
		},


		queryScroll: function() {
			return { top: this.scroller.getScrollTop() };
		},


		setScroll: function(scroll) {
			this.scroller.setScrollTop(scroll.top);
		},


		/* Hit Areas
	------------------------------------------------------------------------------------------------------------------*/
		// forward all hit-related method calls to the grids (dayGrid might not be defined)


		prepareHits: function() {
			this.timeGrid.prepareHits();
			if (this.dayGrid) {
				this.dayGrid.prepareHits();
			}
		},


		releaseHits: function() {
			this.timeGrid.releaseHits();
			if (this.dayGrid) {
				this.dayGrid.releaseHits();
			}
		},


		queryHit: function(left, top) {
			var hit = this.timeGrid.queryHit(left, top);

			if (!hit && this.dayGrid) {
				hit = this.dayGrid.queryHit(left, top);
			}

			return hit;
		},


		getHitSpan: function(hit) {
			// TODO: hit.component is set as a hack to identify where the hit came from
			return hit.component.getHitSpan(hit);
		},


		getHitEl: function(hit) {
			// TODO: hit.component is set as a hack to identify where the hit came from
			return hit.component.getHitEl(hit);
		},


		/* Events
	------------------------------------------------------------------------------------------------------------------*/


		// Renders events onto the view and populates the View's segment array
		renderEvents: function(events) {
			var dayEvents = [];
			var timedEvents = [];
			var daySegs = [];
			var timedSegs;
			var i;

			// separate the events into all-day and timed
			for (i = 0; i < events.length; i++) {
				if (events[i].allDay) {
					dayEvents.push(events[i]);
				}
				else {
					timedEvents.push(events[i]);
				}
			}

			// render the events in the subcomponents
			timedSegs = this.timeGrid.renderEvents(timedEvents);
			if (this.dayGrid) {
				daySegs = this.dayGrid.renderEvents(dayEvents);
			}

			// the all-day area is flexible and might have a lot of events, so shift the height
			this.updateHeight();
		},


		// Retrieves all segment objects that are rendered in the view
		getEventSegs: function() {
			return this.timeGrid.getEventSegs().concat(
				this.dayGrid ? this.dayGrid.getEventSegs() : []
			);
		},


		// Unrenders all event elements and clears internal segment data
		unrenderEvents: function() {

			// unrender the events in the subcomponents
			this.timeGrid.unrenderEvents();
			if (this.dayGrid) {
				this.dayGrid.unrenderEvents();
			}

			// we DON'T need to call updateHeight() because
			// a renderEvents() call always happens after this, which will eventually call updateHeight()
		},


		/* Dragging (for events and external elements)
	------------------------------------------------------------------------------------------------------------------*/


		// A returned value of `true` signals that a mock "helper" event has been rendered.
		renderDrag: function(dropLocation, seg) {
			if (dropLocation.start.hasTime()) {
				return this.timeGrid.renderDrag(dropLocation, seg);
			}
			else if (this.dayGrid) {
				return this.dayGrid.renderDrag(dropLocation, seg);
			}
		},


		unrenderDrag: function() {
			this.timeGrid.unrenderDrag();
			if (this.dayGrid) {
				this.dayGrid.unrenderDrag();
			}
		},


		/* Selection
	------------------------------------------------------------------------------------------------------------------*/


		// Renders a visual indication of a selection
		renderSelection: function(span) {
			if (span.start.hasTime() || span.end.hasTime()) {
				this.timeGrid.renderSelection(span);
			}
			else if (this.dayGrid) {
				this.dayGrid.renderSelection(span);
			}
		},


		// Unrenders a visual indications of a selection
		unrenderSelection: function() {
			this.timeGrid.unrenderSelection();
			if (this.dayGrid) {
				this.dayGrid.unrenderSelection();
			}
		}

	});


// Methods that will customize the rendering behavior of the AgendaView's timeGrid
// TODO: move into TimeGrid
	var agendaTimeGridMethods = {


		// Generates the HTML that will go before the day-of week header cells
		renderHeadIntroHtml: function() {
			var view = this.view;
			var weekText;

			if (view.opt('weekNumbers')) {
				weekText = this.start.format(view.opt('smallWeekFormat'));

				return '' +
					'<th class="fc-axis fc-week-number ' + view.widgetHeaderClass + '" ' + view.axisStyleAttr() + '>' +
					view.buildGotoAnchorHtml( // aside from link, important for matchCellWidths
						{ date: this.start, type: 'week', forceOff: this.colCnt > 1 },
						htmlEscape(weekText) // inner HTML
					) +
					'</th>';
			}
			else {
				return '<th class="fc-axis ' + view.widgetHeaderClass + '" ' + view.axisStyleAttr() + '></th>';
			}
		},


		// Generates the HTML that goes before the bg of the TimeGrid slot area. Long vertical column.
		renderBgIntroHtml: function() {
			var view = this.view;

			return '<td class="fc-axis ' + view.widgetContentClass + '" ' + view.axisStyleAttr() + '></td>';
		},


		// Generates the HTML that goes before all other types of cells.
		// Affects content-skeleton, helper-skeleton, highlight-skeleton for both the time-grid and day-grid.
		renderIntroHtml: function() {
			var view = this.view;

			return '<td class="fc-axis" ' + view.axisStyleAttr() + '></td>';
		}

	};


// Methods that will customize the rendering behavior of the AgendaView's dayGrid
	var agendaDayGridMethods = {


		// Generates the HTML that goes before the all-day cells
		renderBgIntroHtml: function() {
			var view = this.view;

			return '' +
				'<td class="fc-axis ' + view.widgetContentClass + '" ' + view.axisStyleAttr() + '>' +
				'<span>' + // needed for matchCellWidths
				view.getAllDayHtml() +
				'</span>' +
				'</td>';
		},


		// Generates the HTML that goes before all other types of cells.
		// Affects content-skeleton, helper-skeleton, highlight-skeleton for both the time-grid and day-grid.
		renderIntroHtml: function() {
			var view = this.view;

			return '<td class="fc-axis" ' + view.axisStyleAttr() + '></td>';
		}

	};

	;;

	var AGENDA_ALL_DAY_EVENT_LIMIT = 5;

// potential nice values for the slot-duration and interval-duration
// from largest to smallest
	var AGENDA_STOCK_SUB_DURATIONS = [
		{ hours: 1 },
		{ minutes: 30 },
		{ minutes: 15 },
		{ seconds: 30 },
		{ seconds: 15 }
	];

	fcViews.agenda = {
		'class': AgendaView,
		defaults: {
			allDaySlot: true,
			slotDuration: '00:30:00',
			minTime: '00:00:00',
			maxTime: '24:00:00',
			slotEventOverlap: true // a bad name. confused with overlap/constraint system
		}
	};

	fcViews.agendaDay = {
		type: 'agenda',
		duration: { days: 1 }
	};

	fcViews.agendaWeek = {
		type: 'agenda',
		duration: { weeks: 1 }
	};
	;;

	/*
Responsible for the scroller, and forwarding event-related actions into the "grid"
*/
	var ListView = View.extend({

		grid: null,
		scroller: null,

		initialize: function() {
			this.grid = new ListViewGrid(this);
			this.scroller = new Scroller({
				overflowX: 'hidden',
				overflowY: 'auto'
			});
		},

		setRange: function(range) {
			View.prototype.setRange.call(this, range); // super

			this.grid.setRange(range); // needs to process range-related options
		},

		renderSkeleton: function() {
			this.el.addClass(
				'fc-list-view ' +
				this.widgetContentClass
			);

			this.scroller.render();
			this.scroller.el.appendTo(this.el);

			this.grid.setElement(this.scroller.scrollEl);
		},

		unrenderSkeleton: function() {
			this.scroller.destroy(); // will remove the Grid too
		},

		setHeight: function(totalHeight, isAuto) {
			this.scroller.setHeight(this.computeScrollerHeight(totalHeight));
		},

		computeScrollerHeight: function(totalHeight) {
			return totalHeight -
				subtractInnerElHeight(this.el, this.scroller.el); // everything that's NOT the scroller
		},

		renderEvents: function(events) {
			this.grid.renderEvents(events);
		},

		unrenderEvents: function() {
			this.grid.unrenderEvents();
		},

		isEventResizable: function(event) {
			return false;
		},

		isEventDraggable: function(event) {
			return false;
		}

	});

	/*
Responsible for event rendering and user-interaction.
Its "el" is the inner-content of the above view's scroller.
*/
	var ListViewGrid = Grid.extend({

		segSelector: '.fc-list-item', // which elements accept event actions
		hasDayInteractions: false, // no day selection or day clicking

		// slices by day
		spanToSegs: function(span) {
			var view = this.view;
			var dayStart = view.start.clone().time(0); // timed, so segs get times!
			var dayIndex = 0;
			var seg;
			var segs = [];

			while (dayStart < view.end) {

				seg = intersectRanges(span, {
					start: dayStart,
					end: dayStart.clone().add(1, 'day')
				});

				if (seg) {
					seg.dayIndex = dayIndex;
					segs.push(seg);
				}

				dayStart.add(1, 'day');
				dayIndex++;

				// detect when span won't go fully into the next day,
				// and mutate the latest seg to the be the end.
				if (
					seg && !seg.isEnd && span.end.hasTime() &&
					span.end < dayStart.clone().add(this.view.nextDayThreshold)
				) {
					seg.end = span.end.clone();
					seg.isEnd = true;
					break;
				}
			}

			return segs;
		},

		// like "4:00am"
		computeEventTimeFormat: function() {
			return this.view.opt('mediumTimeFormat');
		},

		// for events with a url, the whole <tr> should be clickable,
		// but it's impossible to wrap with an <a> tag. simulate this.
		handleSegClick: function(seg, ev) {
			var url;

			Grid.prototype.handleSegClick.apply(this, arguments); // super. might prevent the default action

			// not clicking on or within an <a> with an href
			if (!$(ev.target).closest('a[href]').length) {
				url = seg.event.url;
				if (url && !ev.isDefaultPrevented()) { // jsEvent not cancelled in handler
					window.location.href = url; // simulate link click
				}
			}
		},

		// returns list of foreground segs that were actually rendered
		renderFgSegs: function(segs) {
			segs = this.renderFgSegEls(segs); // might filter away hidden events

			if (!segs.length) {
				this.renderEmptyMessage();
			}
			else {
				this.renderSegList(segs);
			}

			return segs;
		},

		renderEmptyMessage: function() {
			this.el.html(
				'<div class="fc-list-empty-wrap2">' + // TODO: try less wraps
				'<div class="fc-list-empty-wrap1">' +
				'<div class="fc-list-empty">' +
				htmlEscape(this.view.opt('noEventsMessage')) +
				'</div>' +
				'</div>' +
				'</div>'
			);
		},

		// render the event segments in the view
		renderSegList: function(allSegs) {
			var segsByDay = this.groupSegsByDay(allSegs); // sparse array
			var dayIndex;
			var daySegs;
			var i;
			var tableEl = $('<table class="fc-list-table"><tbody/></table>');
			var tbodyEl = tableEl.find('tbody');

			for (dayIndex = 0; dayIndex < segsByDay.length; dayIndex++) {
				daySegs = segsByDay[dayIndex];
				if (daySegs) { // sparse array, so might be undefined

					// append a day header
					tbodyEl.append(this.dayHeaderHtml(
						this.view.start.clone().add(dayIndex, 'days')
					));

					this.sortEventSegs(daySegs);

					for (i = 0; i < daySegs.length; i++) {
						tbodyEl.append(daySegs[i].el); // append event row
					}
				}
			}

			this.el.empty().append(tableEl);
		},

		// Returns a sparse array of arrays, segs grouped by their dayIndex
		groupSegsByDay: function(segs) {
			var segsByDay = []; // sparse array
			var i, seg;

			for (i = 0; i < segs.length; i++) {
				seg = segs[i];
				(segsByDay[seg.dayIndex] || (segsByDay[seg.dayIndex] = []))
					.push(seg);
			}

			return segsByDay;
		},

		// generates the HTML for the day headers that live amongst the event rows
		dayHeaderHtml: function(dayDate) {
			var view = this.view;
			var mainFormat = view.opt('listDayFormat');
			var altFormat = view.opt('listDayAltFormat');

			return '<tr class="fc-list-heading" data-date="' + dayDate.format('YYYY-MM-DD') + '">' +
				'<td class="' + view.widgetHeaderClass + '" colspan="3">' +
				(mainFormat ?
					view.buildGotoAnchorHtml(
						dayDate,
						{ 'class': 'fc-list-heading-main' },
						htmlEscape(dayDate.format(mainFormat)) // inner HTML
					) :
					'') +
				(altFormat ?
					view.buildGotoAnchorHtml(
						dayDate,
						{ 'class': 'fc-list-heading-alt' },
						htmlEscape(dayDate.format(altFormat)) // inner HTML
					) :
					'') +
				'</td>' +
				'</tr>';
		},

		// generates the HTML for a single event row
		fgSegHtml: function(seg) {
			var view = this.view;
			var classes = [ 'fc-list-item' ].concat(this.getSegCustomClasses(seg));
			var bgColor = this.getSegBackgroundColor(seg);
			var event = seg.event;
			var url = event.url;
			var timeHtml;

			if (event.allDay) {
				timeHtml = view.getAllDayHtml();
			}
			else if (view.isMultiDayEvent(event)) { // if the event appears to span more than one day
				if (seg.isStart || seg.isEnd) { // outer segment that probably lasts part of the day
					timeHtml = htmlEscape(this.getEventTimeText(seg));
				}
				else { // inner segment that lasts the whole day
					timeHtml = view.getAllDayHtml();
				}
			}
			else {
				// Display the normal time text for the *event's* times
				timeHtml = htmlEscape(this.getEventTimeText(event));
			}

			if (url) {
				classes.push('fc-has-url');
			}

			return '<tr class="' + classes.join(' ') + '">' +
				(this.displayEventTime ?
					'<td class="fc-list-item-time ' + view.widgetContentClass + '">' +
					(timeHtml || '') +
					'</td>' :
					'') +
				'<td class="fc-list-item-marker ' + view.widgetContentClass + '">' +
				'<span class="fc-event-dot"' +
				(bgColor ?
					' style="background-color:' + bgColor + '"' :
					'') +
				'></span>' +
				'</td>' +
				'<td class="fc-list-item-title ' + view.widgetContentClass + '">' +
				'<a' + (url ? ' href="' + htmlEscape(url) + '"' : '') + '>' +
				htmlEscape(seg.event.title || '') +
				'</a>' +
				'</td>' +
				'</tr>';
		}

	});

	;;

	fcViews.list = {
		'class': ListView,
		buttonTextKey: 'list', // what to lookup in locale files
		defaults: {
			buttonText: 'list', // text to display for English
			listDayFormat: 'LL', // like "January 1, 2016"
			noEventsMessage: 'No events to display'
		}
	};

	fcViews.listDay = {
		type: 'list',
		duration: { days: 1 },
		defaults: {
			listDayFormat: 'dddd' // day-of-week is all we need. full date is probably in header
		}
	};

	fcViews.listWeek = {
		type: 'list',
		duration: { weeks: 1 },
		defaults: {
			listDayFormat: 'dddd', // day-of-week is more important
			listDayAltFormat: 'LL'
		}
	};

	fcViews.listMonth = {
		type: 'list',
		duration: { month: 1 },
		defaults: {
			listDayAltFormat: 'dddd' // day-of-week is nice-to-have
		}
	};

	fcViews.listYear = {
		type: 'list',
		duration: { year: 1 },
		defaults: {
			listDayAltFormat: 'dddd' // day-of-week is nice-to-have
		}
	};

	;;

	return FC; // export for Node/CommonJS
});


//! moment.js
//! version : 2.20.1
//! authors : Tim Wood, Iskren Chernev, Moment.js contributors
//! license : MIT
//! momentjs.com

;(function (global, factory) {
	typeof exports === 'object' && typeof module !== 'undefined' ? module.exports = factory() :
		typeof define === 'function' && define.amd ? define(factory) :
			global.moment = factory()
}(this, (function () { 'use strict';

	var hookCallback;

	function hooks () {
		return hookCallback.apply(null, arguments);
	}

// This is done to register the method called with moment()
// without creating circular dependencies.
	function setHookCallback (callback) {
		hookCallback = callback;
	}

	function isArray(input) {
		return input instanceof Array || Object.prototype.toString.call(input) === '[object Array]';
	}

	function isObject(input) {
		// IE8 will treat undefined and null as object if it wasn't for
		// input != null
		return input != null && Object.prototype.toString.call(input) === '[object Object]';
	}

	function isObjectEmpty(obj) {
		if (Object.getOwnPropertyNames) {
			return (Object.getOwnPropertyNames(obj).length === 0);
		} else {
			var k;
			for (k in obj) {
				if (obj.hasOwnProperty(k)) {
					return false;
				}
			}
			return true;
		}
	}

	function isUndefined(input) {
		return input === void 0;
	}

	function isNumber(input) {
		return typeof input === 'number' || Object.prototype.toString.call(input) === '[object Number]';
	}

	function isDate(input) {
		return input instanceof Date || Object.prototype.toString.call(input) === '[object Date]';
	}

	function map(arr, fn) {
		var res = [], i;
		for (i = 0; i < arr.length; ++i) {
			res.push(fn(arr[i], i));
		}
		return res;
	}

	function hasOwnProp(a, b) {
		return Object.prototype.hasOwnProperty.call(a, b);
	}

	function extend(a, b) {
		for (var i in b) {
			if (hasOwnProp(b, i)) {
				a[i] = b[i];
			}
		}

		if (hasOwnProp(b, 'toString')) {
			a.toString = b.toString;
		}

		if (hasOwnProp(b, 'valueOf')) {
			a.valueOf = b.valueOf;
		}

		return a;
	}

	function createUTC (input, format, locale, strict) {
		return createLocalOrUTC(input, format, locale, strict, true).utc();
	}

	function defaultParsingFlags() {
		// We need to deep clone this object.
		return {
			empty           : false,
			unusedTokens    : [],
			unusedInput     : [],
			overflow        : -2,
			charsLeftOver   : 0,
			nullInput       : false,
			invalidMonth    : null,
			invalidFormat   : false,
			userInvalidated : false,
			iso             : false,
			parsedDateParts : [],
			meridiem        : null,
			rfc2822         : false,
			weekdayMismatch : false
		};
	}

	function getParsingFlags(m) {
		if (m._pf == null) {
			m._pf = defaultParsingFlags();
		}
		return m._pf;
	}

	var some;
	if (Array.prototype.some) {
		some = Array.prototype.some;
	} else {
		some = function (fun) {
			var t = Object(this);
			var len = t.length >>> 0;

			for (var i = 0; i < len; i++) {
				if (i in t && fun.call(this, t[i], i, t)) {
					return true;
				}
			}

			return false;
		};
	}

	function isValid(m) {
		if (m._isValid == null) {
			var flags = getParsingFlags(m);
			var parsedParts = some.call(flags.parsedDateParts, function (i) {
				return i != null;
			});
			var isNowValid = !isNaN(m._d.getTime()) &&
				flags.overflow < 0 &&
				!flags.empty &&
				!flags.invalidMonth &&
				!flags.invalidWeekday &&
				!flags.weekdayMismatch &&
				!flags.nullInput &&
				!flags.invalidFormat &&
				!flags.userInvalidated &&
				(!flags.meridiem || (flags.meridiem && parsedParts));

			if (m._strict) {
				isNowValid = isNowValid &&
					flags.charsLeftOver === 0 &&
					flags.unusedTokens.length === 0 &&
					flags.bigHour === undefined;
			}

			if (Object.isFrozen == null || !Object.isFrozen(m)) {
				m._isValid = isNowValid;
			}
			else {
				return isNowValid;
			}
		}
		return m._isValid;
	}

	function createInvalid (flags) {
		var m = createUTC(NaN);
		if (flags != null) {
			extend(getParsingFlags(m), flags);
		}
		else {
			getParsingFlags(m).userInvalidated = true;
		}

		return m;
	}

// Plugins that add properties should also add the key here (null value),
// so we can properly clone ourselves.
	var momentProperties = hooks.momentProperties = [];

	function copyConfig(to, from) {
		var i, prop, val;

		if (!isUndefined(from._isAMomentObject)) {
			to._isAMomentObject = from._isAMomentObject;
		}
		if (!isUndefined(from._i)) {
			to._i = from._i;
		}
		if (!isUndefined(from._f)) {
			to._f = from._f;
		}
		if (!isUndefined(from._l)) {
			to._l = from._l;
		}
		if (!isUndefined(from._strict)) {
			to._strict = from._strict;
		}
		if (!isUndefined(from._tzm)) {
			to._tzm = from._tzm;
		}
		if (!isUndefined(from._isUTC)) {
			to._isUTC = from._isUTC;
		}
		if (!isUndefined(from._offset)) {
			to._offset = from._offset;
		}
		if (!isUndefined(from._pf)) {
			to._pf = getParsingFlags(from);
		}
		if (!isUndefined(from._locale)) {
			to._locale = from._locale;
		}

		if (momentProperties.length > 0) {
			for (i = 0; i < momentProperties.length; i++) {
				prop = momentProperties[i];
				val = from[prop];
				if (!isUndefined(val)) {
					to[prop] = val;
				}
			}
		}

		return to;
	}

	var updateInProgress = false;

// Moment prototype object
	function Moment(config) {
		copyConfig(this, config);
		this._d = new Date(config._d != null ? config._d.getTime() : NaN);
		if (!this.isValid()) {
			this._d = new Date(NaN);
		}
		// Prevent infinite loop in case updateOffset creates new moment
		// objects.
		if (updateInProgress === false) {
			updateInProgress = true;
			hooks.updateOffset(this);
			updateInProgress = false;
		}
	}

	function isMoment (obj) {
		return obj instanceof Moment || (obj != null && obj._isAMomentObject != null);
	}

	function absFloor (number) {
		if (number < 0) {
			// -0 -> 0
			return Math.ceil(number) || 0;
		} else {
			return Math.floor(number);
		}
	}

	function toInt(argumentForCoercion) {
		var coercedNumber = +argumentForCoercion,
			value = 0;

		if (coercedNumber !== 0 && isFinite(coercedNumber)) {
			value = absFloor(coercedNumber);
		}

		return value;
	}

// compare two arrays, return the number of differences
	function compareArrays(array1, array2, dontConvert) {
		var len = Math.min(array1.length, array2.length),
			lengthDiff = Math.abs(array1.length - array2.length),
			diffs = 0,
			i;
		for (i = 0; i < len; i++) {
			if ((dontConvert && array1[i] !== array2[i]) ||
				(!dontConvert && toInt(array1[i]) !== toInt(array2[i]))) {
				diffs++;
			}
		}
		return diffs + lengthDiff;
	}

	function warn(msg) {
		if (hooks.suppressDeprecationWarnings === false &&
			(typeof console !==  'undefined') && console.warn) {
			console.warn('Deprecation warning: ' + msg);
		}
	}

	function deprecate(msg, fn) {
		var firstTime = true;

		return extend(function () {
			if (hooks.deprecationHandler != null) {
				hooks.deprecationHandler(null, msg);
			}
			if (firstTime) {
				var args = [];
				var arg;
				for (var i = 0; i < arguments.length; i++) {
					arg = '';
					if (typeof arguments[i] === 'object') {
						arg += '\n[' + i + '] ';
						for (var key in arguments[0]) {
							arg += key + ': ' + arguments[0][key] + ', ';
						}
						arg = arg.slice(0, -2); // Remove trailing comma and space
					} else {
						arg = arguments[i];
					}
					args.push(arg);
				}
				warn(msg + '\nArguments: ' + Array.prototype.slice.call(args).join('') + '\n' + (new Error()).stack);
				firstTime = false;
			}
			return fn.apply(this, arguments);
		}, fn);
	}

	var deprecations = {};

	function deprecateSimple(name, msg) {
		if (hooks.deprecationHandler != null) {
			hooks.deprecationHandler(name, msg);
		}
		if (!deprecations[name]) {
			warn(msg);
			deprecations[name] = true;
		}
	}

	hooks.suppressDeprecationWarnings = false;
	hooks.deprecationHandler = null;

	function isFunction(input) {
		return input instanceof Function || Object.prototype.toString.call(input) === '[object Function]';
	}

	function set (config) {
		var prop, i;
		for (i in config) {
			prop = config[i];
			if (isFunction(prop)) {
				this[i] = prop;
			} else {
				this['_' + i] = prop;
			}
		}
		this._config = config;
		// Lenient ordinal parsing accepts just a number in addition to
		// number + (possibly) stuff coming from _dayOfMonthOrdinalParse.
		// TODO: Remove "ordinalParse" fallback in next major release.
		this._dayOfMonthOrdinalParseLenient = new RegExp(
			(this._dayOfMonthOrdinalParse.source || this._ordinalParse.source) +
			'|' + (/\d{1,2}/).source);
	}

	function mergeConfigs(parentConfig, childConfig) {
		var res = extend({}, parentConfig), prop;
		for (prop in childConfig) {
			if (hasOwnProp(childConfig, prop)) {
				if (isObject(parentConfig[prop]) && isObject(childConfig[prop])) {
					res[prop] = {};
					extend(res[prop], parentConfig[prop]);
					extend(res[prop], childConfig[prop]);
				} else if (childConfig[prop] != null) {
					res[prop] = childConfig[prop];
				} else {
					delete res[prop];
				}
			}
		}
		for (prop in parentConfig) {
			if (hasOwnProp(parentConfig, prop) &&
				!hasOwnProp(childConfig, prop) &&
				isObject(parentConfig[prop])) {
				// make sure changes to properties don't modify parent config
				res[prop] = extend({}, res[prop]);
			}
		}
		return res;
	}

	function Locale(config) {
		if (config != null) {
			this.set(config);
		}
	}

	var keys;

	if (Object.keys) {
		keys = Object.keys;
	} else {
		keys = function (obj) {
			var i, res = [];
			for (i in obj) {
				if (hasOwnProp(obj, i)) {
					res.push(i);
				}
			}
			return res;
		};
	}

	var defaultCalendar = {
		sameDay : '[Today at] LT',
		nextDay : '[Tomorrow at] LT',
		nextWeek : 'dddd [at] LT',
		lastDay : '[Yesterday at] LT',
		lastWeek : '[Last] dddd [at] LT',
		sameElse : 'L'
	};

	function calendar (key, mom, now) {
		var output = this._calendar[key] || this._calendar['sameElse'];
		return isFunction(output) ? output.call(mom, now) : output;
	}

	var defaultLongDateFormat = {
		LTS  : 'h:mm:ss A',
		LT   : 'h:mm A',
		L    : 'MM/DD/YYYY',
		LL   : 'MMMM D, YYYY',
		LLL  : 'MMMM D, YYYY h:mm A',
		LLLL : 'dddd, MMMM D, YYYY h:mm A'
	};

	function longDateFormat (key) {
		var format = this._longDateFormat[key],
			formatUpper = this._longDateFormat[key.toUpperCase()];

		if (format || !formatUpper) {
			return format;
		}

		this._longDateFormat[key] = formatUpper.replace(/MMMM|MM|DD|dddd/g, function (val) {
			return val.slice(1);
		});

		return this._longDateFormat[key];
	}

	var defaultInvalidDate = 'Invalid date';

	function invalidDate () {
		return this._invalidDate;
	}

	var defaultOrdinal = '%d';
	var defaultDayOfMonthOrdinalParse = /\d{1,2}/;

	function ordinal (number) {
		return this._ordinal.replace('%d', number);
	}

	var defaultRelativeTime = {
		future : 'in %s',
		past   : '%s ago',
		s  : 'a few seconds',
		ss : '%d seconds',
		m  : 'a minute',
		mm : '%d minutes',
		h  : 'an hour',
		hh : '%d hours',
		d  : 'a day',
		dd : '%d days',
		M  : 'a month',
		MM : '%d months',
		y  : 'a year',
		yy : '%d years'
	};

	function relativeTime (number, withoutSuffix, string, isFuture) {
		var output = this._relativeTime[string];
		return (isFunction(output)) ?
			output(number, withoutSuffix, string, isFuture) :
			output.replace(/%d/i, number);
	}

	function pastFuture (diff, output) {
		var format = this._relativeTime[diff > 0 ? 'future' : 'past'];
		return isFunction(format) ? format(output) : format.replace(/%s/i, output);
	}

	var aliases = {};

	function addUnitAlias (unit, shorthand) {
		var lowerCase = unit.toLowerCase();
		aliases[lowerCase] = aliases[lowerCase + 's'] = aliases[shorthand] = unit;
	}

	function normalizeUnits(units) {
		return typeof units === 'string' ? aliases[units] || aliases[units.toLowerCase()] : undefined;
	}

	function normalizeObjectUnits(inputObject) {
		var normalizedInput = {},
			normalizedProp,
			prop;

		for (prop in inputObject) {
			if (hasOwnProp(inputObject, prop)) {
				normalizedProp = normalizeUnits(prop);
				if (normalizedProp) {
					normalizedInput[normalizedProp] = inputObject[prop];
				}
			}
		}

		return normalizedInput;
	}

	var priorities = {};

	function addUnitPriority(unit, priority) {
		priorities[unit] = priority;
	}

	function getPrioritizedUnits(unitsObj) {
		var units = [];
		for (var u in unitsObj) {
			units.push({unit: u, priority: priorities[u]});
		}
		units.sort(function (a, b) {
			return a.priority - b.priority;
		});
		return units;
	}

	function zeroFill(number, targetLength, forceSign) {
		var absNumber = '' + Math.abs(number),
			zerosToFill = targetLength - absNumber.length,
			sign = number >= 0;
		return (sign ? (forceSign ? '+' : '') : '-') +
			Math.pow(10, Math.max(0, zerosToFill)).toString().substr(1) + absNumber;
	}

	var formattingTokens = /(\[[^\[]*\])|(\\)?([Hh]mm(ss)?|Mo|MM?M?M?|Do|DDDo|DD?D?D?|ddd?d?|do?|w[o|w]?|W[o|W]?|Qo?|YYYYYY|YYYYY|YYYY|YY|gg(ggg?)?|GG(GGG?)?|e|E|a|A|hh?|HH?|kk?|mm?|ss?|S{1,9}|x|X|zz?|ZZ?|.)/g;

	var localFormattingTokens = /(\[[^\[]*\])|(\\)?(LTS|LT|LL?L?L?|l{1,4})/g;

	var formatFunctions = {};

	var formatTokenFunctions = {};

// token:    'M'
// padded:   ['MM', 2]
// ordinal:  'Mo'
// callback: function () { this.month() + 1 }
	function addFormatToken (token, padded, ordinal, callback) {
		var func = callback;
		if (typeof callback === 'string') {
			func = function () {
				return this[callback]();
			};
		}
		if (token) {
			formatTokenFunctions[token] = func;
		}
		if (padded) {
			formatTokenFunctions[padded[0]] = function () {
				return zeroFill(func.apply(this, arguments), padded[1], padded[2]);
			};
		}
		if (ordinal) {
			formatTokenFunctions[ordinal] = function () {
				return this.localeData().ordinal(func.apply(this, arguments), token);
			};
		}
	}

	function removeFormattingTokens(input) {
		if (input.match(/\[[\s\S]/)) {
			return input.replace(/^\[|\]$/g, '');
		}
		return input.replace(/\\/g, '');
	}

	function makeFormatFunction(format) {
		var array = format.match(formattingTokens), i, length;

		for (i = 0, length = array.length; i < length; i++) {
			if (formatTokenFunctions[array[i]]) {
				array[i] = formatTokenFunctions[array[i]];
			} else {
				array[i] = removeFormattingTokens(array[i]);
			}
		}

		return function (mom) {
			var output = '', i;
			for (i = 0; i < length; i++) {
				output += isFunction(array[i]) ? array[i].call(mom, format) : array[i];
			}
			return output;
		};
	}

// format date using native date object
	function formatMoment(m, format) {
		if (!m.isValid()) {
			return m.localeData().invalidDate();
		}

		format = expandFormat(format, m.localeData());
		formatFunctions[format] = formatFunctions[format] || makeFormatFunction(format);

		return formatFunctions[format](m);
	}

	function expandFormat(format, locale) {
		var i = 5;

		function replaceLongDateFormatTokens(input) {
			return locale.longDateFormat(input) || input;
		}

		localFormattingTokens.lastIndex = 0;
		while (i >= 0 && localFormattingTokens.test(format)) {
			format = format.replace(localFormattingTokens, replaceLongDateFormatTokens);
			localFormattingTokens.lastIndex = 0;
			i -= 1;
		}

		return format;
	}

	var match1         = /\d/;            //       0 - 9
	var match2         = /\d\d/;          //      00 - 99
	var match3         = /\d{3}/;         //     000 - 999
	var match4         = /\d{4}/;         //    0000 - 9999
	var match6         = /[+-]?\d{6}/;    // -999999 - 999999
	var match1to2      = /\d\d?/;         //       0 - 99
	var match3to4      = /\d\d\d\d?/;     //     999 - 9999
	var match5to6      = /\d\d\d\d\d\d?/; //   99999 - 999999
	var match1to3      = /\d{1,3}/;       //       0 - 999
	var match1to4      = /\d{1,4}/;       //       0 - 9999
	var match1to6      = /[+-]?\d{1,6}/;  // -999999 - 999999

	var matchUnsigned  = /\d+/;           //       0 - inf
	var matchSigned    = /[+-]?\d+/;      //    -inf - inf

	var matchOffset    = /Z|[+-]\d\d:?\d\d/gi; // +00:00 -00:00 +0000 -0000 or Z
	var matchShortOffset = /Z|[+-]\d\d(?::?\d\d)?/gi; // +00 -00 +00:00 -00:00 +0000 -0000 or Z

	var matchTimestamp = /[+-]?\d+(\.\d{1,3})?/; // 123456789 123456789.123

// any word (or two) characters or numbers including two/three word month in arabic.
// includes scottish gaelic two word and hyphenated months
	var matchWord = /[0-9]{0,256}['a-z\u00A0-\u05FF\u0700-\uD7FF\uF900-\uFDCF\uFDF0-\uFF07\uFF10-\uFFEF]{1,256}|[\u0600-\u06FF\/]{1,256}(\s*?[\u0600-\u06FF]{1,256}){1,2}/i;


	var regexes = {};

	function addRegexToken (token, regex, strictRegex) {
		regexes[token] = isFunction(regex) ? regex : function (isStrict, localeData) {
			return (isStrict && strictRegex) ? strictRegex : regex;
		};
	}

	function getParseRegexForToken (token, config) {
		if (!hasOwnProp(regexes, token)) {
			return new RegExp(unescapeFormat(token));
		}

		return regexes[token](config._strict, config._locale);
	}

// Code from http://stackoverflow.com/questions/3561493/is-there-a-regexp-escape-function-in-javascript
	function unescapeFormat(s) {
		return regexEscape(s.replace('\\', '').replace(/\\(\[)|\\(\])|\[([^\]\[]*)\]|\\(.)/g, function (matched, p1, p2, p3, p4) {
			return p1 || p2 || p3 || p4;
		}));
	}

	function regexEscape(s) {
		return s.replace(/[-\/\\^$*+?.()|[\]{}]/g, '\\$&');
	}

	var tokens = {};

	function addParseToken (token, callback) {
		var i, func = callback;
		if (typeof token === 'string') {
			token = [token];
		}
		if (isNumber(callback)) {
			func = function (input, array) {
				array[callback] = toInt(input);
			};
		}
		for (i = 0; i < token.length; i++) {
			tokens[token[i]] = func;
		}
	}

	function addWeekParseToken (token, callback) {
		addParseToken(token, function (input, array, config, token) {
			config._w = config._w || {};
			callback(input, config._w, config, token);
		});
	}

	function addTimeToArrayFromToken(token, input, config) {
		if (input != null && hasOwnProp(tokens, token)) {
			tokens[token](input, config._a, config, token);
		}
	}

	var YEAR = 0;
	var MONTH = 1;
	var DATE = 2;
	var HOUR = 3;
	var MINUTE = 4;
	var SECOND = 5;
	var MILLISECOND = 6;
	var WEEK = 7;
	var WEEKDAY = 8;

// FORMATTING

	addFormatToken('Y', 0, 0, function () {
		var y = this.year();
		return y <= 9999 ? '' + y : '+' + y;
	});

	addFormatToken(0, ['YY', 2], 0, function () {
		return this.year() % 100;
	});

	addFormatToken(0, ['YYYY',   4],       0, 'year');
	addFormatToken(0, ['YYYYY',  5],       0, 'year');
	addFormatToken(0, ['YYYYYY', 6, true], 0, 'year');

// ALIASES

	addUnitAlias('year', 'y');

// PRIORITIES

	addUnitPriority('year', 1);

// PARSING

	addRegexToken('Y',      matchSigned);
	addRegexToken('YY',     match1to2, match2);
	addRegexToken('YYYY',   match1to4, match4);
	addRegexToken('YYYYY',  match1to6, match6);
	addRegexToken('YYYYYY', match1to6, match6);

	addParseToken(['YYYYY', 'YYYYYY'], YEAR);
	addParseToken('YYYY', function (input, array) {
		array[YEAR] = input.length === 2 ? hooks.parseTwoDigitYear(input) : toInt(input);
	});
	addParseToken('YY', function (input, array) {
		array[YEAR] = hooks.parseTwoDigitYear(input);
	});
	addParseToken('Y', function (input, array) {
		array[YEAR] = parseInt(input, 10);
	});

// HELPERS

	function daysInYear(year) {
		return isLeapYear(year) ? 366 : 365;
	}

	function isLeapYear(year) {
		return (year % 4 === 0 && year % 100 !== 0) || year % 400 === 0;
	}

// HOOKS

	hooks.parseTwoDigitYear = function (input) {
		return toInt(input) + (toInt(input) > 68 ? 1900 : 2000);
	};

// MOMENTS

	var getSetYear = makeGetSet('FullYear', true);

	function getIsLeapYear () {
		return isLeapYear(this.year());
	}

	function makeGetSet (unit, keepTime) {
		return function (value) {
			if (value != null) {
				set$1(this, unit, value);
				hooks.updateOffset(this, keepTime);
				return this;
			} else {
				return get(this, unit);
			}
		};
	}

	function get (mom, unit) {
		return mom.isValid() ?
			mom._d['get' + (mom._isUTC ? 'UTC' : '') + unit]() : NaN;
	}

	function set$1 (mom, unit, value) {
		if (mom.isValid() && !isNaN(value)) {
			if (unit === 'FullYear' && isLeapYear(mom.year()) && mom.month() === 1 && mom.date() === 29) {
				mom._d['set' + (mom._isUTC ? 'UTC' : '') + unit](value, mom.month(), daysInMonth(value, mom.month()));
			}
			else {
				mom._d['set' + (mom._isUTC ? 'UTC' : '') + unit](value);
			}
		}
	}

// MOMENTS

	function stringGet (units) {
		units = normalizeUnits(units);
		if (isFunction(this[units])) {
			return this[units]();
		}
		return this;
	}


	function stringSet (units, value) {
		if (typeof units === 'object') {
			units = normalizeObjectUnits(units);
			var prioritized = getPrioritizedUnits(units);
			for (var i = 0; i < prioritized.length; i++) {
				this[prioritized[i].unit](units[prioritized[i].unit]);
			}
		} else {
			units = normalizeUnits(units);
			if (isFunction(this[units])) {
				return this[units](value);
			}
		}
		return this;
	}

	function mod(n, x) {
		return ((n % x) + x) % x;
	}

	var indexOf;

	if (Array.prototype.indexOf) {
		indexOf = Array.prototype.indexOf;
	} else {
		indexOf = function (o) {
			// I know
			var i;
			for (i = 0; i < this.length; ++i) {
				if (this[i] === o) {
					return i;
				}
			}
			return -1;
		};
	}

	function daysInMonth(year, month) {
		if (isNaN(year) || isNaN(month)) {
			return NaN;
		}
		var modMonth = mod(month, 12);
		year += (month - modMonth) / 12;
		return modMonth === 1 ? (isLeapYear(year) ? 29 : 28) : (31 - modMonth % 7 % 2);
	}

// FORMATTING

	addFormatToken('M', ['MM', 2], 'Mo', function () {
		return this.month() + 1;
	});

	addFormatToken('MMM', 0, 0, function (format) {
		return this.localeData().monthsShort(this, format);
	});

	addFormatToken('MMMM', 0, 0, function (format) {
		return this.localeData().months(this, format);
	});

// ALIASES

	addUnitAlias('month', 'M');

// PRIORITY

	addUnitPriority('month', 8);

// PARSING

	addRegexToken('M',    match1to2);
	addRegexToken('MM',   match1to2, match2);
	addRegexToken('MMM',  function (isStrict, locale) {
		return locale.monthsShortRegex(isStrict);
	});
	addRegexToken('MMMM', function (isStrict, locale) {
		return locale.monthsRegex(isStrict);
	});

	addParseToken(['M', 'MM'], function (input, array) {
		array[MONTH] = toInt(input) - 1;
	});

	addParseToken(['MMM', 'MMMM'], function (input, array, config, token) {
		var month = config._locale.monthsParse(input, token, config._strict);
		// if we didn't find a month name, mark the date as invalid.
		if (month != null) {
			array[MONTH] = month;
		} else {
			getParsingFlags(config).invalidMonth = input;
		}
	});

// LOCALES

	var MONTHS_IN_FORMAT = /D[oD]?(\[[^\[\]]*\]|\s)+MMMM?/;
	var defaultLocaleMonths = 'January_February_March_April_May_June_July_August_September_October_November_December'.split('_');
	function localeMonths (m, format) {
		if (!m) {
			return isArray(this._months) ? this._months :
				this._months['standalone'];
		}
		return isArray(this._months) ? this._months[m.month()] :
			this._months[(this._months.isFormat || MONTHS_IN_FORMAT).test(format) ? 'format' : 'standalone'][m.month()];
	}

	var defaultLocaleMonthsShort = 'Jan_Feb_Mar_Apr_May_Jun_Jul_Aug_Sep_Oct_Nov_Dec'.split('_');
	function localeMonthsShort (m, format) {
		if (!m) {
			return isArray(this._monthsShort) ? this._monthsShort :
				this._monthsShort['standalone'];
		}
		return isArray(this._monthsShort) ? this._monthsShort[m.month()] :
			this._monthsShort[MONTHS_IN_FORMAT.test(format) ? 'format' : 'standalone'][m.month()];
	}

	function handleStrictParse(monthName, format, strict) {
		var i, ii, mom, llc = monthName.toLocaleLowerCase();
		if (!this._monthsParse) {
			// this is not used
			this._monthsParse = [];
			this._longMonthsParse = [];
			this._shortMonthsParse = [];
			for (i = 0; i < 12; ++i) {
				mom = createUTC([2000, i]);
				this._shortMonthsParse[i] = this.monthsShort(mom, '').toLocaleLowerCase();
				this._longMonthsParse[i] = this.months(mom, '').toLocaleLowerCase();
			}
		}

		if (strict) {
			if (format === 'MMM') {
				ii = indexOf.call(this._shortMonthsParse, llc);
				return ii !== -1 ? ii : null;
			} else {
				ii = indexOf.call(this._longMonthsParse, llc);
				return ii !== -1 ? ii : null;
			}
		} else {
			if (format === 'MMM') {
				ii = indexOf.call(this._shortMonthsParse, llc);
				if (ii !== -1) {
					return ii;
				}
				ii = indexOf.call(this._longMonthsParse, llc);
				return ii !== -1 ? ii : null;
			} else {
				ii = indexOf.call(this._longMonthsParse, llc);
				if (ii !== -1) {
					return ii;
				}
				ii = indexOf.call(this._shortMonthsParse, llc);
				return ii !== -1 ? ii : null;
			}
		}
	}

	function localeMonthsParse (monthName, format, strict) {
		var i, mom, regex;

		if (this._monthsParseExact) {
			return handleStrictParse.call(this, monthName, format, strict);
		}

		if (!this._monthsParse) {
			this._monthsParse = [];
			this._longMonthsParse = [];
			this._shortMonthsParse = [];
		}

		// TODO: add sorting
		// Sorting makes sure if one month (or abbr) is a prefix of another
		// see sorting in computeMonthsParse
		for (i = 0; i < 12; i++) {
			// make the regex if we don't have it already
			mom = createUTC([2000, i]);
			if (strict && !this._longMonthsParse[i]) {
				this._longMonthsParse[i] = new RegExp('^' + this.months(mom, '').replace('.', '') + '$', 'i');
				this._shortMonthsParse[i] = new RegExp('^' + this.monthsShort(mom, '').replace('.', '') + '$', 'i');
			}
			if (!strict && !this._monthsParse[i]) {
				regex = '^' + this.months(mom, '') + '|^' + this.monthsShort(mom, '');
				this._monthsParse[i] = new RegExp(regex.replace('.', ''), 'i');
			}
			// test the regex
			if (strict && format === 'MMMM' && this._longMonthsParse[i].test(monthName)) {
				return i;
			} else if (strict && format === 'MMM' && this._shortMonthsParse[i].test(monthName)) {
				return i;
			} else if (!strict && this._monthsParse[i].test(monthName)) {
				return i;
			}
		}
	}

// MOMENTS

	function setMonth (mom, value) {
		var dayOfMonth;

		if (!mom.isValid()) {
			// No op
			return mom;
		}

		if (typeof value === 'string') {
			if (/^\d+$/.test(value)) {
				value = toInt(value);
			} else {
				value = mom.localeData().monthsParse(value);
				// TODO: Another silent failure?
				if (!isNumber(value)) {
					return mom;
				}
			}
		}

		dayOfMonth = Math.min(mom.date(), daysInMonth(mom.year(), value));
		mom._d['set' + (mom._isUTC ? 'UTC' : '') + 'Month'](value, dayOfMonth);
		return mom;
	}

	function getSetMonth (value) {
		if (value != null) {
			setMonth(this, value);
			hooks.updateOffset(this, true);
			return this;
		} else {
			return get(this, 'Month');
		}
	}

	function getDaysInMonth () {
		return daysInMonth(this.year(), this.month());
	}

	var defaultMonthsShortRegex = matchWord;
	function monthsShortRegex (isStrict) {
		if (this._monthsParseExact) {
			if (!hasOwnProp(this, '_monthsRegex')) {
				computeMonthsParse.call(this);
			}
			if (isStrict) {
				return this._monthsShortStrictRegex;
			} else {
				return this._monthsShortRegex;
			}
		} else {
			if (!hasOwnProp(this, '_monthsShortRegex')) {
				this._monthsShortRegex = defaultMonthsShortRegex;
			}
			return this._monthsShortStrictRegex && isStrict ?
				this._monthsShortStrictRegex : this._monthsShortRegex;
		}
	}

	var defaultMonthsRegex = matchWord;
	function monthsRegex (isStrict) {
		if (this._monthsParseExact) {
			if (!hasOwnProp(this, '_monthsRegex')) {
				computeMonthsParse.call(this);
			}
			if (isStrict) {
				return this._monthsStrictRegex;
			} else {
				return this._monthsRegex;
			}
		} else {
			if (!hasOwnProp(this, '_monthsRegex')) {
				this._monthsRegex = defaultMonthsRegex;
			}
			return this._monthsStrictRegex && isStrict ?
				this._monthsStrictRegex : this._monthsRegex;
		}
	}

	function computeMonthsParse () {
		function cmpLenRev(a, b) {
			return b.length - a.length;
		}

		var shortPieces = [], longPieces = [], mixedPieces = [],
			i, mom;
		for (i = 0; i < 12; i++) {
			// make the regex if we don't have it already
			mom = createUTC([2000, i]);
			shortPieces.push(this.monthsShort(mom, ''));
			longPieces.push(this.months(mom, ''));
			mixedPieces.push(this.months(mom, ''));
			mixedPieces.push(this.monthsShort(mom, ''));
		}
		// Sorting makes sure if one month (or abbr) is a prefix of another it
		// will match the longer piece.
		shortPieces.sort(cmpLenRev);
		longPieces.sort(cmpLenRev);
		mixedPieces.sort(cmpLenRev);
		for (i = 0; i < 12; i++) {
			shortPieces[i] = regexEscape(shortPieces[i]);
			longPieces[i] = regexEscape(longPieces[i]);
		}
		for (i = 0; i < 24; i++) {
			mixedPieces[i] = regexEscape(mixedPieces[i]);
		}

		this._monthsRegex = new RegExp('^(' + mixedPieces.join('|') + ')', 'i');
		this._monthsShortRegex = this._monthsRegex;
		this._monthsStrictRegex = new RegExp('^(' + longPieces.join('|') + ')', 'i');
		this._monthsShortStrictRegex = new RegExp('^(' + shortPieces.join('|') + ')', 'i');
	}

	function createDate (y, m, d, h, M, s, ms) {
		// can't just apply() to create a date:
		// https://stackoverflow.com/q/181348
		var date = new Date(y, m, d, h, M, s, ms);

		// the date constructor remaps years 0-99 to 1900-1999
		if (y < 100 && y >= 0 && isFinite(date.getFullYear())) {
			date.setFullYear(y);
		}
		return date;
	}

	function createUTCDate (y) {
		var date = new Date(Date.UTC.apply(null, arguments));

		// the Date.UTC function remaps years 0-99 to 1900-1999
		if (y < 100 && y >= 0 && isFinite(date.getUTCFullYear())) {
			date.setUTCFullYear(y);
		}
		return date;
	}

// start-of-first-week - start-of-year
	function firstWeekOffset(year, dow, doy) {
		var // first-week day -- which january is always in the first week (4 for iso, 1 for other)
			fwd = 7 + dow - doy,
			// first-week day local weekday -- which local weekday is fwd
			fwdlw = (7 + createUTCDate(year, 0, fwd).getUTCDay() - dow) % 7;

		return -fwdlw + fwd - 1;
	}

// https://en.wikipedia.org/wiki/ISO_week_date#Calculating_a_date_given_the_year.2C_week_number_and_weekday
	function dayOfYearFromWeeks(year, week, weekday, dow, doy) {
		var localWeekday = (7 + weekday - dow) % 7,
			weekOffset = firstWeekOffset(year, dow, doy),
			dayOfYear = 1 + 7 * (week - 1) + localWeekday + weekOffset,
			resYear, resDayOfYear;

		if (dayOfYear <= 0) {
			resYear = year - 1;
			resDayOfYear = daysInYear(resYear) + dayOfYear;
		} else if (dayOfYear > daysInYear(year)) {
			resYear = year + 1;
			resDayOfYear = dayOfYear - daysInYear(year);
		} else {
			resYear = year;
			resDayOfYear = dayOfYear;
		}

		return {
			year: resYear,
			dayOfYear: resDayOfYear
		};
	}

	function weekOfYear(mom, dow, doy) {
		var weekOffset = firstWeekOffset(mom.year(), dow, doy),
			week = Math.floor((mom.dayOfYear() - weekOffset - 1) / 7) + 1,
			resWeek, resYear;

		if (week < 1) {
			resYear = mom.year() - 1;
			resWeek = week + weeksInYear(resYear, dow, doy);
		} else if (week > weeksInYear(mom.year(), dow, doy)) {
			resWeek = week - weeksInYear(mom.year(), dow, doy);
			resYear = mom.year() + 1;
		} else {
			resYear = mom.year();
			resWeek = week;
		}

		return {
			week: resWeek,
			year: resYear
		};
	}

	function weeksInYear(year, dow, doy) {
		var weekOffset = firstWeekOffset(year, dow, doy),
			weekOffsetNext = firstWeekOffset(year + 1, dow, doy);
		return (daysInYear(year) - weekOffset + weekOffsetNext) / 7;
	}

// FORMATTING

	addFormatToken('w', ['ww', 2], 'wo', 'week');
	addFormatToken('W', ['WW', 2], 'Wo', 'isoWeek');

// ALIASES

	addUnitAlias('week', 'w');
	addUnitAlias('isoWeek', 'W');

// PRIORITIES

	addUnitPriority('week', 5);
	addUnitPriority('isoWeek', 5);

// PARSING

	addRegexToken('w',  match1to2);
	addRegexToken('ww', match1to2, match2);
	addRegexToken('W',  match1to2);
	addRegexToken('WW', match1to2, match2);

	addWeekParseToken(['w', 'ww', 'W', 'WW'], function (input, week, config, token) {
		week[token.substr(0, 1)] = toInt(input);
	});

// HELPERS

// LOCALES

	function localeWeek (mom) {
		return weekOfYear(mom, this._week.dow, this._week.doy).week;
	}

	var defaultLocaleWeek = {
		dow : 0, // Sunday is the first day of the week.
		doy : 6  // The week that contains Jan 1st is the first week of the year.
	};

	function localeFirstDayOfWeek () {
		return this._week.dow;
	}

	function localeFirstDayOfYear () {
		return this._week.doy;
	}

// MOMENTS

	function getSetWeek (input) {
		var week = this.localeData().week(this);
		return input == null ? week : this.add((input - week) * 7, 'd');
	}

	function getSetISOWeek (input) {
		var week = weekOfYear(this, 1, 4).week;
		return input == null ? week : this.add((input - week) * 7, 'd');
	}

// FORMATTING

	addFormatToken('d', 0, 'do', 'day');

	addFormatToken('dd', 0, 0, function (format) {
		return this.localeData().weekdaysMin(this, format);
	});

	addFormatToken('ddd', 0, 0, function (format) {
		return this.localeData().weekdaysShort(this, format);
	});

	addFormatToken('dddd', 0, 0, function (format) {
		return this.localeData().weekdays(this, format);
	});

	addFormatToken('e', 0, 0, 'weekday');
	addFormatToken('E', 0, 0, 'isoWeekday');

// ALIASES

	addUnitAlias('day', 'd');
	addUnitAlias('weekday', 'e');
	addUnitAlias('isoWeekday', 'E');

// PRIORITY
	addUnitPriority('day', 11);
	addUnitPriority('weekday', 11);
	addUnitPriority('isoWeekday', 11);

// PARSING

	addRegexToken('d',    match1to2);
	addRegexToken('e',    match1to2);
	addRegexToken('E',    match1to2);
	addRegexToken('dd',   function (isStrict, locale) {
		return locale.weekdaysMinRegex(isStrict);
	});
	addRegexToken('ddd',   function (isStrict, locale) {
		return locale.weekdaysShortRegex(isStrict);
	});
	addRegexToken('dddd',   function (isStrict, locale) {
		return locale.weekdaysRegex(isStrict);
	});

	addWeekParseToken(['dd', 'ddd', 'dddd'], function (input, week, config, token) {
		var weekday = config._locale.weekdaysParse(input, token, config._strict);
		// if we didn't get a weekday name, mark the date as invalid
		if (weekday != null) {
			week.d = weekday;
		} else {
			getParsingFlags(config).invalidWeekday = input;
		}
	});

	addWeekParseToken(['d', 'e', 'E'], function (input, week, config, token) {
		week[token] = toInt(input);
	});

// HELPERS

	function parseWeekday(input, locale) {
		if (typeof input !== 'string') {
			return input;
		}

		if (!isNaN(input)) {
			return parseInt(input, 10);
		}

		input = locale.weekdaysParse(input);
		if (typeof input === 'number') {
			return input;
		}

		return null;
	}

	function parseIsoWeekday(input, locale) {
		if (typeof input === 'string') {
			return locale.weekdaysParse(input) % 7 || 7;
		}
		return isNaN(input) ? null : input;
	}

// LOCALES

	var defaultLocaleWeekdays = 'Sunday_Monday_Tuesday_Wednesday_Thursday_Friday_Saturday'.split('_');
	function localeWeekdays (m, format) {
		if (!m) {
			return isArray(this._weekdays) ? this._weekdays :
				this._weekdays['standalone'];
		}
		return isArray(this._weekdays) ? this._weekdays[m.day()] :
			this._weekdays[this._weekdays.isFormat.test(format) ? 'format' : 'standalone'][m.day()];
	}

	var defaultLocaleWeekdaysShort = 'Sun_Mon_Tue_Wed_Thu_Fri_Sat'.split('_');
	function localeWeekdaysShort (m) {
		return (m) ? this._weekdaysShort[m.day()] : this._weekdaysShort;
	}

	var defaultLocaleWeekdaysMin = 'Su_Mo_Tu_We_Th_Fr_Sa'.split('_');
	function localeWeekdaysMin (m) {
		return (m) ? this._weekdaysMin[m.day()] : this._weekdaysMin;
	}

	function handleStrictParse$1(weekdayName, format, strict) {
		var i, ii, mom, llc = weekdayName.toLocaleLowerCase();
		if (!this._weekdaysParse) {
			this._weekdaysParse = [];
			this._shortWeekdaysParse = [];
			this._minWeekdaysParse = [];

			for (i = 0; i < 7; ++i) {
				mom = createUTC([2000, 1]).day(i);
				this._minWeekdaysParse[i] = this.weekdaysMin(mom, '').toLocaleLowerCase();
				this._shortWeekdaysParse[i] = this.weekdaysShort(mom, '').toLocaleLowerCase();
				this._weekdaysParse[i] = this.weekdays(mom, '').toLocaleLowerCase();
			}
		}

		if (strict) {
			if (format === 'dddd') {
				ii = indexOf.call(this._weekdaysParse, llc);
				return ii !== -1 ? ii : null;
			} else if (format === 'ddd') {
				ii = indexOf.call(this._shortWeekdaysParse, llc);
				return ii !== -1 ? ii : null;
			} else {
				ii = indexOf.call(this._minWeekdaysParse, llc);
				return ii !== -1 ? ii : null;
			}
		} else {
			if (format === 'dddd') {
				ii = indexOf.call(this._weekdaysParse, llc);
				if (ii !== -1) {
					return ii;
				}
				ii = indexOf.call(this._shortWeekdaysParse, llc);
				if (ii !== -1) {
					return ii;
				}
				ii = indexOf.call(this._minWeekdaysParse, llc);
				return ii !== -1 ? ii : null;
			} else if (format === 'ddd') {
				ii = indexOf.call(this._shortWeekdaysParse, llc);
				if (ii !== -1) {
					return ii;
				}
				ii = indexOf.call(this._weekdaysParse, llc);
				if (ii !== -1) {
					return ii;
				}
				ii = indexOf.call(this._minWeekdaysParse, llc);
				return ii !== -1 ? ii : null;
			} else {
				ii = indexOf.call(this._minWeekdaysParse, llc);
				if (ii !== -1) {
					return ii;
				}
				ii = indexOf.call(this._weekdaysParse, llc);
				if (ii !== -1) {
					return ii;
				}
				ii = indexOf.call(this._shortWeekdaysParse, llc);
				return ii !== -1 ? ii : null;
			}
		}
	}

	function localeWeekdaysParse (weekdayName, format, strict) {
		var i, mom, regex;

		if (this._weekdaysParseExact) {
			return handleStrictParse$1.call(this, weekdayName, format, strict);
		}

		if (!this._weekdaysParse) {
			this._weekdaysParse = [];
			this._minWeekdaysParse = [];
			this._shortWeekdaysParse = [];
			this._fullWeekdaysParse = [];
		}

		for (i = 0; i < 7; i++) {
			// make the regex if we don't have it already

			mom = createUTC([2000, 1]).day(i);
			if (strict && !this._fullWeekdaysParse[i]) {
				this._fullWeekdaysParse[i] = new RegExp('^' + this.weekdays(mom, '').replace('.', '\.?') + '$', 'i');
				this._shortWeekdaysParse[i] = new RegExp('^' + this.weekdaysShort(mom, '').replace('.', '\.?') + '$', 'i');
				this._minWeekdaysParse[i] = new RegExp('^' + this.weekdaysMin(mom, '').replace('.', '\.?') + '$', 'i');
			}
			if (!this._weekdaysParse[i]) {
				regex = '^' + this.weekdays(mom, '') + '|^' + this.weekdaysShort(mom, '') + '|^' + this.weekdaysMin(mom, '');
				this._weekdaysParse[i] = new RegExp(regex.replace('.', ''), 'i');
			}
			// test the regex
			if (strict && format === 'dddd' && this._fullWeekdaysParse[i].test(weekdayName)) {
				return i;
			} else if (strict && format === 'ddd' && this._shortWeekdaysParse[i].test(weekdayName)) {
				return i;
			} else if (strict && format === 'dd' && this._minWeekdaysParse[i].test(weekdayName)) {
				return i;
			} else if (!strict && this._weekdaysParse[i].test(weekdayName)) {
				return i;
			}
		}
	}

// MOMENTS

	function getSetDayOfWeek (input) {
		if (!this.isValid()) {
			return input != null ? this : NaN;
		}
		var day = this._isUTC ? this._d.getUTCDay() : this._d.getDay();
		if (input != null) {
			input = parseWeekday(input, this.localeData());
			return this.add(input - day, 'd');
		} else {
			return day;
		}
	}

	function getSetLocaleDayOfWeek (input) {
		if (!this.isValid()) {
			return input != null ? this : NaN;
		}
		var weekday = (this.day() + 7 - this.localeData()._week.dow) % 7;
		return input == null ? weekday : this.add(input - weekday, 'd');
	}

	function getSetISODayOfWeek (input) {
		if (!this.isValid()) {
			return input != null ? this : NaN;
		}

		// behaves the same as moment#day except
		// as a getter, returns 7 instead of 0 (1-7 range instead of 0-6)
		// as a setter, sunday should belong to the previous week.

		if (input != null) {
			var weekday = parseIsoWeekday(input, this.localeData());
			return this.day(this.day() % 7 ? weekday : weekday - 7);
		} else {
			return this.day() || 7;
		}
	}

	var defaultWeekdaysRegex = matchWord;
	function weekdaysRegex (isStrict) {
		if (this._weekdaysParseExact) {
			if (!hasOwnProp(this, '_weekdaysRegex')) {
				computeWeekdaysParse.call(this);
			}
			if (isStrict) {
				return this._weekdaysStrictRegex;
			} else {
				return this._weekdaysRegex;
			}
		} else {
			if (!hasOwnProp(this, '_weekdaysRegex')) {
				this._weekdaysRegex = defaultWeekdaysRegex;
			}
			return this._weekdaysStrictRegex && isStrict ?
				this._weekdaysStrictRegex : this._weekdaysRegex;
		}
	}

	var defaultWeekdaysShortRegex = matchWord;
	function weekdaysShortRegex (isStrict) {
		if (this._weekdaysParseExact) {
			if (!hasOwnProp(this, '_weekdaysRegex')) {
				computeWeekdaysParse.call(this);
			}
			if (isStrict) {
				return this._weekdaysShortStrictRegex;
			} else {
				return this._weekdaysShortRegex;
			}
		} else {
			if (!hasOwnProp(this, '_weekdaysShortRegex')) {
				this._weekdaysShortRegex = defaultWeekdaysShortRegex;
			}
			return this._weekdaysShortStrictRegex && isStrict ?
				this._weekdaysShortStrictRegex : this._weekdaysShortRegex;
		}
	}

	var defaultWeekdaysMinRegex = matchWord;
	function weekdaysMinRegex (isStrict) {
		if (this._weekdaysParseExact) {
			if (!hasOwnProp(this, '_weekdaysRegex')) {
				computeWeekdaysParse.call(this);
			}
			if (isStrict) {
				return this._weekdaysMinStrictRegex;
			} else {
				return this._weekdaysMinRegex;
			}
		} else {
			if (!hasOwnProp(this, '_weekdaysMinRegex')) {
				this._weekdaysMinRegex = defaultWeekdaysMinRegex;
			}
			return this._weekdaysMinStrictRegex && isStrict ?
				this._weekdaysMinStrictRegex : this._weekdaysMinRegex;
		}
	}


	function computeWeekdaysParse () {
		function cmpLenRev(a, b) {
			return b.length - a.length;
		}

		var minPieces = [], shortPieces = [], longPieces = [], mixedPieces = [],
			i, mom, minp, shortp, longp;
		for (i = 0; i < 7; i++) {
			// make the regex if we don't have it already
			mom = createUTC([2000, 1]).day(i);
			minp = this.weekdaysMin(mom, '');
			shortp = this.weekdaysShort(mom, '');
			longp = this.weekdays(mom, '');
			minPieces.push(minp);
			shortPieces.push(shortp);
			longPieces.push(longp);
			mixedPieces.push(minp);
			mixedPieces.push(shortp);
			mixedPieces.push(longp);
		}
		// Sorting makes sure if one weekday (or abbr) is a prefix of another it
		// will match the longer piece.
		minPieces.sort(cmpLenRev);
		shortPieces.sort(cmpLenRev);
		longPieces.sort(cmpLenRev);
		mixedPieces.sort(cmpLenRev);
		for (i = 0; i < 7; i++) {
			shortPieces[i] = regexEscape(shortPieces[i]);
			longPieces[i] = regexEscape(longPieces[i]);
			mixedPieces[i] = regexEscape(mixedPieces[i]);
		}

		this._weekdaysRegex = new RegExp('^(' + mixedPieces.join('|') + ')', 'i');
		this._weekdaysShortRegex = this._weekdaysRegex;
		this._weekdaysMinRegex = this._weekdaysRegex;

		this._weekdaysStrictRegex = new RegExp('^(' + longPieces.join('|') + ')', 'i');
		this._weekdaysShortStrictRegex = new RegExp('^(' + shortPieces.join('|') + ')', 'i');
		this._weekdaysMinStrictRegex = new RegExp('^(' + minPieces.join('|') + ')', 'i');
	}

// FORMATTING

	function hFormat() {
		return this.hours() % 12 || 12;
	}

	function kFormat() {
		return this.hours() || 24;
	}

	addFormatToken('H', ['HH', 2], 0, 'hour');
	addFormatToken('h', ['hh', 2], 0, hFormat);
	addFormatToken('k', ['kk', 2], 0, kFormat);

	addFormatToken('hmm', 0, 0, function () {
		return '' + hFormat.apply(this) + zeroFill(this.minutes(), 2);
	});

	addFormatToken('hmmss', 0, 0, function () {
		return '' + hFormat.apply(this) + zeroFill(this.minutes(), 2) +
			zeroFill(this.seconds(), 2);
	});

	addFormatToken('Hmm', 0, 0, function () {
		return '' + this.hours() + zeroFill(this.minutes(), 2);
	});

	addFormatToken('Hmmss', 0, 0, function () {
		return '' + this.hours() + zeroFill(this.minutes(), 2) +
			zeroFill(this.seconds(), 2);
	});

	function meridiem (token, lowercase) {
		addFormatToken(token, 0, 0, function () {
			return this.localeData().meridiem(this.hours(), this.minutes(), lowercase);
		});
	}

	meridiem('a', true);
	meridiem('A', false);

// ALIASES

	addUnitAlias('hour', 'h');

// PRIORITY
	addUnitPriority('hour', 13);

// PARSING

	function matchMeridiem (isStrict, locale) {
		return locale._meridiemParse;
	}

	addRegexToken('a',  matchMeridiem);
	addRegexToken('A',  matchMeridiem);
	addRegexToken('H',  match1to2);
	addRegexToken('h',  match1to2);
	addRegexToken('k',  match1to2);
	addRegexToken('HH', match1to2, match2);
	addRegexToken('hh', match1to2, match2);
	addRegexToken('kk', match1to2, match2);

	addRegexToken('hmm', match3to4);
	addRegexToken('hmmss', match5to6);
	addRegexToken('Hmm', match3to4);
	addRegexToken('Hmmss', match5to6);

	addParseToken(['H', 'HH'], HOUR);
	addParseToken(['k', 'kk'], function (input, array, config) {
		var kInput = toInt(input);
		array[HOUR] = kInput === 24 ? 0 : kInput;
	});
	addParseToken(['a', 'A'], function (input, array, config) {
		config._isPm = config._locale.isPM(input);
		config._meridiem = input;
	});
	addParseToken(['h', 'hh'], function (input, array, config) {
		array[HOUR] = toInt(input);
		getParsingFlags(config).bigHour = true;
	});
	addParseToken('hmm', function (input, array, config) {
		var pos = input.length - 2;
		array[HOUR] = toInt(input.substr(0, pos));
		array[MINUTE] = toInt(input.substr(pos));
		getParsingFlags(config).bigHour = true;
	});
	addParseToken('hmmss', function (input, array, config) {
		var pos1 = input.length - 4;
		var pos2 = input.length - 2;
		array[HOUR] = toInt(input.substr(0, pos1));
		array[MINUTE] = toInt(input.substr(pos1, 2));
		array[SECOND] = toInt(input.substr(pos2));
		getParsingFlags(config).bigHour = true;
	});
	addParseToken('Hmm', function (input, array, config) {
		var pos = input.length - 2;
		array[HOUR] = toInt(input.substr(0, pos));
		array[MINUTE] = toInt(input.substr(pos));
	});
	addParseToken('Hmmss', function (input, array, config) {
		var pos1 = input.length - 4;
		var pos2 = input.length - 2;
		array[HOUR] = toInt(input.substr(0, pos1));
		array[MINUTE] = toInt(input.substr(pos1, 2));
		array[SECOND] = toInt(input.substr(pos2));
	});

// LOCALES

	function localeIsPM (input) {
		// IE8 Quirks Mode & IE7 Standards Mode do not allow accessing strings like arrays
		// Using charAt should be more compatible.
		return ((input + '').toLowerCase().charAt(0) === 'p');
	}

	var defaultLocaleMeridiemParse = /[ap]\.?m?\.?/i;
	function localeMeridiem (hours, minutes, isLower) {
		if (hours > 11) {
			return isLower ? 'pm' : 'PM';
		} else {
			return isLower ? 'am' : 'AM';
		}
	}


// MOMENTS

// Setting the hour should keep the time, because the user explicitly
// specified which hour he wants. So trying to maintain the same hour (in
// a new timezone) makes sense. Adding/subtracting hours does not follow
// this rule.
	var getSetHour = makeGetSet('Hours', true);

// months
// week
// weekdays
// meridiem
	var baseConfig = {
		calendar: defaultCalendar,
		longDateFormat: defaultLongDateFormat,
		invalidDate: defaultInvalidDate,
		ordinal: defaultOrdinal,
		dayOfMonthOrdinalParse: defaultDayOfMonthOrdinalParse,
		relativeTime: defaultRelativeTime,

		months: defaultLocaleMonths,
		monthsShort: defaultLocaleMonthsShort,

		week: defaultLocaleWeek,

		weekdays: defaultLocaleWeekdays,
		weekdaysMin: defaultLocaleWeekdaysMin,
		weekdaysShort: defaultLocaleWeekdaysShort,

		meridiemParse: defaultLocaleMeridiemParse
	};

// internal storage for locale config files
	var locales = {};
	var localeFamilies = {};
	var globalLocale;

	function normalizeLocale(key) {
		return key ? key.toLowerCase().replace('_', '-') : key;
	}

// pick the locale from the array
// try ['en-au', 'en-gb'] as 'en-au', 'en-gb', 'en', as in move through the list trying each
// substring from most specific to least, but move to the next array item if it's a more specific variant than the current root
	function chooseLocale(names) {
		var i = 0, j, next, locale, split;

		while (i < names.length) {
			split = normalizeLocale(names[i]).split('-');
			j = split.length;
			next = normalizeLocale(names[i + 1]);
			next = next ? next.split('-') : null;
			while (j > 0) {
				locale = loadLocale(split.slice(0, j).join('-'));
				if (locale) {
					return locale;
				}
				if (next && next.length >= j && compareArrays(split, next, true) >= j - 1) {
					//the next array item is better than a shallower substring of this one
					break;
				}
				j--;
			}
			i++;
		}
		return null;
	}

	function loadLocale(name) {
		var oldLocale = null;
		// TODO: Find a better way to register and load all the locales in Node
		if (!locales[name] && (typeof module !== 'undefined') &&
			module && module.exports) {
			try {
				oldLocale = globalLocale._abbr;
				var aliasedRequire = require;
				aliasedRequire('./locale/' + name);
				getSetGlobalLocale(oldLocale);
			} catch (e) {}
		}
		return locales[name];
	}

// This function will load locale and then set the global locale.  If
// no arguments are passed in, it will simply return the current global
// locale key.
	function getSetGlobalLocale (key, values) {
		var data;
		if (key) {
			if (isUndefined(values)) {
				data = getLocale(key);
			}
			else {
				data = defineLocale(key, values);
			}

			if (data) {
				// moment.duration._locale = moment._locale = data;
				globalLocale = data;
			}
		}

		return globalLocale._abbr;
	}

	function defineLocale (name, config) {
		if (config !== null) {
			var parentConfig = baseConfig;
			config.abbr = name;
			if (locales[name] != null) {
				deprecateSimple('defineLocaleOverride',
					'use moment.updateLocale(localeName, config) to change ' +
					'an existing locale. moment.defineLocale(localeName, ' +
					'config) should only be used for creating a new locale ' +
					'See http://momentjs.com/guides/#/warnings/define-locale/ for more info.');
				parentConfig = locales[name]._config;
			} else if (config.parentLocale != null) {
				if (locales[config.parentLocale] != null) {
					parentConfig = locales[config.parentLocale]._config;
				} else {
					if (!localeFamilies[config.parentLocale]) {
						localeFamilies[config.parentLocale] = [];
					}
					localeFamilies[config.parentLocale].push({
						name: name,
						config: config
					});
					return null;
				}
			}
			locales[name] = new Locale(mergeConfigs(parentConfig, config));

			if (localeFamilies[name]) {
				localeFamilies[name].forEach(function (x) {
					defineLocale(x.name, x.config);
				});
			}

			// backwards compat for now: also set the locale
			// make sure we set the locale AFTER all child locales have been
			// created, so we won't end up with the child locale set.
			getSetGlobalLocale(name);


			return locales[name];
		} else {
			// useful for testing
			delete locales[name];
			return null;
		}
	}

	function updateLocale(name, config) {
		if (config != null) {
			var locale, tmpLocale, parentConfig = baseConfig;
			// MERGE
			tmpLocale = loadLocale(name);
			if (tmpLocale != null) {
				parentConfig = tmpLocale._config;
			}
			config = mergeConfigs(parentConfig, config);
			locale = new Locale(config);
			locale.parentLocale = locales[name];
			locales[name] = locale;

			// backwards compat for now: also set the locale
			getSetGlobalLocale(name);
		} else {
			// pass null for config to unupdate, useful for tests
			if (locales[name] != null) {
				if (locales[name].parentLocale != null) {
					locales[name] = locales[name].parentLocale;
				} else if (locales[name] != null) {
					delete locales[name];
				}
			}
		}
		return locales[name];
	}

// returns locale data
	function getLocale (key) {
		var locale;

		if (key && key._locale && key._locale._abbr) {
			key = key._locale._abbr;
		}

		if (!key) {
			return globalLocale;
		}

		if (!isArray(key)) {
			//short-circuit everything else
			locale = loadLocale(key);
			if (locale) {
				return locale;
			}
			key = [key];
		}

		return chooseLocale(key);
	}

	function listLocales() {
		return keys(locales);
	}

	function checkOverflow (m) {
		var overflow;
		var a = m._a;

		if (a && getParsingFlags(m).overflow === -2) {
			overflow =
				a[MONTH]       < 0 || a[MONTH]       > 11  ? MONTH :
					a[DATE]        < 1 || a[DATE]        > daysInMonth(a[YEAR], a[MONTH]) ? DATE :
						a[HOUR]        < 0 || a[HOUR]        > 24 || (a[HOUR] === 24 && (a[MINUTE] !== 0 || a[SECOND] !== 0 || a[MILLISECOND] !== 0)) ? HOUR :
							a[MINUTE]      < 0 || a[MINUTE]      > 59  ? MINUTE :
								a[SECOND]      < 0 || a[SECOND]      > 59  ? SECOND :
									a[MILLISECOND] < 0 || a[MILLISECOND] > 999 ? MILLISECOND :
										-1;

			if (getParsingFlags(m)._overflowDayOfYear && (overflow < YEAR || overflow > DATE)) {
				overflow = DATE;
			}
			if (getParsingFlags(m)._overflowWeeks && overflow === -1) {
				overflow = WEEK;
			}
			if (getParsingFlags(m)._overflowWeekday && overflow === -1) {
				overflow = WEEKDAY;
			}

			getParsingFlags(m).overflow = overflow;
		}

		return m;
	}

// Pick the first defined of two or three arguments.
	function defaults(a, b, c) {
		if (a != null) {
			return a;
		}
		if (b != null) {
			return b;
		}
		return c;
	}

	function currentDateArray(config) {
		// hooks is actually the exported moment object
		var nowValue = new Date(hooks.now());
		if (config._useUTC) {
			return [nowValue.getUTCFullYear(), nowValue.getUTCMonth(), nowValue.getUTCDate()];
		}
		return [nowValue.getFullYear(), nowValue.getMonth(), nowValue.getDate()];
	}

// convert an array to a date.
// the array should mirror the parameters below
// note: all values past the year are optional and will default to the lowest possible value.
// [year, month, day , hour, minute, second, millisecond]
	function configFromArray (config) {
		var i, date, input = [], currentDate, expectedWeekday, yearToUse;

		if (config._d) {
			return;
		}

		currentDate = currentDateArray(config);

		//compute day of the year from weeks and weekdays
		if (config._w && config._a[DATE] == null && config._a[MONTH] == null) {
			dayOfYearFromWeekInfo(config);
		}

		//if the day of the year is set, figure out what it is
		if (config._dayOfYear != null) {
			yearToUse = defaults(config._a[YEAR], currentDate[YEAR]);

			if (config._dayOfYear > daysInYear(yearToUse) || config._dayOfYear === 0) {
				getParsingFlags(config)._overflowDayOfYear = true;
			}

			date = createUTCDate(yearToUse, 0, config._dayOfYear);
			config._a[MONTH] = date.getUTCMonth();
			config._a[DATE] = date.getUTCDate();
		}

		// Default to current date.
		// * if no year, month, day of month are given, default to today
		// * if day of month is given, default month and year
		// * if month is given, default only year
		// * if year is given, don't default anything
		for (i = 0; i < 3 && config._a[i] == null; ++i) {
			config._a[i] = input[i] = currentDate[i];
		}

		// Zero out whatever was not defaulted, including time
		for (; i < 7; i++) {
			config._a[i] = input[i] = (config._a[i] == null) ? (i === 2 ? 1 : 0) : config._a[i];
		}

		// Check for 24:00:00.000
		if (config._a[HOUR] === 24 &&
			config._a[MINUTE] === 0 &&
			config._a[SECOND] === 0 &&
			config._a[MILLISECOND] === 0) {
			config._nextDay = true;
			config._a[HOUR] = 0;
		}

		config._d = (config._useUTC ? createUTCDate : createDate).apply(null, input);
		expectedWeekday = config._useUTC ? config._d.getUTCDay() : config._d.getDay();

		// Apply timezone offset from input. The actual utcOffset can be changed
		// with parseZone.
		if (config._tzm != null) {
			config._d.setUTCMinutes(config._d.getUTCMinutes() - config._tzm);
		}

		if (config._nextDay) {
			config._a[HOUR] = 24;
		}

		// check for mismatching day of week
		if (config._w && typeof config._w.d !== 'undefined' && config._w.d !== expectedWeekday) {
			getParsingFlags(config).weekdayMismatch = true;
		}
	}

	function dayOfYearFromWeekInfo(config) {
		var w, weekYear, week, weekday, dow, doy, temp, weekdayOverflow;

		w = config._w;
		if (w.GG != null || w.W != null || w.E != null) {
			dow = 1;
			doy = 4;

			// TODO: We need to take the current isoWeekYear, but that depends on
			// how we interpret now (local, utc, fixed offset). So create
			// a now version of current config (take local/utc/offset flags, and
			// create now).
			weekYear = defaults(w.GG, config._a[YEAR], weekOfYear(createLocal(), 1, 4).year);
			week = defaults(w.W, 1);
			weekday = defaults(w.E, 1);
			if (weekday < 1 || weekday > 7) {
				weekdayOverflow = true;
			}
		} else {
			dow = config._locale._week.dow;
			doy = config._locale._week.doy;

			var curWeek = weekOfYear(createLocal(), dow, doy);

			weekYear = defaults(w.gg, config._a[YEAR], curWeek.year);

			// Default to current week.
			week = defaults(w.w, curWeek.week);

			if (w.d != null) {
				// weekday -- low day numbers are considered next week
				weekday = w.d;
				if (weekday < 0 || weekday > 6) {
					weekdayOverflow = true;
				}
			} else if (w.e != null) {
				// local weekday -- counting starts from begining of week
				weekday = w.e + dow;
				if (w.e < 0 || w.e > 6) {
					weekdayOverflow = true;
				}
			} else {
				// default to begining of week
				weekday = dow;
			}
		}
		if (week < 1 || week > weeksInYear(weekYear, dow, doy)) {
			getParsingFlags(config)._overflowWeeks = true;
		} else if (weekdayOverflow != null) {
			getParsingFlags(config)._overflowWeekday = true;
		} else {
			temp = dayOfYearFromWeeks(weekYear, week, weekday, dow, doy);
			config._a[YEAR] = temp.year;
			config._dayOfYear = temp.dayOfYear;
		}
	}

// iso 8601 regex
// 0000-00-00 0000-W00 or 0000-W00-0 + T + 00 or 00:00 or 00:00:00 or 00:00:00.000 + +00:00 or +0000 or +00)
	var extendedIsoRegex = /^\s*((?:[+-]\d{6}|\d{4})-(?:\d\d-\d\d|W\d\d-\d|W\d\d|\d\d\d|\d\d))(?:(T| )(\d\d(?::\d\d(?::\d\d(?:[.,]\d+)?)?)?)([\+\-]\d\d(?::?\d\d)?|\s*Z)?)?$/;
	var basicIsoRegex = /^\s*((?:[+-]\d{6}|\d{4})(?:\d\d\d\d|W\d\d\d|W\d\d|\d\d\d|\d\d))(?:(T| )(\d\d(?:\d\d(?:\d\d(?:[.,]\d+)?)?)?)([\+\-]\d\d(?::?\d\d)?|\s*Z)?)?$/;

	var tzRegex = /Z|[+-]\d\d(?::?\d\d)?/;

	var isoDates = [
		['YYYYYY-MM-DD', /[+-]\d{6}-\d\d-\d\d/],
		['YYYY-MM-DD', /\d{4}-\d\d-\d\d/],
		['GGGG-[W]WW-E', /\d{4}-W\d\d-\d/],
		['GGGG-[W]WW', /\d{4}-W\d\d/, false],
		['YYYY-DDD', /\d{4}-\d{3}/],
		['YYYY-MM', /\d{4}-\d\d/, false],
		['YYYYYYMMDD', /[+-]\d{10}/],
		['YYYYMMDD', /\d{8}/],
		// YYYYMM is NOT allowed by the standard
		['GGGG[W]WWE', /\d{4}W\d{3}/],
		['GGGG[W]WW', /\d{4}W\d{2}/, false],
		['YYYYDDD', /\d{7}/]
	];

// iso time formats and regexes
	var isoTimes = [
		['HH:mm:ss.SSSS', /\d\d:\d\d:\d\d\.\d+/],
		['HH:mm:ss,SSSS', /\d\d:\d\d:\d\d,\d+/],
		['HH:mm:ss', /\d\d:\d\d:\d\d/],
		['HH:mm', /\d\d:\d\d/],
		['HHmmss.SSSS', /\d\d\d\d\d\d\.\d+/],
		['HHmmss,SSSS', /\d\d\d\d\d\d,\d+/],
		['HHmmss', /\d\d\d\d\d\d/],
		['HHmm', /\d\d\d\d/],
		['HH', /\d\d/]
	];

	var aspNetJsonRegex = /^\/?Date\((\-?\d+)/i;

// date from iso format
	function configFromISO(config) {
		var i, l,
			string = config._i,
			match = extendedIsoRegex.exec(string) || basicIsoRegex.exec(string),
			allowTime, dateFormat, timeFormat, tzFormat;

		if (match) {
			getParsingFlags(config).iso = true;

			for (i = 0, l = isoDates.length; i < l; i++) {
				if (isoDates[i][1].exec(match[1])) {
					dateFormat = isoDates[i][0];
					allowTime = isoDates[i][2] !== false;
					break;
				}
			}
			if (dateFormat == null) {
				config._isValid = false;
				return;
			}
			if (match[3]) {
				for (i = 0, l = isoTimes.length; i < l; i++) {
					if (isoTimes[i][1].exec(match[3])) {
						// match[2] should be 'T' or space
						timeFormat = (match[2] || ' ') + isoTimes[i][0];
						break;
					}
				}
				if (timeFormat == null) {
					config._isValid = false;
					return;
				}
			}
			if (!allowTime && timeFormat != null) {
				config._isValid = false;
				return;
			}
			if (match[4]) {
				if (tzRegex.exec(match[4])) {
					tzFormat = 'Z';
				} else {
					config._isValid = false;
					return;
				}
			}
			config._f = dateFormat + (timeFormat || '') + (tzFormat || '');
			configFromStringAndFormat(config);
		} else {
			config._isValid = false;
		}
	}

// RFC 2822 regex: For details see https://tools.ietf.org/html/rfc2822#section-3.3
	var rfc2822 = /^(?:(Mon|Tue|Wed|Thu|Fri|Sat|Sun),?\s)?(\d{1,2})\s(Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec)\s(\d{2,4})\s(\d\d):(\d\d)(?::(\d\d))?\s(?:(UT|GMT|[ECMP][SD]T)|([Zz])|([+-]\d{4}))$/;

	function extractFromRFC2822Strings(yearStr, monthStr, dayStr, hourStr, minuteStr, secondStr) {
		var result = [
			untruncateYear(yearStr),
			defaultLocaleMonthsShort.indexOf(monthStr),
			parseInt(dayStr, 10),
			parseInt(hourStr, 10),
			parseInt(minuteStr, 10)
		];

		if (secondStr) {
			result.push(parseInt(secondStr, 10));
		}

		return result;
	}

	function untruncateYear(yearStr) {
		var year = parseInt(yearStr, 10);
		if (year <= 49) {
			return 2000 + year;
		} else if (year <= 999) {
			return 1900 + year;
		}
		return year;
	}

	function preprocessRFC2822(s) {
		// Remove comments and folding whitespace and replace multiple-spaces with a single space
		return s.replace(/\([^)]*\)|[\n\t]/g, ' ').replace(/(\s\s+)/g, ' ').trim();
	}

	function checkWeekday(weekdayStr, parsedInput, config) {
		if (weekdayStr) {
			// TODO: Replace the vanilla JS Date object with an indepentent day-of-week check.
			var weekdayProvided = defaultLocaleWeekdaysShort.indexOf(weekdayStr),
				weekdayActual = new Date(parsedInput[0], parsedInput[1], parsedInput[2]).getDay();
			if (weekdayProvided !== weekdayActual) {
				getParsingFlags(config).weekdayMismatch = true;
				config._isValid = false;
				return false;
			}
		}
		return true;
	}

	var obsOffsets = {
		UT: 0,
		GMT: 0,
		EDT: -4 * 60,
		EST: -5 * 60,
		CDT: -5 * 60,
		CST: -6 * 60,
		MDT: -6 * 60,
		MST: -7 * 60,
		PDT: -7 * 60,
		PST: -8 * 60
	};

	function calculateOffset(obsOffset, militaryOffset, numOffset) {
		if (obsOffset) {
			return obsOffsets[obsOffset];
		} else if (militaryOffset) {
			// the only allowed military tz is Z
			return 0;
		} else {
			var hm = parseInt(numOffset, 10);
			var m = hm % 100, h = (hm - m) / 100;
			return h * 60 + m;
		}
	}

// date and time from ref 2822 format
	function configFromRFC2822(config) {
		var match = rfc2822.exec(preprocessRFC2822(config._i));
		if (match) {
			var parsedArray = extractFromRFC2822Strings(match[4], match[3], match[2], match[5], match[6], match[7]);
			if (!checkWeekday(match[1], parsedArray, config)) {
				return;
			}

			config._a = parsedArray;
			config._tzm = calculateOffset(match[8], match[9], match[10]);

			config._d = createUTCDate.apply(null, config._a);
			config._d.setUTCMinutes(config._d.getUTCMinutes() - config._tzm);

			getParsingFlags(config).rfc2822 = true;
		} else {
			config._isValid = false;
		}
	}

// date from iso format or fallback
	function configFromString(config) {
		var matched = aspNetJsonRegex.exec(config._i);

		if (matched !== null) {
			config._d = new Date(+matched[1]);
			return;
		}

		configFromISO(config);
		if (config._isValid === false) {
			delete config._isValid;
		} else {
			return;
		}

		configFromRFC2822(config);
		if (config._isValid === false) {
			delete config._isValid;
		} else {
			return;
		}

		// Final attempt, use Input Fallback
		hooks.createFromInputFallback(config);
	}

	hooks.createFromInputFallback = deprecate(
		'value provided is not in a recognized RFC2822 or ISO format. moment construction falls back to js Date(), ' +
		'which is not reliable across all browsers and versions. Non RFC2822/ISO date formats are ' +
		'discouraged and will be removed in an upcoming major release. Please refer to ' +
		'http://momentjs.com/guides/#/warnings/js-date/ for more info.',
		function (config) {
			config._d = new Date(config._i + (config._useUTC ? ' UTC' : ''));
		}
	);

// constant that refers to the ISO standard
	hooks.ISO_8601 = function () {};

// constant that refers to the RFC 2822 form
	hooks.RFC_2822 = function () {};

// date from string and format string
	function configFromStringAndFormat(config) {
		// TODO: Move this to another part of the creation flow to prevent circular deps
		if (config._f === hooks.ISO_8601) {
			configFromISO(config);
			return;
		}
		if (config._f === hooks.RFC_2822) {
			configFromRFC2822(config);
			return;
		}
		config._a = [];
		getParsingFlags(config).empty = true;

		// This array is used to make a Date, either with `new Date` or `Date.UTC`
		var string = '' + config._i,
			i, parsedInput, tokens, token, skipped,
			stringLength = string.length,
			totalParsedInputLength = 0;

		tokens = expandFormat(config._f, config._locale).match(formattingTokens) || [];

		for (i = 0; i < tokens.length; i++) {
			token = tokens[i];
			parsedInput = (string.match(getParseRegexForToken(token, config)) || [])[0];
			// console.log('token', token, 'parsedInput', parsedInput,
			//         'regex', getParseRegexForToken(token, config));
			if (parsedInput) {
				skipped = string.substr(0, string.indexOf(parsedInput));
				if (skipped.length > 0) {
					getParsingFlags(config).unusedInput.push(skipped);
				}
				string = string.slice(string.indexOf(parsedInput) + parsedInput.length);
				totalParsedInputLength += parsedInput.length;
			}
			// don't parse if it's not a known token
			if (formatTokenFunctions[token]) {
				if (parsedInput) {
					getParsingFlags(config).empty = false;
				}
				else {
					getParsingFlags(config).unusedTokens.push(token);
				}
				addTimeToArrayFromToken(token, parsedInput, config);
			}
			else if (config._strict && !parsedInput) {
				getParsingFlags(config).unusedTokens.push(token);
			}
		}

		// add remaining unparsed input length to the string
		getParsingFlags(config).charsLeftOver = stringLength - totalParsedInputLength;
		if (string.length > 0) {
			getParsingFlags(config).unusedInput.push(string);
		}

		// clear _12h flag if hour is <= 12
		if (config._a[HOUR] <= 12 &&
			getParsingFlags(config).bigHour === true &&
			config._a[HOUR] > 0) {
			getParsingFlags(config).bigHour = undefined;
		}

		getParsingFlags(config).parsedDateParts = config._a.slice(0);
		getParsingFlags(config).meridiem = config._meridiem;
		// handle meridiem
		config._a[HOUR] = meridiemFixWrap(config._locale, config._a[HOUR], config._meridiem);

		configFromArray(config);
		checkOverflow(config);
	}


	function meridiemFixWrap (locale, hour, meridiem) {
		var isPm;

		if (meridiem == null) {
			// nothing to do
			return hour;
		}
		if (locale.meridiemHour != null) {
			return locale.meridiemHour(hour, meridiem);
		} else if (locale.isPM != null) {
			// Fallback
			isPm = locale.isPM(meridiem);
			if (isPm && hour < 12) {
				hour += 12;
			}
			if (!isPm && hour === 12) {
				hour = 0;
			}
			return hour;
		} else {
			// this is not supposed to happen
			return hour;
		}
	}

// date from string and array of format strings
	function configFromStringAndArray(config) {
		var tempConfig,
			bestMoment,

			scoreToBeat,
			i,
			currentScore;

		if (config._f.length === 0) {
			getParsingFlags(config).invalidFormat = true;
			config._d = new Date(NaN);
			return;
		}

		for (i = 0; i < config._f.length; i++) {
			currentScore = 0;
			tempConfig = copyConfig({}, config);
			if (config._useUTC != null) {
				tempConfig._useUTC = config._useUTC;
			}
			tempConfig._f = config._f[i];
			configFromStringAndFormat(tempConfig);

			if (!isValid(tempConfig)) {
				continue;
			}

			// if there is any input that was not parsed add a penalty for that format
			currentScore += getParsingFlags(tempConfig).charsLeftOver;

			//or tokens
			currentScore += getParsingFlags(tempConfig).unusedTokens.length * 10;

			getParsingFlags(tempConfig).score = currentScore;

			if (scoreToBeat == null || currentScore < scoreToBeat) {
				scoreToBeat = currentScore;
				bestMoment = tempConfig;
			}
		}

		extend(config, bestMoment || tempConfig);
	}

	function configFromObject(config) {
		if (config._d) {
			return;
		}

		var i = normalizeObjectUnits(config._i);
		config._a = map([i.year, i.month, i.day || i.date, i.hour, i.minute, i.second, i.millisecond], function (obj) {
			return obj && parseInt(obj, 10);
		});

		configFromArray(config);
	}

	function createFromConfig (config) {
		var res = new Moment(checkOverflow(prepareConfig(config)));
		if (res._nextDay) {
			// Adding is smart enough around DST
			res.add(1, 'd');
			res._nextDay = undefined;
		}

		return res;
	}

	function prepareConfig (config) {
		var input = config._i,
			format = config._f;

		config._locale = config._locale || getLocale(config._l);

		if (input === null || (format === undefined && input === '')) {
			return createInvalid({nullInput: true});
		}

		if (typeof input === 'string') {
			config._i = input = config._locale.preparse(input);
		}

		if (isMoment(input)) {
			return new Moment(checkOverflow(input));
		} else if (isDate(input)) {
			config._d = input;
		} else if (isArray(format)) {
			configFromStringAndArray(config);
		} else if (format) {
			configFromStringAndFormat(config);
		}  else {
			configFromInput(config);
		}

		if (!isValid(config)) {
			config._d = null;
		}

		return config;
	}

	function configFromInput(config) {
		var input = config._i;
		if (isUndefined(input)) {
			config._d = new Date(hooks.now());
		} else if (isDate(input)) {
			config._d = new Date(input.valueOf());
		} else if (typeof input === 'string') {
			configFromString(config);
		} else if (isArray(input)) {
			config._a = map(input.slice(0), function (obj) {
				return parseInt(obj, 10);
			});
			configFromArray(config);
		} else if (isObject(input)) {
			configFromObject(config);
		} else if (isNumber(input)) {
			// from milliseconds
			config._d = new Date(input);
		} else {
			hooks.createFromInputFallback(config);
		}
	}

	function createLocalOrUTC (input, format, locale, strict, isUTC) {
		var c = {};

		if (locale === true || locale === false) {
			strict = locale;
			locale = undefined;
		}

		if ((isObject(input) && isObjectEmpty(input)) ||
			(isArray(input) && input.length === 0)) {
			input = undefined;
		}
		// object construction must be done this way.
		// https://github.com/moment/moment/issues/1423
		c._isAMomentObject = true;
		c._useUTC = c._isUTC = isUTC;
		c._l = locale;
		c._i = input;
		c._f = format;
		c._strict = strict;

		return createFromConfig(c);
	}

	function createLocal (input, format, locale, strict) {
		return createLocalOrUTC(input, format, locale, strict, false);
	}

	var prototypeMin = deprecate(
		'moment().min is deprecated, use moment.max instead. http://momentjs.com/guides/#/warnings/min-max/',
		function () {
			var other = createLocal.apply(null, arguments);
			if (this.isValid() && other.isValid()) {
				return other < this ? this : other;
			} else {
				return createInvalid();
			}
		}
	);

	var prototypeMax = deprecate(
		'moment().max is deprecated, use moment.min instead. http://momentjs.com/guides/#/warnings/min-max/',
		function () {
			var other = createLocal.apply(null, arguments);
			if (this.isValid() && other.isValid()) {
				return other > this ? this : other;
			} else {
				return createInvalid();
			}
		}
	);

// Pick a moment m from moments so that m[fn](other) is true for all
// other. This relies on the function fn to be transitive.
//
// moments should either be an array of moment objects or an array, whose
// first element is an array of moment objects.
	function pickBy(fn, moments) {
		var res, i;
		if (moments.length === 1 && isArray(moments[0])) {
			moments = moments[0];
		}
		if (!moments.length) {
			return createLocal();
		}
		res = moments[0];
		for (i = 1; i < moments.length; ++i) {
			if (!moments[i].isValid() || moments[i][fn](res)) {
				res = moments[i];
			}
		}
		return res;
	}

// TODO: Use [].sort instead?
	function min () {
		var args = [].slice.call(arguments, 0);

		return pickBy('isBefore', args);
	}

	function max () {
		var args = [].slice.call(arguments, 0);

		return pickBy('isAfter', args);
	}

	var now = function () {
		return Date.now ? Date.now() : +(new Date());
	};

	var ordering = ['year', 'quarter', 'month', 'week', 'day', 'hour', 'minute', 'second', 'millisecond'];

	function isDurationValid(m) {
		for (var key in m) {
			if (!(indexOf.call(ordering, key) !== -1 && (m[key] == null || !isNaN(m[key])))) {
				return false;
			}
		}

		var unitHasDecimal = false;
		for (var i = 0; i < ordering.length; ++i) {
			if (m[ordering[i]]) {
				if (unitHasDecimal) {
					return false; // only allow non-integers for smallest unit
				}
				if (parseFloat(m[ordering[i]]) !== toInt(m[ordering[i]])) {
					unitHasDecimal = true;
				}
			}
		}

		return true;
	}

	function isValid$1() {
		return this._isValid;
	}

	function createInvalid$1() {
		return createDuration(NaN);
	}

	function Duration (duration) {
		var normalizedInput = normalizeObjectUnits(duration),
			years = normalizedInput.year || 0,
			quarters = normalizedInput.quarter || 0,
			months = normalizedInput.month || 0,
			weeks = normalizedInput.week || 0,
			days = normalizedInput.day || 0,
			hours = normalizedInput.hour || 0,
			minutes = normalizedInput.minute || 0,
			seconds = normalizedInput.second || 0,
			milliseconds = normalizedInput.millisecond || 0;

		this._isValid = isDurationValid(normalizedInput);

		// representation for dateAddRemove
		this._milliseconds = +milliseconds +
			seconds * 1e3 + // 1000
			minutes * 6e4 + // 1000 * 60
			hours * 1000 * 60 * 60; //using 1000 * 60 * 60 instead of 36e5 to avoid floating point rounding errors https://github.com/moment/moment/issues/2978
		// Because of dateAddRemove treats 24 hours as different from a
		// day when working around DST, we need to store them separately
		this._days = +days +
			weeks * 7;
		// It is impossible to translate months into days without knowing
		// which months you are are talking about, so we have to store
		// it separately.
		this._months = +months +
			quarters * 3 +
			years * 12;

		this._data = {};

		this._locale = getLocale();

		this._bubble();
	}

	function isDuration (obj) {
		return obj instanceof Duration;
	}

	function absRound (number) {
		if (number < 0) {
			return Math.round(-1 * number) * -1;
		} else {
			return Math.round(number);
		}
	}

// FORMATTING

	function offset (token, separator) {
		addFormatToken(token, 0, 0, function () {
			var offset = this.utcOffset();
			var sign = '+';
			if (offset < 0) {
				offset = -offset;
				sign = '-';
			}
			return sign + zeroFill(~~(offset / 60), 2) + separator + zeroFill(~~(offset) % 60, 2);
		});
	}

	offset('Z', ':');
	offset('ZZ', '');

// PARSING

	addRegexToken('Z',  matchShortOffset);
	addRegexToken('ZZ', matchShortOffset);
	addParseToken(['Z', 'ZZ'], function (input, array, config) {
		config._useUTC = true;
		config._tzm = offsetFromString(matchShortOffset, input);
	});

// HELPERS

// timezone chunker
// '+10:00' > ['10',  '00']
// '-1530'  > ['-15', '30']
	var chunkOffset = /([\+\-]|\d\d)/gi;

	function offsetFromString(matcher, string) {
		var matches = (string || '').match(matcher);

		if (matches === null) {
			return null;
		}

		var chunk   = matches[matches.length - 1] || [];
		var parts   = (chunk + '').match(chunkOffset) || ['-', 0, 0];
		var minutes = +(parts[1] * 60) + toInt(parts[2]);

		return minutes === 0 ?
			0 :
			parts[0] === '+' ? minutes : -minutes;
	}

// Return a moment from input, that is local/utc/zone equivalent to model.
	function cloneWithOffset(input, model) {
		var res, diff;
		if (model._isUTC) {
			res = model.clone();
			diff = (isMoment(input) || isDate(input) ? input.valueOf() : createLocal(input).valueOf()) - res.valueOf();
			// Use low-level api, because this fn is low-level api.
			res._d.setTime(res._d.valueOf() + diff);
			hooks.updateOffset(res, false);
			return res;
		} else {
			return createLocal(input).local();
		}
	}

	function getDateOffset (m) {
		// On Firefox.24 Date#getTimezoneOffset returns a floating point.
		// https://github.com/moment/moment/pull/1871
		return -Math.round(m._d.getTimezoneOffset() / 15) * 15;
	}

// HOOKS

// This function will be called whenever a moment is mutated.
// It is intended to keep the offset in sync with the timezone.
	hooks.updateOffset = function () {};

// MOMENTS

// keepLocalTime = true means only change the timezone, without
// affecting the local hour. So 5:31:26 +0300 --[utcOffset(2, true)]-->
// 5:31:26 +0200 It is possible that 5:31:26 doesn't exist with offset
// +0200, so we adjust the time as needed, to be valid.
//
// Keeping the time actually adds/subtracts (one hour)
// from the actual represented time. That is why we call updateOffset
// a second time. In case it wants us to change the offset again
// _changeInProgress == true case, then we have to adjust, because
// there is no such time in the given timezone.
	function getSetOffset (input, keepLocalTime, keepMinutes) {
		var offset = this._offset || 0,
			localAdjust;
		if (!this.isValid()) {
			return input != null ? this : NaN;
		}
		if (input != null) {
			if (typeof input === 'string') {
				input = offsetFromString(matchShortOffset, input);
				if (input === null) {
					return this;
				}
			} else if (Math.abs(input) < 16 && !keepMinutes) {
				input = input * 60;
			}
			if (!this._isUTC && keepLocalTime) {
				localAdjust = getDateOffset(this);
			}
			this._offset = input;
			this._isUTC = true;
			if (localAdjust != null) {
				this.add(localAdjust, 'm');
			}
			if (offset !== input) {
				if (!keepLocalTime || this._changeInProgress) {
					addSubtract(this, createDuration(input - offset, 'm'), 1, false);
				} else if (!this._changeInProgress) {
					this._changeInProgress = true;
					hooks.updateOffset(this, true);
					this._changeInProgress = null;
				}
			}
			return this;
		} else {
			return this._isUTC ? offset : getDateOffset(this);
		}
	}

	function getSetZone (input, keepLocalTime) {
		if (input != null) {
			if (typeof input !== 'string') {
				input = -input;
			}

			this.utcOffset(input, keepLocalTime);

			return this;
		} else {
			return -this.utcOffset();
		}
	}

	function setOffsetToUTC (keepLocalTime) {
		return this.utcOffset(0, keepLocalTime);
	}

	function setOffsetToLocal (keepLocalTime) {
		if (this._isUTC) {
			this.utcOffset(0, keepLocalTime);
			this._isUTC = false;

			if (keepLocalTime) {
				this.subtract(getDateOffset(this), 'm');
			}
		}
		return this;
	}

	function setOffsetToParsedOffset () {
		if (this._tzm != null) {
			this.utcOffset(this._tzm, false, true);
		} else if (typeof this._i === 'string') {
			var tZone = offsetFromString(matchOffset, this._i);
			if (tZone != null) {
				this.utcOffset(tZone);
			}
			else {
				this.utcOffset(0, true);
			}
		}
		return this;
	}

	function hasAlignedHourOffset (input) {
		if (!this.isValid()) {
			return false;
		}
		input = input ? createLocal(input).utcOffset() : 0;

		return (this.utcOffset() - input) % 60 === 0;
	}

	function isDaylightSavingTime () {
		return (
			this.utcOffset() > this.clone().month(0).utcOffset() ||
			this.utcOffset() > this.clone().month(5).utcOffset()
		);
	}

	function isDaylightSavingTimeShifted () {
		if (!isUndefined(this._isDSTShifted)) {
			return this._isDSTShifted;
		}

		var c = {};

		copyConfig(c, this);
		c = prepareConfig(c);

		if (c._a) {
			var other = c._isUTC ? createUTC(c._a) : createLocal(c._a);
			this._isDSTShifted = this.isValid() &&
				compareArrays(c._a, other.toArray()) > 0;
		} else {
			this._isDSTShifted = false;
		}

		return this._isDSTShifted;
	}

	function isLocal () {
		return this.isValid() ? !this._isUTC : false;
	}

	function isUtcOffset () {
		return this.isValid() ? this._isUTC : false;
	}

	function isUtc () {
		return this.isValid() ? this._isUTC && this._offset === 0 : false;
	}

// ASP.NET json date format regex
	var aspNetRegex = /^(\-|\+)?(?:(\d*)[. ])?(\d+)\:(\d+)(?:\:(\d+)(\.\d*)?)?$/;

// from http://docs.closure-library.googlecode.com/git/closure_goog_date_date.js.source.html
// somewhat more in line with 4.4.3.2 2004 spec, but allows decimal anywhere
// and further modified to allow for strings containing both week and day
	var isoRegex = /^(-|\+)?P(?:([-+]?[0-9,.]*)Y)?(?:([-+]?[0-9,.]*)M)?(?:([-+]?[0-9,.]*)W)?(?:([-+]?[0-9,.]*)D)?(?:T(?:([-+]?[0-9,.]*)H)?(?:([-+]?[0-9,.]*)M)?(?:([-+]?[0-9,.]*)S)?)?$/;

	function createDuration (input, key) {
		var duration = input,
			// matching against regexp is expensive, do it on demand
			match = null,
			sign,
			ret,
			diffRes;

		if (isDuration(input)) {
			duration = {
				ms : input._milliseconds,
				d  : input._days,
				M  : input._months
			};
		} else if (isNumber(input)) {
			duration = {};
			if (key) {
				duration[key] = input;
			} else {
				duration.milliseconds = input;
			}
		} else if (!!(match = aspNetRegex.exec(input))) {
			sign = (match[1] === '-') ? -1 : 1;
			duration = {
				y  : 0,
				d  : toInt(match[DATE])                         * sign,
				h  : toInt(match[HOUR])                         * sign,
				m  : toInt(match[MINUTE])                       * sign,
				s  : toInt(match[SECOND])                       * sign,
				ms : toInt(absRound(match[MILLISECOND] * 1000)) * sign // the millisecond decimal point is included in the match
			};
		} else if (!!(match = isoRegex.exec(input))) {
			sign = (match[1] === '-') ? -1 : (match[1] === '+') ? 1 : 1;
			duration = {
				y : parseIso(match[2], sign),
				M : parseIso(match[3], sign),
				w : parseIso(match[4], sign),
				d : parseIso(match[5], sign),
				h : parseIso(match[6], sign),
				m : parseIso(match[7], sign),
				s : parseIso(match[8], sign)
			};
		} else if (duration == null) {// checks for null or undefined
			duration = {};
		} else if (typeof duration === 'object' && ('from' in duration || 'to' in duration)) {
			diffRes = momentsDifference(createLocal(duration.from), createLocal(duration.to));

			duration = {};
			duration.ms = diffRes.milliseconds;
			duration.M = diffRes.months;
		}

		ret = new Duration(duration);

		if (isDuration(input) && hasOwnProp(input, '_locale')) {
			ret._locale = input._locale;
		}

		return ret;
	}

	createDuration.fn = Duration.prototype;
	createDuration.invalid = createInvalid$1;

	function parseIso (inp, sign) {
		// We'd normally use ~~inp for this, but unfortunately it also
		// converts floats to ints.
		// inp may be undefined, so careful calling replace on it.
		var res = inp && parseFloat(inp.replace(',', '.'));
		// apply sign while we're at it
		return (isNaN(res) ? 0 : res) * sign;
	}

	function positiveMomentsDifference(base, other) {
		var res = {milliseconds: 0, months: 0};

		res.months = other.month() - base.month() +
			(other.year() - base.year()) * 12;
		if (base.clone().add(res.months, 'M').isAfter(other)) {
			--res.months;
		}

		res.milliseconds = +other - +(base.clone().add(res.months, 'M'));

		return res;
	}

	function momentsDifference(base, other) {
		var res;
		if (!(base.isValid() && other.isValid())) {
			return {milliseconds: 0, months: 0};
		}

		other = cloneWithOffset(other, base);
		if (base.isBefore(other)) {
			res = positiveMomentsDifference(base, other);
		} else {
			res = positiveMomentsDifference(other, base);
			res.milliseconds = -res.milliseconds;
			res.months = -res.months;
		}

		return res;
	}

// TODO: remove 'name' arg after deprecation is removed
	function createAdder(direction, name) {
		return function (val, period) {
			var dur, tmp;
			//invert the arguments, but complain about it
			if (period !== null && !isNaN(+period)) {
				deprecateSimple(name, 'moment().' + name  + '(period, number) is deprecated. Please use moment().' + name + '(number, period). ' +
					'See http://momentjs.com/guides/#/warnings/add-inverted-param/ for more info.');
				tmp = val; val = period; period = tmp;
			}

			val = typeof val === 'string' ? +val : val;
			dur = createDuration(val, period);
			addSubtract(this, dur, direction);
			return this;
		};
	}

	function addSubtract (mom, duration, isAdding, updateOffset) {
		var milliseconds = duration._milliseconds,
			days = absRound(duration._days),
			months = absRound(duration._months);

		if (!mom.isValid()) {
			// No op
			return;
		}

		updateOffset = updateOffset == null ? true : updateOffset;

		if (months) {
			setMonth(mom, get(mom, 'Month') + months * isAdding);
		}
		if (days) {
			set$1(mom, 'Date', get(mom, 'Date') + days * isAdding);
		}
		if (milliseconds) {
			mom._d.setTime(mom._d.valueOf() + milliseconds * isAdding);
		}
		if (updateOffset) {
			hooks.updateOffset(mom, days || months);
		}
	}

	var add      = createAdder(1, 'add');
	var subtract = createAdder(-1, 'subtract');

	function getCalendarFormat(myMoment, now) {
		var diff = myMoment.diff(now, 'days', true);
		return diff < -6 ? 'sameElse' :
			diff < -1 ? 'lastWeek' :
				diff < 0 ? 'lastDay' :
					diff < 1 ? 'sameDay' :
						diff < 2 ? 'nextDay' :
							diff < 7 ? 'nextWeek' : 'sameElse';
	}

	function calendar$1 (time, formats) {
		// We want to compare the start of today, vs this.
		// Getting start-of-today depends on whether we're local/utc/offset or not.
		var now = time || createLocal(),
			sod = cloneWithOffset(now, this).startOf('day'),
			format = hooks.calendarFormat(this, sod) || 'sameElse';

		var output = formats && (isFunction(formats[format]) ? formats[format].call(this, now) : formats[format]);

		return this.format(output || this.localeData().calendar(format, this, createLocal(now)));
	}

	function clone () {
		return new Moment(this);
	}

	function isAfter (input, units) {
		var localInput = isMoment(input) ? input : createLocal(input);
		if (!(this.isValid() && localInput.isValid())) {
			return false;
		}
		units = normalizeUnits(!isUndefined(units) ? units : 'millisecond');
		if (units === 'millisecond') {
			return this.valueOf() > localInput.valueOf();
		} else {
			return localInput.valueOf() < this.clone().startOf(units).valueOf();
		}
	}

	function isBefore (input, units) {
		var localInput = isMoment(input) ? input : createLocal(input);
		if (!(this.isValid() && localInput.isValid())) {
			return false;
		}
		units = normalizeUnits(!isUndefined(units) ? units : 'millisecond');
		if (units === 'millisecond') {
			return this.valueOf() < localInput.valueOf();
		} else {
			return this.clone().endOf(units).valueOf() < localInput.valueOf();
		}
	}

	function isBetween (from, to, units, inclusivity) {
		inclusivity = inclusivity || '()';
		return (inclusivity[0] === '(' ? this.isAfter(from, units) : !this.isBefore(from, units)) &&
			(inclusivity[1] === ')' ? this.isBefore(to, units) : !this.isAfter(to, units));
	}

	function isSame (input, units) {
		var localInput = isMoment(input) ? input : createLocal(input),
			inputMs;
		if (!(this.isValid() && localInput.isValid())) {
			return false;
		}
		units = normalizeUnits(units || 'millisecond');
		if (units === 'millisecond') {
			return this.valueOf() === localInput.valueOf();
		} else {
			inputMs = localInput.valueOf();
			return this.clone().startOf(units).valueOf() <= inputMs && inputMs <= this.clone().endOf(units).valueOf();
		}
	}

	function isSameOrAfter (input, units) {
		return this.isSame(input, units) || this.isAfter(input,units);
	}

	function isSameOrBefore (input, units) {
		return this.isSame(input, units) || this.isBefore(input,units);
	}

	function diff (input, units, asFloat) {
		var that,
			zoneDelta,
			delta, output;

		if (!this.isValid()) {
			return NaN;
		}

		that = cloneWithOffset(input, this);

		if (!that.isValid()) {
			return NaN;
		}

		zoneDelta = (that.utcOffset() - this.utcOffset()) * 6e4;

		units = normalizeUnits(units);

		switch (units) {
			case 'year': output = monthDiff(this, that) / 12; break;
			case 'month': output = monthDiff(this, that); break;
			case 'quarter': output = monthDiff(this, that) / 3; break;
			case 'second': output = (this - that) / 1e3; break; // 1000
			case 'minute': output = (this - that) / 6e4; break; // 1000 * 60
			case 'hour': output = (this - that) / 36e5; break; // 1000 * 60 * 60
			case 'day': output = (this - that - zoneDelta) / 864e5; break; // 1000 * 60 * 60 * 24, negate dst
			case 'week': output = (this - that - zoneDelta) / 6048e5; break; // 1000 * 60 * 60 * 24 * 7, negate dst
			default: output = this - that;
		}

		return asFloat ? output : absFloor(output);
	}

	function monthDiff (a, b) {
		// difference in months
		var wholeMonthDiff = ((b.year() - a.year()) * 12) + (b.month() - a.month()),
			// b is in (anchor - 1 month, anchor + 1 month)
			anchor = a.clone().add(wholeMonthDiff, 'months'),
			anchor2, adjust;

		if (b - anchor < 0) {
			anchor2 = a.clone().add(wholeMonthDiff - 1, 'months');
			// linear across the month
			adjust = (b - anchor) / (anchor - anchor2);
		} else {
			anchor2 = a.clone().add(wholeMonthDiff + 1, 'months');
			// linear across the month
			adjust = (b - anchor) / (anchor2 - anchor);
		}

		//check for negative zero, return zero if negative zero
		return -(wholeMonthDiff + adjust) || 0;
	}

	hooks.defaultFormat = 'YYYY-MM-DDTHH:mm:ssZ';
	hooks.defaultFormatUtc = 'YYYY-MM-DDTHH:mm:ss[Z]';

	function toString () {
		return this.clone().locale('en').format('ddd MMM DD YYYY HH:mm:ss [GMT]ZZ');
	}

	function toISOString(keepOffset) {
		if (!this.isValid()) {
			return null;
		}
		var utc = keepOffset !== true;
		var m = utc ? this.clone().utc() : this;
		if (m.year() < 0 || m.year() > 9999) {
			return formatMoment(m, utc ? 'YYYYYY-MM-DD[T]HH:mm:ss.SSS[Z]' : 'YYYYYY-MM-DD[T]HH:mm:ss.SSSZ');
		}
		if (isFunction(Date.prototype.toISOString)) {
			// native implementation is ~50x faster, use it when we can
			if (utc) {
				return this.toDate().toISOString();
			} else {
				return new Date(this._d.valueOf()).toISOString().replace('Z', formatMoment(m, 'Z'));
			}
		}
		return formatMoment(m, utc ? 'YYYY-MM-DD[T]HH:mm:ss.SSS[Z]' : 'YYYY-MM-DD[T]HH:mm:ss.SSSZ');
	}

	/**
	 * Return a human readable representation of a moment that can
	 * also be evaluated to get a new moment which is the same
	 *
	 * @link https://nodejs.org/dist/latest/docs/api/util.html#util_custom_inspect_function_on_objects
	 */
	function inspect () {
		if (!this.isValid()) {
			return 'moment.invalid(/* ' + this._i + ' */)';
		}
		var func = 'moment';
		var zone = '';
		if (!this.isLocal()) {
			func = this.utcOffset() === 0 ? 'moment.utc' : 'moment.parseZone';
			zone = 'Z';
		}
		var prefix = '[' + func + '("]';
		var year = (0 <= this.year() && this.year() <= 9999) ? 'YYYY' : 'YYYYYY';
		var datetime = '-MM-DD[T]HH:mm:ss.SSS';
		var suffix = zone + '[")]';

		return this.format(prefix + year + datetime + suffix);
	}

	function format (inputString) {
		if (!inputString) {
			inputString = this.isUtc() ? hooks.defaultFormatUtc : hooks.defaultFormat;
		}
		var output = formatMoment(this, inputString);
		return this.localeData().postformat(output);
	}

	function from (time, withoutSuffix) {
		if (this.isValid() &&
			((isMoment(time) && time.isValid()) ||
				createLocal(time).isValid())) {
			return createDuration({to: this, from: time}).locale(this.locale()).humanize(!withoutSuffix);
		} else {
			return this.localeData().invalidDate();
		}
	}

	function fromNow (withoutSuffix) {
		return this.from(createLocal(), withoutSuffix);
	}

	function to (time, withoutSuffix) {
		if (this.isValid() &&
			((isMoment(time) && time.isValid()) ||
				createLocal(time).isValid())) {
			return createDuration({from: this, to: time}).locale(this.locale()).humanize(!withoutSuffix);
		} else {
			return this.localeData().invalidDate();
		}
	}

	function toNow (withoutSuffix) {
		return this.to(createLocal(), withoutSuffix);
	}

// If passed a locale key, it will set the locale for this
// instance.  Otherwise, it will return the locale configuration
// variables for this instance.
	function locale (key) {
		var newLocaleData;

		if (key === undefined) {
			return this._locale._abbr;
		} else {
			newLocaleData = getLocale(key);
			if (newLocaleData != null) {
				this._locale = newLocaleData;
			}
			return this;
		}
	}

	var lang = deprecate(
		'moment().lang() is deprecated. Instead, use moment().localeData() to get the language configuration. Use moment().locale() to change languages.',
		function (key) {
			if (key === undefined) {
				return this.localeData();
			} else {
				return this.locale(key);
			}
		}
	);

	function localeData () {
		return this._locale;
	}

	function startOf (units) {
		units = normalizeUnits(units);
		// the following switch intentionally omits break keywords
		// to utilize falling through the cases.
		switch (units) {
			case 'year':
				this.month(0);
			/* falls through */
			case 'quarter':
			case 'month':
				this.date(1);
			/* falls through */
			case 'week':
			case 'isoWeek':
			case 'day':
			case 'date':
				this.hours(0);
			/* falls through */
			case 'hour':
				this.minutes(0);
			/* falls through */
			case 'minute':
				this.seconds(0);
			/* falls through */
			case 'second':
				this.milliseconds(0);
		}

		// weeks are a special case
		if (units === 'week') {
			this.weekday(0);
		}
		if (units === 'isoWeek') {
			this.isoWeekday(1);
		}

		// quarters are also special
		if (units === 'quarter') {
			this.month(Math.floor(this.month() / 3) * 3);
		}

		return this;
	}

	function endOf (units) {
		units = normalizeUnits(units);
		if (units === undefined || units === 'millisecond') {
			return this;
		}

		// 'date' is an alias for 'day', so it should be considered as such.
		if (units === 'date') {
			units = 'day';
		}

		return this.startOf(units).add(1, (units === 'isoWeek' ? 'week' : units)).subtract(1, 'ms');
	}

	function valueOf () {
		return this._d.valueOf() - ((this._offset || 0) * 60000);
	}

	function unix () {
		return Math.floor(this.valueOf() / 1000);
	}

	function toDate () {
		return new Date(this.valueOf());
	}

	function toArray () {
		var m = this;
		return [m.year(), m.month(), m.date(), m.hour(), m.minute(), m.second(), m.millisecond()];
	}

	function toObject () {
		var m = this;
		return {
			years: m.year(),
			months: m.month(),
			date: m.date(),
			hours: m.hours(),
			minutes: m.minutes(),
			seconds: m.seconds(),
			milliseconds: m.milliseconds()
		};
	}

	function toJSON () {
		// new Date(NaN).toJSON() === null
		return this.isValid() ? this.toISOString() : null;
	}

	function isValid$2 () {
		return isValid(this);
	}

	function parsingFlags () {
		return extend({}, getParsingFlags(this));
	}

	function invalidAt () {
		return getParsingFlags(this).overflow;
	}

	function creationData() {
		return {
			input: this._i,
			format: this._f,
			locale: this._locale,
			isUTC: this._isUTC,
			strict: this._strict
		};
	}

// FORMATTING

	addFormatToken(0, ['gg', 2], 0, function () {
		return this.weekYear() % 100;
	});

	addFormatToken(0, ['GG', 2], 0, function () {
		return this.isoWeekYear() % 100;
	});

	function addWeekYearFormatToken (token, getter) {
		addFormatToken(0, [token, token.length], 0, getter);
	}

	addWeekYearFormatToken('gggg',     'weekYear');
	addWeekYearFormatToken('ggggg',    'weekYear');
	addWeekYearFormatToken('GGGG',  'isoWeekYear');
	addWeekYearFormatToken('GGGGG', 'isoWeekYear');

// ALIASES

	addUnitAlias('weekYear', 'gg');
	addUnitAlias('isoWeekYear', 'GG');

// PRIORITY

	addUnitPriority('weekYear', 1);
	addUnitPriority('isoWeekYear', 1);


// PARSING

	addRegexToken('G',      matchSigned);
	addRegexToken('g',      matchSigned);
	addRegexToken('GG',     match1to2, match2);
	addRegexToken('gg',     match1to2, match2);
	addRegexToken('GGGG',   match1to4, match4);
	addRegexToken('gggg',   match1to4, match4);
	addRegexToken('GGGGG',  match1to6, match6);
	addRegexToken('ggggg',  match1to6, match6);

	addWeekParseToken(['gggg', 'ggggg', 'GGGG', 'GGGGG'], function (input, week, config, token) {
		week[token.substr(0, 2)] = toInt(input);
	});

	addWeekParseToken(['gg', 'GG'], function (input, week, config, token) {
		week[token] = hooks.parseTwoDigitYear(input);
	});

// MOMENTS

	function getSetWeekYear (input) {
		return getSetWeekYearHelper.call(this,
			input,
			this.week(),
			this.weekday(),
			this.localeData()._week.dow,
			this.localeData()._week.doy);
	}

	function getSetISOWeekYear (input) {
		return getSetWeekYearHelper.call(this,
			input, this.isoWeek(), this.isoWeekday(), 1, 4);
	}

	function getISOWeeksInYear () {
		return weeksInYear(this.year(), 1, 4);
	}

	function getWeeksInYear () {
		var weekInfo = this.localeData()._week;
		return weeksInYear(this.year(), weekInfo.dow, weekInfo.doy);
	}

	function getSetWeekYearHelper(input, week, weekday, dow, doy) {
		var weeksTarget;
		if (input == null) {
			return weekOfYear(this, dow, doy).year;
		} else {
			weeksTarget = weeksInYear(input, dow, doy);
			if (week > weeksTarget) {
				week = weeksTarget;
			}
			return setWeekAll.call(this, input, week, weekday, dow, doy);
		}
	}

	function setWeekAll(weekYear, week, weekday, dow, doy) {
		var dayOfYearData = dayOfYearFromWeeks(weekYear, week, weekday, dow, doy),
			date = createUTCDate(dayOfYearData.year, 0, dayOfYearData.dayOfYear);

		this.year(date.getUTCFullYear());
		this.month(date.getUTCMonth());
		this.date(date.getUTCDate());
		return this;
	}

// FORMATTING

	addFormatToken('Q', 0, 'Qo', 'quarter');

// ALIASES

	addUnitAlias('quarter', 'Q');

// PRIORITY

	addUnitPriority('quarter', 7);

// PARSING

	addRegexToken('Q', match1);
	addParseToken('Q', function (input, array) {
		array[MONTH] = (toInt(input) - 1) * 3;
	});

// MOMENTS

	function getSetQuarter (input) {
		return input == null ? Math.ceil((this.month() + 1) / 3) : this.month((input - 1) * 3 + this.month() % 3);
	}

// FORMATTING

	addFormatToken('D', ['DD', 2], 'Do', 'date');

// ALIASES

	addUnitAlias('date', 'D');

// PRIOROITY
	addUnitPriority('date', 9);

// PARSING

	addRegexToken('D',  match1to2);
	addRegexToken('DD', match1to2, match2);
	addRegexToken('Do', function (isStrict, locale) {
		// TODO: Remove "ordinalParse" fallback in next major release.
		return isStrict ?
			(locale._dayOfMonthOrdinalParse || locale._ordinalParse) :
			locale._dayOfMonthOrdinalParseLenient;
	});

	addParseToken(['D', 'DD'], DATE);
	addParseToken('Do', function (input, array) {
		array[DATE] = toInt(input.match(match1to2)[0]);
	});

// MOMENTS

	var getSetDayOfMonth = makeGetSet('Date', true);

// FORMATTING

	addFormatToken('DDD', ['DDDD', 3], 'DDDo', 'dayOfYear');

// ALIASES

	addUnitAlias('dayOfYear', 'DDD');

// PRIORITY
	addUnitPriority('dayOfYear', 4);

// PARSING

	addRegexToken('DDD',  match1to3);
	addRegexToken('DDDD', match3);
	addParseToken(['DDD', 'DDDD'], function (input, array, config) {
		config._dayOfYear = toInt(input);
	});

// HELPERS

// MOMENTS

	function getSetDayOfYear (input) {
		var dayOfYear = Math.round((this.clone().startOf('day') - this.clone().startOf('year')) / 864e5) + 1;
		return input == null ? dayOfYear : this.add((input - dayOfYear), 'd');
	}

// FORMATTING

	addFormatToken('m', ['mm', 2], 0, 'minute');

// ALIASES

	addUnitAlias('minute', 'm');

// PRIORITY

	addUnitPriority('minute', 14);

// PARSING

	addRegexToken('m',  match1to2);
	addRegexToken('mm', match1to2, match2);
	addParseToken(['m', 'mm'], MINUTE);

// MOMENTS

	var getSetMinute = makeGetSet('Minutes', false);

// FORMATTING

	addFormatToken('s', ['ss', 2], 0, 'second');

// ALIASES

	addUnitAlias('second', 's');

// PRIORITY

	addUnitPriority('second', 15);

// PARSING

	addRegexToken('s',  match1to2);
	addRegexToken('ss', match1to2, match2);
	addParseToken(['s', 'ss'], SECOND);

// MOMENTS

	var getSetSecond = makeGetSet('Seconds', false);

// FORMATTING

	addFormatToken('S', 0, 0, function () {
		return ~~(this.millisecond() / 100);
	});

	addFormatToken(0, ['SS', 2], 0, function () {
		return ~~(this.millisecond() / 10);
	});

	addFormatToken(0, ['SSS', 3], 0, 'millisecond');
	addFormatToken(0, ['SSSS', 4], 0, function () {
		return this.millisecond() * 10;
	});
	addFormatToken(0, ['SSSSS', 5], 0, function () {
		return this.millisecond() * 100;
	});
	addFormatToken(0, ['SSSSSS', 6], 0, function () {
		return this.millisecond() * 1000;
	});
	addFormatToken(0, ['SSSSSSS', 7], 0, function () {
		return this.millisecond() * 10000;
	});
	addFormatToken(0, ['SSSSSSSS', 8], 0, function () {
		return this.millisecond() * 100000;
	});
	addFormatToken(0, ['SSSSSSSSS', 9], 0, function () {
		return this.millisecond() * 1000000;
	});


// ALIASES

	addUnitAlias('millisecond', 'ms');

// PRIORITY

	addUnitPriority('millisecond', 16);

// PARSING

	addRegexToken('S',    match1to3, match1);
	addRegexToken('SS',   match1to3, match2);
	addRegexToken('SSS',  match1to3, match3);

	var token;
	for (token = 'SSSS'; token.length <= 9; token += 'S') {
		addRegexToken(token, matchUnsigned);
	}

	function parseMs(input, array) {
		array[MILLISECOND] = toInt(('0.' + input) * 1000);
	}

	for (token = 'S'; token.length <= 9; token += 'S') {
		addParseToken(token, parseMs);
	}
// MOMENTS

	var getSetMillisecond = makeGetSet('Milliseconds', false);

// FORMATTING

	addFormatToken('z',  0, 0, 'zoneAbbr');
	addFormatToken('zz', 0, 0, 'zoneName');

// MOMENTS

	function getZoneAbbr () {
		return this._isUTC ? 'UTC' : '';
	}

	function getZoneName () {
		return this._isUTC ? 'Coordinated Universal Time' : '';
	}

	var proto = Moment.prototype;

	proto.add               = add;
	proto.calendar          = calendar$1;
	proto.clone             = clone;
	proto.diff              = diff;
	proto.endOf             = endOf;
	proto.format            = format;
	proto.from              = from;
	proto.fromNow           = fromNow;
	proto.to                = to;
	proto.toNow             = toNow;
	proto.get               = stringGet;
	proto.invalidAt         = invalidAt;
	proto.isAfter           = isAfter;
	proto.isBefore          = isBefore;
	proto.isBetween         = isBetween;
	proto.isSame            = isSame;
	proto.isSameOrAfter     = isSameOrAfter;
	proto.isSameOrBefore    = isSameOrBefore;
	proto.isValid           = isValid$2;
	proto.lang              = lang;
	proto.locale            = locale;
	proto.localeData        = localeData;
	proto.max               = prototypeMax;
	proto.min               = prototypeMin;
	proto.parsingFlags      = parsingFlags;
	proto.set               = stringSet;
	proto.startOf           = startOf;
	proto.subtract          = subtract;
	proto.toArray           = toArray;
	proto.toObject          = toObject;
	proto.toDate            = toDate;
	proto.toISOString       = toISOString;
	proto.inspect           = inspect;
	proto.toJSON            = toJSON;
	proto.toString          = toString;
	proto.unix              = unix;
	proto.valueOf           = valueOf;
	proto.creationData      = creationData;

// Year
	proto.year       = getSetYear;
	proto.isLeapYear = getIsLeapYear;

// Week Year
	proto.weekYear    = getSetWeekYear;
	proto.isoWeekYear = getSetISOWeekYear;

// Quarter
	proto.quarter = proto.quarters = getSetQuarter;

// Month
	proto.month       = getSetMonth;
	proto.daysInMonth = getDaysInMonth;

// Week
	proto.week           = proto.weeks        = getSetWeek;
	proto.isoWeek        = proto.isoWeeks     = getSetISOWeek;
	proto.weeksInYear    = getWeeksInYear;
	proto.isoWeeksInYear = getISOWeeksInYear;

// Day
	proto.date       = getSetDayOfMonth;
	proto.day        = proto.days             = getSetDayOfWeek;
	proto.weekday    = getSetLocaleDayOfWeek;
	proto.isoWeekday = getSetISODayOfWeek;
	proto.dayOfYear  = getSetDayOfYear;

// Hour
	proto.hour = proto.hours = getSetHour;

// Minute
	proto.minute = proto.minutes = getSetMinute;

// Second
	proto.second = proto.seconds = getSetSecond;

// Millisecond
	proto.millisecond = proto.milliseconds = getSetMillisecond;

// Offset
	proto.utcOffset            = getSetOffset;
	proto.utc                  = setOffsetToUTC;
	proto.local                = setOffsetToLocal;
	proto.parseZone            = setOffsetToParsedOffset;
	proto.hasAlignedHourOffset = hasAlignedHourOffset;
	proto.isDST                = isDaylightSavingTime;
	proto.isLocal              = isLocal;
	proto.isUtcOffset          = isUtcOffset;
	proto.isUtc                = isUtc;
	proto.isUTC                = isUtc;

// Timezone
	proto.zoneAbbr = getZoneAbbr;
	proto.zoneName = getZoneName;

// Deprecations
	proto.dates  = deprecate('dates accessor is deprecated. Use date instead.', getSetDayOfMonth);
	proto.months = deprecate('months accessor is deprecated. Use month instead', getSetMonth);
	proto.years  = deprecate('years accessor is deprecated. Use year instead', getSetYear);
	proto.zone   = deprecate('moment().zone is deprecated, use moment().utcOffset instead. http://momentjs.com/guides/#/warnings/zone/', getSetZone);
	proto.isDSTShifted = deprecate('isDSTShifted is deprecated. See http://momentjs.com/guides/#/warnings/dst-shifted/ for more information', isDaylightSavingTimeShifted);

	function createUnix (input) {
		return createLocal(input * 1000);
	}

	function createInZone () {
		return createLocal.apply(null, arguments).parseZone();
	}

	function preParsePostFormat (string) {
		return string;
	}

	var proto$1 = Locale.prototype;

	proto$1.calendar        = calendar;
	proto$1.longDateFormat  = longDateFormat;
	proto$1.invalidDate     = invalidDate;
	proto$1.ordinal         = ordinal;
	proto$1.preparse        = preParsePostFormat;
	proto$1.postformat      = preParsePostFormat;
	proto$1.relativeTime    = relativeTime;
	proto$1.pastFuture      = pastFuture;
	proto$1.set             = set;

// Month
	proto$1.months            =        localeMonths;
	proto$1.monthsShort       =        localeMonthsShort;
	proto$1.monthsParse       =        localeMonthsParse;
	proto$1.monthsRegex       = monthsRegex;
	proto$1.monthsShortRegex  = monthsShortRegex;

// Week
	proto$1.week = localeWeek;
	proto$1.firstDayOfYear = localeFirstDayOfYear;
	proto$1.firstDayOfWeek = localeFirstDayOfWeek;

// Day of Week
	proto$1.weekdays       =        localeWeekdays;
	proto$1.weekdaysMin    =        localeWeekdaysMin;
	proto$1.weekdaysShort  =        localeWeekdaysShort;
	proto$1.weekdaysParse  =        localeWeekdaysParse;

	proto$1.weekdaysRegex       =        weekdaysRegex;
	proto$1.weekdaysShortRegex  =        weekdaysShortRegex;
	proto$1.weekdaysMinRegex    =        weekdaysMinRegex;

// Hours
	proto$1.isPM = localeIsPM;
	proto$1.meridiem = localeMeridiem;

	function get$1 (format, index, field, setter) {
		var locale = getLocale();
		var utc = createUTC().set(setter, index);
		return locale[field](utc, format);
	}

	function listMonthsImpl (format, index, field) {
		if (isNumber(format)) {
			index = format;
			format = undefined;
		}

		format = format || '';

		if (index != null) {
			return get$1(format, index, field, 'month');
		}

		var i;
		var out = [];
		for (i = 0; i < 12; i++) {
			out[i] = get$1(format, i, field, 'month');
		}
		return out;
	}

// ()
// (5)
// (fmt, 5)
// (fmt)
// (true)
// (true, 5)
// (true, fmt, 5)
// (true, fmt)
	function listWeekdaysImpl (localeSorted, format, index, field) {
		if (typeof localeSorted === 'boolean') {
			if (isNumber(format)) {
				index = format;
				format = undefined;
			}

			format = format || '';
		} else {
			format = localeSorted;
			index = format;
			localeSorted = false;

			if (isNumber(format)) {
				index = format;
				format = undefined;
			}

			format = format || '';
		}

		var locale = getLocale(),
			shift = localeSorted ? locale._week.dow : 0;

		if (index != null) {
			return get$1(format, (index + shift) % 7, field, 'day');
		}

		var i;
		var out = [];
		for (i = 0; i < 7; i++) {
			out[i] = get$1(format, (i + shift) % 7, field, 'day');
		}
		return out;
	}

	function listMonths (format, index) {
		return listMonthsImpl(format, index, 'months');
	}

	function listMonthsShort (format, index) {
		return listMonthsImpl(format, index, 'monthsShort');
	}

	function listWeekdays (localeSorted, format, index) {
		return listWeekdaysImpl(localeSorted, format, index, 'weekdays');
	}

	function listWeekdaysShort (localeSorted, format, index) {
		return listWeekdaysImpl(localeSorted, format, index, 'weekdaysShort');
	}

	function listWeekdaysMin (localeSorted, format, index) {
		return listWeekdaysImpl(localeSorted, format, index, 'weekdaysMin');
	}

	getSetGlobalLocale('en', {
		dayOfMonthOrdinalParse: /\d{1,2}(th|st|nd|rd)/,
		ordinal : function (number) {
			var b = number % 10,
				output = (toInt(number % 100 / 10) === 1) ? 'th' :
					(b === 1) ? 'st' :
						(b === 2) ? 'nd' :
							(b === 3) ? 'rd' : 'th';
			return number + output;
		}
	});

// Side effect imports
	hooks.lang = deprecate('moment.lang is deprecated. Use moment.locale instead.', getSetGlobalLocale);
	hooks.langData = deprecate('moment.langData is deprecated. Use moment.localeData instead.', getLocale);

	var mathAbs = Math.abs;

	function abs () {
		var data           = this._data;

		this._milliseconds = mathAbs(this._milliseconds);
		this._days         = mathAbs(this._days);
		this._months       = mathAbs(this._months);

		data.milliseconds  = mathAbs(data.milliseconds);
		data.seconds       = mathAbs(data.seconds);
		data.minutes       = mathAbs(data.minutes);
		data.hours         = mathAbs(data.hours);
		data.months        = mathAbs(data.months);
		data.years         = mathAbs(data.years);

		return this;
	}

	function addSubtract$1 (duration, input, value, direction) {
		var other = createDuration(input, value);

		duration._milliseconds += direction * other._milliseconds;
		duration._days         += direction * other._days;
		duration._months       += direction * other._months;

		return duration._bubble();
	}

// supports only 2.0-style add(1, 's') or add(duration)
	function add$1 (input, value) {
		return addSubtract$1(this, input, value, 1);
	}

// supports only 2.0-style subtract(1, 's') or subtract(duration)
	function subtract$1 (input, value) {
		return addSubtract$1(this, input, value, -1);
	}

	function absCeil (number) {
		if (number < 0) {
			return Math.floor(number);
		} else {
			return Math.ceil(number);
		}
	}

	function bubble () {
		var milliseconds = this._milliseconds;
		var days         = this._days;
		var months       = this._months;
		var data         = this._data;
		var seconds, minutes, hours, years, monthsFromDays;

		// if we have a mix of positive and negative values, bubble down first
		// check: https://github.com/moment/moment/issues/2166
		if (!((milliseconds >= 0 && days >= 0 && months >= 0) ||
			(milliseconds <= 0 && days <= 0 && months <= 0))) {
			milliseconds += absCeil(monthsToDays(months) + days) * 864e5;
			days = 0;
			months = 0;
		}

		// The following code bubbles up values, see the tests for
		// examples of what that means.
		data.milliseconds = milliseconds % 1000;

		seconds           = absFloor(milliseconds / 1000);
		data.seconds      = seconds % 60;

		minutes           = absFloor(seconds / 60);
		data.minutes      = minutes % 60;

		hours             = absFloor(minutes / 60);
		data.hours        = hours % 24;

		days += absFloor(hours / 24);

		// convert days to months
		monthsFromDays = absFloor(daysToMonths(days));
		months += monthsFromDays;
		days -= absCeil(monthsToDays(monthsFromDays));

		// 12 months -> 1 year
		years = absFloor(months / 12);
		months %= 12;

		data.days   = days;
		data.months = months;
		data.years  = years;

		return this;
	}

	function daysToMonths (days) {
		// 400 years have 146097 days (taking into account leap year rules)
		// 400 years have 12 months === 4800
		return days * 4800 / 146097;
	}

	function monthsToDays (months) {
		// the reverse of daysToMonths
		return months * 146097 / 4800;
	}

	function as (units) {
		if (!this.isValid()) {
			return NaN;
		}
		var days;
		var months;
		var milliseconds = this._milliseconds;

		units = normalizeUnits(units);

		if (units === 'month' || units === 'year') {
			days   = this._days   + milliseconds / 864e5;
			months = this._months + daysToMonths(days);
			return units === 'month' ? months : months / 12;
		} else {
			// handle milliseconds separately because of floating point math errors (issue #1867)
			days = this._days + Math.round(monthsToDays(this._months));
			switch (units) {
				case 'week'   : return days / 7     + milliseconds / 6048e5;
				case 'day'    : return days         + milliseconds / 864e5;
				case 'hour'   : return days * 24    + milliseconds / 36e5;
				case 'minute' : return days * 1440  + milliseconds / 6e4;
				case 'second' : return days * 86400 + milliseconds / 1000;
				// Math.floor prevents floating point math errors here
				case 'millisecond': return Math.floor(days * 864e5) + milliseconds;
				default: throw new Error('Unknown unit ' + units);
			}
		}
	}

// TODO: Use this.as('ms')?
	function valueOf$1 () {
		if (!this.isValid()) {
			return NaN;
		}
		return (
			this._milliseconds +
			this._days * 864e5 +
			(this._months % 12) * 2592e6 +
			toInt(this._months / 12) * 31536e6
		);
	}

	function makeAs (alias) {
		return function () {
			return this.as(alias);
		};
	}

	var asMilliseconds = makeAs('ms');
	var asSeconds      = makeAs('s');
	var asMinutes      = makeAs('m');
	var asHours        = makeAs('h');
	var asDays         = makeAs('d');
	var asWeeks        = makeAs('w');
	var asMonths       = makeAs('M');
	var asYears        = makeAs('y');

	function clone$1 () {
		return createDuration(this);
	}

	function get$2 (units) {
		units = normalizeUnits(units);
		return this.isValid() ? this[units + 's']() : NaN;
	}

	function makeGetter(name) {
		return function () {
			return this.isValid() ? this._data[name] : NaN;
		};
	}

	var milliseconds = makeGetter('milliseconds');
	var seconds      = makeGetter('seconds');
	var minutes      = makeGetter('minutes');
	var hours        = makeGetter('hours');
	var days         = makeGetter('days');
	var months       = makeGetter('months');
	var years        = makeGetter('years');

	function weeks () {
		return absFloor(this.days() / 7);
	}

	var round = Math.round;
	var thresholds = {
		ss: 44,         // a few seconds to seconds
		s : 45,         // seconds to minute
		m : 45,         // minutes to hour
		h : 22,         // hours to day
		d : 26,         // days to month
		M : 11          // months to year
	};

// helper function for moment.fn.from, moment.fn.fromNow, and moment.duration.fn.humanize
	function substituteTimeAgo(string, number, withoutSuffix, isFuture, locale) {
		return locale.relativeTime(number || 1, !!withoutSuffix, string, isFuture);
	}

	function relativeTime$1 (posNegDuration, withoutSuffix, locale) {
		var duration = createDuration(posNegDuration).abs();
		var seconds  = round(duration.as('s'));
		var minutes  = round(duration.as('m'));
		var hours    = round(duration.as('h'));
		var days     = round(duration.as('d'));
		var months   = round(duration.as('M'));
		var years    = round(duration.as('y'));

		var a = seconds <= thresholds.ss && ['s', seconds]  ||
			seconds < thresholds.s   && ['ss', seconds] ||
			minutes <= 1             && ['m']           ||
			minutes < thresholds.m   && ['mm', minutes] ||
			hours   <= 1             && ['h']           ||
			hours   < thresholds.h   && ['hh', hours]   ||
			days    <= 1             && ['d']           ||
			days    < thresholds.d   && ['dd', days]    ||
			months  <= 1             && ['M']           ||
			months  < thresholds.M   && ['MM', months]  ||
			years   <= 1             && ['y']           || ['yy', years];

		a[2] = withoutSuffix;
		a[3] = +posNegDuration > 0;
		a[4] = locale;
		return substituteTimeAgo.apply(null, a);
	}

// This function allows you to set the rounding function for relative time strings
	function getSetRelativeTimeRounding (roundingFunction) {
		if (roundingFunction === undefined) {
			return round;
		}
		if (typeof(roundingFunction) === 'function') {
			round = roundingFunction;
			return true;
		}
		return false;
	}

// This function allows you to set a threshold for relative time strings
	function getSetRelativeTimeThreshold (threshold, limit) {
		if (thresholds[threshold] === undefined) {
			return false;
		}
		if (limit === undefined) {
			return thresholds[threshold];
		}
		thresholds[threshold] = limit;
		if (threshold === 's') {
			thresholds.ss = limit - 1;
		}
		return true;
	}

	function humanize (withSuffix) {
		if (!this.isValid()) {
			return this.localeData().invalidDate();
		}

		var locale = this.localeData();
		var output = relativeTime$1(this, !withSuffix, locale);

		if (withSuffix) {
			output = locale.pastFuture(+this, output);
		}

		return locale.postformat(output);
	}

	var abs$1 = Math.abs;

	function sign(x) {
		return ((x > 0) - (x < 0)) || +x;
	}

	function toISOString$1() {
		// for ISO strings we do not use the normal bubbling rules:
		//  * milliseconds bubble up until they become hours
		//  * days do not bubble at all
		//  * months bubble up until they become years
		// This is because there is no context-free conversion between hours and days
		// (think of clock changes)
		// and also not between days and months (28-31 days per month)
		if (!this.isValid()) {
			return this.localeData().invalidDate();
		}

		var seconds = abs$1(this._milliseconds) / 1000;
		var days         = abs$1(this._days);
		var months       = abs$1(this._months);
		var minutes, hours, years;

		// 3600 seconds -> 60 minutes -> 1 hour
		minutes           = absFloor(seconds / 60);
		hours             = absFloor(minutes / 60);
		seconds %= 60;
		minutes %= 60;

		// 12 months -> 1 year
		years  = absFloor(months / 12);
		months %= 12;


		// inspired by https://github.com/dordille/moment-isoduration/blob/master/moment.isoduration.js
		var Y = years;
		var M = months;
		var D = days;
		var h = hours;
		var m = minutes;
		var s = seconds ? seconds.toFixed(3).replace(/\.?0+$/, '') : '';
		var total = this.asSeconds();

		if (!total) {
			// this is the same as C#'s (Noda) and python (isodate)...
			// but not other JS (goog.date)
			return 'P0D';
		}

		var totalSign = total < 0 ? '-' : '';
		var ymSign = sign(this._months) !== sign(total) ? '-' : '';
		var daysSign = sign(this._days) !== sign(total) ? '-' : '';
		var hmsSign = sign(this._milliseconds) !== sign(total) ? '-' : '';

		return totalSign + 'P' +
			(Y ? ymSign + Y + 'Y' : '') +
			(M ? ymSign + M + 'M' : '') +
			(D ? daysSign + D + 'D' : '') +
			((h || m || s) ? 'T' : '') +
			(h ? hmsSign + h + 'H' : '') +
			(m ? hmsSign + m + 'M' : '') +
			(s ? hmsSign + s + 'S' : '');
	}

	var proto$2 = Duration.prototype;

	proto$2.isValid        = isValid$1;
	proto$2.abs            = abs;
	proto$2.add            = add$1;
	proto$2.subtract       = subtract$1;
	proto$2.as             = as;
	proto$2.asMilliseconds = asMilliseconds;
	proto$2.asSeconds      = asSeconds;
	proto$2.asMinutes      = asMinutes;
	proto$2.asHours        = asHours;
	proto$2.asDays         = asDays;
	proto$2.asWeeks        = asWeeks;
	proto$2.asMonths       = asMonths;
	proto$2.asYears        = asYears;
	proto$2.valueOf        = valueOf$1;
	proto$2._bubble        = bubble;
	proto$2.clone          = clone$1;
	proto$2.get            = get$2;
	proto$2.milliseconds   = milliseconds;
	proto$2.seconds        = seconds;
	proto$2.minutes        = minutes;
	proto$2.hours          = hours;
	proto$2.days           = days;
	proto$2.weeks          = weeks;
	proto$2.months         = months;
	proto$2.years          = years;
	proto$2.humanize       = humanize;
	proto$2.toISOString    = toISOString$1;
	proto$2.toString       = toISOString$1;
	proto$2.toJSON         = toISOString$1;
	proto$2.locale         = locale;
	proto$2.localeData     = localeData;

// Deprecations
	proto$2.toIsoString = deprecate('toIsoString() is deprecated. Please use toISOString() instead (notice the capitals)', toISOString$1);
	proto$2.lang = lang;

// Side effect imports

// FORMATTING

	addFormatToken('X', 0, 0, 'unix');
	addFormatToken('x', 0, 0, 'valueOf');

// PARSING

	addRegexToken('x', matchSigned);
	addRegexToken('X', matchTimestamp);
	addParseToken('X', function (input, array, config) {
		config._d = new Date(parseFloat(input, 10) * 1000);
	});
	addParseToken('x', function (input, array, config) {
		config._d = new Date(toInt(input));
	});

// Side effect imports


	hooks.version = '2.20.1';

	setHookCallback(createLocal);

	hooks.fn                    = proto;
	hooks.min                   = min;
	hooks.max                   = max;
	hooks.now                   = now;
	hooks.utc                   = createUTC;
	hooks.unix                  = createUnix;
	hooks.months                = listMonths;
	hooks.isDate                = isDate;
	hooks.locale                = getSetGlobalLocale;
	hooks.invalid               = createInvalid;
	hooks.duration              = createDuration;
	hooks.isMoment              = isMoment;
	hooks.weekdays              = listWeekdays;
	hooks.parseZone             = createInZone;
	hooks.localeData            = getLocale;
	hooks.isDuration            = isDuration;
	hooks.monthsShort           = listMonthsShort;
	hooks.weekdaysMin           = listWeekdaysMin;
	hooks.defineLocale          = defineLocale;
	hooks.updateLocale          = updateLocale;
	hooks.locales               = listLocales;
	hooks.weekdaysShort         = listWeekdaysShort;
	hooks.normalizeUnits        = normalizeUnits;
	hooks.relativeTimeRounding  = getSetRelativeTimeRounding;
	hooks.relativeTimeThreshold = getSetRelativeTimeThreshold;
	hooks.calendarFormat        = getCalendarFormat;
	hooks.prototype             = proto;

// currently HTML5 input type only supports 24-hour formats
	hooks.HTML5_FMT = {
		DATETIME_LOCAL: 'YYYY-MM-DDTHH:mm',             // <input type="datetime-local" />
		DATETIME_LOCAL_SECONDS: 'YYYY-MM-DDTHH:mm:ss',  // <input type="datetime-local" step="1" />
		DATETIME_LOCAL_MS: 'YYYY-MM-DDTHH:mm:ss.SSS',   // <input type="datetime-local" step="0.001" />
		DATE: 'YYYY-MM-DD',                             // <input type="date" />
		TIME: 'HH:mm',                                  // <input type="time" />
		TIME_SECONDS: 'HH:mm:ss',                       // <input type="time" step="1" />
		TIME_MS: 'HH:mm:ss.SSS',                        // <input type="time" step="0.001" />
		WEEK: 'YYYY-[W]WW',                             // <input type="week" />
		MONTH: 'YYYY-MM'                                // <input type="month" />
	};

	return hooks;

})));

/*!
FullCalendar Scheduler v1.5.0
Docs & License: http://fullcalendar.io/scheduler/
(c) 2016 Adam Shaw
 */
(function(factory) {
	if (typeof define === 'function' && define.amd) {
		define([ 'jquery', 'moment', 'fullcalendar' ], factory);
	}
	else if (typeof exports === 'object') { // Node/CommonJS
		module.exports = factory(
			require('jquery'),
			require('moment'),
			require('fullcalendar')
		);
	}
	else {
		factory(jQuery, moment);
	}
})(function($, moment) {;
	var COL_MIN_WIDTH, Calendar, CalendarExtension, Class, ClippedScroller, CoordCache, DEFAULT_GRID_DURATION, DragListener, EmitterMixin, EnhancedScroller, EventRow, FC, Grid, HRowGroup, LICENSE_INFO_URL, ListenerMixin, MAX_AUTO_CELLS, MAX_AUTO_SLOTS_PER_LABEL, MAX_CELLS, MIN_AUTO_LABELS, PRESET_LICENSE_KEYS, Promise, RELEASE_DATE, ResourceAgendaView, ResourceBasicView, ResourceDayGrid, ResourceDayTableMixin, ResourceGridMixin, ResourceManager, ResourceMonthView, ResourceRow, ResourceTimeGrid, ResourceTimelineGrid, ResourceTimelineView, ResourceViewMixin, RowGroup, RowParent, STOCK_SUB_DURATIONS, ScrollFollower, ScrollFollowerSprite, ScrollJoiner, ScrollerCanvas, Spreadsheet, TaskQueue, TimelineGrid, TimelineView, UPGRADE_WINDOW, VRowGroup, VertResourceViewMixin, View, _filterResourcesWithEvents, applyAll, capitaliseFirstLetter, compareByFieldSpecs, computeIntervalUnit, computeOffsetForSeg, computeOffsetForSegs, copyRect, createObject, cssToStr, debounce, detectWarningInContainer, divideDurationByDuration, divideRangeByDuration, durationHasTime, flexibleCompare, getContentRect, getOuterRect, getOwnCells, getRectHeight, getRectWidth, getScrollbarWidths, hContainRect, htmlEscape, intersectRanges, intersectRects, isImmuneUrl, isInt, isValidKey, joinRects, multiplyDuration, origExecuteEventsRender, origGetSegCustomClasses, origGetSegDefaultBackgroundColor, origGetSegDefaultBorderColor, origGetSegDefaultTextColor, origHandleDate, origOnDateRender, origRemoveElement, origSetElement, parseFieldSpecs, processLicenseKey, proxy, renderingWarningInContainer, testRectContains, testRectHContains, testRectVContains, timeRowSegsCollide, vContainRect,
		extend = function(child, parent) { for (var key in parent) { if (hasProp.call(parent, key)) child[key] = parent[key]; } function ctor() { this.constructor = child; } ctor.prototype = parent.prototype; child.prototype = new ctor(); child.__super__ = parent.prototype; return child; },
		hasProp = {}.hasOwnProperty,
		indexOf = [].indexOf || function(item) { for (var i = 0, l = this.length; i < l; i++) { if (i in this && this[i] === item) return i; } return -1; },
		slice = [].slice;

	FC = $.fullCalendar;

	FC.schedulerVersion = "1.5.0";

	if (FC.internalApiVersion !== 7) {
		FC.warn('v' + FC.schedulerVersion + ' of FullCalendar Scheduler ' + 'is incompatible with v' + FC.version + ' of the core.\n' + 'Please see http://fullcalendar.io/support/ for more information.');
		return;
	}

	Calendar = FC.Calendar;

	Class = FC.Class;

	View = FC.View;

	Grid = FC.Grid;

	intersectRanges = FC.intersectRanges;

	debounce = FC.debounce;

	isInt = FC.isInt;

	getScrollbarWidths = FC.getScrollbarWidths;

	DragListener = FC.DragListener;

	htmlEscape = FC.htmlEscape;

	computeIntervalUnit = FC.computeIntervalUnit;

	proxy = FC.proxy;

	capitaliseFirstLetter = FC.capitaliseFirstLetter;

	applyAll = FC.applyAll;

	EmitterMixin = FC.EmitterMixin;

	ListenerMixin = FC.ListenerMixin;

	durationHasTime = FC.durationHasTime;

	divideRangeByDuration = FC.divideRangeByDuration;

	divideDurationByDuration = FC.divideDurationByDuration;

	multiplyDuration = FC.multiplyDuration;

	parseFieldSpecs = FC.parseFieldSpecs;

	compareByFieldSpecs = FC.compareByFieldSpecs;

	flexibleCompare = FC.flexibleCompare;

	intersectRects = FC.intersectRects;

	CoordCache = FC.CoordCache;

	getContentRect = FC.getContentRect;

	getOuterRect = FC.getOuterRect;

	createObject = FC.createObject;

	Promise = FC.Promise;

	TaskQueue = FC.TaskQueue;


	/*
Given a jQuery <tr> set, returns the <td>'s that do not have multi-line rowspans.
Would use the [rowspan] selector, but never not defined in IE8.
 */

	getOwnCells = function(trs) {
		return trs.find('> td').filter(function(i, tdNode) {
			return tdNode.rowSpan <= 1;
		});
	};


	/*
A Scroller with additional functionality:
- optional ScrollerCanvas for content
- fired events for scroll start/end
- cross-browser normalization of horizontal scroll for RTL
 */

	EnhancedScroller = (function(superClass) {
		var detectRtlScrollSystem, rtlScrollSystem;

		extend(EnhancedScroller, superClass);

		EnhancedScroller.mixin(EmitterMixin);

		EnhancedScroller.mixin(ListenerMixin);

		EnhancedScroller.prototype.canvas = null;

		EnhancedScroller.prototype.isScrolling = false;

		EnhancedScroller.prototype.isTouching = false;

		EnhancedScroller.prototype.isMoving = false;

		EnhancedScroller.prototype.isTouchScrollEnabled = true;

		EnhancedScroller.prototype.preventTouchScrollHandler = null;

		function EnhancedScroller() {
			EnhancedScroller.__super__.constructor.apply(this, arguments);
			this.requestMovingEnd = debounce(this.reportMovingEnd, 500);
		}

		EnhancedScroller.prototype.render = function() {
			EnhancedScroller.__super__.render.apply(this, arguments);
			if (this.canvas) {
				this.canvas.render();
				this.canvas.el.appendTo(this.scrollEl);
			}
			return this.bindHandlers();
		};

		EnhancedScroller.prototype.destroy = function() {
			EnhancedScroller.__super__.destroy.apply(this, arguments);
			return this.unbindHandlers();
		};

		EnhancedScroller.prototype.disableTouchScroll = function() {
			this.isTouchScrollEnabled = false;
			return this.bindPreventTouchScroll();
		};

		EnhancedScroller.prototype.enableTouchScroll = function() {
			this.isTouchScrollEnabled = true;
			if (!this.isTouching) {
				return this.unbindPreventTouchScroll();
			}
		};

		EnhancedScroller.prototype.bindPreventTouchScroll = function() {
			if (!this.preventTouchScrollHandler) {
				return this.scrollEl.on('touchmove', this.preventTouchScrollHandler = FC.preventDefault);
			}
		};

		EnhancedScroller.prototype.unbindPreventTouchScroll = function() {
			if (this.preventTouchScrollHandler) {
				this.scrollEl.off('touchmove', this.preventTouchScrollHandler);
				return this.preventTouchScrollHandler = null;
			}
		};

		EnhancedScroller.prototype.bindHandlers = function() {
			return this.listenTo(this.scrollEl, {
				scroll: this.reportScroll,
				touchstart: this.reportTouchStart,
				touchend: this.reportTouchEnd
			});
		};

		EnhancedScroller.prototype.unbindHandlers = function() {
			return this.stopListeningTo(this.scrollEl);
		};

		EnhancedScroller.prototype.reportScroll = function() {
			if (!this.isScrolling) {
				this.reportScrollStart();
			}
			this.trigger('scroll');
			this.isMoving = true;
			return this.requestMovingEnd();
		};

		EnhancedScroller.prototype.reportScrollStart = function() {
			if (!this.isScrolling) {
				this.isScrolling = true;
				return this.trigger('scrollStart', this.isTouching);
			}
		};

		EnhancedScroller.prototype.requestMovingEnd = null;

		EnhancedScroller.prototype.reportMovingEnd = function() {
			this.isMoving = false;
			if (!this.isTouching) {
				return this.reportScrollEnd();
			}
		};

		EnhancedScroller.prototype.reportScrollEnd = function() {
			if (this.isScrolling) {
				this.trigger('scrollEnd');
				return this.isScrolling = false;
			}
		};

		EnhancedScroller.prototype.reportTouchStart = function() {
			return this.isTouching = true;
		};

		EnhancedScroller.prototype.reportTouchEnd = function() {
			if (this.isTouching) {
				this.isTouching = false;
				if (this.isTouchScrollEnabled) {
					this.unbindPreventTouchScroll();
				}
				if (!this.isMoving) {
					return this.reportScrollEnd();
				}
			}
		};


		/*
  	If RTL, and scrolled to the left, returns NEGATIVE value (like Firefox)
   */

		EnhancedScroller.prototype.getScrollLeft = function() {
			var direction, node, val;
			direction = this.scrollEl.css('direction');
			node = this.scrollEl[0];
			val = node.scrollLeft;
			if (direction === 'rtl') {
				switch (rtlScrollSystem) {
					case 'positive':
						val = val + node.clientWidth - node.scrollWidth;
						break;
					case 'reverse':
						val = -val;
				}
			}
			return val;
		};


		/*
  	Accepts a NEGATIVE value for when scrolled in RTL
   */

		EnhancedScroller.prototype.setScrollLeft = function(val) {
			var direction, node;
			direction = this.scrollEl.css('direction');
			node = this.scrollEl[0];
			if (direction === 'rtl') {
				switch (rtlScrollSystem) {
					case 'positive':
						val = val - node.clientWidth + node.scrollWidth;
						break;
					case 'reverse':
						val = -val;
				}
			}
			return node.scrollLeft = val;
		};


		/*
  	Always returns the number of pixels scrolled from the leftmost position (even if RTL).
  	Always positive.
   */

		EnhancedScroller.prototype.getScrollFromLeft = function() {
			var direction, node, val;
			direction = this.scrollEl.css('direction');
			node = this.scrollEl[0];
			val = node.scrollLeft;
			if (direction === 'rtl') {
				switch (rtlScrollSystem) {
					case 'negative':
						val = val - node.clientWidth + node.scrollWidth;
						break;
					case 'reverse':
						val = -val - node.clientWidth + node.scrollWidth;
				}
			}
			return val;
		};

		EnhancedScroller.prototype.getNativeScrollLeft = function() {
			return this.scrollEl[0].scrollLeft;
		};

		EnhancedScroller.prototype.setNativeScrollLeft = function(val) {
			return this.scrollEl[0].scrollLeft = val;
		};

		rtlScrollSystem = null;

		detectRtlScrollSystem = function() {
			var el, node, system;
			el = $('<div style=" position: absolute top: -1000px; width: 1px; height: 1px; overflow: scroll; direction: rtl; font-size: 14px; ">A</div>').appendTo('body');
			node = el[0];
			system = node.scrollLeft > 0 ? 'positive' : (node.scrollLeft = 1, el.scrollLeft > 0 ? 'reverse' : 'negative');
			el.remove();
			return system;
		};

		$(function() {
			return rtlScrollSystem = detectRtlScrollSystem();
		});

		return EnhancedScroller;

	})(FC.Scroller);


	/*
A Scroller, but with a wrapping div that allows "clipping" away of native scrollbars,
giving the appearance that there are no scrollbars.
 */

	ClippedScroller = (function(superClass) {
		extend(ClippedScroller, superClass);

		ClippedScroller.prototype.isHScrollbarsClipped = false;

		ClippedScroller.prototype.isVScrollbarsClipped = false;


		/*
  	Received overflows can be set to 'clipped', meaning scrollbars shouldn't be visible
  	to the user, but the area should still scroll.
   */

		function ClippedScroller() {
			ClippedScroller.__super__.constructor.apply(this, arguments);
			if (this.overflowX === 'clipped-scroll') {
				this.overflowX = 'scroll';
				this.isHScrollbarsClipped = true;
			}
			if (this.overflowY === 'clipped-scroll') {
				this.overflowY = 'scroll';
				this.isVScrollbarsClipped = true;
			}
		}

		ClippedScroller.prototype.renderEl = function() {
			var scrollEl;
			scrollEl = ClippedScroller.__super__.renderEl.apply(this, arguments);
			return $('<div class="fc-scroller-clip" />').append(scrollEl);
		};

		ClippedScroller.prototype.updateSize = function() {
			var cssProps, scrollEl, scrollbarWidths;
			scrollEl = this.scrollEl;
			scrollbarWidths = getScrollbarWidths(scrollEl);
			cssProps = {
				marginLeft: 0,
				marginRight: 0,
				marginTop: 0,
				marginBottom: 0
			};
			if (this.isHScrollbarsClipped) {
				cssProps.marginTop = -scrollbarWidths.top;
				cssProps.marginBottom = -scrollbarWidths.bottom;
			}
			if (this.isVScrollbarsClipped) {
				cssProps.marginLeft = -scrollbarWidths.left;
				cssProps.marginRight = -scrollbarWidths.right;
			}
			scrollEl.css(cssProps);
			return scrollEl.toggleClass('fc-no-scrollbars', (this.isHScrollbarsClipped || this.overflowX === 'hidden') && (this.isVScrollbarsClipped || this.overflowY === 'hidden') && !(scrollbarWidths.top || scrollbarWidths.bottom || scrollbarWidths.left || scrollbarWidths.right));
		};


		/*
  	Accounts for 'clipped' scrollbars
   */

		ClippedScroller.prototype.getScrollbarWidths = function() {
			var widths;
			widths = getScrollbarWidths(this.scrollEl);
			if (this.isHScrollbarsClipped) {
				widths.top = 0;
				widths.bottom = 0;
			}
			if (this.isVScrollbarsClipped) {
				widths.left = 0;
				widths.right = 0;
			}
			return widths;
		};

		return ClippedScroller;

	})(EnhancedScroller);


	/*
A rectangular area of content that lives within a Scroller.
Can have "gutters", areas of dead spacing around the perimeter.
Also very useful for forcing a width, which a Scroller cannot do alone.
Has a content area that lives above a background area.
 */

	ScrollerCanvas = (function() {
		ScrollerCanvas.prototype.el = null;

		ScrollerCanvas.prototype.contentEl = null;

		ScrollerCanvas.prototype.bgEl = null;

		ScrollerCanvas.prototype.gutters = null;

		ScrollerCanvas.prototype.width = null;

		ScrollerCanvas.prototype.minWidth = null;

		function ScrollerCanvas() {
			this.gutters = {};
		}

		ScrollerCanvas.prototype.render = function() {
			this.el = $('<div class="fc-scroller-canvas"> <div class="fc-content"></div> <div class="fc-bg"></div> </div>');
			this.contentEl = this.el.find('.fc-content');
			return this.bgEl = this.el.find('.fc-bg');
		};


		/*
  	If falsy, resets all the gutters to 0
   */

		ScrollerCanvas.prototype.setGutters = function(gutters) {
			if (!gutters) {
				this.gutters = {};
			} else {
				$.extend(this.gutters, gutters);
			}
			return this.updateSize();
		};

		ScrollerCanvas.prototype.setWidth = function(width1) {
			this.width = width1;
			return this.updateSize();
		};

		ScrollerCanvas.prototype.setMinWidth = function(minWidth1) {
			this.minWidth = minWidth1;
			return this.updateSize();
		};

		ScrollerCanvas.prototype.clearWidth = function() {
			this.width = null;
			this.minWidth = null;
			return this.updateSize();
		};

		ScrollerCanvas.prototype.updateSize = function() {
			var gutters;
			gutters = this.gutters;
			this.el.toggleClass('fc-gutter-left', Boolean(gutters.left)).toggleClass('fc-gutter-right', Boolean(gutters.right)).toggleClass('fc-gutter-top', Boolean(gutters.top)).toggleClass('fc-gutter-bottom', Boolean(gutters.bottom)).css({
				paddingLeft: gutters.left || '',
				paddingRight: gutters.right || '',
				paddingTop: gutters.top || '',
				paddingBottom: gutters.bottom || '',
				width: this.width != null ? this.width + (gutters.left || 0) + (gutters.right || 0) : '',
				minWidth: this.minWidth != null ? this.minWidth + (gutters.left || 0) + (gutters.right || 0) : ''
			});
			return this.bgEl.css({
				left: gutters.left || '',
				right: gutters.right || '',
				top: gutters.top || '',
				bottom: gutters.bottom || ''
			});
		};

		return ScrollerCanvas;

	})();

	ScrollJoiner = (function() {
		ScrollJoiner.prototype.axis = null;

		ScrollJoiner.prototype.scrollers = null;

		ScrollJoiner.prototype.masterScroller = null;

		function ScrollJoiner(axis, scrollers) {
			var j, len, ref, scroller;
			this.axis = axis;
			this.scrollers = scrollers;
			ref = this.scrollers;
			for (j = 0, len = ref.length; j < len; j++) {
				scroller = ref[j];
				this.initScroller(scroller);
			}
			return;
		}

		ScrollJoiner.prototype.initScroller = function(scroller) {
			scroller.scrollEl.on('wheel mousewheel DomMouseScroll MozMousePixelScroll', (function(_this) {
				return function() {
					_this.assignMasterScroller(scroller);
				};
			})(this));
			return scroller.on('scrollStart', (function(_this) {
				return function() {
					if (!_this.masterScroller) {
						return _this.assignMasterScroller(scroller);
					}
				};
			})(this)).on('scroll', (function(_this) {
				return function() {
					var j, len, otherScroller, ref, results;
					if (scroller === _this.masterScroller) {
						ref = _this.scrollers;
						results = [];
						for (j = 0, len = ref.length; j < len; j++) {
							otherScroller = ref[j];
							if (otherScroller !== scroller) {
								switch (_this.axis) {
									case 'horizontal':
										results.push(otherScroller.setNativeScrollLeft(scroller.getNativeScrollLeft()));
										break;
									case 'vertical':
										results.push(otherScroller.setScrollTop(scroller.getScrollTop()));
										break;
									default:
										results.push(void 0);
								}
							} else {
								results.push(void 0);
							}
						}
						return results;
					}
				};
			})(this)).on('scrollEnd', (function(_this) {
				return function() {
					if (scroller === _this.masterScroller) {
						return _this.unassignMasterScroller();
					}
				};
			})(this));
		};

		ScrollJoiner.prototype.assignMasterScroller = function(scroller) {
			var j, len, otherScroller, ref;
			this.unassignMasterScroller();
			this.masterScroller = scroller;
			ref = this.scrollers;
			for (j = 0, len = ref.length; j < len; j++) {
				otherScroller = ref[j];
				if (otherScroller !== scroller) {
					otherScroller.disableTouchScroll();
				}
			}
		};

		ScrollJoiner.prototype.unassignMasterScroller = function() {
			var j, len, otherScroller, ref;
			if (this.masterScroller) {
				ref = this.scrollers;
				for (j = 0, len = ref.length; j < len; j++) {
					otherScroller = ref[j];
					otherScroller.enableTouchScroll();
				}
				this.masterScroller = null;
			}
		};

		ScrollJoiner.prototype.update = function() {
			var allWidths, i, j, k, len, len1, maxBottom, maxLeft, maxRight, maxTop, ref, scroller, widths;
			allWidths = (function() {
				var j, len, ref, results;
				ref = this.scrollers;
				results = [];
				for (j = 0, len = ref.length; j < len; j++) {
					scroller = ref[j];
					results.push(scroller.getScrollbarWidths());
				}
				return results;
			}).call(this);
			maxLeft = maxRight = maxTop = maxBottom = 0;
			for (j = 0, len = allWidths.length; j < len; j++) {
				widths = allWidths[j];
				maxLeft = Math.max(maxLeft, widths.left);
				maxRight = Math.max(maxRight, widths.right);
				maxTop = Math.max(maxTop, widths.top);
				maxBottom = Math.max(maxBottom, widths.bottom);
			}
			ref = this.scrollers;
			for (i = k = 0, len1 = ref.length; k < len1; i = ++k) {
				scroller = ref[i];
				widths = allWidths[i];
				scroller.canvas.setGutters(this.axis === 'horizontal' ? {
					left: maxLeft - widths.left,
					right: maxRight - widths.right
				} : {
					top: maxTop - widths.top,
					bottom: maxBottom - widths.bottom
				});
			}
		};

		return ScrollJoiner;

	})();

	ScrollFollower = (function() {
		ScrollFollower.prototype.scroller = null;

		ScrollFollower.prototype.scrollbarWidths = null;

		ScrollFollower.prototype.sprites = null;

		ScrollFollower.prototype.viewportRect = null;

		ScrollFollower.prototype.contentOffset = null;

		ScrollFollower.prototype.isHFollowing = true;

		ScrollFollower.prototype.isVFollowing = false;

		ScrollFollower.prototype.allowPointerEvents = false;

		ScrollFollower.prototype.containOnNaturalLeft = false;

		ScrollFollower.prototype.containOnNaturalRight = false;

		ScrollFollower.prototype.minTravel = 0;

		ScrollFollower.prototype.isTouch = false;

		ScrollFollower.prototype.isForcedRelative = false;

		function ScrollFollower(scroller, allowPointerEvents) {
			this.allowPointerEvents = allowPointerEvents != null ? allowPointerEvents : false;
			this.scroller = scroller;
			this.sprites = [];
			scroller.on('scroll', (function(_this) {
				return function() {
					if (scroller.isTouching) {
						_this.isTouch = true;
						return _this.isForcedRelative = true;
					} else {
						_this.isTouch = false;
						_this.isForcedRelative = false;
						return _this.handleScroll();
					}
				};
			})(this));
			scroller.on('scrollEnd', (function(_this) {
				return function() {
					return _this.handleScroll();
				};
			})(this));
		}

		ScrollFollower.prototype.setSprites = function(sprites) {
			var j, len, sprite;
			this.clearSprites();
			if (sprites instanceof $) {
				return this.sprites = (function() {
					var j, len, results;
					results = [];
					for (j = 0, len = sprites.length; j < len; j++) {
						sprite = sprites[j];
						results.push(new ScrollFollowerSprite($(sprite), this));
					}
					return results;
				}).call(this);
			} else {
				for (j = 0, len = sprites.length; j < len; j++) {
					sprite = sprites[j];
					sprite.follower = this;
				}
				return this.sprites = sprites;
			}
		};

		ScrollFollower.prototype.clearSprites = function() {
			var j, len, ref, sprite;
			ref = this.sprites;
			for (j = 0, len = ref.length; j < len; j++) {
				sprite = ref[j];
				sprite.clear();
			}
			return this.sprites = [];
		};

		ScrollFollower.prototype.handleScroll = function() {
			this.updateViewport();
			return this.updatePositions();
		};

		ScrollFollower.prototype.cacheDimensions = function() {
			var j, len, ref, sprite;
			this.updateViewport();
			this.scrollbarWidths = this.scroller.getScrollbarWidths();
			this.contentOffset = this.scroller.canvas.el.offset();
			ref = this.sprites;
			for (j = 0, len = ref.length; j < len; j++) {
				sprite = ref[j];
				sprite.cacheDimensions();
			}
		};

		ScrollFollower.prototype.updateViewport = function() {
			var left, scroller, top;
			scroller = this.scroller;
			left = scroller.getScrollFromLeft();
			top = scroller.getScrollTop();
			return this.viewportRect = {
				left: left,
				right: left + scroller.getClientWidth(),
				top: top,
				bottom: top + scroller.getClientHeight()
			};
		};

		ScrollFollower.prototype.forceRelative = function() {
			var j, len, ref, results, sprite;
			if (!this.isForcedRelative) {
				this.isForcedRelative = true;
				ref = this.sprites;
				results = [];
				for (j = 0, len = ref.length; j < len; j++) {
					sprite = ref[j];
					if (sprite.doAbsolute) {
						results.push(sprite.assignPosition());
					} else {
						results.push(void 0);
					}
				}
				return results;
			}
		};

		ScrollFollower.prototype.clearForce = function() {
			var j, len, ref, results, sprite;
			if (this.isForcedRelative && !this.isTouch) {
				this.isForcedRelative = false;
				ref = this.sprites;
				results = [];
				for (j = 0, len = ref.length; j < len; j++) {
					sprite = ref[j];
					results.push(sprite.assignPosition());
				}
				return results;
			}
		};

		ScrollFollower.prototype.update = function() {
			this.cacheDimensions();
			return this.updatePositions();
		};

		ScrollFollower.prototype.updatePositions = function() {
			var j, len, ref, sprite;
			ref = this.sprites;
			for (j = 0, len = ref.length; j < len; j++) {
				sprite = ref[j];
				sprite.updatePosition();
			}
		};

		ScrollFollower.prototype.getContentRect = function(el) {
			return getContentRect(el, this.contentOffset);
		};

		ScrollFollower.prototype.getBoundingRect = function(el) {
			return getOuterRect(el, this.contentOffset);
		};

		return ScrollFollower;

	})();

	ScrollFollowerSprite = (function() {
		ScrollFollowerSprite.prototype.follower = null;

		ScrollFollowerSprite.prototype.el = null;

		ScrollFollowerSprite.prototype.absoluteEl = null;

		ScrollFollowerSprite.prototype.naturalRect = null;

		ScrollFollowerSprite.prototype.parentRect = null;

		ScrollFollowerSprite.prototype.containerRect = null;

		ScrollFollowerSprite.prototype.isEnabled = true;

		ScrollFollowerSprite.prototype.isHFollowing = false;

		ScrollFollowerSprite.prototype.isVFollowing = false;

		ScrollFollowerSprite.prototype.doAbsolute = false;

		ScrollFollowerSprite.prototype.isAbsolute = false;

		ScrollFollowerSprite.prototype.isCentered = false;

		ScrollFollowerSprite.prototype.rect = null;

		ScrollFollowerSprite.prototype.isBlock = false;

		ScrollFollowerSprite.prototype.naturalWidth = null;

		function ScrollFollowerSprite(el1, follower1) {
			this.el = el1;
			this.follower = follower1 != null ? follower1 : null;
			this.isBlock = this.el.css('display') === 'block';
			this.el.css('position', 'relative');
		}

		ScrollFollowerSprite.prototype.disable = function() {
			if (this.isEnabled) {
				this.isEnabled = false;
				this.resetPosition();
				return this.unabsolutize();
			}
		};

		ScrollFollowerSprite.prototype.enable = function() {
			if (!this.isEnabled) {
				this.isEnabled = true;
				return this.assignPosition();
			}
		};

		ScrollFollowerSprite.prototype.clear = function() {
			this.disable();
			this.follower = null;
			return this.absoluteEl = null;
		};

		ScrollFollowerSprite.prototype.cacheDimensions = function() {
			var containerRect, follower, isCentered, isHFollowing, isVFollowing, minTravel, naturalRect, parentEl;
			isHFollowing = false;
			isVFollowing = false;
			isCentered = false;
			this.naturalWidth = this.el.width();
			this.resetPosition();
			follower = this.follower;
			naturalRect = this.naturalRect = follower.getBoundingRect(this.el);
			parentEl = this.el.parent();
			this.parentRect = follower.getBoundingRect(parentEl);
			containerRect = this.containerRect = joinRects(follower.getContentRect(parentEl), naturalRect);
			minTravel = follower.minTravel;
			if (follower.containOnNaturalLeft) {
				containerRect.left = naturalRect.left;
			}
			if (follower.containOnNaturalRight) {
				containerRect.right = naturalRect.right;
			}
			if (follower.isHFollowing) {
				if (getRectWidth(containerRect) - getRectWidth(naturalRect) >= minTravel) {
					isCentered = this.el.css('text-align') === 'center';
					isHFollowing = true;
				}
			}
			if (follower.isVFollowing) {
				if (getRectHeight(containerRect) - getRectHeight(naturalRect) >= minTravel) {
					isVFollowing = true;
				}
			}
			this.isHFollowing = isHFollowing;
			this.isVFollowing = isVFollowing;
			return this.isCentered = isCentered;
		};

		ScrollFollowerSprite.prototype.updatePosition = function() {
			this.computePosition();
			return this.assignPosition();
		};

		ScrollFollowerSprite.prototype.resetPosition = function() {
			return this.el.css({
				top: '',
				left: ''
			});
		};

		ScrollFollowerSprite.prototype.computePosition = function() {
			var containerRect, doAbsolute, parentRect, rect, rectWidth, subjectRect, viewportRect, visibleParentRect;
			viewportRect = this.follower.viewportRect;
			parentRect = this.parentRect;
			containerRect = this.containerRect;
			visibleParentRect = intersectRects(viewportRect, parentRect);
			rect = null;
			doAbsolute = false;
			if (visibleParentRect) {
				rect = copyRect(this.naturalRect);
				subjectRect = intersectRects(rect, parentRect);
				if ((this.isCentered && !testRectContains(viewportRect, parentRect)) || (subjectRect && !testRectContains(viewportRect, subjectRect))) {
					doAbsolute = true;
					if (this.isHFollowing) {
						if (this.isCentered) {
							rectWidth = getRectWidth(rect);
							rect.left = (visibleParentRect.left + visibleParentRect.right) / 2 - rectWidth / 2;
							rect.right = rect.left + rectWidth;
						} else {
							if (!hContainRect(rect, viewportRect)) {
								doAbsolute = false;
							}
						}
						if (hContainRect(rect, containerRect)) {
							doAbsolute = false;
						}
					}
					if (this.isVFollowing) {
						if (!vContainRect(rect, viewportRect)) {
							doAbsolute = false;
						}
						if (vContainRect(rect, containerRect)) {
							doAbsolute = false;
						}
					}
					if (!testRectContains(viewportRect, rect)) {
						doAbsolute = false;
					}
				}
			}
			this.rect = rect;
			return this.doAbsolute = doAbsolute;
		};

		ScrollFollowerSprite.prototype.assignPosition = function() {
			var left, top;
			if (this.isEnabled) {
				if (!this.rect) {
					return this.unabsolutize();
				} else if (this.doAbsolute && !this.follower.isForcedRelative) {
					this.absolutize();
					return this.absoluteEl.css({
						top: this.rect.top - this.follower.viewportRect.top + this.follower.scrollbarWidths.top,
						left: this.rect.left - this.follower.viewportRect.left + this.follower.scrollbarWidths.left,
						width: this.isBlock ? this.naturalWidth : ''
					});
				} else {
					top = this.rect.top - this.naturalRect.top;
					left = this.rect.left - this.naturalRect.left;
					this.unabsolutize();
					return this.el.toggleClass('fc-following', Boolean(top || left)).css({
						top: top,
						left: left
					});
				}
			}
		};

		ScrollFollowerSprite.prototype.absolutize = function() {
			if (!this.isAbsolute) {
				if (!this.absoluteEl) {
					this.absoluteEl = this.buildAbsoluteEl();
				}
				this.absoluteEl.appendTo(this.follower.scroller.el);
				this.el.css('visibility', 'hidden');
				return this.isAbsolute = true;
			}
		};

		ScrollFollowerSprite.prototype.unabsolutize = function() {
			if (this.isAbsolute) {
				this.absoluteEl.detach();
				this.el.css('visibility', '');
				return this.isAbsolute = false;
			}
		};

		ScrollFollowerSprite.prototype.buildAbsoluteEl = function() {
			var el;
			el = this.el.clone().addClass('fc-following');
			el.css({
				'position': 'absolute',
				'z-index': 1000,
				'font-weight': this.el.css('font-weight'),
				'font-size': this.el.css('font-size'),
				'font-family': this.el.css('font-family'),
				'text-decoration': this.el.css('text-decoration'),
				'color': this.el.css('color'),
				'padding-top': this.el.css('padding-top'),
				'padding-bottom': this.el.css('padding-bottom'),
				'padding-left': this.el.css('padding-left'),
				'padding-right': this.el.css('padding-right')
			});
			if (!this.follower.allowPointerEvents) {
				el.css('pointer-events', 'none');
			}
			return el;
		};

		return ScrollFollowerSprite;

	})();

	copyRect = function(rect) {
		return {
			left: rect.left,
			right: rect.right,
			top: rect.top,
			bottom: rect.bottom
		};
	};

	getRectWidth = function(rect) {
		return rect.right - rect.left;
	};

	getRectHeight = function(rect) {
		return rect.bottom - rect.top;
	};

	testRectContains = function(rect, innerRect) {
		return testRectHContains(rect, innerRect) && testRectVContains(rect, innerRect);
	};

	testRectHContains = function(rect, innerRect) {
		return innerRect.left >= rect.left && innerRect.right <= rect.right;
	};

	testRectVContains = function(rect, innerRect) {
		return innerRect.top >= rect.top && innerRect.bottom <= rect.bottom;
	};

	hContainRect = function(rect, outerRect) {
		if (rect.left < outerRect.left) {
			rect.right = outerRect.left + getRectWidth(rect);
			rect.left = outerRect.left;
			return true;
		} else if (rect.right > outerRect.right) {
			rect.left = outerRect.right - getRectWidth(rect);
			rect.right = outerRect.right;
			return true;
		} else {
			return false;
		}
	};

	vContainRect = function(rect, outerRect) {
		if (rect.top < outerRect.top) {
			rect.bottom = outerRect.top + getRectHeight(rect);
			rect.top = outerRect.top;
			return true;
		} else if (rect.bottom > outerRect.bottom) {
			rect.top = outerRect.bottom - getRectHeight(rect);
			rect.bottom = outerRect.bottom;
			return true;
		} else {
			return false;
		}
	};

	joinRects = function(rect1, rect2) {
		return {
			left: Math.min(rect1.left, rect2.left),
			right: Math.max(rect1.right, rect2.right),
			top: Math.min(rect1.top, rect2.top),
			bottom: Math.max(rect1.bottom, rect2.bottom)
		};
	};

	CalendarExtension = (function(superClass) {
		extend(CalendarExtension, superClass);

		function CalendarExtension() {
			return CalendarExtension.__super__.constructor.apply(this, arguments);
		}

		CalendarExtension.prototype.resourceManager = null;

		CalendarExtension.prototype.initialize = function() {
			return this.resourceManager = new ResourceManager(this);
		};

		CalendarExtension.prototype.instantiateView = function(viewType) {
			var spec, viewClass;
			spec = this.getViewSpec(viewType);
			viewClass = spec['class'];
			if (this.options.resources && spec.options.resources !== false) {
				if (spec.queryResourceClass) {
					viewClass = spec.queryResourceClass(spec) || viewClass;
				} else if (spec.resourceClass) {
					viewClass = spec.resourceClass;
				}
			}
			return new viewClass(this, viewType, spec.options, spec.duration);
		};

		CalendarExtension.prototype.getResources = function() {
			return Array.prototype.slice.call(this.resourceManager.topLevelResources);
		};

		CalendarExtension.prototype.addResource = function(resourceInput, scroll) {
			var promise;
			if (scroll == null) {
				scroll = false;
			}
			promise = this.resourceManager.addResource(resourceInput);
			if (scroll && this.view.scrollToResource) {
				promise.then((function(_this) {
					return function(resource) {
						return _this.view.scrollToResource(resource);
					};
				})(this));
			}
		};

		CalendarExtension.prototype.removeResource = function(idOrResource) {
			return this.resourceManager.removeResource(idOrResource);
		};

		CalendarExtension.prototype.refetchResources = function() {
			this.resourceManager.fetchResources();
		};

		CalendarExtension.prototype.rerenderResources = function() {
			this.resourceManager.resetCurrentResources();
		};

		CalendarExtension.prototype.isSpanAllowed = function(span, constraint) {
			var constrainToResourceIds, ref;
			if (typeof constraint === 'object') {
				constrainToResourceIds = this.getEventResourceIds(constraint);
				if (constrainToResourceIds.length && (!span.resourceId || !(ref = span.resourceId, indexOf.call(constrainToResourceIds, ref) >= 0))) {
					return false;
				}
			}
			return CalendarExtension.__super__.isSpanAllowed.apply(this, arguments);
		};

		CalendarExtension.prototype.getPeerEvents = function(span, event) {
			var filteredPeerEvents, isPeer, j, k, l, len, len1, len2, newResourceId, newResourceIds, peerEvent, peerEvents, peerResourceId, peerResourceIds;
			peerEvents = CalendarExtension.__super__.getPeerEvents.apply(this, arguments);
			newResourceIds = span.resourceId ? [span.resourceId] : event ? this.getEventResourceIds(event) : [];
			filteredPeerEvents = [];
			for (j = 0, len = peerEvents.length; j < len; j++) {
				peerEvent = peerEvents[j];
				isPeer = false;
				peerResourceIds = this.getEventResourceIds(peerEvent);
				if (!peerResourceIds.length || !newResourceIds.length) {
					isPeer = true;
				} else {
					for (k = 0, len1 = peerResourceIds.length; k < len1; k++) {
						peerResourceId = peerResourceIds[k];
						for (l = 0, len2 = newResourceIds.length; l < len2; l++) {
							newResourceId = newResourceIds[l];
							if (newResourceId === peerResourceId) {
								isPeer = true;
								break;
							}
						}
					}
				}
				if (isPeer) {
					filteredPeerEvents.push(peerEvent);
				}
			}
			return filteredPeerEvents;
		};

		CalendarExtension.prototype.spanContainsSpan = function(outerSpan, innerSpan) {
			if (outerSpan.resourceId && outerSpan.resourceId !== innerSpan.resourceId) {
				return false;
			} else {
				return CalendarExtension.__super__.spanContainsSpan.apply(this, arguments);
			}
		};

		CalendarExtension.prototype.getCurrentBusinessHourEvents = function(wholeDay) {
			var allEvents, anyCustomBusinessHours, event, events, flatResources, j, k, l, len, len1, len2, resource;
			flatResources = this.resourceManager.getFlatResources();
			anyCustomBusinessHours = false;
			for (j = 0, len = flatResources.length; j < len; j++) {
				resource = flatResources[j];
				if (resource.businessHours) {
					anyCustomBusinessHours = true;
				}
			}
			if (anyCustomBusinessHours) {
				allEvents = [];
				for (k = 0, len1 = flatResources.length; k < len1; k++) {
					resource = flatResources[k];
					events = this.computeBusinessHourEvents(wholeDay, resource.businessHours || this.options.businessHours);
					for (l = 0, len2 = events.length; l < len2; l++) {
						event = events[l];
						event.resourceId = resource.id;
						allEvents.push(event);
					}
				}
				return allEvents;
			} else {
				return CalendarExtension.__super__.getCurrentBusinessHourEvents.apply(this, arguments);
			}
		};

		CalendarExtension.prototype.buildSelectSpan = function(startInput, endInput, resourceId) {
			var span;
			span = CalendarExtension.__super__.buildSelectSpan.apply(this, arguments);
			if (resourceId) {
				span.resourceId = resourceId;
			}
			return span;
		};

		CalendarExtension.prototype.getResourceById = function(id) {
			return this.resourceManager.getResourceById(id);
		};

		CalendarExtension.prototype.normalizeEvent = function(event) {
			CalendarExtension.__super__.normalizeEvent.apply(this, arguments);
			if (event.resourceId == null) {
				event.resourceId = null;
			}
			return event.resourceIds != null ? event.resourceIds : event.resourceIds = null;
		};

		CalendarExtension.prototype.getEventResourceId = function(event) {
			return this.getEventResourceIds(event)[0];
		};

		CalendarExtension.prototype.getEventResourceIds = function(event) {
			var j, len, normalResourceId, normalResourceIds, ref, ref1, ref2, resourceId;
			resourceId = String((ref = (ref1 = event[this.getEventResourceField()]) != null ? ref1 : event.resourceId) != null ? ref : '');
			if (resourceId) {
				return [resourceId];
			} else if (event.resourceIds) {
				normalResourceIds = [];
				ref2 = event.resourceIds;
				for (j = 0, len = ref2.length; j < len; j++) {
					resourceId = ref2[j];
					normalResourceId = String(resourceId != null ? resourceId : '');
					if (normalResourceId) {
						normalResourceIds.push(normalResourceId);
					}
				}
				return normalResourceIds;
			} else {
				return [];
			}
		};

		CalendarExtension.prototype.setEventResourceId = function(event, resourceId) {
			return event[this.getEventResourceField()] = String(resourceId != null ? resourceId : '');
		};

		CalendarExtension.prototype.getEventResourceField = function() {
			return this.options['eventResourceField'] || 'resourceId';
		};

		CalendarExtension.prototype.getResourceEvents = function(idOrResource) {
			var resource;
			resource = typeof idOrResource === 'object' ? idOrResource : this.getResourceById(idOrResource);
			if (resource) {
				return this.clientEvents((function(_this) {
					return function(event) {
						return $.inArray(resource.id, _this.getEventResourceIds(event)) !== -1;
					};
				})(this));
			} else {
				return [];
			}
		};

		CalendarExtension.prototype.getEventResource = function(idOrEvent) {
			return this.getEventResources(idOrEvent)[0];
		};

		CalendarExtension.prototype.getEventResources = function(idOrEvent) {
			var event, j, len, ref, resource, resourceId, resources;
			event = typeof idOrEvent === 'object' ? idOrEvent : this.clientEvents(idOrEvent)[0];
			resources = [];
			if (event) {
				ref = this.getEventResourceIds(event);
				for (j = 0, len = ref.length; j < len; j++) {
					resourceId = ref[j];
					resource = this.getResourceById(resourceId);
					if (resource) {
						resources.push(resource);
					}
				}
			}
			return resources;
		};

		return CalendarExtension;

	})(Calendar);

	Calendar.prototype = CalendarExtension.prototype;

	origSetElement = View.prototype.setElement;

	origRemoveElement = View.prototype.removeElement;

	origHandleDate = View.prototype.handleDate;

	origOnDateRender = View.prototype.onDateRender;

	origExecuteEventsRender = View.prototype.executeEventsRender;

	View.prototype.isResourcesBound = false;

	View.prototype.isResourcesSet = false;

	Calendar.defaults.refetchResourcesOnNavigate = false;

	View.prototype.setElement = function() {
		var promise;
		promise = origSetElement.apply(this, arguments);
		this.bindResources();
		return promise;
	};

	View.prototype.removeElement = function() {
		this.unbindResources({
			skipRerender: true
		});
		return origRemoveElement.apply(this, arguments);
	};

	View.prototype.handleDate = function(date, isReset) {
		if (isReset && this.opt('refetchResourcesOnNavigate')) {
			this.unsetResources({
				skipUnrender: true
			});
			this.fetchResources();
		}
		return origHandleDate.apply(this, arguments);
	};

	View.prototype.onDateRender = function() {
		return origOnDateRender.apply(this, arguments);
	};

	View.prototype.executeEventsRender = function(events) {
		return this.whenResourcesSet().then((function(_this) {
			return function() {
				return origExecuteEventsRender.call(_this, events);
			};
		})(this));
	};

	View.prototype.bindResources = function() {
		var promise;
		if (!this.isResourcesBound) {
			this.isResourcesBound = true;
			this.trigger('resourcesBind');
			promise = this.opt('refetchResourcesOnNavigate') ? this.fetchResources() : this.requestResources();
			return this.rejectOn('resourcesUnbind', promise).then((function(_this) {
				return function(resources) {
					_this.listenTo(_this.calendar.resourceManager, {
						set: _this.setResources,
						reset: _this.setResources,
						unset: _this.unsetResources,
						add: _this.addResource,
						remove: _this.removeResource
					});
					return _this.setResources(resources);
				};
			})(this));
		}
	};

	View.prototype.unbindResources = function(teardownOptions) {
		if (this.isResourcesBound) {
			this.isResourcesBound = false;
			this.stopListeningTo(this.calendar.resourceManager);
			this.unsetResources(teardownOptions);
			return this.trigger('resourcesUnbind');
		}
	};

	View.prototype.setResources = function(resources) {
		var isReset;
		isReset = this.isResourcesSet;
		this.isResourcesSet = true;
		this.handleResources(resources, isReset);
		return this.trigger(isReset ? 'resourcesReset' : 'resourcesSet', resources);
	};

	View.prototype.unsetResources = function(teardownOptions) {
		if (this.isResourcesSet) {
			this.isResourcesSet = false;
			this.handleResourcesUnset(teardownOptions);
			return this.trigger('resourcesUnset');
		}
	};

	View.prototype.whenResourcesSet = function() {
		if (this.isResourcesSet) {
			return Promise.resolve();
		} else {
			return new Promise((function(_this) {
				return function(resolve) {
					return _this.one('resourcesSet', resolve);
				};
			})(this));
		}
	};

	View.prototype.addResource = function(resource) {
		if (this.isResourcesSet) {
			this.handleResourceAdd(resource);
			return this.trigger('resourceAdd', resource);
		}
	};

	View.prototype.removeResource = function(resource) {
		if (this.isResourcesSet) {
			this.handleResourceRemove(resource);
			return this.trigger('resourceRemove', resource);
		}
	};

	View.prototype.handleResources = function(resources) {
		if (this.isEventsRendered) {
			return this.requestCurrentEventsRender();
		}
	};

	View.prototype.handleResourcesUnset = function(teardownOptions) {
		if (teardownOptions == null) {
			teardownOptions = {};
		}
		return this.requestEventsUnrender();
	};

	View.prototype.handleResourceAdd = function(resource) {
		if (this.isEventsRendered) {
			return this.requestCurrentEventsRender();
		}
	};

	View.prototype.handleResourceRemove = function(resource) {
		if (this.isEventsRendered) {
			return this.requestCurrentEventsRender();
		}
	};

	View.prototype.requestResources = function() {
		return this.calendar.resourceManager.getResources();
	};

	View.prototype.fetchResources = function() {
		return this.calendar.resourceManager.fetchResources();
	};

	View.prototype.getCurrentResources = function() {
		return this.calendar.resourceManager.topLevelResources;
	};

	origGetSegCustomClasses = Grid.prototype.getSegCustomClasses;

	origGetSegDefaultBackgroundColor = Grid.prototype.getSegDefaultBackgroundColor;

	origGetSegDefaultBorderColor = Grid.prototype.getSegDefaultBorderColor;

	origGetSegDefaultTextColor = Grid.prototype.getSegDefaultTextColor;

	Grid.prototype.getSegCustomClasses = function(seg) {
		var classes, j, len, ref, resource;
		classes = origGetSegCustomClasses.apply(this, arguments);
		ref = this.getSegResources(seg);
		for (j = 0, len = ref.length; j < len; j++) {
			resource = ref[j];
			classes = classes.concat(resource.eventClassName || []);
		}
		return classes;
	};

	Grid.prototype.getSegDefaultBackgroundColor = function(seg) {
		var currentResource, j, len, resources, val;
		resources = this.getSegResources(seg);
		for (j = 0, len = resources.length; j < len; j++) {
			currentResource = resources[j];
			while (currentResource) {
				val = currentResource.eventBackgroundColor || currentResource.eventColor;
				if (val) {
					return val;
				}
				currentResource = currentResource._parent;
			}
		}
		return origGetSegDefaultBackgroundColor.apply(this, arguments);
	};

	Grid.prototype.getSegDefaultBorderColor = function(seg) {
		var currentResource, j, len, resources, val;
		resources = this.getSegResources(seg);
		for (j = 0, len = resources.length; j < len; j++) {
			currentResource = resources[j];
			while (currentResource) {
				val = currentResource.eventBorderColor || currentResource.eventColor;
				if (val) {
					return val;
				}
				currentResource = currentResource._parent;
			}
		}
		return origGetSegDefaultBorderColor.apply(this, arguments);
	};

	Grid.prototype.getSegDefaultTextColor = function(seg) {
		var currentResource, j, len, resources, val;
		resources = this.getSegResources(seg);
		for (j = 0, len = resources.length; j < len; j++) {
			currentResource = resources[j];
			while (currentResource) {
				val = currentResource.eventTextColor;
				if (val) {
					return val;
				}
				currentResource = currentResource._parent;
			}
		}
		return origGetSegDefaultTextColor.apply(this, arguments);
	};

	Grid.prototype.getSegResources = function(seg) {
		if (seg.resource) {
			return [seg.resource];
		} else {
			return this.view.calendar.getEventResources(seg.event);
		}
	};

	ResourceManager = (function(superClass) {
		extend(ResourceManager, superClass);

		ResourceManager.mixin(EmitterMixin);

		ResourceManager.resourceGuid = 1;

		ResourceManager.ajaxDefaults = {
			dataType: 'json',
			cache: false
		};

		ResourceManager.prototype.calendar = null;

		ResourceManager.prototype.fetchId = 0;

		ResourceManager.prototype.topLevelResources = null;

		ResourceManager.prototype.resourcesById = null;

		ResourceManager.prototype.fetching = null;

		function ResourceManager(calendar1) {
			this.calendar = calendar1;
			this.initializeCache();
		}

		ResourceManager.prototype.getResources = function() {
			return this.fetching || this.fetchResources();
		};

		ResourceManager.prototype.fetchResources = function() {
			var currentFetchId;
			currentFetchId = (this.fetchId += 1);
			return this.fetching = new Promise((function(_this) {
				return function(resolve, reject) {
					return _this.fetchResourceInputs(function(resourceInputs) {
						if (currentFetchId === _this.fetchId) {
							_this.setResources(resourceInputs);
							return resolve(_this.topLevelResources);
						} else {
							return reject();
						}
					});
				};
			})(this));
		};

		ResourceManager.prototype.fetchResourceInputs = function(callback) {
			var source;
			source = this.calendar.options['resources'];
			if ($.type(source) === 'string') {
				source = {
					url: source
				};
			}
			switch ($.type(source)) {
				case 'function':
					this.calendar.pushLoading();
					return source((function(_this) {
						return function(resourceInputs) {
							_this.calendar.popLoading();
							return callback(resourceInputs);
						};
					})(this));
				case 'object':
					this.calendar.pushLoading();
					return $.ajax($.extend({}, ResourceManager.ajaxDefaults, source)).then((function(_this) {
						return function(resourceInputs) {
							_this.calendar.popLoading();
							return callback(resourceInputs);
						};
					})(this));
				case 'array':
					return callback(source);
				default:
					return callback([]);
			}
		};

		ResourceManager.prototype.getResourceById = function(id) {
			return this.resourcesById[id];
		};

		ResourceManager.prototype.getFlatResources = function() {
			var id, results;
			results = [];
			for (id in this.resourcesById) {
				results.push(this.resourcesById[id]);
			}
			return results;
		};

		ResourceManager.prototype.initializeCache = function() {
			this.topLevelResources = [];
			return this.resourcesById = {};
		};

		ResourceManager.prototype.setResources = function(resourceInputs) {
			var j, len, resource, resourceInput, resources, validResources, wasSet;
			wasSet = Boolean(this.topLevelResources);
			this.initializeCache();
			resources = (function() {
				var j, len, results;
				results = [];
				for (j = 0, len = resourceInputs.length; j < len; j++) {
					resourceInput = resourceInputs[j];
					results.push(this.buildResource(resourceInput));
				}
				return results;
			}).call(this);
			validResources = (function() {
				var j, len, results;
				results = [];
				for (j = 0, len = resources.length; j < len; j++) {
					resource = resources[j];
					if (this.addResourceToIndex(resource)) {
						results.push(resource);
					}
				}
				return results;
			}).call(this);
			for (j = 0, len = validResources.length; j < len; j++) {
				resource = validResources[j];
				this.addResourceToTree(resource);
			}
			if (wasSet) {
				this.trigger('reset', this.topLevelResources);
			} else {
				this.trigger('set', this.topLevelResources);
			}
			return this.calendar.publiclyTrigger('resourcesSet', null, this.topLevelResources);
		};

		ResourceManager.prototype.resetCurrentResources = function() {
			if (this.topLevelResources) {
				return this.trigger('reset', this.topLevelResources);
			}
		};

		ResourceManager.prototype.addResource = function(resourceInput) {
			return this.getResources().then((function(_this) {
				return function() {
					var resource;
					resource = _this.buildResource(resourceInput);
					if (_this.addResourceToIndex(resource)) {
						_this.addResourceToTree(resource);
						_this.trigger('add', resource);
						return resource;
					} else {
						return false;
					}
				};
			})(this));
		};

		ResourceManager.prototype.addResourceToIndex = function(resource) {
			var child, j, len, ref;
			if (this.resourcesById[resource.id]) {
				return false;
			} else {
				this.resourcesById[resource.id] = resource;
				ref = resource.children;
				for (j = 0, len = ref.length; j < len; j++) {
					child = ref[j];
					this.addResourceToIndex(child);
				}
				return true;
			}
		};

		ResourceManager.prototype.addResourceToTree = function(resource) {
			var parent, parentId, ref, siblings;
			if (!resource.parent) {
				parentId = String((ref = resource['parentId']) != null ? ref : '');
				if (parentId) {
					parent = this.resourcesById[parentId];
					if (parent) {
						resource.parent = parent;
						siblings = parent.children;
					} else {
						return false;
					}
				} else {
					siblings = this.topLevelResources;
				}
				siblings.push(resource);
			}
			return true;
		};

		ResourceManager.prototype.removeResource = function(idOrResource) {
			var id;
			id = typeof idOrResource === 'object' ? idOrResource.id : idOrResource;
			return this.getResources().then((function(_this) {
				return function() {
					var resource;
					resource = _this.removeResourceFromIndex(id);
					if (resource) {
						_this.removeResourceFromTree(resource);
						_this.trigger('remove', resource);
					}
					return resource;
				};
			})(this));
		};

		ResourceManager.prototype.removeResourceFromIndex = function(resourceId) {
			var child, j, len, ref, resource;
			resource = this.resourcesById[resourceId];
			if (resource) {
				delete this.resourcesById[resourceId];
				ref = resource.children;
				for (j = 0, len = ref.length; j < len; j++) {
					child = ref[j];
					this.removeResourceFromIndex(child.id);
				}
				return resource;
			} else {
				return false;
			}
		};

		ResourceManager.prototype.removeResourceFromTree = function(resource, siblings) {
			var i, j, len, sibling;
			if (siblings == null) {
				siblings = this.topLevelResources;
			}
			for (i = j = 0, len = siblings.length; j < len; i = ++j) {
				sibling = siblings[i];
				if (sibling === resource) {
					resource.parent = null;
					siblings.splice(i, 1);
					return true;
				}
				if (this.removeResourceFromTree(resource, sibling.children)) {
					return true;
				}
			}
			return false;
		};

		ResourceManager.prototype.buildResource = function(resourceInput) {
			var child, childInput, rawClassName, ref, resource;
			resource = $.extend({}, resourceInput);
			resource.id = String((ref = resourceInput.id) != null ? ref : '_fc' + ResourceManager.resourceGuid++);
			rawClassName = resourceInput.eventClassName;
			resource.eventClassName = (function() {
				switch ($.type(rawClassName)) {
					case 'string':
						return rawClassName.split(/\s+/);
					case 'array':
						return rawClassName;
					default:
						return [];
				}
			})();
			resource.children = (function() {
				var j, len, ref1, ref2, results;
				ref2 = (ref1 = resourceInput.children) != null ? ref1 : [];
				results = [];
				for (j = 0, len = ref2.length; j < len; j++) {
					childInput = ref2[j];
					child = this.buildResource(childInput);
					child.parent = resource;
					results.push(child);
				}
				return results;
			}).call(this);
			return resource;
		};

		return ResourceManager;

	})(Class);

	Calendar.defaults.filterResourcesWithEvents = false;


	/*
A view that structurally distinguishes events by resource
 */

	ResourceViewMixin = {
		isResourcesRendered: false,
		isResourcesDirty: false,
		resourceRenderQueue: null,
		resourceTextFunc: null,
		canRenderSpecificResources: false,
		setElement: function() {
			this.resourceRenderQueue = new TaskQueue();
			return View.prototype.setElement.apply(this, arguments);
		},
		onDateRender: function() {
			return this.rejectOn('dateUnrender', this.whenResourcesRendered()).then((function(_this) {
				return function() {
					return View.prototype.onDateRender.apply(_this, arguments);
				};
			})(this));
		},
		handleEvents: function(events) {
			var filteredResources, resources;
			if (this.opt('filterResourcesWithEvents')) {
				if (this.isResourcesSet) {
					resources = this.getCurrentResources();
					filteredResources = this.filterResourcesWithEvents(resources, events);
					return this.requestResourcesRender(filteredResources);
				}
			} else {
				if (this.isResourcesRendered && !this.isResourcesDirty) {
					return this.requestEventsRender(events);
				}
			}
		},
		handleResources: function(resources) {
			var events, filteredResources;
			if (this.opt('filterResourcesWithEvents')) {
				if (this.isEventsSet) {
					events = this.getCurrentEvents();
					filteredResources = this.filterResourcesWithEvents(resources, events);
					return this.requestResourcesRender(filteredResources);
				}
			} else {
				return this.requestResourcesRender(resources);
			}
		},
		handleResourcesUnset: function(teardownOptions) {
			if (teardownOptions == null) {
				teardownOptions = {};
			}
			if (teardownOptions.skipUnrender) {
				return this.isResourcesDirty = this.isResourcesRendered;
			} else {
				return this.requestResourcesUnrender(teardownOptions);
			}
		},
		handleResourceAdd: function(resource) {
			var a, events;
			if (this.canRenderSpecificResources) {
				if (this.opt('filterResourcesWithEvents')) {
					if (this.isEventsSet) {
						events = this.getCurrentEvents();
						a = this.filterResourcesWithEvents([resource], events);
						if (a.length) {
							return this.requestResourceRender(a[0]);
						}
					}
				} else {
					return this.requestResourceRender(resource);
				}
			} else {
				return this.handleResources(this.getCurrentResources());
			}
		},
		handleResourceRemove: function(resource) {
			if (this.canRenderSpecificResources) {
				return this.requestResourceUnrender(resource);
			} else {
				return this.handleResources(this.getCurrentResources());
			}
		},
		requestResourcesRender: function(resources) {
			return this.resourceRenderQueue.add((function(_this) {
				return function() {
					return _this.executeResourcesRender(resources);
				};
			})(this));
		},
		requestResourcesUnrender: function(teardownOptions) {
			if (this.isResourcesRendered) {
				return this.resourceRenderQueue.add((function(_this) {
					return function() {
						return _this.executeResourcesUnrender(teardownOptions);
					};
				})(this));
			} else {
				return Promise.resolve();
			}
		},
		requestResourceRender: function(resource) {
			return this.resourceRenderQueue.add((function(_this) {
				return function() {
					return _this.executeResourceRender(resource);
				};
			})(this));
		},
		requestResourceUnrender: function(resource) {
			return this.resourceRenderQueue.add((function(_this) {
				return function() {
					return _this.executeResourceUnrender(resource);
				};
			})(this));
		},
		executeResourcesRender: function(resources) {
			this.captureScroll();
			this.freezeHeight();
			return this.executeResourcesUnrender().then((function(_this) {
				return function() {
					_this.renderResources(resources);
					_this.thawHeight();
					_this.releaseScroll();
					return _this.reportResourcesRender();
				};
			})(this));
		},
		executeResourcesUnrender: function(teardownOptions) {
			if (this.isResourcesRendered) {
				return this.requestEventsUnrender().then((function(_this) {
					return function() {
						_this.captureScroll();
						_this.freezeHeight();
						_this.unrenderResources(teardownOptions);
						_this.thawHeight();
						_this.releaseScroll();
						return _this.reportResourcesUnrender();
					};
				})(this));
			} else {
				return Promise.resolve();
			}
		},
		executeResourceRender: function(resource) {
			if (this.isResourcesRendered) {
				this.captureScroll();
				this.freezeHeight();
				this.renderResource(resource);
				this.thawHeight();
				return this.releaseScroll();
			} else {
				return Promise.reject();
			}
		},
		executeResourceUnrender: function(resource) {
			if (this.isResourcesRendered) {
				this.captureScroll();
				this.freezeHeight();
				this.unrenderResource(resource);
				this.thawHeight();
				return this.releaseScroll();
			} else {
				return Promise.reject();
			}
		},
		reportResourcesRender: function() {
			this.isResourcesRendered = true;
			this.trigger('resourcesRender');
			if (this.isEventsSet) {
				this.requestEventsRender(this.getCurrentEvents());
			}
		},
		reportResourcesUnrender: function() {
			this.isResourcesRendered = false;
			return this.isResourcesDirty = false;
		},
		whenResourcesRendered: function() {
			if (this.isResourcesRendered && !this.isResourcesDirty) {
				return Promise.resolve();
			} else {
				return new Promise((function(_this) {
					return function(resolve) {
						return _this.one('resourcesRender', resolve);
					};
				})(this));
			}
		},
		renderResources: function(resources) {},
		unrenderResources: function(teardownOptions) {},
		renderResource: function(resource) {},
		unrenderResource: function(resource) {},
		isEventDraggable: function(event) {
			return this.isEventResourceEditable(event) || View.prototype.isEventDraggable.call(this, event);
		},
		isEventResourceEditable: function(event) {
			var ref, ref1, ref2;
			return (ref = (ref1 = (ref2 = event.resourceEditable) != null ? ref2 : (event.source || {}).resourceEditable) != null ? ref1 : this.opt('eventResourceEditable')) != null ? ref : this.isEventGenerallyEditable(event);
		},
		getResourceText: function(resource) {
			return this.getResourceTextFunc()(resource);
		},
		getResourceTextFunc: function() {
			var func;
			if (this.resourceTextFunc) {
				return this.resourceTextFunc;
			} else {
				func = this.opt('resourceText');
				if (typeof func !== 'function') {
					func = function(resource) {
						return resource.title || resource.id;
					};
				}
				return this.resourceTextFunc = func;
			}
		},
		triggerDayClick: function(span, dayEl, ev) {
			var resourceManager;
			resourceManager = this.calendar.resourceManager;
			return this.publiclyTrigger('dayClick', dayEl, this.calendar.applyTimezone(span.start), ev, this, resourceManager.getResourceById(span.resourceId));
		},
		triggerSelect: function(span, ev) {
			var resourceManager;
			resourceManager = this.calendar.resourceManager;
			return this.publiclyTrigger('select', null, this.calendar.applyTimezone(span.start), this.calendar.applyTimezone(span.end), ev, this, resourceManager.getResourceById(span.resourceId));
		},
		triggerExternalDrop: function(event, dropLocation, el, ev, ui) {
			this.publiclyTrigger('drop', el[0], dropLocation.start, ev, ui, dropLocation.resourceId);
			if (event) {
				return this.publiclyTrigger('eventReceive', null, event);
			}
		},

		/* Hacks
  	 * ------------------------------------------------------------------------------------------------------------------
  	These triggers usually call mutateEvent with dropLocation, which causes an event modification and rerender.
  	But mutateEvent isn't aware of eventResourceField, so it might be setting the wrong property. Workaround.
  	TODO: normalize somewhere else. maybe make a hook in core.
   */
		reportEventDrop: function() {
			var dropLocation, event, otherArgs, ref;
			event = arguments[0], dropLocation = arguments[1], otherArgs = 3 <= arguments.length ? slice.call(arguments, 2) : [];
			dropLocation = this.normalizeDropLocation(dropLocation);
			if (dropLocation.resourceId && event.resourceIds) {
				dropLocation.resourceIds = null;
			}
			return (ref = View.prototype.reportEventDrop).call.apply(ref, [this, event, dropLocation].concat(slice.call(otherArgs)));
		},
		reportExternalDrop: function() {
			var dropLocation, meta, otherArgs, ref;
			meta = arguments[0], dropLocation = arguments[1], otherArgs = 3 <= arguments.length ? slice.call(arguments, 2) : [];
			dropLocation = this.normalizeDropLocation(dropLocation);
			return (ref = View.prototype.reportExternalDrop).call.apply(ref, [this, meta, dropLocation].concat(slice.call(otherArgs)));
		},
		normalizeDropLocation: function(dropLocation) {
			var out;
			out = $.extend({}, dropLocation);
			delete out.resourceId;
			this.calendar.setEventResourceId(out, dropLocation.resourceId);
			return out;
		},
		filterResourcesWithEvents: function(resources, events) {
			var event, j, k, len, len1, ref, resourceId, resourceIdHits;
			resourceIdHits = {};
			for (j = 0, len = events.length; j < len; j++) {
				event = events[j];
				ref = this.calendar.getEventResourceIds(event);
				for (k = 0, len1 = ref.length; k < len1; k++) {
					resourceId = ref[k];
					resourceIdHits[resourceId] = true;
				}
			}
			return _filterResourcesWithEvents(resources, resourceIdHits);
		}
	};

	_filterResourcesWithEvents = function(sourceResources, resourceIdHits) {
		var filteredChildren, filteredResource, filteredResources, j, len, sourceResource;
		filteredResources = [];
		for (j = 0, len = sourceResources.length; j < len; j++) {
			sourceResource = sourceResources[j];
			if (sourceResource.children.length) {
				filteredChildren = _filterResourcesWithEvents(sourceResource.children, resourceIdHits);
				if (filteredChildren.length || resourceIdHits[sourceResource.id]) {
					filteredResource = createObject(sourceResource);
					filteredResource.children = filteredChildren;
					filteredResources.push(filteredResource);
				}
			} else {
				if (resourceIdHits[sourceResource.id]) {
					filteredResources.push(sourceResource);
				}
			}
		}
		return filteredResources;
	};


	/*
For vertical resource view.
For views that rely on grids that don't render their resources and dates together.
 */

	VertResourceViewMixin = $.extend({}, ResourceViewMixin, {
		executeResourcesRender: function(resources) {
			this.setResourcesOnGrids(resources);
			if (this.isDateRendered) {
				return this.requestDateRender().then((function(_this) {
					return function() {
						return _this.reportResourcesRender();
					};
				})(this));
			} else {
				return Promise.resolve();
			}
		},
		executeResourcesUnrender: function(teardownOptions) {
			if (teardownOptions == null) {
				teardownOptions = {};
			}
			this.unsetResourcesOnGrids();
			if (this.isDateRendered && !teardownOptions.skipRerender) {
				return this.requestDateRender().then((function(_this) {
					return function() {
						return _this.reportResourcesUnrender();
					};
				})(this));
			} else {
				this.reportResourcesUnrender();
				return Promise.resolve();
			}
		},
		executeDateRender: function(date) {
			return View.prototype.executeDateRender.apply(this, arguments).then((function(_this) {
				return function() {
					if (_this.isResourcesSet) {
						return _this.reportResourcesRender();
					}
				};
			})(this));
		},
		executeDateUnrender: function(date) {
			return View.prototype.executeDateUnrender.apply(this, arguments).then((function(_this) {
				return function() {
					if (_this.isResourcesSet) {
						return _this.reportResourcesUnrender();
					}
				};
			})(this));
		},
		setResourcesOnGrids: function(resources) {},
		unsetResourcesOnGrids: function() {}
	});

	ResourceGridMixin = {
		allowCrossResource: true,
		eventRangeToSpans: function(range, event) {
			var j, len, resourceId, resourceIds, results;
			resourceIds = this.view.calendar.getEventResourceIds(event);
			if (resourceIds.length) {
				results = [];
				for (j = 0, len = resourceIds.length; j < len; j++) {
					resourceId = resourceIds[j];
					results.push($.extend({}, range, {
						resourceId: resourceId
					}));
				}
				return results;
			} else if (FC.isBgEvent(event)) {
				return Grid.prototype.eventRangeToSpans.apply(this, arguments);
			} else {
				return [];
			}
		},
		fabricateHelperEvent: function(eventLocation, seg) {
			var event;
			event = Grid.prototype.fabricateHelperEvent.apply(this, arguments);
			this.view.calendar.setEventResourceId(event, eventLocation.resourceId);
			return event;
		},
		computeEventDrop: function(startSpan, endSpan, event) {
			var dropLocation;
			if (this.view.isEventStartEditable(event)) {
				dropLocation = Grid.prototype.computeEventDrop.apply(this, arguments);
			} else {
				dropLocation = FC.pluckEventDateProps(event);
			}
			if (dropLocation) {
				if (this.view.isEventResourceEditable(event)) {
					dropLocation.resourceId = endSpan.resourceId;
				} else {
					dropLocation.resourceId = startSpan.resourceId;
				}
			}
			return dropLocation;
		},
		computeExternalDrop: function(span, meta) {
			var dropLocation;
			dropLocation = Grid.prototype.computeExternalDrop.apply(this, arguments);
			if (dropLocation) {
				dropLocation.resourceId = span.resourceId;
			}
			return dropLocation;
		},
		computeEventResize: function(type, startSpan, endSpan, event) {
			var resizeLocation;
			if (!this.allowCrossResource && startSpan.resourceId !== endSpan.resourceId) {
				return;
			}
			resizeLocation = Grid.prototype.computeEventResize.apply(this, arguments);
			if (resizeLocation) {
				resizeLocation.resourceId = startSpan.resourceId;
			}
			return resizeLocation;
		},
		computeSelectionSpan: function(startSpan, endSpan) {
			var selectionSpan;
			if (!this.allowCrossResource && startSpan.resourceId !== endSpan.resourceId) {
				return;
			}
			selectionSpan = Grid.prototype.computeSelectionSpan.apply(this, arguments);
			if (selectionSpan) {
				selectionSpan.resourceId = startSpan.resourceId;
			}
			return selectionSpan;
		}
	};


	/*
Requirements:
- must be a Grid
- grid must have a view that's a ResourceView
- DayTableMixin must already be mixed in
 */

	ResourceDayTableMixin = {
		flattenedResources: null,
		resourceCnt: 0,
		datesAboveResources: false,
		allowCrossResource: false,
		setResources: function(resources) {
			this.flattenedResources = this.flattenResources(resources);
			this.resourceCnt = this.flattenedResources.length;
			return this.updateDayTableCols();
		},
		unsetResources: function() {
			this.flattenedResources = null;
			this.resourceCnt = 0;
			return this.updateDayTableCols();
		},
		flattenResources: function(resources) {
			var orderSpecs, orderVal, res, sortFunc;
			orderVal = this.view.opt('resourceOrder');
			if (orderVal) {
				orderSpecs = parseFieldSpecs(orderVal);
				sortFunc = function(a, b) {
					return compareByFieldSpecs(a, b, orderSpecs);
				};
			} else {
				sortFunc = null;
			}
			res = [];
			this.accumulateResources(resources, sortFunc, res);
			return res;
		},
		accumulateResources: function(resources, sortFunc, res) {
			var j, len, resource, results, sortedResources;
			if (sortFunc) {
				sortedResources = resources.slice(0);
				sortedResources.sort(sortFunc);
			} else {
				sortedResources = resources;
			}
			results = [];
			for (j = 0, len = sortedResources.length; j < len; j++) {
				resource = sortedResources[j];
				res.push(resource);
				results.push(this.accumulateResources(resource.children, sortFunc, res));
			}
			return results;
		},
		updateDayTableCols: function() {
			this.datesAboveResources = this.view.opt('groupByDateAndResource');
			return FC.DayTableMixin.updateDayTableCols.call(this);
		},
		computeColCnt: function() {
			return (this.resourceCnt || 1) * this.daysPerRow;
		},
		getColDayIndex: function(col) {
			if (this.isRTL) {
				col = this.colCnt - 1 - col;
			}
			if (this.datesAboveResources) {
				return Math.floor(col / (this.resourceCnt || 1));
			} else {
				return col % this.daysPerRow;
			}
		},
		getColResource: function(col) {
			return this.flattenedResources[this.getColResourceIndex(col)];
		},
		getColResourceIndex: function(col) {
			if (this.isRTL) {
				col = this.colCnt - 1 - col;
			}
			if (this.datesAboveResources) {
				return col % (this.resourceCnt || 1);
			} else {
				return Math.floor(col / this.daysPerRow);
			}
		},
		indicesToCol: function(resourceIndex, dayIndex) {
			var col;
			col = this.datesAboveResources ? dayIndex * (this.resourceCnt || 1) + resourceIndex : resourceIndex * this.daysPerRow + dayIndex;
			if (this.isRTL) {
				col = this.colCnt - 1 - col;
			}
			return col;
		},
		renderHeadTrHtml: function() {
			if (!this.resourceCnt) {
				return FC.DayTableMixin.renderHeadTrHtml.call(this);
			} else {
				if (this.daysPerRow > 1) {
					if (this.datesAboveResources) {
						return this.renderHeadDateAndResourceHtml();
					} else {
						return this.renderHeadResourceAndDateHtml();
					}
				} else {
					return this.renderHeadResourceHtml();
				}
			}
		},
		renderHeadResourceHtml: function() {
			var j, len, ref, resource, resourceHtmls;
			resourceHtmls = [];
			ref = this.flattenedResources;
			for (j = 0, len = ref.length; j < len; j++) {
				resource = ref[j];
				resourceHtmls.push(this.renderHeadResourceCellHtml(resource));
			}
			return this.wrapTr(resourceHtmls, 'renderHeadIntroHtml');
		},
		renderHeadResourceAndDateHtml: function() {
			var date, dateHtmls, dayIndex, j, k, len, ref, ref1, resource, resourceHtmls;
			resourceHtmls = [];
			dateHtmls = [];
			ref = this.flattenedResources;
			for (j = 0, len = ref.length; j < len; j++) {
				resource = ref[j];
				resourceHtmls.push(this.renderHeadResourceCellHtml(resource, null, this.daysPerRow));
				for (dayIndex = k = 0, ref1 = this.daysPerRow; k < ref1; dayIndex = k += 1) {
					date = this.dayDates[dayIndex].clone();
					dateHtmls.push(this.renderHeadResourceDateCellHtml(date, resource));
				}
			}
			return this.wrapTr(resourceHtmls, 'renderHeadIntroHtml') + this.wrapTr(dateHtmls, 'renderHeadIntroHtml');
		},
		renderHeadDateAndResourceHtml: function() {
			var date, dateHtmls, dayIndex, j, k, len, ref, ref1, resource, resourceHtmls;
			dateHtmls = [];
			resourceHtmls = [];
			for (dayIndex = j = 0, ref = this.daysPerRow; j < ref; dayIndex = j += 1) {
				date = this.dayDates[dayIndex].clone();
				dateHtmls.push(this.renderHeadDateCellHtml(date, this.resourceCnt));
				ref1 = this.flattenedResources;
				for (k = 0, len = ref1.length; k < len; k++) {
					resource = ref1[k];
					resourceHtmls.push(this.renderHeadResourceCellHtml(resource, date));
				}
			}
			return this.wrapTr(dateHtmls, 'renderHeadIntroHtml') + this.wrapTr(resourceHtmls, 'renderHeadIntroHtml');
		},
		renderHeadResourceCellHtml: function(resource, date, colspan) {
			return '<th class="fc-resource-cell"' + ' data-resource-id="' + resource.id + '"' + (date ? ' data-date="' + date.format('YYYY-MM-DD') + '"' : '') + (colspan > 1 ? ' colspan="' + colspan + '"' : '') + '>' + htmlEscape(this.view.getResourceText(resource)) + '</th>';
		},
		renderHeadResourceDateCellHtml: function(date, resource, colspan) {
			return this.renderHeadDateCellHtml(date, colspan, 'data-resource-id="' + resource.id + '"');
		},
		processHeadResourceEls: function(containerEl) {
			return containerEl.find('.fc-resource-cell').each((function(_this) {
				return function(col, node) {
					var resource;
					if (_this.datesAboveResources) {
						resource = _this.getColResource(col);
					} else {
						resource = _this.flattenedResources[_this.isRTL ? _this.flattenedResources.length - 1 - col : col];
					}
					return _this.view.publiclyTrigger('resourceRender', resource, resource, $(node), $());
				};
			})(this));
		},
		renderBgCellsHtml: function(row) {
			var col, date, htmls, j, ref, resource;
			if (!this.resourceCnt) {
				return FC.DayTableMixin.renderBgCellsHtml.call(this, row);
			} else {
				htmls = [];
				for (col = j = 0, ref = this.colCnt; j < ref; col = j += 1) {
					date = this.getCellDate(row, col);
					resource = this.getColResource(col);
					htmls.push(this.renderResourceBgCellHtml(date, resource));
				}
				return htmls.join('');
			}
		},
		renderResourceBgCellHtml: function(date, resource) {
			return this.renderBgCellHtml(date, 'data-resource-id="' + resource.id + '"');
		},
		wrapTr: function(cellHtmls, introMethodName) {
			if (this.isRTL) {
				cellHtmls.reverse();
				return '<tr>' + cellHtmls.join('') + this[introMethodName]() + '</tr>';
			} else {
				return '<tr>' + this[introMethodName]() + cellHtmls.join('') + '</tr>';
			}
		},

		/*
  	If there are no per-resource business hour definitions, returns null.
  	Otherwise, returns a list of business hours segs for *every* resource.
   */
		computePerResourceBusinessHourSegs: function(wholeDay) {
			var allSegs, anyCustomBusinessHours, businessHours, event, events, j, k, l, len, len1, len2, ref, ref1, resource, segs;
			if (this.flattenedResources) {
				anyCustomBusinessHours = false;
				ref = this.flattenedResources;
				for (j = 0, len = ref.length; j < len; j++) {
					resource = ref[j];
					if (resource.businessHours) {
						anyCustomBusinessHours = true;
					}
				}
				if (anyCustomBusinessHours) {
					allSegs = [];
					ref1 = this.flattenedResources;
					for (k = 0, len1 = ref1.length; k < len1; k++) {
						resource = ref1[k];
						businessHours = resource.businessHours || this.view.calendar.options.businessHours;
						events = this.buildBusinessHourEvents(wholeDay, businessHours);
						for (l = 0, len2 = events.length; l < len2; l++) {
							event = events[l];
							event.resourceId = resource.id;
						}
						segs = this.eventsToSegs(events);
						allSegs.push.apply(allSegs, segs);
					}
					return allSegs;
				}
			}
			return null;
		}
	};

	ResourceDayGrid = (function(superClass) {
		extend(ResourceDayGrid, superClass);

		function ResourceDayGrid() {
			return ResourceDayGrid.__super__.constructor.apply(this, arguments);
		}

		ResourceDayGrid.mixin(ResourceGridMixin);

		ResourceDayGrid.mixin(ResourceDayTableMixin);

		ResourceDayGrid.prototype.getHitSpan = function(hit) {
			var span;
			span = ResourceDayGrid.__super__.getHitSpan.apply(this, arguments);
			if (this.resourceCnt) {
				span.resourceId = this.getColResource(hit.col).id;
			}
			return span;
		};

		ResourceDayGrid.prototype.spanToSegs = function(span) {
			var copy, genericSegs, j, k, l, len, len1, ref, resourceCnt, resourceIndex, resourceObj, resourceSegs, seg;
			resourceCnt = this.resourceCnt;
			genericSegs = this.datesAboveResources ? this.sliceRangeByDay(span) : this.sliceRangeByRow(span);
			if (!resourceCnt) {
				for (j = 0, len = genericSegs.length; j < len; j++) {
					seg = genericSegs[j];
					if (this.isRTL) {
						seg.leftCol = seg.lastRowDayIndex;
						seg.rightCol = seg.firstRowDayIndex;
					} else {
						seg.leftCol = seg.firstRowDayIndex;
						seg.rightCol = seg.lastRowDayIndex;
					}
				}
				return genericSegs;
			} else {
				resourceSegs = [];
				for (k = 0, len1 = genericSegs.length; k < len1; k++) {
					seg = genericSegs[k];
					for (resourceIndex = l = 0, ref = resourceCnt; l < ref; resourceIndex = l += 1) {
						resourceObj = this.flattenedResources[resourceIndex];
						if (!span.resourceId || span.resourceId === resourceObj.id) {
							copy = $.extend({}, seg);
							copy.resource = resourceObj;
							if (this.isRTL) {
								copy.leftCol = this.indicesToCol(resourceIndex, seg.lastRowDayIndex);
								copy.rightCol = this.indicesToCol(resourceIndex, seg.firstRowDayIndex);
							} else {
								copy.leftCol = this.indicesToCol(resourceIndex, seg.firstRowDayIndex);
								copy.rightCol = this.indicesToCol(resourceIndex, seg.lastRowDayIndex);
							}
							resourceSegs.push(copy);
						}
					}
				}
				return resourceSegs;
			}
		};

		ResourceDayGrid.prototype.renderBusinessHours = function() {
			var segs;
			segs = this.computePerResourceBusinessHourSegs(true);
			if (segs) {
				return this.renderFill('businessHours', segs, 'bgevent');
			} else {
				return ResourceDayGrid.__super__.renderBusinessHours.apply(this, arguments);
			}
		};

		return ResourceDayGrid;

	})(FC.DayGrid);

	ResourceTimeGrid = (function(superClass) {
		extend(ResourceTimeGrid, superClass);

		function ResourceTimeGrid() {
			return ResourceTimeGrid.__super__.constructor.apply(this, arguments);
		}

		ResourceTimeGrid.mixin(ResourceGridMixin);

		ResourceTimeGrid.mixin(ResourceDayTableMixin);

		ResourceTimeGrid.prototype.getHitSpan = function(hit) {
			var span;
			span = ResourceTimeGrid.__super__.getHitSpan.apply(this, arguments);
			if (this.resourceCnt) {
				span.resourceId = this.getColResource(hit.col).id;
			}
			return span;
		};

		ResourceTimeGrid.prototype.spanToSegs = function(span) {
			var copy, genericSegs, j, k, l, len, len1, ref, resourceCnt, resourceIndex, resourceObj, resourceSegs, seg;
			resourceCnt = this.resourceCnt;
			genericSegs = this.sliceRangeByTimes(span);
			if (!resourceCnt) {
				for (j = 0, len = genericSegs.length; j < len; j++) {
					seg = genericSegs[j];
					seg.col = seg.dayIndex;
				}
				return genericSegs;
			} else {
				resourceSegs = [];
				for (k = 0, len1 = genericSegs.length; k < len1; k++) {
					seg = genericSegs[k];
					for (resourceIndex = l = 0, ref = resourceCnt; l < ref; resourceIndex = l += 1) {
						resourceObj = this.flattenedResources[resourceIndex];
						if (!span.resourceId || span.resourceId === resourceObj.id) {
							copy = $.extend({}, seg);
							copy.resource = resourceObj;
							copy.col = this.indicesToCol(resourceIndex, seg.dayIndex);
							resourceSegs.push(copy);
						}
					}
				}
				return resourceSegs;
			}
		};

		ResourceTimeGrid.prototype.renderBusinessHours = function() {
			var segs;
			segs = this.computePerResourceBusinessHourSegs(false);
			if (segs) {
				return this.renderBusinessSegs(segs);
			} else {
				return ResourceTimeGrid.__super__.renderBusinessHours.apply(this, arguments);
			}
		};

		return ResourceTimeGrid;

	})(FC.TimeGrid);

	TimelineView = (function(superClass) {
		extend(TimelineView, superClass);

		function TimelineView() {
			return TimelineView.__super__.constructor.apply(this, arguments);
		}

		TimelineView.prototype.timeGrid = null;

		TimelineView.prototype.isScrolled = false;

		TimelineView.prototype.initialize = function() {
			this.timeGrid = this.instantiateGrid();
			return this.intervalDuration = this.timeGrid.duration;
		};

		TimelineView.prototype.instantiateGrid = function() {
			return new TimelineGrid(this);
		};

		TimelineView.prototype.setRange = function(range) {
			TimelineView.__super__.setRange.apply(this, arguments);
			return this.timeGrid.setRange(range);
		};

		TimelineView.prototype.renderSkeleton = function() {
			this.el.addClass('fc-timeline');
			if (this.opt('eventOverlap') === false) {
				this.el.addClass('fc-no-overlap');
			}
			this.el.html(this.renderSkeletonHtml());
			return this.renderTimeGridSkeleton();
		};

		TimelineView.prototype.renderSkeletonHtml = function() {
			return '<table> <thead class="fc-head"> <tr> <td class="fc-time-area ' + this.widgetHeaderClass + '"></td> </tr> </thead> <tbody class="fc-body"> <tr> <td class="fc-time-area ' + this.widgetContentClass + '"></td> </tr> </tbody> </table>';
		};

		TimelineView.prototype.renderTimeGridSkeleton = function() {
			this.timeGrid.setElement(this.el.find('tbody .fc-time-area'));
			this.timeGrid.headEl = this.el.find('thead .fc-time-area');
			this.timeGrid.renderSkeleton();
			this.isScrolled = false;
			return this.timeGrid.bodyScroller.on('scroll', proxy(this, 'handleBodyScroll'));
		};

		TimelineView.prototype.handleBodyScroll = function(top, left) {
			if (top) {
				if (!this.isScrolled) {
					this.isScrolled = true;
					return this.el.addClass('fc-scrolled');
				}
			} else {
				if (this.isScrolled) {
					this.isScrolled = false;
					return this.el.removeClass('fc-scrolled');
				}
			}
		};

		TimelineView.prototype.unrenderSkeleton = function() {
			this.timeGrid.removeElement();
			this.handleBodyScroll(0);
			return TimelineView.__super__.unrenderSkeleton.apply(this, arguments);
		};

		TimelineView.prototype.renderDates = function() {
			return this.timeGrid.renderDates();
		};

		TimelineView.prototype.unrenderDates = function() {
			return this.timeGrid.unrenderDates();
		};

		TimelineView.prototype.renderBusinessHours = function() {
			return this.timeGrid.renderBusinessHours();
		};

		TimelineView.prototype.unrenderBusinessHours = function() {
			return this.timeGrid.unrenderBusinessHours();
		};

		TimelineView.prototype.getNowIndicatorUnit = function() {
			return this.timeGrid.getNowIndicatorUnit();
		};

		TimelineView.prototype.renderNowIndicator = function(date) {
			return this.timeGrid.renderNowIndicator(date);
		};

		TimelineView.prototype.unrenderNowIndicator = function() {
			return this.timeGrid.unrenderNowIndicator();
		};

		TimelineView.prototype.prepareHits = function() {
			return this.timeGrid.prepareHits();
		};

		TimelineView.prototype.releaseHits = function() {
			return this.timeGrid.releaseHits();
		};

		TimelineView.prototype.queryHit = function(leftOffset, topOffset) {
			return this.timeGrid.queryHit(leftOffset, topOffset);
		};

		TimelineView.prototype.getHitSpan = function(hit) {
			return this.timeGrid.getHitSpan(hit);
		};

		TimelineView.prototype.getHitEl = function(hit) {
			return this.timeGrid.getHitEl(hit);
		};

		TimelineView.prototype.updateWidth = function() {
			return this.timeGrid.updateWidth();
		};

		TimelineView.prototype.setHeight = function(totalHeight, isAuto) {
			var bodyHeight;
			if (isAuto) {
				bodyHeight = 'auto';
			} else {
				bodyHeight = totalHeight - this.timeGrid.headHeight() - this.queryMiscHeight();
			}
			return this.timeGrid.bodyScroller.setHeight(bodyHeight);
		};

		TimelineView.prototype.queryMiscHeight = function() {
			return this.el.outerHeight() - this.timeGrid.headScroller.el.outerHeight() - this.timeGrid.bodyScroller.el.outerHeight();
		};

		TimelineView.prototype.computeInitialScroll = function() {
			var left, scrollTime;
			left = 0;
			if (this.timeGrid.isTimeScale) {
				scrollTime = this.opt('scrollTime');
				if (scrollTime) {
					scrollTime = moment.duration(scrollTime);
					left = this.timeGrid.dateToCoord(this.start.clone().time(scrollTime));
				}
			}
			return {
				left: left,
				top: 0
			};
		};

		TimelineView.prototype.queryScroll = function() {
			return {
				left: this.timeGrid.bodyScroller.getScrollLeft(),
				top: this.timeGrid.bodyScroller.getScrollTop()
			};
		};

		TimelineView.prototype.setScroll = function(scroll) {
			this.timeGrid.headScroller.setScrollLeft(scroll.left);
			this.timeGrid.bodyScroller.setScrollLeft(scroll.left);
			return this.timeGrid.bodyScroller.setScrollTop(scroll.top);
		};

		TimelineView.prototype.renderEvents = function(events) {
			this.timeGrid.renderEvents(events);
			return this.updateWidth();
		};

		TimelineView.prototype.unrenderEvents = function() {
			this.timeGrid.unrenderEvents();
			return this.updateWidth();
		};

		TimelineView.prototype.renderDrag = function(dropLocation, seg) {
			return this.timeGrid.renderDrag(dropLocation, seg);
		};

		TimelineView.prototype.unrenderDrag = function() {
			return this.timeGrid.unrenderDrag();
		};

		TimelineView.prototype.getEventSegs = function() {
			return this.timeGrid.getEventSegs();
		};

		TimelineView.prototype.renderSelection = function(range) {
			return this.timeGrid.renderSelection(range);
		};

		TimelineView.prototype.unrenderSelection = function() {
			return this.timeGrid.unrenderSelection();
		};

		return TimelineView;

	})(View);

	cssToStr = FC.cssToStr;

	TimelineGrid = (function(superClass) {
		extend(TimelineGrid, superClass);

		TimelineGrid.prototype.slotDates = null;

		TimelineGrid.prototype.slotCnt = null;

		TimelineGrid.prototype.snapCnt = null;

		TimelineGrid.prototype.snapsPerSlot = null;

		TimelineGrid.prototype.snapDiffToIndex = null;

		TimelineGrid.prototype.snapIndexToDiff = null;

		TimelineGrid.prototype.headEl = null;

		TimelineGrid.prototype.slatContainerEl = null;

		TimelineGrid.prototype.slatEls = null;

		TimelineGrid.prototype.containerCoordCache = null;

		TimelineGrid.prototype.slatCoordCache = null;

		TimelineGrid.prototype.slatInnerCoordCache = null;

		TimelineGrid.prototype.headScroller = null;

		TimelineGrid.prototype.bodyScroller = null;

		TimelineGrid.prototype.joiner = null;

		TimelineGrid.prototype.follower = null;

		TimelineGrid.prototype.eventTitleFollower = null;

		TimelineGrid.prototype.minTime = null;

		TimelineGrid.prototype.maxTime = null;

		TimelineGrid.prototype.timeWindowMs = null;

		TimelineGrid.prototype.slotDuration = null;

		TimelineGrid.prototype.snapDuration = null;

		TimelineGrid.prototype.duration = null;

		TimelineGrid.prototype.labelInterval = null;

		TimelineGrid.prototype.headerFormats = null;

		TimelineGrid.prototype.isTimeScale = null;

		TimelineGrid.prototype.largeUnit = null;

		TimelineGrid.prototype.emphasizeWeeks = false;

		TimelineGrid.prototype.titleFollower = null;

		TimelineGrid.prototype.segContainerEl = null;

		TimelineGrid.prototype.segContainerHeight = null;

		TimelineGrid.prototype.bgSegContainerEl = null;

		TimelineGrid.prototype.helperEls = null;

		TimelineGrid.prototype.innerEl = null;

		function TimelineGrid() {
			var input;
			TimelineGrid.__super__.constructor.apply(this, arguments);
			this.initScaleProps();
			this.minTime = moment.duration(this.opt('minTime') || '00:00');
			this.maxTime = moment.duration(this.opt('maxTime') || '24:00');
			this.timeWindowMs = this.maxTime - this.minTime;
			this.snapDuration = (input = this.opt('snapDuration')) ? moment.duration(input) : this.slotDuration;
			this.minResizeDuration = this.snapDuration;
			this.snapsPerSlot = divideDurationByDuration(this.slotDuration, this.snapDuration);
			this.slotWidth = this.opt('slotWidth');
		}

		TimelineGrid.prototype.opt = function(name) {
			return this.view.opt(name);
		};

		TimelineGrid.prototype.isValidDate = function(date) {
			var ms;
			if (this.view.isHiddenDay(date)) {
				return false;
			} else if (this.isTimeScale) {
				ms = date.time() - this.minTime;
				ms = ((ms % 86400000) + 86400000) % 86400000;
				return ms < this.timeWindowMs;
			} else {
				return true;
			}
		};

		TimelineGrid.prototype.computeDisplayEventTime = function() {
			return !this.isTimeScale;
		};

		TimelineGrid.prototype.computeDisplayEventEnd = function() {
			return false;
		};

		TimelineGrid.prototype.computeEventTimeFormat = function() {
			return this.opt('extraSmallTimeFormat');
		};


		/*
  	Makes the given date consistent with isTimeScale/largeUnit,
  	so, either removes the times, ensures a time, or makes it the startOf largeUnit.
  	Strips all timezones. Returns new copy.
  	TODO: should maybe be called "normalizeRangeDate".
   */

		TimelineGrid.prototype.normalizeGridDate = function(date) {
			var normalDate;
			if (this.isTimeScale) {
				normalDate = date.clone();
				if (!normalDate.hasTime()) {
					normalDate.time(0);
				}
			} else {
				normalDate = date.clone().stripTime();
				if (this.largeUnit) {
					normalDate.startOf(this.largeUnit);
				}
			}
			return normalDate;
		};

		TimelineGrid.prototype.normalizeGridRange = function(range) {
			var adjustedEnd, normalRange;
			if (this.isTimeScale) {
				normalRange = {
					start: this.normalizeGridDate(range.start),
					end: this.normalizeGridDate(range.end)
				};
			} else {
				normalRange = this.view.computeDayRange(range);
				if (this.largeUnit) {
					normalRange.start.startOf(this.largeUnit);
					adjustedEnd = normalRange.end.clone().startOf(this.largeUnit);
					if (!adjustedEnd.isSame(normalRange.end) || !adjustedEnd.isAfter(normalRange.start)) {
						adjustedEnd.add(this.slotDuration);
					}
					normalRange.end = adjustedEnd;
				}
			}
			return normalRange;
		};

		TimelineGrid.prototype.rangeUpdated = function() {
			var date, slotDates;
			this.start = this.normalizeGridDate(this.start);
			this.end = this.normalizeGridDate(this.end);
			if (this.isTimeScale) {
				this.start.add(this.minTime);
				this.end.subtract(1, 'day').add(this.maxTime);
			}
			slotDates = [];
			date = this.start.clone();
			while (date < this.end) {
				if (this.isValidDate(date)) {
					slotDates.push(date.clone());
				}
				date.add(this.slotDuration);
			}
			this.slotDates = slotDates;
			return this.updateGridDates();
		};

		TimelineGrid.prototype.updateGridDates = function() {
			var date, snapDiff, snapDiffToIndex, snapIndex, snapIndexToDiff;
			snapIndex = -1;
			snapDiff = 0;
			snapDiffToIndex = [];
			snapIndexToDiff = [];
			date = this.start.clone();
			while (date < this.end) {
				if (this.isValidDate(date)) {
					snapIndex++;
					snapDiffToIndex.push(snapIndex);
					snapIndexToDiff.push(snapDiff);
				} else {
					snapDiffToIndex.push(snapIndex + 0.5);
				}
				date.add(this.snapDuration);
				snapDiff++;
			}
			this.snapDiffToIndex = snapDiffToIndex;
			this.snapIndexToDiff = snapIndexToDiff;
			this.snapCnt = snapIndex + 1;
			return this.slotCnt = this.snapCnt / this.snapsPerSlot;
		};

		TimelineGrid.prototype.spanToSegs = function(span) {
			var normalRange, seg;
			normalRange = this.normalizeGridRange(span);
			if (this.computeDateSnapCoverage(span.start) < this.computeDateSnapCoverage(span.end)) {
				seg = intersectRanges(normalRange, this);
				if (seg) {
					if (seg.isStart && !this.isValidDate(seg.start)) {
						seg.isStart = false;
					}
					if (seg.isEnd && seg.end && !this.isValidDate(seg.end.clone().subtract(1))) {
						seg.isEnd = false;
					}
					return [seg];
				}
			}
			return [];
		};

		TimelineGrid.prototype.prepareHits = function() {
			return this.buildCoords();
		};

		TimelineGrid.prototype.queryHit = function(leftOffset, topOffset) {
			var containerCoordCache, localSnapIndex, partial, slatCoordCache, slatIndex, slatLeft, slatRight, slatWidth, snapIndex, snapLeft, snapRight, snapsPerSlot;
			snapsPerSlot = this.snapsPerSlot;
			slatCoordCache = this.slatCoordCache;
			containerCoordCache = this.containerCoordCache;
			if (containerCoordCache.isTopInBounds(topOffset)) {
				slatIndex = slatCoordCache.getHorizontalIndex(leftOffset);
				if (slatIndex != null) {
					slatWidth = slatCoordCache.getWidth(slatIndex);
					if (this.isRTL) {
						slatRight = slatCoordCache.getRightOffset(slatIndex);
						partial = (slatRight - leftOffset) / slatWidth;
						localSnapIndex = Math.floor(partial * snapsPerSlot);
						snapIndex = slatIndex * snapsPerSlot + localSnapIndex;
						snapRight = slatRight - (localSnapIndex / snapsPerSlot) * slatWidth;
						snapLeft = snapRight - ((localSnapIndex + 1) / snapsPerSlot) * slatWidth;
					} else {
						slatLeft = slatCoordCache.getLeftOffset(slatIndex);
						partial = (leftOffset - slatLeft) / slatWidth;
						localSnapIndex = Math.floor(partial * snapsPerSlot);
						snapIndex = slatIndex * snapsPerSlot + localSnapIndex;
						snapLeft = slatLeft + (localSnapIndex / snapsPerSlot) * slatWidth;
						snapRight = slatLeft + ((localSnapIndex + 1) / snapsPerSlot) * slatWidth;
					}
					return {
						snap: snapIndex,
						component: this,
						left: snapLeft,
						right: snapRight,
						top: containerCoordCache.getTopOffset(0),
						bottom: containerCoordCache.getBottomOffset(0)
					};
				}
			}
		};

		TimelineGrid.prototype.getHitSpan = function(hit) {
			return this.getSnapRange(hit.snap);
		};

		TimelineGrid.prototype.getHitEl = function(hit) {
			return this.getSnapEl(hit.snap);
		};

		TimelineGrid.prototype.getSnapRange = function(snapIndex) {
			var end, start;
			start = this.start.clone();
			start.add(multiplyDuration(this.snapDuration, this.snapIndexToDiff[snapIndex]));
			end = start.clone().add(this.snapDuration);
			return {
				start: start,
				end: end
			};
		};

		TimelineGrid.prototype.getSnapEl = function(snapIndex) {
			return this.slatEls.eq(Math.floor(snapIndex / this.snapsPerSlot));
		};

		TimelineGrid.prototype.renderSkeleton = function() {
			this.headScroller = new ClippedScroller({
				overflowX: 'clipped-scroll',
				overflowY: 'hidden'
			});
			this.headScroller.canvas = new ScrollerCanvas();
			this.headScroller.render();
			this.headEl.append(this.headScroller.el);
			this.bodyScroller = new ClippedScroller();
			this.bodyScroller.canvas = new ScrollerCanvas();
			this.bodyScroller.render();
			this.el.append(this.bodyScroller.el);
			this.innerEl = this.bodyScroller.canvas.contentEl;
			this.slatContainerEl = $('<div class="fc-slats"/>').appendTo(this.bodyScroller.canvas.bgEl);
			this.segContainerEl = $('<div class="fc-event-container"/>').appendTo(this.bodyScroller.canvas.contentEl);
			this.bgSegContainerEl = this.bodyScroller.canvas.bgEl;
			this.containerCoordCache = new CoordCache({
				els: this.bodyScroller.canvas.el,
				isHorizontal: true,
				isVertical: true
			});
			this.joiner = new ScrollJoiner('horizontal', [this.headScroller, this.bodyScroller]);
			if (true) {
				this.follower = new ScrollFollower(this.headScroller, true);
			}
			if (true) {
				this.eventTitleFollower = new ScrollFollower(this.bodyScroller);
				this.eventTitleFollower.minTravel = 50;
				if (this.isRTL) {
					this.eventTitleFollower.containOnNaturalRight = true;
				} else {
					this.eventTitleFollower.containOnNaturalLeft = true;
				}
			}
			return TimelineGrid.__super__.renderSkeleton.apply(this, arguments);
		};

		TimelineGrid.prototype.headColEls = null;

		TimelineGrid.prototype.slatColEls = null;

		TimelineGrid.prototype.renderDates = function() {
			var date, i, j, len, ref;
			this.headScroller.canvas.contentEl.html(this.renderHeadHtml());
			this.headColEls = this.headScroller.canvas.contentEl.find('col');
			this.slatContainerEl.html(this.renderSlatHtml());
			this.slatColEls = this.slatContainerEl.find('col');
			this.slatEls = this.slatContainerEl.find('td');
			this.slatCoordCache = new CoordCache({
				els: this.slatEls,
				isHorizontal: true
			});
			this.slatInnerCoordCache = new CoordCache({
				els: this.slatEls.find('> div'),
				isHorizontal: true,
				offsetParent: this.bodyScroller.canvas.el
			});
			ref = this.slotDates;
			for (i = j = 0, len = ref.length; j < len; i = ++j) {
				date = ref[i];
				this.view.publiclyTrigger('dayRender', null, date, this.slatEls.eq(i));
			}
			if (this.follower) {
				return this.follower.setSprites(this.headEl.find('tr:not(:last-child) .fc-cell-text'));
			}
		};

		TimelineGrid.prototype.unrenderDates = function() {
			if (this.follower) {
				this.follower.clearSprites();
			}
			this.headScroller.canvas.contentEl.empty();
			this.slatContainerEl.empty();
			this.headScroller.canvas.clearWidth();
			return this.bodyScroller.canvas.clearWidth();
		};

		TimelineGrid.prototype.renderHeadHtml = function() {
			var cell, cellRows, date, format, formats, headerCellClassNames, html, i, isChrono, isLast, isSingleDay, isSuperRow, isWeekStart, j, k, l, labelInterval, leadingCell, len, len1, len2, len3, len4, len5, len6, m, n, newCell, p, prevWeekNumber, q, row, rowCells, rowUnits, slatHtml, slotCells, slotDates, text, weekNumber;
			labelInterval = this.labelInterval;
			formats = this.headerFormats;
			cellRows = (function() {
				var j, len, results;
				results = [];
				for (j = 0, len = formats.length; j < len; j++) {
					format = formats[j];
					results.push([]);
				}
				return results;
			})();
			leadingCell = null;
			prevWeekNumber = null;
			slotDates = this.slotDates;
			slotCells = [];
			rowUnits = (function() {
				var j, len, results;
				results = [];
				for (j = 0, len = formats.length; j < len; j++) {
					format = formats[j];
					results.push(FC.queryMostGranularFormatUnit(format));
				}
				return results;
			})();
			for (j = 0, len = slotDates.length; j < len; j++) {
				date = slotDates[j];
				weekNumber = date.week();
				isWeekStart = this.emphasizeWeeks && prevWeekNumber !== null && prevWeekNumber !== weekNumber;
				for (row = k = 0, len1 = formats.length; k < len1; row = ++k) {
					format = formats[row];
					rowCells = cellRows[row];
					leadingCell = rowCells[rowCells.length - 1];
					isSuperRow = formats.length > 1 && row < formats.length - 1;
					newCell = null;
					if (isSuperRow) {
						text = date.format(format);
						if (!leadingCell || leadingCell.text !== text) {
							newCell = this.buildCellObject(date, text, rowUnits[row]);
						} else {
							leadingCell.colspan += 1;
						}
					} else {
						if (!leadingCell || isInt(divideRangeByDuration(this.start, date, labelInterval))) {
							text = date.format(format);
							newCell = this.buildCellObject(date, text, rowUnits[row]);
						} else {
							leadingCell.colspan += 1;
						}
					}
					if (newCell) {
						newCell.weekStart = isWeekStart;
						rowCells.push(newCell);
					}
				}
				slotCells.push({
					weekStart: isWeekStart
				});
				prevWeekNumber = weekNumber;
			}
			isChrono = labelInterval > this.slotDuration;
			isSingleDay = this.slotDuration.as('days') === 1;
			html = '<table>';
			html += '<colgroup>';
			for (l = 0, len2 = slotDates.length; l < len2; l++) {
				date = slotDates[l];
				html += '<col/>';
			}
			html += '</colgroup>';
			html += '<tbody>';
			for (i = m = 0, len3 = cellRows.length; m < len3; i = ++m) {
				rowCells = cellRows[i];
				isLast = i === cellRows.length - 1;
				html += '<tr' + (isChrono && isLast ? ' class="fc-chrono"' : '') + '>';
				for (n = 0, len4 = rowCells.length; n < len4; n++) {
					cell = rowCells[n];
					headerCellClassNames = [this.view.widgetHeaderClass];
					if (cell.weekStart) {
						headerCellClassNames.push('fc-em-cell');
					}
					if (isSingleDay) {
						headerCellClassNames = headerCellClassNames.concat(this.getDayClasses(cell.date, true));
					}
					html += '<th class="' + headerCellClassNames.join(' ') + '"' + ' data-date="' + cell.date.format() + '"' + (cell.colspan > 1 ? ' colspan="' + cell.colspan + '"' : '') + '>' + '<div class="fc-cell-content">' + cell.spanHtml + '</div>' + '</th>';
				}
				html += '</tr>';
			}
			html += '</tbody></table>';
			slatHtml = '<table>';
			slatHtml += '<colgroup>';
			for (p = 0, len5 = slotCells.length; p < len5; p++) {
				cell = slotCells[p];
				slatHtml += '<col/>';
			}
			slatHtml += '</colgroup>';
			slatHtml += '<tbody><tr>';
			for (i = q = 0, len6 = slotCells.length; q < len6; i = ++q) {
				cell = slotCells[i];
				date = slotDates[i];
				slatHtml += this.slatCellHtml(date, cell.weekStart);
			}
			slatHtml += '</tr></tbody></table>';
			this._slatHtml = slatHtml;
			return html;
		};

		TimelineGrid.prototype.buildCellObject = function(date, text, rowUnit) {
			var spanHtml;
			date = date.clone();
			spanHtml = this.view.buildGotoAnchorHtml({
				date: date,
				type: rowUnit,
				forceOff: !rowUnit
			}, {
				'class': 'fc-cell-text'
			}, htmlEscape(text));
			return {
				text: text,
				spanHtml: spanHtml,
				date: date,
				colspan: 1
			};
		};

		TimelineGrid.prototype.renderSlatHtml = function() {
			return this._slatHtml;
		};

		TimelineGrid.prototype.slatCellHtml = function(date, isEm) {
			var classes;
			if (this.isTimeScale) {
				classes = [];
				classes.push(isInt(divideRangeByDuration(this.start, date, this.labelInterval)) ? 'fc-major' : 'fc-minor');
			} else {
				classes = this.getDayClasses(date);
				classes.push('fc-day');
			}
			classes.unshift(this.view.widgetContentClass);
			if (isEm) {
				classes.push('fc-em-cell');
			}
			return '<td class="' + classes.join(' ') + '"' + ' data-date="' + date.format() + '"' + '><div /></td>';
		};

		TimelineGrid.prototype.businessHourSegs = null;

		TimelineGrid.prototype.renderBusinessHours = function() {
			var segs;
			if (!this.largeUnit) {
				segs = this.businessHourSegs = this.buildBusinessHourSegs(!this.isTimeScale);
				return this.renderFill('businessHours', segs, 'bgevent');
			}
		};

		TimelineGrid.prototype.unrenderBusinessHours = function() {
			return this.unrenderFill('businessHours');
		};

		TimelineGrid.prototype.nowIndicatorEls = null;

		TimelineGrid.prototype.getNowIndicatorUnit = function() {
			if (this.isTimeScale) {
				return computeIntervalUnit(this.slotDuration);
			}
		};

		TimelineGrid.prototype.renderNowIndicator = function(date) {
			var coord, css, nodes;
			nodes = [];
			date = this.normalizeGridDate(date);
			if (date >= this.start && date < this.end) {
				coord = this.dateToCoord(date);
				css = this.isRTL ? {
					right: -coord
				} : {
					left: coord
				};
				nodes.push($("<div class='fc-now-indicator fc-now-indicator-arrow'></div>").css(css).appendTo(this.headScroller.canvas.el)[0]);
				nodes.push($("<div class='fc-now-indicator fc-now-indicator-line'></div>").css(css).appendTo(this.bodyScroller.canvas.el)[0]);
			}
			return this.nowIndicatorEls = $(nodes);
		};

		TimelineGrid.prototype.unrenderNowIndicator = function() {
			if (this.nowIndicatorEls) {
				this.nowIndicatorEls.remove();
				return this.nowIndicatorEls = null;
			}
		};

		TimelineGrid.prototype.explicitSlotWidth = null;

		TimelineGrid.prototype.defaultSlotWidth = null;

		TimelineGrid.prototype.updateWidth = function() {
			var availableWidth, containerMinWidth, containerWidth, isDatesRendered, nonLastSlotWidth, slotWidth;
			isDatesRendered = this.headColEls;
			if (isDatesRendered) {
				slotWidth = Math.round(this.slotWidth || (this.slotWidth = this.computeSlotWidth()));
				containerWidth = slotWidth * this.slotDates.length;
				containerMinWidth = '';
				nonLastSlotWidth = slotWidth;
				availableWidth = this.bodyScroller.getClientWidth();
				if (availableWidth > containerWidth) {
					containerMinWidth = availableWidth;
					containerWidth = '';
					nonLastSlotWidth = Math.floor(availableWidth / this.slotDates.length);
				}
			} else {
				containerWidth = '';
				containerMinWidth = '';
			}
			this.headScroller.canvas.setWidth(containerWidth);
			this.headScroller.canvas.setMinWidth(containerMinWidth);
			this.bodyScroller.canvas.setWidth(containerWidth);
			this.bodyScroller.canvas.setMinWidth(containerMinWidth);
			if (isDatesRendered) {
				this.headColEls.slice(0, -1).add(this.slatColEls.slice(0, -1)).width(nonLastSlotWidth);
			}
			this.headScroller.updateSize();
			this.bodyScroller.updateSize();
			this.joiner.update();
			if (isDatesRendered) {
				this.buildCoords();
				this.updateSegPositions();
				this.view.updateNowIndicator();
			}
			if (this.follower) {
				this.follower.update();
			}
			if (this.eventTitleFollower) {
				return this.eventTitleFollower.update();
			}
		};

		TimelineGrid.prototype.computeSlotWidth = function() {
			var headerWidth, innerEls, maxInnerWidth, minWidth, slotWidth, slotsPerLabel;
			maxInnerWidth = 0;
			innerEls = this.headEl.find('tr:last-child th .fc-cell-text');
			innerEls.each(function(i, node) {
				var innerWidth;
				innerWidth = $(node).outerWidth();
				return maxInnerWidth = Math.max(maxInnerWidth, innerWidth);
			});
			headerWidth = maxInnerWidth + 1;
			slotsPerLabel = divideDurationByDuration(this.labelInterval, this.slotDuration);
			slotWidth = Math.ceil(headerWidth / slotsPerLabel);
			minWidth = this.headColEls.eq(0).css('min-width');
			if (minWidth) {
				minWidth = parseInt(minWidth, 10);
				if (minWidth) {
					slotWidth = Math.max(slotWidth, minWidth);
				}
			}
			return slotWidth;
		};

		TimelineGrid.prototype.buildCoords = function() {
			this.containerCoordCache.build();
			this.slatCoordCache.build();
			return this.slatInnerCoordCache.build();
		};

		TimelineGrid.prototype.computeDateSnapCoverage = function(date) {
			var snapCoverage, snapDiff, snapDiffInt;
			snapDiff = divideRangeByDuration(this.start, date, this.snapDuration);
			if (snapDiff < 0) {
				return 0;
			} else if (snapDiff >= this.snapDiffToIndex.length) {
				return this.snapCnt;
			} else {
				snapDiffInt = Math.floor(snapDiff);
				snapCoverage = this.snapDiffToIndex[snapDiffInt];
				if (isInt(snapCoverage)) {
					snapCoverage += snapDiff - snapDiffInt;
				} else {
					snapCoverage = Math.ceil(snapCoverage);
				}
				return snapCoverage;
			}
		};

		TimelineGrid.prototype.dateToCoord = function(date) {
			var coordCache, partial, slotCoverage, slotIndex, snapCoverage;
			snapCoverage = this.computeDateSnapCoverage(date);
			slotCoverage = snapCoverage / this.snapsPerSlot;
			slotIndex = Math.floor(slotCoverage);
			slotIndex = Math.min(slotIndex, this.slotCnt - 1);
			partial = slotCoverage - slotIndex;
			coordCache = this.slatInnerCoordCache;
			if (this.isRTL) {
				return (coordCache.getRightPosition(slotIndex) - coordCache.getWidth(slotIndex) * partial) - this.containerCoordCache.getWidth(0);
			} else {
				return coordCache.getLeftPosition(slotIndex) + coordCache.getWidth(slotIndex) * partial;
			}
		};

		TimelineGrid.prototype.rangeToCoords = function(range) {
			if (this.isRTL) {
				return {
					right: this.dateToCoord(range.start),
					left: this.dateToCoord(range.end)
				};
			} else {
				return {
					left: this.dateToCoord(range.start),
					right: this.dateToCoord(range.end)
				};
			}
		};

		TimelineGrid.prototype.headHeight = function() {
			var table;
			table = this.headScroller.canvas.contentEl.find('table');
			return table.height.apply(table, arguments);
		};

		TimelineGrid.prototype.updateSegPositions = function() {
			var coords, j, len, seg, segs;
			segs = (this.segs || []).concat(this.businessHourSegs || []);
			for (j = 0, len = segs.length; j < len; j++) {
				seg = segs[j];
				coords = this.rangeToCoords(seg);
				seg.el.css({
					left: (seg.left = coords.left),
					right: -(seg.right = coords.right)
				});
			}
		};

		TimelineGrid.prototype.renderFgSegs = function(segs) {
			segs = this.renderFgSegEls(segs);
			this.renderFgSegsInContainers([[this, segs]]);
			this.updateSegFollowers(segs);
			return segs;
		};

		TimelineGrid.prototype.unrenderFgSegs = function() {
			this.clearSegFollowers();
			return this.unrenderFgContainers([this]);
		};

		TimelineGrid.prototype.renderFgSegsInContainers = function(pairs) {
			var container, coords, j, k, l, len, len1, len2, len3, len4, len5, len6, len7, m, n, p, q, r, ref, ref1, ref2, ref3, results, seg, segs;
			for (j = 0, len = pairs.length; j < len; j++) {
				ref = pairs[j], container = ref[0], segs = ref[1];
				for (k = 0, len1 = segs.length; k < len1; k++) {
					seg = segs[k];
					coords = this.rangeToCoords(seg);
					seg.el.css({
						left: (seg.left = coords.left),
						right: -(seg.right = coords.right)
					});
				}
			}
			for (l = 0, len2 = pairs.length; l < len2; l++) {
				ref1 = pairs[l], container = ref1[0], segs = ref1[1];
				for (m = 0, len3 = segs.length; m < len3; m++) {
					seg = segs[m];
					seg.el.appendTo(container.segContainerEl);
				}
			}
			for (n = 0, len4 = pairs.length; n < len4; n++) {
				ref2 = pairs[n], container = ref2[0], segs = ref2[1];
				for (p = 0, len5 = segs.length; p < len5; p++) {
					seg = segs[p];
					seg.height = seg.el.outerHeight(true);
				}
				this.buildSegLevels(segs);
				container.segContainerHeight = computeOffsetForSegs(segs);
			}
			results = [];
			for (q = 0, len6 = pairs.length; q < len6; q++) {
				ref3 = pairs[q], container = ref3[0], segs = ref3[1];
				for (r = 0, len7 = segs.length; r < len7; r++) {
					seg = segs[r];
					seg.el.css('top', seg.top);
				}
				results.push(container.segContainerEl.height(container.segContainerHeight));
			}
			return results;
		};

		TimelineGrid.prototype.buildSegLevels = function(segs) {
			var belowSeg, isLevelCollision, j, k, l, len, len1, len2, level, placedSeg, ref, ref1, segLevels, unplacedSeg;
			segLevels = [];
			this.sortEventSegs(segs);
			for (j = 0, len = segs.length; j < len; j++) {
				unplacedSeg = segs[j];
				unplacedSeg.above = [];
				level = 0;
				while (level < segLevels.length) {
					isLevelCollision = false;
					ref = segLevels[level];
					for (k = 0, len1 = ref.length; k < len1; k++) {
						placedSeg = ref[k];
						if (timeRowSegsCollide(unplacedSeg, placedSeg)) {
							unplacedSeg.above.push(placedSeg);
							isLevelCollision = true;
						}
					}
					if (isLevelCollision) {
						level += 1;
					} else {
						break;
					}
				}
				(segLevels[level] || (segLevels[level] = [])).push(unplacedSeg);
				level += 1;
				while (level < segLevels.length) {
					ref1 = segLevels[level];
					for (l = 0, len2 = ref1.length; l < len2; l++) {
						belowSeg = ref1[l];
						if (timeRowSegsCollide(unplacedSeg, belowSeg)) {
							belowSeg.above.push(unplacedSeg);
						}
					}
					level += 1;
				}
			}
			return segLevels;
		};

		TimelineGrid.prototype.unrenderFgContainers = function(containers) {
			var container, j, len, results;
			results = [];
			for (j = 0, len = containers.length; j < len; j++) {
				container = containers[j];
				container.segContainerEl.empty();
				container.segContainerEl.height('');
				results.push(container.segContainerHeight = null);
			}
			return results;
		};

		TimelineGrid.prototype.fgSegHtml = function(seg, disableResizing) {
			var classes, event, isDraggable, isResizableFromEnd, isResizableFromStart, timeText;
			event = seg.event;
			isDraggable = this.view.isEventDraggable(event);
			isResizableFromStart = seg.isStart && this.view.isEventResizableFromStart(event);
			isResizableFromEnd = seg.isEnd && this.view.isEventResizableFromEnd(event);
			classes = this.getSegClasses(seg, isDraggable, isResizableFromStart || isResizableFromEnd);
			classes.unshift('fc-timeline-event', 'fc-h-event');
			timeText = this.getEventTimeText(event);
			return '<a class="' + classes.join(' ') + '" style="' + cssToStr(this.getSegSkinCss(seg)) + '"' + (event.url ? ' href="' + htmlEscape(event.url) + '"' : '') + '>' + '<div class="fc-content">' + (timeText ? '<span class="fc-time">' + htmlEscape(timeText) + '</span>' : '') + '<span class="fc-title">' + (event.title ? htmlEscape(event.title) : '&nbsp;') + '</span>' + '</div>' + '<div class="fc-bg" />' + (isResizableFromStart ? '<div class="fc-resizer fc-start-resizer"></div>' : '') + (isResizableFromEnd ? '<div class="fc-resizer fc-end-resizer"></div>' : '') + '</a>';
		};

		TimelineGrid.prototype.updateSegFollowers = function(segs) {
			var j, len, seg, sprites, titleEl;
			if (this.eventTitleFollower) {
				sprites = [];
				for (j = 0, len = segs.length; j < len; j++) {
					seg = segs[j];
					titleEl = seg.el.find('.fc-title');
					if (titleEl.length) {
						sprites.push(new ScrollFollowerSprite(titleEl));
					}
				}
				return this.eventTitleFollower.setSprites(sprites);
			}
		};

		TimelineGrid.prototype.clearSegFollowers = function() {
			if (this.eventTitleFollower) {
				return this.eventTitleFollower.clearSprites();
			}
		};

		TimelineGrid.prototype.segDragStart = function() {
			TimelineGrid.__super__.segDragStart.apply(this, arguments);
			if (this.eventTitleFollower) {
				return this.eventTitleFollower.forceRelative();
			}
		};

		TimelineGrid.prototype.segDragEnd = function() {
			TimelineGrid.__super__.segDragEnd.apply(this, arguments);
			if (this.eventTitleFollower) {
				return this.eventTitleFollower.clearForce();
			}
		};

		TimelineGrid.prototype.segResizeStart = function() {
			TimelineGrid.__super__.segResizeStart.apply(this, arguments);
			if (this.eventTitleFollower) {
				return this.eventTitleFollower.forceRelative();
			}
		};

		TimelineGrid.prototype.segResizeEnd = function() {
			TimelineGrid.__super__.segResizeEnd.apply(this, arguments);
			if (this.eventTitleFollower) {
				return this.eventTitleFollower.clearForce();
			}
		};

		TimelineGrid.prototype.renderHelper = function(event, sourceSeg) {
			var segs;
			segs = this.eventToSegs(event);
			segs = this.renderFgSegEls(segs);
			return this.renderHelperSegsInContainers([[this, segs]], sourceSeg);
		};

		TimelineGrid.prototype.renderHelperSegsInContainers = function(pairs, sourceSeg) {
			var containerObj, coords, helperContainerEl, helperNodes, j, k, l, len, len1, len2, len3, m, ref, ref1, ref2, seg, segNodes, segs;
			helperNodes = [];
			segNodes = [];
			for (j = 0, len = pairs.length; j < len; j++) {
				ref = pairs[j], containerObj = ref[0], segs = ref[1];
				for (k = 0, len1 = segs.length; k < len1; k++) {
					seg = segs[k];
					coords = this.rangeToCoords(seg);
					seg.el.css({
						left: (seg.left = coords.left),
						right: -(seg.right = coords.right)
					});
					if (sourceSeg && sourceSeg.resourceId === ((ref1 = containerObj.resource) != null ? ref1.id : void 0)) {
						seg.el.css('top', sourceSeg.el.css('top'));
					} else {
						seg.el.css('top', 0);
					}
				}
			}
			for (l = 0, len2 = pairs.length; l < len2; l++) {
				ref2 = pairs[l], containerObj = ref2[0], segs = ref2[1];
				helperContainerEl = $('<div class="fc-event-container fc-helper-container"/>').appendTo(containerObj.innerEl);
				helperNodes.push(helperContainerEl[0]);
				for (m = 0, len3 = segs.length; m < len3; m++) {
					seg = segs[m];
					helperContainerEl.append(seg.el);
					segNodes.push(seg.el[0]);
				}
			}
			if (this.helperEls) {
				this.helperEls = this.helperEls.add($(helperNodes));
			} else {
				this.helperEls = $(helperNodes);
			}
			return $(segNodes);
		};

		TimelineGrid.prototype.unrenderHelper = function() {
			if (this.helperEls) {
				this.helperEls.remove();
				return this.helperEls = null;
			}
		};

		TimelineGrid.prototype.renderEventResize = function(resizeLocation, seg) {
			this.renderHighlight(this.eventToSpan(resizeLocation));
			return this.renderEventLocationHelper(resizeLocation, seg);
		};

		TimelineGrid.prototype.unrenderEventResize = function() {
			this.unrenderHighlight();
			return this.unrenderHelper();
		};

		TimelineGrid.prototype.renderFill = function(type, segs, className) {
			segs = this.renderFillSegEls(type, segs);
			this.renderFillInContainers(type, [[this, segs]], className);
			return segs;
		};

		TimelineGrid.prototype.renderFillInContainers = function(type, pairs, className) {
			var containerObj, j, len, ref, results, segs;
			results = [];
			for (j = 0, len = pairs.length; j < len; j++) {
				ref = pairs[j], containerObj = ref[0], segs = ref[1];
				results.push(this.renderFillInContainer(type, containerObj, segs, className));
			}
			return results;
		};

		TimelineGrid.prototype.renderFillInContainer = function(type, containerObj, segs, className) {
			var containerEl, coords, j, len, seg;
			if (segs.length) {
				className || (className = type.toLowerCase());
				containerEl = $('<div class="fc-' + className + '-container" />').appendTo(containerObj.bgSegContainerEl);
				for (j = 0, len = segs.length; j < len; j++) {
					seg = segs[j];
					coords = this.rangeToCoords(seg);
					seg.el.css({
						left: (seg.left = coords.left),
						right: -(seg.right = coords.right)
					});
					seg.el.appendTo(containerEl);
				}
				if (this.elsByFill[type]) {
					return this.elsByFill[type] = this.elsByFill[type].add(containerEl);
				} else {
					return this.elsByFill[type] = containerEl;
				}
			}
		};

		TimelineGrid.prototype.renderDrag = function(dropLocation, seg) {
			if (seg) {
				return this.renderEventLocationHelper(dropLocation, seg);
			} else {
				this.renderHighlight(this.eventToSpan(dropLocation));
				return null;
			}
		};

		TimelineGrid.prototype.unrenderDrag = function() {
			this.unrenderHelper();
			return this.unrenderHighlight();
		};

		return TimelineGrid;

	})(Grid);

	computeOffsetForSegs = function(segs) {
		var j, len, max, seg;
		max = 0;
		for (j = 0, len = segs.length; j < len; j++) {
			seg = segs[j];
			max = Math.max(max, computeOffsetForSeg(seg));
		}
		return max;
	};

	computeOffsetForSeg = function(seg) {
		if (seg.top == null) {
			seg.top = computeOffsetForSegs(seg.above);
		}
		return seg.top + seg.height;
	};

	timeRowSegsCollide = function(seg0, seg1) {
		return seg0.left < seg1.right && seg0.right > seg1.left;
	};

	MIN_AUTO_LABELS = 18;

	MAX_AUTO_SLOTS_PER_LABEL = 6;

	MAX_AUTO_CELLS = 200;

	MAX_CELLS = 1000;

	DEFAULT_GRID_DURATION = {
		months: 1
	};

	STOCK_SUB_DURATIONS = [
		{
			years: 1
		}, {
			months: 1
		}, {
			days: 1
		}, {
			hours: 1
		}, {
			minutes: 30
		}, {
			minutes: 15
		}, {
			minutes: 10
		}, {
			minutes: 5
		}, {
			minutes: 1
		}, {
			seconds: 30
		}, {
			seconds: 15
		}, {
			seconds: 10
		}, {
			seconds: 5
		}, {
			seconds: 1
		}, {
			milliseconds: 500
		}, {
			milliseconds: 100
		}, {
			milliseconds: 10
		}, {
			milliseconds: 1
		}
	];

	TimelineGrid.prototype.initScaleProps = function() {
		var input, slotUnit, type;
		this.labelInterval = this.queryDurationOption('slotLabelInterval');
		this.slotDuration = this.queryDurationOption('slotDuration');
		this.ensureGridDuration();
		this.validateLabelAndSlot();
		this.ensureLabelInterval();
		this.ensureSlotDuration();
		input = this.opt('slotLabelFormat');
		type = $.type(input);
		this.headerFormats = type === 'array' ? input : type === 'string' ? [input] : this.computeHeaderFormats();
		this.isTimeScale = durationHasTime(this.slotDuration);
		this.largeUnit = !this.isTimeScale ? (slotUnit = computeIntervalUnit(this.slotDuration), /year|month|week/.test(slotUnit) ? slotUnit : void 0) : void 0;
		return this.emphasizeWeeks = this.slotDuration.as('days') === 1 && this.duration.as('weeks') >= 2 && !this.opt('businessHours');

		/*
  	console.log('view duration =', @duration.humanize())
  	console.log('label interval =', @labelInterval.humanize())
  	console.log('slot duration =', @slotDuration.humanize())
  	console.log('header formats =', @headerFormats)
  	console.log('isTimeScale', @isTimeScale)
  	console.log('largeUnit', @largeUnit)
   */
	};

	TimelineGrid.prototype.queryDurationOption = function(name) {
		var dur, input;
		input = this.opt(name);
		if (input != null) {
			dur = moment.duration(input);
			if (+dur) {
				return dur;
			}
		}
	};

	TimelineGrid.prototype.validateLabelAndSlot = function() {
		var labelCnt, slotCnt, slotsPerLabel;
		if (this.labelInterval) {
			labelCnt = divideDurationByDuration(this.duration, this.labelInterval);
			if (labelCnt > MAX_CELLS) {
				FC.warn('slotLabelInterval results in too many cells');
				this.labelInterval = null;
			}
		}
		if (this.slotDuration) {
			slotCnt = divideDurationByDuration(this.duration, this.slotDuration);
			if (slotCnt > MAX_CELLS) {
				FC.warn('slotDuration results in too many cells');
				this.slotDuration = null;
			}
		}
		if (this.labelInterval && this.slotDuration) {
			slotsPerLabel = divideDurationByDuration(this.labelInterval, this.slotDuration);
			if (!isInt(slotsPerLabel) || slotsPerLabel < 1) {
				FC.warn('slotLabelInterval must be a multiple of slotDuration');
				return this.slotDuration = null;
			}
		}
	};

	TimelineGrid.prototype.ensureGridDuration = function() {
		var gridDuration, input, j, labelCnt, labelInterval;
		gridDuration = this.duration;
		if (!gridDuration) {
			gridDuration = this.view.intervalDuration;
			if (!gridDuration) {
				if (!this.labelInterval && !this.slotDuration) {
					gridDuration = moment.duration(DEFAULT_GRID_DURATION);
				} else {
					labelInterval = this.ensureLabelInterval();
					for (j = STOCK_SUB_DURATIONS.length - 1; j >= 0; j += -1) {
						input = STOCK_SUB_DURATIONS[j];
						gridDuration = moment.duration(input);
						labelCnt = divideDurationByDuration(gridDuration, labelInterval);
						if (labelCnt >= MIN_AUTO_LABELS) {
							break;
						}
					}
				}
			}
			this.duration = gridDuration;
		}
		return gridDuration;
	};

	TimelineGrid.prototype.ensureLabelInterval = function() {
		var input, j, k, labelCnt, labelInterval, len, len1, slotsPerLabel, tryLabelInterval;
		labelInterval = this.labelInterval;
		if (!labelInterval) {
			if (!this.duration && !this.slotDuration) {
				this.ensureGridDuration();
			}
			if (this.slotDuration) {
				for (j = 0, len = STOCK_SUB_DURATIONS.length; j < len; j++) {
					input = STOCK_SUB_DURATIONS[j];
					tryLabelInterval = moment.duration(input);
					slotsPerLabel = divideDurationByDuration(tryLabelInterval, this.slotDuration);
					if (isInt(slotsPerLabel) && slotsPerLabel <= MAX_AUTO_SLOTS_PER_LABEL) {
						labelInterval = tryLabelInterval;
						break;
					}
				}
				if (!labelInterval) {
					labelInterval = this.slotDuration;
				}
			} else {
				for (k = 0, len1 = STOCK_SUB_DURATIONS.length; k < len1; k++) {
					input = STOCK_SUB_DURATIONS[k];
					labelInterval = moment.duration(input);
					labelCnt = divideDurationByDuration(this.duration, labelInterval);
					if (labelCnt >= MIN_AUTO_LABELS) {
						break;
					}
				}
			}
			this.labelInterval = labelInterval;
		}
		return labelInterval;
	};

	TimelineGrid.prototype.ensureSlotDuration = function() {
		var input, j, labelInterval, len, slotCnt, slotDuration, slotsPerLabel, trySlotDuration;
		slotDuration = this.slotDuration;
		if (!slotDuration) {
			labelInterval = this.ensureLabelInterval();
			for (j = 0, len = STOCK_SUB_DURATIONS.length; j < len; j++) {
				input = STOCK_SUB_DURATIONS[j];
				trySlotDuration = moment.duration(input);
				slotsPerLabel = divideDurationByDuration(labelInterval, trySlotDuration);
				if (isInt(slotsPerLabel) && slotsPerLabel > 1 && slotsPerLabel <= MAX_AUTO_SLOTS_PER_LABEL) {
					slotDuration = trySlotDuration;
					break;
				}
			}
			if (slotDuration && this.duration) {
				slotCnt = divideDurationByDuration(this.duration, slotDuration);
				if (slotCnt > MAX_AUTO_CELLS) {
					slotDuration = null;
				}
			}
			if (!slotDuration) {
				slotDuration = labelInterval;
			}
			this.slotDuration = slotDuration;
		}
		return slotDuration;
	};

	TimelineGrid.prototype.computeHeaderFormats = function() {
		var format0, format1, format2, gridDuration, labelInterval, unit, view, weekNumbersVisible;
		view = this.view;
		gridDuration = this.duration;
		labelInterval = this.labelInterval;
		unit = computeIntervalUnit(labelInterval);
		weekNumbersVisible = this.opt('weekNumbers');
		format0 = format1 = format2 = null;
		if (unit === 'week' && !weekNumbersVisible) {
			unit = 'day';
		}
		switch (unit) {
			case 'year':
				format0 = 'YYYY';
				break;
			case 'month':
				if (gridDuration.asYears() > 1) {
					format0 = 'YYYY';
				}
				format1 = 'MMM';
				break;
			case 'week':
				if (gridDuration.asYears() > 1) {
					format0 = 'YYYY';
				}
				format1 = this.opt('shortWeekFormat');
				break;
			case 'day':
				if (gridDuration.asYears() > 1) {
					format0 = this.opt('monthYearFormat');
				} else if (gridDuration.asMonths() > 1) {
					format0 = 'MMMM';
				}
				if (weekNumbersVisible) {
					format1 = this.opt('weekFormat');
				}
				format2 = 'dd D';
				break;
			case 'hour':
				if (weekNumbersVisible) {
					format0 = this.opt('weekFormat');
				}
				if (gridDuration.asDays() > 1) {
					format1 = this.opt('dayOfMonthFormat');
				}
				format2 = this.opt('smallTimeFormat');
				break;
			case 'minute':
				if (labelInterval.asMinutes() / 60 >= MAX_AUTO_SLOTS_PER_LABEL) {
					format0 = this.opt('hourFormat');
					format1 = '[:]mm';
				} else {
					format0 = this.opt('mediumTimeFormat');
				}
				break;
			case 'second':
				if (labelInterval.asSeconds() / 60 >= MAX_AUTO_SLOTS_PER_LABEL) {
					format0 = 'LT';
					format1 = '[:]ss';
				} else {
					format0 = 'LTS';
				}
				break;
			case 'millisecond':
				format0 = 'LTS';
				format1 = '[.]SSS';
		}
		return [].concat(format0 || [], format1 || [], format2 || []);
	};

	FC.views.timeline = {
		"class": TimelineView,
		defaults: {
			eventResizableFromStart: true
		}
	};

	FC.views.timelineDay = {
		type: 'timeline',
		duration: {
			days: 1
		}
	};

	FC.views.timelineWeek = {
		type: 'timeline',
		duration: {
			weeks: 1
		}
	};

	FC.views.timelineMonth = {
		type: 'timeline',
		duration: {
			months: 1
		}
	};

	FC.views.timelineYear = {
		type: 'timeline',
		duration: {
			years: 1
		}
	};

	ResourceTimelineView = (function(superClass) {
		extend(ResourceTimelineView, superClass);

		function ResourceTimelineView() {
			return ResourceTimelineView.__super__.constructor.apply(this, arguments);
		}

		ResourceTimelineView.mixin(ResourceViewMixin);

		ResourceTimelineView.prototype.canRenderSpecificResources = true;

		ResourceTimelineView.prototype.resourceGrid = null;

		ResourceTimelineView.prototype.tbodyHash = null;

		ResourceTimelineView.prototype.joiner = null;

		ResourceTimelineView.prototype.dividerEls = null;

		ResourceTimelineView.prototype.superHeaderText = null;

		ResourceTimelineView.prototype.isVGrouping = null;

		ResourceTimelineView.prototype.isHGrouping = null;

		ResourceTimelineView.prototype.groupSpecs = null;

		ResourceTimelineView.prototype.colSpecs = null;

		ResourceTimelineView.prototype.orderSpecs = null;

		ResourceTimelineView.prototype.rowHierarchy = null;

		ResourceTimelineView.prototype.resourceRowHash = null;

		ResourceTimelineView.prototype.nestingCnt = 0;

		ResourceTimelineView.prototype.isNesting = null;

		ResourceTimelineView.prototype.dividerWidth = null;

		ResourceTimelineView.prototype.initialize = function() {
			ResourceTimelineView.__super__.initialize.apply(this, arguments);
			this.processResourceOptions();
			this.resourceGrid = new Spreadsheet(this);
			this.rowHierarchy = new RowParent(this);
			return this.resourceRowHash = {};
		};

		ResourceTimelineView.prototype.instantiateGrid = function() {
			return new ResourceTimelineGrid(this);
		};

		ResourceTimelineView.prototype.processResourceOptions = function() {
			var allColSpecs, allOrderSpecs, colSpec, defaultLabelText, groupColSpecs, groupSpec, groupSpecs, hGroupField, isGroup, isHGrouping, isVGrouping, j, k, l, labelText, len, len1, len2, orderSpec, plainColSpecs, plainOrderSpecs, superHeaderText;
			allColSpecs = this.opt('resourceColumns') || [];
			labelText = this.opt('resourceLabelText');
			defaultLabelText = 'Resources';
			superHeaderText = null;
			if (!allColSpecs.length) {
				allColSpecs.push({
					labelText: labelText || defaultLabelText,
					text: this.getResourceTextFunc()
				});
			} else {
				superHeaderText = labelText;
			}
			plainColSpecs = [];
			groupColSpecs = [];
			groupSpecs = [];
			isVGrouping = false;
			isHGrouping = false;
			for (j = 0, len = allColSpecs.length; j < len; j++) {
				colSpec = allColSpecs[j];
				if (colSpec.group) {
					groupColSpecs.push(colSpec);
				} else {
					plainColSpecs.push(colSpec);
				}
			}
			plainColSpecs[0].isMain = true;
			if (groupColSpecs.length) {
				groupSpecs = groupColSpecs;
				isVGrouping = true;
			} else {
				hGroupField = this.opt('resourceGroupField');
				if (hGroupField) {
					isHGrouping = true;
					groupSpecs.push({
						field: hGroupField,
						text: this.opt('resourceGroupText'),
						render: this.opt('resourceGroupRender')
					});
				}
			}
			allOrderSpecs = parseFieldSpecs(this.opt('resourceOrder'));
			plainOrderSpecs = [];
			for (k = 0, len1 = allOrderSpecs.length; k < len1; k++) {
				orderSpec = allOrderSpecs[k];
				isGroup = false;
				for (l = 0, len2 = groupSpecs.length; l < len2; l++) {
					groupSpec = groupSpecs[l];
					if (groupSpec.field === orderSpec.field) {
						groupSpec.order = orderSpec.order;
						isGroup = true;
						break;
					}
				}
				if (!isGroup) {
					plainOrderSpecs.push(orderSpec);
				}
			}
			this.superHeaderText = superHeaderText;
			this.isVGrouping = isVGrouping;
			this.isHGrouping = isHGrouping;
			this.groupSpecs = groupSpecs;
			this.colSpecs = groupColSpecs.concat(plainColSpecs);
			return this.orderSpecs = plainOrderSpecs;
		};

		ResourceTimelineView.prototype.renderSkeleton = function() {
			ResourceTimelineView.__super__.renderSkeleton.apply(this, arguments);
			this.renderResourceGridSkeleton();
			this.tbodyHash = {
				spreadsheet: this.resourceGrid.tbodyEl,
				event: this.timeGrid.tbodyEl
			};
			this.joiner = new ScrollJoiner('vertical', [this.resourceGrid.bodyScroller, this.timeGrid.bodyScroller]);
			return this.initDividerMoving();
		};

		ResourceTimelineView.prototype.renderSkeletonHtml = function() {
			return '<table> <thead class="fc-head"> <tr> <td class="fc-resource-area ' + this.widgetHeaderClass + '"></td> <td class="fc-divider fc-col-resizer ' + this.widgetHeaderClass + '"></td> <td class="fc-time-area ' + this.widgetHeaderClass + '"></td> </tr> </thead> <tbody class="fc-body"> <tr> <td class="fc-resource-area ' + this.widgetContentClass + '"></td> <td class="fc-divider fc-col-resizer ' + this.widgetHeaderClass + '"></td> <td class="fc-time-area ' + this.widgetContentClass + '"></td> </tr> </tbody> </table>';
		};

		ResourceTimelineView.prototype.renderResourceGridSkeleton = function() {
			this.resourceGrid.el = this.el.find('tbody .fc-resource-area');
			this.resourceGrid.headEl = this.el.find('thead .fc-resource-area');
			return this.resourceGrid.renderSkeleton();
		};

		ResourceTimelineView.prototype.initDividerMoving = function() {
			var ref;
			this.dividerEls = this.el.find('.fc-divider');
			this.dividerWidth = (ref = this.opt('resourceAreaWidth')) != null ? ref : this.resourceGrid.tableWidth;
			if (this.dividerWidth != null) {
				this.positionDivider(this.dividerWidth);
			}
			return this.dividerEls.on('mousedown', (function(_this) {
				return function(ev) {
					return _this.dividerMousedown(ev);
				};
			})(this));
		};

		ResourceTimelineView.prototype.dividerMousedown = function(ev) {
			var dragListener, isRTL, maxWidth, minWidth, origWidth;
			isRTL = this.opt('isRTL');
			minWidth = 30;
			maxWidth = this.el.width() - 30;
			origWidth = this.getNaturalDividerWidth();
			dragListener = new DragListener({
				dragStart: (function(_this) {
					return function() {
						return _this.dividerEls.addClass('fc-active');
					};
				})(this),
				drag: (function(_this) {
					return function(dx, dy) {
						var width;
						if (isRTL) {
							width = origWidth - dx;
						} else {
							width = origWidth + dx;
						}
						width = Math.max(width, minWidth);
						width = Math.min(width, maxWidth);
						_this.dividerWidth = width;
						_this.positionDivider(width);
						return _this.updateWidth();
					};
				})(this),
				dragEnd: (function(_this) {
					return function() {
						return _this.dividerEls.removeClass('fc-active');
					};
				})(this)
			});
			return dragListener.startInteraction(ev);
		};

		ResourceTimelineView.prototype.getNaturalDividerWidth = function() {
			return this.el.find('.fc-resource-area').width();
		};

		ResourceTimelineView.prototype.positionDivider = function(w) {
			return this.el.find('.fc-resource-area').width(w);
		};

		ResourceTimelineView.prototype.renderEvents = function(events) {
			this.timeGrid.renderEvents(events);
			this.syncRowHeights();
			return this.updateWidth();
		};

		ResourceTimelineView.prototype.unrenderEvents = function() {
			this.timeGrid.unrenderEvents();
			this.syncRowHeights();
			return this.updateWidth();
		};

		ResourceTimelineView.prototype.updateWidth = function() {
			ResourceTimelineView.__super__.updateWidth.apply(this, arguments);
			this.resourceGrid.updateWidth();
			this.joiner.update();
			if (this.cellFollower) {
				return this.cellFollower.update();
			}
		};

		ResourceTimelineView.prototype.updateHeight = function(isResize) {
			ResourceTimelineView.__super__.updateHeight.apply(this, arguments);
			if (isResize) {
				return this.syncRowHeights();
			}
		};

		ResourceTimelineView.prototype.setHeight = function(totalHeight, isAuto) {
			var bodyHeight, headHeight;
			headHeight = this.syncHeadHeights();
			if (isAuto) {
				bodyHeight = 'auto';
			} else {
				bodyHeight = totalHeight - headHeight - this.queryMiscHeight();
			}
			this.timeGrid.bodyScroller.setHeight(bodyHeight);
			return this.resourceGrid.bodyScroller.setHeight(bodyHeight);
		};

		ResourceTimelineView.prototype.queryMiscHeight = function() {
			return this.el.outerHeight() - Math.max(this.resourceGrid.headScroller.el.outerHeight(), this.timeGrid.headScroller.el.outerHeight()) - Math.max(this.resourceGrid.bodyScroller.el.outerHeight(), this.timeGrid.bodyScroller.el.outerHeight());
		};

		ResourceTimelineView.prototype.syncHeadHeights = function() {
			var headHeight;
			this.resourceGrid.headHeight('auto');
			this.timeGrid.headHeight('auto');
			headHeight = Math.max(this.resourceGrid.headHeight(), this.timeGrid.headHeight());
			this.resourceGrid.headHeight(headHeight);
			this.timeGrid.headHeight(headHeight);
			return headHeight;
		};

		ResourceTimelineView.prototype.renderResources = function(resources) {
			var j, len, resource;
			this.batchRows();
			for (j = 0, len = resources.length; j < len; j++) {
				resource = resources[j];
				this.insertResource(resource);
			}
			this.rowHierarchy.show();
			this.unbatchRows();
			return this.reinitializeCellFollowers();
		};

		ResourceTimelineView.prototype.unrenderResources = function() {
			this.batchRows();
			this.rowHierarchy.removeChildren();
			this.unbatchRows();
			return this.reinitializeCellFollowers();
		};


		/*
  	TODO: the scenario where there were previously unassociated events that are now
  	 attached to this resource. should render those events immediately.

  	Responsible for rendering the new resource
   */

		ResourceTimelineView.prototype.renderResource = function(resource) {
			this.insertResource(resource);
			return this.reinitializeCellFollowers();
		};

		ResourceTimelineView.prototype.unrenderResource = function(resource) {
			var row;
			row = this.getResourceRow(resource.id);
			if (row) {
				this.batchRows();
				row.remove();
				this.unbatchRows();
				return this.reinitializeCellFollowers();
			}
		};

		ResourceTimelineView.prototype.cellFollower = null;

		ResourceTimelineView.prototype.reinitializeCellFollowers = function() {
			var cellContent, j, len, nodes, ref, row;
			if (this.cellFollower) {
				this.cellFollower.clearSprites();
			}
			this.cellFollower = new ScrollFollower(this.resourceGrid.bodyScroller, true);
			this.cellFollower.isHFollowing = false;
			this.cellFollower.isVFollowing = true;
			nodes = [];
			ref = this.rowHierarchy.getNodes();
			for (j = 0, len = ref.length; j < len; j++) {
				row = ref[j];
				if (row instanceof VRowGroup) {
					if (row.groupTd) {
						cellContent = row.groupTd.find('.fc-cell-content');
						if (cellContent.length) {
							nodes.push(cellContent[0]);
						}
					}
				}
			}
			return this.cellFollower.setSprites($(nodes));
		};

		ResourceTimelineView.prototype.insertResource = function(resource, parentResourceRow) {
			var childResource, j, len, parentId, ref, results, row;
			row = new ResourceRow(this, resource);
			if (parentResourceRow == null) {
				parentId = resource.parentId;
				if (parentId) {
					parentResourceRow = this.getResourceRow(parentId);
				}
			}
			if (parentResourceRow) {
				this.insertRowAsChild(row, parentResourceRow);
			} else {
				this.insertRow(row);
			}
			ref = resource.children;
			results = [];
			for (j = 0, len = ref.length; j < len; j++) {
				childResource = ref[j];
				results.push(this.insertResource(childResource, row));
			}
			return results;
		};

		ResourceTimelineView.prototype.insertRow = function(row, parent, groupSpecs) {
			var group;
			if (parent == null) {
				parent = this.rowHierarchy;
			}
			if (groupSpecs == null) {
				groupSpecs = this.groupSpecs;
			}
			if (groupSpecs.length) {
				group = this.ensureResourceGroup(row, parent, groupSpecs[0]);
				if (group instanceof HRowGroup) {
					return this.insertRowAsChild(row, group);
				} else {
					return this.insertRow(row, group, groupSpecs.slice(1));
				}
			} else {
				return this.insertRowAsChild(row, parent);
			}
		};

		ResourceTimelineView.prototype.insertRowAsChild = function(row, parent) {
			return parent.addChild(row, this.computeChildRowPosition(row, parent));
		};

		ResourceTimelineView.prototype.computeChildRowPosition = function(child, parent) {
			var cmp, i, j, len, ref, sibling;
			if (this.orderSpecs.length) {
				ref = parent.children;
				for (i = j = 0, len = ref.length; j < len; i = ++j) {
					sibling = ref[i];
					cmp = this.compareResources(sibling.resource || {}, child.resource || {});
					if (cmp > 0) {
						return i;
					}
				}
			}
			return null;
		};

		ResourceTimelineView.prototype.compareResources = function(a, b) {
			return compareByFieldSpecs(a, b, this.orderSpecs);
		};

		ResourceTimelineView.prototype.ensureResourceGroup = function(row, parent, spec) {
			var cmp, group, groupValue, i, j, k, len, len1, ref, ref1, testGroup;
			groupValue = (row.resource || {})[spec.field];
			group = null;
			if (spec.order) {
				ref = parent.children;
				for (i = j = 0, len = ref.length; j < len; i = ++j) {
					testGroup = ref[i];
					cmp = flexibleCompare(testGroup.groupValue, groupValue) * spec.order;
					if (cmp === 0) {
						group = testGroup;
						break;
					} else if (cmp > 0) {
						break;
					}
				}
			} else {
				ref1 = parent.children;
				for (i = k = 0, len1 = ref1.length; k < len1; i = ++k) {
					testGroup = ref1[i];
					if (testGroup.groupValue === groupValue) {
						group = testGroup;
						break;
					}
				}
			}
			if (!group) {
				if (this.isVGrouping) {
					group = new VRowGroup(this, spec, groupValue);
				} else {
					group = new HRowGroup(this, spec, groupValue);
				}
				parent.addChild(group, i);
			}
			return group;
		};

		ResourceTimelineView.prototype.pairSegsWithRows = function(segs) {
			var j, len, pair, pairs, pairsById, resourceId, rowObj, seg;
			pairs = [];
			pairsById = {};
			for (j = 0, len = segs.length; j < len; j++) {
				seg = segs[j];
				resourceId = seg.resourceId;
				if (resourceId) {
					rowObj = this.getResourceRow(resourceId);
					if (rowObj) {
						pair = pairsById[resourceId];
						if (!pair) {
							pair = [rowObj, []];
							pairs.push(pair);
							pairsById[resourceId] = pair;
						}
						pair[1].push(seg);
					}
				}
			}
			return pairs;
		};

		ResourceTimelineView.prototype.rowAdded = function(row) {
			var isNesting, wasNesting;
			if (row instanceof ResourceRow) {
				this.resourceRowHash[row.resource.id] = row;
				this.timeGrid.assignRowBusinessHourSegs(row);
			}
			wasNesting = this.isNesting;
			isNesting = Boolean(this.nestingCnt += row.depth ? 1 : 0);
			if (wasNesting !== isNesting) {
				this.el.toggleClass('fc-nested', isNesting);
				this.el.toggleClass('fc-flat', !isNesting);
			}
			return this.isNesting = isNesting;
		};

		ResourceTimelineView.prototype.rowRemoved = function(row) {
			var isNesting, wasNesting;
			if (row instanceof ResourceRow) {
				delete this.resourceRowHash[row.resource.id];
				this.timeGrid.destroyRowBusinessHourSegs(row);
			}
			wasNesting = this.isNesting;
			isNesting = Boolean(this.nestingCnt -= row.depth ? 1 : 0);
			if (wasNesting !== isNesting) {
				this.el.toggleClass('fc-nested', isNesting);
				this.el.toggleClass('fc-flat', !isNesting);
			}
			return this.isNesting = isNesting;
		};

		ResourceTimelineView.prototype.batchRowDepth = 0;

		ResourceTimelineView.prototype.shownRowBatch = null;

		ResourceTimelineView.prototype.hiddenRowBatch = null;

		ResourceTimelineView.prototype.batchRows = function() {
			if (!(this.batchRowDepth++)) {
				this.shownRowBatch = [];
				return this.hiddenRowBatch = [];
			}
		};

		ResourceTimelineView.prototype.unbatchRows = function() {
			if (!(--this.batchRowDepth)) {
				if (this.hiddenRowBatch.length) {
					this.rowsHidden(this.hiddenRowBatch);
				}
				if (this.shownRowBatch.length) {
					this.rowsShown(this.shownRowBatch);
				}
				this.hiddenRowBatch = null;
				return this.shownRowBatch = null;
			}
		};

		ResourceTimelineView.prototype.rowShown = function(row) {
			if (this.shownRowBatch) {
				return this.shownRowBatch.push(row);
			} else {
				return this.rowsShown([row]);
			}
		};

		ResourceTimelineView.prototype.rowHidden = function(row) {
			if (this.hiddenRowBatch) {
				return this.hiddenRowBatch.push(row);
			} else {
				return this.rowsHidden([row]);
			}
		};

		ResourceTimelineView.prototype.rowsShown = function(rows) {
			this.syncRowHeights(rows);
			return this.updateWidth();
		};

		ResourceTimelineView.prototype.rowsHidden = function(rows) {
			return this.updateWidth();
		};

		ResourceTimelineView.prototype.syncRowHeights = function(visibleRows, safe) {
			var h, h1, h2, i, innerHeights, j, k, len, len1, row;
			if (safe == null) {
				safe = false;
			}
			if (visibleRows == null) {
				visibleRows = this.getVisibleRows();
			}
			for (j = 0, len = visibleRows.length; j < len; j++) {
				row = visibleRows[j];
				row.setTrInnerHeight('');
			}
			innerHeights = (function() {
				var k, len1, results;
				results = [];
				for (k = 0, len1 = visibleRows.length; k < len1; k++) {
					row = visibleRows[k];
					h = row.getMaxTrInnerHeight();
					if (safe) {
						h += h % 2;
					}
					results.push(h);
				}
				return results;
			})();
			for (i = k = 0, len1 = visibleRows.length; k < len1; i = ++k) {
				row = visibleRows[i];
				row.setTrInnerHeight(innerHeights[i]);
			}
			if (!safe) {
				h1 = this.resourceGrid.tbodyEl.height();
				h2 = this.timeGrid.tbodyEl.height();
				if (Math.abs(h1 - h2) > 1) {
					return this.syncRowHeights(visibleRows, true);
				}
			}
		};

		ResourceTimelineView.prototype.getVisibleRows = function() {
			var j, len, ref, results, row;
			ref = this.rowHierarchy.getRows();
			results = [];
			for (j = 0, len = ref.length; j < len; j++) {
				row = ref[j];
				if (row.isShown) {
					results.push(row);
				}
			}
			return results;
		};

		ResourceTimelineView.prototype.getEventRows = function() {
			var j, len, ref, results, row;
			ref = this.rowHierarchy.getRows();
			results = [];
			for (j = 0, len = ref.length; j < len; j++) {
				row = ref[j];
				if (row instanceof EventRow) {
					results.push(row);
				}
			}
			return results;
		};

		ResourceTimelineView.prototype.getResourceRow = function(resourceId) {
			return this.resourceRowHash[resourceId];
		};

		ResourceTimelineView.prototype.queryScroll = function() {
			var el, elBottom, j, len, ref, rowObj, scroll, scrollerTop;
			scroll = ResourceTimelineView.__super__.queryScroll.apply(this, arguments);
			scrollerTop = this.timeGrid.bodyScroller.scrollEl.offset().top;
			ref = this.getVisibleRows();
			for (j = 0, len = ref.length; j < len; j++) {
				rowObj = ref[j];
				if (rowObj.resource) {
					el = rowObj.getTr('event');
					elBottom = el.offset().top + el.outerHeight();
					if (elBottom > scrollerTop) {
						scroll.resourceId = rowObj.resource.id;
						scroll.bottom = elBottom - scrollerTop;
						break;
					}
				}
			}
			return scroll;
		};

		ResourceTimelineView.prototype.setScroll = function(scroll) {
			var el, elBottom, innerTop, row;
			if (scroll.resourceId) {
				row = this.getResourceRow(scroll.resourceId);
				if (row) {
					el = row.getTr('event');
					if (el) {
						innerTop = this.timeGrid.bodyScroller.canvas.el.offset().top;
						elBottom = el.offset().top + el.outerHeight();
						scroll.top = elBottom - scroll.bottom - innerTop;
					}
				}
			}
			ResourceTimelineView.__super__.setScroll.call(this, scroll);
			return this.resourceGrid.bodyScroller.setScrollTop(scroll.top);
		};

		ResourceTimelineView.prototype.scrollToResource = function(resource) {
			var el, innerTop, row, scrollTop;
			row = this.getResourceRow(resource.id);
			if (row) {
				el = row.getTr('event');
				if (el) {
					innerTop = this.timeGrid.bodyScroller.canvas.el.offset().top;
					scrollTop = el.offset().top - innerTop;
					this.timeGrid.bodyScroller.setScrollTop(scrollTop);
					return this.resourceGrid.bodyScroller.setScrollTop(scrollTop);
				}
			}
		};

		return ResourceTimelineView;

	})(TimelineView);

	ResourceTimelineGrid = (function(superClass) {
		extend(ResourceTimelineGrid, superClass);

		function ResourceTimelineGrid() {
			return ResourceTimelineGrid.__super__.constructor.apply(this, arguments);
		}

		ResourceTimelineGrid.mixin(ResourceGridMixin);

		ResourceTimelineGrid.prototype.eventRows = null;

		ResourceTimelineGrid.prototype.shownEventRows = null;

		ResourceTimelineGrid.prototype.tbodyEl = null;

		ResourceTimelineGrid.prototype.rowCoordCache = null;

		ResourceTimelineGrid.prototype.spanToSegs = function(span) {
			var calendar, j, len, resourceId, seg, segs;
			segs = ResourceTimelineGrid.__super__.spanToSegs.apply(this, arguments);
			calendar = this.view.calendar;
			resourceId = span.resourceId;
			if (resourceId) {
				for (j = 0, len = segs.length; j < len; j++) {
					seg = segs[j];
					seg.resource = calendar.getResourceById(resourceId);
					seg.resourceId = resourceId;
				}
			}
			return segs;
		};

		ResourceTimelineGrid.prototype.prepareHits = function() {
			var row, trArray;
			ResourceTimelineGrid.__super__.prepareHits.apply(this, arguments);
			this.eventRows = this.view.getEventRows();
			this.shownEventRows = (function() {
				var j, len, ref, results;
				ref = this.eventRows;
				results = [];
				for (j = 0, len = ref.length; j < len; j++) {
					row = ref[j];
					if (row.isShown) {
						results.push(row);
					}
				}
				return results;
			}).call(this);
			trArray = (function() {
				var j, len, ref, results;
				ref = this.shownEventRows;
				results = [];
				for (j = 0, len = ref.length; j < len; j++) {
					row = ref[j];
					results.push(row.getTr('event')[0]);
				}
				return results;
			}).call(this);
			this.rowCoordCache = new CoordCache({
				els: trArray,
				isVertical: true
			});
			return this.rowCoordCache.build();
		};

		ResourceTimelineGrid.prototype.releaseHits = function() {
			ResourceTimelineGrid.__super__.releaseHits.apply(this, arguments);
			this.eventRows = null;
			this.shownEventRows = null;
			return this.rowCoordCache.clear();
		};

		ResourceTimelineGrid.prototype.queryHit = function(leftOffset, topOffset) {
			var rowIndex, simpleHit;
			simpleHit = ResourceTimelineGrid.__super__.queryHit.apply(this, arguments);
			if (simpleHit) {
				rowIndex = this.rowCoordCache.getVerticalIndex(topOffset);
				if (rowIndex != null) {
					return {
						resourceId: this.shownEventRows[rowIndex].resource.id,
						snap: simpleHit.snap,
						component: this,
						left: simpleHit.left,
						right: simpleHit.right,
						top: this.rowCoordCache.getTopOffset(rowIndex),
						bottom: this.rowCoordCache.getBottomOffset(rowIndex)
					};
				}
			}
		};

		ResourceTimelineGrid.prototype.getHitSpan = function(hit) {
			var span;
			span = this.getSnapRange(hit.snap);
			span.resourceId = hit.resourceId;
			return span;
		};

		ResourceTimelineGrid.prototype.getHitEl = function(hit) {
			return this.getSnapEl(hit.snap);
		};

		ResourceTimelineGrid.prototype.renderSkeleton = function() {
			var rowContainerEl;
			ResourceTimelineGrid.__super__.renderSkeleton.apply(this, arguments);
			this.segContainerEl.remove();
			this.segContainerEl = null;
			rowContainerEl = $('<div class="fc-rows"><table><tbody/></table></div>').appendTo(this.bodyScroller.canvas.contentEl);
			return this.tbodyEl = rowContainerEl.find('tbody');
		};

		ResourceTimelineGrid.prototype.renderFgSegs = function(segs) {
			var containerObj, containerSegs, j, len, pair, pairs, visiblePairs;
			segs = this.renderFgSegEls(segs);
			pairs = this.view.pairSegsWithRows(segs);
			visiblePairs = [];
			for (j = 0, len = pairs.length; j < len; j++) {
				pair = pairs[j];
				containerObj = pair[0], containerSegs = pair[1];
				containerObj.fgSegs = containerSegs;
				if (containerObj.isShown) {
					containerObj.isSegsRendered = true;
					visiblePairs.push(pair);
				}
			}
			this.renderFgSegsInContainers(visiblePairs);
			this.updateSegFollowers(segs);
			return segs;
		};

		ResourceTimelineGrid.prototype.unrenderFgSegs = function() {
			var eventRow, eventRows, j, len;
			this.clearSegFollowers();
			eventRows = this.view.getEventRows();
			for (j = 0, len = eventRows.length; j < len; j++) {
				eventRow = eventRows[j];
				eventRow.fgSegs = null;
				eventRow.isSegsRendered = false;
			}
			return this.unrenderFgContainers(eventRows);
		};

		ResourceTimelineGrid.prototype.rowCntWithCustomBusinessHours = 0;

		ResourceTimelineGrid.prototype.renderBusinessHours = function() {
			if (this.rowCntWithCustomBusinessHours) {
				return this.ensureIndividualBusinessHours();
			} else {
				return ResourceTimelineGrid.__super__.renderBusinessHours.apply(this, arguments);
			}
		};

		ResourceTimelineGrid.prototype.unrenderBusinessHours = function() {
			if (this.rowCntWithCustomBusinessHours) {
				return this.clearIndividualBusinessHours();
			} else {
				return ResourceTimelineGrid.__super__.unrenderBusinessHours.apply(this, arguments);
			}
		};


		/*
  	Ensures that all rows have their individual business hours DISPLAYED.
   */

		ResourceTimelineGrid.prototype.ensureIndividualBusinessHours = function() {
			var j, len, ref, results, row;
			ref = this.view.getEventRows();
			results = [];
			for (j = 0, len = ref.length; j < len; j++) {
				row = ref[j];
				if (this.view.isDateSet && !row.businessHourSegs) {
					this.populateRowBusinessHoursSegs(row);
				}
				if (row.isShown) {
					results.push(row.ensureBusinessHourSegsRendered());
				} else {
					results.push(void 0);
				}
			}
			return results;
		};


		/*
  	Ensures that all rows have their individual business hours CLEARED.
   */

		ResourceTimelineGrid.prototype.clearIndividualBusinessHours = function() {
			var j, len, ref, results, row;
			ref = this.view.getEventRows();
			results = [];
			for (j = 0, len = ref.length; j < len; j++) {
				row = ref[j];
				results.push(row.clearBusinessHourSegs());
			}
			return results;
		};


		/*
  	Called when a row has been added to the tree data structure, but before it's rendered.
  	Computes and assigns business hour data *if necessary*. To be rendered soon after.
   */

		ResourceTimelineGrid.prototype.assignRowBusinessHourSegs = function(row) {
			if (row.resource.businessHours) {
				if (!this.rowCntWithCustomBusinessHours) {
					TimelineGrid.prototype.unrenderBusinessHours.call(this);
					this.ensureIndividualBusinessHours();
				}
				this.rowCntWithCustomBusinessHours += 1;
			}
			if (this.view.isDateSet && this.rowCntWithCustomBusinessHours) {
				return this.populateRowBusinessHoursSegs(row);
			}
		};


		/*
  	Called when a row has been removed from the tree data structure.
  	Unrenders the row's segs and, if necessary, forces businessHours back to generic rendering.
   */

		ResourceTimelineGrid.prototype.destroyRowBusinessHourSegs = function(row) {
			row.clearBusinessHourSegs();
			if (row.resource.businessHours) {
				this.rowCntWithCustomBusinessHours -= 1;
				if (!this.rowCntWithCustomBusinessHours) {
					this.clearIndividualBusinessHours();
					return TimelineGrid.prototype.renderBusinessHours.call(this);
				}
			}
		};


		/*
  	Compute and assign to row.businessHourSegs unconditionally
   */

		ResourceTimelineGrid.prototype.populateRowBusinessHoursSegs = function(row) {
			var businessHourSegs, businessHours, businessHoursEvents;
			businessHours = row.resource.businessHours || this.view.opt('businessHours');
			businessHoursEvents = this.view.calendar.computeBusinessHourEvents(!this.isTimeScale, businessHours);
			businessHourSegs = this.eventsToSegs(businessHoursEvents);
			businessHourSegs = this.renderFillSegEls('businessHours', businessHourSegs);
			row.businessHourSegs = businessHourSegs;
		};

		ResourceTimelineGrid.prototype.renderFill = function(type, segs, className) {
			var j, k, len, len1, nonResourceSegs, pair, pairs, resourceSegs, rowObj, rowSegs, seg, visiblePairs;
			segs = this.renderFillSegEls(type, segs);
			resourceSegs = [];
			nonResourceSegs = [];
			for (j = 0, len = segs.length; j < len; j++) {
				seg = segs[j];
				if (seg.resourceId) {
					resourceSegs.push(seg);
				} else {
					nonResourceSegs.push(seg);
				}
			}
			pairs = this.view.pairSegsWithRows(resourceSegs);
			visiblePairs = [];
			for (k = 0, len1 = pairs.length; k < len1; k++) {
				pair = pairs[k];
				rowObj = pair[0], rowSegs = pair[1];
				if (type === 'bgEvent') {
					rowObj.bgSegs = rowSegs;
				}
				if (rowObj.isShown) {
					visiblePairs.push(pair);
				}
			}
			if (nonResourceSegs.length) {
				visiblePairs.unshift([this, nonResourceSegs]);
			}
			this.renderFillInContainers(type, visiblePairs, className);
			return segs;
		};

		ResourceTimelineGrid.prototype.renderHelper = function(event, sourceSeg) {
			var pairs, segs;
			segs = this.eventToSegs(event);
			segs = this.renderFgSegEls(segs);
			pairs = this.view.pairSegsWithRows(segs);
			return this.renderHelperSegsInContainers(pairs, sourceSeg);
		};

		return ResourceTimelineGrid;

	})(TimelineGrid);

	COL_MIN_WIDTH = 30;

	Spreadsheet = (function() {
		Spreadsheet.prototype.view = null;

		Spreadsheet.prototype.headEl = null;

		Spreadsheet.prototype.el = null;

		Spreadsheet.prototype.tbodyEl = null;

		Spreadsheet.prototype.headScroller = null;

		Spreadsheet.prototype.bodyScroller = null;

		Spreadsheet.prototype.joiner = null;

		function Spreadsheet(view1) {
			var colSpec;
			this.view = view1;
			this.isRTL = this.view.opt('isRTL');
			this.givenColWidths = this.colWidths = (function() {
				var j, len, ref, results;
				ref = this.view.colSpecs;
				results = [];
				for (j = 0, len = ref.length; j < len; j++) {
					colSpec = ref[j];
					results.push(colSpec.width);
				}
				return results;
			}).call(this);
		}

		Spreadsheet.prototype.colGroupHtml = '';

		Spreadsheet.prototype.headTable = null;

		Spreadsheet.prototype.headColEls = null;

		Spreadsheet.prototype.headCellEls = null;

		Spreadsheet.prototype.bodyColEls = null;

		Spreadsheet.prototype.bodyTable = null;

		Spreadsheet.prototype.renderSkeleton = function() {
			this.headScroller = new ClippedScroller({
				overflowX: 'clipped-scroll',
				overflowY: 'hidden'
			});
			this.headScroller.canvas = new ScrollerCanvas();
			this.headScroller.render();
			this.headScroller.canvas.contentEl.html(this.renderHeadHtml());
			this.headEl.append(this.headScroller.el);
			this.bodyScroller = new ClippedScroller({
				overflowY: 'clipped-scroll'
			});
			this.bodyScroller.canvas = new ScrollerCanvas();
			this.bodyScroller.render();
			this.bodyScroller.canvas.contentEl.html('<div class="fc-rows"><table>' + this.colGroupHtml + '<tbody/></table></div>');
			this.tbodyEl = this.bodyScroller.canvas.contentEl.find('tbody');
			this.el.append(this.bodyScroller.el);
			this.joiner = new ScrollJoiner('horizontal', [this.headScroller, this.bodyScroller]);
			this.headTable = this.headEl.find('table');
			this.headColEls = this.headEl.find('col');
			this.headCellEls = this.headScroller.canvas.contentEl.find('tr:last-child th');
			this.bodyColEls = this.el.find('col');
			this.bodyTable = this.el.find('table');
			this.colMinWidths = this.computeColMinWidths();
			this.applyColWidths();
			return this.initColResizing();
		};

		Spreadsheet.prototype.renderHeadHtml = function() {
			var colGroupHtml, colSpecs, html, i, isLast, isMainCol, j, k, len, len1, o;
			colSpecs = this.view.colSpecs;
			html = '<table>';
			colGroupHtml = '<colgroup>';
			for (j = 0, len = colSpecs.length; j < len; j++) {
				o = colSpecs[j];
				if (o.isMain) {
					colGroupHtml += '<col class="fc-main-col"/>';
				} else {
					colGroupHtml += '<col/>';
				}
			}
			colGroupHtml += '</colgroup>';
			this.colGroupHtml = colGroupHtml;
			html += colGroupHtml;
			html += '<tbody>';
			if (this.view.superHeaderText) {
				html += '<tr class="fc-super">' + '<th class="' + this.view.widgetHeaderClass + '" colspan="' + colSpecs.length + '">' + '<div class="fc-cell-content">' + '<span class="fc-cell-text">' + htmlEscape(this.view.superHeaderText) + '</span>' + '</div>' + '</th>' + '</tr>';
			}
			html += '<tr>';
			isMainCol = true;
			for (i = k = 0, len1 = colSpecs.length; k < len1; i = ++k) {
				o = colSpecs[i];
				isLast = i === colSpecs.length - 1;
				html += '<th class="' + this.view.widgetHeaderClass + '">' + '<div>' + '<div class="fc-cell-content">' + (o.isMain ? '<span class="fc-expander-space">' + '<span class="fc-icon"></span>' + '</span>' : '') + '<span class="fc-cell-text">' + htmlEscape(o.labelText || '') + '</span>' + '</div>' + (!isLast ? '<div class="fc-col-resizer"></div>' : '') + '</div>' + '</th>';
			}
			html += '</tr>';
			html += '</tbody></table>';
			return html;
		};

		Spreadsheet.prototype.givenColWidths = null;

		Spreadsheet.prototype.colWidths = null;

		Spreadsheet.prototype.colMinWidths = null;

		Spreadsheet.prototype.tableWidth = null;

		Spreadsheet.prototype.tableMinWidth = null;

		Spreadsheet.prototype.initColResizing = function() {
			return this.headEl.find('th .fc-col-resizer').each((function(_this) {
				return function(i, resizerEl) {
					resizerEl = $(resizerEl);
					return resizerEl.on('mousedown', function(ev) {
						return _this.colResizeMousedown(i, ev, resizerEl);
					});
				};
			})(this));
		};

		Spreadsheet.prototype.colResizeMousedown = function(i, ev, resizerEl) {
			var colWidths, dragListener, minWidth, origColWidth;
			colWidths = this.colWidths = this.queryColWidths();
			colWidths.pop();
			colWidths.push(null);
			origColWidth = colWidths[i];
			minWidth = Math.min(this.colMinWidths[i], COL_MIN_WIDTH);
			dragListener = new DragListener({
				dragStart: (function(_this) {
					return function() {
						return resizerEl.addClass('fc-active');
					};
				})(this),
				drag: (function(_this) {
					return function(dx, dy) {
						var width;
						width = origColWidth + (_this.isRTL ? -dx : dx);
						width = Math.max(width, minWidth);
						colWidths[i] = width;
						return _this.applyColWidths();
					};
				})(this),
				dragEnd: (function(_this) {
					return function() {
						return resizerEl.removeClass('fc-active');
					};
				})(this)
			});
			return dragListener.startInteraction(ev);
		};

		Spreadsheet.prototype.applyColWidths = function() {
			var allNumbers, anyPercentages, colMinWidths, colWidth, colWidths, cssWidth, cssWidths, defaultCssWidth, i, j, k, l, len, len1, len2, tableMinWidth, total;
			colMinWidths = this.colMinWidths;
			colWidths = this.colWidths;
			allNumbers = true;
			anyPercentages = false;
			total = 0;
			for (j = 0, len = colWidths.length; j < len; j++) {
				colWidth = colWidths[j];
				if (typeof colWidth === 'number') {
					total += colWidth;
				} else {
					allNumbers = false;
					if (colWidth) {
						anyPercentages = true;
					}
				}
			}
			defaultCssWidth = anyPercentages && !this.view.isHGrouping ? 'auto' : '';
			cssWidths = (function() {
				var k, len1, results;
				results = [];
				for (i = k = 0, len1 = colWidths.length; k < len1; i = ++k) {
					colWidth = colWidths[i];
					results.push(colWidth != null ? colWidth : defaultCssWidth);
				}
				return results;
			})();
			tableMinWidth = 0;
			for (i = k = 0, len1 = cssWidths.length; k < len1; i = ++k) {
				cssWidth = cssWidths[i];
				tableMinWidth += typeof cssWidth === 'number' ? cssWidth : colMinWidths[i];
			}
			for (i = l = 0, len2 = cssWidths.length; l < len2; i = ++l) {
				cssWidth = cssWidths[i];
				this.headColEls.eq(i).width(cssWidth);
				this.bodyColEls.eq(i).width(cssWidth);
			}
			this.headScroller.canvas.setMinWidth(tableMinWidth);
			this.bodyScroller.canvas.setMinWidth(tableMinWidth);
			this.tableMinWidth = tableMinWidth;
			return this.tableWidth = allNumbers ? total : void 0;
		};

		Spreadsheet.prototype.computeColMinWidths = function() {
			var i, j, len, ref, results, width;
			ref = this.givenColWidths;
			results = [];
			for (i = j = 0, len = ref.length; j < len; i = ++j) {
				width = ref[i];
				if (typeof width === 'number') {
					results.push(width);
				} else {
					results.push(parseInt(this.headColEls.eq(i).css('min-width')) || COL_MIN_WIDTH);
				}
			}
			return results;
		};

		Spreadsheet.prototype.queryColWidths = function() {
			var j, len, node, ref, results;
			ref = this.headCellEls;
			results = [];
			for (j = 0, len = ref.length; j < len; j++) {
				node = ref[j];
				results.push($(node).outerWidth());
			}
			return results;
		};

		Spreadsheet.prototype.updateWidth = function() {
			this.headScroller.updateSize();
			this.bodyScroller.updateSize();
			this.joiner.update();
			if (this.follower) {
				return this.follower.update();
			}
		};

		Spreadsheet.prototype.headHeight = function() {
			var table;
			table = this.headScroller.canvas.contentEl.find('table');
			return table.height.apply(table, arguments);
		};

		return Spreadsheet;

	})();


	/*
An abstract node in a row-hierarchy tree.
May be a self-contained single row, a row with subrows,
OR a grouping of rows without its own distinct row.
 */

	RowParent = (function() {
		RowParent.prototype.view = null;

		RowParent.prototype.parent = null;

		RowParent.prototype.prevSibling = null;

		RowParent.prototype.children = null;

		RowParent.prototype.depth = 0;

		RowParent.prototype.hasOwnRow = false;

		RowParent.prototype.trHash = null;

		RowParent.prototype.trs = null;

		RowParent.prototype.isRendered = false;

		RowParent.prototype.isExpanded = true;

		RowParent.prototype.isShown = false;

		function RowParent(view1) {
			this.view = view1;
			this.children = [];
			this.trHash = {};
			this.trs = $();
		}


		/*
  	Adds the given node as a child.
  	Will be inserted at the `index`. If not given, will be appended to the end.
   */

		RowParent.prototype.addChild = function(child, index) {
			var children, j, len, node, ref;
			child.remove();
			children = this.children;
			if (index != null) {
				children.splice(index, 0, child);
			} else {
				index = children.length;
				children.push(child);
			}
			child.prevSibling = index > 0 ? children[index - 1] : null;
			if (index < children.length - 1) {
				children[index + 1].prevSibling = child;
			}
			child.parent = this;
			child.depth = this.depth + (this.hasOwnRow ? 1 : 0);
			ref = child.getNodes();
			for (j = 0, len = ref.length; j < len; j++) {
				node = ref[j];
				node.added();
			}
			if (this.isShown && this.isExpanded) {
				return child.show();
			}
		};


		/*
  	Removes the given child from the node. Assumes it is a direct child.
  	If not a direct child, returns false and nothing happens.
  	Unrenders the child and triggers handlers.
   */

		RowParent.prototype.removeChild = function(child) {
			var children, i, isFound, j, k, len, len1, ref, row, testChild;
			children = this.children;
			isFound = false;
			for (i = j = 0, len = children.length; j < len; i = ++j) {
				testChild = children[i];
				if (testChild === child) {
					isFound = true;
					break;
				}
			}
			if (!isFound) {
				return false;
			} else {
				if (i < children.length - 1) {
					children[i + 1].prevSibling = child.prevSibling;
				}
				children.splice(i, 1);
				child.recursivelyUnrender();
				ref = child.getNodes();
				for (k = 0, len1 = ref.length; k < len1; k++) {
					row = ref[k];
					row.removed();
				}
				child.parent = null;
				child.prevSibling = null;
				return child;
			}
		};


		/*
  	Removes all of the node's children from the hierarchy. Unrenders them and triggers callbacks.
  	NOTE: batchRows/unbatchRows should probably be called before this happens :(
   */

		RowParent.prototype.removeChildren = function() {
			var child, j, k, len, len1, ref, ref1;
			ref = this.children;
			for (j = 0, len = ref.length; j < len; j++) {
				child = ref[j];
				child.recursivelyUnrender();
			}
			ref1 = this.getDescendants();
			for (k = 0, len1 = ref1.length; k < len1; k++) {
				child = ref1[k];
				child.removed();
			}
			return this.children = [];
		};


		/*
  	Removes this node from its parent
   */

		RowParent.prototype.remove = function() {
			if (this.parent) {
				return this.parent.removeChild(this);
			}
		};


		/*
  	Gets the last direct child node
   */

		RowParent.prototype.getLastChild = function() {
			var children;
			children = this.children;
			return children[children.length - 1];
		};


		/*
  	Walks backward in the hierarchy to find the previous row leaf node.
  	When looking at the hierarchy in a flat linear fashion, this is the revealed row just before the current.
   */

		RowParent.prototype.getPrevRow = function() {
			var lastChild, node;
			node = this;
			while (node) {
				if (node.prevSibling) {
					node = node.prevSibling;
					while ((lastChild = node.getLastChild())) {
						node = lastChild;
					}
				} else {
					node = node.parent;
				}
				if (node && node.hasOwnRow && node.isShown) {
					return node;
				}
			}
			return null;
		};


		/*
  	Returns the first node in the subtree that has a revealed row
   */

		RowParent.prototype.getLeadingRow = function() {
			if (this.hasOwnRow) {
				return this;
			} else if (this.isExpanded && this.children.length) {
				return this.children[0].getLeadingRow();
			}
		};


		/*
  	Generates a flat array containing all the row-nodes of the subtree. Descendants + self
   */

		RowParent.prototype.getRows = function(batchArray) {
			var child, j, len, ref;
			if (batchArray == null) {
				batchArray = [];
			}
			if (this.hasOwnRow) {
				batchArray.push(this);
			}
			ref = this.children;
			for (j = 0, len = ref.length; j < len; j++) {
				child = ref[j];
				child.getRows(batchArray);
			}
			return batchArray;
		};


		/*
  	Generates a flat array containing all the nodes (row/non-row) of the subtree. Descendants + self
   */

		RowParent.prototype.getNodes = function(batchArray) {
			var child, j, len, ref;
			if (batchArray == null) {
				batchArray = [];
			}
			batchArray.push(this);
			ref = this.children;
			for (j = 0, len = ref.length; j < len; j++) {
				child = ref[j];
				child.getNodes(batchArray);
			}
			return batchArray;
		};


		/*
  	Generates a flat array containing all the descendant nodes the current node
   */

		RowParent.prototype.getDescendants = function() {
			var batchArray, child, j, len, ref;
			batchArray = [];
			ref = this.children;
			for (j = 0, len = ref.length; j < len; j++) {
				child = ref[j];
				child.getNodes(batchArray);
			}
			return batchArray;
		};


		/*
  	Builds and populates the TRs for each row type. Inserts them into the DOM.
  	Does this only for this single row. Not recursive. If not a row (hasOwnRow=false), does not render anything.
  	PRECONDITION: assumes the parent has already been rendered.
   */

		RowParent.prototype.render = function() {
			var prevRow, ref, renderMethodName, tbody, tr, trNodes, type;
			this.trHash = {};
			trNodes = [];
			if (this.hasOwnRow) {
				prevRow = this.getPrevRow();
				ref = this.view.tbodyHash;
				for (type in ref) {
					tbody = ref[type];
					tr = $('<tr/>');
					this.trHash[type] = tr;
					trNodes.push(tr[0]);
					renderMethodName = 'render' + capitaliseFirstLetter(type) + 'Content';
					if (this[renderMethodName]) {
						this[renderMethodName](tr);
					}
					if (prevRow) {
						prevRow.trHash[type].after(tr);
					} else {
						tbody.prepend(tr);
					}
				}
			}
			this.trs = $(trNodes).on('click', '.fc-expander', proxy(this, 'toggleExpanded'));
			return this.isRendered = true;
		};


		/*
  	Unpopulates and removes all of this row's TRs from the DOM. Only for this single row. Not recursive.
  	Will trigger "hidden".
   */

		RowParent.prototype.unrender = function() {
			var ref, tr, type, unrenderMethodName;
			if (this.isRendered) {
				ref = this.trHash;
				for (type in ref) {
					tr = ref[type];
					unrenderMethodName = 'unrender' + capitaliseFirstLetter(type) + 'Content';
					if (this[unrenderMethodName]) {
						this[unrenderMethodName](tr);
					}
				}
				this.trHash = {};
				this.trs.remove();
				this.trs = $();
				this.isRendered = false;
				this.isShown = false;
				return this.hidden();
			}
		};


		/*
  	Like unrender(), but does it for this row AND all descendants.
  	NOTE: batchRows/unbatchRows should probably be called before this happens :(
   */

		RowParent.prototype.recursivelyUnrender = function() {
			var child, j, len, ref, results;
			this.unrender();
			ref = this.children;
			results = [];
			for (j = 0, len = ref.length; j < len; j++) {
				child = ref[j];
				results.push(child.recursivelyUnrender());
			}
			return results;
		};


		/*
  	A simple getter for retrieving a TR jQuery object of a certain row type
   */

		RowParent.prototype.getTr = function(type) {
			return this.trHash[type];
		};


		/*
  	Renders this row if not already rendered, making sure it is visible.
  	Also renders descendants of this subtree, based on whether they are expanded or not.
  	NOTE: If called externally, batchRows/unbatchRows should probably be called before this happens :(
   */

		RowParent.prototype.show = function() {
			var child, j, len, ref, results;
			if (!this.isShown) {
				if (!this.isRendered) {
					this.render();
				} else {
					this.trs.css('display', '');
				}
				if (this.ensureSegsRendered) {
					this.ensureSegsRendered();
				}
				if (this.isExpanded) {
					this.indicateExpanded();
				} else {
					this.indicateCollapsed();
				}
				this.isShown = true;
				this.shown();
				if (this.isExpanded) {
					ref = this.children;
					results = [];
					for (j = 0, len = ref.length; j < len; j++) {
						child = ref[j];
						results.push(child.show());
					}
					return results;
				}
			}
		};


		/*
  	Temporarily hides this node's TRs (if applicable) as well as all nodes in the subtree
   */

		RowParent.prototype.hide = function() {
			var child, j, len, ref, results;
			if (this.isShown) {
				if (this.isRendered) {
					this.trs.hide();
				}
				this.isShown = false;
				this.hidden();
				if (this.isExpanded) {
					ref = this.children;
					results = [];
					for (j = 0, len = ref.length; j < len; j++) {
						child = ref[j];
						results.push(child.hide());
					}
					return results;
				}
			}
		};


		/*
  	Reveals this node's children if they have not already been revealed. Changes any expander icon.
   */

		RowParent.prototype.expand = function() {
			var child, j, len, ref;
			if (!this.isExpanded) {
				this.isExpanded = true;
				this.indicateExpanded();
				this.view.batchRows();
				ref = this.children;
				for (j = 0, len = ref.length; j < len; j++) {
					child = ref[j];
					child.show();
				}
				this.view.unbatchRows();
				return this.animateExpand();
			}
		};


		/*
  	Hides this node's children if they are not already hidden. Changes any expander icon.
   */

		RowParent.prototype.collapse = function() {
			var child, j, len, ref;
			if (this.isExpanded) {
				this.isExpanded = false;
				this.indicateCollapsed();
				this.view.batchRows();
				ref = this.children;
				for (j = 0, len = ref.length; j < len; j++) {
					child = ref[j];
					child.hide();
				}
				return this.view.unbatchRows();
			}
		};


		/*
  	Switches between expanded/collapsed states
   */

		RowParent.prototype.toggleExpanded = function() {
			if (this.isExpanded) {
				return this.collapse();
			} else {
				return this.expand();
			}
		};


		/*
  	Changes the expander icon to the "expanded" state
   */

		RowParent.prototype.indicateExpanded = function() {
			return this.trs.find('.fc-expander .fc-icon').removeClass(this.getCollapsedIcon()).addClass(this.getExpandedIcon());
		};


		/*
  	Changes the expander icon to the "collapsed" state
   */

		RowParent.prototype.indicateCollapsed = function() {
			return this.trs.find('.fc-expander .fc-icon').removeClass(this.getExpandedIcon()).addClass(this.getCollapsedIcon());
		};


		/*
   */

		RowParent.prototype.enableExpanding = function() {
			return this.trs.find('.fc-expander-space').addClass('fc-expander');
		};


		/*
   */

		RowParent.prototype.disableExpanding = function() {
			return this.trs.find('.fc-expander-space').removeClass('fc-expander').find('.fc-icon').removeClass(this.getExpandedIcon()).removeClass(this.getCollapsedIcon());
		};

		RowParent.prototype.getExpandedIcon = function() {
			return 'fc-icon-down-triangle';
		};

		RowParent.prototype.getCollapsedIcon = function() {
			var dir;
			dir = this.view.isRTL ? 'left' : 'right';
			return 'fc-icon-' + dir + '-triangle';
		};


		/*
  	Causes a slide-down CSS transition to demonstrate that the expand has happened
   */

		RowParent.prototype.animateExpand = function() {
			var ref, ref1, trs;
			trs = (ref = this.children[0]) != null ? (ref1 = ref.getLeadingRow()) != null ? ref1.trs : void 0 : void 0;
			if (trs) {
				trs.addClass('fc-collapsed');
				setTimeout(function() {
					trs.addClass('fc-transitioning');
					return trs.removeClass('fc-collapsed');
				});
				return trs.one('webkitTransitionEnd otransitionend oTransitionEnd msTransitionEnd transitionend', function() {
					return trs.removeClass('fc-transitioning');
				});
			}
		};


		/*
  	Find each TRs "inner div" (div within first cell). This div controls each TRs height.
  	Returns the max pixel height.
   */

		RowParent.prototype.getMaxTrInnerHeight = function() {
			var max;
			max = 0;
			$.each(this.trHash, (function(_this) {
				return function(type, tr) {
					var innerEl;
					innerEl = getOwnCells(tr).find('> div:not(.fc-cell-content):first');
					return max = Math.max(innerEl.height(), max);
				};
			})(this));
			return max;
		};


		/*
  	Find each TRs "inner div" and sets all of their heights to the same value.
   */

		RowParent.prototype.setTrInnerHeight = function(height) {
			return $.each(this.trHash, (function(_this) {
				return function(type, tr) {
					return getOwnCells(tr).find('> div:not(.fc-cell-content):first').height(height);
				};
			})(this));
		};


		/*
  	Triggered when the current node has been shown (either freshly rendered or re-shown)
  	when it had previously been unrendered or hidden. `shown` does not bubble up the hierarchy.
   */

		RowParent.prototype.shown = function() {
			if (this.hasOwnRow) {
				return this.rowShown(this);
			}
		};


		/*
  	Triggered when the current node has been hidden (either temporarily or permanently)
  	when it had previously been shown. `hidden` does not bubble up the hierarchy.
   */

		RowParent.prototype.hidden = function() {
			if (this.hasOwnRow) {
				return this.rowHidden(this);
			}
		};


		/*
  	Just like `shown`, but only triggered for nodes that are actual rows. Bubbles up the hierarchy.
   */

		RowParent.prototype.rowShown = function(row) {
			return (this.parent || this.view).rowShown(row);
		};


		/*
  	Just like `hidden`, but only triggered for nodes that are actual rows. Bubbles up the hierarchy.
   */

		RowParent.prototype.rowHidden = function(row) {
			return (this.parent || this.view).rowHidden(row);
		};


		/*
  	Triggered when the current node has been added to the hierarchy. `added` does not bubble up.
   */

		RowParent.prototype.added = function() {
			if (this.hasOwnRow) {
				return this.rowAdded(this);
			}
		};


		/*
  	Triggered when the current node has been removed from the hierarchy. `removed` does not bubble up.
   */

		RowParent.prototype.removed = function() {
			if (this.hasOwnRow) {
				return this.rowRemoved(this);
			}
		};


		/*
  	Just like `added`, but only triggered for nodes that are actual rows. Bubbles up the hierarchy.
   */

		RowParent.prototype.rowAdded = function(row) {
			return (this.parent || this.view).rowAdded(row);
		};


		/*
  	Just like `removed`, but only triggered for nodes that are actual rows. Bubbles up the hierarchy.
   */

		RowParent.prototype.rowRemoved = function(row) {
			return (this.parent || this.view).rowRemoved(row);
		};

		return RowParent;

	})();


	/*
An abstract node in a row-hierarchy tree that contains other nodes.
Will have some sort of rendered label indicating the grouping,
up to the subclass for determining what to do with it.
 */

	RowGroup = (function(superClass) {
		extend(RowGroup, superClass);

		RowGroup.prototype.groupSpec = null;

		RowGroup.prototype.groupValue = null;

		function RowGroup(view, groupSpec1, groupValue1) {
			this.groupSpec = groupSpec1;
			this.groupValue = groupValue1;
			RowGroup.__super__.constructor.apply(this, arguments);
		}


		/*
  	Called when this row (if it renders a row) or a subrow is removed
   */

		RowGroup.prototype.rowRemoved = function(row) {
			RowGroup.__super__.rowRemoved.apply(this, arguments);
			if (row !== this && !this.children.length) {
				return this.remove();
			}
		};


		/*
  	Renders the content wrapper element that will be inserted into this row's TD cell
   */

		RowGroup.prototype.renderGroupContentEl = function() {
			var contentEl, filter;
			contentEl = $('<div class="fc-cell-content" />').append(this.renderGroupTextEl());
			filter = this.groupSpec.render;
			if (typeof filter === 'function') {
				contentEl = filter(contentEl, this.groupValue) || contentEl;
			}
			return contentEl;
		};


		/*
  	Renders the text span element that will be inserted into this row's TD cell.
  	Goes within the content element.
   */

		RowGroup.prototype.renderGroupTextEl = function() {
			var filter, text;
			text = this.groupValue || '';
			filter = this.groupSpec.text;
			if (typeof filter === 'function') {
				text = filter(text) || text;
			}
			return $('<span class="fc-cell-text" />').text(text);
		};

		return RowGroup;

	})(RowParent);


	/*
A row grouping that renders as a single solid row that spans width-wise (like a horizontal rule)
 */

	HRowGroup = (function(superClass) {
		extend(HRowGroup, superClass);

		function HRowGroup() {
			return HRowGroup.__super__.constructor.apply(this, arguments);
		}

		HRowGroup.prototype.hasOwnRow = true;


		/*
  	Renders this row's TR for the "spreadsheet" quadrant, the area with info about each resource
   */

		HRowGroup.prototype.renderSpreadsheetContent = function(tr) {
			var contentEl;
			contentEl = this.renderGroupContentEl();
			contentEl.prepend('<span class="fc-expander">' + '<span class="fc-icon"></span>' + '</span>');
			return $('<td class="fc-divider" />').attr('colspan', this.view.colSpecs.length).append($('<div/>').append(contentEl)).appendTo(tr);
		};


		/*
  	Renders this row's TR for the quadrant that contains a resource's events
   */

		HRowGroup.prototype.renderEventContent = function(tr) {
			return tr.append('<td class="fc-divider"> <div/> </td>');
		};

		return HRowGroup;

	})(RowGroup);


	/*
A row grouping that renders as a tall multi-cell vertical span in the "spreadsheet" area
 */

	VRowGroup = (function(superClass) {
		extend(VRowGroup, superClass);

		function VRowGroup() {
			return VRowGroup.__super__.constructor.apply(this, arguments);
		}

		VRowGroup.prototype.rowspan = 0;

		VRowGroup.prototype.leadingTr = null;

		VRowGroup.prototype.groupTd = null;


		/*
  	Called when a row somewhere within the grouping is shown
   */

		VRowGroup.prototype.rowShown = function(row) {
			this.rowspan += 1;
			this.renderRowspan();
			return VRowGroup.__super__.rowShown.apply(this, arguments);
		};


		/*
  	Called when a row somewhere within the grouping is hidden
   */

		VRowGroup.prototype.rowHidden = function(row) {
			this.rowspan -= 1;
			this.renderRowspan();
			return VRowGroup.__super__.rowHidden.apply(this, arguments);
		};


		/*
  	Makes sure the groupTd has the correct rowspan / place in the DOM.
  	PRECONDITION: in the case of multiple group nesting, a child's renderRowspan()
  	will be called before the parent's renderRowspan().
   */

		VRowGroup.prototype.renderRowspan = function() {
			var leadingTr;
			if (this.rowspan) {
				if (!this.groupTd) {
					this.groupTd = $('<td class="' + this.view.widgetContentClass + '"/>').append(this.renderGroupContentEl());
				}
				this.groupTd.attr('rowspan', this.rowspan);
				leadingTr = this.getLeadingRow().getTr('spreadsheet');
				if (leadingTr !== this.leadingTr) {
					if (leadingTr) {
						leadingTr.prepend(this.groupTd);
					}
					return this.leadingTr = leadingTr;
				}
			} else {
				if (this.groupTd) {
					this.groupTd.remove();
					this.groupTd = null;
				}
				return this.leadingTr = null;
			}
		};

		return VRowGroup;

	})(RowGroup);

	EventRow = (function(superClass) {
		extend(EventRow, superClass);

		function EventRow() {
			return EventRow.__super__.constructor.apply(this, arguments);
		}

		EventRow.prototype.hasOwnRow = true;

		EventRow.prototype.segContainerEl = null;

		EventRow.prototype.segContainerHeight = null;

		EventRow.prototype.innerEl = null;

		EventRow.prototype.bgSegContainerEl = null;

		EventRow.prototype.isSegsRendered = false;

		EventRow.prototype.isBusinessHourSegsRendered = false;

		EventRow.prototype.businessHourSegs = null;

		EventRow.prototype.bgSegs = null;

		EventRow.prototype.fgSegs = null;

		EventRow.prototype.renderEventContent = function(tr) {
			tr.html('<td class="' + this.view.widgetContentClass + '"> <div> <div class="fc-event-container" /> </div> </td>');
			this.segContainerEl = tr.find('.fc-event-container');
			this.innerEl = this.bgSegContainerEl = tr.find('td > div');
			return this.ensureSegsRendered();
		};

		EventRow.prototype.ensureSegsRendered = function() {
			if (!this.isSegsRendered) {
				this.ensureBusinessHourSegsRendered();
				if (this.bgSegs) {
					this.view.timeGrid.renderFillInContainer('bgEvent', this, this.bgSegs);
				}
				if (this.fgSegs) {
					this.view.timeGrid.renderFgSegsInContainers([[this, this.fgSegs]]);
				}
				return this.isSegsRendered = true;
			}
		};

		EventRow.prototype.ensureBusinessHourSegsRendered = function() {
			if (this.businessHourSegs && !this.isBusinessHourSegsRendered) {
				this.view.timeGrid.renderFillInContainer('businessHours', this, this.businessHourSegs, 'bgevent');
				return this.isBusinessHourSegsRendered = true;
			}
		};

		EventRow.prototype.unrenderEventContent = function() {
			this.clearBusinessHourSegs();
			this.bgSegs = null;
			this.fgSegs = null;
			return this.isSegsRendered = false;
		};

		EventRow.prototype.clearBusinessHourSegs = function() {
			var j, len, ref, seg;
			if (this.businessHourSegs) {
				ref = this.businessHourSegs;
				for (j = 0, len = ref.length; j < len; j++) {
					seg = ref[j];
					if (seg.el) {
						seg.el.remove();
					}
				}
				this.businessHourSegs = null;
			}
			return this.isBusinessHourSegsRendered = false;
		};

		return EventRow;

	})(RowParent);


	/*
A row that renders information about a particular resource, as well as it events (handled by superclass)
 */

	ResourceRow = (function(superClass) {
		extend(ResourceRow, superClass);

		ResourceRow.prototype.resource = null;

		function ResourceRow(view, resource1) {
			this.resource = resource1;
			ResourceRow.__super__.constructor.apply(this, arguments);
		}


		/*
  	Called when a row in the tree has been added
   */

		ResourceRow.prototype.rowAdded = function(row) {
			ResourceRow.__super__.rowAdded.apply(this, arguments);
			if (row !== this && this.isRendered) {
				if (this.children.length === 1) {
					this.enableExpanding();
					if (this.isExpanded) {
						return this.indicateExpanded();
					} else {
						return this.indicateCollapsed();
					}
				}
			}
		};


		/*
  	Called when a row in the tree has been removed
   */

		ResourceRow.prototype.rowRemoved = function(row) {
			ResourceRow.__super__.rowRemoved.apply(this, arguments);
			if (row !== this && this.isRendered) {
				if (!this.children.length) {
					return this.disableExpanding();
				}
			}
		};

		ResourceRow.prototype.render = function() {
			ResourceRow.__super__.render.apply(this, arguments);
			if (this.children.length > 0) {
				this.enableExpanding();
			} else {
				this.disableExpanding();
			}
			return this.view.publiclyTrigger('resourceRender', this.resource, this.resource, this.getTr('spreadsheet').find('> td'), this.getTr('event').find('> td'));
		};

		ResourceRow.prototype.renderEventContent = function(tr) {
			ResourceRow.__super__.renderEventContent.apply(this, arguments);
			return tr.attr('data-resource-id', this.resource.id);
		};


		/*
  	Populates the TR with cells containing data about the resource
   */

		ResourceRow.prototype.renderSpreadsheetContent = function(tr) {
			var colSpec, contentEl, input, j, len, ref, resource, td, text;
			resource = this.resource;
			ref = this.view.colSpecs;
			for (j = 0, len = ref.length; j < len; j++) {
				colSpec = ref[j];
				if (colSpec.group) {
					continue;
				}
				input = colSpec.field ? resource[colSpec.field] || null : resource;
				text = typeof colSpec.text === 'function' ? colSpec.text(resource, input) : input;
				contentEl = $('<div class="fc-cell-content">' + (colSpec.isMain ? this.renderGutterHtml() : '') + '<span class="fc-cell-text">' + (text ? htmlEscape(text) : '&nbsp;') + '</span>' + '</div>');
				if (typeof colSpec.render === 'function') {
					contentEl = colSpec.render(resource, contentEl, input) || contentEl;
				}
				td = $('<td class="' + this.view.widgetContentClass + '"/>').append(contentEl);
				if (colSpec.isMain) {
					td.wrapInner('<div/>');
				}
				tr.append(td);
			}
			return tr.attr('data-resource-id', resource.id);
		};


		/*
  	Renders the HTML responsible for the subrow expander area,
  	as well as the space before it (used to align expanders of similar depths)
   */

		ResourceRow.prototype.renderGutterHtml = function() {
			var html, i, j, ref;
			html = '';
			for (i = j = 0, ref = this.depth; j < ref; i = j += 1) {
				html += '<span class="fc-icon"/>';
			}
			html += '<span class="fc-expander-space">' + '<span class="fc-icon"></span>' + '</span>';
			return html;
		};

		return ResourceRow;

	})(EventRow);

	FC.views.timeline.resourceClass = ResourceTimelineView;

	ResourceAgendaView = (function(superClass) {
		extend(ResourceAgendaView, superClass);

		function ResourceAgendaView() {
			return ResourceAgendaView.__super__.constructor.apply(this, arguments);
		}

		ResourceAgendaView.mixin(VertResourceViewMixin);

		ResourceAgendaView.prototype.timeGridClass = ResourceTimeGrid;

		ResourceAgendaView.prototype.dayGridClass = ResourceDayGrid;

		ResourceAgendaView.prototype.renderHead = function() {
			ResourceAgendaView.__super__.renderHead.apply(this, arguments);
			return this.timeGrid.processHeadResourceEls(this.headContainerEl);
		};

		ResourceAgendaView.prototype.setResourcesOnGrids = function(resources) {
			this.timeGrid.setResources(resources);
			if (this.dayGrid) {
				return this.dayGrid.setResources(resources);
			}
		};

		ResourceAgendaView.prototype.unsetResourcesOnGrids = function() {
			this.timeGrid.unsetResources();
			if (this.dayGrid) {
				return this.dayGrid.unsetResources();
			}
		};

		return ResourceAgendaView;

	})(FC.AgendaView);

	FC.views.agenda.queryResourceClass = function(viewSpec) {
		var ref;
		if ((ref = viewSpec.options.groupByResource || viewSpec.options.groupByDateAndResource) != null ? ref : viewSpec.duration.as('days') === 1) {
			return ResourceAgendaView;
		}
	};

	ResourceBasicView = (function(superClass) {
		extend(ResourceBasicView, superClass);

		function ResourceBasicView() {
			return ResourceBasicView.__super__.constructor.apply(this, arguments);
		}

		ResourceBasicView.mixin(VertResourceViewMixin);

		ResourceBasicView.prototype.dayGridClass = ResourceDayGrid;

		ResourceBasicView.prototype.renderHead = function() {
			ResourceBasicView.__super__.renderHead.apply(this, arguments);
			return this.dayGrid.processHeadResourceEls(this.headContainerEl);
		};

		ResourceBasicView.prototype.setResourcesOnGrids = function(resources) {
			return this.dayGrid.setResources(resources);
		};

		ResourceBasicView.prototype.unsetResourcesOnGrids = function() {
			return this.dayGrid.unsetResources();
		};

		return ResourceBasicView;

	})(FC.BasicView);

	ResourceMonthView = (function(superClass) {
		extend(ResourceMonthView, superClass);

		function ResourceMonthView() {
			return ResourceMonthView.__super__.constructor.apply(this, arguments);
		}

		ResourceMonthView.mixin(VertResourceViewMixin);

		ResourceMonthView.prototype.dayGridClass = ResourceDayGrid;

		ResourceMonthView.prototype.renderHead = function() {
			ResourceMonthView.__super__.renderHead.apply(this, arguments);
			return this.dayGrid.processHeadResourceEls(this.headContainerEl);
		};

		ResourceMonthView.prototype.setResourcesOnGrids = function(resources) {
			return this.dayGrid.setResources(resources);
		};

		ResourceMonthView.prototype.unsetResourcesOnGrids = function() {
			return this.dayGrid.unsetResources();
		};

		return ResourceMonthView;

	})(FC.MonthView);

	FC.views.basic.queryResourceClass = function(viewSpec) {
		var ref;
		if ((ref = viewSpec.options.groupByResource || viewSpec.options.groupByDateAndResource) != null ? ref : viewSpec.duration.as('days') === 1) {
			return ResourceBasicView;
		}
	};

	FC.views.month.queryResourceClass = function(viewSpec) {
		if (viewSpec.options.groupByResource || viewSpec.options.groupByDateAndResource) {
			return ResourceMonthView;
		}
	};

	RELEASE_DATE = '2018-03-05';

	UPGRADE_WINDOW = {
		years: 1,
		weeks: 1
	}

});
