<?php namespace Visiosoft\BookingModule\Http\Controller\Admin;

use Anomaly\SettingsModule\Setting\Contract\SettingRepositoryInterface;
use Anomaly\Streams\Platform\Entry\EntryModel;
use Anomaly\UsersModule\User\UserModel;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use PHPUnit\Util\Json;
use Visiosoft\BookingModule\Booking\BookingModel;
use Illuminate\Contracts\Config\Repository;
use Visiosoft\BookingModule\Booking\Contract\BookingRepositoryInterface;
use Visiosoft\BookingModule\Booking\Events\CancellationBooking;
use Visiosoft\BookingModule\Booking\Form\BookingFormBuilder;
use Visiosoft\BookingModule\Booking\Table\BookingTableBuilder;
use Anomaly\Streams\Platform\Http\Controller\AdminController;
use Visiosoft\BookingModule\Service\ServiceModel;
use Illuminate\Http\Request;
use Visiosoft\BookingModule\Staff\StaffModel;

class BookingController extends AdminController
{
    private $model;
    private $request_params;
    private $repository;

    public function __construct(BookingModel $model, Request $request, BookingRepositoryInterface $repository)
    {
        parent::__construct();
        $this->model = $model;
        $this->repository = $repository;
        $this->request_params = $request->all();
    }

    /**
     * Display an index of existing entries.
     *
     * @param BookingTableBuilder $table
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function index(BookingTableBuilder $table)
    {
        if (!isset($this->request->view) || $this->request->view !== "booking") {
            $table->setColumns(['staff', 'booking_datetime_start',
                'services_select' => [
                    'value' => function (EntryModel $entry) {
                        return $this->getServiceName($entry->services_select);
                    }
                ]
                , 'name', 'phone', 'email']);
        }

        return $table->render();
    }

    public function bookingJSON()
    {
        $all_bookings = $this->model
            ->newQuery()
            ->join('booking_services', function ($join) {
                $join->on('booking_booking.services_select', '=', 'booking_services.id');
                if ($locationId = \request()->filter_location) {
                    $join->where('location_id', $locationId);
                }
            })->select([
                'booking_booking.name as title',
                'booking_booking.phone as phone',
                'booking_booking.email as email',
                'booking_booking.booking_datetime_start as start',
                'booking_booking.booking_datetime_end as end',
                'booking_booking.id as id',
                'booking_booking.staff_id as resourceId',
                'booking_services.color as color',
            ]);

        if($this->request->has('group'))
        {
            $all_bookings = $all_bookings->groupBy($this->request->get('group'));
        }

        $all_bookings = $all_bookings->get();

        return $this->response->json($all_bookings);
    }

    public function staffJSON()
    {
        $all_bookings = StaffModel::query()
            ->newQuery()
            ->select([
                'booking_staff.id as id',
                'booking_staff.name as title',
            ])
            ->get();
        return $this->response->json($all_bookings);
    }

    /**
     * Create a new entry.
     *
     * @param BookingFormBuilder $form
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function create(BookingFormBuilder $form)
    {
        return $form->render();
    }

    /**
     * Edit an existing entry.
     *
     * @param BookingFormBuilder $form
     * @param        $id
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function edit(BookingFormBuilder $form, $id)
    {
        return $form->render($id);
    }

    /**
     * @param $id
     * @param $setting
     * @return string
     */
    public function getServiceName($id)
    {
        $serviceModel = new ServiceModel();
        return $serviceModel->getServiceName($id) . " - " . $serviceModel->getServicePrice($id) . " " . substr(setting_value('streams::currency'), 0, 3);
    }

    /**
     * @param BookingModel $bookingModel
     * @param $id
     */
    public function getDetails(BookingModel $bookingModel, $id)
    {
        return $this->response->json($bookingModel->getDetail($id));
    }

    public function createAjax()
    {
        $params = $this->request_params;
        $status = "error_create";
        $modal = 'show';
        unset($params['locations']);

        $search = $this->model->isBooking($params['booking_datetime_start'], $params['staff'], $params['booking_datetime_end']);

        if ($search == 0) {
            $this->model->create($params);
            $status = "create_success";
            $modal = 'hide';
        }

        return response()->json(['status' => $status, 'form' => $modal]);
    }

    public function editAjax()
    {
        $params = $this->request_params;
        $status = "found_booking";
        $modal = 'hide';
        unset($params['locations']);
        $booking = $this->model->find($params['id']);
        if ($booking) {
            unset($params['id']);
            $params['payment_type'] = $booking->payment_type;
            $params['email'] = $booking->email;
            $params['total_price'] = $booking->total_price;

            $userDetail = $this->request->ip() . "-#" . Auth::id();

            $booking->update(['updated_by_id' => Auth::id(), 'deleted_by_user' => $userDetail]);
            $booking->delete();

            $booking->create($params);
            $status = "edit_success";
        }
        return response()->json(['status' => $status, 'form' => $modal]);

    }

    public function deleteAjax($id)
    {
        $booking = $this->model->newQuery()->find($id);
        $userDetail = $this->request->ip() . "-#" . Auth::id();

        $booking->update(['updated_by_id' => Auth::id(), 'deleted_by_user' => $userDetail]);
        $booking->delete();

        event(new CancellationBooking($booking));


        $this->messages->success(trans('streams::message.delete_success', ['count' => 1]));
        return $this->redirect->to('/admin/booking/booking');
    }

    public function approveAjax($id)
    {
        $this->model->activateById($id);
        return response()->json(['status' => true]);
    }

    public function ApprovePayment($id)
    {
        $this->repository->changePaymentStatus($id, 'success');
        $this->messages->success([trans('visiosoft.module.booking::messages.approvedPaymentMsg')]);
        return $this->redirect->to('/admin/booking/booking');
    }

    public function ApprovePaymentAjax($id)
    {
        $this->repository->changePaymentStatus($id, 'success');
        return response()->json(['status' => true]);
    }
}
