<?php namespace Visiosoft\BookingModule\Booking;

use Anomaly\UsersModule\User\UserModel;
use DateTime;
use Visiosoft\BookingModule\Booking\Contract\BookingInterface;
use Anomaly\Streams\Platform\Model\Booking\BookingBookingEntryModel;

class BookingModel extends BookingBookingEntryModel implements BookingInterface
{
    /**
     * @param array $params
     * @param array $search_fields
     * @return array
     * Control Required Field
     */
    public function requiredFieldControl(array $params, array $search_fields)
    {
        $error = array();

        foreach ($search_fields as $field) {
            if (isset($params[$field])) {
                $params[$field] = str_replace(' ', '', $params[$field]);
                if (empty($params[$field]) || is_null($params[$field])) {
                    $error[] = trans('visiosoft.module.booking::error.required_field',
                        ['field' => trans('visiosoft.module.booking::field.' . $field . '.name')]);
                }
            } else {
                $error[] = trans('visiosoft.module.booking::error.required_field', [
                    'field' => trans('visiosoft.module.booking::field.' . $field . '.name')
                ]);
            }
        }
        return $error;

    }

    /**
     * @param array $params
     * @param array $search_fields
     * @return array
     * Control Required Field
     */
    public function uniqueFieldControl(array $params, array $search_fields)
    {
        $error = array();

        foreach ($search_fields as $field) {
            if ($field == 'phone') {
                $exists = UserModel::where('phone_number', $params[$field])->first();
                if (!is_null($exists)) {
                    $error[] = trans('visiosoft.module.booking::error.exist_field', [
                        'field' => trans('visiosoft.module.booking::field.' . $field . '.name')
                    ]);
                }
            } else {
                $exists = UserModel::where($field, $params[$field])->first();
                if (!is_null($exists)) {
                    $error[] = trans('visiosoft.module.booking::error.exist_field', [
                        'field' => trans('visiosoft.module.booking::field.' . $field . '.name')
                    ]);
                }
            }
        }
        return $error;

    }

    /**
     * @param $service_id
     * @param $staff_id
     * @return mixed
     * Get Old reservation With Staff and Service ID
     */
    public function getBookingDateListServiceAndStaff($service_id, $staff_id)
    {
        $bokingDayLimit = setting_value('visiosoft.module.booking::bookingDateLimit') + 1;
        $start_date = date('Y-m-d');
        $end_date = date('Y-m-d', strtotime($start_date . ' + ' . $bokingDayLimit . ' day'));

        return $this
            ->where('staff_id', $staff_id)
            ->whereBetween('booking_datetime_start', [$start_date, $end_date])
            ->whereIn('activated', [0, 1])
            ->whereIn('status', ['active', 'pending'])
            ->select('id', 'booking_datetime_start', 'booking_datetime_end', 'services_select', 'staff_id')
            ->get()->toArray();
    }

    /**
     * @param array $params
     * @return array
     * Create Booking with control
     */
    public function createBooking(array $params)
    {
        //Create Save Parameters
        $saveParams['total_price'] = $params['total_price'];
        $saveParams['created_at'] = date('Y-m-d H:i:s');
        $saveParams['name'] = $params['username'];
        $saveParams['phone'] = $params['phone'];
        $saveParams['email'] = $params['email'];
        $saveParams['user_id'] = $params['user_id'];
        $saveParams['services_select'] = $params['service_id'];
        $saveParams['staff_id'] = $params['staff_id'];
        $saveParams['booking_datetime_start'] = $params['select_time'];
        $saveParams['booking_datetime_end'] = $params['booking_datetime_end'];
        $saveParams['payment_type'] = $params['payment_type'];
        $saveParams['activation_code'] = $params['activation_code'];

        if (!setting_value('visiosoft.module.booking::activation_system')) {
            $saveParams['activated'] = $params['activated'];
            $saveParams['status'] = $params['status'];
        }

        //Create Booking is Save Parameters
        $booking = $this->create($saveParams);
        return $booking->id;
    }

    /**
     * @param $date
     * @param $staff_id
     * @param $service_id
     * @return mixed
     * Control is Bookingwith Date
     */
    public function isBooking($start, $staff_id, $end)
    {

        $end = new DateTime($end);
        $end = $end->modify('-1 minutes')->format('Y-m-d H:i:s');
        $start = new DateTime($start);
        $start = $start->modify('+1 minutes')->format('Y-m-d H:i:s');

        $search = $this->newQuery()
            ->where(function ($query) use ($start, $end) {
                $query->WhereBetween('booking_datetime_start', [$start, $end]);
                $query->orWhereBetween('booking_datetime_end', [$start, $end]);
                $query->orWhereRaw('booking_datetime_start < "' . $start . '" and booking_datetime_end > "' . $end . '"');

            })
            ->where(function ($query) use ($staff_id) {
                $query->where('staff_id', $staff_id);
            })
            ->count();

        return $search;
    }

    public function activate($activation_code, $id)
    {
        $search = $this->where('id', $id)
            ->where('activation_code', $activation_code)
            ->first();

        if ($search != null) {
            $search->update(['activated' => 1, 'status' => 'active']);
            return true;
        }
        return false;
    }

    public function activateById($id)
    {
        $search = $this->where('id', $id)->first();

        if ($search != null) {
            $search->update(['activated' => 1, 'status' => 'active']);
            return true;
        }
        return false;
    }

    public function activatePayPal($id, $payment)
    {
        $payment_success_params = ['activated' => 1, 'status' => 'active', 'payment_status' => "success"];
        $payment_failed_params = ['activated' => 0, 'status' => 'pending', 'payment_status' => "failed"];

        if (!setting_value('visiosoft.module.booking::activation_system')) {
            $payment_failed_params = $payment_success_params;
            $payment_failed_params['payment_status'] = 'failed';
        }

        $search = $this->where('id', $id)
            ->first();

        if ($search != null) {
            if ($payment) {
                $search->update($payment_success_params);
            } else {
                $search->update($payment_failed_params);
            }
            return true;
        }
        return false;
    }

    /**
     * Disable timestamps for this model.
     *
     * @var bool
     */
    public $timestamps = false;

    /**
     * The number of minutes to cache query results.
     *
     * @var null|false|int
     */
    protected $ttl = false;

    /**
     * The attributes that are
     * not mass assignable.
     *
     * @var array
     */
    protected $guarded = [];

    /**
     * Runtime cache.
     *
     * @var array
     */
    protected $cache = [];

    /**
     * @param $id
     * @return mixed
     */
    public function getDetail($id)
    {
        return $this->find($id);
    }

    public function getBookingForDate($date)
    {
        $date = date("Y-m-d H:i:s", strtotime($date));
        return $this->newQuery()->whereRaw(
            "(booking_datetime_start <= ? AND booking_datetime_end > ?)",
            [$date, $date]
        )->get();
    }

}
