<?php namespace Visiosoft\BiddingModule\Http\Controller;

use Anomaly\Streams\Platform\Entry\Contract\EntryInterface;
use Anomaly\Streams\Platform\Http\Controller\PublicController;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Visiosoft\AdvsModule\Adv\AdvModel;
use Visiosoft\AdvsModule\Adv\Contract\AdvRepositoryInterface;
use Visiosoft\BiddingModule\AdDetail\Contract\AdDetailRepositoryInterface;
use Visiosoft\BiddingModule\Offer\Contract\OfferRepositoryInterface;

class AuctionController extends PublicController
{
    protected $entryRepository;
    protected $repository;
    protected $auctionEntryDetailRepository;
    protected $advModel;

    public function __construct(
        AdvRepositoryInterface      $entryRepository,
        OfferRepositoryInterface    $repository,
        AdDetailRepositoryInterface $auctionEntryDetailRepository,
        AdvModel                    $advModel

    )
    {
        $this->entryRepository = $entryRepository;
        $this->repository = $repository;
        $this->auctionEntryDetailRepository = $auctionEntryDetailRepository;
        $this->advModel = $advModel;
        parent::__construct();
    }

    public function appendBid()
    {
        $validator = Validator::make(request()->all(), [
            'entry_id' => 'required|integer',
            'price' => 'required',
        ]);

        if ($validator->fails()) {
            return $this->response->json([
                'success' => false,
                'message' => $validator->errors(),
            ], 400);
        }

        /**
         * Check Entry
         * Check Entry Finish At
         *  Check Approved Ad
         * Check Auction Ad
         */
        if (!$entry = $this->entryRepository->find($this->request->get('entry_id'))) {
            return $this->response->json([
                'success' => false,
                'message' => ['entry' => [
                    trans('visiosoft.module.bidding::message.entry_not_found')
                ]],
            ], 404);
        }

        if ($entry->getAttribute('finish_at') <= Carbon::now()
            || is_null($entry->getAttribute('slug'))
            || $entry->getAttribute('status') != 'approved') {
            return $this->response->json([
                'success' => false,
                'message' => ['entry' => [
                    trans('visiosoft.module.bidding::message.ad_is_not_live')
                ]],
            ], 406);
        }

        if (!$entryDetail = $this->auctionEntryDetailRepository->findByEntryID($entry->id)) {
            return $this->response->json([
                'success' => false,
                'message' => ['entry' => [
                    trans('visiosoft.module.bidding::message.ad_is_not_available_bidding')
                ]],
            ], 406);
        }

        /**
         * Check bid rules
         * Check Old Bid
         */
        $is_bigger_bid = $this->repository->newQuery()
            ->where('parent_adv_id', $entry->getId())
            ->where('bid_price', '>=', $this->request->get('price'))
            ->first();

        if ($is_bigger_bid) {
            return $this->response->json([
                'success' => false,
                'message' => ['price' => [
                    trans('visiosoft.module.bidding::message.cannot_lower_bid_than_previous_bid')
                ]],
            ], 406);
        }

        if ($entryDetail->getAttribute('starting_bid_price') > $this->request->get('price')) {
            return $this->response->json([
                'success' => false,
                'message' => ['price' => [
                    trans('visiosoft.module.bidding::message.cannot_bid_below_starting_price')
                ]],
            ], 406);
        }

        if ($entryDetail->getAttribute('buy_it_now_price') < $this->request->get('price')) {
            return $this->response->json([
                'success' => false,
                'message' => ['price' => [
                    trans('visiosoft.module.bidding::message.cannot_higher_buy_it_now')
                ]],
            ], 406);
        }

        $this->repository->newQuery()
            ->create([
                'parent_adv_id' => $entry->getId(),
                'bid_price' => $this->request->get('price')
            ]);

        return $this->response->json([
            'success' => true,
            'message' => ['entry' => [
                trans('visiosoft.module.bidding::message.bid_added_success')
            ]],
        ], 201);
    }

    public function bids()
    {
        $validator = Validator::make(request()->all(), [
            'entry_id' => 'required|integer',
        ]);

        if ($validator->fails()) {
            return $this->response->json([
                'success' => false,
                'message' => $validator->errors(),
            ], 400);
        }

        if (!$entry = $this->entryRepository->find($this->request->get('entry_id'))) {
            return $this->response->json([
                'success' => false,
                'message' => ['entry' => [
                    trans('visiosoft.module.bidding::message.entry_not_found')
                ]],
            ], 404);
        }

        if ($entry->getAttribute('finish_at') <= Carbon::now()
            || is_null($entry->getAttribute('slug'))
            || $entry->getAttribute('status') != 'approved') {
            return $this->response->json([
                'success' => false,
                'message' => ['entry' => [
                    trans('visiosoft.module.bidding::message.ad_is_not_live')
                ]],
            ], 406);
        }

        if (!$entryDetail = $this->auctionEntryDetailRepository->findByEntryID($entry->id)) {
            return $this->response->json([
                'success' => false,
                'message' => ['entry' => [
                    trans('visiosoft.module.bidding::message.ad_is_not_available_bidding')
                ]],
            ], 406);
        }

        $entries = $this->repository->newQuery()
            ->where('parent_adv_id', $entry->getId())
            ->orderBy('bid_price', 'DESC')
            ->select(['bid_price', DB::raw('(select concat(u.first_name," ",u.last_name) from default_users_users as u where u.id = default_bidding_offers.created_by_id limit 1) as name')])
            ->limit($this->request->get('limit', 5))->get();

        $entries = $entries->map(function ($item) {
            $item['bid_price'] = number_format($item['bid_price'], 0, ',', '.');
            return $item;
        });

        //Set currency symbol
        $currencySymbol = $this->request->get('currency', '$');

        return $this->view->make('visiosoft.module.bidding::ad-detail/partials/handle/bids', compact('entries', 'currencySymbol'));
    }

    public function myAuctionItems()
    {
        $entries = $this->advModel->myAdvsByUser()
            ->selectRaw("`default_advs_advs`.*, `default_advs_advs_translations`.`locale`, 
            `default_advs_advs_translations`.`name`, `default_advs_advs_translations`.`advs_desc`, 
            `default_advs_advs_translations`.`seo_title`, `default_advs_advs_translations`.`seo_description`,
            `default_bidding_ad_detail`.`buy_it_now_price`,`default_bidding_ad_detail`.`auction_duration`,
            `default_bidding_ad_detail`.`starting_bid_price`,`default_bidding_ad_detail`.`reserve_price`")
            ->rightJoin('bidding_ad_detail', 'bidding_ad_detail.entry_id', 'advs_advs.id')
            ->whereRaw('NOW() < DATE_ADD(default_advs_advs.publish_at, INTERVAL default_bidding_ad_detail.auction_duration DAY)')
            ->paginate(setting_value('streams::per_page', 15));

        return $this->view->make('module::profile.my-auction-items', compact('entries'));
    }

    public function purchased()
    {
        // Tüm kazananları getirir.
        $entries = $this->advModel->newQuery()
            ->rightJoin('bidding_ad_detail', 'bidding_ad_detail.entry_id', 'advs_advs.id')
            ->addSelect('bidding_ad_detail.auction_duration')
            ->addSelect('bidding_ad_detail.starting_bid_price')
            ->addSelect('bidding_ad_detail.reserve_price')
            ->leftJoin('bidding_offers','bidding_offers.parent_adv_id','advs_advs.id')
            ->addSelect(DB::raw('max(default_bidding_offers.bid_price) as winning_price'))
            ->addSelect(DB::raw('default_bidding_offers.created_by_id as winning_user_id'))
            ->whereRaw('NOW() > DATE_ADD(default_advs_advs.publish_at, INTERVAL default_bidding_ad_detail.auction_duration DAY)')
            ->get();

        // Tüm kazananlar arasında benim kazandıklarımı filtreler
        $entries = $entries->filter(
            function (EntryInterface $entry) {
                return $entry->getAttribute('winning_user_id') == Auth::id();
            }
        );

        return $this->view->make('module::profile.purchased', compact('entries'));
    }
}
