<?php namespace Visiosoft\BankidModule\Bankid;

use Anomaly\UsersModule\User\Contract\UserRepositoryInterface;
use Dimafe6\BankID\Service\BankIDService;
use Monolog\Handler\StreamHandler;
use Monolog\Logger;
use Visiosoft\BankidModule\BankIDInterface;
use Visiosoft\BankidModule\SignRequest\Contract\SignRequestRepositoryInterface;

class BankID implements BankIDInterface
{
    /**
     * @var BankIDService $bankIDService
     */
    private $bankIDService;
    protected $signRequestRepository;

    public function __construct()
    {
        $this->signRequestRepository = app(SignRequestRepositoryInterface::class);

        $apiUrl = setting_value('visiosoft.module.bankid::api_url');
        $debug = setting_value('visiosoft.module.bankid::environment');
        if ($debug) {
            $apiUrl = setting_value('visiosoft.module.bankid::test_api_url');
        }

        $endUserIp = isset($_SERVER["REMOTE_ADDR"]) ? $_SERVER["REMOTE_ADDR"] : '127.0.0.1';

        $this->bankIDService = new BankIDService($apiUrl, $endUserIp, $this->getSecurityContents($debug));
    }


    /**
     * @return array
     */
    public function getSecurityContents($debug = false): array
    {
        $certPath = __DIR__ . '/ssl/';
        $certPass = setting_value('visiosoft.module.bankid::ssl_password');
        $certName = 'live.p12';

        if ($debug) {
            $certName = 'debug.p12';
            $certPass = "qwerty123";
        }


        $cert = $certPath . $certName;
        return [
            'verify' => false,
            'cert' => [$cert, $certPass]
        ];
    }

    /**
     * @param string $ssn
     * @param string $orderSlug
     * @param string $orderName
     * @param string $orderContent
     * @return false|string (Provide Order Ref For Check Scenario)
     */
    public function sign(string $ssn, string $orderSlug, string $orderName, string $orderContent)
    {
        try {
            $orderRef = $this->bankIDService->getSignResponse($ssn, $orderContent)->orderRef;
            $this->signRequestRepository->createOrder($ssn, $orderRef, $orderSlug, $orderName, $orderContent);
            return $orderRef;
        } catch (\Exception $e) {
            $this->logError($e);
            return false;
        }
    }


    /**
     * @param $orderRef
     * @return void
     */
    public function saveSignToRelatedModule($orderRef)
    {
        $signRequestRepository = app(SignRequestRepositoryInterface::class);
        $request = $signRequestRepository->getRequestByOrderRef($orderRef);
        if ($request) {
            switch ($request->order_slug) {
                case "disabled_card_photo":
                    $usersModule = app(UserRepositoryInterface::class);
                    $user = $usersModule->findBySsn($request->ssn);
                    $user->update(['is_disabled' => 1]);
                    break;
            }
        }
    }

    /**
     * @param string $orderRef
     * @param string|null $type
     * @return false|string (Status => complete, failed, cancelled)
     */
    public function checkSign(string $orderRef)
    {
        try {
            $status = $this->bankIDService->collectResponse($orderRef)->status;;
            $request = $this->signRequestRepository->findBy('order_ref', $orderRef);
            if ($request) {
                $params = ['status' => $status];
                if ($status == "complete") {
                    $params['sign_date'] = date("Y-m-d H:i:s");
                    $this->saveSignToRelatedModule($orderRef);
                }
                $request->update($params);
            }

            return $status;

        } catch (\Exception $e) {
            $this->logError($e);
            return false;
        }

    }

    public function logError($error)
    {
        $log = new Logger('bankid_module');
        $log->pushHandler(new StreamHandler(storage_path('logs/bankid_module.log')), Logger::ERROR);
        $log->error($error);
    }

}