<?php namespace Visiosoft\AuctionsModule\Http\Controller;

use Anomaly\Streams\Platform\Http\Controller\ResourceController;
use Anomaly\UsersModule\User\Contract\UserRepositoryInterface;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Visiosoft\AdvsModule\Adv\AdvModel;
use Visiosoft\AdvsModule\Adv\Contract\AdvRepositoryInterface;
use Visiosoft\AuctionsModule\LiveBid\Contract\LiveBidRepositoryInterface;
use Visiosoft\CatsModule\Category\CategoryModel;
use Visiosoft\CatsModule\Category\Contract\CategoryRepositoryInterface;
use Visiosoft\CustomfieldsModule\CustomField\Contract\CustomFieldRepositoryInterface;
use Visiosoft\StoreModule\Store\Contract\StoreRepositoryInterface;

class StoreController extends ResourceController
{

    protected $repository;
    protected $adv;
    protected $advModel;
    protected $customFieldRepository;
    protected $categoryModel;
    protected $categoryRepository;
    protected $liveBidRepository;
    protected $userRepository;
    protected $storeUserRepository;

    public function __construct(
        StoreRepositoryInterface $repository,
        AdvRepositoryInterface $advRepository,
        AdvModel $advModel,
        CustomFieldRepositoryInterface $customFieldRepository,
        CategoryModel $categoryModel,
        CategoryRepositoryInterface $categoryRepository,
        UserRepositoryInterface $userRepository,
        \Visiosoft\StoreModule\User\Contract\UserRepositoryInterface $storeUserRepository,
        LiveBidRepositoryInterface $liveBidRepository)
    {
        $this->repository = $repository;
        $this->adv = $advRepository;
        $this->advModel = $advModel;
        $this->customFieldRepository = $customFieldRepository;
        $this->categoryModel = $categoryModel;
        $this->categoryRepository = $categoryRepository;
        $this->userRepository = $userRepository;
        $this->storeUserRepository = $storeUserRepository;
        $this->liveBidRepository = $liveBidRepository;
        parent::__construct();
    }

    public function panel($slug)
    {
        $store = $this->repository->findBySlug($slug);

        if ($store->user_id !== \auth()->id()) {
            abort(403, trans('visiosoft.module.store::message.unauthorized_access'));
        }

        $auction_entries = $this->liveBidRepository->getByStoreId($store->id);

        $live = $auction_entries->filter(
            function ($entry) {
                return $entry->status == null;
            }
        );

        $completed = $auction_entries->filter(
            function ($entry) {
                return $entry->status != null;
            }
        );

        $sold = $auction_entries->filter(
            function ($entry) {
                return $entry->status == "sold";
            }
        );
        $sold = $sold->pluck('adv_id')->all();

        $unsold = $auction_entries->filter(
            function ($entry) {
                return $entry->status == "unsold";
            }
        );
        $unsold = $unsold->pluck('adv_id')->all();

        $pending_auction_products = $this->customFieldRepository
            ->QueryAdsWithCFValue('urun_tipi', setting_value('visiosoft.module.auctions::auction_customfields_value'))
            ->whereNotIn('advs_advs.id', $auction_entries->pluck('adv_id')->all())
            ->get();

        $pending_auction_products = $pending_auction_products->filter(
            function ($entry) {
                return $entry->created_by_id == Auth::id();
            }
        );

        $pending_auction_products = $this->adv->addAttributes($pending_auction_products);


        $live_auction_products = $this->customFieldRepository
            ->QueryAdsWithCFValue('urun_tipi', setting_value('visiosoft.module.auctions::auction_customfields_value'))
            ->whereIn('advs_advs.id', $live->pluck('adv_id')->all())
            ->get();

        $live_auction_products = $live_auction_products->filter(
            function ($entry) {
                return $entry->created_by_id == Auth::id();
            }
        );

        $live_auction_products = $this->adv->addAttributes($live_auction_products);

        $completed_auction_products = $this->customFieldRepository
            ->QueryAdsWithCFValue('urun_tipi', setting_value('visiosoft.module.auctions::auction_customfields_value'))
            ->whereIn('advs_advs.id', $completed->pluck('adv_id')->all())
            ->get();

        $completed_auction_products = $completed_auction_products->filter(
            function ($entry) {
                return $entry->created_by_id == Auth::id();
            }
        );

        $completed_auction_products = $this->adv->addAttributes($completed_auction_products);


        $storeUsers = $store->getStoreUsers();
        $usersIds = $storeUsers->pluck('id');

        $ads = $this->adv
            ->getByUsersIDs($usersIds, false, true)
            ->whereNotIn('advs_advs.id', $auction_entries->pluck('adv_id')->all())
            ->whereNotIn('advs_advs.id', $pending_auction_products->pluck('id')->all())
            ->get();

        $ads = $this->adv->addAttributes($ads);

        return view('visiosoft.module.auctions::store.panel',
            compact('store', 'ads', 'pending_auction_products', 'live_auction_products', 'completed_auction_products', 'sold', 'unsold'));
    }

    public function actionsAuctions($slug)
    {
        $store = $this->repository->findBySlug($slug);

        if ($this->request->has(['items', 'action']) and $count = count($this->request->items)) {

            $action = $this->request->action;
            $items = $this->request->items;

            if ($action == "unpublish") {
                $this->unpublish($count, $items);
            } elseif ($action == "publish") {
                $this->publish($items, $count, $store);
            } elseif ($action == "sold") {
                $this->sold($items, $count);
            } elseif ($action == "unsold") {
                $this->unsold($items, $count);
            } elseif ($action == "move") {
                $this->move($items, $count);
            }
        }
        return $this->redirect->route('store::store_panel', ['slug' => $store->slug]);
    }

    public function unpublish($count, $items)
    {
        $this->liveBidRepository->newQuery()->whereIn('adv_id', $items)->delete();
        $this->messages->success(trans('module::messages.removed_from_display', ['count' => $count]));
    }

    public function publish($items, $count, $store)
    {
        foreach ($items as $item) {
            if (!$entry = $this->liveBidRepository->findBy('adv_id', $item)) {
                $this->liveBidRepository->createNew($store->id, $item);
            } else {
                $entry->update(['status' => null]);
            }
        }
        $this->messages->success(trans('module::messages.products_on_display', ['count' => $count]));
    }

    public function sold($items, $count)
    {
        $this->liveBidRepository->getByAdsAndUpdateStatus($items, 'sold');
        $this->messages->success(trans('module::messages.sold', ['count' => $count]));
    }

    public function unsold($items, $count)
    {
        $this->liveBidRepository->getByAdsAndUpdateStatus($items, 'unsold');
        $this->messages->success(trans('module::messages.not_sold', ['count' => $count]));
    }

    public function move($items, $count)
    {
        $customfield = $this->customFieldRepository->findBySlug('urun_tipi');

        foreach ($items as $item) {
            if ($adv = $this->adv->newQuery()->find($item) and $customfield) {
                $customfield_values_by_adv = json_decode($adv->cf_json, true);
                $customfield_values_by_adv["cf" . $customfield->id] = setting_value('visiosoft.module.auctions::auction_customfields_value');
                $adv->cf_json = json_encode($customfield_values_by_adv);
                $adv->save();
            }
        }

        $this->messages->success(trans('module::messages.updated_as_auctions', ['count' => $count]));
    }

    public function stream($slug)
    {
        if ($store = $this->repository->findBySlug($slug)) {

            $auction_entries = $this->liveBidRepository->getByStoreId($store->id);

            $live = $auction_entries->filter(
                function ($entry) {
                    return $entry->status == null;
                }
            );

            $live_auction_products = $this->customFieldRepository
                ->QueryAdsWithCFValue('urun_tipi', setting_value('visiosoft.module.auctions::auction_customfields_value'))
                ->whereIn('advs_advs.id', $live->pluck('adv_id')->all())
                ->get();

            if (count($live_auction_products)) {
                $live_auction_products = $this->adv->addAttributes($live_auction_products);

                return $this->view->make('visiosoft.module.auctions::stream', compact('store', 'live_auction_products'));
            }
        }
        $this->messages->error(trans('visiosoft.module.auctions::messages.error_stream'));
        return redirect('/');
    }

    public function index($slug)
    {
        // Redirect if subdomain is allowed
        $domain = setting_value('streams::domain');
        if (\request()->getHttpHost() == $domain && setting_value('visiosoft.module.store::use_subdomain_routing')) {
            return redirect(\request()->getScheme() . '://' . $slug . '.' . $domain);
        }

        $slug = str_slug($slug, '_');
        $store = $this->repository->findBy('slug', $slug);
        if ($store && ($store->isApproved() || $store->getOwner()->id === \auth()->id())) {
            $storeUsers = $store->getStoreUsers();
            $users = $storeUsers;

            $user = null;
            if ($userId = \request()->userId) {
                $isStoreUser = $store->user_id === $userId;
                if (!$isStoreUser) {
                    $isStoreUser = $this->storeUserRepository->findUser($userId, $store->id, true);
                }
                if ($isStoreUser) {
                    $user = $this->userRepository->newQuery()->where('id', $userId)->get();
                    if (count($user)) {
                        $users = $user;
                    }
                }
            }

            $usersIds = $users->pluck('id');

            $auction_entries = $this->liveBidRepository->getByStoreId($store->id);

            $auction_products = $this->customFieldRepository
                ->QueryAdsWithCFValue('urun_tipi', setting_value('visiosoft.module.auctions::auction_customfields_value'))
                ->whereIn('advs_advs.created_by_id', $usersIds)
                ->whereNotIn('advs_advs.id', $auction_entries->pluck('adv_id')->all())
                ->paginate(setting_value('streams::per_page'));

            $auction_products = $this->adv->addAttributes($auction_products);

            $rawAds = $this->adv->getByUsersIDs($usersIds);
            if ($catReq = \request()->cat) {
                $rawAds = $this->checkCatParam($catReq, $rawAds);
            }
            $ads = clone $rawAds;
            $ads = $ads->whereNotIn('advs_advs.id', $auction_products->pluck('id')->all());
            $ads = $ads->paginate(setting_value('streams::per_page'));

            if (setting_value('visiosoft.module.advs::hide_out_of_stock_products_without_listing')) {
                $ads = $ads->filter(
                    function ($entry) {
                        return (($entry->is_get_adv == true && $entry->stock > 0) || ($entry->is_get_adv == false));
                    }
                );
            }

            $advs = $this->adv->addAttributes($ads);

            $previousCat = null;
            $subCats = array();
            for ($i = 1; $i <= 10; $i++) {
                $cats = $rawAds
                    ->groupBy('cat' . $i)
                    ->pluck('cat' . $i)
                    ->toArray();
                if (count($cats) === 1 && !is_null($cats[0])) {
                    $previousCat = $cats;
                    continue;
                }
                if (!is_null($previousCat)) {
                    if ($mainCats = $this->categoryModel->getMains($previousCat)) {
                        $currentCat = $this->categoryModel->getCat($previousCat);
                        $mainCats[] = [
                            'id' => $currentCat->id,
                            'val' => $currentCat->name,
                        ];
                        $subCats = array();
                        foreach ($cats as $cat) {
                            $subCats[] = $this->categoryRepository->find($cat);
                        }
                    }
                } else {
                    $rawCats = array();
                    foreach ($cats as $cat) {
                        $rawCats[] = $this->categoryRepository->find($cat);
                    }
                    $mainCats = array();
                    $subCats = array();
                    foreach ($rawCats as $rawCat) {
                        if ($rawCat) {
                            $mainCats[] = [
                                'id' => $rawCat->id,
                                'val' => $rawCat->name,
                                'parentCat' => true
                            ];
                        }
                    }
                }
                break;
            }

            $default_detail_page = setting_value('visiosoft.module.store::default_detail_page');
            $default_detail_page = $this->renderStoreHtml($store, $default_detail_page);

            $compact = compact('advs', 'mainCats', 'subCats', 'store', 'auction_products', 'storeUsers', 'user', 'default_detail_page');
            return $this->view->make('visiosoft.module.auctions::ads-list/list', $compact);
        }

        return $this->view->make('visiosoft.module.advs::error-msg')
            ->with('message', trans('visiosoft.module.store::message.this_store_is_not_available'));
    }

    public function checkCatParam($catParam, $userAds)
    {
        $cat_d = $this->categoryRepository->find($catParam);

        if ($cat_d->parent_category_id == null) {
            $catLevel = 1;
        } else {
            $catLevel = $this->categoryRepository->getModel()->getCatLevel($catParam);
        }

        $catLevel = "cat" . $catLevel;

        return $userAds->where($catLevel, $catParam);
    }

    public function renderStoreHtml($store, $default_detail_page)
    {
        $fields = array(
            "land_phone", "gsm_phone", "slug", "summary", "detailed_description", "address",
            "email", "web_site", "facebook", "instagram", "twitter", "tax_number", "lat", "lng"
        );

        foreach ($fields as $field) {
            $default_detail_page = str_replace("{{" . $field . "}}", $store->{$field}, $default_detail_page);
        }

        return $default_detail_page;
    }

    public function extendAd(AdvRepositoryInterface $advRepository, $id)
    {
        if ($ad = $advRepository->newQuery()->find($id)) {
            $defaultAdPublishTime = setting_value('visiosoft.module.advs::default_published_time');
            $ad->update([
                "publish_at" => Carbon::now(),
                'finish_at' => date('Y-m-d H:i:s', strtotime(date('Y-m-d H:i:s') . ' + ' . $defaultAdPublishTime . ' day')),
            ]);

            $this->messages->success(trans('visiosoft.module.advs::message.extended', ['number' => 1]));
        } else {
            $this->messages->error(trans('visiosoft.module.advs::message.ad_doesnt_exist'));
        }

        return $this->redirect->to('/profile/ads');
    }
}
