<?php namespace Visiosoft\ArabamTheme;

use Anomaly\SettingsModule\Setting\Contract\SettingRepositoryInterface;
use Anomaly\Streams\Platform\Assignment\Contract\AssignmentRepositoryInterface;
use Anomaly\Streams\Platform\Database\Seeder\Seeder;
use Anomaly\Streams\Platform\Field\Contract\FieldRepositoryInterface;
use Anomaly\Streams\Platform\Model\Repeater\RepeaterLocationsEntryModel;
use Anomaly\Streams\Platform\Stream\Contract\StreamRepositoryInterface;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Visiosoft\CatsModule\Category\CategoryModel;
use Visiosoft\CustomfieldsModule\CustomField\Contract\CustomFieldRepositoryInterface;
use Visiosoft\CustomfieldsModule\CustomField\CustomFieldModel;
use Visiosoft\LocationModule\City\CityModel;

class ArabamThemeSeeder extends Seeder
{

    private $fieldRepository;
    private $streamRepository;
    private $assignmentRepository;
    private $customFieldRepository;
    private $settingRepository;

    public function __construct(
        FieldRepositoryInterface $fieldRepository,
        StreamRepositoryInterface $streamRepository,
        AssignmentRepositoryInterface $assignmentRepository,
        CustomFieldRepositoryInterface $customFieldRepository,
        SettingRepositoryInterface $settingRepository
    )
    {
        parent::__construct();
        $this->fieldRepository = $fieldRepository;
        $this->streamRepository = $streamRepository;
        $this->assignmentRepository = $assignmentRepository;
        $this->customFieldRepository = $customFieldRepository;
        $this->settingRepository = $settingRepository;
    }

    /**
     * Run the seeder.
     */
    public function run()
    {
        $defaultPagesStream = $this->streamRepository->findBySlugAndNamespace('default_pages', 'pages');

        // Create locations repeater
        $locationsRepeater = $this->streamRepository->findBySlugAndNamespace('locations', 'repeater');
        if (!$locationsRepeater) {
            $locationsRepeater = $this->streamRepository->create([
                'name' => 'Locations',
                'namespace' => 'repeater',
                'slug' => 'locations',
                'prefix' => 'repeater_',
            ]);
        }


        // Assign repeater fields
        $locationsFields = [
            'city' => [
                'name' => 'City',
                'namespace' => 'repeater',
                'slug' => 'city',
                'type' => 'anomaly.field_type.relationship',
                "config" => [
                    "related" => CityModel::class,
                    "mode" => "search",
                ]
            ],
            'cover_photo' => [
                'name' => 'Cover Photo',
                'namespace' => 'repeater',
                'slug' => 'cover_photo',
                'type' => 'anomaly.field_type.file',
                "config" => [
                    "folders" => ['images'],
                ]
            ]
        ];

        foreach ($locationsFields as $locationsField) {
            $field = $this->fieldRepository->findBySlugAndNamespace($locationsField['slug'], $locationsField['namespace']);
            if (!$field) {
                $field = $this->fieldRepository->create([
                    'name' => $locationsField['name'],
                    'namespace' => $locationsField['namespace'],
                    'slug' => $locationsField['slug'],
                    'type' => $locationsField['type'],
                    'locked' => 0,
                    "config" => $locationsField['config']
                ]);

                $this->assignmentRepository->create([
                    'stream_id' => $locationsRepeater->getId(),
                    'field_id' => $field->id,
                    'label' => $locationsField['name'],
                    'config' => [
                        'required' => true,
                        'unique' => true
                    ]
                ]);
            }

        }

        // Assign pages fields
        $pagesFields = [
            'featured_makes' => [
                'name' => 'Featured Makes',
                'namespace' => 'pages',
                'slug' => 'featured_makes',
                'type' => 'anomaly.field_type.multiple',
                "config" => [
                    "related" => CategoryModel::class,
                    "mode" => "lookup",
                ]
            ],
            'locations' => [
                'name' => 'Locations',
                'namespace' => 'pages',
                'slug' => 'locations',
                'type' => 'anomaly.field_type.repeater',
                "config" => [
                    "related" => RepeaterLocationsEntryModel::class,
                    "add_row" => "visiosoft.theme.arabam::button.add_location",
                ]
            ]
        ];


        foreach ($pagesFields as $pagesField) {
            $field = $this->fieldRepository->findBySlugAndNamespace($pagesField['slug'], $pagesField['namespace']);
            if (!$field) {
                $field = $this->fieldRepository->create([
                    'name' => $pagesField['name'],
                    'namespace' => $pagesField['namespace'],
                    'slug' => $pagesField['slug'],
                    'type' => $pagesField['type'],
                    'locked' => 0,
                    "config" => $pagesField['config']
                ]);
                $this->assignmentRepository->create([
                    'stream_id' => $defaultPagesStream->getId(),
                    'field_id' => $field->id,
                    'label' => $pagesField['name'],
                ]);
            }
        }


        // Activate
        $this->settingRepository->set('streams::standard_theme', 'visiosoft.theme.arabam');

        // Seeder
        Model::unguard();
        DB::unprepared(file_get_contents(__DIR__ . '/arabam_demodata.sql'));
        Model::reguard();

        $zip = new \ZipArchive();
        $zip->open(__DIR__ . '/images.zip', \ZipArchive::CREATE);
        $zip->extractTo(storage_path('streams/default/files-module/local/images/'));
        $zip->close();

        Artisan::call('assets:clear');
    }
}