---
title: Security Checks
---

### Security Checks

Security checks are responsible for filtering login attempts and users. For example a security check could enforce that a certain criteria is met by the user. Or check that the login form is not being flooded.

#### Security Check Extension

This section will go over the `\Anomaly\UsersModule\User\Security\Contract\SecurityCheckInterface` class.

##### SecurityCheckExtension::attempt()

The `attempt` method is used to check security during a login attempt.

###### Returns: `\Illuminate\Http\RedirectResponse` or `true`

##### SecurityCheckExtension::check()

The `check` method is run during each request against a user.

###### Returns: `\Illuminate\Http\RedirectResponse` or `true`

###### Arguments

<table class="table table-bordered table-striped">

<thead>

<tr>

<th>Key</th>

<th>Required</th>

<th>Type</th>

<th>Default</th>

<th>Description</th>

</tr>

</thead>

<tbody>

<tr>

<td>

$user

</td>

<td>

false

</td>

<td>

`UserInterface`

</td>

<td>

null

</td>

<td>

The user to check security for.

</td>

</tr>

</tbody>

</table>

#### Writing Security Checks

This section will show you how to write your own custom security check extension.

##### Creating the extension

The first thing we need to do is to use the `make:addon` command to create our extension:

    php artisan make:addon anomaly.extension.user_security_check

##### Extending the security check extension

The extension you create must extend the `\Anomaly\UsersModule\User\Security\SecurityCheckExtension` class:

    <?php namespace Anomaly\UserSecurityCheckExtension;

    use Anomaly\UserSecurityCheckExtension\Command\CheckUser;
    use Anomaly\UsersModule\User\Contract\UserInterface;
    use Anomaly\UsersModule\User\Security\SecurityCheckExtension;
    use Symfony\Component\HttpFoundation\Response;

    class UserSecurityCheckExtension extends SecurityCheckExtension
    {

        /**
         * This extension provides a security check that
         * assures the user is active, enabled, etc.
         *
         * @var null|string
         */
        protected $provides = 'anomaly.module.users::security_check.user';

        /**
         * Check an HTTP request.
         *
         * @param  UserInterface $user
         * @return bool|Response
         */
        public function check(UserInterface $user = null)
        {
            if (!$user) {
                return true;
            }

            return $this->dispatch(new CheckUser($user));
        }

    }

You must define the `provides` property as `anomaly.module.users::security_check.your_widget_slug` so that it's picked up as a supported extension.

##### Validating security

The primary task of any security check is to validate a user. In this example we will use a command thats dispatched within the `check` method to check the user over and make sure they are valid and allowed:

    public function check(UserInterface $user = null)
    {
        if (!$user) {
            return true;
        }

        return $this->dispatch(new CheckUser($user));
    }

Our `CheckUser` command is responsible for the actual work:

    <?php namespace Anomaly\UserSecurityCheckExtension\Command;

    use Anomaly\Streams\Platform\Message\MessageBag;
    use Anomaly\UsersModule\User\Contract\UserInterface;
    use Anomaly\UsersModule\User\UserAuthenticator;
    use Illuminate\Routing\Redirector;

    class CheckUser
    {

        /**
         * The user instance.
         *
         * @var UserInterface
         */
        protected $user;

        /**
         * Create a new CheckUser instance.
         *
         * @param UserInterface $user
         */
        public function __construct(UserInterface $user)
        {
            $this->user = $user;
        }

        /**
         * @param  UserAuthenticator                      $authenticator
         * @param  MessageBag                             $message
         * @param  Redirector                             $redirect
         * @return bool|\Illuminate\Http\RedirectResponse
         */
        public function handle(UserAuthenticator $authenticator, MessageBag $message, Redirector $redirect)
        {
            if (!$this->user->isActivated()) {

                $message->error('Your account has not been activated.');

                $authenticator->logout(); // Just in case.

                return $redirect->back();
            }

            if (!$this->user->isEnabled()) {

                $message->error('Your account has been disabled.');

                $authenticator->logout(); // Just in case.

                return $redirect->back();
            }

            return true;
        }
    }
