---
title: Authenticators
---

### Authenticators

Authenticators are responsible for authenticating credentials and login attempts.

#### Authenticator Extension

This section will go over the `\Anomaly\UsersModule\User\Authenticator\AuthenticatorExtension` class.

##### AuthenticatorExtension::authenticate()

The `authenticate` method is responsible for authenticating the `credentials` and returning `null`, a `user`, or a `redirect`.

###### Returns: `\Anomaly\UsersModule\User\Contract\UserInterface` or `\Illuminate\Http\RedirectResponse` or `null`

###### Arguments

<table class="table table-bordered table-striped">

<thead>

<tr>

<th>Key</th>

<th>Required</th>

<th>Type</th>

<th>Default</th>

<th>Description</th>

</tr>

</thead>

<tbody>

<tr>

<td>

$credentials

</td>

<td>

true

</td>

<td>

array

</td>

<td>

none

</td>

<td>

The login information.

</td>

</tr>

</tbody>

</table>

#### Writing Authenticators

This section will show you how to write your own custom authenticator extension.

##### Creating the extension

The first thing we need to do is to use the `make:addon` command to create our extension:

    php artisan make:addon anomaly.extension.default_authenticator

##### Extending the authenticator extension

The extension you create must extend the `\Anomaly\UsersModule\User\Authenticator\AuthenticatorExtension` class:

    <?php namespace Anomaly\DefaultAuthenticatorExtension;

    use Anomaly\DefaultAuthenticatorExtension\Command\AuthenticateCredentials;
    use Anomaly\UsersModule\User\Authenticator\AuthenticatorExtension;
    use Anomaly\UsersModule\User\Contract\UserInterface;

    class DefaultAuthenticatorExtension extends AuthenticatorExtension
    {

        /**
         * This extensions provides a basic
         * authenticator for the users module.
         *
         * @var string
         */
        protected $provides = 'anomaly.module.users::authenticator.default';

        /**
         * Authenticate a set of credentials.
         *
         * @param array $credentials
         * @return null|UserInterface
         */
        public function authenticate(array $credentials)
        {
            return $this->dispatch(new AuthenticateCredentials($credentials));
        }
    }

You must define the `provides` property as `anomaly.module.users::authenticator.your_widget_slug` so that it's picked up as a supported extension.

##### Authenticating credentials

The primary task of any authenticators is to authenticate a login request. In this example we will use a command thats dispatched within the `authenticate` method to check the credentials:

    public function authenticate(array $credentials)
    {
        return $this->dispatch(new AuthenticateCredentials($credentials));
    }

Our `AuthenticateCredentials` command is responsible for the actual work:

    <?php namespace Anomaly\DefaultAuthenticatorExtension\Command;

    use Anomaly\UsersModule\User\Contract\UserRepositoryInterface;

    class AuthenticateCredentials
    {

        /**
         * The credentials to authenticate.
         *
         * @var array
         */
        protected $credentials;

        /**
         * Create a new AuthenticateCredentials instance.
         *
         * @param array $credentials
         */
        public function __construct(array $credentials)
        {
            $this->credentials = $credentials;
        }

        /**
         * Handle the command.
         *
         * @param  UserRepositoryInterface                               $users
         * @return \Anomaly\UsersModule\User\Contract\UserInterface|null
         */
        public function handle(UserRepositoryInterface $users)
        {
            if (!isset($this->credentials['password']) && !isset($this->credentials['email'])) {
                return null;
            }

            return $users->findByCredentials($this->credentials);
        }
    }

##### Redirecting authentication requests

The `authenticate` method can return an instance of the user, null, or a redirect instance. In the case a redirect is returns the request will be redirected immediately. After the redirect is made the authentication will be in your hands!
