<?php namespace Anomaly\ApiModule;

use Anomaly\ApiModule\Command\LoadKeys;
use Anomaly\ApiModule\Command\LoadScopes;
use Anomaly\ApiModule\Passport\PassportServiceProvider;
use Anomaly\ApiModule\User\UserModel;
use Anomaly\Streams\Platform\Addon\AddonServiceProvider;
use Anomaly\Streams\Platform\Model\EloquentModel;
use Illuminate\Contracts\Config\Repository;
use Illuminate\Contracts\View\Factory;
use Illuminate\Http\Request;
use Illuminate\Routing\Router;
use Laravel\Passport\Http\Middleware\CheckForAnyScope;
use Laravel\Passport\Http\Middleware\CheckScopes;
use Laravel\Passport\Passport;

/**
 * Class ApiModuleServiceProvider
 *
 * @link   http://pyrocms.com/
 * @author PyroCMS, Inc. <support@pyrocms.com>
 * @author Ryan Thompson <ryan@pyrocms.com>
 */
class ApiModuleServiceProvider extends AddonServiceProvider
{

    /**
     * The addon providers.
     *
     * @var array
     */
    protected $providers = [
        PassportServiceProvider::class,
    ];

    /**
     * The addon route middlewares.
     *
     * @var array
     */
    protected $routeMiddleware = [
        'scopes' => CheckScopes::class,
        'scope'  => CheckForAnyScope::class,
    ];

    /**
     * The addon routes.
     *
     * @var array
     */
    protected $routes = [
        'oauth/request'       => 'Anomaly\ApiModule\Http\Controller\OAuthController@request',
        'admin/api'           => 'Anomaly\ApiModule\Http\Controller\Admin\ClientsController@index',
        'admin/api/create'    => 'Anomaly\ApiModule\Http\Controller\Admin\ClientsController@create',
        'admin/api/edit/{id}' => 'Anomaly\ApiModule\Http\Controller\Admin\ClientsController@edit',
    ];

    /**
     * Register the addon.
     *
     * @param Request $request
     * @param Repository $config
     * @param ApiModule $module
     * @param Factory $views
     */
    public function register(
        Request $request,
        Repository $config,
        ApiModule $module,
        Factory $views,
        EloquentModel $model
    ) {
        $views->addNamespace('passport', $module->getPath('resources/views'));

        $config->set('auth.guards.api.driver', 'passport');

        if ($request->segment(1) == 'api') {
            $config->set('auth.providers.users.model', UserModel::class);
        }

        $model->bind(
            'to_array_for_api',
            function () {

                /* @var EloquentModel $this */
                return $this->toArray();
            }
        );
    }

    /**
     * Boot the addon.
     */
    public function boot()
    {
        $this->dispatch(new LoadKeys());
        $this->dispatch(new LoadScopes());
    }

    /**
     * Map additional routes.
     *
     * @param Router $router
     */
    public function map(Router $router)
    {
        $this->mapStreamsApi($router);
        $this->mapEntriesApi($router);

        Passport::routes();
    }

    /**
     * Map the entries API.
     *
     * @param Router $router
     */
    protected function mapEntriesApi(Router $router)
    {
        $router->get(
            'api/entries/{namespace}/{stream}',
            [
                'uses'           => 'Anomaly\ApiModule\Http\Controller\Resource\EntriesController@index',
                'streams::addon' => 'anomaly.module.api',
            ]
        )->middleware('auth:api');

        $router->post(
            'api/entries/{namespace}/{stream}',
            [
                'uses'           => 'Anomaly\ApiModule\Http\Controller\Resource\EntriesController@store',
                'streams::addon' => 'anomaly.module.api',
            ]
        )->middleware('auth:api');

        $router->get(
            'api/entries/{namespace}/{stream}/{id}/{map?}',
            [
                'uses'           => 'Anomaly\ApiModule\Http\Controller\Resource\EntriesController@show',
                'streams::addon' => 'anomaly.module.api',
            ]
        )->where(['map' => '(.*)'])->middleware('auth:api');

        $router->put(
            'api/entries/{namespace}/{stream}/{id}',
            [
                'uses'           => 'Anomaly\ApiModule\Http\Controller\Resource\EntriesController@update',
                'streams::addon' => 'anomaly.module.api',
            ]
        )->middleware('auth:api');

        $router->patch(
            'api/entries/{namespace}/{stream}/{id}',
            [
                'uses'           => 'Anomaly\ApiModule\Http\Controller\Resource\EntriesController@update',
                'streams::addon' => 'anomaly.module.api',
            ]
        )->middleware('auth:api');

        $router->delete(
            'api/entries/{namespace}/{stream}/{id}',
            [
                'uses'           => 'Anomaly\ApiModule\Http\Controller\Resource\EntriesController@delete',
                'streams::addon' => 'anomaly.module.api',
            ]
        )->middleware('auth:api');
    }

    /**
     * Map the streams API.
     *
     * @param Router $router
     */
    protected function mapStreamsApi(Router $router)
    {
        $router->get(
            'api/streams',
            [
                'uses'           => 'Anomaly\ApiModule\Http\Controller\Resource\StreamsController@index',
                'streams::addon' => 'anomaly.module.api',
            ]
        )->middleware('auth:api');

        $router->post(
            'api/streams',
            [
                'uses'           => 'Anomaly\ApiModule\Http\Controller\Resource\StreamsController@store',
                'streams::addon' => 'anomaly.module.api',
            ]
        )->middleware('auth:api');

        $router->get(
            'api/streams/{namespace}',
            [
                'uses'           => 'Anomaly\ApiModule\Http\Controller\Resource\StreamsController@streams',
                'streams::addon' => 'anomaly.module.api',
            ]
        )->middleware('auth:api');

        $router->get(
            'api/streams/{namespace}/{slug}',
            [
                'uses'           => 'Anomaly\ApiModule\Http\Controller\Resource\StreamsController@show',
                'streams::addon' => 'anomaly.module.api',
            ]
        )->middleware('auth:api');

        $router->put(
            'api/streams/{namespace}/{slug}',
            [
                'uses'           => 'Anomaly\ApiModule\Http\Controller\Resource\StreamsController@update',
                'streams::addon' => 'anomaly.module.api',
            ]
        )->middleware('auth:api');

        $router->patch(
            'api/streams/{namespace}/{slug}',
            [
                'uses'           => 'Anomaly\ApiModule\Http\Controller\Resource\StreamsController@update',
                'streams::addon' => 'anomaly.module.api',
            ]
        )->middleware('auth:api');

        $router->delete(
            'api/streams/{namespace}/{slug}',
            [
                'uses'           => 'Anomaly\ApiModule\Http\Controller\Resource\StreamsController@delete',
                'streams::addon' => 'anomaly.module.api',
            ]
        )->middleware('auth:api');
    }
}
